#!/usr/bin/perl
#------------------------------------------------------------------------------
#    ThreadPlus, simply forum CGI.
#    Copyright (C) 2012,2013 Kaga, Hiroaki
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#------------------------------------------------------------------------------
use strict;
use warnings;

use constant {
    LOCK_SH => 1,
    LOCK_EX => 2,
    LOCK_NB => 4,
    LOCK_UN => 8,
};

use CGI;
use Encode;

use HTML::Template;

require "config.cgi";
require "subs.cgi";

# ユーザーエージェントから携帯かどうかを判定
my $device = 'pc';
my $ua = $ENV{'HTTP_USER_AGENT'};
if (defined $ua && $ua =~ /\ADoCoMo|\AUP.Browser|\AKDDI|\AJ-PHONE|\AVodafone|\ASoftBank/) {
    $device = 'mobile';
}

my $cgi = CGI->new();

# パスワードが設定されてる場合はチェック
my $sid = _pass_check($device, $cgi);

my $a = $cgi->param('a');
if (defined $a && $a eq 'exec') {
    _exec($cgi);
}
else {
    _view($cgi);
}

exit;

# ----- control functions -----
sub _view {
    my ($cgi) = @_;

    my $func = $cgi->param('func');
    $func = '' if (!defined $func);
    if ($func eq 'new') {
        _view_new();
    }
    elsif ($func eq 'thread') {
        _view_thread($cgi);
    }
    elsif ($func eq 'delete') {
        _view_delete($cgi);
    }
    elsif ($func eq 'remove') {
        _view_remove($cgi);
    }
    else {
        _view_list($cgi);
    }
}

sub _exec {
    my ($cgi) = @_;

    my $func = $cgi->param('func');
    if ($func eq 'new') {
        _exec_new($cgi);
    }
    elsif ($func eq 'thread') {
        _exec_thread($cgi);
    }
    elsif ($func eq 'delete') {
        _exec_delete($cgi);
    }
    elsif ($func eq 'remove') {
        _exec_remove($cgi);
    }
}

# ----- list functions -----
sub _view_list {
    my ($cgi) = @_;

    my $page = $cgi->param('page');
    if (!(defined $page) || $page eq '') {
        $page = 1;
    }
    if ($page !~ /\A[0-9]+\z/) {
        _error($device);
    }

    my @items = _get_update_items();
    _view_html($page, @items);
}

sub _view_html {
    my ($page, @items) = @_;
#    _logger("_view_html : items = " .  scalar(@items));   # for debug

    my $listtitle = 'スレッド一覧';
    my $toplist  = '|&lt; 先頭へ';
    my $prevlist = '&lt; 前へ';
    my $nextlist = '次へ &gt;';

    my $pagenavi = '';
    my @loop_data = ();
    my $itemnum = scalar(@items);
    if ($itemnum > 0) {
        my $linenum = get_line_number();
        my $pagenum = int($itemnum / $linenum); 
        if (($itemnum / $linenum) > $pagenum) {
            $pagenum++;
        }
        my @pageitems = ();
        my $start = ($page - 1) * $linenum;
        my $end = $page * $linenum - 1;
        if ($itemnum < $end) {
            $end = $itemnum;
        }
        for my $i ($start .. $end) {
            push @pageitems, $items[$i];
        }

        foreach my $threadid (@pageitems) {
            my @datarecs = _get_threaddata($threadid);
            next if (!scalar(@datarecs));
            my ($id, $datetime, $subject, $username, $mailaddr, $body, $ipaddr) = split(/,/, $datarecs[0]);
            $subject  =~ s/enc_conma/,/g;
            $mailaddr =~ s/enc_conma/,/g;
            $username =~ s/enc_conma/,/g;
            $body     =~ s/enc_conma/,/g;
            $body     =~ s/enc_crlf/<br \/>/g;

            $body = _url_convert($body); # URLをリンクに変換

            my $number = scalar(@datarecs) - 1;

            if ($username eq '') {
                $username = "[" . get_anonymous() . "]";
            }

            my %row_data;
            $row_data{THREADID} = $threadid;
            $row_data{SUBJECT} = $subject;
            $row_data{BODY} = $body;
            $row_data{USERNAME} = $username;
            $row_data{DATETIME} = $datetime;
            $row_data{NUMBER} = $number;
            $row_data{SID} = $sid;
            push @loop_data, \%row_data;
        }

        my $base = 1;
        my $currentnum = ($pagenum < 5) ? $pagenum : 5;
        if ($page > 5) {
            $base = $page - 4;
        }
        my @pagelist = ();
        for my $pagecount ($base .. ($base + $currentnum - 1)) {
            push @pagelist, $pagecount;
        }
        my $current = '';
        foreach my $pagecount (@pagelist) {
            if ($pagecount != $page) {
                $current .= "<a href=\"bbs.cgi?page=$pagecount&s=$sid\">$pagecount</a>&nbsp;";
            }
            else {
                $current .= "$pagecount&nbsp;";
            }
        }
        my $prev = ''; 
        if ($page > 1) {
            my $prevpage = $page - 1;
            $prev = "<a href=\"bbs.cgi?page=1&s=$sid\">$toplist</a>&nbsp;&nbsp;";
            $prev .= "<a href=\"bbs.cgi?page=$prevpage&s=$sid\">$prevlist</a>";
        }
        my $next = ''; 
        if ($page < $pagenum) {
            my $nextpage = $page + 1;
            $next = "<a href=\"bbs.cgi?page=$nextpage&s=$sid\">$nextlist</a>";
        }
        $pagenavi = <<"END_PAGENAVI";
<strong>$prev&nbsp;&nbsp;$current&nbsp;$next</strong>
END_PAGENAVI
    }

    my $template = HTML::Template->new(filename => get_tmpl_dir() . '/list.tmpl');

    $template->param(SITETITLE => get_site_title());
    $template->param(SITEURL => get_site_url());
    $template->param(SUBTITLE => get_sub_title());
    $template->param(LISTTITLE => $listtitle);
    $template->param(THREAD_NUM => scalar(@loop_data));
    $template->param(THREAD_LIST => \@loop_data);
    $template->param(PAGENAVI => $pagenavi);
    $template->param(SESSIONID => $sid);
    if ($device eq 'mobile') {
        $template->param(CHARSET => 'SHIFT_JIS');
    }
    else {
        $template->param(CHARSET => 'UTF-8');
    }

    my $pagesource = $template->output;
    if ($device eq 'mobile') {
        Encode::from_to($pagesource, 'utf8', 'shiftjis'); # シフトJISに変換
    }
    print "Content-Type: text/html\n\n", $pagesource;
}

# ----- thread functions -----
sub _view_thread {
    my ($cgi) = @_;
    my $threadid = $cgi->param('threadid');
    if ($threadid !~ /\A[0-9]+\z/) {
        _error($device);
    }

    my $formdata;
    $formdata->{threadid} = $threadid;
    $formdata->{username} = '';
    $formdata->{mailaddr} = '';
    $formdata->{body}     = '';

    _show_thread($formdata, '');
}

sub _exec_thread {
    my ($cgi) = @_;
#    _logger("_exec_thread : ");   # for debug

    my $formdata;
    $formdata->{threadid} = $cgi->param('threadid');
    if ($formdata->{threadid} !~ /\A[0-9]+\z/) {
        _error($device);
    }
    $formdata->{threadtitle} = _get_threadtitle($formdata->{threadid});
    $formdata->{username} = _tag_invalidate($cgi->param('username'));
    $formdata->{mailaddr} = _tag_invalidate($cgi->param('mailaddr'));
    $formdata->{body}     = _tag_invalidate($cgi->param('body'));

    if (($formdata->{mailaddr} ne '') && ($formdata->{mailaddr} !~ /.+\@.+[.].+/)) {
        _show_thread($formdata, 'メールアドレスの形式が間違っています');
    }
    if ($formdata->{body} eq '') {
        _show_thread($formdata, '本文を入力してください');
    }
    else {
        if ($formdata->{body} =~ /\A[\x20-\x7E\r\n]+\z/) {
            _show_thread($formdata, '本文は日本語で入力してください');
        }
    }

    # リファラーをチェック（スパム対策）
    my $referer = $ENV{'HTTP_REFERER'};
    my $site_url = get_site_url();
    if ($referer !~ /\A$site_url/) {
        _show_thread($formdata, '外部サイトからは投稿できません');
    }
    # ブラックリストをチェック（スパム対策）
    if (_check_blacklist("$formdata->{username} $formdata->{mailaddr} $formdata->{body} " . $ENV{'REMOTE_ADDR'})) {
        _write_spam($ENV{'REMOTE_ADDR'}, $formdata->{body});
        _show_thread($formdata, 'ブラックリストに登録されています');
    }

    _add_comment($formdata);
    _add_update($formdata->{threadid});

    print "Location: bbs.cgi?a=view&func=thread&threadid=$formdata->{threadid}&s=$sid", "\n\n";
}

sub _show_thread {
    my ($formdata, $msg) = @_;
    _logger("_show_thread : ");   # for debug

    my @datarecs = _get_threaddata($formdata->{threadid});
    my ($threadid, $threaddate, $subject, $threaduser, $mailaddr, $threadbody, $ipaddr) = split(/,/, $datarecs[0]);
    $subject =~ s/enc_conma/,/g;
    $threaduser =~ s/enc_conma/,/g;
    $mailaddr =~ s/enc_conma/,/g;
    $threadbody =~ s/enc_conma/,/g;
    $threadbody =~ s/enc_crlf/<br \/>/g;

    $threadbody = _url_convert($threadbody); # URLをリンクに変換

    if ($threaduser eq '') {
        $threaduser = "[" . get_anonymous() . "]";
    }

    my @loop_data = ();
    for (my $i = 1; $i < scalar(@datarecs); $i++) {
        my ($commentid, $postdate, $name, $mailaddr, $commentbody, $ipaddr) = split(/,/, $datarecs[$i]);
        $mailaddr =~ s/enc_conma/,/g;
        $name =~ s/enc_conma/,/g;
        if ($name eq '') {
            $name = "[" . get_anonymous() . "]";
        }
        $commentbody =~ s/enc_conma/,/g;
        $commentbody =~ s/enc_crlf/<br \/>/g;

        $commentbody = _url_convert($commentbody); # URLをリンクに変換

        # >>$commentidに対してリンクを張る
        $commentbody =~ s/>>([0-9]+)/<a href="#comment$1">&gt;&gt;$1<\/a>/g;
        $commentbody =~ s/&gt;&gt;([0-9]+)/<a href="#comment$1">&gt;&gt;$1<\/a>/g;

        my %row_data;
        $row_data{COMMENTID} = $commentid;
        $row_data{COMMENTBODY} = $commentbody;
        $row_data{NAME} = $name;
        $row_data{POSTDATE} = $postdate;
        $row_data{DELMENU} = " &nbsp;<a href=\"bbs.cgi?a=view&func=remove&threadid=$formdata->{threadid}&commentid=$commentid&s=$sid\">削除</a>";
        push @loop_data, \%row_data;
    }

    if ($msg ne '') {
        $msg = "<p><div id=\"message\">$msg</div></p>";
    }

    _access_log($threadid, $device);   # アクセスログを記録
    my $template = HTML::Template->new(filename => get_tmpl_dir() . '/thread.tmpl');

    $template->param(SUBJECT => $subject);
    $template->param(THREADID => $formdata->{threadid});
    $template->param(THREADBODY => $threadbody);
    $template->param(THREADUSER => $threaduser);
    $template->param(THREADDATE => $threaddate);
    $template->param(COMMENT_NUM => scalar(@loop_data));
    $template->param(COMMENT_LIST => \@loop_data);
    $template->param(MSG => $msg);
    $template->param(USERNAME => $formdata->{username});
    $template->param(MAILADDR => $formdata->{mailaddr});
    $template->param(BODY => $formdata->{body});
    $template->param(SESSIONID => $sid);
    if ($device eq 'mobile') {
        $template->param(CHARSET => 'SHIFT_JIS');
    }
    else {
        $template->param(CHARSET => 'UTF-8');
    }

    my $pagesource = $template->output;
    if ($device eq 'mobile') {
        Encode::from_to($pagesource, 'utf8', 'shiftjis'); # シフトJISに変換
    }
    print "Content-Type: text/html\n\n", $pagesource;

    exit;
}

sub _add_comment {
    my ($formdata) = @_;
#    _logger("_add_comment : ");   # for debug

    my $threadid = $formdata->{threadid};
    my $datafile = get_data_dir() . "/thread_$threadid.data";

    my $threadtitle = _get_threadtitle($threadid);
    my $body = $formdata->{body};
    my $name = $formdata->{username};
    if ($name eq '') {
        $name = "[" . get_anonymous() . "]";
    }

    $formdata->{username} =~ s/[ ]/&nbsp;/g;
    $formdata->{mailaddr} =~ s/[ ]/&nbsp;/g;
    $formdata->{body}     =~ s/[ ]/&nbsp;/g;

    $formdata->{username} =~ s/,/enc_conma/g;
    $formdata->{mailaddr} =~ s/,/enc_conma/g;
    $formdata->{body}     =~ s/,/enc_conma/g;
    $formdata->{username} =~ s/\r?\n//g;
    $formdata->{mailaddr} =~ s/\r?\n//g;
    $formdata->{body}     =~ s/\r?\n/enc_crlf/g;

    my $newcommentid = _get_newcommentid($threadid);
    my $ipaddr = $ENV{'REMOTE_ADDR'};
    my $postdate = _get_datetime();
    my $datarec = "$newcommentid,$postdate,$formdata->{username},$formdata->{mailaddr},$formdata->{body},$ipaddr";

    open my $datafh, '>>', $datafile;
    flock $datafh, LOCK_EX;
    print {$datafh} "$datarec\n";
    flock $datafh, LOCK_UN;
    close $datafh;
    chmod 0766, $datafile;

    my $sitetitle = get_site_title();
    my $siteurl = get_site_url();
    my $threadurl = $siteurl . "bbs.cgi?a=view&func=thread&threadid=$formdata->{threadid}&s=$sid";
    my $subject = 'コメントが投稿されました';
    my $mailbody = <<"END_MAILBODY";
$sitetitle
$siteurl

$threadtitle($threadurl)に$nameさんが以下のコメントを投稿しました。

-----

$newcommentid：

$body

Posted by $name at $postdate

-----
END_MAILBODY

    _mailto($subject, get_webmaster_email(), $mailbody, get_webmaster_email());
}

# ----- remove comment functions -----
sub _view_remove {
    my ($cgi) = @_;
#    _logger("_view_remove : ");   # for debug

    my $formdata;
    $formdata->{threadid} = $cgi->param('threadid');
    if ($formdata->{threadid} !~ /\A[0-9]+\z/) {
        _error($device);
    }
    $formdata->{commentid} = $cgi->param('commentid');
    if ($formdata->{commentid} !~ /\A[0-9]+\z/) {
        _error($device);
    }

    _show_remove($formdata, '');
}

sub _exec_remove {
    my ($cgi) = @_;
#    _logger("_exec_remove : ");   # for debug

    my $formdata;
    $formdata->{threadid} = $cgi->param('threadid');
    if ($formdata->{threadid} !~ /\A[0-9]+\z/) {
        _error($device);
    }
    $formdata->{commentid} = $cgi->param('commentid');
    if ($formdata->{commentid} !~ /\A[0-9]+\z/) {
        _error($device);
    }
    my $adminpass = $cgi->param('adminpass');
    if ($adminpass ne get_admin_password()) {
        _show_remove($formdata, '管理者パスワードが違います');
    }

    _remove_comment($formdata->{threadid}, $formdata->{commentid});

    print "Location: bbs.cgi?a=view&func=thread&threadid=$formdata->{threadid}&s=$sid", "\n\n";
}

sub _show_remove {
    my ($formdata, $msg) = @_;
#    _logger("_show_remove : ");   # for debug

    if ($msg ne '') {
        $msg = "<p><div id=\"message\">$msg</div></p>";
    }

    my @hitrec = ();
    my @datarecs = _get_threaddata($formdata->{threadid});
    shift @datarecs;
    foreach my $datarec (@datarecs) {
        my @fields = split(/,/, $datarec);
        if ($fields[0] == $formdata->{commentid}) {
            @hitrec = @fields;
            last;
        }
    }

    my ($id, $postdate, $name, $mailaddr, $body, $ipaddr) = @hitrec;
    $name =~ s/enc_conma/,/g;
    $mailaddr =~ s/enc_conma/,/g;
    $body =~ s/enc_conma/,/g;
    $body =~ s/enc_crlf/<br \/>/g;

    if ($name eq '') {
        $name = "[" . get_anonymous() . "]";
    }

    my $template = HTML::Template->new(filename => get_tmpl_dir() . '/remove.tmpl');

    $template->param(MSG => $msg);
    $template->param(THREADID => $formdata->{threadid});
    $template->param(COMMENTID => $formdata->{commentid});
    $template->param(BODY => $body);
    $template->param(NAME => $name);
    $template->param(POSTDATE => $postdate);
    $template->param(SESSIONID => $sid);
    if ($device eq 'mobile') {
        $template->param(CHARSET => 'SHIFT_JIS');
    }
    else {
        $template->param(CHARSET => 'UTF-8');
    }

    my $pagesource = $template->output;
    if ($device eq 'mobile') {
        Encode::from_to($pagesource, 'utf8', 'shiftjis'); # シフトJISに変換
    }
    print "Content-Type: text/html\n\n", $pagesource;

    exit;
}

sub _remove_comment {
    my ($threadid, $commentid) = @_;
#    _logger("_remove_comment : ");   # for debug
    my @datarecs = _get_threaddata($threadid);
    my @newrecs = ();
    push @newrecs, $datarecs[0];

    shift @datarecs;
    foreach my $datarec (@datarecs) {
        my @fields = split(/,/, $datarec);
        next if ($fields[0] == $commentid);
        push @newrecs, $datarec;
    }

    _update_threaddata($threadid, @newrecs);
}

# ----- thread functions -----
sub _view_new {

    my ($title, $quote) = ('', '');
    my $url = $cgi->param('url');
    if ($url ne '') {
        ($title, $quote) = _get_quote($url);
        $quote = $cgi->param('quote');
        $quote .= "\n$url\n";
    }

    my $formdata;
    $formdata->{subject}  = $title;
    $formdata->{username} = '';
    $formdata->{mailaddr} = '';
    $formdata->{body}     = $quote;
    _show_new($formdata, '');
}

sub _exec_new {
    my ($cgi) = @_;
#    _logger("_exec_new : ");   # for debug

    my $formdata;
    $formdata->{subject}  = _tag_invalidate($cgi->param('subject'));
    $formdata->{username} = _tag_invalidate($cgi->param('username'));
    $formdata->{mailaddr} = _tag_invalidate($cgi->param('mailaddr'));
    $formdata->{body}     = _tag_invalidate($cgi->param('body'));

    if ($formdata->{subject} eq '') {
        _show_new($formdata, '件名を入力してください');
    }
    if (($formdata->{mailaddr} ne '') && ($formdata->{mailaddr} !~ /.+\@.+[.].+/)) {
        _show_new($formdata, 'メールアドレスの形式が間違っています');
    }
    if ($formdata->{body} eq '') {
        _show_new($formdata, '本文を入力してください');
    }
    else {
        if ($formdata->{body} =~ /\A[\x20-\x7E\r\n]+\z/) {
            _show_new($formdata, '本文は日本語で入力してください');
        }
    }

    # リファラーをチェック（スパム対策）
    my $referer = $ENV{'HTTP_REFERER'};
    my $site_url = get_site_url();
    if ($referer !~ /\A$site_url/) {
        _show_new($formdata, '外部サイトからは投稿できません');
    }
    # ブラックリストをチェック（スパム対策）
    if (_check_blacklist("$formdata->{subject} $formdata->{username} $formdata->{mailaddr} $formdata->{body} " . $ENV{'REMOTE_ADDR'})) {
        _write_spam($ENV{'REMOTE_ADDR'}, $formdata->{body});
        _show_new($formdata, 'ブラックリストに登録されています');
    }

    $formdata->{threadid} = _get_threadid() + 1;
    _update_threadid($formdata->{threadid});

    # add thread data
    _add_thread($formdata);
    _add_update($formdata->{threadid});

    print "Location: bbs.cgi?a=view&func=thread&threadid=$formdata->{threadid}&s=$sid", "\n\n";
}

sub _show_new {
    my ($formdata, $msg) = @_;
#    _logger("_show_new : ");   # for debug

    if ($msg ne '') {
        $msg = "<p><div id=\"message\">$msg</div></p>";
    }

    my $template = HTML::Template->new(filename => get_tmpl_dir() . '/new.tmpl');

    $template->param(MSG => $msg);
    $template->param(SUBJECT => $formdata->{subject});
    $template->param(USERNAME => $formdata->{username});
    $template->param(MAILADDR => $formdata->{mailaddr});
    $template->param(BODY => $formdata->{body});
    $template->param(SESSIONID => $sid);
    if ($device eq 'mobile') {
        $template->param(CHARSET => 'SHIFT_JIS');
    }
    else {
        $template->param(CHARSET => 'UTF-8');
    }

    my $pagesource = $template->output;
    if ($device eq 'mobile') {
        Encode::from_to($pagesource, 'utf8', 'shiftjis'); # シフトJISに変換
    }
    print "Content-Type: text/html\n\n", $pagesource;

    exit;
}

sub _add_thread {
    my ($formdata) = @_;
#    _logger("_add_thread : ");   # for debug

    my $name = $formdata->{username};
    if ($name eq '') {
        $name = "[" . get_anonymous() . "]";
    }
    my $threadtitle = $formdata->{subject};
    my $body = $formdata->{body};

    $formdata->{subject}  =~ s/[ ]/&nbsp;/g;
    $formdata->{username} =~ s/[ ]/&nbsp;/g;
    $formdata->{mailaddr} =~ s/[ ]/&nbsp;/g;
    $formdata->{body}     =~ s/[ ]/&nbsp;/g;

    $formdata->{subject}  =~ s/,/enc_conma/g;
    $formdata->{username} =~ s/,/enc_conma/g;
    $formdata->{mailaddr} =~ s/,/enc_conma/g;
    $formdata->{body}     =~ s/,/enc_conma/g;
    $formdata->{subject}  =~ s/\r?\n//g;
    $formdata->{username} =~ s/\r?\n//g;
    $formdata->{mailaddr} =~ s/\r?\n//g;
    $formdata->{body}     =~ s/\r?\n/enc_crlf/g;

    my $ipaddr = $ENV{'REMOTE_ADDR'};
    my $postdate = _get_datetime();
    my $datarec = "$formdata->{threadid},$postdate,$formdata->{subject},$formdata->{username},$formdata->{mailaddr},$formdata->{body},$ipaddr";

    my $datafile = get_data_dir() . "/thread_$formdata->{threadid}.data";
    open my $datafh, '>', $datafile;
    flock $datafh, LOCK_EX;
    print {$datafh} "$datarec\n";
    flock $datafh, LOCK_UN;
    close $datafh;
    chmod 0766, $datafile;

    my $sitetitle = get_site_title();
    my $siteurl = get_site_url();
    my $threadurl = $siteurl . "bbs.cgi?a=view&func=thread&threadid=$formdata->{threadid}&s=$sid";
    my $subject = 'スレッドが投稿されました';
    my $mailbody = <<"END_MAILBODY";
$sitetitle
$siteurl

$nameさんが以下のスレッドを投稿しました。

-----

[$formdata->{threadid}] $threadtitle
$threadurl

$body

Posted by $name at $postdate

-----
END_MAILBODY

    _mailto($subject, get_webmaster_email(), $mailbody, get_webmaster_email());

}

# ----- delete functions -----
sub _view_delete {
    my ($cgi) = @_;
#    _logger("_view_delete : ");   # for debug

    my $formdata;
    $formdata->{threadid} = $cgi->param('threadid');
    if ($formdata->{threadid} !~ /\A[0-9]+\z/) {
        _error($device);
    }

    _show_delete($formdata, '');
}

sub _exec_delete {
    my ($cgi) = @_;
#    _logger("_exec_delete : ");   # for debug

    my $formdata;
    $formdata->{threadid} = $cgi->param('threadid');
    if ($formdata->{threadid} !~ /\A[0-9]+\z/) {
        _error($device);
    }

    my $adminpass = $cgi->param('adminpass');
    if ($adminpass ne get_admin_password()) {
        _show_delete($formdata, '管理者パスワードが違います');
    }

    _delete_thread($formdata->{threadid});
    _delete_update($formdata->{threadid});

    print "Location: bbs.cgi?s=$sid", "\n\n";
}

sub _show_delete {
    my ($formdata, $msg) = @_;
#    _logger("_show_delete : ");   # for debug

    if ($msg ne '') {
        $msg = "<p><div id=\"message\">$msg</div></p>";
    }

    my @recs = _get_threaddata($formdata->{threadid});
    ($formdata->{datetime}, $formdata->{subject}, $formdata->{username}, $formdata->{mailaddr}, $formdata->{body}) = (split(/,/, $recs[0]))[1..5];

    $formdata->{subject}  =~ s/enc_conma/,/g;
    $formdata->{username} =~ s/enc_conma/,/g;
    $formdata->{mailaddr} =~ s/enc_conma/,/g;
    $formdata->{body}     =~ s/enc_conma/,/g;
    $formdata->{body}     =~ s/enc_crlf/<br \/>/g;

    my $name = $formdata->{username};
    if ($name eq '') {
        $name = "[" . get_anonymous() . "]";
    }

     my $template = HTML::Template->new(filename => get_tmpl_dir() . '/delete.tmpl');

    $template->param(MSG => $msg);
    $template->param(SUBJECT => $formdata->{subject});
    $template->param(THREADID => $formdata->{threadid});
    $template->param(BODY => $formdata->{body});
    $template->param(USERNAME => $name);
    $template->param(DATETIME => $formdata->{datetime});
    $template->param(SESSIONID => $sid);
    if ($device eq 'mobile') {
        $template->param(CHARSET => 'SHIFT_JIS');
    }
    else {
        $template->param(CHARSET => 'UTF-8');
    }

    my $pagesource = $template->output;
    if ($device eq 'mobile') {
        Encode::from_to($pagesource, 'utf8', 'shiftjis'); # シフトJISに変換
    }
    print "Content-Type: text/html\n\n", $pagesource;

    exit;
}

sub _delete_thread {
    my $threadid = shift;
#    _logger("_delete_thread : ");   # for debug
    my $datafile = get_data_dir() . "/thread_$threadid.data";
    my $deletefile = get_data_dir() . "/thread_$threadid.deleted";
    rename $datafile, $deletefile;
}

