/*
 * Copyright 2009- kensir0u.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package net.sf.thirdi.validation.message;

import java.lang.reflect.Array;
import java.util.Locale;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.ResourceBundle;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import net.sf.thirdi.validation.spec.ConstraintDescriptor;
import net.sf.thirdi.validation.spec.MessageEditor;

public class MessageEditorImpl implements MessageEditor {

	private static final String BIND_SYMBOL = "#";

	private static final Pattern messagePattern = Pattern
			.compile("\\{([a-zA-Z_0-9" + BIND_SYMBOL + ".]+)\\}");

	private static final String DEFAULT_MESSAGE_BUNDLE_NAME = "ValidationMessages";

	private ResourceBundle messageBundle;

	private Locale locale;

	private String messageBundleName;

	public String bind(String message,
			ConstraintDescriptor constraintdescriptor, Object obj) {
		if (message == null || message.length() == 0)
			return message;
		else
			return messageBind(message, constraintdescriptor, obj);
	}

	public String bind(String message,
			ConstraintDescriptor constraintdescriptor, Object obj, Locale ignore) {
		if (message == null || message.length() == 0) {
			return message;
		} else {
			return messageBind(message, constraintdescriptor, obj);
		}
	}

	public MessageEditorImpl() {
		this.locale = Locale.getDefault();
		this.messageBundleName = DEFAULT_MESSAGE_BUNDLE_NAME;
		initialize();
	}

	public MessageEditorImpl(Locale locale) {
		if (locale == null) {
			this.locale = Locale.getDefault();
		} else {
			this.locale = locale;
		}

		this.messageBundleName = DEFAULT_MESSAGE_BUNDLE_NAME;

		initialize();
	}

	public MessageEditorImpl(Locale locale, String messageBundleName) {
		if (locale == null) {
			this.locale = Locale.getDefault();
		} else {
			this.locale = locale;
		}
		if (messageBundleName == null)
			throw new NullPointerException("Resource bundle file name is null.");
		if (messageBundleName.length() == 0)
			throw new RuntimeException("Resource bundle file name is empty.");

		this.messageBundleName = messageBundleName;

		initialize();
	}

	public MessageEditorImpl(String messageBundleName) {

		this.locale = Locale.getDefault();

		if (messageBundleName == null)
			throw new NullPointerException("Resource bundle file name is null.");
		if (messageBundleName.length() == 0)
			throw new RuntimeException("Resource bundle file name is empty.");

		this.messageBundleName = messageBundleName;

		initialize();
	}

	private void initialize() {

		BundleLoader bl = new BundleLoaderImpl();
		this.messageBundle = bl.load(getMessageBundleName(), getLocale());

	}

	private String messageBind(String message,
			ConstraintDescriptor constraintDescriptor, Object obj) {

		return replace(message, constraintDescriptor.getParameters(), obj);
	}

	private String replace(String message, Map<String, Object> parameters,
			Object obj) {
		Matcher matcher = messagePattern.matcher(message);
		StringBuffer sb = new StringBuffer();
		while (matcher.find()) {
			matcher.appendReplacement(sb, resolveParameter(matcher.group(1),
					parameters, obj));
		}
		matcher.appendTail(sb);
		return sb.toString();
	}

	private String resolveParameter(String token,
			Map<String, Object> parameters, Object obj) {

		if (checkParameterKey(token, obj)) {
			if (obj != null) {
				// #+index
				String string = getBindValue(token, obj);

				return replace(string, parameters, obj);

			}
		}

		Object variable = parameters.get(token);
		if (variable != null) {
			return replace(variable.toString(), parameters, obj);
		}

		StringBuffer buffer = new StringBuffer();
		String string = null;
		try {
			string = messageBundle != null ? messageBundle.getString(token)
					: null;
		} catch (MissingResourceException e) {
		}

		if (string != null) {

			buffer.append(replace(string, parameters, obj));
		} else {

			buffer.append("{" + token + "}");

		}

		return buffer.toString();
	}

	private int getBindIndex(String key) {

		if (BIND_SYMBOL.equals(key)) {
			return 0;
		} else {
			return Integer.parseInt(key.substring(1));
		}

	}

	private String getBindValue(String key, Object obj) {

		if (obj.getClass().isArray()) {
			int index = getBindIndex(key);
			Object value = Array.get(obj, index);
			if (value != null) {
				return value.toString();
			} else {
				return null;
			}
		} else {

			return obj.toString();

		}
	}

	private boolean checkParameterKey(String key, Object obj) {

		if (obj == null)
			return false;

		if (!key.startsWith(BIND_SYMBOL))
			return false;

		if (key.length() > 1) {
			String index = key.substring(1);
			int i;
			try {
				i = Integer.parseInt(index);
			} catch (NumberFormatException e) {
				return false;
			}

			if (i == 0) {
				return true;
			} else {
				if (obj.getClass().isArray()) {
					if (Array.getLength(obj) > i) {
						return true;
					} else {
						return false;
					}
				} else {
					return false;
				}
			}

		} else {
			return true;
		}
	}

	public Locale getLocale() {
		return locale;
	}

	public String getMessageBundleName() {
		return messageBundleName;
	}

	interface BundleLoader {

		ResourceBundle load(String bundleName, Locale locale);
	}

	static class BundleLoaderImpl implements BundleLoader {

		public ResourceBundle load(String bundleName, Locale locale) {
			ResourceBundle rb = null;
			ClassLoader classLoader = Thread.currentThread()
					.getContextClassLoader();

			if (classLoader != null) {
				rb = loadBundle(classLoader, bundleName, locale);
				if (rb != null) {
					return rb;
				}
			}

			classLoader = this.getClass().getClassLoader();

			if (classLoader != null) {
				rb = loadBundle(classLoader, bundleName, locale);
				if (rb != null) {
					return rb;
				}
			}

			throw new MissingResourceException("Not found resource file  ["
					+ bundleName + ".properties].", bundleName, "");

		}

		private ResourceBundle loadBundle(ClassLoader classLoader,
				String bundleName, Locale locale) {
			ResourceBundle rb = null;
			try {
				rb = ResourceBundle.getBundle(bundleName, locale, classLoader);
			} catch (MissingResourceException ignore) {
			}

			return rb;
		}

	}

}
