/*
 * Copyright 2009- kensir0u.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package net.sf.thirdi.validation.core.meta;

import java.lang.annotation.Annotation;
import java.lang.reflect.AccessibleObject;
import java.lang.reflect.Array;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

import net.sf.thirdi.validation.config.ConstraintID;
import net.sf.thirdi.validation.config.UsingJPAColumn;
import net.sf.thirdi.validation.core.meta.util.IndexComparator;
import net.sf.thirdi.validation.core.meta.util.ValidationUtils;
import net.sf.thirdi.validation.spec.ConstraintValidator;
import net.sf.thirdi.validation.spec.Valid;
import net.sf.thirdi.validation.spec.ValidationException;
import net.sf.thirdi.validation.util.ReflectionUtils;

public class ConstraintDescBuilder {

	private static ConstraintDescBuilder cdf = new ConstraintDescBuilder();

	private static boolean isJPA_Column = ValidationUtils
			.isPresent("javax.persistence.Column");

	private static Map<String, ConstraintClassDesc> pool = new ConcurrentHashMap<String, ConstraintClassDesc>();

	public static ConstraintClassDesc getDesc(Class<?> c) {
		if (cdf == null)
			cdf = new ConstraintDescBuilder();
		return cdf.getConstraintDesc(c);
	}

	private ConstraintClassDesc getConstraintDesc(Class<?> c) {

		if (c == null)
			throw new NullPointerException("class parameter is null.");

		if (!pool.containsKey(c.getName())) {
			ConstraintClassDesc ccd = createConstraintClassDesc(c);
			pool.put(c.getName(), ccd);
		}

		return pool.get(c.getName());
	}

	private ConstraintClassDesc createConstraintClassDesc(Class<?> c) {

		ConstraintClassDesc cc = new ConstraintClassDesc();
		cc.setAttribute(processClassAttributeList(c));
		cc.setField(processConstraintFieldDescList(c));
		cc.setProperty(processConstraintPropertyDescList(c));
		cc.setName(c.getName());
		cc.setConstraintClass(c);
		cc.setId(c.getName());
		cc.setGroupseaquence(getGroupseaquence(cc.getAttribute()));

		if (c.getSuperclass() != null) {
			Class<?> sc = c.getSuperclass();
			if (!sc.isAssignableFrom(Object.class)) {
				cc.setSuperclass(getConstraintDesc(c.getSuperclass()));
			}

		}

		Class<?>[] interfaces = c.getInterfaces();
		if (interfaces != null && interfaces.length != 0) {
			List<ConstraintClassDesc> list = new ArrayList<ConstraintClassDesc>();
			for (int i = 0; i < interfaces.length; i++) {
				Class<?> iface = interfaces[i];
				ConstraintClassDesc ccdi = getConstraintDesc(iface);
				if (ccdi != null) {
					list.add(ccdi);
				}
			}
			cc.setInterfaces(list);
		}

		return cc;

	}

	private Map<String, Set<String>> getGroupseaquence(
			List<ConstraintAttributeDesc> attribute) {

		Map<String, Set<String>> m = new HashMap<String, Set<String>>();

		for (ConstraintAttributeDesc cad : attribute) {
			if (cad.isGroups()) {
				m.put(cad.getName(), cad.getGroups());
			}
		}

		return m;
	}

	private List<ConstraintAttributeDesc> processClassAttributeList(Class<?> c) {

		Annotation[] classAno = c.getDeclaredAnnotations();
		List<ConstraintAttributeDesc> list = new ArrayList<ConstraintAttributeDesc>();

		for (int i = 0; i < classAno.length; i++) {
			Annotation annotation = classAno[i];
			if (isUsingJPAColumn(annotation)) {
				ConstraintAttributeDesc cad = processUsingJPAColumn((UsingJPAColumn) annotation);
				list.add(cad);
			}
		}

		return list;
	}

	private ConstraintAttributeDesc processUsingJPAColumn(
			UsingJPAColumn annotation) {

		ConstraintAttributeDesc cad = new ConstraintAttributeDesc();
		cad.setAnnotation(annotation);
		cad.setUsingJPAColumn(true);
		// cad.setGroups(true);
		// cad.setName(annotation.name());
		// cad.setGroups(createGroupSet(annotation.sequence()));

		return cad;

	}

	private Set<String> createGroupSet(String[] ary) {
		Set<String> set = new LinkedHashSet<String>();
		if (ary == null || ary.length == 0) {
			set.add("default");
		} else {
			for (int i = 0; i < ary.length; i++) {
				String string = ary[i];
				set.add(string);
			}
		}
		return set;
	}

	private boolean isUsingJPAColumn(Annotation annotation) {
		if (annotation instanceof UsingJPAColumn) {
			return true;
		} else {
			return false;
		}
	}

	private void processAttributeList(AccessibleObject ao,
			AbstractConstraintMemberDesc cfd) {

		List<ConstraintAttributeDesc> list = new ArrayList<ConstraintAttributeDesc>();

		for (Annotation annotation : ao.getAnnotations()) {
			ConstraintAttributeDesc cad = processAttribute(annotation, cfd);
			if (cad != null) {
				list.add(cad);
			}
			Set<Annotation> s = getInnerList(annotation);
			if (s != null) {
				for (Annotation ano:s){
					ConstraintAttributeDesc cadoverride = processAttribute(ano, cfd);
					if (cadoverride != null) {
						list.add(cadoverride);
					}
				}
			}
			for (Annotation ano : annotation.annotationType().getAnnotations()) {
				ConstraintAttributeDesc cadoverride = processAttribute(ano, cfd);
				if (cadoverride != null) {
					list.add(cadoverride);
				}
				Set<Annotation> soverride = getInnerList(ano);
				if (soverride != null) {
					for (Annotation innerano:soverride){
						ConstraintAttributeDesc cadoverridelist = processAttribute(innerano, cfd);
						if (cadoverridelist != null) {
							list.add(cadoverridelist);
						}
					}
				}
			}
		}

		IndexComparator.sort(list);

		cfd.setAttribute(list);
	}

	private <A extends Annotation> ConstraintAttributeDesc processAttribute(
			A annotation, AbstractConstraintMemberDesc cfd) {
		
		if (ValidationUtils.isConstraintAnnotation(annotation)) {
			ConstraintValidator vcAnno = annotation.annotationType()
					.getAnnotation(ConstraintValidator.class);
			ConstraintAttributeDesc cad = new ConstraintAttributeDesc();

			cad.setAnnotation(annotation);

			cad.setConstraintClass(vcAnno.value());
			// index
			cad.setIndex(getIndex(annotation));

			// groups
			cad.setGroups(getGroups(annotation));

			// parameter include message
			cad.setParameters(getParameter(annotation, cfd));

			// bean value parameter
			cad.setBeanValueParameter(ValidationUtils
					.isBeanValueParameterAnnotation(annotation));

			return cad;
		} else {

			if (isJPA_Column) {
				//TODO
				return null;
			} else {
				return null;
			}

		}
	}
	private Set<Annotation> getInnerList(Annotation annotation) {
		
		Object o = getAnnotationValue(annotation, "value");
		if (o != null) {
			if (o.getClass().isArray()) {
				if (Array.getLength(o) > 0) {
					
					Object ano = Array.get(o, 0);
					if (ano == null) {
						return null;
					} else {
						if (Annotation.class.isAssignableFrom(ano.getClass())){
							Set<Annotation> set = new LinkedHashSet<Annotation>();
							set.add((Annotation)ano);
							for (int i = 1;i < Array.getLength(o);i++){
								set.add((Annotation)Array.get(o, i));
							}
							return set;
						} else {
							return null;
						}
					}
				} else {
					return null;
				}
			} else {
				return null;
			}
		}else {
			return null;
		}
	}
	

	private int getIndex(Annotation annotation) {
		Object o = getAnnotationValue(annotation, "index");
		if (o != null) {
			if (!o.getClass().isArray()) {
				if (o.getClass().isAssignableFrom(Integer.class)) {
					return (Integer) o;
				} else {
					try {
						int index = Integer.parseInt(o.toString());
						return index;
					} catch (Exception e) {
					}
					return -1;
				}
			} else {
				return -1;
			}
		} else {
			return -1;
		}
	}

	private Map<String, Object> getParameter(Annotation annotation,
			AbstractConstraintMemberDesc cfd) {

		Map<String, Object> param = new HashMap<String, Object>();

		Method[] declaredMethods = annotation.annotationType()
				.getDeclaredMethods();

		for (Method m : declaredMethods) {
			try {

				if ("name".equals(m.getName())) {

					Object o = m.invoke(annotation);

					if (o == null || "".equals(o.toString())) {
						param.put(m.getName(), cfd.getName());
					} else {
						param.put(m.getName(), o);
					}

				} else {
					param.put(m.getName(), m.invoke(annotation));
				}

			} catch (Exception e) {
				throw new ValidationException(
						"Unable to read annotation parameters: "
								+ annotation.getClass(), e);
			}
		}

		return param;

	}

	private Set<String> getGroups(Annotation annotation) {
		String[] ary = null;
		Object o = getAnnotationValue(annotation, "groups");
		if (o != null) {
			if (o.getClass().isArray()) {
				if (o.getClass().isAssignableFrom(String[].class)) {
					ary = (String[]) o;
				}
			}
		}
		return createGroupSet(ary);
	}

	private ConstraintFieldDesc processConstraintFieldDesc(Field field) {

		ConstraintFieldDesc cfd = new ConstraintFieldDesc();
		cfd.setField(field);
		cfd.setFieldName(field.getName());
		cfd.setId(field.getDeclaringClass().getName() + "#" + field.getName());
		cfd.setValid(isValid(field));
		cfd.setGraph(isGraph(field));
		processConstraintID(field, cfd);
		cfd.setAttType(field.getType());
		processAttributeList(field, cfd);

		return cfd;
	}

	private boolean isGraph(Field f) {
		return ReflectionUtils.isGraph(f.getType());
	}

	private boolean isGraph(Method m) {
		return ReflectionUtils.isGraph(m.getReturnType());
	}

	private void processConstraintID(AccessibleObject ao,
			AbstractConstraintMemberDesc cfd) {
		if (ao == null)
			return;
		ConstraintID constraintid = ao.getAnnotation(ConstraintID.class);
		if (constraintid == null)
			return;
		cfd.setIndex(constraintid.index());
		cfd.setName(constraintid.name());
	}

	private boolean isValid(AccessibleObject ao) {
		if (ao == null)
			return false;
		Valid valid = ao.getAnnotation(Valid.class);

		if (valid == null) {
			return false;
		} else {
			return true;
		}
	}

	private List<ConstraintFieldDesc> processConstraintFieldDescList(Class<?> c) {

		List<ConstraintFieldDesc> list = new ArrayList<ConstraintFieldDesc>();

		Field[] field = c.getDeclaredFields();
		for (int i = 0; i < field.length; i++) {
			Field field2 = field[i];

			ConstraintFieldDesc cfd = processConstraintFieldDesc(field2);
			list.add(cfd);

		}

		IndexComparator.sort(list);

		return list;
	}

	private ConstraintPropertyDesc processConstraintPropertyDesc(Method m) {
		ConstraintPropertyDesc cpd = new ConstraintPropertyDesc();
		cpd.setPropertyName(m.getName());
		cpd.setMethod(m);
		cpd.setId(m.getDeclaringClass().getName() + "#" + m.getName());
		cpd.setValid(isValid(m));
		cpd.setGraph(isGraph(m));
		cpd.setAttType(m.getReturnType());
		processConstraintID(m, cpd);

		processAttributeList(m, cpd);

		return cpd;
	}

	private List<ConstraintPropertyDesc> processConstraintPropertyDescList(
			Class<?> c) {
		List<ConstraintPropertyDesc> list = new ArrayList<ConstraintPropertyDesc>();

		Method[] method = c.getDeclaredMethods();
		for (int i = 0; i < method.length; i++) {
			Method method2 = method[i];

			ConstraintPropertyDesc cpd = processConstraintPropertyDesc(method2);
			list.add(cpd);

		}

		IndexComparator.sort(list);

		return list;
	}

	private Object getAnnotationValue(Annotation annotation, String name) {
		Method valueMethod = null;
		try {
			valueMethod = annotation.annotationType().getDeclaredMethod(name);
		} catch (NoSuchMethodException ignore) {
		}

		if (valueMethod != null) {
			try {
				return valueMethod.invoke(annotation);
			} catch (Exception ignore) {
			}
		}
		return null;
	}

}
