/*
 * Copyright 2009- kensir0u.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package net.sf.thirdi.validation.core;

import java.lang.annotation.Annotation;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import net.sf.thirdi.validation.core.meta.AbstractConstraintMemberDesc;
import net.sf.thirdi.validation.core.meta.ConstraintAttributeDesc;
import net.sf.thirdi.validation.core.meta.ConstraintClassDesc;
import net.sf.thirdi.validation.core.meta.ConstraintDescBuilder;
import net.sf.thirdi.validation.core.meta.ConstraintFieldDesc;
import net.sf.thirdi.validation.core.meta.ConstraintPropertyDesc;
import net.sf.thirdi.validation.core.meta.ConstraintType;
import net.sf.thirdi.validation.core.meta.util.ValidationUtils;
import net.sf.thirdi.validation.spec.Constraint;
import net.sf.thirdi.validation.spec.ConstraintContext;
import net.sf.thirdi.validation.spec.ConstraintFactory;
import net.sf.thirdi.validation.spec.ConstraintViolation;
import net.sf.thirdi.validation.spec.MessageEditor;
import net.sf.thirdi.validation.spec.Validator;
import net.sf.thirdi.validation.util.ReflectionUtils;

public class TIValidator implements Validator {

	private static final long serialVersionUID = 1L;

	private MessageEditor messageresolver;

	private ConstraintFactory constraintfactory;

	@SuppressWarnings("unused")
	private TIValidator() {
	}

	TIValidator(MessageEditor messageresolver,
			ConstraintFactory constraintfactory) {
		this.messageresolver = messageresolver;
		this.constraintfactory = constraintfactory;

	}

	public <T> Set<ConstraintViolation<T>> validate(T arg0, String... groups) {

		if (groups == null || groups.length == 0) {
			return valid(arg0, "default");
		} else {
			return valid(arg0, groups);
		}
	}

	private <T> Set<ConstraintViolation<T>> valid(T bean, String... groups) {

		if (bean == null)
			throw new IllegalArgumentException("bean object is null.");

		Set<ConstraintViolation<T>> set = new LinkedHashSet<ConstraintViolation<T>>();

		ConstraintClassDesc ccd = ConstraintDescBuilder
				.getDesc(bean.getClass());

		List<ConstraintFieldDesc> fieldlist = ccd.getField();

		for (int i = 0; i < fieldlist.size(); i++) {
			ConstraintFieldDesc cfd = fieldlist.get(i);
			vaild(bean, cfd, ccd, set, groups);

			if (cfd.isValid()) {
				Object innerbean = ValidationUtils.getFieldValue(cfd.getField(),bean);
				if (innerbean != null) {
					innervalid(bean,innerbean, set, cfd.getAttType(), groups);
				}
			}
		}

		List<ConstraintPropertyDesc> propertylist = ccd.getProperty();

		for (int i = 0; i < propertylist.size(); i++) {
			ConstraintPropertyDesc cpd = propertylist.get(i);

			vaild(bean, cpd, ccd, set, groups);

			if (cpd.isValid()) {
				Object innerbean = ValidationUtils.getMethodValue(cpd.getMethod(),bean);
				if (innerbean != null) {
					innervalid(bean,innerbean, set, cpd.getAttType(), groups);
				}
			}
		}

		if (ccd.getSuperclass() != null) {
			set = travarseVaild(bean, set, ccd.getSuperclass(), groups);
		}

		List<ConstraintClassDesc> l = ccd.getInterfaces();
		if (l != null && l.size() != 0) {
			for (int i = 0; i < l.size(); i++) {
				set = travarseVaild(bean, set, l.get(i), groups);
			}
		}

		return set;

	}

	private <T> Set<ConstraintViolation<T>> innervalid(T beans,
			Object innerbean, Set<ConstraintViolation<T>> set, Class<?> type,
			String... groups) {

		if (innerbean == null)
			throw new IllegalArgumentException("bean object is null.");

		ConstraintClassDesc ccd = ConstraintDescBuilder.getDesc(type);

		List<ConstraintFieldDesc> list = ccd.getField();

		for (int i = 0; i < list.size(); i++) {
			ConstraintFieldDesc cfd = list.get(i);

			innervaild(beans, innerbean, cfd, ccd, set, groups);
		}

		List<ConstraintPropertyDesc> list2 = ccd.getProperty();

		for (int i = 0; i < list2.size(); i++) {
			ConstraintPropertyDesc cpd = list2.get(i);

			innervaild(beans, innerbean, cpd, ccd, set, groups);
		}

		if (ccd.getSuperclass() != null) {
			set = innerTravarseVaild(beans, innerbean, set,
					ccd.getSuperclass(), groups);
		}

		List<ConstraintClassDesc> l = ccd.getInterfaces();
		if (l != null && l.size() != 0) {
			for (int i = 0; i < l.size(); i++) {
				set = innerTravarseVaild(beans, innerbean, set, l.get(i),
						groups);
			}
		}

		return set;

	}

	private <T> Set<ConstraintViolation<T>> innerTravarseVaild(T beans,
			Object bean, Set<ConstraintViolation<T>> set,
			ConstraintClassDesc c, String... groups) {

		if (set == null)
			set = new LinkedHashSet<ConstraintViolation<T>>();

		ConstraintClassDesc ccd = ConstraintDescBuilder.getDesc(c
				.getConstraintClass());

		List<ConstraintFieldDesc> list = ccd.getField();

		for (int i = 0; i < list.size(); i++) {
			ConstraintFieldDesc cfd = list.get(i);

			innervaild(beans, bean, cfd, ccd, set, groups);

		}

		List<ConstraintPropertyDesc> list2 = ccd.getProperty();

		for (int i = 0; i < list2.size(); i++) {
			ConstraintPropertyDesc cpd = list2.get(i);

			innervaild(beans, bean, cpd, ccd, set, groups);
		}

		if (ccd.getSuperclass() != null) {
			set = innerTravarseVaild(beans, bean, set, ccd.getSuperclass(),
					groups);
		}

		List<ConstraintClassDesc> l = ccd.getInterfaces();
		if (l != null && l.size() != 0) {
			for (int i = 0; i < l.size(); i++) {
				set = innerTravarseVaild(beans, bean, set, l.get(i), groups);
			}
		}

		return set;

	}

	private <T> Set<ConstraintViolation<T>> travarseVaild(T bean,
			Set<ConstraintViolation<T>> set, ConstraintClassDesc c,
			String... groups) {

		if (set == null)
			set = new LinkedHashSet<ConstraintViolation<T>>();

		ConstraintClassDesc ccd = ConstraintDescBuilder.getDesc(c
				.getConstraintClass());

		List<ConstraintFieldDesc> list = ccd.getField();

		for (int i = 0; i < list.size(); i++) {
			ConstraintFieldDesc cfd = list.get(i);
			vaild(bean, cfd, ccd, set, groups);

			if (cfd.isValid()) {
				Object innerbean = ValidationUtils.getFieldValue(cfd.getField(),bean);
				if (innerbean != null) {
					innervalid(bean,innerbean, set, cfd.getAttType(), groups);
				}
			}
		}

		List<ConstraintPropertyDesc> list2 = ccd.getProperty();

		for (int i = 0; i < list2.size(); i++) {
			ConstraintPropertyDesc cpd = list2.get(i);

			vaild(bean, cpd, ccd, set, groups);

			if (cpd.isValid()) {
				Object innerbean = ValidationUtils.getMethodValue(cpd.getMethod(),bean);
				if (innerbean != null) {
					innervalid(bean,innerbean, set, cpd.getAttType(), groups);
				}
			}
		}

		if (ccd.getSuperclass() != null) {
			set = travarseVaild(bean, set, ccd.getSuperclass(), groups);
		}

		List<ConstraintClassDesc> l = ccd.getInterfaces();
		if (l != null && l.size() != 0) {
			for (int i = 0; i < l.size(); i++) {
				set = travarseVaild(bean, set, l.get(i), groups);
			}
		}

		return set;

	}

	private <T> ConstraintViolation<T> vaild(T bean,
			AbstractConstraintMemberDesc cfd, ConstraintClassDesc ccd,
			Set<ConstraintViolation<T>> set, String... groups) {

		List<ConstraintAttributeDesc> list = cfd.getAttribute();
		Map<String, String> m = createGroupMap(groups);
		boolean isGraph = cfd.isGraph();
		for (int i = 0; i < list.size(); i++) {
			ConstraintAttributeDesc cad = list.get(i);

			if (!isInGroup(cad.getGroups(), m))
				continue;

			Constraint<?> cont = getConstraint(cad.getAnnotation(), cad);
			Object o = null;

			if (cfd.getType() == ConstraintType.FIELD) {
				o = ValidationUtils.getFieldValue(((ConstraintFieldDesc) cfd)
						.getField(), bean);
			} else {
				o = ValidationUtils.getMethodValue(
						((ConstraintPropertyDesc) cfd).getMethod(), bean);
			}

			// bean value
			if (cad.isBeanValueParameter()) {
				o = bean;
			}

			ConstraintContext context = getConstraintContext(cad, bean);

			if (isGraph) {
				Iterator<?> ite = createIterator(cfd.getAttType(), o);
				for (int j = 0; ite.hasNext(); j++) {
					Object io = ite.next();

					if (io instanceof Map.Entry) {
						io = ((Map.Entry<?, ?>) io).getValue();
					}

					if (!cont.isValid(io, context)) {

						String message = messageresolver.bind(context
								.getDefaultErrorMessage(), cad, io);

						TIConstraintViolation<T> failingConstraintViolation = new TIConstraintViolation<T>(
								message, bean, io, ccd.getConstraintClass(),
								cfd.getId() + "{" + j + "}", io, cad
										.getGroups(), cad);

						set.add(failingConstraintViolation);
						return null;

					}

				}

			} else {

				if (!cont.isValid(o, context)) {

					String message = messageresolver.bind(context
							.getDefaultErrorMessage(), cad, o);

					TIConstraintViolation<T> failingConstraintViolation = new TIConstraintViolation<T>(
							message, bean, o, ccd.getConstraintClass(),

							cfd.getId(), o, cad.getGroups(), cad);

					set.add(failingConstraintViolation);
					return null;

				}
			}

		}

		return null;
	}

	private <T> ConstraintViolation<T> innervaild(T beans, Object bean,
			AbstractConstraintMemberDesc cfd, ConstraintClassDesc ccd,
			Set<ConstraintViolation<T>> set, String... groups) {

		List<ConstraintAttributeDesc> list = cfd.getAttribute();
		Map<String, String> m = createGroupMap(groups);
		boolean isGraph = cfd.isGraph();
		for (int i = 0; i < list.size(); i++) {
			ConstraintAttributeDesc cad = list.get(i);

			if (!isInGroup(cad.getGroups(), m))
				continue;

			Constraint<?> cont = getConstraint(cad.getAnnotation(), cad);
			Object o = null;

			if (cfd.getType() == ConstraintType.FIELD) {
				o = ValidationUtils.getFieldValue(((ConstraintFieldDesc) cfd)
						.getField(), bean);
			} else {
				o = ValidationUtils.getMethodValue(
						((ConstraintPropertyDesc) cfd).getMethod(), bean);
			}

			// bean value
			if (cad.isBeanValueParameter()) {
				o = bean;
			}

			ConstraintContext context = getConstraintContext(cad, bean);

			if (isGraph) {
				Iterator<?> ite = createIterator(cfd.getAttType(), o);
				for (int j = 0; ite.hasNext(); j++) {
					Object io = ite.next();

					if (io instanceof Map.Entry) {
						io = ((Map.Entry<?, ?>) io).getValue();
					}

					if (!cont.isValid(io, context)) {

						String message = messageresolver.bind(context
								.getDefaultErrorMessage(), cad, io);

						TIConstraintViolation<T> failingConstraintViolation = new TIConstraintViolation<T>(
								message, beans, bean, ccd.getConstraintClass(),
								cfd.getId() + "{" + j + "}", io, cad
										.getGroups(), cad);

						set.add(failingConstraintViolation);
						return null;

					}

				}

			} else {

				if (!cont.isValid(o, context)) {

					String message = messageresolver.bind(context
							.getDefaultErrorMessage(), cad, o);

					TIConstraintViolation<T> failingConstraintViolation = new TIConstraintViolation<T>(
							message, beans, bean, ccd.getConstraintClass(), cfd
									.getId(), o, cad.getGroups(), cad);

					set.add(failingConstraintViolation);
					return null;

				}
			}

		}

		return null;
	}

	private <T> Iterator<?> createIterator(Class<?> type, Object value) {
		Iterator<?> iter;
		if (ReflectionUtils.isCollectionClass(type)) {
			if (value instanceof Iterable) {
				Iterable<?> elements = (Iterable<?>) value;
				iter = elements.iterator();
			} else {
				Map<?, ?> map = (Map<?, ?>) value;

				iter = map.entrySet().iterator();
			}
		} else if (ReflectionUtils.isArray(type)) {
			List<?> arrayList = Arrays.asList(value);
			iter = arrayList.iterator();
		} else {
			List<Object> list = new ArrayList<Object>();
			list.add(value);
			iter = list.iterator();
		}
		return iter;
	}

	private boolean isInGroup(Set<String> s, Map<String, String> m) {
		for (Iterator<String> iterator = s.iterator(); iterator.hasNext();) {
			String string = iterator.next();
			if (m.containsKey(string))
				return true;
		}
		return false;
	}

	private Map<String, String> createGroupMap(String... groups) {
		Map<String, String> m = new HashMap<String, String>();
		if (groups == null || groups.length == 0) {
			return m;
		} else {
			for (String group : groups) {
				m.put(group, group);
			}
		}
		return m;
	}

	@SuppressWarnings("unchecked")
	private <A extends Annotation> Constraint getConstraint(A annotation,
			ConstraintAttributeDesc cad) {

		Constraint<A> constraint = null;
		try {
			// unchecked
			constraint = (Constraint<A>) constraintfactory.getInstance(cad
					.getConstraintClass());
		} catch (RuntimeException e) {
			throw e;
		}

		try {
			constraint.initialize(annotation);
		} catch (RuntimeException e) {
		}

		return constraint;
	}

	private ConstraintContext getConstraintContext(ConstraintAttributeDesc cad,
			Object bean) {

		return new TIConstraintContext((String) cad.getParameters().get(
				"message"), cad.getConstraintClass(), bean);

	}
}
