/** @file
 * c2xml.cpp
 *
 * Generate TestLink testcase XML from C/C++ source codes.
 * C/C++̃\[XR[hTestLink̃eXgP[X쐬B
 *
 * @version 0.1
 * @date 2007/10/01
 *
 * @author Toshiyuki Kawanishi
 */

#include<iostream>
#include<fstream>
#include<string>
#include<vector>

//----------------------------------------------------------------------------------------------------
// Classes and structures definition    NXƍ\̂̒`
//----------------------------------------------------------------------------------------------------

/**
 * TestLink Tag
 * eXgN^O̍\
 *
 * @version 0.1
 * @date 2007/10/01
 *
 * @author Toshiyuki Kawanishi
 */
typedef struct {
  std::string testcaseName;
  std::string summary;
  std::string steps;
  std::string expectedResults;
  std::string keyword;
  std::string notes;
} Tag;

/**
 * Class of generate TestLink testcase XML from C/C++ sorce codes
 * C/C++̃\[XR[hTestLink̃eXgP[X쐬NX
 *
 * @version 0.1
 * @date 2007/10/01
 *
 * @author Toshiyuki Kawanishi
 */
class CToXml {
  public:
    /* constructor    RXgN^ */
    CToXml();
	  
    /* public methods    pubNb\h */
    void parseSorce(const std::string filename);
    void makeXml(void);
    
  private:
    /* private constants    vCx[g萔 */
    // XML Tag    XML^O
    static const std::string TL_XMLHEAD;
    static const std::string TL_TESTCASES;
    static const std::string TL_TESTCASENAME;
    static const std::string TL_SUMMARY;
    static const std::string TL_STEPS;
    static const std::string TL_EXPECTEDRESULTS;
    static const std::string TL_KEYWORDS;
    static const std::string TL_KEYWORD;
    static const std::string TL_NOTES;
    
    static const std::string TL_XMLHEADER;
    static const std::string TL_XMLFOOTER;
    
    // JavaDoc Tag    JavaDoc^O
    static const std::string TL_TESTCASENAME_TAG;
    static const std::string TL_SUMMARY_TAG;
    static const std::string TL_STEPS_TAG;
    static const std::string TL_EXPECTEDRESULTS_TAG;
    static const std::string TL_KEYWORD_TAG;
    static const std::string TL_NOTES_TAG;
    
    // Output file name    o̓t@C
    static const std::string XMLFILE;
    
    /* private variables    vCx[gϐ */
    std::vector<Tag> tagVector;
    std::string xml;
    
    /* private method    vCx[gb\h */
    void writeXml();
};


//----------------------------------------------------------------------------------------------------
// Static constants definition    ÓI萔̒`
//----------------------------------------------------------------------------------------------------

const std::string CToXml::TL_XMLHEAD("<?xml version=\"1.0\" encoding=\"UTF-8\"?>");
const std::string CToXml::TL_TESTCASES("testcases");
const std::string CToXml::TL_TESTCASENAME("testcase");
const std::string CToXml::TL_SUMMARY("summary");
const std::string CToXml::TL_STEPS("steps");
const std::string CToXml::TL_EXPECTEDRESULTS("expectedresults");
const std::string CToXml::TL_KEYWORDS("keywords");
const std::string CToXml::TL_KEYWORD("keyword");
const std::string CToXml::TL_NOTES("notes");

const std::string CToXml::TL_XMLHEADER(TL_XMLHEAD + "\n<" + TL_TESTCASES + ">");
const std::string CToXml::TL_XMLFOOTER("</" + TL_TESTCASES + ">");

const std::string CToXml::TL_TESTCASENAME_TAG("tltestcase");
const std::string CToXml::TL_SUMMARY_TAG("tlsummary");
const std::string CToXml::TL_STEPS_TAG("tlsteps");
const std::string CToXml::TL_EXPECTEDRESULTS_TAG("tlexpectedresults");
const std::string CToXml::TL_KEYWORD_TAG("tlkeyword");
const std::string CToXml::TL_NOTES_TAG("tlnotes");

const std::string CToXml::XMLFILE("testcases.xml");


//----------------------------------------------------------------------------------------------------
// Methods (member functions) definition    b\h̒`
//----------------------------------------------------------------------------------------------------

/**
 * constructor of Class CToXml
 * CToXml̃RXgN^
 *
 * @param none
 * @return none
 *
 * @version 0.1
 * @date 2007/10/01
 *
 * @author Toshiyuki Kawanishi
 */
CToXml::CToXml() {
  xml = "";
}

/**
 * Parsing source C/C++ source codes and obtain JavaDoc tags for TestLink.
 * C/C++̃\[XR[h͂TestLinkpJavaDoc^O擾B
 *
 * @param none
 * @return none
 *
 * @version 0.1
 * @date 2007/10/01
 *
 * @author Toshiyuki Kawanishi
 */
void
CToXml::parseSorce(const std::string filename) {
  std::ifstream inputFile(filename.c_str());
  std::string sourcecode;
  
  int tagStartPosition;
  int tagEndPosition;
  
  int nextSpacePosition;
  int nextTabPosition;
  
  std::string tagName;
  
  bool testcaseNameFlag = false;
  bool summaryFlag = false;
  bool stepsFlag = false;
  bool exepectedResultsFlag = false;
  bool keywordFlag = false;
  bool notesFlag = false;
  
  Tag tags;
  
  while (inputFile && getline(inputFile, sourcecode)) {
    std::cout << sourcecode << std::endl;
    
    tagStartPosition = sourcecode.find("@", 0);
    if (tagStartPosition != std::string::npos) {
      nextSpacePosition = sourcecode.find_first_of(" ", tagStartPosition);
      nextTabPosition = sourcecode.find_first_of("\t", tagStartPosition);
      
      if ((nextSpacePosition != std::string::npos) &&
          (nextTabPosition == std::string::npos)) {
        tagEndPosition = nextSpacePosition;
      }
      else if ((nextSpacePosition == std::string::npos) &&
               (nextTabPosition != std::string::npos)) {
        tagEndPosition = nextTabPosition;
      }
      else if (nextSpacePosition < nextTabPosition) {
        tagEndPosition = nextSpacePosition;
      }
      else {
        tagEndPosition = nextTabPosition;
      }
      
      tagName = sourcecode.substr(tagStartPosition + 1, tagEndPosition - tagStartPosition - 1);
      
      if (tagName == TL_TESTCASENAME_TAG) {
        tags.testcaseName = sourcecode.substr(tagEndPosition + 1, sourcecode.length() - tagEndPosition);
        testcaseNameFlag = true;
      }
      else if (tagName == TL_SUMMARY_TAG) {
        tags.summary = sourcecode.substr(tagEndPosition + 1, sourcecode.length() - tagEndPosition);
        summaryFlag = true;
      }
      else if (tagName == TL_STEPS_TAG) {
        tags.steps = sourcecode.substr(tagEndPosition + 1, sourcecode.length() - tagEndPosition);
        stepsFlag = true;
      }
      else if (tagName == TL_EXPECTEDRESULTS_TAG) {
        tags.expectedResults = sourcecode.substr(tagEndPosition + 1, sourcecode.length() - tagEndPosition);
        exepectedResultsFlag = true;
      }
      else if (tagName == TL_KEYWORD_TAG) {
        tags.keyword = sourcecode.substr(tagEndPosition + 1, sourcecode.length() - tagEndPosition);
        keywordFlag = true;
      }
      else if (tagName == TL_NOTES_TAG) {
        tags.notes = sourcecode.substr(tagEndPosition + 1, sourcecode.length() - tagEndPosition);
        notesFlag = true;
      }
    }
    
    if (testcaseNameFlag && summaryFlag && stepsFlag && exepectedResultsFlag && keywordFlag && notesFlag) {
      tagVector.push_back(tags);
      
      testcaseNameFlag = false;
      summaryFlag = false;
      stepsFlag = false;
      exepectedResultsFlag = false;
      keywordFlag = false;
      notesFlag = false;
    }
  }
  
  inputFile.close();
  
  return;
}

/**
 * Configure XML file to TestLink format.
 * TestLink`XMLt@C𐮌`B
 *
 * @param none
 * @return none
 *
 * @version 0.1
 * @date 2007/10/01
 *
 * @author Toshiyuki Kawanishi
 */
void
CToXml::makeXml(void) {
  std::vector<Tag>::iterator currentTag;
  
  xml = TL_XMLHEADER + "\n";
  
  for (currentTag = tagVector.begin(); currentTag != tagVector.end(); currentTag++) {
    xml += + "<" + TL_TESTCASENAME + " name=\"" + currentTag->testcaseName + "\">"
        + "<" + TL_SUMMARY + "><![CDATA[<p>" + currentTag->summary + "</p>]]></" + TL_SUMMARY + ">"
        + "<" + TL_STEPS + "><![CDATA[<p>" + currentTag->steps + "</p>]]></" + TL_STEPS + ">"
        + "<" + TL_EXPECTEDRESULTS + "><![CDATA[<p>" + currentTag->expectedResults + "</p>]]></" + TL_EXPECTEDRESULTS + ">"
        + "<" + TL_KEYWORDS + ">"
        + "<" + TL_KEYWORD + " name=\"" + currentTag->keyword + "\">"
        + "<" + TL_NOTES + "><![CDATA[<p>" + currentTag->notes + "</p>]]></" + TL_NOTES + ">"
        + "</" + TL_KEYWORD + ">"
        + "</" + TL_KEYWORDS + ">"
        + "</" + TL_TESTCASENAME + ">\n";
  }
  
  xml += TL_XMLFOOTER;
  
  writeXml();
  return;
}

/**
 * Write XML to "xxx.xml" file.
 * XMLt@CɏށB
 *
 * @param none
 * @return none
 *
 * @version 0.1
 * @date 2007/10/01
 *
 * @author Toshiyuki Kawanishi
 */
void
CToXml::writeXml(void) {
  std::ofstream outputFile(XMLFILE.c_str());
  outputFile << xml << std::endl;
  outputFile.close();
  
  return;
}

//----------------------------------------------------------------------------------------------------
// Main routine    C[`
//----------------------------------------------------------------------------------------------------

int main(int argc, std::string argv[]) {
  CToXml ctoxml;
  std::string filename;
  
  if (argc != 2) {
    std::cout << "c2xml: Error! too or few arguments." << std::endl;
    return -1;
  }
  else {
    filename = argv[1];
    ctoxml.parseSorce(filename);
    ctoxml.makeXml();
  }
  
  return 0;
}
