﻿using System;
using System.Data;
using System.Configuration;
using System.Collections;
using System.Web;
using System.Web.Security;
using System.Web.UI;
using System.Web.UI.WebControls;
using System.Web.UI.WebControls.WebParts;
using System.Web.UI.HtmlControls;
using TERASOLUNA.Fw.Web;
using TERASOLUNA.Fw.Web.Configuration.Sql;
using System.Data.SqlClient;
using TutorialServerWeb.Dto;
using TERASOLUNA.Fw.Common.BLogic;
using TERASOLUNA.Fw.Web.Session;

namespace TutorialServerWeb.Form
{
    /// <summary>
    /// 商品選択画面。
    /// </summary>
    public partial class SelectItemForm : System.Web.UI.Page
    {
        /// <summary>
        /// セッション管理クラス。
        /// </summary>
        private SessionManager _sessionManager = null;

        /// <summary>
        /// 初期化メソッドをオーバーライドし、SessionManagerのインスタンスを生成する。
        /// </summary>
        /// <param name="e">イベントデータを格納している<see cref="EventArgs"/>。</param>
        protected override void OnInit(EventArgs e)
        {
            base.OnInit(e);

            // セッション管理クラスの生成
            _sessionManager = new SessionManager(this.Session);
        }

        /// <summary>
        /// 初期化イベント。
        /// </summary>
        /// <param name="sender">イベントのソース。</param>
        /// <param name="e">イベントデータを格納している<see cref="EventArgs"/>。</param>
        protected void Page_Load(object sender, EventArgs e)
        {
            if (_sessionManager["ItemDataSet"] == null)
            {
                // セッションに商品情報格納用のデータセットを格納する
                ItemDataSet itemDataSet = new ItemDataSet();
                _sessionManager.Add("ItemDataSet", itemDataSet, 1);
            }

            if (!IsPostBack)
            {
                // カテゴリID"all"をもとに、ビジネスロジックを実行し、実行結果の商品情報をGridViewにバインドする。
                UpdateGridViewItem("all");

                // セッションに登録された商品に対して、チェックを付ける
                CheckedFromSession();
            }
        }

        /// <summary>
        /// 商品情報をセッションに格納し、商品確認画面に遷移する。
        /// </summary>
        /// <param name="sender">イベントのソース。</param>
        /// <param name="e">イベントデータを格納している<see cref="EventArgs"/>。</param>
        protected void ButtonConfirm_Click(object sender, EventArgs e)
        {
            // 選択された商品情報をセッションに格納する。
            SetItemToSession();

            // 何も商品が選択されなかった場合は、エラーメッセージを表示する
            ItemDataSet itemDataSet = (ItemDataSet)_sessionManager["ItemDataSet"];
            if (itemDataSet.Item.Count == 0)
            {
                LabelSelectError.Visible = true;
            }
            else
            {
                WebUtils.Transit("ConfirmItemForm");
            }
        }

        /// <summary>
        /// GridViewに存在する商品が既にセッションに存在する場合は、選択状態にする。
        /// </summary>
        private void CheckedFromSession()
        {
            // 選択された商品情報（商品IDおよび商品名）をデータセットに格納する
            foreach (GridViewRow row in GridViewItem.Rows)
            {
                // グリッドに設定されている商品IDを取得する
                string itemId = row.Cells[0].Text;

                // セッションから商品情報が格納されたデータセットを取得する
                ItemDataSet itemDataSet = (ItemDataSet)_sessionManager["ItemDataSet"];

                foreach (ItemDataSet.ItemRow itemRow in itemDataSet.Item.Rows)
                {
                    // GridViewの行に存在する商品IDがセッションに存在する場合は、選択状態にする
                    if (itemId.Equals(itemRow.ItemId))
                    {
                        // GridViewの行にあるチェックボックスコントロールを取得する
                        CheckBox checkBox = (CheckBox)row.FindControl("ItemCheck");
                        checkBox.Checked = true;
                    }
                }
            }
        }

        /// <summary>
        /// 選択されたItemIdをデータセットとして取得する。
        /// </summary>
        private void SetItemToSession()
        {
            // 選択された商品情報（商品IDおよび商品名）をデータセットに格納する
            foreach (GridViewRow row in GridViewItem.Rows)
            {
                CheckBox checkBox = (CheckBox)row.FindControl("ItemCheck");

                // グリッドに設定されている商品IDを取得する
                string itemId = row.Cells[0].Text;

                // グリッドに設定されている商品名を取得する
                string itemName = row.Cells[1].Text;

                if (checkBox.Checked)
                {
                    AddDataRowByItem(itemId, itemName);
                }
                else
                {
                    RemoveDataRowByItemId(itemId);
                }
            }
        }

        /// <summary>
        /// 引数で指定された商品IDと商品名をセッションに格納する。
        /// </summary>
        /// <param name="itemId">商品ID</param>
        /// <param name="itemName">商品名</param>
        private void AddDataRowByItem(string itemId, string itemName)
        {
            ItemDataSet itemDataSet = (ItemDataSet)_sessionManager["ItemDataSet"];

            foreach (ItemDataSet.ItemRow row in itemDataSet.Item.Rows)
            {
                if (itemId.Equals(row.ItemId))
                {
                    return;
                }
            }

            itemDataSet.Item.AddItemRow(itemId, itemName);
        }

        /// <summary>
        /// 引数で指定された商品IDをもとに、賞品情報をセッションから削除する。
        /// </summary>
        /// <param name="itemId">商品ID</param>
        private void RemoveDataRowByItemId(string itemId)
        {
            ItemDataSet itemDataSet = (ItemDataSet)_sessionManager["ItemDataSet"];

            foreach (ItemDataSet.ItemRow row in itemDataSet.Item.Rows)
            {
                if (itemId.Equals(row.ItemId))
                {
                    itemDataSet.Item.RemoveItemRow(row);
                    return;
                }
            }
        }

        /// <summary>
        /// エラーメッセージを非表示にする。
        /// </summary>
        /// <param name="sender">イベントのソース。</param>
        /// <param name="e">イベントデータを格納している<see cref="EventArgs"/>。</param>
        protected void DropDownListCategory_SelectedIndexChanged(object sender, EventArgs e)
        {
            LabelSelectError.Visible = false;
        }

        /// <summary>
        /// メニュー画面に遷移する。
        /// </summary>
        /// <param name="sender">イベントのソース。</param>
        /// <param name="e">イベントデータを格納している<see cref="EventArgs"/>。</param>
        protected void ButtonMenu_Click(object sender, EventArgs e)
        {
            WebUtils.Transit("MenuForm");
        }


        /// <summary>
        /// 画面で選択されたカテゴリに属する商品の情報を取得する。
        /// </summary>
        /// <param name="sender">イベントのソース。</param>
        /// <param name="e">イベントデータを格納している<see cref="EventArgs"/>。</param>
        protected void ButtonCategory_Click(object sender, EventArgs e)
        {
            // 商品情報をセッションに登録する
            SetItemToSession();

            // 画面で選択されたカテゴリIDをもとに、ビジネスロジックを実行し、実行結果の商品情報をGridViewにバインドする。
            UpdateGridViewItem(DropDownListCategory.SelectedValue);

            // セッションに登録された商品に対して、チェックを付ける
            CheckedFromSession();
        }

        /// <summary>
        /// 引数のカテゴリIDをもとにビジネスロジックを実行し、実行結果の商品情報をGridViewにバインドする。
        /// </summary>
        /// <param name="selectedCategoryId">カテゴリID</param>
        private void UpdateGridViewItem(string selectedCategoryId)
        {
            // ビジネスロジックの生成
            IBLogic blogic = BLogicFactory.CreateBLogic("selectItem");

            // ビジネスロジック入力データセットへのカテゴリIDの設定
            BLogicParam param = new BLogicParam();
            SelectItemParamDataSet paramDataSet = new SelectItemParamDataSet();
            paramDataSet.SelectItemParam.AddSelectItemParamRow(selectedCategoryId);

            param.ParamData = paramDataSet;

            // ビジネスロジックの実行
            BLogicResult result = blogic.Execute(param);

            if(BLogicResult.SUCCESS.Equals(result.ResultString)){
                ItemDataSet itemDataSet = (ItemDataSet)result.ResultData;

                // GridViewにデータベースから取得した商品情報をバインドする
                GridViewItem.DataSource = itemDataSet.Item;
                GridViewItem.DataBind();
            }
        }

    }
}
