// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Text;
using System.Collections.Generic;
using TERASOLUNA.Fw.Web.Controller;
using System.Web;
using TERASOLUNA.Fw.Common;
using TERASOLUNA.Fw.WebUT;
using TERASOLUNA.Fw.Utlib.Logging;
using TERASOLUNA.Fw.Common.Logging;
using System.Reflection;

namespace TERASOLUNA.Fw.WebUT.Controller
{
    /// <summary>
    ///TERASOLUNA.Fw.Web.Controller.RequestControllerFactory のテスト クラスです。すべての
    ///TERASOLUNA.Fw.Web.Controller.RequestControllerFactory 単体テストをここに含めます
    ///</summary>
 
    [TestClass()]
    public class RequestControllerFactoryTest
    {


        private TestContext testContextInstance;

        /// <summary>
        ///現在のテストの実行についての情報および機能を
        ///提供するテスト コンテキストを取得または設定します。
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }
        #region 追加のテスト属性
        // 
        //テストを作成するときに、次の追加属性を使用することができます:
        //
        //クラスの最初のテストを実行する前にコードを実行するには、ClassInitialize を使用
        //
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //クラスのすべてのテストを実行した後にコードを実行するには、ClassCleanup を使用
        //
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //各テストを実行する前にコードを実行するには、TestInitialize を使用
        //
        [TestInitialize()]
        public void MyTestInitialize()
        {
            LogUTUtil.Clear();
        }
        //
        //各テストを実行した後にコードを実行するには、TestCleanup を使用
        //
        //[TestCleanup()]
        //public void MyTestCleanup()
        //{
        //}
        //
        #endregion


        /// <summary>
        ///CreateDefaultRequestController (string) のテスト
        ///</summary>
        /// <TestTarget>CreateDefaultRequestController</TestTarget>
        /// <TestCondition>引数に"Normal"を指定すること</TestCondition>
        /// <TestAssertion>引数に対応する既定のリクエストコントローラクラスが取得できること</TestAssertion>
        [TestMethod()]
        public void TestCreateDefaultRequestController01()
        {
            RequestControllerFactory target = new RequestControllerFactory();

            TERASOLUNA_Fw_Web_Controller_RequestControllerFactoryAccessor accessor = new TERASOLUNA_Fw_Web_Controller_RequestControllerFactoryAccessor(target);

            string requestControllerName = "Normal";

            Type expected = typeof(BLogicRequestController);
            IHttpHandler actual;

            actual = accessor.CreateDefaultRequestController(requestControllerName);

            Assert.IsInstanceOfType(actual, expected);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(RequestControllerFactory), LogLevel.INFO));
        }

        /// <summary>
        ///CreateDefaultRequestController (string) のテスト
        ///</summary>
        /// <TestTarget>CreateDefaultRequestController</TestTarget>
        /// <TestCondition>引数に"Upload"を指定すること</TestCondition>
        /// <TestAssertion>引数に対応する既定のリクエストコントローラクラスが取得できること</TestAssertion>
        [TestMethod()]
        public void TestCreateDefaultRequestController02()
        {
            RequestControllerFactory target = new RequestControllerFactory();

            TERASOLUNA_Fw_Web_Controller_RequestControllerFactoryAccessor accessor = new TERASOLUNA_Fw_Web_Controller_RequestControllerFactoryAccessor(target);

            string requestControllerName = "Upload";

            Type expected = typeof(FileUploadRequestController);
            IHttpHandler actual;

            actual = accessor.CreateDefaultRequestController(requestControllerName);

            Assert.IsInstanceOfType(actual, expected);
            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(RequestControllerFactory), LogLevel.INFO));
        }

        /// <summary>
        ///CreateDefaultRequestController (string) のテスト
        ///</summary>
        /// <TestTarget>CreateDefaultRequestController</TestTarget>
        /// <TestCondition>引数に"Download"を指定すること</TestCondition>
        /// <TestAssertion>引数に対応する既定のリクエストコントローラクラスが取得できること</TestAssertion>
        [TestMethod()]
        public void TestCreateDefaultRequestController03()
        {
            RequestControllerFactory target = new RequestControllerFactory();

            TERASOLUNA_Fw_Web_Controller_RequestControllerFactoryAccessor accessor = new TERASOLUNA_Fw_Web_Controller_RequestControllerFactoryAccessor(target);

            string requestControllerName = "Download";

            Type expected = typeof(FileDownloadRequestController);
            IHttpHandler actual;

            actual = accessor.CreateDefaultRequestController(requestControllerName);

            Assert.IsInstanceOfType(actual, expected);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(RequestControllerFactory), LogLevel.INFO));
        }

        /// <summary>
        ///CreateDefaultRequestController (string) のテスト
        ///</summary>
        /// <TestTarget>CreateDefaultRequestController</TestTarget>
        /// <TestCondition>引数に"Unknown"を指定すること</TestCondition>
        /// <TestAssertion>引数に対応する既定のリクエストコントローラクラスが取得できること</TestAssertion>
        [TestMethod()]
        public void TestCreateDefaultRequestController04()
        {
            RequestControllerFactory target = new RequestControllerFactory();

            TERASOLUNA_Fw_Web_Controller_RequestControllerFactoryAccessor accessor = new TERASOLUNA_Fw_Web_Controller_RequestControllerFactoryAccessor(target);

            string requestControllerName = "Unknown";

            Type expected = typeof(UnknownRequestController);
            IHttpHandler actual;

            actual = accessor.CreateDefaultRequestController(requestControllerName);

            Assert.IsInstanceOfType(actual, expected);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(RequestControllerFactory), LogLevel.INFO));
        }

        /// <summary>
        ///CreateDefaultRequestController (string) のテスト
        ///</summary>
        /// <TestTarget>CreateDefaultRequestController</TestTarget>
        /// <TestCondition>引数に既定リクエストコントローラに対応しない値を指定すること</TestCondition>
        /// <TestAssertion>TerasolunaExceptionがスローされること</TestAssertion>
        [TestMethod()]
        public void TestCreateDefaultRequestController05()
        {
            RequestControllerFactory target = new RequestControllerFactory();

            TERASOLUNA_Fw_Web_Controller_RequestControllerFactoryAccessor accessor = new TERASOLUNA_Fw_Web_Controller_RequestControllerFactoryAccessor(target);

            string requestControllerName = "Hoge";

            Type expected = typeof(TerasolunaException);
            IHttpHandler actual;

            try
            {
                actual = accessor.CreateDefaultRequestController(requestControllerName);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException ex)
            {
                //TerasolunaException re = ex.InnerException as TerasolunaException;
                Assert.AreEqual(typeof(TerasolunaException).Name, ex.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_UNKNOWN_REQUEST_CONTROLLER", requestControllerName), ex.Message);

            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(RequestControllerFactory), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(
                    typeof(RequestControllerFactory), typeof(TerasolunaException), "E_UNKNOWN_REQUEST_CONTROLLER", requestControllerName));
        }

        /// <summary>
        ///CreateDefaultRequestController (string) のテスト
        ///</summary>
        /// <TestTarget>CreateDefaultRequestController</TestTarget>
        /// <TestCondition>引数に空文字を指定すること</TestCondition>
        /// <TestAssertion>TerasolunaExceptionがスローされること</TestAssertion>
        [TestMethod()]
        public void TestCreateDefaultRequestController06()
        {
            RequestControllerFactory target = new RequestControllerFactory();

            TERASOLUNA_Fw_Web_Controller_RequestControllerFactoryAccessor accessor = new TERASOLUNA_Fw_Web_Controller_RequestControllerFactoryAccessor(target);

            string requestControllerName = "";

            Type expected = typeof(TerasolunaException);
            IHttpHandler actual;

            try
            {
                actual = accessor.CreateDefaultRequestController(requestControllerName);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException ex)
            {
                //TerasolunaException re = ex.InnerException as TerasolunaException;
                Assert.AreEqual(typeof(TerasolunaException).Name, ex.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_UNKNOWN_REQUEST_CONTROLLER", requestControllerName), ex.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(RequestControllerFactory), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(
                    typeof(RequestControllerFactory), typeof(TerasolunaException), "E_UNKNOWN_REQUEST_CONTROLLER", requestControllerName));
        }

        /// <summary>
        ///CreateDefaultRequestController (string) のテスト
        ///</summary>
        /// <TestTarget>CreateDefaultRequestController</TestTarget>
        /// <TestCondition>引数にnullを指定すること</TestCondition>
        /// <TestAssertion>TerasolunaExceptionがスローされること</TestAssertion>
        [TestMethod()]
        public void TestCreateDefaultRequestController07()
        {
            RequestControllerFactory target = new RequestControllerFactory();

            TERASOLUNA_Fw_Web_Controller_RequestControllerFactoryAccessor accessor = new TERASOLUNA_Fw_Web_Controller_RequestControllerFactoryAccessor(target);

            string requestControllerName = null;

            Type expected = typeof(TerasolunaException);
            IHttpHandler actual;

            try
            {
                actual = accessor.CreateDefaultRequestController(requestControllerName);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException ex)
            {
                //TerasolunaException re = ex.InnerException as TerasolunaException;
                Assert.AreEqual(typeof(TerasolunaException).Name, ex.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_UNKNOWN_REQUEST_CONTROLLER", requestControllerName), ex.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(RequestControllerFactory), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(
                    typeof(RequestControllerFactory), typeof(TerasolunaException), "E_UNKNOWN_REQUEST_CONTROLLER", requestControllerName));

        }


        /// <summary>
        /// GetTargetRequestController (string) のテスト
        /// </summary>
        /// <TestCondition>
        /// 引数に構成ファイルのリクエストコントローラ設定で設定されている
        /// リクエストタイプを指定すること
        /// </TestCondition>
        /// <TestAssertion>
        /// 引数に対応する構成ファイルのリクエストコントローラ設定で
        /// 設定されている該当リクエストコントローラクラスが取得できること
        /// </TestAssertion>
        [TestMethod()]
        public void TestGetTargetRequestController01()
        {
            RequestControllerFactory target = new RequestControllerFactory();

            TERASOLUNA_Fw_Web_Controller_RequestControllerFactoryAccessor accessor = new TERASOLUNA_Fw_Web_Controller_RequestControllerFactoryAccessor(target);

            string reqCtrlName = "Normal"; 

            Type expected = typeof(BLogicRequestController);
            IHttpHandler actual;

            actual = accessor.GetTargetRequestController(reqCtrlName);

            Assert.IsInstanceOfType(actual, expected);
       
            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(RequestControllerFactory), LogLevel.INFO));
        }

        /// <summary>
        /// GetTargetRequestController (string) のテスト
        /// </summary>
        /// <TestCondition>
        /// 引数に構成ファイルのリクエストコントローラ設定で設定されていない
        /// リクエストタイプを指定すること
        /// </TestCondition>
        /// <TestAssertion>Terasolunaexceptionをスローすること</TestAssertion>
        [TestMethod()]
        public void TestGetTargetRequestController02()
        {
            RequestControllerFactory target = new RequestControllerFactory();

            TERASOLUNA_Fw_Web_Controller_RequestControllerFactoryAccessor accessor = new TERASOLUNA_Fw_Web_Controller_RequestControllerFactoryAccessor(target);

            string reqCtrlName = "TERASOLUNA";

            Type expected = typeof(TerasolunaException);
            IHttpHandler actual;

            try
            {
                actual = accessor.GetTargetRequestController(reqCtrlName);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException ex)
            {
                //TerasolunaException te = ex.InnerException as TerasolunaException;
                Assert.AreEqual(typeof(TerasolunaException).Name, ex.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_UNKNOWN_REQUEST_CONTROLLER", reqCtrlName),ex.Message);
                Assert.IsInstanceOfType(ex, expected);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(RequestControllerFactory), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(RequestControllerFactory), typeof(TerasolunaException), "E_UNKNOWN_REQUEST_CONTROLLER", reqCtrlName));

        }

        /// <summary>
        /// GetTargetRequestController (string) のテスト
        /// </summary>
        /// <TestCondition>
        /// 引数に空文字を指定すること
        /// </TestCondition>
        /// <TestAssertion>Terasolunaexceptionをスローすること</TestAssertion>
        [TestMethod()]
        public void TestGetTargetRequestController03()
        {
            RequestControllerFactory target = new RequestControllerFactory();

            TERASOLUNA_Fw_Web_Controller_RequestControllerFactoryAccessor accessor = new TERASOLUNA_Fw_Web_Controller_RequestControllerFactoryAccessor(target);

            string reqCtrlName = "";

            Type expected = typeof(TerasolunaException);
            IHttpHandler actual;

            try
            {
                actual = accessor.GetTargetRequestController(reqCtrlName);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException ex)
            {
                //TerasolunaException te = ex.InnerException as TerasolunaException;
                Assert.AreEqual(typeof(TerasolunaException).Name, ex.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_UNKNOWN_REQUEST_CONTROLLER", reqCtrlName), ex.Message);
                Assert.IsInstanceOfType(ex, expected);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(RequestControllerFactory), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(RequestControllerFactory), typeof(TerasolunaException), "E_UNKNOWN_REQUEST_CONTROLLER", reqCtrlName));
        }

        /// <summary>
        /// GetTargetRequestController (string) のテスト
        /// </summary>
        /// <TestCondition>
        /// 引数にnullを指定すること
        /// </TestCondition>
        /// <TestAssertion>ArgumentNullExceptionをスローすること</TestAssertion>
        [TestMethod()]
        public void TestGetTargetRequestController04()
        {
            RequestControllerFactory target = new RequestControllerFactory();

            TERASOLUNA_Fw_Web_Controller_RequestControllerFactoryAccessor accessor = new TERASOLUNA_Fw_Web_Controller_RequestControllerFactoryAccessor(target);

            string reqCtrlName = null;

            Type expected = typeof(ArgumentNullException);
            IHttpHandler actual;

            try
            {
                actual = accessor.GetTargetRequestController(reqCtrlName);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException ex)
            {
                //ArgumentNullException ae = ex.InnerException as ArgumentNullException;
                Assert.AreEqual("name", ex.ParamName);
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(RequestControllerFactory), LogLevel.INFO));
        }

        /// <summary>
        /// BLogicTypes のテスト
        /// </summary>
        /// <TestCondition>
        /// 特になし
        /// </TestCondition>
        /// <TestAssertion>
        /// 初期値を取得すること
        /// </TestAssertion>
        [TestMethod()]
        public void TestBLogicTypes01()
        {
            RequestControllerFactory target = new RequestControllerFactory();
            TERASOLUNA_Fw_Web_Controller_RequestControllerFactoryAccessor accessor = new TERASOLUNA_Fw_Web_Controller_RequestControllerFactoryAccessor(target);
         
   　       // 試験実施
            IDictionary<string, Type> actual = accessor.BLogicTypes;

            // 確認
            Assert.IsInstanceOfType(actual, typeof(Dictionary<string, Type>));
            Assert.AreEqual(0, actual.Count);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(RequestControllerFactory), LogLevel.INFO));
        }
    }
}
