﻿// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using System;
using System.Configuration;
using TERASOLUNA.Fw.Common.Logging;

namespace TERASOLUNA.Fw.Common.Configuration.BLogic
{
    /// <summary>
    /// dbInfo 要素のコレクションを格納する構成要素を表します。
    /// </summary>
    /// <remarks>
    /// <see cref="BLogicDbInfoCollection"/> は、ビジネスロジック設定ファイル内の dbInfo 要素のコレクションを表します。 
    /// 詳細については <see cref="ConfigurationElementCollection"/> を参照してください。
    /// </remarks>
    public class BLogicDbInfoCollection : ConfigurationElementCollection
    {
        /// <summary>
        /// <see cref="ILog"/> 実装クラスのインスタンスです。
        /// </summary>
        /// <remarks>
        /// ログ出力に利用します。
        /// </remarks>
        private static ILog _log = LogFactory.GetLogger(typeof(BLogicCollection));

        /// <summary>
        /// dbInfo 要素の要素名です。
        /// </summary>
        /// <remarks>
        /// この定数の値は、 "dbInfo" です。
        /// </remarks>
        private static readonly string DBINFO_ELEMENT_NAME = "dbInfo";

        /// <summary>
        /// <see cref="BLogicDbInfoCollection"/> クラスの新しいインスタンスを初期化します。
        /// </summary>
        /// <remarks>
        /// デフォルトコンストラクタです。
        /// </remarks>
        public BLogicDbInfoCollection()
        {
        }

        /// <summary>
        /// <see cref="BLogicDbInfoCollection"/> の型を取得します。
        /// </summary>
        /// <remarks>
        /// このコレクションの <see cref="ConfigurationElementCollectionType"/> は BasicMap です。
        /// </remarks>
        /// <value>
        /// このコレクションの <see cref="ConfigurationElementCollectionType"/> 。
        /// </value>
        public override ConfigurationElementCollectionType CollectionType
        {
            get
            {
                return ConfigurationElementCollectionType.BasicMap;
            }
        }

        /// <summary>
        /// 新しい dbInfo 要素を作成します。
        /// </summary>
        /// <returns>新しい dbInfo 要素。</returns>
        /// <remarks>
        /// 新しい <see cref="BLogicDbInfoElement"/> クラスのインスタンスを作成します。
        /// </remarks>
        protected override ConfigurationElement CreateNewElement()
        {
            return new BLogicDbInfoElement();
        }

        /// <summary>
        /// 指定したインデックス位置にある dbInfo 要素を取得します。
        /// </summary>
        /// <param name="index">取得する <see cref="BLogicDbInfoElement"/> のインデックス位置。</param>
        /// <returns>指定したインデックス位置にある <see cref="BLogicDbInfoElement"/> 。</returns>
        /// <remarks>
        /// <seealso cref="ConfigurationElementCollection.BaseGet(int)"/> メソッドを使用して、<see cref="BLogicDbInfoCollection"/> から <see cref="BLogicDbInfoElement"/> を取得します。
        /// </remarks>
        public BLogicDbInfoElement this[int index]
        {
            get
            {
                return (BLogicDbInfoElement)BaseGet(index);
            }
        }

        /// <summary>
        /// 指定したキーを持つ dbInfo 要素を取得します。
        /// </summary>
        /// <param name="name">取得する dbInfo 要素のキー。</param>
        /// <returns>指定したキーを持つ <see cref="BLogicDbInfoElement"/> 。それ以外の場合は null 参照です。</returns>
        /// <remarks>
        /// <seealso cref="ConfigurationElementCollection.BaseGet(object)"/> メソッドを使用して、 <see cref="BLogicDbInfoCollection"/> から <see cref="BLogicDbInfoElement"/> を取得します。
        /// 指定したキーを持つ <see cref="BLogicDbInfoElement"/> がコレクションに存在しない場合、null 参照を返します。
        /// </remarks>
        public new BLogicDbInfoElement this[string name]
        {
            get
            {
                return (BLogicDbInfoElement)BaseGet(name);
            }
        }

        /// <summary>
        /// ビジネスロジック設定ファイル内のこの要素のコレクションを識別するために使用する名前を取得します。
        /// </summary>
        /// <remarks>
        /// 要素名は dbInfo です。
        /// </remarks>
        /// <value>
        /// この要素のコレクションの要素名。
        /// </value>
        protected override string ElementName
        {
            get
            {
                return DBINFO_ELEMENT_NAME;
            }
        }

        /// <summary>
        /// dbInfo 要素の要素キーを取得します。
        /// </summary>
        /// <param name="element">取得する <see cref="BLogicDbInfoElement"/> 要素のキー。</param>
        /// <returns>指定したキーを持つ <see cref="BLogicDbInfoElement"/> 。それ以外の場合は null 参照です。</returns>
        /// <exception cref="System.ArgumentNullException">
        /// <paramref name="element"/> が null 参照です。
        /// </exception>
        /// <exception cref="System.ArgumentException">
        /// <paramref name="element"/> が <see cref="BLogicDbInfoElement"/> にキャストできません。
        /// </exception>
        protected override Object GetElementKey(ConfigurationElement element)
        {
            if (element == null)
            {
                ArgumentNullException exception = new ArgumentNullException("element");
                if (_log.IsErrorEnabled)
                {
                    _log.Error(string.Format(
                        Common.Properties.Resources.E_NULL_ARGUMENT, "element"), exception);
                }
                throw exception;
            }
            BLogicDbInfoElement dbInfoElement = element as BLogicDbInfoElement;
            if (dbInfoElement == null)
            {
                string message = string.Format(
                    TERASOLUNA.Fw.Common.Properties.Resources.E_INVALID_CAST,
                    element.GetType().FullName, typeof(BLogicDbInfoElement).FullName);
                ArgumentException exception = new ArgumentException(message);
                if (_log.IsErrorEnabled)
                {
                    _log.Error(message, exception);
                }
                throw exception;
            }

            return dbInfoElement.Name;
        }

        /// <summary>
        /// <see cref="BLogicDbInfoCollection"/> に dbInfo 要素を追加します。
        /// </summary>
        /// <param name="element">追加する <see cref="BLogicDbInfoElement"/> 。</param>
        /// <remarks>
        /// <seealso cref="ConfigurationElementCollection.BaseAdd(ConfigurationElement)"/> メソッドを使用して、
        /// コレクションに新しい <see cref="BLogicDbInfoElement"/> を追加します。
        /// </remarks>
        public void Add(BLogicDbInfoElement element)
        {
            BaseAdd(element);
        }

        /// <summary>
        /// <see cref="BLogicDbInfoCollection"/> に dbInfo 要素を追加します。
        /// </summary>
        /// <param name="element">追加する <see cref="BLogicDbInfoElement"/> 。</param>
        /// <remarks>
        /// 指定した <paramref name="element"/> が dbInfo 要素コレクションに既に存在する場合、例外は発生しません。
        /// </remarks>
        protected override void BaseAdd(ConfigurationElement element)
        {
            BaseAdd(element, false);
        }

        /// <summary>
        /// <see cref="BLogicDbInfoCollection"/> が読み取り専用かどうかを示す値を取得します。 
        /// </summary>
        /// <returns>
        /// <see cref="BLogicDbInfoCollection"/> が読み取り専用の場合は true 。それ以外の場合は false 。
        /// </returns>
        /// <remarks>
        /// <see cref="BLogicDbInfoCollection"/> は読み取り専用ではありません。
        /// </remarks>
        public override bool IsReadOnly()
        {
            return false;
        }
    }
}
