/*
 * Copyright (c) 2012 NTT DATA Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package jp.terasoluna.fw.ex.web.struts.action;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.jsp.PageContext;

import jp.terasoluna.fw.web.RequestUtil;

import org.apache.struts.Globals;
import org.apache.struts.action.ActionForm;
import org.apache.struts.action.ActionForward;
import org.apache.struts.action.ActionMapping;
import org.apache.struts.action.ActionMessages;
import org.apache.struts.action.ExceptionHandler;
import org.apache.struts.config.ExceptionConfig;
import org.apache.struts.util.MessageResources;

/**
 * 抽象例外ハンドラクラスです。 
 * 提供する各例外ハンドラクラスに共通する処理を集約しています。
 * 
 * @version 1.0.0
 */
public abstract class AbstractExceptionHandler extends ExceptionHandler {

    /**
     * 以下の例外処理を順次実行します。<br>
     * <ol>
     * <li>遷移先を表すActionForwardインスタンスを取得します。</li>
     * <li>リソースバンドルからエラーメッセージの一覧を取得し、
     * 指定されたスコープに登録します。</li>
     * <li>発生した例外をログに出力します。</li>
     * <li>発生した例外の情報をリクエストに登録します。</li>
     * </ol>
     * 
     * @param ex 例外オブジェクト
     * @param ae ExceptionConfig
     * @param mapping アクションマッピング
     * @param form アクションフォーム
     * @param request リクエスト
     * @param response レスポンス
     * @return 遷移先を表すActionFowardインスタンス
     */
    @Override
    public ActionForward execute(Exception ex, ExceptionConfig ae,
            ActionMapping mapping, ActionForm form, HttpServletRequest request,
            HttpServletResponse response) throws ServletException {
        ActionForward forward = getActionForward(ex, ae, mapping, request);
        setModule(forward, ex, ae, mapping);
        String bundleKey = getBundleKey(ae);
        MessageResources resources = getMessageResources(request, bundleKey);
        ActionMessages errors = getErrors(ex, ae, resources);
        storeException(request, errors, forward, ae.getScope());
        logException(ex);
        setRequestAttribute(request, ex, ae, mapping);
        return forward;
    }

    /**
     * 遷移先を表すActionForwardインスタンスを取得します。 
     * 以下の順序で遷移先を特定します。
     * <ol>
     * <li>exception要素のpath属性</li>
     * <li>実行中のaction要素のinput属性</li>
     * <li>いずれも取得できない場合はnull</li>
     * </ol>
     * 
     * @param ex 例外オブジェクト
     * @param ae ExceptionConfig
     * @param mapping アクションマッピング
     * @param request リクエスト
     * @return 遷移先を表すActionForwardインスタンス<br>
     */
    protected ActionForward getActionForward(Exception ex, ExceptionConfig ae,
            ActionMapping mapping, HttpServletRequest request) {
        ActionForward forward = null;
        if (ae.getPath() != null) {
            forward = new ActionForward(ae.getPath());
        } else {
            forward = mapping.getInputForward();
        }
        return forward;
    }

    /**
     * リソースバンドルのキーを取得します。
     * 
     * @param ae ExceptionConfig
     * @return リソースバンドルのキー
     */
    protected String getBundleKey(ExceptionConfig ae) {
        if (ae.getBundle() != null) {
            return ae.getBundle();
        }
        return Globals.MESSAGES_KEY;
    }

    /**
     * メッセージリソースを取得します。
     * 
     * @param request リクエスト
     * @param bundleKey リソースバンドルのキー
     * @return メッセージリソース
     */
    protected MessageResources getMessageResources(HttpServletRequest request,
            String bundleKey) {
        MessageResources messageResources = (MessageResources) request
                .getAttribute(bundleKey);
        if (messageResources != null) {
            return messageResources;
        }
        return (MessageResources) RequestUtil.getServletContext(request)
                .getAttribute(bundleKey);
    }

    /**
     * Globals.ERROR_KEYをキーに、エラーメッセージの一覧を
     * 指定されたスコープに登録します。
     * 
     * @param request リクエスト
     * @param errors エラーメッセージ一覧
     * @param forward ActionFowardインスタンス
     * @param scope エラーメッセージの一覧を登録するスコープ
     */
    protected void storeException(HttpServletRequest request,
            ActionMessages errors, ActionForward forward, String scope) {
        if ("request".equals(scope)) {
            request.setAttribute(Globals.ERROR_KEY, errors);
        } else {
            request.getSession().setAttribute(Globals.ERROR_KEY, errors);
        }
    }

    /**
     * Globals.EXCEPTION_KEY、およびPageContext.EXCEPTIONをキーに、
     * 発生した例外をリクエストに登録します。
     * 
     * @param request リクエスト
     * @param ex 例外
     * @param ae ExceptionConfig
     * @param mapping アクションマッピング
     */
    protected void setRequestAttribute(HttpServletRequest request,
            Exception ex, ExceptionConfig ae, ActionMapping mapping) {
        request.setAttribute(Globals.EXCEPTION_KEY, ex);
        request.setAttribute(PageContext.EXCEPTION, ex);
    }

    /**
     * エラーメッセージの一覧を保持するActionMessagesのインスタンスを取得します。
     * 
     * @param ex 例外
     * @param ae ExceptionConfig
     * @param resources メッセージリソース
     * @return エラーメッセージの一覧を保持するActionMessagesインスタンス
     */
    protected abstract ActionMessages getErrors(Exception ex,
            ExceptionConfig ae, MessageResources resources);

    /**
     * ActionForwardインスタンスに、遷移先モジュールを設定します。
     * 
     * @param forward ActionForwardインスタンス
     * @param ex 例外
     * @param ae ExceptionConfig
     * @param mapping アクションマッピング
     */
    protected abstract void setModule(ActionForward forward, Exception ex,
            ExceptionConfig ae, ActionMapping mapping);
}
