/*
 * Copyright (c) 2011 NTT DATA Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package jp.terasoluna.fw.collector;

import java.beans.Introspector;
import java.io.Closeable;
import java.io.IOException;
import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.Queue;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;

import jp.terasoluna.fw.collector.exception.CollectorException;
import jp.terasoluna.fw.collector.exception.CollectorExceptionHandler;
import jp.terasoluna.fw.collector.exception.CollectorExceptionHandlerStatus;
import jp.terasoluna.fw.collector.validate.ValidateErrorStatus;
import jp.terasoluna.fw.collector.validate.ValidationErrorHandler;
import jp.terasoluna.fw.collector.vo.CollectorStatus;
import jp.terasoluna.fw.collector.vo.DataValueObject;
import jp.terasoluna.fw.exception.SystemException;
import jp.terasoluna.fw.logger.TLogger;

import org.springframework.validation.BindException;
import org.springframework.validation.Errors;
import org.springframework.validation.Validator;

/**
 * AbstractCollectorۃNX
 * @param &lt;P&gt;
 */
public abstract class AbstractCollector<P> implements Collector<P>, Closeable,
                                           Callable<Integer>, Cloneable {
    /**
     * Log.
     */
    private static final TLogger LOGGER = TLogger
            .getLogger(AbstractCollector.class);

    /** ftHg̃L[TCY */
    public static final int DEFAULT_QUEUE_SIZE = 20;

    /** ftHgX[v(msec) */
    protected static final int DEFAULT_SLEEP_WAIT = 1;

    /** ݃L[ێ`FbNTCY */
    protected static final int CURRENT_QUEUE_CHECK_SIZE = 1;

    /** L[ێ`FbNTCY */
    protected static final int PREVIOUS_QUEUE_CHECK_SIZE = 2;

    /** 璷Oo̓tO. */
    protected static AtomicBoolean verboseLog = new AtomicBoolean(false);

    /** L[TCY */
    protected int queueSize = DEFAULT_QUEUE_SIZE;

    /** X[v(msec) */
    protected int sleepWait = DEFAULT_SLEEP_WAIT;

    /** L[ */
    protected Queue<DataValueObject> queue = null;

    /** ݃L[ */
    protected Queue<DataValueObject> currentQueue = null;

    /** L[ */
    protected Queue<DataValueObject> previousQueue = null;

    /** 񓯊̌ʂ擾邽߂̃NX */
    protected volatile Future<?> fo = null;

    /** ItO */
    protected volatile boolean finish = false;

    /** sJntO */
    protected volatile boolean beginning = false;

    /** Validator */
    protected Validator validator = null;

    /** ValidationErrorHandler */
    protected ValidationErrorHandler validationErrorHandler = null;

    /** CollectorExceptionHandler */
    protected CollectorExceptionHandler exceptionHandler = null;

    /** qXbhCX^X */
    protected AbstractCollector<?> child = null;

    /** f[^JEg */
    protected AtomicLong acquireDataCount = new AtomicLong(0);

    /**
     * AbstractCollectorsB
     */
    @SuppressWarnings("unchecked")
    protected void execute() {
        if (this.beginning) {
            return;
        }
        synchronized (this) {
            if (!this.beginning) {

                if (this.queue == null) {
                    // L[
                    this.queue = createQueue();
                }

                if (this.fo == null) {
                    // g̃N[쐬
                    Callable<Integer> callable = null;
                    try {
                        callable = (Callable<Integer>) this.clone();
                    } catch (CloneNotSupportedException e) {
                        SystemException exception = new SystemException(e);
                        exception.setMessage("The clone cannot be made.");
                        throw exception;
                    }

                    if (callable instanceof AbstractCollector) {
                        this.child = (AbstractCollector) callable;
                    }

                    // ExecutorService擾
                    ExecutorService ex = getExecutor();

                    try {
                        // ʃXbhŎs
                        this.fo = ex.submit(callable);
                    } catch (Throwable e) {
                        SystemException exception = new SystemException(e);
                        exception.setMessage("The thread cannot be started.");
                        throw exception;
                    } finally {
                        ex.shutdown();
                    }
                }

                this.beginning = true;
            }
        }
    }

    /**
     * JԂłɗvfꍇ true Ԃ܂B<br>
     * ܂Anext ̌ĂяoOX[邱ƂȂvfԂꍇ́Atrue Ԃ܂B
     * <p>
     * <b>{\bh̓}`XbhZ[tł܂B</b>
     * </p>
     * @return qɗvfꍇ true
     * @see java.util.Iterator#hasNext()
     */
    public boolean hasNext() {
        // sJnî݁j
        execute();

        while (true) {
            // L[łȂǂ
            if (this.queue != null && !this.queue.isEmpty()) {
                DataValueObject value = null;
                // L[1f[^擾i폜Ȃj
                value = this.queue.peek();

                // RN^Xe[^X擾
                if (value != null && value.getCollectorStatus() != null) {
                    if (verboseLog.get() && LOGGER.isTraceEnabled()) {
                        LOGGER.trace(LogId.TAL041017);
                    }

                    this.queue.poll();
                    CollectorStatus collectorStatus = value
                            .getCollectorStatus();
                    if (CollectorStatus.END.equals(collectorStatus)) {
                        if (verboseLog.get() && LOGGER.isTraceEnabled()) {
                            LOGGER.trace(LogId.TAL041016);
                        }
                        // ItO𗧂Ă
                        setFinish(true);

                        if (verboseLog.get() && LOGGER.isTraceEnabled()) {
                            LOGGER.trace(LogId.TAL041015, this.queue.size());
                        }
                        if (!this.queue.isEmpty()) {
                            continue;
                        }
                        break;
                    }
                }
                return true;
            }

            // ItO
            if (isFinish() && this.queue.isEmpty()) {
                if (verboseLog.get() && LOGGER.isTraceEnabled()) {
                    LOGGER.trace(LogId.TAL041014);
                }
                break;
            }

            try {
                if (verboseLog.get() && LOGGER.isTraceEnabled()) {
                    LOGGER.trace(LogId.TAL041006, sleepWait);
                }
                // sleepWait ms҂
                Thread.sleep(sleepWait);
            } catch (InterruptedException e) {
                LOGGER.warn(LogId.WAL041003, e);
                break;
            }
        }

        return false;
    }

    /**
     * JԂŎ̗vfԂ܂B
     * <p>
     * <b>{\bh̓}`XbhZ[tł܂B</b>
     * </p>
     * @return JԂŎ̗vf
     * @throws NoSuchElementException JԂłȏvfȂꍇ
     * @see java.util.Iterator#next()
     */
    @SuppressWarnings("unchecked")
    public P next() {
        // sJnî݁j
        execute();

        P pn = getNext();
        if (pn != null) {
            if (this.previousQueue != null) {
                while (this.previousQueue.size() > PREVIOUS_QUEUE_CHECK_SIZE) {
                    this.previousQueue.remove();
                }
                this.previousQueue.add(new DataValueObject(pn));
            }
            if (this.currentQueue != null) {
                while (this.currentQueue.size() > CURRENT_QUEUE_CHECK_SIZE) {
                    this.currentQueue.remove();
                }
                this.currentQueue.add(new DataValueObject(pn));
            }
        }

        DataValueObject value = null;
        CollectorExceptionHandlerStatus es = null;
        do {
            // L[1f[^擾
            if (this.queue instanceof BlockingQueue) {
                try {
                    value = ((BlockingQueue<DataValueObject>) this.queue).poll(
                            sleepWait, TimeUnit.MILLISECONDS);
                } catch (InterruptedException e) {
                    LOGGER.warn(LogId.WAL041003, e);
                    break;
                }
            } else {
                if (this.queue != null) {
                    value = this.queue.poll();
                }
            }

            // ItO
            if (isFinish() && this.queue.isEmpty()) {
                if (verboseLog.get() && LOGGER.isTraceEnabled()) {
                    LOGGER.trace(LogId.TAL041014);
                }
                break;
            }

            if (value == null && verboseLog.get() && LOGGER.isTraceEnabled()) {
                LOGGER.trace(LogId.TAL041007, this.queue.size());
            }

            if (value != null) {
                // XLbv
                if (value.getValidateStatus() != null) {
                    ValidateErrorStatus validateStatus = value
                            .getValidateStatus();
                    // XLbv̎
                    if (ValidateErrorStatus.SKIP.equals(validateStatus)) {
                        acquireDataCount.incrementAndGet();
                        value = null;
                        continue;
                    }
                } else if (value.getThrowable() != null
                        && this.exceptionHandler != null) {
                    try {
                        // Onhs
                        es = handleException(value);
                    } catch (Throwable e) {
                        LOGGER.warn(LogId.WAL041004, e);
                        // ł̗O̓OɎĉ݂ŉȂ
                    }
                    // XLbv̎
                    if (CollectorExceptionHandlerStatus.SKIP.equals(es)) {
                        acquireDataCount.incrementAndGet();
                        es = null;
                        value = null;
                        continue;
                    }
                }
            }
        } while (value == null);

        if (value == null) {
            return null;
        } else if (value.getCollectorStatus() != null) {
            CollectorStatus collectorStatus = value.getCollectorStatus();
            if (CollectorStatus.END.equals(collectorStatus)) {
                setFinish(true);
            }
            return null;
        } else if (value.getValue() == null) {
            acquireDataCount.incrementAndGet();
            if (value.getValidateStatus() != null) {
                // DataValueObjectɓ̓`FbNXe[^Xi~ji[ĂN[Y
                if (ValidateErrorStatus.END.equals(value.getValidateStatus())) {
                    this.close();
                }
            } else if (value.getThrowable() != null) {
                Throwable throwable = value.getThrowable();

                if (es == null
                        || CollectorExceptionHandlerStatus.THROW.equals(es)) {
                    // OX[
                    if (throwable instanceof RuntimeException) {
                        throw (RuntimeException) throwable;
                    } else {
                        throw new SystemException(throwable);
                    }
                } else if (CollectorExceptionHandlerStatus.END.equals(es)) {
                    // ~iȍ~̃L[CO~j
                    this.close();
                }
            }
            return null;
        }

        if (value != null) {
            long dtcnt = acquireDataCount.incrementAndGet();
            if (dtcnt != value.getDataCount()) {
                StringBuilder sb = new StringBuilder();
                sb.append("dtcnt:");
                sb.append(dtcnt);
                sb.append(" DataValueObject dataCount:");
                sb.append(value.getDataCount());

                DataValueObject next = this.queue.peek();
                if (next != null) {
                    sb.append(" getNext() dataCount:");
                    sb.append(next.getDataCount());
                }

                DataValueObject previous = this.previousQueue.peek();
                if (previous != null) {
                    sb.append(" getPrevious dataCount:");
                    sb.append(previous.getDataCount());
                }

                DataValueObject current = this.currentQueue.peek();
                if (current != null) {
                    sb.append(" getCurrent dataCount:");
                    sb.append(current.getDataCount());
                }
                throw new CollectorException(sb.toString());
            }
        }

        return (P) value.getValue();
    }

    /**
     * |C^̗vfɈڂɎ̗vfԂ܂B<br>
     * <p>
     * null̏ꍇ͎̗vf݂ȂƂ܂B<br>
     * |C^͈ړ܂B
     * </p>
     * <p>
     * <b>{\bh̓}`XbhZ[tł܂B</b>
     * </p>
     * @return &lt;P&gt;
     * @see jp.terasoluna.fw.collector.Collector#getNext()
     */
    @SuppressWarnings("unchecked")
    public P getNext() {
        // sJnî݁j
        execute();

        DataValueObject value = null;
        do {
            // L[1f[^擾i폜Ȃj
            if (this.queue != null) {
                value = this.queue.peek();
            }

            // ItO
            if (isFinish() && this.queue.isEmpty()) {
                if (verboseLog.get() && LOGGER.isTraceEnabled()) {
                    LOGGER.trace(LogId.TAL041014);
                }
                break;
            }

            // null̏ꍇ̓X[v
            if (value == null) {
                try {
                    Thread.sleep(sleepWait);
                } catch (InterruptedException e) {
                    LOGGER.warn(LogId.WAL041003, e);
                    break;
                }
                if (verboseLog.get() && LOGGER.isTraceEnabled()) {
                    LOGGER.trace(LogId.TAL041008, this.queue.size());
                }
            }
        } while (value == null);

        if (value == null) {
            return null;
        } else if (value.getCollectorStatus() != null) {
            CollectorStatus collectorStatus = value.getCollectorStatus();
            if (CollectorStatus.END.equals(collectorStatus)) {
                setFinish(true);
            }
            return null;
        } else if (value.getValue() == null) {
            // peekȂ̂ŗO͖
            return null;
        }

        return (P) value.getValue();
    }

    /**
     * 1O̗vfԂ܂B<br>
     * <p>
     * 1ڂ̏ꍇnullԂ܂B<br>
     * |C^͈ړ܂B
     * </p>
     * <p>
     * <b>{\bh̓}`XbhZ[tł܂B</b>
     * </p>
     * @return &lt;P&gt;
     * @see jp.terasoluna.fw.collector.Collector#getPrevious()
     */
    @SuppressWarnings("unchecked")
    public P getPrevious() {
        // sJnî݁j
        execute();

        if (this.previousQueue != null && this.previousQueue.size() > 1) {
            while (this.previousQueue.size() > PREVIOUS_QUEUE_CHECK_SIZE) {
                this.previousQueue.remove();
            }
            DataValueObject dvo = this.previousQueue.peek();

            if (dvo != null) {
                return (P) dvo.getValue();
            }
        }
        return null;
    }

    /**
     * ݂̗vfԂ܂B<br>
     * <p>
     * null̏ꍇ݂̗͌vf݂ȂƂ܂B<br>
     * |C^͈ړ܂B
     * </p>
     * <p>
     * <b>{\bh̓}`XbhZ[tł܂B</b>
     * </p>
     * @return &lt;P&gt;
     * @see jp.terasoluna.fw.collector.Collector#getCurrent()
     */
    @SuppressWarnings("unchecked")
    public P getCurrent() {
        // sJnî݁j
        execute();

        if (this.currentQueue != null && this.currentQueue.size() > 0) {
            while (this.currentQueue.size() > CURRENT_QUEUE_CHECK_SIZE) {
                this.currentQueue.remove();
            }
            DataValueObject dvo = this.currentQueue.peek();

            if (dvo != null) {
                return (P) dvo.getValue();
            }
        }
        return null;
    }

    /**
     * ̃Xg[āAɊ֘A邷ׂẴVXe\[X܂B<br>
     * Xg[łɕĂꍇ́Ã\bhĂяoĂ̌ʂ܂B
     * <p>
     * <b>{\bh̓}`XbhZ[tł܂B</b>
     * </p>
     * @throws IOException o̓G[ꍇ
     * @see java.io.Closeable#close()
     */
    public void close() {
        if (!isFinish()) {
            if (this.fo != null) {
                this.fo.cancel(true);
            }
        }
    }

    /**
     * ɂȂRNVAqɂčŌɕԂꂽvf폜܂ (Cӂ̃Iy[V)B<br>
     * ̃\bh́Anext ̌ĂяoƂ 1 񂾂ĂяoƂł܂Bq̓́A<br>
     * JԂ̃\bȟĂяoȊO̕@ŎsĂƂɊɂȂRNVύXꂽꍇ͕ۏ؂܂B
     * <p>
     * <b>{\bh̓}`XbhZ[tł܂B</b>
     * </p>
     * @throws UnsupportedOperationException Iterator  remove Iy[VT|[gȂꍇ
     * @throws IllegalStateException next \bh܂ĂяoĂȂꍇA܂ next \bh̍Ō̌ĂяôƂ remove \bhłɌĂяoĂꍇ
     * @see java.util.Iterator#remove()
     */
    public void remove() {
        // ǂݎ̂Ă
        next();
    }

    /*
     * (non-Javadoc)
     * @see java.lang.Object#finalize()
     */
    @Override
    protected void finalize() throws Throwable {
        if (verboseLog.get() && LOGGER.isTraceEnabled()) {
            LOGGER.trace(LogId.TAL041011, Thread.currentThread().getName());
        }
        if (!isFinish()) {
            if (LOGGER.isWarnEnabled()) {
                LOGGER.warn(LogId.WAL041005, Thread.currentThread().getName());
            }
        }
        super.finalize();

        // bN댯̂close()̌Ăяo߂
    }

    /*
     * (non-Javadoc)
     * @see java.lang.Iterable#iterator()
     */
    public Iterator<P> iterator() {
        return this;
    }

    /**
     * getExecutor\bh.
     * @return ExecutorService
     */
    protected ExecutorService getExecutor() {
        // VGO[L[^ԋp
        return Executors.newSingleThreadExecutor(new CollectorThreadFactory());
    }

    /**
     * L[쐬
     * @return
     */
    protected Queue<DataValueObject> createQueue() {
        if (this.currentQueue == null) {
            // currentL[
            this.currentQueue = createCurrentQueue();
        }
        if (this.previousQueue == null) {
            // previousL[
            this.previousQueue = createPreviousQueue();
        }
        return new ArrayBlockingQueue<DataValueObject>(this.queueSize);
    }

    /**
     * currentL[쐬
     * @return Queue&lt;DataValueObject&gt;
     */
    protected Queue<DataValueObject> createCurrentQueue() {
        return new ConcurrentLinkedQueue<DataValueObject>();
    }

    /**
     * previousL[쐬
     * @return Queue&lt;DataValueObject&gt;
     */
    protected Queue<DataValueObject> createPreviousQueue() {
        return new ConcurrentLinkedQueue<DataValueObject>();
    }

    /**
     * L[擾B
     * @return Queue&lt;DataValueObject&gt;
     */
    protected Queue<DataValueObject> getQueue() {
        return this.queue;
    }

    /**
     * L[TCYw肷B<br>
     * @param queueSize int
     */
    protected void setQueueSize(int queueSize) {
        this.queueSize = queueSize;
    }

    /**
     * X[v(msec)<br>
     * @return X[v(msec)
     */
    protected int getSleepWait() {
        return sleepWait;
    }

    /**
     * X[v(msec)<br>
     * @param sleepWait X[v(msec)
     */
    protected void setSleepWait(int sleepWait) {
        this.sleepWait = sleepWait;
    }

    /**
     * L[Ƀf[^ǉB
     * @param dataValueObject DataValueObject
     * @throws InterruptedException
     */
    protected void addQueue(DataValueObject dataValueObject)
                                                            throws InterruptedException {
        addQueue(dataValueObject, false);
    }

    /**
     * L[Ƀf[^ǉB
     * @param dataValueObject DataValueObject
     * @param force boolean L[COtO
     * @throws InterruptedException
     */
    protected void addQueue(DataValueObject dataValueObject, boolean force)
                                                                           throws InterruptedException {
        if (force && this.queue != null) {
            this.queue.offer(dataValueObject);
            return;
        }

        boolean finish = isFinish();

        if (!finish) {
            // ̓`FbN
            ValidateErrorStatus vs = null;
            if (this.validator != null) {
                try {
                    vs = validate(dataValueObject);
                } catch (Throwable e) {
                    // O1L[ɂ߂
                    if (this.queue instanceof BlockingQueue) {
                        ((BlockingQueue<DataValueObject>) this.queue)
                                .put(new DataValueObject(e));
                    } else {
                        this.queue.add(new DataValueObject(e));
                    }
                    return;
                }
            }

            if (vs == null || ValidateErrorStatus.CONTINUE.equals(vs)) {
                // 擾f[^1L[ɂ߂
                if (this.queue instanceof BlockingQueue) {
                    ((BlockingQueue<DataValueObject>) this.queue)
                            .put(dataValueObject);
                } else {
                    this.queue.add(dataValueObject);
                }
            } else if (ValidateErrorStatus.END.equals(vs)) {
                DataValueObject errorStop = new DataValueObject(vs);
                if (this.queue instanceof BlockingQueue) {
                    ((BlockingQueue<DataValueObject>) this.queue)
                            .put(errorStop);
                } else {
                    this.queue.add(errorStop);
                }
                // ~iȍ~̃L[CO~j
                setFinish(true);
            } else if (ValidateErrorStatus.SKIP.equals(vs)) {
                DataValueObject skip = new DataValueObject(vs);
                if (this.queue instanceof BlockingQueue) {
                    ((BlockingQueue<DataValueObject>) this.queue).put(skip);
                } else {
                    this.queue.add(skip);
                }
            }
        } else {
            if (LOGGER.isTraceEnabled()) {
                long dc = -1;
                if (dataValueObject != null) {
                    dc = dataValueObject.getDataCount();
                }
                LOGGER.trace(LogId.TAL041013, finish, "", dc);
            }
            throw new InterruptedException(
                    "The stop demand of the thread is carried out.");
        }
    }

    /**
     * ̓`FbNs.<br>
     * @param dataValueObject DataValueObject
     * @return ValidateStatus
     */
    protected ValidateErrorStatus validate(DataValueObject dataValueObject) {
        ValidateErrorStatus vs = ValidateErrorStatus.CONTINUE;

        if (this.validator != null) {
            Class<?> clazz = null;
            String objectName = null;
            Errors errors = null;

            // ̓IuWFNg̃NX^擾
            if (dataValueObject != null && dataValueObject.getValue() != null) {
                clazz = dataValueObject.getValue().getClass();

                if (clazz != null) {
                    objectName = clazz.getSimpleName();
                    if (objectName != null) {
                        objectName = Introspector.decapitalize(objectName);
                        // ErrorsIuWFNg
                        errors = new BindException(dataValueObject.getValue(),
                                objectName);
                    }
                }
            }

            if (clazz != null && errors != null
                    && this.validator.supports(clazz)) {
                // ̓`FbN
                this.validator.validate(dataValueObject.getValue(), errors);

                if (errors.hasErrors()) {
                    vs = handleValidationError(dataValueObject, errors);
                }
            }
        }

        return vs;
    }

    /**
     * ̓`FbNG[̏.<br>
     * @param dataValueObject DataValueObject
     * @param errors Errors
     * @return ValidateErrorStatus
     */
    protected ValidateErrorStatus handleValidationError(
            DataValueObject dataValueObject, Errors errors) {

        if (this.validationErrorHandler != null) {
            return this.validationErrorHandler.handleValidationError(
                    dataValueObject, errors);
        }

        return ValidateErrorStatus.SKIP;
    }

    /**
     * Ȍ
     * @param dataValueObject DataValueObject
     * @return CollectorExceptionHandlerStatus
     */
    protected CollectorExceptionHandlerStatus handleException(
            DataValueObject dataValueObject) {
        CollectorExceptionHandlerStatus result = null;
        if (this.exceptionHandler != null) {
            result = this.exceptionHandler.handleException(dataValueObject);
        }
        return result;
    }

    /**
     * ItȌԂmFB
     * @return boolean
     */
    protected boolean isFinish() {
        boolean finish = this.finish;
        AbstractCollector<?> localChild = this.child;
        Future<?> future = this.fo;

        if (future != null) {
            boolean done = future.isDone();

            if (localChild != null) {
                // qXbh̏ItOQƂ
                if (localChild.isFinish()) {
                    finish = localChild.isFinish();
                }
            }
            return finish || done;
        }

        if (localChild != null) {
            // qXbh̏ItOQƂ
            if (localChild.isFinish()) {
                finish = localChild.isFinish();
            }
        }
        return finish;
    }

    /**
     * ItOݒ肷B
     */
    protected void setFinish() {
        if (verboseLog.get() && LOGGER.isTraceEnabled()) {
            LOGGER.trace(LogId.TAL041012, Thread.currentThread().getName());
        }
        setFinish(true);

        // ItOL[ɂ߂
        try {
            addQueue(new DataValueObject(CollectorStatus.END), true);
        } catch (InterruptedException ie) {
            if (LOGGER.isTraceEnabled()) {
                LOGGER.trace(LogId.TAL041012, ie, Thread.currentThread()
                        .getName());
            }
        }
    }

    /**
     * ItOݒ肷B
     * @param finish
     */
    protected void setFinish(boolean finish) {
        this.finish = finish;
    }

    /**
     * CollectorN[YB<br>
     * <p>
     * ɓnꂽcollectornullłȂ΃N[YB<br>
     * ܂AN[YۂIOOꍇ͖B<br>
     * </p>
     * @param collector Collector
     */
    public static void closeQuietly(Collector<?> collector) {
        try {
            if (collector != null) {
                collector.close();
            }
        } catch (IOException e) {
            // ȂɂȂ
        }
    }

    /**
     * 璷Oo̓tOݒ肷B
     * @param verbose 璷Oo̓tO
     */
    public static void setVerbose(boolean verbose) {
        verboseLog.set(verbose);
    }
}
