/*
 * Copyright (c) 2011 NTT DATA Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package jp.terasoluna.fw.batch.dao.support;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Queue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import jp.terasoluna.fw.dao.SqlHolder;
import jp.terasoluna.fw.dao.UpdateDAO;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * ob`XVT|[gNX<br>
 * <p>
 * {NX𗘗pƂUpdateDAÕob`XV̎sSqlIDɐ񂳂ꂽԂōsƂłB<br>
 * SqlIDŃ\[g邱ƂŃvyA[hXe[ggLbVpꐫ\オ߂B
 * </p>
 * @see UpdateDAO
 */
public class BatchUpdateSupportImpl implements BatchUpdateSupport {

    /**
     * O.
     */
    private static Log logger = LogFactory.getLog(BatchUpdateExecutor.class);

    /** UpdateDAO */
    protected UpdateDAO updateDAO = null;

    /** SqlID\[g鎞ɎgpComparator */
    protected Comparator<String> comparator = null;

    /** ob`sSQLێ. */
    protected final ConcurrentHashMap<String, Queue<SqlHolder>> batchSqlsMap = new ConcurrentHashMap<String, Queue<SqlHolder>>();

    /** ob`sSQLo^ */
    protected long count = 0;

    /** bNIuWFNg */
    protected ReentrantReadWriteLock lock = new ReentrantReadWriteLock();

    /**
     * obWXVT|[gNXRXgN^.
     */
    public BatchUpdateSupportImpl() {
        // Ȃ
    }

    /**
     * obWXVT|[gNXRXgN^.
     * @param updateDAO UpdateDAO
     */
    public BatchUpdateSupportImpl(UpdateDAO updateDAO) {
        this(updateDAO, null);
    }

    /**
     * obWXVT|[gNXRXgN^.
     * @param updateDAO UpdateDAO
     * @param comparator Comparator&lt;String&gt;
     */
    public BatchUpdateSupportImpl(UpdateDAO updateDAO,
            Comparator<String> comparator) {
        this.updateDAO = updateDAO;
        this.comparator = comparator;

        if (this.updateDAO == null && logger.isWarnEnabled()) {
            StringBuilder sb = new StringBuilder();
            sb.append("UpdateDAO is null.");
            logger.warn(sb.toString());
        }
    }

    /*
     * (non-Javadoc)
     * @see jp.terasoluna.fw.batch.dao.support.BatchUpdateSupportIf#addBatch(java.lang.String, java.lang.Object)
     */
    public void addBatch(final String sqlID, final Object bindParams) {
        if (sqlID == null || sqlID.length() == 0) {
            if (logger.isWarnEnabled()) {
                StringBuilder sb = new StringBuilder();
                sb.append("SqlID is null or empty.");
                logger.warn(sb.toString());
            }
            return;
        }
        try {
            this.lock.readLock().lock();

            Queue<SqlHolder> sqlQueue = this.batchSqlsMap.get(sqlID);

            if (sqlQueue == null) {
                try {
                    this.lock.readLock().unlock();
                    this.lock.writeLock().lock();

                    sqlQueue = new ConcurrentLinkedQueue<SqlHolder>();
                    this.batchSqlsMap.put(sqlID, sqlQueue);
                } finally {
                    this.lock.readLock().lock();
                    this.lock.writeLock().unlock();
                }

                if (sqlQueue != null) {
                    sqlQueue.add(new SqlHolder(sqlID, bindParams));
                    this.count++;
                }
            } else {
                sqlQueue.add(new SqlHolder(sqlID, bindParams));
                this.count++;
            }
        } finally {
            this.lock.readLock().unlock();
        }
    }

    /*
     * (non-Javadoc)
     * @see jp.terasoluna.fw.batch.dao.support.BatchUpdateSupport#executeBatch()
     */
    public int executeBatch() {
        return executeBatch(this.updateDAO, this.comparator);
    }

    /*
     * (non-Javadoc)
     * @see jp.terasoluna.fw.batch.dao.support.BatchUpdateSupport#executeBatch(jp.terasoluna.fw.dao.UpdateDAO)
     */
    public int executeBatch(UpdateDAO updateDAO) {
        return executeBatch(updateDAO, this.comparator);
    }

    /*
     * (non-Javadoc)
     * @see jp.terasoluna.fw.batch.dao.support.BatchUpdateSupport#executeBatch(jp.terasoluna.fw.dao.UpdateDAO,
     * java.util.Comparator)
     */
    public int executeBatch(UpdateDAO updateDAO, Comparator<String> comparator) {
        int result = -1000;

        if (updateDAO == null) {
            if (logger.isWarnEnabled()) {
                StringBuilder sb = new StringBuilder();
                sb.append("UpdateDAO is null.");
                logger.warn(sb.toString());
            }
            return ERROR_UPDATE_DAO_IS_NULL;
        }

        List<SqlHolder> sqlHolderList = new ArrayList<SqlHolder>();

        try {
            this.lock.writeLock().lock();

            // SQL-IDŃ\[g
            List<String> keyList = new ArrayList<String>(this.batchSqlsMap
                    .keySet());
            if (comparator != null) {
                Collections.sort(keyList, comparator);
            } else {
                Collections.sort(keyList);
            }

            for (String key : keyList) {
                sqlHolderList.addAll(this.batchSqlsMap.get(key));
            }

            // ob`XVs
            result = updateDAO.executeBatch(sqlHolderList);

            this.batchSqlsMap.clear();
            this.count = 0;
        } finally {
            this.lock.writeLock().unlock();
        }

        return result;
    }

    /*
     * (non-Javadoc)
     * @see jp.terasoluna.fw.batch.dao.support.BatchUpdateSupportIf#clear()
     */
    public void clear() {
        try {
            this.lock.writeLock().lock();

            this.batchSqlsMap.clear();
            this.count = 0;
        } finally {
            this.lock.writeLock().unlock();
        }
    }

    /*
     * (non-Javadoc)
     * @see jp.terasoluna.fw.batch.dao.support.BatchUpdateSupport#size()
     */
    public long size() {
        return this.count;
    }

    /*
     * (non-Javadoc)
     * @see jp.terasoluna.fw.batch.dao.support.BatchUpdateSupport#getSqlHolderList()
     */
    public List<SqlHolder> getSqlHolderList() {
        return getSqlHolderList(null);
    }

    /*
     * (non-Javadoc)
     * @see jp.terasoluna.fw.batch.dao.support.BatchUpdateSupport#getSqlHolderList(java.util.Comparator)
     */
    public List<SqlHolder> getSqlHolderList(Comparator<String> comparator) {
        List<SqlHolder> sqlHolderList = new ArrayList<SqlHolder>();

        try {
            this.lock.writeLock().lock();

            // SQL-IDŃ\[g
            List<String> keyList = new ArrayList<String>(this.batchSqlsMap
                    .keySet());

            if (comparator != null) {
                Collections.sort(keyList, comparator);
            } else {
                Collections.sort(keyList);
            }

            for (String key : keyList) {
                sqlHolderList.addAll(this.batchSqlsMap.get(key));
            }

        } finally {
            this.lock.writeLock().unlock();
        }

        return sqlHolderList;
    }

}
