/*
 * Copyright (c) 2011 NTT DATA Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package jp.terasoluna.fw.batch.executor;

import java.io.File;
import java.util.List;

import jp.terasoluna.fw.batch.executor.concurrent.BatchServant;
import jp.terasoluna.fw.batch.executor.vo.BatchJobListResult;
import jp.terasoluna.fw.batch.util.JobUtil;
import jp.terasoluna.fw.dao.QueryDAO;
import jp.terasoluna.fw.dao.UpdateDAO;
import jp.terasoluna.fw.util.PropertyUtil;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.context.ApplicationContext;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.transaction.PlatformTransactionManager;

/**
 * 񓯊ob`GO[L[^B<br>
 * <br>
 * 풓vZXƂċNAWuǗe[uɓo^ꂽWu擾AWu̎sBatchServantNXɈڏB<br>
 * ܂WuǗe[uɃWusʂXVB<br>
 * @see jp.terasoluna.fw.batch.executor.AbstractJobBatchExecutor
 */
public class AsyncBatchExecutor extends AbstractJobBatchExecutor {

    /**
     * O.
     */
    private static Log logging = LogFactory.getLog(AsyncBatchExecutor.class);

    /**
     * ^XNGO[L[^Bean
     */
    private static final String BATCH_TASK_EXECUTOR = "batchTaskExecutor.default";

    /**
     * XbhspBatchServantNXBean
     */
    private static final String BATCH_TASK_SERVANT = "batchTaskExecutor.batchServant";

    /**
     * vZXIR[hij
     */
    private static final int PROCESS_END_STATUS_NORMAL = 0;

    /**
     * vZXIR[hiُj
     */
    private static final int PROCESS_END_STATUS_FAILURE = 255;

    /**
     * RXgN^
     */
    protected AsyncBatchExecutor() {
        super();
    }

    /**
     * C\bh.
     * @param args
     */
    public static void main(String[] args) {
        int status = PROCESS_END_STATUS_FAILURE;
        Throwable throwable = null;
        String jobAppCd = null;
        String batchTaskExecutorName = null;
        String batchTaskServantName = null;
        ThreadPoolTaskExecutor taskExecutor = null;

        if (logging.isInfoEnabled()) {
            logging.info("AsyncBatchExecutor START");
        }

        // 1WuƖR[h擾
        if (args.length > 0) {
            jobAppCd = args[0];
        }

        // Ɏw肳ĂȂꍇ͊ϐWuƖR[h擾
        if (jobAppCd == null || jobAppCd.length() == 0) {
            jobAppCd = JobUtil.getenv(ENV_JOB_APP_CD);
            if (jobAppCd != null && jobAppCd.length() == 0) {
                jobAppCd = null;
            }
        }

        if (logging.isInfoEnabled()) {
            StringBuilder sb = new StringBuilder();
            sb.append("jobAppCd:[");
            sb.append(jobAppCd == null ? "" : jobAppCd);
            sb.append("]");
            logging.info(sb.toString());
        }

        // GO[L[^
        AsyncBatchExecutor executor = new AsyncBatchExecutor();

        // QueryDAO擾
        QueryDAO queryDAO = executor.getSysQueryDAO();
        if (queryDAO == null) {
            if (logging.isInfoEnabled()) {
                logging.info("QueryDAO is null.");
            }
            System.exit(status);
            return;
        }

        // UpdateDAO擾
        UpdateDAO updateDAO = executor.getSysUpdateDAO();
        if (updateDAO == null) {
            if (logging.isInfoEnabled()) {
                logging.info("UpdateDAO is null.");
            }
            System.exit(status);
            return;
        }

        // PlatformTransactionManager擾
        PlatformTransactionManager transactionManager = executor
                .getSysTransactionManager();
        if (transactionManager == null) {
            if (logging.isInfoEnabled()) {
                logging.info("PlatformTransactionManager is null.");
            }
            System.exit(status);
            return;
        }

        // ^XNGO[L[^BatchServantNXBean擾
        batchTaskExecutorName = PropertyUtil.getProperty(BATCH_TASK_EXECUTOR);
        batchTaskServantName = PropertyUtil.getProperty(BATCH_TASK_SERVANT);

        // ^XNGO[L[^擾
        ApplicationContext ctx = executor.getDefaultApplicationContext();
        if (ctx != null) {
            if (ctx.containsBean(batchTaskExecutorName)) {
                Object batchTaskExecutorObj = null;
                try {
                    batchTaskExecutorObj = ctx.getBean(batchTaskExecutorName,
                            ThreadPoolTaskExecutor.class);
                } catch (Throwable e) {
                    if (logging.isErrorEnabled()) {
                        StringBuilder sb = new StringBuilder();
                        sb.append("ThreadPoolTaskExecutor bean not found.");
                        sb.append(" beanName:[");
                        sb.append(batchTaskExecutorName);
                        sb.append("]");
                        logging.error(sb.toString(), e);
                    }
                }
                if (batchTaskExecutorObj instanceof ThreadPoolTaskExecutor) {
                    taskExecutor = (ThreadPoolTaskExecutor) batchTaskExecutorObj;
                }
            }
        }
        if (taskExecutor == null) {
            if (logging.isInfoEnabled()) {
                logging.info("BatchTaskExecutor not found.");
            }
            System.exit(status);
            return;
        }

        try {
            do {
                // WuXg1̂ݎ擾isXbhɋ󂫂ꍇ̂ݎ擾j
                List<BatchJobListResult> jobList = null;
                if (checkTaskQueue(taskExecutor)) {
                    if (jobAppCd == null) {
                        jobList = JobUtil.selectJobList(queryDAO, 0, 1);
                    } else {
                        jobList = JobUtil.selectJobList(jobAppCd, queryDAO, 0,
                                1);
                    }
                }

                if (jobList != null && !jobList.isEmpty()) {
                    // XĝPڂ̂ݎ擾
                    BatchJobListResult batchJobListResult = jobList.get(0);

                    if (batchJobListResult != null) {
                        if (logging.isDebugEnabled()) {
                            StringBuilder sb = new StringBuilder();
                            sb.append("jobSequenceId:[");
                            sb.append(batchJobListResult.getJobSequenceId());
                            sb.append("]");
                            logging.debug(sb.toString());
                        }

                        if (logging.isDebugEnabled()) {
                            // Xbhv[^XNGO[L[^̃Xe[^XfobOOɏo
                            logOutputTaskExecutor(logging, taskExecutor);
                        }

                        // sXbhɋ󂫂΃ob`s
                        if (checkTaskQueue(taskExecutor)) {
                            BatchServant job = null;

                            // ReLXgBatchServantCX^X擾
                            if (ctx != null) {
                                try {
                                    job = (BatchServant) ctx.getBean(
                                            batchTaskServantName,
                                            BatchServant.class);
                                } catch (Throwable e) {
                                    if (logging.isErrorEnabled()) {
                                        StringBuilder sb = new StringBuilder();
                                        sb.append("BatchServant not found.");
                                        sb.append(" bean name:[");
                                        sb.append(batchTaskServantName);
                                        sb.append("]");
                                        logging.error(sb.toString(), e);
                                        // [v𔲂ďI
                                        break;
                                    }
                                }
                            }

                            if (job == null) {
                                if (logging.isErrorEnabled()) {
                                    StringBuilder sb = new StringBuilder();
                                    sb.append("BatchServant not found.");
                                    sb.append(" bean name:[");
                                    sb.append(batchTaskServantName);
                                    sb.append("]");
                                    logging.error(sb.toString());
                                    // [v𔲂ďI
                                    break;
                                }
                            } else {
                                // WuXe[^XݒiJnj
                                boolean st = executor
                                        .startBatchStatus(batchJobListResult
                                                .getJobSequenceId(), queryDAO,
                                                updateDAO, transactionManager);
                                if (!st) {
                                    if (logging.isInfoEnabled()) {
                                        StringBuilder sb = new StringBuilder();
                                        sb.append("Job status update error.");
                                        sb.append("(JOB_SEQ_CD:");
                                        sb.append(batchJobListResult
                                                .getJobSequenceId());
                                        sb.append(")");
                                        logging.info(sb.toString());
                                    }
                                } else {
                                    // BatchServantɃWuV[PXR[hݒ
                                    job.setJobSequenceId(batchJobListResult
                                            .getJobSequenceId());
                                    // Wus
                                    taskExecutor.execute(job);
                                }
                            }
                        }
                    }
                }

                // ItOt@C`FbN
                if (checkEndFile(executor.getExecutorEndMonitoringFile())) {
                    // ItOt@C
                    if (logging.isInfoEnabled()) {
                        StringBuilder sb = new StringBuilder();
                        sb.append("End flag file discovery.");
                        logging.info(sb.toString());
                    }
                    break;
                }

                // WuXĝƂ or sXbhɋ󂫂ꍇ ͎w莞ԃEFCg
                if (jobList == null || jobList.size() == 0) {
                    // Wu̎sԊui~bj
                    if (executor.getJobIntervalTime() >= 0) {
                        try {
                            Thread.sleep(executor.getJobIntervalTime());
                        } catch (InterruptedException e) {
                            // 荞ݎMŏI
                            if (logging.isInfoEnabled()) {
                                StringBuilder sb = new StringBuilder();
                                sb.append("Interrupt is received.");
                                sb.append(" exception message:[");
                                sb.append(e.getMessage());
                                sb.append("]");
                                logging.info(sb.toString());
                            }
                            break;
                        }
                    }
                }

                // 풓pɃ[v
            } while (true);
        } catch (Throwable e) {
            // UۑĂ
            throwable = e;
        } finally {
            if (logging.isDebugEnabled()) {
                logging.debug("task wait.");
            }

            // IɃ^XN͂܂ő҂
            taskExecutor.setWaitForTasksToCompleteOnShutdown(true);

            if (logging.isDebugEnabled()) {
                logging.debug("task shutdown.");
            }

            taskExecutor.shutdown();

            if (logging.isDebugEnabled()) {
                logging.debug("task shutdown after.");
            }

            while (taskExecutor.getActiveCount() != 0) {
                try {
                    if (logging.isDebugEnabled()) {
                        StringBuilder sb = new StringBuilder();
                        sb.append("AsyncBatchExecutor WAIT ENDS activeCount:[");
                        sb.append(taskExecutor.getActiveCount());
                        sb.append("]");
                        logging.debug(sb.toString());
                    }
                    Thread.sleep(executor
                            .getExecutorJobTerminateWaitIntervalTime());
                } catch (InterruptedException e) {
                }
            }
        }

        if (throwable != null) {
            if (logging.isErrorEnabled()) {
                logging.error("AsyncBatchExecutor ABNORMAL END", throwable);
            }
        } else {
            if (logging.isInfoEnabled()) {
                logging.info("AsyncBatchExecutor END");
            }
            status = PROCESS_END_STATUS_NORMAL;
        }

        System.exit(status);
        return;
    }

    /**
     * ^XNGO[L[^ɋ󂫂邩`FbN
     * @param taskExecutor ^XNGO[L[^
     * @return 󂫂̗L
     */
    private static boolean checkTaskQueue(ThreadPoolTaskExecutor taskExecutor) {
        // Xbh̋󂫃`FbN
        if (taskExecutor.getActiveCount() < taskExecutor.getMaxPoolSize()) {
            // 󂫂
            return true;
        }
        // L[̋󂫃`FbN
        if (taskExecutor.getThreadPoolExecutor().getQueue().remainingCapacity() > 0) {
            // 󂫂
            return true;
        }
        // 󂫂Ȃ
        return false;
    }

    /**
     * ItOt@C`FbN
     * @param endFilePath ItOt@CpX
     * @return ItOt@C`FbN
     */
    private static boolean checkEndFile(String endFilePath) {
        if (endFilePath != null && endFilePath.length() != 0) {
            File endFile = new File(endFilePath);
            return endFile.exists();
        }
        return false;
    }

    /**
     * Xbhv[^XNGO[L[^̃Xe[^XfobOOɏo
     * @param log Log
     * @param taskExec ThreadPoolTaskExecutor
     */
    private static void logOutputTaskExecutor(Log log,
            ThreadPoolTaskExecutor taskExec) {
        StringBuilder sb = new StringBuilder();
        sb.append("activeCount:[");
        sb.append(taskExec.getActiveCount());
        sb.append("]");
        sb.append(" corePoolSize:[");
        sb.append(taskExec.getCorePoolSize());
        sb.append("]");
        sb.append(" maxPoolSize:[");
        sb.append(taskExec.getMaxPoolSize());
        sb.append("]");
        sb.append(" poolSize:[");
        sb.append(taskExec.getPoolSize());
        sb.append("]");
        sb.append(" activeCount:[");
        sb.append(taskExec.getThreadPoolExecutor().getActiveCount());
        sb.append("]");
        sb.append(" taskCount:[");
        sb.append(taskExec.getThreadPoolExecutor().getTaskCount());
        sb.append("]");
        sb.append(" queueSize:[");
        sb.append(taskExec.getThreadPoolExecutor().getQueue().size());
        sb.append("]");
        sb.append(" remainingCapacity:[");
        sb.append(taskExec.getThreadPoolExecutor().getQueue()
                .remainingCapacity());
        sb.append("]");
        logging.debug(sb.toString());
    }
}
