package jp.terasoluna.toursample.server.interfaces.ws.exception.handler.impl;

import java.util.ArrayList;
import java.util.List;

import jp.terasoluna.fw.exception.SystemException;
import jp.terasoluna.toursample.server.application.exception.ApplicationException;
import jp.terasoluna.toursample.server.application.message.MessageAccessor;
import jp.terasoluna.toursample.server.interfaces.ws.exception.ErrorMessage;
import jp.terasoluna.toursample.server.interfaces.ws.exception.ErrorMessages;
import jp.terasoluna.toursample.server.interfaces.ws.exception.ErrorType;
import jp.terasoluna.toursample.server.interfaces.ws.exception.SampleFaultBean;
import jp.terasoluna.toursample.server.interfaces.ws.exception.SampleSoapFaultException;
import jp.terasoluna.toursample.server.interfaces.ws.exception.handler.SampleSoapFaultExceptionHandler;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.context.MessageSourceResolvable;
import org.springframework.validation.BindException;
import org.springframework.validation.ObjectError;
/**
*
* 例外を集約的にハンドリングしSoapFaultを返却するインタフェース
*
*/
public class SampleSoapFaultExceptionHandlerImpl implements
        SampleSoapFaultExceptionHandler {
    private static final String COMMA = ",";
    private static Log log = LogFactory
            .getLog(SampleSoapFaultExceptionHandlerImpl.class);
    private MessageAccessor messageAccessor;

    public void setMessageAccessor(MessageAccessor messageAccessor) {
        this.messageAccessor = messageAccessor;
    }

    @Override
    public void handleException(Exception e) throws SampleSoapFaultException {
        if (e instanceof BindException) {
            // 入力チェックエラー
            throw createSoapFaultValidationException((BindException) e);
        } else if (e instanceof ApplicationException) {
            // 業務エラー
            throw createSoapFaultAppException((ApplicationException) e);
        } else if (e instanceof SystemException) {
            // システムエラー
            throw createSoapFaultSysException((SystemException) e);
        } else {
            // その他予期せぬシステムエラー
            if (log.isErrorEnabled()) {
                log.error(e.getMessage(), e);
            }
            throw new SampleSoapFaultException(messageAccessor
                    .getMessage("ERROR_COMMON_0001"));
        }
    }

    /**
     * 入力チェックエラー用のSOAPFaultを作成
     *
     */
    private SampleSoapFaultException createSoapFaultValidationException(
            BindException e) {
        SampleFaultBean faultBean = new SampleFaultBean();
        ErrorMessages errorMessages = new ErrorMessages();
        List<ErrorMessage> messages = new ArrayList<ErrorMessage>();

        List<ObjectError> errors = e.getAllErrors();
        for (ObjectError error : errors) {
            ErrorMessage errorMessage = new ErrorMessage();
            // errorMessage.setErrorId(ErrorType.VALIDATION_FAILURE);
            errorMessage.setErrorId(error.getCode());
            errorMessage.setMessage(messageAccessor.getMessage(error));
            errorMessage
                    .setArguments(formatToCommaSepratedStringForValidationError(error
                            .getArguments()));
            messages.add(errorMessage);
        }
        errorMessages.setErrorMessage(messages);
        // Terasoluna Client FW for .NET 3.0では、ErrorTypeが
        // 「ValidationFailure」の場合、入力チェック例外であることを判定する
        faultBean.setErrorType(ErrorType.VALIDATION_FAILURE);
        faultBean.setErrorMessages(errorMessages);
        return new SampleSoapFaultException(messageAccessor
                .getMessage("ERROR_COMMON_0002"), faultBean);
    }

    /**
     * 業務エラー用のSOAPFaultを作成
     *
     */
    private SampleSoapFaultException createSoapFaultAppException(
            ApplicationException e) {
        SampleFaultBean faultBean = new SampleFaultBean();
        ErrorMessages errorMessages = new ErrorMessages();
        List<ErrorMessage> messages = new ArrayList<ErrorMessage>();

        ErrorMessage errorMessage = new ErrorMessage();
        errorMessage.setErrorId(e.getErrorCode());
        errorMessage.setMessage(e.getMessage());
        errorMessage.setArguments(formatToCommaSeparatedString(e.getOptions()));
        messages.add(errorMessage);
        errorMessages.setErrorMessage(messages);
        // Terasoluna Client FW for .NET 3.0では、ErrorTypeが
        // 「BizLogicFailure」の場合、業務例外であることを判定する
        faultBean.setErrorType(ErrorType.BIZ_LOGIC_FAILURE);
        // 業務エラーを分類したければ分類可能
        faultBean.setBizLogicErrorType(messageAccessor
                .getMessage("ERROR_COMMON_APP_ERROR"));
        faultBean.setErrorMessages(errorMessages);
        return new SampleSoapFaultException(messageAccessor
                .getMessage("ERROR_COMMON_0003"), faultBean);
    }

    /**
     * システムエラー用のSOAPFaultを作成
     *
     */
    private SampleSoapFaultException createSoapFaultSysException(
            SystemException e) {
        SampleFaultBean faultBean = new SampleFaultBean();
        ErrorMessages errorMessages = new ErrorMessages();
        List<ErrorMessage> messages = new ArrayList<ErrorMessage>();

        ErrorMessage errorMessage = new ErrorMessage();
        errorMessage.setErrorId(e.getErrorCode());
        errorMessage.setMessage(e.getMessage());
        errorMessage.setArguments(formatToCommaSeparatedString(e.getOptions()));
        messages.add(errorMessage);
        errorMessages.setErrorMessage(messages);
        faultBean.setErrorMessages(errorMessages);
        return new SampleSoapFaultException(messageAccessor
                .getMessage("ERROR_COMMON_0004"), faultBean);
    }

    /**
     * 入力チェックエラー時の引数をリストをカンマ区切り形式の文字列に変換
     */
    private String formatToCommaSepratedStringForValidationError(Object[] args) {
        if (args == null) {
            return null;
        }
        ArrayList<String> list = new ArrayList<String>();
        for (Object arg : args) {
            if (arg instanceof MessageSourceResolvable) {
                list.add(((MessageSourceResolvable) arg).getDefaultMessage());
            }
        }
        return formatToCommaSeparatedString(list
                .toArray(new String[list.size()]));

    }

    /**
     * リストをカンマ区切り形式の文字列に変換
     */
    private String formatToCommaSeparatedString(String[] options) {

        return StringUtils.join(options, COMMA);
    }

}
