// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Text;
using System.Collections.Generic;
using TERASOLUNA.Fw.Client.Forms;
using System.Windows.Forms;
using TERASOLUNA.Fw.Common;
using System.Configuration;
using TERASOLUNA.Fw.Client.Conversion;
using System.Data;
using System.Reflection;
using TERASOLUNA.Fw.Client.Configuration.View;
using TERASOLUNA.Fw.Client.Configuration;
using TERASOLUNA.Fw.ClientUT.Configuration;
using System.Collections;
using TERASOLUNA.Fw.Utlib.Logging;
using TERASOLUNA.Fw.Common.Logging;
using System.ComponentModel;

namespace TERASOLUNA.Fw.ClientUT.Forms
{
    ///<summary>
    /// FormForwarder のテスト クラスです。
    ///</summary>

    [TestClass()]
    public class FormForwarderTest
    {
        private TestContext testContextInstance;

        private bool _formClosed1Called = false;
        private bool _formClosed2Called = false;
        private bool _formClosed3Called = false;
        private bool _formClosing1Called = false;
        private bool _formClosing2Called = false;
        private bool _formClosing3Called = false;
        private bool _oldInit;
        private string _oldSetting;

        private static string TEST_CONVERTER_TYPE =
            "TERASOLUNA.Fw.ClientUT.Forms.FormForwarder_ConverterStub01, TERASOLUNA.Fw.ClientUT";
        private static string VIEW_CONFIG_FILE = "FormForwarder_ViewConfigSection01.config";

        /// <summary>
        ///現在のテストの実行についての情報および機能を
        ///提供するテスト コンテキストを取得または設定します。
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }
        #region 追加のテスト属性
        // 
        //テストを作成するときに、次の追加属性を使用することができます:
        //
        //クラスの最初のテストを実行する前にコードを実行するには、ClassInitialize を使用
        //
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //クラスのすべてのテストを実行した後にコードを実行するには、ClassCleanup を使用
        //
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //各テストを実行する前にコードを実行するには、TestInitialize を使用
        //
        [TestInitialize()]
        public void MyTestInitialize()
        {
            TERASOLUNA_Fw_Client_Conversion_ConverterFactoryAccessor._factory = null;
            TERASOLUNA_Fw_Client_Forms_ViewFactoryAccessor._factory = null;

            _oldInit = FormForwarder_ForwardableStub01._init;
            _oldSetting = ConfigurationManager.AppSettings.Get(ConverterFactory.CONVERTER_TYPENAME);
            this._formClosed1Called = false;
            this._formClosed2Called = false;
            this._formClosed3Called = false;
            this._formClosing1Called = false;
            this._formClosing2Called = false;
            this._formClosing3Called = false;
            LogUTUtil.Clear();
        }

        //
        //各テストを実行した後にコードを実行するには、TestCleanup を使用
        //
        [TestCleanup()]
        public void MyTestCleanup()
        {
            FormForwarder_ForwardableStub01._init = _oldInit;
            ConfigurationManager.AppSettings.Set(ConverterFactory.CONVERTER_TYPENAME, _oldSetting);
            FormForwarder_ForwardableStub01._items.Clear();
            FormForwarder_FormStub01._items.Clear();
            FormForwarder_FormStub02._items.Clear();
            FormForwarder_ConverterStub01.Reset();
        }
        #endregion

        ///<summary>
        /// FormForwarder () のテスト
        ///</summary>
        ///<remarks>
        /// インスタンスが生成されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ConstructorTest01()
        {
            FormForwarder target = new FormForwarder();

            Assert.IsNotNull(target);
            Assert.AreEqual(Modality.Modeless, target.Modality);
            Assert.IsNull(target.OwnerForm);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(FormForwarder), LogLevel.INFO));

        }

        /// <summary>
        /// FormForwarder(IContainer)のテスト
        /// </summary>
        /// <remarks>
        /// 正常にインスタンス化できることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ConstructorIContainerTest01()
        {
            FormForwarder_IContainerStub01 arg = new FormForwarder_IContainerStub01();
            FormForwarder target = new FormForwarder(arg);

            // 確認
            Assert.IsNotNull(target);
            Assert.AreSame(target, arg.Component);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(FormForwarder), LogLevel.INFO));
        }

        /// <summary>
        /// FormForwarder(IContainer)のテスト
        /// </summary>
        /// <remarks>
        /// 引数にnullを与えた場合、正常にインスタンス化できることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ConstructorIContainerTest02()
        {
            FormForwarder_IContainerStub01 arg = null;
            FormForwarder target = new FormForwarder(arg);

            // 確認
            Assert.IsNotNull(target);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(FormForwarder), LogLevel.INFO));
        }

        ///<summary>
        /// DoClosed (object, FormClosedEventArgs) のテスト
        ///</summary>
        ///<remarks>
        /// senderがnullの場合、
        /// 例外ArgumentNullExceptionが起こることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void DoClosedTest01()
        {
            FormForwarder target = new FormForwarder();
            TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor accessor =
                new TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor(target);
            object sender = null;
            FormClosedEventArgs e = new FormClosedEventArgs(new CloseReason());
            target.ConvertId = "temp";
            target.ForwardHost = new FormBase();
            TERASOLUNA_Fw_Client_Forms_FormBaseAccessor formBaseAccessor =
                new TERASOLUNA_Fw_Client_Forms_FormBaseAccessor((FormBase)target.ForwardHost);
            formBaseAccessor._viewData = new DataSet();

            try
            {
                accessor.DoClosed(sender, e);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException ex)
            {
                // メッセージ確認
                //ArgumentNullException exception = ex.InnerException as ArgumentNullException;
                Assert.IsInstanceOfType(ex, typeof(ArgumentNullException));
                Assert.AreEqual("sender", ex.ParamName);
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(FormForwarder), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(FormForwarder), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "sender"));

        }

        ///<summary>
        /// DoClosed (object, FormClosedEventArgs) のテスト
        ///</summary>
        ///<remarks>
        /// senderがIFowardable を実装しないクラスのインスタンスの場合、
        /// 例外ArgumentNullExceptionが起こることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void DoClosedTest02()
        {
            FormForwarder target = new FormForwarder();
            TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor accessor =
                new TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor(target);
            object sender = new object();
            FormClosedEventArgs e = new FormClosedEventArgs(new CloseReason());
            target.ConvertId = "temp";
            target.ForwardHost = new FormBase();
            TERASOLUNA_Fw_Client_Forms_FormBaseAccessor formBaseAccessor =
                new TERASOLUNA_Fw_Client_Forms_FormBaseAccessor((FormBase)target.ForwardHost);
            formBaseAccessor._viewData = new DataSet();

            try
            {
                accessor.DoClosed(sender, e);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException ex)
            {
                // メッセージ確認
                //TerasolunaException exception = ex.InnerException as TerasolunaException;
                Assert.IsInstanceOfType(ex, typeof(TerasolunaException));
                Assert.AreEqual(MessageManager.GetMessage(
                    "E_INVALID_CAST",
                    sender.GetType().FullName,
                    typeof(IForwardable).FullName), ex.Message);

            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(FormForwarder), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(FormForwarder), typeof(TerasolunaException), "E_INVALID_CAST",
                    sender.GetType().FullName,
                    typeof(IForwardable).FullName));

        }

        ///<summary>
        /// DoClosed (object, FormClosedEventArgs) のテスト
        ///</summary>
        ///<remarks>
        /// senderがIFowardable を実装しないFormのインスタンスの場合、
        /// 例外ArgumentNullExceptionが起こることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void DoClosedTest02_2()
        {
            FormForwarder target = new FormForwarder();
            TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor accessor =
                new TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor(target);
            object sender = new Form();
            (sender as Form).DialogResult = DialogResult.OK;
            FormClosedEventArgs e = new FormClosedEventArgs(new CloseReason());
            target.ConvertId = "temp";
            target.ForwardHost = new FormBase();
            TERASOLUNA_Fw_Client_Forms_FormBaseAccessor formBaseAccessor =
                new TERASOLUNA_Fw_Client_Forms_FormBaseAccessor((FormBase)target.ForwardHost);
            formBaseAccessor._viewData = new DataSet();

            try
            {
                accessor.DoClosed(sender, e);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException ex)
            {
                //TerasolunaException exception = ex.InnerException as TerasolunaException;
                Assert.IsInstanceOfType(ex, typeof(TerasolunaException));
                Assert.AreEqual(MessageManager.GetMessage(
                    "E_INVALID_CAST",
                    sender.GetType().FullName,
                    typeof(IForwardable).FullName), ex.Message);

            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(FormForwarder), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(FormForwarder), typeof(TerasolunaException), "E_INVALID_CAST",
                    sender.GetType().FullName,
                    typeof(IForwardable).FullName));

        }

        ///<summary>
        /// DoClosed (object, FormClosedEventArgs) のテスト
        ///</summary>
        ///<remarks>
        /// senderがIFowardable を実装するクラス、
        /// 且つeがnot null、
        /// 且つプロパティForwardHostがnot null、
        /// 且つプロパティConvertIdが空文字列でない文字列
        /// 且つFormClosedイベントが登録されていない場合、
        /// 正常に処理されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void DoClosedTest03()
        {
            FormForwarder target = new FormForwarder();
            TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor accessor = new TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor(target);

            FormBase sender = new FormBase();
            (sender as Form).DialogResult = DialogResult.OK;

            sender.ViewData = new DataSet();
            FormClosedEventArgs e = new FormClosedEventArgs(new CloseReason());
            target.ForwardHost = new FormBase();
            TERASOLUNA_Fw_Client_Forms_FormBaseAccessor formBaseAccessor =
                new TERASOLUNA_Fw_Client_Forms_FormBaseAccessor((FormBase)target.ForwardHost);
            formBaseAccessor._viewData = new DataSet();

            string convertId = "temp";
            target.ConvertId = convertId;
            ConfigurationManager.AppSettings.Set(ConverterFactory.CONVERTER_TYPENAME,
                TEST_CONVERTER_TYPE);

            accessor.DoClosed(sender, e);

            Assert.IsTrue(FormForwarder_ConverterStub01.reverseCalled);
            Assert.AreEqual(convertId, FormForwarder_ConverterStub01.id);
            Assert.AreSame(target.ForwardHost.ViewData, FormForwarder_ConverterStub01.srcData);
            Assert.AreSame(((IForwardable)sender).ViewData, FormForwarder_ConverterStub01.destData);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(FormForwarder), LogLevel.INFO));
        }

        ///<summary>
        /// DoClosed (object, FormClosedEventArgs) のテスト
        ///</summary>
        ///<remarks>
        /// [バグ改修で追加]
        /// senderがIFowardable を実装するクラスであり、sender.DialogResultがOKではない。
        /// 且つeがnot null、
        /// 且つプロパティForwardHostがnot null、
        /// 且つプロパティConvertIdが空文字列でない文字列
        /// 且つFormClosedイベントが登録されていない場合、
        /// 正常に処理されることを確認するテスト。
        /// 例外が発生しないこと、Reverseメソッドが呼び出されないことを確認する。
        ///</remarks>
        [TestMethod()]
        public void DoClosedTest03_2()
        {
            FormForwarder target = new FormForwarder();
            TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor accessor = new TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor(target);

            FormBase sender = new FormBase();
            (sender as Form).DialogResult = DialogResult.No;
            sender.ViewData = new DataSet();
            FormClosedEventArgs e = new FormClosedEventArgs(new CloseReason());
            target.ForwardHost = new FormBase();
            TERASOLUNA_Fw_Client_Forms_FormBaseAccessor formBaseAccessor =
                new TERASOLUNA_Fw_Client_Forms_FormBaseAccessor((FormBase)target.ForwardHost);
            formBaseAccessor._viewData = new DataSet();

            string convertId = "temp";
            target.ConvertId = convertId;
            ConfigurationManager.AppSettings.Set(ConverterFactory.CONVERTER_TYPENAME,
                TEST_CONVERTER_TYPE);

            accessor.DoClosed(sender, e);

            Assert.IsFalse(FormForwarder_ConverterStub01.reverseCalled);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(FormForwarder), LogLevel.INFO));

        }

        ///<summary>
        /// DoClosed (object, FormClosedEventArgs) のテスト
        ///</summary>
        ///<remarks>
        /// eがnullの場合、
        /// 正常に処理されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void DoClosedTest04()
        {
            FormForwarder target = new FormForwarder();
            TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor accessor =
                new TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor(target);

            FormBase sender = new FormBase();
            (sender as Form).DialogResult = DialogResult.OK;
            sender.ViewData = new DataSet();
            FormClosedEventArgs e = null;
            target.ForwardHost = new FormBase();
            TERASOLUNA_Fw_Client_Forms_FormBaseAccessor formBaseAccessor =
                new TERASOLUNA_Fw_Client_Forms_FormBaseAccessor((FormBase)target.ForwardHost);
            formBaseAccessor._viewData = new DataSet();

            string convertId = "temp";
            target.ConvertId = convertId;
            ConfigurationManager.AppSettings.Set(ConverterFactory.CONVERTER_TYPENAME, TEST_CONVERTER_TYPE);

            accessor.DoClosed(sender, e);
            Assert.IsTrue(FormForwarder_ConverterStub01.reverseCalled);
            Assert.AreEqual(convertId, FormForwarder_ConverterStub01.id);
            Assert.AreSame(target.ForwardHost.ViewData, FormForwarder_ConverterStub01.srcData);
            Assert.AreSame(((IForwardable)sender).ViewData, FormForwarder_ConverterStub01.destData);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(FormForwarder), LogLevel.INFO));
        }

        ///<summary>
        /// DoClosed (object, FormClosedEventArgs) のテスト
        ///</summary>
        ///<remarks>
        /// ForwardHostがnullの場合、
        /// Reverseが呼び出されないことを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void DoClosedTest05()
        {
            FormForwarder target = new FormForwarder();
            TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor accessor = new TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor(target);

            FormBase sender = new FormBase();
            (sender as Form).DialogResult = DialogResult.OK;
            sender.ViewData = new DataSet();
            FormClosedEventArgs e = new FormClosedEventArgs(new CloseReason());
            target.ForwardHost = null;
            target.ConvertId = "temp";
            ConfigurationManager.AppSettings.Set(ConverterFactory.CONVERTER_TYPENAME, TEST_CONVERTER_TYPE);

            accessor.DoClosed(sender, e);

            Assert.IsFalse(FormForwarder_ConverterStub01.reverseCalled);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(FormForwarder), LogLevel.INFO));

        }

        ///<summary>
        /// DoClosed (object, FormClosedEventArgs) のテスト
        ///</summary>
        ///<remarks>
        /// ConvertIdがnullの場合、
        /// Reverseが呼び出されないことを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void DoClosedTest06()
        {
            FormForwarder target = new FormForwarder();
            TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor accessor = new TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor(target);

            FormBase sender = new FormBase();
            (sender as Form).DialogResult = DialogResult.OK;
            sender.ViewData = new DataSet();
            FormClosedEventArgs e = new FormClosedEventArgs(new CloseReason());
            target.ForwardHost = new FormBase();
            TERASOLUNA_Fw_Client_Forms_FormBaseAccessor formBaseAccessor =
                new TERASOLUNA_Fw_Client_Forms_FormBaseAccessor((FormBase)target.ForwardHost);
            formBaseAccessor._viewData = new DataSet();

            target.ConvertId = null;
            ConfigurationManager.AppSettings.Set(ConverterFactory.CONVERTER_TYPENAME, TEST_CONVERTER_TYPE);

            accessor.DoClosed(sender, e);

            Assert.IsFalse(FormForwarder_ConverterStub01.reverseCalled);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(FormForwarder), LogLevel.INFO));

        }

        ///<summary>
        /// DoClosed (object, FormClosedEventArgs) のテスト
        ///</summary>
        ///<remarks>
        /// ConvertIdが空文字列の場合、
        /// Reverseが呼び出されないことを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void DoClosedTest07()
        {
            FormForwarder target = new FormForwarder();
            TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor accessor = new TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor(target);

            FormBase sender = new FormBase();
            (sender as Form).DialogResult = DialogResult.OK;

            sender.ViewData = new DataSet();
            FormClosedEventArgs e = null;
            target.ForwardHost = new FormBase();
            TERASOLUNA_Fw_Client_Forms_FormBaseAccessor formBaseAccessor =
                new TERASOLUNA_Fw_Client_Forms_FormBaseAccessor((FormBase)target.ForwardHost);
            formBaseAccessor._viewData = new DataSet();

            target.ConvertId = string.Empty;
            ConfigurationManager.AppSettings.Set(ConverterFactory.CONVERTER_TYPENAME, TEST_CONVERTER_TYPE);

            accessor.DoClosed(sender, e);

            Assert.IsFalse(FormForwarder_ConverterStub01.reverseCalled);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(FormForwarder), LogLevel.INFO));

        }

        ///<summary>
        /// DoClosed (object, FormClosedEventArgs) のテスト
        ///</summary>
        ///<remarks>
        /// FormClosedイベントにイベントが１つ登録されている場合、
        /// 登録したイベントが実行されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void DoClosedTest08()
        {
            FormForwarder target = new FormForwarder();
            TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor accessor = new TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor(target);

            FormBase sender = new FormBase();
            (sender as Form).DialogResult = DialogResult.OK;
            sender.ViewData = new DataSet();
            FormClosedEventArgs e = new FormClosedEventArgs(new CloseReason()); ;
            target.ForwardHost = new FormBase();
            TERASOLUNA_Fw_Client_Forms_FormBaseAccessor formBaseAccessor =
                new TERASOLUNA_Fw_Client_Forms_FormBaseAccessor((FormBase)target.ForwardHost);
            formBaseAccessor._viewData = new DataSet();

            string convertId = "temp";
            target.ConvertId = convertId;
            target.FormClosed += new ForwardableFormCloseEventHandler(FormClosed1);
            ConfigurationManager.AppSettings.Set(ConverterFactory.CONVERTER_TYPENAME, TEST_CONVERTER_TYPE);

            accessor.DoClosed(sender, e);

            Assert.IsTrue(FormForwarder_ConverterStub01.reverseCalled);
            Assert.AreEqual(convertId, FormForwarder_ConverterStub01.id);
            Assert.AreSame(target.ForwardHost.ViewData, FormForwarder_ConverterStub01.srcData);
            Assert.AreSame(((IForwardable)sender).ViewData, FormForwarder_ConverterStub01.destData);
            Assert.IsTrue(this._formClosed1Called);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(FormForwarder), LogLevel.INFO));

        }

        ///<summary>
        /// DoClosed (object, FormClosedEventArgs) のテスト
        ///</summary>
        ///<remarks>
        /// FormClosedイベントにイベントが３つ登録されている場合、
        /// すべてのイベントが実行されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void DoClosedTest09()
        {
            FormForwarder target = new FormForwarder();
            TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor accessor = new TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor(target);

            FormBase sender = new FormBase();
            (sender as Form).DialogResult = DialogResult.OK;
            sender.ViewData = new DataSet();
            FormClosedEventArgs e = new FormClosedEventArgs(new CloseReason()); ;
            target.ForwardHost = new FormBase();
            TERASOLUNA_Fw_Client_Forms_FormBaseAccessor formBaseAccessor =
                new TERASOLUNA_Fw_Client_Forms_FormBaseAccessor((FormBase)target.ForwardHost);
            formBaseAccessor._viewData = new DataSet();

            string convertId = "temp";
            target.ConvertId = convertId;

            target.FormClosed += new ForwardableFormCloseEventHandler(FormClosed1);
            target.FormClosed += new ForwardableFormCloseEventHandler(FormClosed2);
            target.FormClosed += new ForwardableFormCloseEventHandler(FormClosed3);

            ConfigurationManager.AppSettings.Set(ConverterFactory.CONVERTER_TYPENAME, TEST_CONVERTER_TYPE);

            accessor.DoClosed(sender, e);

            Assert.IsTrue(FormForwarder_ConverterStub01.reverseCalled);
            Assert.AreEqual(convertId, FormForwarder_ConverterStub01.id);
            Assert.AreSame(target.ForwardHost.ViewData, FormForwarder_ConverterStub01.srcData);
            Assert.AreSame(((IForwardable)sender).ViewData, FormForwarder_ConverterStub01.destData);

            Assert.IsTrue(this._formClosed1Called);
            Assert.IsTrue(this._formClosed2Called);
            Assert.IsTrue(this._formClosed3Called);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(FormForwarder), LogLevel.INFO));

        }

        ///<summary>
        /// DoClosed (object, FormClosedEventArgs) のテスト
        ///</summary>
        ///<remarks>
        /// ConvertIdがnull、
        /// 且つsenderのViewDataプロパティがnullの場合、
        /// Reverseが呼びだされずに正常に実行されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void DoClosedTest10()
        {
            FormForwarder target = new FormForwarder();
            TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor accessor = new TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor(target);

            FormBase sender = new FormBase();
            (sender as Form).DialogResult = DialogResult.OK;
            sender.ViewData = null;
            FormClosedEventArgs e = new FormClosedEventArgs(new CloseReason()); ;
            target.ForwardHost = new FormBase();
            TERASOLUNA_Fw_Client_Forms_FormBaseAccessor formBaseAccessor =
                new TERASOLUNA_Fw_Client_Forms_FormBaseAccessor((FormBase)target.ForwardHost);
            formBaseAccessor._viewData = new DataSet();
            target.ConvertId = null;

            ConfigurationManager.AppSettings.Set(ConverterFactory.CONVERTER_TYPENAME, TEST_CONVERTER_TYPE);

            accessor.DoClosed(sender, e);

            Assert.IsFalse(FormForwarder_ConverterStub01.reverseCalled);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(FormForwarder), LogLevel.INFO));

        }

        ///<summary>
        /// DoClosed (object, FormClosedEventArgs) のテスト
        ///</summary>
        ///<remarks>
        /// ConvertIdが指定されているが、senderのViewDataがnullの場合
        /// 例外TerasolunaExcepionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void DoClosedTest11()
        {
            FormForwarder target = new FormForwarder();
            TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor accessor = new TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor(target);

            FormBase sender = new FormBase();
            (sender as FormBase).DialogResult = DialogResult.OK;
            sender.ViewData = null;
            FormClosedEventArgs e = new FormClosedEventArgs(new CloseReason()); ;
            target.ForwardHost = new FormBase();
            TERASOLUNA_Fw_Client_Forms_FormBaseAccessor formBaseAccessor =
                new TERASOLUNA_Fw_Client_Forms_FormBaseAccessor((FormBase)target.ForwardHost);
            formBaseAccessor._viewData = new DataSet();
            target.ConvertId = "temp";

            ConfigurationManager.AppSettings.Set(ConverterFactory.CONVERTER_TYPENAME, TEST_CONVERTER_TYPE);

            try
            {
                accessor.DoClosed(sender, e);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException ex)
            {
                //TerasolunaException exception = ex.InnerException as TerasolunaException;
                Assert.IsInstanceOfType(ex, typeof(TerasolunaException));
                Assert.AreEqual(MessageManager.GetMessage(
                    "E_FORMS_NULL_TARGET_OBJECT",
                    "ConvertId", "ViewData"), ex.Message);

            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(FormForwarder), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(FormForwarder), typeof(TerasolunaException), "E_FORMS_NULL_TARGET_OBJECT",
                    "ConvertId", "ViewData"));

        }

        ///<summary>
        /// DoClosed (object, FormClosedEventArgs) のテスト
        ///</summary>
        ///<remarks>
        /// ConvertIdがnull、
        /// 且つForwardHostのViewDataプロパティがnullの場合、
        /// Reverseが呼びだされずに正常実行されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void DoClosedTest12()
        {
            FormForwarder target = new FormForwarder();
            TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor accessor = new TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor(target);

            FormBase sender = new FormBase();
            (sender as Form).DialogResult = DialogResult.OK;
            sender.ViewData = new DataSet();
            FormClosedEventArgs e = new FormClosedEventArgs(new CloseReason()); ;
            target.ForwardHost = new FormBase();
            TERASOLUNA_Fw_Client_Forms_FormBaseAccessor formBaseAccessor =
                new TERASOLUNA_Fw_Client_Forms_FormBaseAccessor((FormBase)target.ForwardHost);
            formBaseAccessor._viewData = null;
            target.ConvertId = null;

            ConfigurationManager.AppSettings.Set(ConverterFactory.CONVERTER_TYPENAME, TEST_CONVERTER_TYPE);

            accessor.DoClosed(sender, e);

            Assert.IsFalse(FormForwarder_ConverterStub01.reverseCalled);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(FormForwarder), LogLevel.INFO));

        }

        ///<summary>
        /// DoClosed (object, FormClosedEventArgs) のテスト
        ///</summary>
        ///<remarks>
        /// ConverteIdが指定されているが、ForwardHostのViewDataがnullの場合
        /// 例外TerasolunaExcepionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void DoClosedTest13()
        {
            FormForwarder target = new FormForwarder();
            TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor accessor = new TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor(target);

            FormBase sender = new FormBase();
            (sender as FormBase).DialogResult = DialogResult.OK;
            sender.ViewData = new DataSet();
            FormClosedEventArgs e = new FormClosedEventArgs(new CloseReason()); ;
            target.ForwardHost = new FormBase();
            TERASOLUNA_Fw_Client_Forms_FormBaseAccessor formBaseAccessor =
                new TERASOLUNA_Fw_Client_Forms_FormBaseAccessor((FormBase)target.ForwardHost);
            formBaseAccessor._viewData = null;
            target.ConvertId = "temp";

            ConfigurationManager.AppSettings.Set(ConverterFactory.CONVERTER_TYPENAME, TEST_CONVERTER_TYPE);

            try
            {
                accessor.DoClosed(sender, e);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException ex)
            {
                //TerasolunaException exception = ex.InnerException as TerasolunaException;
                Assert.IsInstanceOfType(ex, typeof(TerasolunaException));
                Assert.AreEqual(MessageManager.GetMessage(
                    "E_FORMS_NULL_HOST_OBJECT",
                    "ConvertId", "ViewData"), ex.Message);

            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(FormForwarder), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(FormForwarder), typeof(TerasolunaException), "E_FORMS_NULL_HOST_OBJECT",
                    "ConvertId", "ViewData"));

        }

        ///<summary>
        /// DoClosing (object, FormClosingEventArgs) のテスト
        ///</summary>
        ///<remarks>
        /// senderがnullの場合、
        /// 例外ArgumentNullExceptionが起こることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void DoClosingTest01()
        {
            FormForwarder target = new FormForwarder();

            TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor accessor = new TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor(target);

            object sender = null;

            FormClosingEventArgs e = new FormClosingEventArgs(new CloseReason(), false);

            try
            {
                accessor.DoClosing(sender, e);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException ex)
            {
                // メッセージ確認
                //ArgumentNullException exception = ex.InnerException as ArgumentNullException;
                Assert.IsInstanceOfType(ex, typeof(ArgumentNullException));
                Assert.AreEqual("sender", ex.ParamName);
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(FormForwarder), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(FormForwarder), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "sender"));

        }

        ///<summary>
        /// DoClosing (object, FormClosingEventArgs) のテスト
        ///</summary>
        ///<remarks>
        /// senderがIFowardable を実装しないクラスのインスタンスの場合、
        /// 例外TerasolunaExceptionが起こることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void DoClosingTest02()
        {
            FormForwarder target = new FormForwarder();

            TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor accessor = new TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor(target);

            object sender = new object();

            FormClosingEventArgs e = new FormClosingEventArgs(new CloseReason(), false);

            try
            {
                accessor.DoClosing(sender, e);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException ex)
            {
                // メッセージ確認
                //TerasolunaException exception = ex.InnerException as TerasolunaException;
                Assert.IsInstanceOfType(ex, typeof(TerasolunaException));
                Assert.AreEqual(MessageManager.GetMessage(
                    "E_INVALID_CAST",
                    sender.GetType().FullName,
                    typeof(IForwardable).FullName), ex.Message);

            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(FormForwarder), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(FormForwarder), typeof(TerasolunaException), "E_INVALID_CAST",
                    sender.GetType().FullName,
                    typeof(IForwardable).FullName));

        }

        ///<summary>
        /// DoClosing (object, FormClosingEventArgs) のテスト
        ///</summary>
        ///<remarks>
        /// senderがIForwardable を実装するクラスのインスタンスの場合、
        /// 例外なく処理が終了することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void DoClosingTest03()
        {
            FormForwarder target = new FormForwarder();
            TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor accessor = new TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor(target);
            object sender = new FormBase();
            FormClosingEventArgs e = new FormClosingEventArgs(new CloseReason(), false);
            accessor.DoClosing(sender, e);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(FormForwarder), LogLevel.INFO));

        }

        ///<summary>
        /// DoClosing (object, FormClosingEventArgs) のテスト
        ///</summary>
        ///<remarks>
        /// eがnullの場合、
        /// 例外なく処理が終了することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void DoClosingTest04()
        {
            FormForwarder target = new FormForwarder();
            TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor accessor = new TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor(target);
            object sender = new FormBase();
            FormClosingEventArgs e = null;
            accessor.DoClosing(sender, e);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(FormForwarder), LogLevel.INFO));

        }

        ///<summary>
        /// DoClosing (object, FormClosingEventArgs) のテスト
        ///</summary>
        ///<remarks>
        /// FormClosingイベントが登録されている場合、
        /// イベントが実行され、例外なく処理が終了することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void DoClosingTest05()
        {
            FormForwarder target = new FormForwarder();
            TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor accessor = new TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor(target);
            object sender = new FormBase();
            FormClosingEventArgs e = new FormClosingEventArgs(new CloseReason(), false);
            target.FormClosing += new ForwardableFormCloseEventHandler(FormClosing1);

            accessor.DoClosing(sender, e);

            Assert.IsTrue(this._formClosing1Called);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(FormForwarder), LogLevel.INFO));

        }

        ///<summary>
        /// DoClosing (object, FormClosingEventArgs) のテスト
        ///</summary>
        ///<remarks>
        /// 3つのFormClosingイベントが登録されている場合
        /// すべてのイベントが実行されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void DoClosingTest06()
        {
            FormForwarder target = new FormForwarder();

            TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor accessor = new TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor(target);

            object sender = new FormBase();

            FormClosingEventArgs e = new FormClosingEventArgs(new CloseReason(), false);

            target.FormClosing += new ForwardableFormCloseEventHandler(FormClosing1);
            target.FormClosing += new ForwardableFormCloseEventHandler(FormClosing2);
            target.FormClosing += new ForwardableFormCloseEventHandler(FormClosing3);

            accessor.DoClosing(sender, e);

            Assert.IsTrue(this._formClosing1Called);
            Assert.IsTrue(this._formClosing2Called);
            Assert.IsTrue(this._formClosing3Called);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(FormForwarder), LogLevel.INFO));

        }

        ///<summary>
        /// EnableParent (object, FormClosedEventArgs) のテスト
        ///</summary>
        ///<remarks>
        /// senderがnullの場合、
        /// ForwardHostのEnabledプロパティがtrueになることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void EnableParentTest01()
        {
            FormForwarder target = new FormForwarder();

            TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor accessor =
                new TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor(target);

            object sender = null;
            FormClosedEventArgs e = new FormClosedEventArgs(new CloseReason());
            FormBase host = new FormBase();
            host.Enabled = false;
            target.ForwardHost = host;
            accessor.EnableParent(sender, e);
            Assert.IsTrue(host.Enabled);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(FormForwarder), LogLevel.INFO));

        }

        ///<summary>
        /// EnableParent (object, FormClosedEventArgs) のテスト
        ///</summary>
        ///<remarks>
        /// senderがIFowardable を実装しないクラスのインスタンスの場合、
        /// ForwardHostのEnabledプロパティがtrueになることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void EnableParentTest02()
        {
            FormForwarder target = new FormForwarder();

            TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor accessor =
                new TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor(target);

            object sender = new object();
            FormClosedEventArgs e = new FormClosedEventArgs(new CloseReason());
            FormBase host = new FormBase();
            host.Enabled = false;
            target.ForwardHost = host;

            accessor.EnableParent(sender, e);

            Assert.IsTrue(host.Enabled);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(FormForwarder), LogLevel.INFO));

        }

        /// <summary>
        ///EnableParent (object, FormClosedEventArgs) のテスト
        ///</summary>
        /// <remarks>
        /// senderがIFowardable を実装するクラスのインスタンス、
        /// 且つeがnot null、
        /// 且つプロパティForwardHostがForm 継承クラスのインスタンスの場合、
        /// ForwardHostのEnabledプロパティがtrueになることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void EnableParentTest03()
        {
            FormForwarder target = new FormForwarder();

            TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor accessor =
                new TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor(target);

            object sender = new FormBase();
            FormClosedEventArgs e = new FormClosedEventArgs(new CloseReason());
            FormBase host = new FormBase();
            host.Enabled = false;
            target.ForwardHost = host;
            accessor.EnableParent(sender, e);
            Assert.IsTrue(host.Enabled);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(FormForwarder), LogLevel.INFO));

        }

        ///<summary>
        /// EnableParent (object, FormClosedEventArgs) のテスト
        ///</summary>
        ///<remarks>
        /// eがnullの場合、
        /// ForwardHostのEnabledプロパティがtrueになることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void EnableParentTest04()
        {
            FormForwarder target = new FormForwarder();

            TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor accessor =
                new TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor(target);

            object sender = new FormBase();
            FormClosedEventArgs e = null;
            FormBase host = new FormBase();
            host.Enabled = false;
            target.ForwardHost = host;
            accessor.EnableParent(sender, e);
            Assert.IsTrue(host.Enabled);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(FormForwarder), LogLevel.INFO));

        }

        ///<summary>
        /// EnableParent (object, FormClosedEventArgs) のテスト
        ///</summary>
        ///<remarks>
        /// プロパティForwardHostがnullの場合、
        /// 例外なく処理が終了することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void EnableParentTest05()
        {
            FormForwarder target = new FormForwarder();

            TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor accessor =
                new TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor(target);

            object sender = new FormBase();
            FormClosedEventArgs e = new FormClosedEventArgs(new CloseReason());
            target.ForwardHost = null;

            accessor.EnableParent(sender, e);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(FormForwarder), LogLevel.INFO));

        }

        ///<summary>
        /// EnableParent (object, FormClosedEventArgs) のテスト
        ///</summary>
        ///<remarks>
        /// ForwardHostがForm を継承しないクラスのインスタンスの場合、
        /// 例外なく処理が終了することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void EnableParentTest06()
        {
            FormForwarder target = new FormForwarder();

            TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor accessor =
                new TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor(target);

            object sender = new FormBase();
            FormClosedEventArgs e = new FormClosedEventArgs(new CloseReason());
            target.ForwardHost = new FormForwarder_ForwardableStub01();

            accessor.EnableParent(sender, e);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(FormForwarder), LogLevel.INFO));

        }

        ///<summary>
        /// Execute () のテスト
        ///</summary>
        ///<remarks>
        /// _modalityがFormModalのインスタンスの場合、
        /// 正常に処理されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Forms\FormForwarderTestData\FormForwarder_ViewConfigExternal01.config")]
        [DeploymentItem(@"Forms\FormForwarderTestData\FormForwarder_ViewConfigSection01.config")]
        public void ExecuteTest01()
        {
            ViewConfigurationSection section = ConfigTestHelper.GetViewConfigurationSectionFromCustomFile(VIEW_CONFIG_FILE);
            section.CreateViewElements();
            TERASOLUNA_Fw_Client_Configuration_TerasolunaClientConfigurationAccessor._viewConfigSection = section;

            FormForwarder target = new FormForwarder();

            TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor accessor =
                new TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor(target);

            accessor._modality = Modality.FormModal;
            accessor._ownerForm = null;
            target.ViewId = "testView";
            target.ConvertId = null;
            target.ForwardHost = null;

            IForwardable forwardTarget = ViewFactory.CreateView(target.ViewId);
            Form targetForm = forwardTarget as Form;

            ForwardResult actual = target.Execute();

            Assert.AreEqual(ForwardResult.None, actual);
            Assert.AreEqual(0, FormForwarder_FormStub01._items.Count);

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(FormForwarder), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckInfoUseMessageId(typeof(FormForwarder), "I_FORM_TRANSITION_NO_FORWARD_HOST",
                                Modality.FormModal.ToString(),
                                targetForm.GetType().FullName));

        }

        ///<summary>
        /// Execute () のテスト
        ///</summary>
        ///<remarks>
        /// _modalityがFormModal、且つForwardHostがFormクラスを継承しないクラスのインスタンスの場合、
        /// 正常に処理されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Forms\FormForwarderTestData\FormForwarder_ViewConfigExternal01.config")]
        [DeploymentItem(@"Forms\FormForwarderTestData\FormForwarder_ViewConfigSection01.config")]
        public void ExecuteTest02()
        {
            ViewConfigurationSection section = ConfigTestHelper.GetViewConfigurationSectionFromCustomFile(VIEW_CONFIG_FILE);
            section.CreateViewElements();
            TERASOLUNA_Fw_Client_Configuration_TerasolunaClientConfigurationAccessor._viewConfigSection = section;

            FormForwarder target = new FormForwarder();

            TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor accessor =
                new TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor(target);

            accessor._modality = Modality.FormModal;
            accessor._ownerForm = null;
            target.ViewId = "testView";
            target.ConvertId = null;
            target.ForwardHost = new FormForwarder_ForwardableStub01();
            FormForwarder_ForwardableStub01._items = new Hashtable();
            FormForwarder_FormStub01._items = new Hashtable();

            IForwardable forwardTarget = ViewFactory.CreateView(target.ViewId);
            Form targetForm = forwardTarget as Form;

            ForwardResult actual = target.Execute();

            Assert.AreEqual(ForwardResult.None, actual);
            Assert.AreEqual(0, FormForwarder_FormStub01._items.Count);

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(FormForwarder), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckInfoUseMessageId(typeof(FormForwarder), "I_FORM_TRANSITION",
                                Modality.FormModal.ToString(),
                                target.ForwardHost.GetType().FullName,
                                targetForm.GetType().FullName));
        }

        ///<summary>
        /// Execute () のテスト
        ///</summary>
        ///<remarks>
        /// _modalityがFormModal、
        /// 且つForwardHostがFormクラスを継承するクラスのインスタンス、
        /// ForwardHostに設定されたクラスのViewDataがnot nullの場合、
        /// 正常に処理されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Forms\FormForwarderTestData\FormForwarder_ViewConfigExternal01.config")]
        [DeploymentItem(@"Forms\FormForwarderTestData\FormForwarder_ViewConfigSection01.config")]
        public void ExecuteTest03()
        {
            ViewConfigurationSection section = ConfigTestHelper.GetViewConfigurationSectionFromCustomFile(VIEW_CONFIG_FILE);
            section.CreateViewElements();
            TERASOLUNA_Fw_Client_Configuration_TerasolunaClientConfigurationAccessor._viewConfigSection = section;

            FormForwarder target = new FormForwarder();

            TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor accessor =
                new TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor(target);

            accessor._modality = Modality.FormModal;
            accessor._ownerForm = null;
            target.ViewId = "testView";
            target.ConvertId = null;
            target.ForwardHost = new FormForwarder_FormStub02();
            FormForwarder_FormStub02._items = new Hashtable();
            FormForwarder_FormStub01._items = new Hashtable();

            IForwardable forwardTarget = ViewFactory.CreateView(target.ViewId);
            Form targetForm = forwardTarget as Form;

            ForwardResult actual = target.Execute();

            Assert.AreEqual(ForwardResult.None, actual);
            Assert.AreEqual(0, FormForwarder_FormStub01._items.Count);

            Assert.IsFalse(((Form)target.ForwardHost).Enabled);

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(FormForwarder), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckInfoUseMessageId(typeof(FormForwarder), "I_FORM_TRANSITION",
                                Modality.FormModal.ToString(),
                                target.ForwardHost.GetType().FullName,
                                targetForm.GetType().FullName));
        }

        ///<summary>
        /// Execute () のテスト
        ///</summary>
        ///<remarks>
        /// _modalityがModelessの場合
        /// 正常に処理されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Forms\FormForwarderTestData\FormForwarder_ViewConfigExternal01.config")]
        [DeploymentItem(@"Forms\FormForwarderTestData\FormForwarder_ViewConfigSection01.config")]
        public void ExecuteTest04()
        {
            ViewConfigurationSection section = ConfigTestHelper.GetViewConfigurationSectionFromCustomFile(VIEW_CONFIG_FILE);
            section.CreateViewElements();
            TERASOLUNA_Fw_Client_Configuration_TerasolunaClientConfigurationAccessor._viewConfigSection = section;

            FormForwarder target = new FormForwarder();

            TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor accessor =
                new TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor(target);

            accessor._modality = Modality.Modeless;
            accessor._ownerForm = null;
            target.ViewId = "testView";
            target.ConvertId = null;
            target.ForwardHost = new FormForwarder_FormStub02();
            FormForwarder_FormStub02._items = new Hashtable();
            FormForwarder_FormStub01._items = new Hashtable();

            IForwardable forwardTarget = ViewFactory.CreateView(target.ViewId);
            Form targetForm = forwardTarget as Form;

            ForwardResult actual = target.Execute();

            Assert.AreEqual(ForwardResult.None, actual);
            Assert.AreEqual(0, FormForwarder_FormStub01._items.Count);

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(FormForwarder), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckInfoUseMessageId(typeof(FormForwarder), "I_FORM_TRANSITION",
                                Modality.Modeless.ToString(),
                                target.ForwardHost.GetType().FullName,
                                targetForm.GetType().FullName));
        }

        ///<summary>
        /// Execute () のテスト
        ///</summary>
        ///<remarks>
        /// _ownerFormがnot nullの場合、
        /// 正常に処理されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Forms\FormForwarderTestData\FormForwarder_ViewConfigExternal01.config")]
        [DeploymentItem(@"Forms\FormForwarderTestData\FormForwarder_ViewConfigSection01.config")]
        public void ExecuteTest05()
        {
            ViewConfigurationSection section = ConfigTestHelper.GetViewConfigurationSectionFromCustomFile(VIEW_CONFIG_FILE);
            section.CreateViewElements();
            TERASOLUNA_Fw_Client_Configuration_TerasolunaClientConfigurationAccessor._viewConfigSection = section;

            FormForwarder target = new FormForwarder();

            TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor accessor =
                new TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor(target);

            accessor._modality = Modality.Modeless;
            accessor._ownerForm = new Form();
            target.ViewId = "testView";
            target.ConvertId = null;
            target.ForwardHost = new FormForwarder_FormStub02();
            FormForwarder_FormStub02._items = new Hashtable();
            FormForwarder_FormStub01._items = new Hashtable();

            IForwardable forwardTarget = ViewFactory.CreateView(target.ViewId);
            Form targetForm = forwardTarget as Form;

            ForwardResult actual = target.Execute();

            Assert.AreEqual(ForwardResult.None, actual);
            Assert.AreEqual(0, FormForwarder_FormStub01._items.Count);

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(FormForwarder), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckInfoUseMessageId(typeof(FormForwarder), "I_FORM_TRANSITION",
                               Modality.Modeless.ToString(),
                               target.ForwardHost.GetType().FullName,
                               targetForm.GetType().FullName));
        }

        ///<summary>
        /// Execute () のテスト
        ///</summary>
        ///<remarks>
        /// ViewIdがnullの場合、例外TerasolunaExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Forms\FormForwarderTestData\FormForwarder_ViewConfigExternal01.config")]
        [DeploymentItem(@"Forms\FormForwarderTestData\FormForwarder_ViewConfigSection01.config")]
        public void ExecuteTest06()
        {
            ViewConfigurationSection section = ConfigTestHelper.GetViewConfigurationSectionFromCustomFile(VIEW_CONFIG_FILE);
            section.CreateViewElements();
            TERASOLUNA_Fw_Client_Configuration_TerasolunaClientConfigurationAccessor._viewConfigSection = section;

            FormForwarder target = new FormForwarder();

            TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor accessor =
                new TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor(target);

            accessor._modality = Modality.Modeless;
            accessor._ownerForm = null;
            target.ViewId = null;
            target.ConvertId = null;
            target.ForwardHost = new FormBase();

            try
            {
                ForwardResult actual = target.Execute();
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_NULL_OR_EMPTY_PROPERTY_STRING", "Forwarder", "ViewId"), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(FormForwarder), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(FormForwarder), typeof(TerasolunaException), "E_NULL_OR_EMPTY_PROPERTY_STRING", "Forwarder", "ViewId"));

        }

        ///<summary>
        /// Execute () のテスト
        ///</summary>
        ///<remarks>
        /// ViewIdが空文字の場合、例外TerasolunaExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Forms\FormForwarderTestData\FormForwarder_ViewConfigExternal01.config")]
        [DeploymentItem(@"Forms\FormForwarderTestData\FormForwarder_ViewConfigSection01.config")]
        public void ExecuteTest07()
        {
            ViewConfigurationSection section = ConfigTestHelper.GetViewConfigurationSectionFromCustomFile(VIEW_CONFIG_FILE);
            section.CreateViewElements();
            TERASOLUNA_Fw_Client_Configuration_TerasolunaClientConfigurationAccessor._viewConfigSection = section;

            FormForwarder target = new FormForwarder();

            TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor accessor =
                new TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor(target);

            accessor._modality = Modality.Modeless;
            accessor._ownerForm = null;
            target.ViewId = string.Empty;
            target.ConvertId = null;
            target.ForwardHost = new FormBase();

            try
            {
                ForwardResult actual = target.Execute();
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_NULL_OR_EMPTY_PROPERTY_STRING", "Forwarder", "ViewId"), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(FormForwarder), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(FormForwarder), typeof(TerasolunaException), "E_NULL_OR_EMPTY_PROPERTY_STRING", "Forwarder", "ViewId"));

        }

        ///<summary>
        /// Execute () のテスト
        ///</summary>
        ///<remarks>
        /// ViewIdがIForwardable 実装しないクラスの idの場合、
        /// 例外TerasolunaExceptionが発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Forms\FormForwarderTestData\FormForwarder_ViewConfigExternal01.config")]
        [DeploymentItem(@"Forms\FormForwarderTestData\FormForwarder_ViewConfigSection01.config")]
        public void ExecuteTest08()
        {
            ViewConfigurationSection section = ConfigTestHelper.GetViewConfigurationSectionFromCustomFile(VIEW_CONFIG_FILE);
            section.CreateViewElements();
            TERASOLUNA_Fw_Client_Configuration_TerasolunaClientConfigurationAccessor._viewConfigSection = section;

            FormForwarder target = new FormForwarder();

            TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor accessor =
                new TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor(target);

            accessor._modality = Modality.Modeless;
            accessor._ownerForm = null;
            target.ViewId = "testNotForward";
            target.ConvertId = null;
            target.ForwardHost = new FormBase();
            Type targetType = typeof(System.Object);

            try
            {
                ForwardResult actual = target.Execute();
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // メッセージ確認
                Assert.AreEqual(MessageManager.GetMessage("E_INVALID_CAST_TYPE_FROM_GENERICPARAM", targetType.AssemblyQualifiedName, typeof(IForwardable).AssemblyQualifiedName), e.Message);
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(FormForwarder), LogLevel.INFO));
        }

        ///<summary>
        /// Execute () のテスト
        ///</summary>
        ///<remarks>
        /// ViewIdがIForwardable を実装し、Form を継承しないクラスのidの場合、
        /// 例外TerasolunaExceptionが発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Forms\FormForwarderTestData\FormForwarder_ViewConfigExternal01.config")]
        [DeploymentItem(@"Forms\FormForwarderTestData\FormForwarder_ViewConfigSection01.config")]
        public void ExecuteTest09()
        {
            ViewConfigurationSection section = ConfigTestHelper.GetViewConfigurationSectionFromCustomFile(VIEW_CONFIG_FILE);
            section.CreateViewElements();
            TERASOLUNA_Fw_Client_Configuration_TerasolunaClientConfigurationAccessor._viewConfigSection = section;

            FormForwarder target = new FormForwarder();

            TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor accessor =
                new TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor(target);

            accessor._modality = Modality.Modeless;
            accessor._ownerForm = null;
            target.ViewId = "testForward";
            target.ConvertId = null;
            target.ForwardHost = new FormBase();
            IForwardable forwardTarget = ViewFactory.CreateView(target.ViewId);

            try
            {
                ForwardResult actual = target.Execute();
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // メッセージ確認
                Assert.AreEqual(MessageManager.GetMessage("E_INVALID_CAST",
                    forwardTarget.GetType().FullName,
                    typeof(Form).FullName), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(FormForwarder), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(FormForwarder), typeof(TerasolunaException), "E_INVALID_CAST",
                    forwardTarget.GetType().FullName,
                    typeof(Form).FullName));
        }

        ///<summary>
        /// Execute () のテスト
        ///</summary>
        ///<remarks>
        /// ViewIdがIForwardable を実装し、Form を継承するクラスの id、
        /// 且つConvertIdがnullの場合、
        /// Converterが呼ばれずに正常に遷移することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Forms\FormForwarderTestData\FormForwarder_ViewConfigExternal01.config")]
        [DeploymentItem(@"Forms\FormForwarderTestData\FormForwarder_ViewConfigSection01.config")]
        public void ExecuteTest10()
        {
            ViewConfigurationSection section = ConfigTestHelper.GetViewConfigurationSectionFromCustomFile(VIEW_CONFIG_FILE);
            section.CreateViewElements();
            TERASOLUNA_Fw_Client_Configuration_TerasolunaClientConfigurationAccessor._viewConfigSection = section;

            FormForwarder target = new FormForwarder();
            

            TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor accessor =
                new TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor(target);

            accessor._modality = Modality.Modeless;
            accessor._ownerForm = null;
            target.ViewId = "testView";
            target.ConvertId = null;
            target.ForwardHost = new FormBase();
            FormForwarder_FormStub01._items = new Hashtable();

            ConfigurationManager.AppSettings.Set(ConverterFactory.CONVERTER_TYPENAME,
                TEST_CONVERTER_TYPE);

            IForwardable forwardTarget = ViewFactory.CreateView(target.ViewId);
            Form targetForm = forwardTarget as Form;

            ForwardResult actual = target.Execute();

            Assert.AreEqual(ForwardResult.None, actual);
            Assert.AreEqual(0, FormForwarder_FormStub01._items.Count);
            Assert.IsFalse(FormForwarder_ConverterStub01.convertCalled);

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(FormForwarder), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckInfoUseMessageId(typeof(FormForwarder), "I_FORM_TRANSITION",
                             Modality.Modeless.ToString(),
                             target.ForwardHost.GetType().FullName,
                             targetForm.GetType().FullName));
        }

        ///<summary>
        /// Execute () のテスト
        ///</summary>
        ///<remarks>
        /// ConvertIdが空文字の場合、
        /// Converterが呼ばれずに正常に遷移することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Forms\FormForwarderTestData\FormForwarder_ViewConfigExternal01.config")]
        [DeploymentItem(@"Forms\FormForwarderTestData\FormForwarder_ViewConfigSection01.config")]
        public void ExecuteTest11()
        {
            ViewConfigurationSection section = ConfigTestHelper.GetViewConfigurationSectionFromCustomFile(VIEW_CONFIG_FILE);
            section.CreateViewElements();
            TERASOLUNA_Fw_Client_Configuration_TerasolunaClientConfigurationAccessor._viewConfigSection = section;

            FormForwarder target = new FormForwarder();

            TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor accessor =
                new TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor(target);

            accessor._modality = Modality.Modeless;
            accessor._ownerForm = null;
            target.ViewId = "testView";
            target.ConvertId = string.Empty;
            target.ForwardHost = new FormBase();
            FormForwarder_FormStub01._items = new Hashtable();

            ConfigurationManager.AppSettings.Set(ConverterFactory.CONVERTER_TYPENAME,
                TEST_CONVERTER_TYPE);
            
            IForwardable forwardTarget = ViewFactory.CreateView(target.ViewId);
            Form targetForm = forwardTarget as Form;

            ForwardResult actual = target.Execute();

            Assert.AreEqual(ForwardResult.None, actual);
            Assert.AreEqual(0, FormForwarder_FormStub01._items.Count);
            Assert.IsFalse(FormForwarder_ConverterStub01.convertCalled);

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(FormForwarder), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckInfoUseMessageId(typeof(FormForwarder), "I_FORM_TRANSITION",
                             Modality.Modeless.ToString(),
                             target.ForwardHost.GetType().FullName,
                             targetForm.GetType().FullName));

        }

        ///<summary>
        /// Execute () のテスト
        ///</summary>
        ///<remarks>
        /// ConvertIdが空文字列ではない、
        /// 且つForwardHost に設定されたクラスのItems の要素数が0の場合、
        /// Converterが呼ばれ、正常に遷移することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Forms\FormForwarderTestData\FormForwarder_ViewConfigExternal01.config")]
        [DeploymentItem(@"Forms\FormForwarderTestData\FormForwarder_ViewConfigSection01.config")]
        public void ExecuteTest12()
        {
            ViewConfigurationSection section = ConfigTestHelper.GetViewConfigurationSectionFromCustomFile(VIEW_CONFIG_FILE);
            section.CreateViewElements();
            TERASOLUNA_Fw_Client_Configuration_TerasolunaClientConfigurationAccessor._viewConfigSection = section;

            FormForwarder target = new FormForwarder();

            TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor accessor =
                new TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor(target);

            accessor._modality = Modality.Modeless;
            accessor._ownerForm = null;
            target.ViewId = "testView";
            target.ConvertId = "temp";
            target.ForwardHost = new FormBase();
            TERASOLUNA_Fw_Client_Forms_FormBaseAccessor formBaseAccessor =
                new TERASOLUNA_Fw_Client_Forms_FormBaseAccessor((FormBase)target.ForwardHost);
            formBaseAccessor._viewData = new DataSet();

            ConfigurationManager.AppSettings.Set(ConverterFactory.CONVERTER_TYPENAME,
                TEST_CONVERTER_TYPE);

            IForwardable forwardTarget = ViewFactory.CreateView(target.ViewId);
            Form targetForm = forwardTarget as Form;

            ForwardResult actual = target.Execute();

            Assert.AreEqual(ForwardResult.None, actual);
            Assert.AreEqual(0, FormForwarder_FormStub01._items.Count);
            Assert.IsTrue(FormForwarder_ConverterStub01.convertCalled);
            Assert.AreSame(target.ForwardHost.ViewData, FormForwarder_ConverterStub01.srcData);
            Assert.AreSame(FormForwarder_FormStub01._dataSet, FormForwarder_ConverterStub01.destData);
            Assert.AreEqual(target.ConvertId, FormForwarder_ConverterStub01.id);

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(FormForwarder), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckInfoUseMessageId(typeof(FormForwarder), "I_FORM_TRANSITION",
                                         Modality.Modeless.ToString(),
                                         target.ForwardHost.GetType().FullName,
                                         targetForm.GetType().FullName));

        }

        ///<summary>
        /// Execute()のテスト
        ///</summary>
        ///<remarks>
        /// ForwardHost に設定されたクラスのItems の要素数が1の場合、
        /// 正常に遷移することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Forms\FormForwarderTestData\FormForwarder_ViewConfigExternal01.config")]
        [DeploymentItem(@"Forms\FormForwarderTestData\FormForwarder_ViewConfigSection01.config")]
        public void ExecuteTest13()
        {
            ViewConfigurationSection section = ConfigTestHelper.GetViewConfigurationSectionFromCustomFile(VIEW_CONFIG_FILE);
            section.CreateViewElements();
            TERASOLUNA_Fw_Client_Configuration_TerasolunaClientConfigurationAccessor._viewConfigSection = section;

            FormForwarder target = new FormForwarder();

            TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor accessor =
                new TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor(target);

            accessor._modality = Modality.Modeless;
            accessor._ownerForm = null;
            target.ViewId = "testView";
            target.ConvertId = null;
            target.ForwardHost = new FormForwarder_FormStub02();
            FormForwarder_FormStub02._items = new Hashtable();
            FormForwarder_FormStub02._items.Add("aaa", "bbb");
            FormForwarder_FormStub01._items = new Hashtable();

            ConfigurationManager.AppSettings.Set(ConverterFactory.CONVERTER_TYPENAME,
                TEST_CONVERTER_TYPE);

            IForwardable forwardTarget = ViewFactory.CreateView(target.ViewId);
            Form targetForm = forwardTarget as Form;

            ForwardResult actual = target.Execute();

            Assert.AreEqual(ForwardResult.None, actual);
            Assert.AreEqual(1, FormForwarder_FormStub01._items.Count);
            Assert.AreEqual("bbb", FormForwarder_FormStub01._items["aaa"]);

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(FormForwarder), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckInfoUseMessageId(typeof(FormForwarder), "I_FORM_TRANSITION",
                                       Modality.Modeless.ToString(),
                                       target.ForwardHost.GetType().FullName,
                                       targetForm.GetType().FullName));
        }

        ///<summary>
        /// Execute () のテスト
        ///</summary>
        ///<remarks>
        /// ForwardHost に設定されたクラスのItems の要素数が3の場合、
        /// 正常に遷移することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Forms\FormForwarderTestData\FormForwarder_ViewConfigExternal01.config")]
        [DeploymentItem(@"Forms\FormForwarderTestData\FormForwarder_ViewConfigSection01.config")]
        public void ExecuteTest14()
        {
            ViewConfigurationSection section = ConfigTestHelper.GetViewConfigurationSectionFromCustomFile(VIEW_CONFIG_FILE);
            section.CreateViewElements();
            TERASOLUNA_Fw_Client_Configuration_TerasolunaClientConfigurationAccessor._viewConfigSection = section;

            FormForwarder target = new FormForwarder();

            TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor accessor =
                new TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor(target);

            accessor._modality = Modality.Modeless;
            accessor._ownerForm = null;
            target.ViewId = "testView";
            target.ConvertId = null;
            target.ForwardHost = new FormForwarder_FormStub02();
            FormForwarder_FormStub02._items = new Hashtable();
            FormForwarder_FormStub02._items.Add("aaa", "bbb");
            FormForwarder_FormStub02._items.Add("bbb", "bbb");
            FormForwarder_FormStub02._items.Add("ccc", "bbb");
            FormForwarder_FormStub01._items = new Hashtable();

            IForwardable forwardTarget = ViewFactory.CreateView(target.ViewId);
            Form targetForm = forwardTarget as Form;

            ForwardResult actual = target.Execute();

            Assert.AreEqual(ForwardResult.None, actual);
            Assert.AreEqual(3, FormForwarder_FormStub01._items.Count);
            Assert.AreEqual("bbb", FormForwarder_FormStub01._items["aaa"]);
            Assert.AreEqual("bbb", FormForwarder_FormStub01._items["bbb"]);
            Assert.AreEqual("bbb", FormForwarder_FormStub01._items["ccc"]);

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(FormForwarder), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckInfoUseMessageId(typeof(FormForwarder), "I_FORM_TRANSITION",
                                     Modality.Modeless.ToString(),
                                     target.ForwardHost.GetType().FullName,
                                     targetForm.GetType().FullName));

        }

        ///<summary>
        /// Execute () のテスト
        ///</summary>
        ///<remarks>
        /// _viewId で指定したクラスの Init メソッドの戻り値がfalseの場合、
        /// 正常に遷移することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Forms\FormForwarderTestData\FormForwarder_ViewConfigExternal01.config")]
        [DeploymentItem(@"Forms\FormForwarderTestData\FormForwarder_ViewConfigSection01.config")]
        public void ExecuteTest15()
        {
            ViewConfigurationSection section = ConfigTestHelper.GetViewConfigurationSectionFromCustomFile(VIEW_CONFIG_FILE);
            section.CreateViewElements();
            TERASOLUNA_Fw_Client_Configuration_TerasolunaClientConfigurationAccessor._viewConfigSection = section;

            FormForwarder target = new FormForwarder();

            TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor accessor =
                new TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor(target);

            accessor._modality = Modality.Modeless;
            accessor._ownerForm = null;
            target.ViewId = "testView";
            target.ConvertId = null;
            target.ForwardHost = new FormForwarder_FormStub02();
            FormForwarder_FormStub02._items.Clear();
            FormForwarder_FormStub01._initRet = false;

            ForwardResult actual = target.Execute();

            Assert.AreEqual(ForwardResult.Abort, actual);
            Assert.AreEqual(0, FormForwarder_FormStub01._items.Count);

            FormForwarder_FormStub01._initRet = true;

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(FormForwarder), LogLevel.INFO));

        }

        ///<summary>
        /// Execute () のテスト
        ///</summary>
        ///<remarks>
        /// ConvertIdがnull、
        /// 且つ生成されるクラスのViewDataがnullの場合、
        /// 正常に遷移することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Forms\FormForwarderTestData\FormForwarder_ViewConfigExternal01.config")]
        [DeploymentItem(@"Forms\FormForwarderTestData\FormForwarder_ViewConfigSection01.config")]
        public void ExecuteTest16()
        {
            ViewConfigurationSection section = ConfigTestHelper.GetViewConfigurationSectionFromCustomFile(VIEW_CONFIG_FILE);
            section.CreateViewElements();
            TERASOLUNA_Fw_Client_Configuration_TerasolunaClientConfigurationAccessor._viewConfigSection = section;

            FormForwarder target = new FormForwarder();

            TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor accessor =
                new TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor(target);

            accessor._modality = Modality.Modeless;
            accessor._ownerForm = null;
            target.ViewId = "testNoDataSet";
            target.ConvertId = null;
            target.ForwardHost = new FormForwarder_FormStub02();

            IForwardable forwardTarget = ViewFactory.CreateView(target.ViewId);
            Form targetForm = forwardTarget as Form;

            ForwardResult actual = target.Execute();

            Assert.AreEqual(ForwardResult.None, actual);
            Assert.AreEqual(0, FormForwarder_FormStub03._items.Count);
            Assert.IsFalse(FormForwarder_ConverterStub01.convertCalled);

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(FormForwarder), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckInfoUseMessageId(typeof(FormForwarder), "I_FORM_TRANSITION",
                                    Modality.Modeless.ToString(),
                                    target.ForwardHost.GetType().FullName,
                                    targetForm.GetType().FullName));

        }

        ///<summary>
        /// Execute () のテスト
        ///</summary>
        ///<remarks>
        /// Converterが存在するが、生成されるViewのDataSetがnullの場合
        /// 例外TerasolunaExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Forms\FormForwarderTestData\FormForwarder_ViewConfigExternal01.config")]
        [DeploymentItem(@"Forms\FormForwarderTestData\FormForwarder_ViewConfigSection01.config")]
        public void ExecuteTest17()
        {
            ViewConfigurationSection section = ConfigTestHelper.GetViewConfigurationSectionFromCustomFile(VIEW_CONFIG_FILE);
            section.CreateViewElements();
            TERASOLUNA_Fw_Client_Configuration_TerasolunaClientConfigurationAccessor._viewConfigSection = section;

            FormForwarder target = new FormForwarder();

            TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor accessor =
                new TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor(target);

            accessor._modality = Modality.Modeless;
            accessor._ownerForm = null;
            target.ViewId = "testNoDataSet";
            target.ConvertId = "temp";
            target.ForwardHost = new FormForwarder_FormStub02();

            ConfigurationManager.AppSettings.Set(ConverterFactory.CONVERTER_TYPENAME,
                TEST_CONVERTER_TYPE);

            IForwardable forwardTarget = ViewFactory.CreateView(target.ViewId);

            try
            {
                ForwardResult actual = target.Execute();
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // メッセージ確認
                Assert.AreEqual(MessageManager.GetMessage("E_FORMS_NULL_TARGET_OBJECT","ConvertId","ViewData"), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(FormForwarder), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(FormForwarder), typeof(TerasolunaException), "E_FORMS_NULL_TARGET_OBJECT","ConvertId","ViewData"));

        }

        ///<summary>
        /// Execute () のテスト
        ///</summary>
        ///<remarks>
        /// ConvertIdがnull、
        /// 且つForwardHostに設定されたクラスのViewDataがnullの場合、
        /// 正常に処理されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Forms\FormForwarderTestData\FormForwarder_ViewConfigExternal01.config")]
        [DeploymentItem(@"Forms\FormForwarderTestData\FormForwarder_ViewConfigSection01.config")]
        public void ExecuteTest18()
        {
            FormForwarder_FormStub01._initRet = false;
            ViewConfigurationSection section = ConfigTestHelper.GetViewConfigurationSectionFromCustomFile(VIEW_CONFIG_FILE);
            section.CreateViewElements();
            TERASOLUNA_Fw_Client_Configuration_TerasolunaClientConfigurationAccessor._viewConfigSection = section;

            FormForwarder target = new FormForwarder();

            TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor accessor =
                new TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor(target);

            accessor._modality = Modality.Modeless;
            accessor._ownerForm = null;
            target.ViewId = "testView";
            target.ConvertId = null;
            target.ForwardHost = new FormForwarder_FormStub02();
            FormForwarder_FormStub02._items.Clear();

            ForwardResult actual = target.Execute();

            Assert.AreEqual(ForwardResult.Abort, actual);
            Assert.AreEqual(0, FormForwarder_FormStub01._items.Count);
            Assert.IsFalse(FormForwarder_ConverterStub01.convertCalled);

            FormForwarder_FormStub02._dataSet = new DataSet();

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(FormForwarder), LogLevel.INFO));
        }

        ///<summary>
        /// Execute () のテスト
        ///</summary>
        ///<remarks>
        /// ConvertIdが空文字列ではない、
        /// 且つForwardHostに設定されたクラスのViewDataがnullの場合、
        /// 例外TerasolunaExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Forms\FormForwarderTestData\FormForwarder_ViewConfigExternal01.config")]
        [DeploymentItem(@"Forms\FormForwarderTestData\FormForwarder_ViewConfigSection01.config")]
        public void ExecuteTest19()
        {
            ViewConfigurationSection section = ConfigTestHelper.GetViewConfigurationSectionFromCustomFile(VIEW_CONFIG_FILE);
            section.CreateViewElements();
            TERASOLUNA.Fw.ClientUT.TERASOLUNA_Fw_Client_Configuration_TerasolunaClientConfigurationAccessor._viewConfigSection = section;

            FormForwarder target = new FormForwarder();

            TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor accessor =
                new TERASOLUNA_Fw_Client_Forms_FormForwarderAccessor(target);

            accessor._modality = Modality.Modeless;
            accessor._ownerForm = null;
            target.ViewId = "testView";
            target.ConvertId = "temp";
            target.ForwardHost = new FormForwarder_FormStub03();

            try
            {
                ForwardResult actual = target.Execute();
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // メッセージ確認
                Assert.AreEqual(MessageManager.GetMessage("E_FORMS_NULL_HOST_OBJECT", "ConvertId", "ViewData"), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(FormForwarder), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(FormForwarder), typeof(TerasolunaException), "E_FORMS_NULL_HOST_OBJECT", "ConvertId", "ViewData"));

        }

        ///<summary>
        /// Modality のテスト
        ///</summary>
        ///<remarks>
        /// 取得したModalityの値がセットした値と同じであることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ModalityTest01()
        {
            FormForwarder target = new FormForwarder();

            target.Modality = Modality.Modeless;

            Assert.AreEqual(Modality.Modeless, target.Modality);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(FormForwarder), LogLevel.INFO));

        }

        ///<summary>
        /// OwnerForm のテスト
        ///</summary>
        ///<remarks>
        /// 取得したOwnerFormの値がセットした値と同じであることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void OwnerFormTest01()
        {
            FormForwarder target = new FormForwarder();

            Form val = new FormBase();

            target.OwnerForm = val;

            Assert.AreSame(val, target.OwnerForm);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(FormForwarder), LogLevel.INFO));

        }

        private void FormClosed1(object sender, ForwardableFormCloseEventArgs e)
        {
            this._formClosed1Called = true;
        }

        private void FormClosed2(object sender, ForwardableFormCloseEventArgs e)
        {
            this._formClosed2Called = true;
        }

        private void FormClosed3(object sender, ForwardableFormCloseEventArgs e)
        {
            this._formClosed3Called = true;
        }

        private void FormClosing1(object sender, ForwardableFormCloseEventArgs e)
        {
            this._formClosing1Called = true;
        }

        private void FormClosing2(object sender, ForwardableFormCloseEventArgs e)
        {
            this._formClosing2Called = true;
        }

        private void FormClosing3(object sender, ForwardableFormCloseEventArgs e)
        {
            this._formClosing3Called = true;
        }
    }

    ///<summary>
    /// Converterのスタブ。
    ///</summary>
    public class FormForwarder_ConverterStub01 : IConverter
    {
        public static bool reverseCalled = false;
        public static bool convertCalled = false;
        public static string id;
        public static DataSet srcData;
        public static DataSet destData;

        public static void Reset()
        {
            reverseCalled = false;
            convertCalled = false;
        }

        public void Convert(string id, DataSet srcData, DataSet destData)
        {
            FormForwarder_ConverterStub01.convertCalled = true;
            FormForwarder_ConverterStub01.id = id;
            FormForwarder_ConverterStub01.srcData = srcData;
            FormForwarder_ConverterStub01.destData = destData;
        }

        public void Reverse(string id, DataSet srcData, DataSet destData)
        {
            reverseCalled = true;
            FormForwarder_ConverterStub01.id = id;
            FormForwarder_ConverterStub01.srcData = srcData;
            FormForwarder_ConverterStub01.destData = destData;
        }
    }

    ///<summary>
    /// 正常系を通すためのスタブ。
    ///</summary>
    public class FormForwarder_FormStub01 : Form, IForwardable
    {
        public static IDictionary _items = new Hashtable();
        public static DataSet _dataSet = new DataSet();
        public static bool _initRet = true;

        #region IForwardable メンバ

        public IDictionary Items
        {
            get
            {
                return _items;
            }
            set
            {
                _items = value;
            }
        }

        public DataSet ViewData
        {
            get
            {
                return _dataSet;
            }
        }

        public bool Init(IForwardable forwardable)
        {
            return _initRet;
        }

        #endregion
    }

    ///<summary>
    /// ForwardHostとして利用されるスタブ。
    ///</summary>
    public class FormForwarder_FormStub02 : Form, IForwardable
    {
        public static IDictionary _items = new Hashtable();
        public static DataSet _dataSet = new DataSet();

        public IDictionary Items
        {
            get
            {
                return _items;
            }
            set
            {
                _items = value;
            }
        }

        public DataSet ViewData
        {
            get
            {
                return _dataSet;
            }
        }

        public bool Init(IForwardable forwardable)
        {
            return true;
        }
    }

    ///<summary>
    /// DataSetがnullとなるスタブ。
    ///</summary>
    public class FormForwarder_FormStub03 : Form, IForwardable
    {
        public static IDictionary _items = new Hashtable();

        #region IForwardable メンバ

        public bool Init(IForwardable forwardable)
        {
            return true;
        }

        public IDictionary Items
        {
            get
            {
                return _items;
            }
            set
            {
                _items = value;
            }
        }

        public DataSet ViewData
        {
            get { return null; }
        }

        #endregion

    }

    ///<summary>
    /// Formを継承しないスタブ。
    ///</summary>
    public class FormForwarder_ForwardableStub01 : IForwardable
    {
        public static bool _init = false;
        public static IDictionary _items = new Hashtable();
        public static DataSet _dataSet = new DataSet();

        public System.Collections.IDictionary Items
        {
            get { return _items; }
        }

        public IForwardable ParentForwardable
        {
            get
            {
                return null;
            }
            set
            {

            }
        }

        public DataSet ViewData
        {
            get
            {
                return _dataSet;
            }
        }

        public bool Init(IForwardable forwardable)
        {
            return _init;
        }
    }

    /// <summary>
    /// Constructor のテストで利用する IContainer のスタブクラスです。
    /// </summary>
    public class FormForwarder_IContainerStub01 : IContainer
    {
        private IComponent _component;

        public IComponent Component
        {
            get { return _component; }
            set { _component = value; }
        }

        public void Add(IComponent component, string name)
        {
            throw new Exception("The method or operation is not implemented.");
        }

        public void Add(IComponent component)
        {
            this._component = component;
        }

        public ComponentCollection Components
        {
            get { throw new Exception("The method or operation is not implemented."); }
        }

        public void Remove(IComponent component)
        {
            throw new Exception("The method or operation is not implemented.");
        }

        public void Dispose()
        {
            throw new Exception("The method or operation is not implemented.");
        }
    }
}