// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Text;
using System.Collections.Generic;
using TERASOLUNA.Fw.Client;
using System.ComponentModel;
using TERASOLUNA.Fw.Common.BLogic;
using System.Data;
using TERASOLUNA.Fw.Client.Conversion;
using TERASOLUNA.Fw.Common.Validation;
using TERASOLUNA.Fw.Utlib.Logging;
using TERASOLUNA.Fw.Common.Logging;
using TERASOLUNA.Fw.ClientUT.CoordinatorTestData;
using System.Configuration;
using TERASOLUNA.Fw.Common;
using System.Reflection;
namespace TERASOLUNA.Fw.ClientUT
{
    /// <summary>
    /// TERASOLUNA.Fw.Client.Coordinator のテストクラスです。
    /// </summary>
    [TestClass()]
    public class CoordinatorTest
    {

        private TestContext testContextInstance;

        private readonly Type _blogicParamType = typeof(Coordinator_DataSetStub01);
        private readonly string _blogicResultTypeName = typeof(Coordinator_DataSetStub02).AssemblyQualifiedName;
        private readonly string _blogicName = "blogicName";
        private readonly string _requestName = "requestName";
        private readonly Type _viewDataType = typeof(Coordinator_DataSetStub03);

        /// <summary>
        ///現在のテストの実行についての情報および機能を
        ///提供するテスト コンテキストを取得または設定します。
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }
        #region 追加のテスト属性
        // 
        //テストを作成するときに、次の追加属性を使用することができます:
        //
        //クラスの最初のテストを実行する前にコードを実行するには、ClassInitialize を使用
        //
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //クラスのすべてのテストを実行した後にコードを実行するには、ClassCleanup を使用
        //
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        
        //各テストを実行する前にコードを実行するには、TestInitialize を使用
        [TestInitialize()]
        public void MyTestInitialize()
        {
            LogUTUtil.Clear();
            ConfigurationManager.AppSettings["BLogicFactoryTypeName"] = typeof(Coordinator_BLogicFactoryStub01).AssemblyQualifiedName;
            ConfigurationManager.AppSettings["ValidatorTypeName"] = typeof(Coordinator_IValidatorImpl01).AssemblyQualifiedName;
            ConfigurationManager.AppSettings["ConverterTypeName"] = typeof(Coordinator_IConverterImpl01).AssemblyQualifiedName;
            ConfigurationManager.AppSettings["CoordinatorTypeName"] = null;
            TERASOLUNA_Fw_Common_Validation_ValidatorFactoryAccessor._factory = null;
            TERASOLUNA_Fw_Client_Conversion_ConverterFactoryAccessor._factory = null;
            TERASOLUNA_Fw_Client_CoordinatorAccessor._coordinatorType = null;
        }
        //
        //各テストを実行した後にコードを実行するには、TestCleanup を使用
        //
        [TestCleanup()]
        public void MyTestCleanup()
        {
            ConfigurationManager.AppSettings["BLogicFactoryTypeName"] = null;
            ConfigurationManager.AppSettings["ValidatorTypeName"] = null;
            ConfigurationManager.AppSettings["ConverterTypeName"] = null;
            ConfigurationManager.AppSettings["CoordinatorTypeName"] = null;
            TERASOLUNA_Fw_Common_Validation_ValidatorFactoryAccessor._factory = null;
            TERASOLUNA_Fw_Client_Conversion_ConverterFactoryAccessor._factory = null;
            TERASOLUNA_Fw_Client_CoordinatorAccessor._coordinatorType = null;
        }
        
        #endregion


        /// <summary>
        /// Execute (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController が null の場合、ArgumentNullException
        /// がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest01()
        {
            Coordinator target = new Coordinator();

            EventController eventController = null;

            try
            {
                // 実施
                target.Execute(eventController);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // 成功
                Assert.AreEqual("eventController", e.ParamName);
            }
            catch (Exception e)
            {
                Assert.Fail("予期せぬエラーです。" + e.ToString());
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(
                typeof(Coordinator), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "eventController"));
        }

        /// <summary>
        /// Execute (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController が null でない場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest02()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            ExecutionResult actual = target.Execute(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed); 

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
            Assert.AreEqual(_blogicName, Coordinator_BLogicFactoryStub01._blogicName);
            Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
            Assert.AreEqual(_blogicResultTypeName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

            // _blogic.Execute() が呼び出されており、正しい引数がわたっていること
            Coordinator_IBLogicImpl01 blogicImpl = (Coordinator_IBLogicImpl01)accessor._blogic;
            Assert.IsTrue(blogicImpl.IsCalledExecute);
            Assert.AreEqual(1, blogicImpl.ExecuteBLogicParam.Items.Count);
            Assert.AreEqual(_requestName, blogicImpl.ExecuteBLogicParam.Items[EventController.REQUEST_NAME]);
            Assert.IsInstanceOfType(blogicImpl.ExecuteBLogicParam.ParamData, _blogicParamType);

            // _converter.Reverseメソッドが呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledReverse);

            // eventController.ReportBLogicFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportBLogicFailed);

            // eventController.Itemsが変更されていないこと
            Assert.AreEqual(0, eventController.Items.Count);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 戻り値のexecutionResult.Errorsが_blogic.Execute()の戻り値のBlogicResult.Errorsと要素が同一
            Assert.AreEqual(0, actual.Errors.Count);

            // 戻り値のexecutionResult.ResultStringが_blogic.Execute()の戻り値のBLogicResult.ResultStringと同一
            Assert.AreEqual(BLogicResult.SUCCESS, actual.ResultString);

            // 戻り値のexecutionResult.BLogicResultDataが_blogic.Execute()の戻り値のBLogicResult.ResultDataと同一
            Assert.IsInstanceOfType(actual.BLogicResultData, typeof(Coordinator_DataSetStub04));

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// eventController.ErrorHandler が null でない場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest03()
        {
            // 準備
            Coordinator target = new Coordinator();
    
            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            Coordinator_IErrorHandlerImpl01 errorHandler = new Coordinator_IErrorHandlerImpl01();
            eventController.ErrorHandler = errorHandler;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            ExecutionResult actual = target.Execute(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // eventController.ErrorHandlerのClearErrorメソッドが呼び出され、正しい引数がわたっていること
            Assert.IsTrue(errorHandler.IsCalledClearError);
            Assert.AreSame(eventController.ViewData, errorHandler.ClearErrorView);
            Assert.IsInstanceOfType(errorHandler.ClearErrorView, _viewDataType);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ErrorHandler.HandleError() が呼ばれないこと
            Assert.IsFalse(errorHandler.IsCalledHandleError);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
            Assert.AreEqual(_blogicName, Coordinator_BLogicFactoryStub01._blogicName);
            Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
            Assert.AreEqual(_blogicResultTypeName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

            // _blogic.Execute() が呼び出されており、正しい引数がわたっていること
            Coordinator_IBLogicImpl01 blogicImpl = (Coordinator_IBLogicImpl01)accessor._blogic;
            Assert.IsTrue(blogicImpl.IsCalledExecute);
            Assert.AreEqual(1, blogicImpl.ExecuteBLogicParam.Items.Count);
            Assert.AreEqual(_requestName, blogicImpl.ExecuteBLogicParam.Items[EventController.REQUEST_NAME]);
            Assert.IsInstanceOfType(blogicImpl.ExecuteBLogicParam.ParamData, _blogicParamType);

            // _converter.Reverseメソッドが呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledReverse);

            // eventController.ReportBLogicFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportBLogicFailed);

            // eventController.Itemsが変更されていないこと
            Assert.AreEqual(0, eventController.Items.Count);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 戻り値のexecutionResult.Errorsが_blogic.Execute()の戻り値のBlogicResult.Errorsと要素が同一
            Assert.AreEqual(0, actual.Errors.Count);

            // 戻り値のexecutionResult.ResultStringが_blogic.Execute()の戻り値のBLogicResult.ResultStringと同一
            Assert.AreEqual(BLogicResult.SUCCESS, actual.ResultString);

            // 戻り値のexecutionResult.BLogicResultDataが_blogic.Execute()の戻り値のBLogicResult.ResultDataと同一
            Assert.IsInstanceOfType(actual.BLogicResultData, typeof(Coordinator_DataSetStub04));

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// eventController.ErrorHandler が null でなく、
        /// eventController.ErrorPaths のリスト数が 1 の場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest04()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            Coordinator_IErrorHandlerImpl01 errorHandler = new Coordinator_IErrorHandlerImpl01();
            eventController.ErrorHandler = errorHandler;
            string errorPath01 = "errorPath01";
            eventController.ErrorPaths.Add(errorPath01);

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            ExecutionResult actual = target.Execute(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // eventController.ErrorHandlerのClearErrorメソッドが呼び出され、正しい引数がわたっていること
            Assert.IsTrue(errorHandler.IsCalledClearError);
            Assert.IsInstanceOfType(errorHandler.ClearErrorView, _viewDataType);
            Assert.AreSame(eventController.ViewData, errorHandler.ClearErrorView);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ErrorHandler.HandleError() が呼ばれないこと
            Assert.IsFalse(errorHandler.IsCalledHandleError);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
            Assert.AreEqual(_blogicName, Coordinator_BLogicFactoryStub01._blogicName);
            Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
            Assert.AreEqual(_blogicResultTypeName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

            // _blogic.Execute() が呼び出されており、正しい引数がわたっていること
            Coordinator_IBLogicImpl01 blogicImpl = (Coordinator_IBLogicImpl01)accessor._blogic;
            Assert.IsTrue(blogicImpl.IsCalledExecute);
            Assert.AreEqual(1, blogicImpl.ExecuteBLogicParam.Items.Count);
            Assert.AreEqual(_requestName, blogicImpl.ExecuteBLogicParam.Items[EventController.REQUEST_NAME]);
            Assert.IsInstanceOfType(blogicImpl.ExecuteBLogicParam.ParamData, _blogicParamType);

            // _converter.Reverseメソッドが呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledReverse);

            // eventController.ReportBLogicFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportBLogicFailed);

            // eventController.Itemsが変更されていないこと
            Assert.AreEqual(0, eventController.Items.Count);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 戻り値のexecutionResult.Errorsが_blogic.Execute()の戻り値のBlogicResult.Errorsと要素が同一
            Assert.AreEqual(0, actual.Errors.Count);

            // 戻り値のexecutionResult.ResultStringが_blogic.Execute()の戻り値のBLogicResult.ResultStringと同一
            Assert.AreEqual(BLogicResult.SUCCESS, actual.ResultString);

            // 戻り値のexecutionResult.BLogicResultDataが_blogic.Execute()の戻り値のBLogicResult.ResultDataと同一
            Assert.IsInstanceOfType(actual.BLogicResultData, typeof(Coordinator_DataSetStub04));

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// eventController.ErrorHandler が null でなく、
        /// eventController.ErrorPaths のリスト数が 3 の場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest05()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            Coordinator_IErrorHandlerImpl01 errorHandler = new Coordinator_IErrorHandlerImpl01();
            eventController.ErrorHandler = errorHandler;
            string errorPath01 = "errorPath01";
            string errorPath02 = "errorPath02";
            string errorPath03 = "errorPath03";
            eventController.ErrorPaths.Add(errorPath01);
            eventController.ErrorPaths.Add(errorPath02);
            eventController.ErrorPaths.Add(errorPath03);

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            ExecutionResult actual = target.Execute(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // eventController.ErrorHandlerのClearErrorメソッドが呼び出され、正しい引数がわたっていること
            Assert.IsTrue(errorHandler.IsCalledClearError);
            Assert.IsInstanceOfType(errorHandler.ClearErrorView, _viewDataType);
            Assert.AreSame(eventController.ViewData, errorHandler.ClearErrorView);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ErrorHandler.HandleError() が呼ばれないこと
            Assert.IsFalse(errorHandler.IsCalledHandleError);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
            Assert.AreEqual(_blogicName, Coordinator_BLogicFactoryStub01._blogicName);
            Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
            Assert.AreEqual(_blogicResultTypeName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

            // _blogic.Execute() が呼び出されており、正しい引数がわたっていること
            Coordinator_IBLogicImpl01 blogicImpl = (Coordinator_IBLogicImpl01)accessor._blogic;
            Assert.IsTrue(blogicImpl.IsCalledExecute);
            Assert.AreEqual(1, blogicImpl.ExecuteBLogicParam.Items.Count);
            Assert.AreEqual(_requestName, blogicImpl.ExecuteBLogicParam.Items[EventController.REQUEST_NAME]);
            Assert.IsInstanceOfType(blogicImpl.ExecuteBLogicParam.ParamData, _blogicParamType);

            // _converter.Reverseメソッドが呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledReverse);

            // eventController.ReportBLogicFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportBLogicFailed);

            // eventController.Itemsが変更されていないこと
            Assert.AreEqual(0, eventController.Items.Count);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 戻り値のexecutionResult.Errorsが_blogic.Execute()の戻り値のBlogicResult.Errorsと要素が同一
            Assert.AreEqual(0, actual.Errors.Count);

            // 戻り値のexecutionResult.ResultStringが_blogic.Execute()の戻り値のBLogicResult.ResultStringと同一
            Assert.AreEqual(BLogicResult.SUCCESS, actual.ResultString);

            // 戻り値のexecutionResult.BLogicResultDataが_blogic.Execute()の戻り値のBLogicResult.ResultDataと同一
            Assert.IsInstanceOfType(actual.BLogicResultData, typeof(Coordinator_DataSetStub04));

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// IValidator 実装クラスのインスタンスを生成できない場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest06()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            ConfigurationManager.AppSettings["ValidatorTypeName"] = typeof(Coordinator_IValidatorImpl02).AssemblyQualifiedName;

            try
            {
                // 実施
                target.Execute(eventController);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // 成功
                Assert.AreEqual(MessageManager.GetMessage("E_MEMBER_ACCESS_EXCEPTION", typeof(Coordinator_IValidatorImpl02).AssemblyQualifiedName), e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("予期せぬエラーです。" + e.ToString());
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// IConverter 実装クラスのインスタンスを生成できない場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest07()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            ConfigurationManager.AppSettings["ConverterTypeName"] = typeof(Coordinator_IConverterImpl02).AssemblyQualifiedName;

            try
            {
                // 実施
                target.Execute(eventController);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // 成功
                Assert.AreEqual(MessageManager.GetMessage("E_MEMBER_ACCESS_EXCEPTION", typeof(Coordinator_IConverterImpl02).AssemblyQualifiedName), e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("予期せぬエラーです。" + e.ToString());
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.ValidationFilePath が空文字列の場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest08()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            eventController.ValidationFilePath = string.Empty;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            ExecutionResult actual = target.Execute(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
            Assert.AreEqual(_blogicName, Coordinator_BLogicFactoryStub01._blogicName);
            Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
            Assert.AreEqual(_blogicResultTypeName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

            // _blogic.Execute() が呼び出されており、正しい引数がわたっていること
            Coordinator_IBLogicImpl01 blogicImpl = (Coordinator_IBLogicImpl01)accessor._blogic;
            Assert.IsTrue(blogicImpl.IsCalledExecute);
            Assert.AreEqual(1, blogicImpl.ExecuteBLogicParam.Items.Count);
            Assert.AreEqual(_requestName, blogicImpl.ExecuteBLogicParam.Items[EventController.REQUEST_NAME]);
            Assert.IsInstanceOfType(blogicImpl.ExecuteBLogicParam.ParamData, _blogicParamType);

            // _converter.Reverseメソッドが呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledReverse);

            // eventController.ReportBLogicFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportBLogicFailed);

            // eventController.Itemsが変更されていないこと
            Assert.AreEqual(0, eventController.Items.Count);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 戻り値のexecutionResult.Errorsが_blogic.Execute()の戻り値のBlogicResult.Errorsと要素が同一
            Assert.AreEqual(0, actual.Errors.Count);

            // 戻り値のexecutionResult.ResultStringが_blogic.Execute()の戻り値のBLogicResult.ResultStringと同一
            Assert.AreEqual(BLogicResult.SUCCESS, actual.ResultString);

            // 戻り値のexecutionResult.BLogicResultDataが_blogic.Execute()の戻り値のBLogicResult.ResultDataと同一
            Assert.IsInstanceOfType(actual.BLogicResultData, typeof(Coordinator_DataSetStub04));

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.ValidationFilePath が null・空文字列でなく、
        /// eventController.ViewData が null の場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest09()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            string validationFilePath = "validationFilePath";
            eventController.ValidationFilePath = validationFilePath;
            eventController.ViewData = null;

            try
            {
                // 実施
                target.Execute(eventController);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // 成功
                Assert.AreEqual(MessageManager.GetMessage("E_BLOGIC_NULL_DATASET", "ValidationFilePath", "ViewData"), e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("予期せぬエラーです。" + e.ToString());
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(Coordinator), typeof(TerasolunaException), "E_BLOGIC_NULL_DATASET", "ValidationFilePath", "ViewData"));
        }

        /// <summary>
        /// Execute (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.ValidationFilePath が null・空文字列でなく、
        /// eventController.ViewData が null でない場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest10()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            string validationFilePath = "validationFilePath";
            eventController.ValidationFilePath = validationFilePath;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            ExecutionResult actual = target.Execute(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // validator.ValidationFilePathにeventController.ValidationFilePathの値が設定されていること
            Assert.AreEqual(validationFilePath, accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validateメソッドが呼び出され、正しい引数がわたっていること
            Coordinator_IValidatorImpl01 validatorImpl = (Coordinator_IValidatorImpl01)accessor._validator;
            Assert.IsTrue(validatorImpl.IsCalledValidate);
            Assert.AreSame(eventController.ViewData, validatorImpl.ValidateDataSet);
            Assert.IsInstanceOfType(validatorImpl.ValidateDataSet, _viewDataType);

            // eventController.ReportCustomValidatingメソッドが呼び出され引数が_validator.Validate()の戻り値と等しいこと
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
            Assert.AreEqual(_blogicName, Coordinator_BLogicFactoryStub01._blogicName);
            Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
            Assert.AreEqual(_blogicResultTypeName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

            // _blogic.Execute() が呼び出されており、正しい引数がわたっていること
            Coordinator_IBLogicImpl01 blogicImpl = (Coordinator_IBLogicImpl01)accessor._blogic;
            Assert.IsTrue(blogicImpl.IsCalledExecute);
            Assert.AreEqual(1, blogicImpl.ExecuteBLogicParam.Items.Count);
            Assert.AreEqual(_requestName, blogicImpl.ExecuteBLogicParam.Items[EventController.REQUEST_NAME]);
            Assert.IsInstanceOfType(blogicImpl.ExecuteBLogicParam.ParamData, _blogicParamType);

            // _converter.Reverseメソッドが呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledReverse);

            // eventController.ReportBLogicFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportBLogicFailed);

            // eventController.Itemsが変更されていないこと
            Assert.AreEqual(0, eventController.Items.Count);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 戻り値のexecutionResult.Errorsが_blogic.Execute()の戻り値のBlogicResult.Errorsと要素が同一
            Assert.AreEqual(0, actual.Errors.Count);

            // 戻り値のexecutionResult.ResultStringが_blogic.Execute()の戻り値のBLogicResult.ResultStringと同一
            Assert.AreEqual(BLogicResult.SUCCESS, actual.ResultString);

            // 戻り値のexecutionResult.BLogicResultDataが_blogic.Execute()の戻り値のBLogicResult.ResultDataと同一
            Assert.IsInstanceOfType(actual.BLogicResultData, typeof(Coordinator_DataSetStub04));

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 入力値検証機能による単項目チェックにおいてエラー数が 1 の場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest11()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            string validationFilePath = "validationFilePath";
            eventController.ValidationFilePath = validationFilePath;

            ConfigurationManager.AppSettings["ValidatorTypeName"] = typeof(Coordinator_IValidatorImpl03).AssemblyQualifiedName;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            ExecutionResult actual = target.Execute(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl03));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // validator.ValidationFilePathにeventController.ValidationFilePathの値が設定されていること
            Assert.AreEqual(validationFilePath, accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validateメソッドが呼び出され、正しい引数がわたっていること
            Coordinator_IValidatorImpl03 validatorImpl = (Coordinator_IValidatorImpl03)accessor._validator;
            Assert.IsTrue(validatorImpl.IsCalledValidate);
            Assert.AreSame(eventController.ViewData, validatorImpl.ValidateDataSet);
            Assert.IsInstanceOfType(validatorImpl.ValidateDataSet, _viewDataType);

            // eventController.ReportCustomValidatingメソッドが呼び出され引数が_validator.Validate()の戻り値と等しいこと
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(1, eventController.ReportCustomValidatingValidationResult.Errors.Count);
            Assert.AreSame(validatorImpl.ValidateValidationMessageInfo01, eventController.ReportCustomValidatingValidationResult.Errors[0]);

            // eventController.ErrorPathsにvalidationResult.Errorsの各ValidationMessageInfoのパスが追加されていること
            Assert.AreEqual(1, eventController.ErrorPaths.Count);
            Assert.AreEqual("errorPath01", eventController.ErrorPaths[0]);

            // eventController.ReportValidationFailedメソッドが呼び出され、正しい引数がわたってきていること
            Assert.IsTrue(eventController.IsCalledReportValidationFailed);
            Assert.AreEqual(1, eventController.ReportValidationFailedExecutionResult.Errors.Count);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Key, eventController.ReportValidationFailedExecutionResult.Errors[0].Key);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Message, eventController.ReportValidationFailedExecutionResult.Errors[0].Message);
            Assert.AreEqual(Coordinator.VALIDATION_FAILED, eventController.ReportValidationFailedExecutionResult.ResultString);
            Assert.IsNull(eventController.ReportValidationFailedExecutionResult.BLogicResultData);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessed);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // BLogicFactory.CreateBLogic() が呼び出されていないこと
            Assert.IsFalse(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);

            // _converter.Reverseメソッドが呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledReverse);

            // eventController.ReportBLogicFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportBLogicFailed);

            // eventController.Itemsが変更されていないこと
            Assert.AreEqual(0, eventController.Items.Count);

            // 戻り値のexecutionResult.ErrorsとvalidationResult.Errorsの要素が同一
            Assert.AreEqual(1, actual.Errors.Count);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Key, actual.Errors[0].Key);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Message, actual.Errors[0].Message);

            // 戻り値のexecutionResult.ResultStringがVALIDATION_FAILED
            Assert.AreEqual(Coordinator.VALIDATION_FAILED, actual.ResultString);

            // 戻り値のexecutionResult.BLogicResultDataがnull
            Assert.IsNull(actual.BLogicResultData);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 入力値検証機能による単項目チェックにおいてエラー数が 3 の場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest12()
        {
            // 準備
            Coordinator target = new Coordinator();
            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();

            string validationFilePath = "validationFilePath";
            eventController.ValidationFilePath = validationFilePath;

            ConfigurationManager.AppSettings["ValidatorTypeName"] = typeof(Coordinator_IValidatorImpl04).AssemblyQualifiedName;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            ExecutionResult actual = target.Execute(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl04));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // validator.ValidationFilePathにeventController.ValidationFilePathの値が設定されていること
            Assert.AreEqual(validationFilePath, accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validateメソッドが呼び出され、正しい引数がわたっていること
            Coordinator_IValidatorImpl04 validatorImpl = (Coordinator_IValidatorImpl04)accessor._validator;
            Assert.IsTrue(validatorImpl.IsCalledValidate);
            Assert.AreSame(eventController.ViewData, validatorImpl.ValidateDataSet);
            Assert.IsInstanceOfType(validatorImpl.ValidateDataSet, _viewDataType);

            // eventController.ReportCustomValidatingメソッドが呼び出され引数が_validator.Validate()の戻り値と等しいこと
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(3, eventController.ReportCustomValidatingValidationResult.Errors.Count);
            Assert.AreSame(validatorImpl.ValidateValidationMessageInfo01, eventController.ReportCustomValidatingValidationResult.Errors[0]);
            Assert.AreSame(validatorImpl.ValidateValidationMessageInfo02, eventController.ReportCustomValidatingValidationResult.Errors[1]);
            Assert.AreSame(validatorImpl.ValidateValidationMessageInfo03, eventController.ReportCustomValidatingValidationResult.Errors[2]);

            // eventController.ErrorPathsにvalidationResult.Errorsの各ValidationMessageInfoのパスが追加されていること
            Assert.AreEqual(3, eventController.ErrorPaths.Count);
            Assert.AreEqual("errorPath01", eventController.ErrorPaths[0]);
            Assert.AreEqual("errorPath02", eventController.ErrorPaths[1]);
            Assert.AreEqual("errorPath03", eventController.ErrorPaths[2]);

            // eventController.ReportValidationFailedメソッドが呼び出され、正しい引数がわたってきていること
            Assert.IsTrue(eventController.IsCalledReportValidationFailed);
            Assert.AreEqual(3, eventController.ReportValidationFailedExecutionResult.Errors.Count);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Key, eventController.ReportValidationFailedExecutionResult.Errors[0].Key);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo02.Key, eventController.ReportValidationFailedExecutionResult.Errors[1].Key);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo03.Key, eventController.ReportValidationFailedExecutionResult.Errors[2].Key);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Message, eventController.ReportValidationFailedExecutionResult.Errors[0].Message);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo02.Message, eventController.ReportValidationFailedExecutionResult.Errors[1].Message);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo03.Message, eventController.ReportValidationFailedExecutionResult.Errors[2].Message);
            
            Assert.AreEqual(Coordinator.VALIDATION_FAILED, eventController.ReportValidationFailedExecutionResult.ResultString);
            Assert.IsNull(eventController.ReportValidationFailedExecutionResult.BLogicResultData);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessed);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // BLogicFactory.CreateBLogic() が呼び出されていないこと
            Assert.IsFalse(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);

            // _converter.Reverseメソッドが呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledReverse);

            // eventController.ReportBLogicFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportBLogicFailed);

            // eventController.Itemsが変更されていないこと
            Assert.AreEqual(0, eventController.Items.Count);

            // 戻り値のexecutionResult.ErrorsとvalidationResult.Errorsの要素が同一
            Assert.AreEqual(3, actual.Errors.Count);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Key, actual.Errors[0].Key);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo02.Key, actual.Errors[1].Key);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo03.Key, actual.Errors[2].Key);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Message, actual.Errors[0].Message);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo02.Message, actual.Errors[1].Message);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo03.Message, actual.Errors[2].Message);

            // 戻り値のexecutionResult.ResultStringがVALIDATION_FAILED
            Assert.AreEqual(Coordinator.VALIDATION_FAILED, actual.ResultString);

            // 戻り値のexecutionResult.BLogicResultDataがnull
            Assert.IsNull(actual.BLogicResultData);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.EventHander が null でなく、
        /// 入力値検証機能による単項目チェックにおいてエラー数が 1 の場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest13()
        {
            // 準備
            Coordinator target = new Coordinator();
            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();

            string validationFilePath = "validationFilePath";
            eventController.ValidationFilePath = validationFilePath;

            Coordinator_IErrorHandlerImpl01 errorHandler = new Coordinator_IErrorHandlerImpl01();
            eventController.ErrorHandler = errorHandler;

            ConfigurationManager.AppSettings["ValidatorTypeName"] = typeof(Coordinator_IValidatorImpl03).AssemblyQualifiedName;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            ExecutionResult actual = target.Execute(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // eventController.ErrorHandlerのClearErrorメソッドが呼び出され、正しい引数がわたっていること
            Assert.IsTrue(errorHandler.IsCalledClearError);
            Assert.AreSame(eventController.ViewData, errorHandler.ClearErrorView);
            Assert.IsInstanceOfType(errorHandler.ClearErrorView, _viewDataType);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl03));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // validator.ValidationFilePathにeventController.ValidationFilePathの値が設定されていること
            Assert.AreEqual(validationFilePath, accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validateメソッドが呼び出され、正しい引数がわたっていること
            Coordinator_IValidatorImpl03 validatorImpl = (Coordinator_IValidatorImpl03)accessor._validator;
            Assert.IsTrue(validatorImpl.IsCalledValidate);
            Assert.AreSame(eventController.ViewData, validatorImpl.ValidateDataSet);
            Assert.IsInstanceOfType(validatorImpl.ValidateDataSet, _viewDataType);

            // eventController.ReportCustomValidatingメソッドが呼び出され引数が_validator.Validate()の戻り値と等しいこと
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(1, eventController.ReportCustomValidatingValidationResult.Errors.Count);
            Assert.AreSame(validatorImpl.ValidateValidationMessageInfo01, eventController.ReportCustomValidatingValidationResult.Errors[0]);

            // eventController.ErrorHandler.HandleError()が呼ばれ引数が正しいこと
            Assert.IsTrue(errorHandler.IsCalledHandleError);
            Assert.AreEqual(Coordinator.VALIDATION_FAILED, errorHandler.HandleErrorResultString);
            Assert.AreEqual(1, errorHandler.HandleErrorMessages.Count);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Key, errorHandler.HandleErrorMessages[0].Key);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Message, errorHandler.HandleErrorMessages[0].Message);
            Assert.AreSame(eventController.ViewData, errorHandler.HandleErrorView);
            Assert.IsInstanceOfType(errorHandler.HandleErrorView, _viewDataType);

            // eventController.ErrorPathsにvalidationResult.Errorsの各ValidationMessageInfoのパスが追加されていること
            Assert.AreEqual(1, eventController.ErrorPaths.Count);
            Assert.AreEqual("errorPath01", eventController.ErrorPaths[0]);

            // eventController.ReportValidationFailedメソッドが呼び出され、正しい引数がわたってきていること
            Assert.IsTrue(eventController.IsCalledReportValidationFailed);
            Assert.AreEqual(1, eventController.ReportValidationFailedExecutionResult.Errors.Count);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Key, eventController.ReportValidationFailedExecutionResult.Errors[0].Key);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Message, eventController.ReportValidationFailedExecutionResult.Errors[0].Message);
            Assert.AreEqual(Coordinator.VALIDATION_FAILED, eventController.ReportValidationFailedExecutionResult.ResultString);
            Assert.IsNull(eventController.ReportValidationFailedExecutionResult.BLogicResultData);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessed);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // BLogicFactory.CreateBLogic() が呼び出されていないこと
            Assert.IsFalse(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);

            // _converter.Reverseメソッドが呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledReverse);

            // eventController.ReportBLogicFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportBLogicFailed);

            // eventController.Itemsが変更されていないこと
            Assert.AreEqual(0, eventController.Items.Count);

            // 戻り値のexecutionResult.ErrorsとvalidationResult.Errorsの要素が同一
            Assert.AreEqual(1, actual.Errors.Count);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Key, actual.Errors[0].Key);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Message, actual.Errors[0].Message);

            // 戻り値のexecutionResult.ResultStringがVALIDATION_FAILED
            Assert.AreEqual(Coordinator.VALIDATION_FAILED, actual.ResultString);

            // 戻り値のexecutionResult.BLogicResultDataがnull
            Assert.IsNull(actual.BLogicResultData);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 入力値検証機能による単項目チェックにおいてエラー数が 1 であり
        /// カスタムチェックイベントによる入力値検証においてエラー数が 1 の場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest14()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub02 eventController = new Coordinator_EventControllerStub02();
            string validationFilePath = "validationFilePath";
            eventController.ValidationFilePath = validationFilePath;

            ConfigurationManager.AppSettings["ValidatorTypeName"] = typeof(Coordinator_IValidatorImpl03).AssemblyQualifiedName;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            ExecutionResult actual = target.Execute(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl03));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // validator.ValidationFilePathにeventController.ValidationFilePathの値が設定されていること
            Assert.AreEqual(validationFilePath, accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validateメソッドが呼び出され、正しい引数がわたっていること
            Coordinator_IValidatorImpl03 validatorImpl = (Coordinator_IValidatorImpl03)accessor._validator;
            Assert.IsTrue(validatorImpl.IsCalledValidate);
            Assert.AreSame(eventController.ViewData, validatorImpl.ValidateDataSet);
            Assert.IsInstanceOfType(validatorImpl.ValidateDataSet, _viewDataType);

            // eventController.ReportCustomValidatingメソッドが呼び出され引数が_validator.Validate()の戻り値と等しいこと
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(1, eventController.ReportCustomValidatingValidationResult.Errors.Count);
            Assert.AreSame(validatorImpl.ValidateValidationMessageInfo01, eventController.ReportCustomValidatingValidationResult.Errors[0]);

            // eventController.ErrorPathsにvalidationResult.Errorsの各ValidationMessageInfoのパスが追加されていること
            Assert.AreEqual(2, eventController.ErrorPaths.Count);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.ErrorPath, eventController.ErrorPaths[0]);
            Assert.AreEqual(((ValidationMessageInfo)eventController.ReportCustomValidatingMessageInfo01).ErrorPath, eventController.ErrorPaths[1]);

            // eventController.ReportValidationFailedメソッドが呼び出され、正しい引数がわたってきていること
            Assert.IsTrue(eventController.IsCalledReportValidationFailed);
            Assert.AreEqual(2, eventController.ReportValidationFailedExecutionResult.Errors.Count);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Key, eventController.ReportValidationFailedExecutionResult.Errors[0].Key);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Message, eventController.ReportValidationFailedExecutionResult.Errors[0].Message);
            Assert.AreEqual(eventController.ReportCustomValidatingMessageInfo01.Key, eventController.ReportValidationFailedExecutionResult.Errors[1].Key);
            Assert.AreEqual(eventController.ReportCustomValidatingMessageInfo01.Message, eventController.ReportValidationFailedExecutionResult.Errors[1].Message);         
            Assert.AreEqual(Coordinator.VALIDATION_FAILED, eventController.ReportValidationFailedExecutionResult.ResultString);
            Assert.IsNull(eventController.ReportValidationFailedExecutionResult.BLogicResultData);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessed);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // BLogicFactory.CreateBLogic() が呼び出されていないこと
            Assert.IsFalse(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);

            // _converter.Reverseメソッドが呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledReverse);

            // eventController.ReportBLogicFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportBLogicFailed);

            // eventController.Itemsが変更されていないこと
            Assert.AreEqual(0, eventController.Items.Count);

            // 戻り値のexecutionResult.ErrorsとvalidationResult.Errorsの要素が同一
            Assert.AreEqual(2, actual.Errors.Count);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Key, actual.Errors[0].Key);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Message, actual.Errors[0].Message);
            Assert.AreEqual(eventController.ReportCustomValidatingMessageInfo01.Key, actual.Errors[1].Key);
            Assert.AreEqual(eventController.ReportCustomValidatingMessageInfo01.Message, actual.Errors[1].Message);

            // 戻り値のexecutionResult.ResultStringがVALIDATION_FAILED
            Assert.AreEqual(Coordinator.VALIDATION_FAILED, actual.ResultString);

            // 戻り値のexecutionResult.BLogicResultDataがnull
            Assert.IsNull(actual.BLogicResultData);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 入力値検証機能による単項目チェックにおいてエラー数が 1 であり
        /// カスタムチェックイベントによる入力値検証においてエラー数が 3 の場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest15()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub03 eventController = new Coordinator_EventControllerStub03();
            string validationFilePath = "validationFilePath";
            eventController.ValidationFilePath = validationFilePath;

            ConfigurationManager.AppSettings["ValidatorTypeName"] = typeof(Coordinator_IValidatorImpl03).AssemblyQualifiedName;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            ExecutionResult actual = target.Execute(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl03));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // validator.ValidationFilePathにeventController.ValidationFilePathの値が設定されていること
            Assert.AreEqual(validationFilePath, accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validateメソッドが呼び出され、正しい引数がわたっていること
            Coordinator_IValidatorImpl03 validatorImpl = (Coordinator_IValidatorImpl03)accessor._validator;
            Assert.IsTrue(validatorImpl.IsCalledValidate);
            Assert.AreSame(eventController.ViewData, validatorImpl.ValidateDataSet);
            Assert.IsInstanceOfType(validatorImpl.ValidateDataSet, _viewDataType);

            // eventController.ReportCustomValidatingメソッドが呼び出され引数が_validator.Validate()の戻り値と等しいこと
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(1, eventController.ReportCustomValidatingValidationResult.Errors.Count);
            Assert.AreSame(validatorImpl.ValidateValidationMessageInfo01, eventController.ReportCustomValidatingValidationResult.Errors[0]);

            // eventController.ErrorPathsにvalidationResult.Errorsの各ValidationMessageInfoのパスが追加されていること
            Assert.AreEqual(4, eventController.ErrorPaths.Count);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.ErrorPath, eventController.ErrorPaths[0]);
            Assert.AreEqual(((ValidationMessageInfo)eventController.ReportCustomValidatingMessageInfo01).ErrorPath, eventController.ErrorPaths[1]);
            Assert.AreEqual(((ValidationMessageInfo)eventController.ReportCustomValidatingMessageInfo02).ErrorPath, eventController.ErrorPaths[2]);
            Assert.AreEqual(((ValidationMessageInfo)eventController.ReportCustomValidatingMessageInfo03).ErrorPath, eventController.ErrorPaths[3]);

            // eventController.ReportValidationFailedメソッドが呼び出され、正しい引数がわたってきていること
            Assert.IsTrue(eventController.IsCalledReportValidationFailed);
            Assert.AreEqual(4, eventController.ReportValidationFailedExecutionResult.Errors.Count);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Key, eventController.ReportValidationFailedExecutionResult.Errors[0].Key);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Message, eventController.ReportValidationFailedExecutionResult.Errors[0].Message);
            Assert.AreEqual(eventController.ReportCustomValidatingMessageInfo01.Key, eventController.ReportValidationFailedExecutionResult.Errors[1].Key);
            Assert.AreEqual(eventController.ReportCustomValidatingMessageInfo01.Message, eventController.ReportValidationFailedExecutionResult.Errors[1].Message);
            Assert.AreEqual(eventController.ReportCustomValidatingMessageInfo02.Key, eventController.ReportValidationFailedExecutionResult.Errors[2].Key);
            Assert.AreEqual(eventController.ReportCustomValidatingMessageInfo02.Message, eventController.ReportValidationFailedExecutionResult.Errors[2].Message);
            Assert.AreEqual(eventController.ReportCustomValidatingMessageInfo03.Key, eventController.ReportValidationFailedExecutionResult.Errors[3].Key);
            Assert.AreEqual(eventController.ReportCustomValidatingMessageInfo03.Message, eventController.ReportValidationFailedExecutionResult.Errors[3].Message);           
            Assert.AreEqual(Coordinator.VALIDATION_FAILED, eventController.ReportValidationFailedExecutionResult.ResultString);
            Assert.IsNull(eventController.ReportValidationFailedExecutionResult.BLogicResultData);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessed);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // BLogicFactory.CreateBLogic() が呼び出されていないこと
            Assert.IsFalse(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);

            // _converter.Reverseメソッドが呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledReverse);

            // eventController.ReportBLogicFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportBLogicFailed);

            // eventController.Itemsが変更されていないこと
            Assert.AreEqual(0, eventController.Items.Count);

            // 戻り値のexecutionResult.ErrorsとvalidationResult.Errorsの要素が同一
            Assert.AreEqual(4, actual.Errors.Count);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Key, actual.Errors[0].Key);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Message, actual.Errors[0].Message);
            Assert.AreEqual(eventController.ReportCustomValidatingMessageInfo01.Key, actual.Errors[1].Key);
            Assert.AreEqual(eventController.ReportCustomValidatingMessageInfo01.Message, actual.Errors[1].Message);
            Assert.AreEqual(eventController.ReportCustomValidatingMessageInfo02.Key, actual.Errors[2].Key);
            Assert.AreEqual(eventController.ReportCustomValidatingMessageInfo02.Message, actual.Errors[2].Message);
            Assert.AreEqual(eventController.ReportCustomValidatingMessageInfo03.Key, actual.Errors[3].Key);
            Assert.AreEqual(eventController.ReportCustomValidatingMessageInfo03.Message, actual.Errors[3].Message);

            // 戻り値のexecutionResult.ResultStringがVALIDATION_FAILED
            Assert.AreEqual(Coordinator.VALIDATION_FAILED, actual.ResultString);

            // 戻り値のexecutionResult.BLogicResultDataがnull
            Assert.IsNull(actual.BLogicResultData);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.RuleSet が空文字であり、
        /// 入力値検証機能による単項目チェックにおいてエラー数が 1 の場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest16()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            string validationFilePath = "validationFilePath";
            eventController.ValidationFilePath = validationFilePath;
            eventController.RuleSet = string.Empty;

            ConfigurationManager.AppSettings["ValidatorTypeName"] = typeof(Coordinator_IValidatorImpl03).AssemblyQualifiedName;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            ExecutionResult actual = target.Execute(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl03));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // validator.ValidationFilePathにeventController.ValidationFilePathの値が設定されていること
            Assert.AreEqual(validationFilePath, accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validateメソッドが呼び出され、正しい引数がわたっていること
            Coordinator_IValidatorImpl03 validatorImpl = (Coordinator_IValidatorImpl03)accessor._validator;
            Assert.IsTrue(validatorImpl.IsCalledValidate);
            Assert.AreSame(eventController.ViewData, validatorImpl.ValidateDataSet);
            Assert.IsInstanceOfType(validatorImpl.ValidateDataSet, _viewDataType);

            // eventController.ReportCustomValidatingメソッドが呼び出され引数が_validator.Validate()の戻り値と等しいこと
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(1, eventController.ReportCustomValidatingValidationResult.Errors.Count);
            Assert.AreSame(validatorImpl.ValidateValidationMessageInfo01, eventController.ReportCustomValidatingValidationResult.Errors[0]);

            // eventController.ErrorPathsにvalidationResult.Errorsの各ValidationMessageInfoのパスが追加されていること
            Assert.AreEqual(1, eventController.ErrorPaths.Count);
            Assert.AreEqual("errorPath01", eventController.ErrorPaths[0]);

            // eventController.ReportValidationFailedメソッドが呼び出され、正しい引数がわたってきていること
            Assert.IsTrue(eventController.IsCalledReportValidationFailed);
            Assert.AreEqual(1, eventController.ReportValidationFailedExecutionResult.Errors.Count);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Key, eventController.ReportValidationFailedExecutionResult.Errors[0].Key);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Message, eventController.ReportValidationFailedExecutionResult.Errors[0].Message);
            Assert.AreEqual(Coordinator.VALIDATION_FAILED, eventController.ReportValidationFailedExecutionResult.ResultString);
            Assert.IsNull(eventController.ReportValidationFailedExecutionResult.BLogicResultData);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessed);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // BLogicFactory.CreateBLogic() が呼び出されていないこと
            Assert.IsFalse(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);

            // _converter.Reverseメソッドが呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledReverse);

            // eventController.ReportBLogicFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportBLogicFailed);

            // eventController.Itemsが変更されていないこと
            Assert.AreEqual(0, eventController.Items.Count);

            // 戻り値のexecutionResult.ErrorsとvalidationResult.Errorsの要素が同一
            Assert.AreEqual(1, actual.Errors.Count);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Key, actual.Errors[0].Key);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Message, actual.Errors[0].Message);

            // 戻り値のexecutionResult.ResultStringがVALIDATION_FAILED
            Assert.AreEqual(Coordinator.VALIDATION_FAILED, actual.ResultString);

            // 戻り値のexecutionResult.BLogicResultDataがnull
            Assert.IsNull(actual.BLogicResultData);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.RuleSet が null・空文字でなく、
        /// 入力値検証機能による単項目チェックにおいてエラー数が 1 の場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest17()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            string validationFilePath = "validationFilePath";
            eventController.ValidationFilePath = validationFilePath;
            string ruleSet = "RuleSet";
            eventController.RuleSet = ruleSet;

            ConfigurationManager.AppSettings["ValidatorTypeName"] = typeof(Coordinator_IValidatorImpl03).AssemblyQualifiedName;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            ExecutionResult actual = target.Execute(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl03));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // validator.ValidationFilePathにeventController.ValidationFilePathの値が設定されていること
            Assert.AreEqual(validationFilePath, accessor._validator.ValidationFilePath);

            // _validator.RuleSetにeventController.RuleSetの値が設定されていること
            Assert.AreEqual(ruleSet, accessor._validator.RuleSet);

            // _validator.Validateメソッドが呼び出され、正しい引数がわたっていること
            Coordinator_IValidatorImpl03 validatorImpl = (Coordinator_IValidatorImpl03)accessor._validator;
            Assert.IsTrue(validatorImpl.IsCalledValidate);
            Assert.AreSame(eventController.ViewData, validatorImpl.ValidateDataSet);
            Assert.IsInstanceOfType(validatorImpl.ValidateDataSet, _viewDataType);

            // eventController.ReportCustomValidatingメソッドが呼び出され引数が_validator.Validate()の戻り値と等しいこと
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(1, eventController.ReportCustomValidatingValidationResult.Errors.Count);
            Assert.AreSame(validatorImpl.ValidateValidationMessageInfo01, eventController.ReportCustomValidatingValidationResult.Errors[0]);

            // eventController.ErrorPathsにvalidationResult.Errorsの各ValidationMessageInfoのパスが追加されていること
            Assert.AreEqual(1, eventController.ErrorPaths.Count);
            Assert.AreEqual("errorPath01", eventController.ErrorPaths[0]);

            // eventController.ReportValidationFailedメソッドが呼び出され、正しい引数がわたってきていること
            Assert.IsTrue(eventController.IsCalledReportValidationFailed);
            Assert.AreEqual(1, eventController.ReportValidationFailedExecutionResult.Errors.Count);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Key, eventController.ReportValidationFailedExecutionResult.Errors[0].Key);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Message, eventController.ReportValidationFailedExecutionResult.Errors[0].Message);
            Assert.AreEqual(Coordinator.VALIDATION_FAILED, eventController.ReportValidationFailedExecutionResult.ResultString);
            Assert.IsNull(eventController.ReportValidationFailedExecutionResult.BLogicResultData);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessed);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // BLogicFactory.CreateBLogic() が呼び出されていないこと
            Assert.IsFalse(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);

            // _converter.Reverseメソッドが呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledReverse);

            // eventController.ReportBLogicFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportBLogicFailed);

            // eventController.Itemsが変更されていないこと
            Assert.AreEqual(0, eventController.Items.Count);

            // 戻り値のexecutionResult.ErrorsとvalidationResult.Errorsの要素が同一
            Assert.AreEqual(1, actual.Errors.Count);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Key, actual.Errors[0].Key);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Message, actual.Errors[0].Message);

            // 戻り値のexecutionResult.ResultStringがVALIDATION_FAILED
            Assert.AreEqual(Coordinator.VALIDATION_FAILED, actual.ResultString);

            // 戻り値のexecutionResult.BLogicResultDataがnull
            Assert.IsNull(actual.BLogicResultData);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 入力値検証機能による単項目チェックにおいてエラー数が 1 、
        /// カスタムチェックイベントによる入力値検証においてエラー数が 1 であり、
        /// かつ、エラーの型が ValidationMessageInfo でない場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest18()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub04 eventController = new Coordinator_EventControllerStub04();
            string validationFilePath = "validationFilePath";
            eventController.ValidationFilePath = validationFilePath;

            ConfigurationManager.AppSettings["ValidatorTypeName"] = typeof(Coordinator_IValidatorImpl05).AssemblyQualifiedName;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            ExecutionResult actual = target.Execute(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl05));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // validator.ValidationFilePathにeventController.ValidationFilePathの値が設定されていること
            Assert.AreEqual(validationFilePath, accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validateメソッドが呼び出され、正しい引数がわたっていること
            Coordinator_IValidatorImpl05 validatorImpl = (Coordinator_IValidatorImpl05)accessor._validator;
            Assert.IsTrue(validatorImpl.IsCalledValidate);
            Assert.AreSame(eventController.ViewData, validatorImpl.ValidateDataSet);
            Assert.IsInstanceOfType(validatorImpl.ValidateDataSet, _viewDataType);

            // eventController.ReportCustomValidatingメソッドが呼び出され引数が_validator.Validate()の戻り値と等しいこと
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(1, eventController.ReportCustomValidatingValidationResult.Errors.Count);
            Assert.AreSame(validatorImpl.ValidateMessageInfo01, eventController.ReportCustomValidatingValidationResult.Errors[0]);

            // eventController.ReportValidationFailedメソッドが呼び出され、正しい引数がわたってきていること
            Assert.IsTrue(eventController.IsCalledReportValidationFailed);
            Assert.AreEqual(2, eventController.ReportValidationFailedExecutionResult.Errors.Count);
            Assert.AreEqual(validatorImpl.ValidateMessageInfo01.Key, eventController.ReportValidationFailedExecutionResult.Errors[0].Key);
            Assert.AreEqual(validatorImpl.ValidateMessageInfo01.Message, eventController.ReportValidationFailedExecutionResult.Errors[0].Message);
            Assert.AreEqual(eventController.ReportCustomValidatingMessageInfo01.Key, eventController.ReportValidationFailedExecutionResult.Errors[1].Key);
            Assert.AreEqual(eventController.ReportCustomValidatingMessageInfo01.Message, eventController.ReportValidationFailedExecutionResult.Errors[1].Message);
            Assert.AreEqual(Coordinator.VALIDATION_FAILED, eventController.ReportValidationFailedExecutionResult.ResultString);
            Assert.IsNull(eventController.ReportValidationFailedExecutionResult.BLogicResultData);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessed);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // BLogicFactory.CreateBLogic() が呼び出されていないこと
            Assert.IsFalse(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);

            // _converter.Reverseメソッドが呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledReverse);

            // eventController.ReportBLogicFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportBLogicFailed);

            // eventController.Itemsが変更されていないこと
            Assert.AreEqual(0, eventController.Items.Count);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 戻り値のexecutionResult.ErrorsとvalidationResult.Errorsの要素が同一
            Assert.AreEqual(2, actual.Errors.Count);
            Assert.AreEqual(validatorImpl.ValidateMessageInfo01.Key, actual.Errors[0].Key);
            Assert.AreEqual(validatorImpl.ValidateMessageInfo01.Message, actual.Errors[0].Message);
            Assert.AreEqual(eventController.ReportCustomValidatingMessageInfo01.Key, actual.Errors[1].Key);
            Assert.AreEqual(eventController.ReportCustomValidatingMessageInfo01.Message, actual.Errors[1].Message);

            // 戻り値のexecutionResult.ResultStringがVALIDATION_FAILED
            Assert.AreEqual(Coordinator.VALIDATION_FAILED, actual.ResultString);

            // 戻り値のexecutionResult.BLogicResultDataがnull
            Assert.IsNull(actual.BLogicResultData);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

         /// <summary>
        /// Execute (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.BLogicParamTypeName が null であり、
        /// eventController.ViewData が null の場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest19()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            eventController.BLogicParamTypeName = null;
            eventController.ViewData = null;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            ExecutionResult actual = target.Execute(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataの型がDataSet
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, typeof(DataSet));

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
            Assert.AreEqual(_blogicName, Coordinator_BLogicFactoryStub01._blogicName);
            Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
            Assert.AreEqual(_blogicResultTypeName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

            // _blogic.Execute() が呼び出されており、正しい引数がわたっていること
            Coordinator_IBLogicImpl01 blogicImpl = (Coordinator_IBLogicImpl01)accessor._blogic;
            Assert.IsTrue(blogicImpl.IsCalledExecute);
            Assert.AreEqual(1, blogicImpl.ExecuteBLogicParam.Items.Count);
            Assert.AreEqual(_requestName, blogicImpl.ExecuteBLogicParam.Items[EventController.REQUEST_NAME]);
            Assert.IsInstanceOfType(blogicImpl.ExecuteBLogicParam.ParamData, typeof(DataSet));

            // _converter.Reverseメソッドが呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledReverse);

            // eventController.ReportBLogicFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportBLogicFailed);

            // eventController.Itemsが変更されていないこと
            Assert.AreEqual(0, eventController.Items.Count);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 戻り値のexecutionResult.Errorsが_blogic.Execute()の戻り値のBlogicResult.Errorsと要素が同一
            Assert.AreEqual(0, actual.Errors.Count);

            // 戻り値のexecutionResult.ResultStringが_blogic.Execute()の戻り値のBLogicResult.ResultStringと同一
            Assert.AreEqual(BLogicResult.SUCCESS, actual.ResultString);

            // 戻り値のexecutionResult.BLogicResultDataが_blogic.Execute()の戻り値のBLogicResult.ResultDataと同一
            Assert.IsInstanceOfType(actual.BLogicResultData, typeof(Coordinator_DataSetStub04));

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.BLogicParamTypeName が null であり、
        /// eventController.ViewData が null でない場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest20()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            eventController.BLogicParamTypeName = null;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            ExecutionResult actual = target.Execute(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataの型がViewDataで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _viewDataType);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
            Assert.AreEqual(_blogicName, Coordinator_BLogicFactoryStub01._blogicName);
            Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
            Assert.AreEqual(_blogicResultTypeName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

            // _blogic.Execute() が呼び出されており、正しい引数がわたっていること
            Coordinator_IBLogicImpl01 blogicImpl = (Coordinator_IBLogicImpl01)accessor._blogic;
            Assert.IsTrue(blogicImpl.IsCalledExecute);
            Assert.AreEqual(1, blogicImpl.ExecuteBLogicParam.Items.Count);
            Assert.AreEqual(_requestName, blogicImpl.ExecuteBLogicParam.Items[EventController.REQUEST_NAME]);
            Assert.IsInstanceOfType(blogicImpl.ExecuteBLogicParam.ParamData, _viewDataType);

            // _converter.Reverseメソッドが呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledReverse);

            // eventController.ReportBLogicFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportBLogicFailed);

            // eventController.Itemsが変更されていないこと
            Assert.AreEqual(0, eventController.Items.Count);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 戻り値のexecutionResult.Errorsが_blogic.Execute()の戻り値のBlogicResult.Errorsと要素が同一
            Assert.AreEqual(0, actual.Errors.Count);

            // 戻り値のexecutionResult.ResultStringが_blogic.Execute()の戻り値のBLogicResult.ResultStringと同一
            Assert.AreEqual(BLogicResult.SUCCESS, actual.ResultString);

            // 戻り値のexecutionResult.BLogicResultDataが_blogic.Execute()の戻り値のBLogicResult.ResultDataと同一
            Assert.IsInstanceOfType(actual.BLogicResultData, typeof(Coordinator_DataSetStub04));

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.BLogicParamTypeName が 空文字 であり、
        /// eventController.ViewData が null の場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest21()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            eventController.BLogicParamTypeName = string.Empty;
            eventController.ViewData = null;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            ExecutionResult actual = target.Execute(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataの型がDataSet
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, typeof(DataSet));

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
            Assert.AreEqual(_blogicName, Coordinator_BLogicFactoryStub01._blogicName);
            Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
            Assert.AreEqual(_blogicResultTypeName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

            // _blogic.Execute() が呼び出されており、正しい引数がわたっていること
            Coordinator_IBLogicImpl01 blogicImpl = (Coordinator_IBLogicImpl01)accessor._blogic;
            Assert.IsTrue(blogicImpl.IsCalledExecute);
            Assert.AreEqual(1, blogicImpl.ExecuteBLogicParam.Items.Count);
            Assert.AreEqual(_requestName, blogicImpl.ExecuteBLogicParam.Items[EventController.REQUEST_NAME]);
            Assert.IsInstanceOfType(blogicImpl.ExecuteBLogicParam.ParamData, typeof(DataSet));

            // _converter.Reverseメソッドが呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledReverse);

            // eventController.ReportBLogicFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportBLogicFailed);

            // eventController.Itemsが変更されていないこと
            Assert.AreEqual(0, eventController.Items.Count);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 戻り値のexecutionResult.Errorsが_blogic.Execute()の戻り値のBlogicResult.Errorsと要素が同一
            Assert.AreEqual(0, actual.Errors.Count);

            // 戻り値のexecutionResult.ResultStringが_blogic.Execute()の戻り値のBLogicResult.ResultStringと同一
            Assert.AreEqual(BLogicResult.SUCCESS, actual.ResultString);

            // 戻り値のexecutionResult.BLogicResultDataが_blogic.Execute()の戻り値のBLogicResult.ResultDataと同一
            Assert.IsInstanceOfType(actual.BLogicResultData, typeof(Coordinator_DataSetStub04));

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// ビジネスロジック入力 DataSet のインスタンスを生成できない場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest22()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            string blogicParamenTypeName = typeof(Coordinator_DataSetStub05).AssemblyQualifiedName;
            eventController.BLogicParamTypeName = blogicParamenTypeName;
            
            try
            {
                // 実施
                target.Execute(eventController);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // 成功
                Assert.AreEqual(MessageManager.GetMessage("E_INVALID_CAST_TYPE_FROM_GENERICPARAM", blogicParamenTypeName, typeof(DataSet).AssemblyQualifiedName), e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("予期せぬエラーです。" + e.ToString());
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.Items の要素数が 1 で EventController.REQUEST_NAME がキーに存在している場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest23()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            string item = "Item01";
            eventController.Items.Add(EventController.REQUEST_NAME, item);

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            ExecutionResult actual = target.Execute(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
            Assert.AreEqual(_blogicName, Coordinator_BLogicFactoryStub01._blogicName);
            Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
            Assert.AreEqual(_blogicResultTypeName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

            // _blogic.Execute() が呼び出されており、正しい引数がわたっていること
            Coordinator_IBLogicImpl01 blogicImpl = (Coordinator_IBLogicImpl01)accessor._blogic;
            Assert.IsTrue(blogicImpl.IsCalledExecute);
            Assert.AreEqual(1, blogicImpl.ExecuteBLogicParam.Items.Count);
            Assert.AreEqual(_requestName, blogicImpl.ExecuteBLogicParam.Items[EventController.REQUEST_NAME]);
            Assert.IsInstanceOfType(blogicImpl.ExecuteBLogicParam.ParamData, _blogicParamType);

            // _converter.Reverseメソッドが呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledReverse);

            // eventController.ReportBLogicFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportBLogicFailed);

            // eventController.Itemsが変更されていないこと
            Assert.AreEqual(1, eventController.Items.Count);
            Assert.AreEqual(item, eventController.Items[EventController.REQUEST_NAME]);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 戻り値のexecutionResult.Errorsが_blogic.Execute()の戻り値のBlogicResult.Errorsと要素が同一
            Assert.AreEqual(0, actual.Errors.Count);

            // 戻り値のexecutionResult.ResultStringが_blogic.Execute()の戻り値のBLogicResult.ResultStringと同一
            Assert.AreEqual(BLogicResult.SUCCESS, actual.ResultString);

            // 戻り値のexecutionResult.BLogicResultDataが_blogic.Execute()の戻り値のBLogicResult.ResultDataと同一
            Assert.IsInstanceOfType(actual.BLogicResultData, typeof(Coordinator_DataSetStub04));

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.Items の要素数が 1 で EventController.REQUEST_NAME がキーに存在していない場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest24()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            string key = "key01";
            string item = "Item01";
            eventController.Items.Add(key, item);

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            ExecutionResult actual = target.Execute(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
            Assert.AreEqual(_blogicName, Coordinator_BLogicFactoryStub01._blogicName);
            Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
            Assert.AreEqual(_blogicResultTypeName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

            // _blogic.Execute() が呼び出されており、正しい引数がわたっていること
            Coordinator_IBLogicImpl01 blogicImpl = (Coordinator_IBLogicImpl01)accessor._blogic;
            Assert.IsTrue(blogicImpl.IsCalledExecute);
            Assert.AreEqual(2, blogicImpl.ExecuteBLogicParam.Items.Count);
            Assert.AreEqual(_requestName, blogicImpl.ExecuteBLogicParam.Items[EventController.REQUEST_NAME]);
            Assert.AreEqual(item, blogicImpl.ExecuteBLogicParam.Items[key]);
            Assert.IsInstanceOfType(blogicImpl.ExecuteBLogicParam.ParamData, _blogicParamType);

            // _converter.Reverseメソッドが呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledReverse);

            // eventController.ReportBLogicFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportBLogicFailed);

            // eventController.Itemsが変更されていないこと
            Assert.AreEqual(1, eventController.Items.Count);
            Assert.AreEqual(item, eventController.Items[key]);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 戻り値のexecutionResult.Errorsが_blogic.Execute()の戻り値のBlogicResult.Errorsと要素が同一
            Assert.AreEqual(0, actual.Errors.Count);

            // 戻り値のexecutionResult.ResultStringが_blogic.Execute()の戻り値のBLogicResult.ResultStringと同一
            Assert.AreEqual(BLogicResult.SUCCESS, actual.ResultString);

            // 戻り値のexecutionResult.BLogicResultDataが_blogic.Execute()の戻り値のBLogicResult.ResultDataと同一
            Assert.IsInstanceOfType(actual.BLogicResultData, typeof(Coordinator_DataSetStub04));

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.Items の要素数が 3 で EventController.REQUEST_NAME がキーに存在している場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest25()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            string key01 = "key01";
            string key02 = "key02";
            string item01 = "Item01";
            string item02 = "Item02";
            string item03 = "Item03";
            eventController.Items.Add(key01, item01);
            eventController.Items.Add(key02, item02);
            eventController.Items.Add(EventController.REQUEST_NAME, item03);

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            ExecutionResult actual = target.Execute(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
            Assert.AreEqual(_blogicName, Coordinator_BLogicFactoryStub01._blogicName);
            Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
            Assert.AreEqual(_blogicResultTypeName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

            // _blogic.Execute() が呼び出されており、正しい引数がわたっていること
            Coordinator_IBLogicImpl01 blogicImpl = (Coordinator_IBLogicImpl01)accessor._blogic;
            Assert.IsTrue(blogicImpl.IsCalledExecute);
            Assert.AreEqual(3, blogicImpl.ExecuteBLogicParam.Items.Count);
            Assert.AreEqual(item01, blogicImpl.ExecuteBLogicParam.Items[key01]);
            Assert.AreEqual(item02, blogicImpl.ExecuteBLogicParam.Items[key02]);
            Assert.AreEqual(_requestName, blogicImpl.ExecuteBLogicParam.Items[EventController.REQUEST_NAME]);
            Assert.IsInstanceOfType(blogicImpl.ExecuteBLogicParam.ParamData, _blogicParamType);

            // _converter.Reverseメソッドが呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledReverse);

            // eventController.ReportBLogicFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportBLogicFailed);

            // eventController.Itemsが変更されていないこと
            Assert.AreEqual(3, eventController.Items.Count);
            Assert.AreEqual(item01, eventController.Items[key01]);
            Assert.AreEqual(item02, eventController.Items[key02]);
            Assert.AreEqual(item03, eventController.Items[EventController.REQUEST_NAME]);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 戻り値のexecutionResult.Errorsが_blogic.Execute()の戻り値のBlogicResult.Errorsと要素が同一
            Assert.AreEqual(0, actual.Errors.Count);

            // 戻り値のexecutionResult.ResultStringが_blogic.Execute()の戻り値のBLogicResult.ResultStringと同一
            Assert.AreEqual(BLogicResult.SUCCESS, actual.ResultString);

            // 戻り値のexecutionResult.BLogicResultDataが_blogic.Execute()の戻り値のBLogicResult.ResultDataと同一
            Assert.IsInstanceOfType(actual.BLogicResultData, typeof(Coordinator_DataSetStub04));

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.Items の要素数が 3 で EventController.REQUEST_NAME がキーに存在していない場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest26()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            string key01 = "key01";
            string key02 = "key02";
            string key03 = "key03";
            string item01 = "Item01";
            string item02 = "Item02";
            string item03 = "Item03";
            eventController.Items.Add(key01, item01);
            eventController.Items.Add(key02, item02);
            eventController.Items.Add(key03, item03);

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            ExecutionResult actual = target.Execute(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
            Assert.AreEqual(_blogicName, Coordinator_BLogicFactoryStub01._blogicName);
            Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
            Assert.AreEqual(_blogicResultTypeName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

            // _blogic.Execute() が呼び出されており、正しい引数がわたっていること
            Coordinator_IBLogicImpl01 blogicImpl = (Coordinator_IBLogicImpl01)accessor._blogic;
            Assert.IsTrue(blogicImpl.IsCalledExecute);
            Assert.AreEqual(4, blogicImpl.ExecuteBLogicParam.Items.Count);
            Assert.AreEqual(item01, blogicImpl.ExecuteBLogicParam.Items[key01]);
            Assert.AreEqual(item02, blogicImpl.ExecuteBLogicParam.Items[key02]);
            Assert.AreEqual(item03, blogicImpl.ExecuteBLogicParam.Items[key03]);
            Assert.AreEqual(_requestName, blogicImpl.ExecuteBLogicParam.Items[EventController.REQUEST_NAME]);
            Assert.IsInstanceOfType(blogicImpl.ExecuteBLogicParam.ParamData, _blogicParamType);

            // _converter.Reverseメソッドが呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledReverse);

            // eventController.ReportBLogicFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportBLogicFailed);

            // eventController.Itemsが変更されていないこと
            Assert.AreEqual(3, eventController.Items.Count);
            Assert.AreEqual(item01, eventController.Items[key01]);
            Assert.AreEqual(item02, eventController.Items[key02]);
            Assert.AreEqual(item03, eventController.Items[key03]);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 戻り値のexecutionResult.Errorsが_blogic.Execute()の戻り値のBlogicResult.Errorsと要素が同一
            Assert.AreEqual(0, actual.Errors.Count);

            // 戻り値のexecutionResult.ResultStringが_blogic.Execute()の戻り値のBLogicResult.ResultStringと同一
            Assert.AreEqual(BLogicResult.SUCCESS, actual.ResultString);

            // 戻り値のexecutionResult.BLogicResultDataが_blogic.Execute()の戻り値のBLogicResult.ResultDataと同一
            Assert.IsInstanceOfType(actual.BLogicResultData, typeof(Coordinator_DataSetStub04));

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.ConvertId が空文字列の場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest27()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            eventController.ConvertId = string.Empty;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            ExecutionResult actual = target.Execute(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
            Assert.AreEqual(_blogicName, Coordinator_BLogicFactoryStub01._blogicName);
            Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
            Assert.AreEqual(_blogicResultTypeName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

            // _blogic.Execute() が呼び出されており、正しい引数がわたっていること
            Coordinator_IBLogicImpl01 blogicImpl = (Coordinator_IBLogicImpl01)accessor._blogic;
            Assert.IsTrue(blogicImpl.IsCalledExecute);
            Assert.AreEqual(1, blogicImpl.ExecuteBLogicParam.Items.Count);
            Assert.AreEqual(_requestName, blogicImpl.ExecuteBLogicParam.Items[EventController.REQUEST_NAME]);
            Assert.IsInstanceOfType(blogicImpl.ExecuteBLogicParam.ParamData, _blogicParamType);

            // _converter.Reverseメソッドが呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledReverse);

            // eventController.ReportBLogicFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportBLogicFailed);

            // eventController.Itemsが変更されていないこと
            Assert.AreEqual(0, eventController.Items.Count);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 戻り値のexecutionResult.Errorsが_blogic.Execute()の戻り値のBlogicResult.Errorsと要素が同一
            Assert.AreEqual(0, actual.Errors.Count);

            // 戻り値のexecutionResult.ResultStringが_blogic.Execute()の戻り値のBLogicResult.ResultStringと同一
            Assert.AreEqual(BLogicResult.SUCCESS, actual.ResultString);

            // 戻り値のexecutionResult.BLogicResultDataが_blogic.Execute()の戻り値のBLogicResult.ResultDataと同一
            Assert.IsInstanceOfType(actual.BLogicResultData, typeof(Coordinator_DataSetStub04));

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.ConvertId が null・空文字列でなく
        /// eventController.ViewData が null の場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest28()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            string convertId = "convertId";
            eventController.ConvertId = convertId;
            eventController.ViewData = null;

            try
            {
                // 実施
                target.Execute(eventController);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // 成功
                Assert.AreEqual(MessageManager.GetMessage("E_BLOGIC_NULL_DATASET", "ConvertId", "ViewData"), e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("予期せぬエラーです。" + e.ToString());
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(
                typeof(Coordinator), typeof(TerasolunaException), "E_BLOGIC_NULL_DATASET", "ConvertId", "ViewData"));
        }

        /// <summary>
        /// Execute (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.ConvertId が null・空文字列でなく
        /// eventController.ViewData が null でない場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest30()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            string convertId = "convertId";
            eventController.ConvertId = convertId; 

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            ExecutionResult actual = target.Execute(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されており、正しい引数がわたっていること
            Coordinator_IConverterImpl01 converter = (Coordinator_IConverterImpl01)accessor._converter;
            Assert.IsTrue(converter.IsCalledConvert);
            Assert.AreEqual(convertId, converter.ConvertId);
            Assert.AreSame(eventController.ViewData, converter.ConvertSrcData);
            Assert.IsInstanceOfType(converter.ConvertDestData, _blogicParamType);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
            Assert.AreEqual(_blogicName, Coordinator_BLogicFactoryStub01._blogicName);
            Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
            Assert.AreEqual(_blogicResultTypeName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

            // _blogic.Execute() が呼び出されており、正しい引数がわたっていること
            Coordinator_IBLogicImpl01 blogicImpl = (Coordinator_IBLogicImpl01)accessor._blogic;
            Assert.IsTrue(blogicImpl.IsCalledExecute);
            Assert.AreEqual(1, blogicImpl.ExecuteBLogicParam.Items.Count);
            Assert.AreEqual(_requestName, blogicImpl.ExecuteBLogicParam.Items[EventController.REQUEST_NAME]);
            Assert.IsInstanceOfType(blogicImpl.ExecuteBLogicParam.ParamData, _blogicParamType);

            // _converter.Reverse() が呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(converter.IsCalledReverse);
            Assert.AreEqual(convertId, converter.ReverseId);
            Assert.AreSame(eventController.ViewData, converter.ReverseSrcData);
            Assert.IsInstanceOfType(converter.ReverseDestData, typeof(Coordinator_DataSetStub04));

            // eventController.ReportBLogicFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportBLogicFailed);

            // eventController.Itemsが変更されていないこと
            Assert.AreEqual(0, eventController.Items.Count);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 戻り値のexecutionResult.Errorsが_blogic.Execute()の戻り値のBlogicResult.Errorsと要素が同一
            Assert.AreEqual(0, actual.Errors.Count);

            // 戻り値のexecutionResult.ResultStringが_blogic.Execute()の戻り値のBLogicResult.ResultStringと同一
            Assert.AreEqual(BLogicResult.SUCCESS, actual.ResultString);

            // 戻り値のexecutionResult.BLogicResultDataが_blogic.Execute()の戻り値のBLogicResult.ResultDataと同一
            Assert.IsInstanceOfType(actual.BLogicResultData, typeof(Coordinator_DataSetStub04));

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 前処理完了イベントでのエラー数が 1 でエラーの型が ValidationMessageInfo の場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest31()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub05 eventController = new Coordinator_EventControllerStub05();

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            ExecutionResult actual = target.Execute(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ErrorPathsにexecutionResult.Errorsの各ValidationMessageInfoのパスが追加されていること
            Assert.AreEqual(1, eventController.ErrorPaths.Count);
            Assert.AreEqual(((ValidationMessageInfo)eventController.ReportPreprocessedMessageInfo01).ErrorPath, eventController.ErrorPaths[0]);

            // eventController.ReportPreprocessedFailedメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(eventController.IsCalledReportPreprocessedFailed);
            Assert.AreEqual(1, eventController.ReportPreprocessedFailedExecutionResult.Errors.Count);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo01.Key, eventController.ReportPreprocessedFailedExecutionResult.Errors[0].Key);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo01.Message, eventController.ReportPreprocessedFailedExecutionResult.Errors[0].Message);
            Assert.AreEqual(Coordinator.PREPROCESSED_FAILED, eventController.ReportPreprocessedFailedExecutionResult.ResultString);
            Assert.IsNull(eventController.ReportPreprocessedFailedExecutionResult.BLogicResultData);

            // BLogicFactory.CreateBLogicメソッドが呼び出されないこと
            Assert.IsFalse(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);

            // _converter.Reverseメソッドが呼び出されないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledReverse);

            // eventController.ReportBLogicFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportBLogicFailed);

            // eventController.Itemsが変更されていないこと
            Assert.AreEqual(0, eventController.Items.Count);

            // 戻り値のexecutionResult.ErrorsとeventController.ReportPreprocessed()で追加されたexecutionResult.Errorsの中身が同じ
            Assert.AreEqual(1, actual.Errors.Count);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo01.Key, actual.Errors[0].Key);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo01.Message, actual.Errors[0].Message);

            // 戻り値のexecutionResult.ResultStringがPREPROCESSED_FAILED
            Assert.AreEqual(Coordinator.PREPROCESSED_FAILED, actual.ResultString);

            // 戻り値のexecutionResult.BLogicResultDataがnull
            Assert.IsNull(actual.BLogicResultData);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.ErrorHandler が null でなく、
        /// 前処理完了イベントでのエラー数が 1 でエラーの型が ValidationMessageInfo でない場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest32()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub06 eventController = new Coordinator_EventControllerStub06();
            Coordinator_IErrorHandlerImpl01 errorHandler = new Coordinator_IErrorHandlerImpl01();
            eventController.ErrorHandler = errorHandler;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            ExecutionResult actual = target.Execute(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ErrorHandler.HandleError()が呼ばれ引数が正しいこと
            Assert.IsTrue(errorHandler.IsCalledHandleError);
            Assert.AreEqual(Coordinator.PREPROCESSED_FAILED, errorHandler.HandleErrorResultString);
            Assert.AreEqual(1, errorHandler.HandleErrorMessages.Count);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo01.Key, errorHandler.HandleErrorMessages[0].Key);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo01.Message, errorHandler.HandleErrorMessages[0].Message);
            Assert.IsInstanceOfType(errorHandler.HandleErrorView, _viewDataType);

            // eventController.ReportPreprocessedFailedメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(eventController.IsCalledReportPreprocessedFailed);
            Assert.AreEqual(1, eventController.ReportPreprocessedFailedExecutionResult.Errors.Count);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo01.Key, eventController.ReportPreprocessedFailedExecutionResult.Errors[0].Key);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo01.Message, eventController.ReportPreprocessedFailedExecutionResult.Errors[0].Message);
            Assert.AreEqual(Coordinator.PREPROCESSED_FAILED, eventController.ReportPreprocessedFailedExecutionResult.ResultString);
            Assert.IsNull(eventController.ReportPreprocessedFailedExecutionResult.BLogicResultData);

            // BLogicFactory.CreateBLogicメソッドが呼び出されないこと
            Assert.IsFalse(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);

            // _converter.Reverseメソッドが呼び出されないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledReverse);

            // eventController.ReportBLogicFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportBLogicFailed);

            // eventController.Itemsが変更されていないこと
            Assert.AreEqual(0, eventController.Items.Count);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 戻り値のexecutionResult.ErrorsとeventController.ReportPreprocessed()で追加されたexecutionResult.Errorsの中身が同じ
            Assert.AreEqual(1, actual.Errors.Count);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo01.Key, actual.Errors[0].Key);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo01.Message, actual.Errors[0].Message);

            // 戻り値のexecutionResult.ResultStringがPREPROCESSED_FAILED
            Assert.AreEqual(Coordinator.PREPROCESSED_FAILED, actual.ResultString);

            // 戻り値のexecutionResult.BLogicResultDataがnull
            Assert.IsNull(actual.BLogicResultData);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.ErrorHandler が null でなく、
        /// 前処理完了イベントでのエラー数が 3 でエラーの型が ValidationMessageInfo でない場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest33()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub07 eventController = new Coordinator_EventControllerStub07();
            Coordinator_IErrorHandlerImpl01 errorHandler = new Coordinator_IErrorHandlerImpl01();
            eventController.ErrorHandler = errorHandler;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            ExecutionResult actual = target.Execute(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ErrorHandler.HandleError()が呼ばれ引数が正しいこと
            Assert.IsTrue(errorHandler.IsCalledHandleError);
            Assert.AreEqual(Coordinator.PREPROCESSED_FAILED, errorHandler.HandleErrorResultString);
            Assert.AreEqual(3, errorHandler.HandleErrorMessages.Count);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo01.Key, errorHandler.HandleErrorMessages[0].Key);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo02.Key, errorHandler.HandleErrorMessages[1].Key);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo03.Key, errorHandler.HandleErrorMessages[2].Key);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo01.Message, errorHandler.HandleErrorMessages[0].Message);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo02.Message, errorHandler.HandleErrorMessages[1].Message);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo03.Message, errorHandler.HandleErrorMessages[2].Message);
            Assert.IsInstanceOfType(errorHandler.HandleErrorView, _viewDataType);

            // eventController.ReportPreprocessedFailedメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(eventController.IsCalledReportPreprocessedFailed);
            Assert.AreEqual(3, eventController.ReportPreprocessedFailedExecutionResult.Errors.Count);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo01.Key, eventController.ReportPreprocessedFailedExecutionResult.Errors[0].Key);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo02.Key, eventController.ReportPreprocessedFailedExecutionResult.Errors[1].Key);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo03.Key, eventController.ReportPreprocessedFailedExecutionResult.Errors[2].Key);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo01.Message, eventController.ReportPreprocessedFailedExecutionResult.Errors[0].Message);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo02.Message, eventController.ReportPreprocessedFailedExecutionResult.Errors[1].Message);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo03.Message, eventController.ReportPreprocessedFailedExecutionResult.Errors[2].Message);
            Assert.AreEqual(Coordinator.PREPROCESSED_FAILED, eventController.ReportPreprocessedFailedExecutionResult.ResultString);
            Assert.IsNull(eventController.ReportPreprocessedFailedExecutionResult.BLogicResultData);

            // BLogicFactory.CreateBLogicメソッドが呼び出されないこと
            Assert.IsFalse(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);

            // _converter.Reverseメソッドが呼び出されないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledReverse);

            // eventController.ReportBLogicFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportBLogicFailed);

            // eventController.Itemsが変更されていないこと
            Assert.AreEqual(0, eventController.Items.Count);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 戻り値のexecutionResult.ErrorsとeventController.ReportPreprocessed()で追加されたexecutionResult.Errorsの中身が同じ
            Assert.AreEqual(3, actual.Errors.Count);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo01.Key, actual.Errors[0].Key);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo02.Key, actual.Errors[1].Key);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo03.Key, actual.Errors[2].Key);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo01.Message, actual.Errors[0].Message);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo02.Message, actual.Errors[1].Message);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo03.Message, actual.Errors[2].Message);

            // 戻り値のexecutionResult.ResultStringがPREPROCESSED_FAILED
            Assert.AreEqual(Coordinator.PREPROCESSED_FAILED, actual.ResultString);

            // 戻り値のexecutionResult.BLogicResultDataがnull
            Assert.IsNull(actual.BLogicResultData);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.BLogicResultTypeName が null であり、
        /// eventController.ViewData が null の場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest34()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            eventController.BLogicResultTypeName = null;
            eventController.ViewData = null;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            ExecutionResult actual = target.Execute(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
            Assert.AreEqual(_blogicName, Coordinator_BLogicFactoryStub01._blogicName);
            Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
            Assert.AreEqual(typeof(DataSet).AssemblyQualifiedName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

            // _blogic.Execute() が呼び出されており、正しい引数がわたっていること
            Coordinator_IBLogicImpl01 blogicImpl = (Coordinator_IBLogicImpl01)accessor._blogic;
            Assert.IsTrue(blogicImpl.IsCalledExecute);
            Assert.AreEqual(1, blogicImpl.ExecuteBLogicParam.Items.Count);
            Assert.AreEqual(_requestName, blogicImpl.ExecuteBLogicParam.Items[EventController.REQUEST_NAME]);
            Assert.IsInstanceOfType(blogicImpl.ExecuteBLogicParam.ParamData, _blogicParamType);

            // _converter.Reverseメソッドが呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledReverse);

            // eventController.ReportBLogicFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportBLogicFailed);

            // eventController.Itemsが変更されていないこと
            Assert.AreEqual(0, eventController.Items.Count);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 戻り値のexecutionResult.Errorsが_blogic.Execute()の戻り値のBlogicResult.Errorsと要素が同一
            Assert.AreEqual(0, actual.Errors.Count);

            // 戻り値のexecutionResult.ResultStringが_blogic.Execute()の戻り値のBLogicResult.ResultStringと同一
            Assert.AreEqual(BLogicResult.SUCCESS, actual.ResultString);

            // 戻り値のexecutionResult.BLogicResultDataが_blogic.Execute()の戻り値のBLogicResult.ResultDataと同一
            Assert.IsInstanceOfType(actual.BLogicResultData, typeof(Coordinator_DataSetStub04));

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.BLogicResultTypeName が null であり、
        /// eventController.ViewData が null でない場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest35()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            eventController.BLogicResultTypeName = null;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            ExecutionResult actual = target.Execute(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
            Assert.AreEqual(_blogicName, Coordinator_BLogicFactoryStub01._blogicName);
            Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
            Assert.AreEqual(_viewDataType.AssemblyQualifiedName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

            // _blogic.Execute() が呼び出されており、正しい引数がわたっていること
            Coordinator_IBLogicImpl01 blogicImpl = (Coordinator_IBLogicImpl01)accessor._blogic;
            Assert.IsTrue(blogicImpl.IsCalledExecute);
            Assert.AreEqual(1, blogicImpl.ExecuteBLogicParam.Items.Count);
            Assert.AreEqual(_requestName, blogicImpl.ExecuteBLogicParam.Items[EventController.REQUEST_NAME]);
            Assert.IsInstanceOfType(blogicImpl.ExecuteBLogicParam.ParamData, _blogicParamType);

            // _converter.Reverseメソッドが呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledReverse);

            // eventController.ReportBLogicFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportBLogicFailed);

            // eventController.Itemsが変更されていないこと
            Assert.AreEqual(0, eventController.Items.Count);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 戻り値のexecutionResult.Errorsが_blogic.Execute()の戻り値のBlogicResult.Errorsと要素が同一
            Assert.AreEqual(0, actual.Errors.Count);

            // 戻り値のexecutionResult.ResultStringが_blogic.Execute()の戻り値のBLogicResult.ResultStringと同一
            Assert.AreEqual(BLogicResult.SUCCESS, actual.ResultString);

            // 戻り値のexecutionResult.BLogicResultDataが_blogic.Execute()の戻り値のBLogicResult.ResultDataと同一
            Assert.IsInstanceOfType(actual.BLogicResultData, typeof(Coordinator_DataSetStub04));

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.BLogicResultTypeName が空文字列であり、
        /// eventController.ViewData が null の場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest36()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            eventController.BLogicResultTypeName = string.Empty;
            eventController.ViewData = null;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            ExecutionResult actual = target.Execute(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
            Assert.AreEqual(_blogicName, Coordinator_BLogicFactoryStub01._blogicName);
            Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
            Assert.AreEqual(typeof(DataSet).AssemblyQualifiedName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

            // _blogic.Execute() が呼び出されており、正しい引数がわたっていること
            Coordinator_IBLogicImpl01 blogicImpl = (Coordinator_IBLogicImpl01)accessor._blogic;
            Assert.IsTrue(blogicImpl.IsCalledExecute);
            Assert.AreEqual(1, blogicImpl.ExecuteBLogicParam.Items.Count);
            Assert.AreEqual(_requestName, blogicImpl.ExecuteBLogicParam.Items[EventController.REQUEST_NAME]);
            Assert.IsInstanceOfType(blogicImpl.ExecuteBLogicParam.ParamData, _blogicParamType);

            // _converter.Reverseメソッドが呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledReverse);

            // eventController.ReportBLogicFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportBLogicFailed);

            // eventController.Itemsが変更されていないこと
            Assert.AreEqual(0, eventController.Items.Count);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 戻り値のexecutionResult.Errorsが_blogic.Execute()の戻り値のBlogicResult.Errorsと要素が同一
            Assert.AreEqual(0, actual.Errors.Count);

            // 戻り値のexecutionResult.ResultStringが_blogic.Execute()の戻り値のBLogicResult.ResultStringと同一
            Assert.AreEqual(BLogicResult.SUCCESS, actual.ResultString);

            // 戻り値のexecutionResult.BLogicResultDataが_blogic.Execute()の戻り値のBLogicResult.ResultDataと同一
            Assert.IsInstanceOfType(actual.BLogicResultData, typeof(Coordinator_DataSetStub04));

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.BLogicResultTypeName が空文字列であり、
        /// eventController.ViewData が null でない場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest37()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            eventController.BLogicResultTypeName = string.Empty;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            ExecutionResult actual = target.Execute(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
            Assert.AreEqual(_blogicName, Coordinator_BLogicFactoryStub01._blogicName);
            Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
            Assert.AreEqual(_viewDataType.AssemblyQualifiedName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

            // _blogic.Execute() が呼び出されており、正しい引数がわたっていること
            Coordinator_IBLogicImpl01 blogicImpl = (Coordinator_IBLogicImpl01)accessor._blogic;
            Assert.IsTrue(blogicImpl.IsCalledExecute);
            Assert.AreEqual(1, blogicImpl.ExecuteBLogicParam.Items.Count);
            Assert.AreEqual(_requestName, blogicImpl.ExecuteBLogicParam.Items[EventController.REQUEST_NAME]);
            Assert.IsInstanceOfType(blogicImpl.ExecuteBLogicParam.ParamData, _blogicParamType);

            // _converter.Reverseメソッドが呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledReverse);

            // eventController.ReportBLogicFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportBLogicFailed);

            // eventController.Itemsが変更されていないこと
            Assert.AreEqual(0, eventController.Items.Count);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 戻り値のexecutionResult.Errorsが_blogic.Execute()の戻り値のBlogicResult.Errorsと要素が同一
            Assert.AreEqual(0, actual.Errors.Count);

            // 戻り値のexecutionResult.ResultStringが_blogic.Execute()の戻り値のBLogicResult.ResultStringと同一
            Assert.AreEqual(BLogicResult.SUCCESS, actual.ResultString);

            // 戻り値のexecutionResult.BLogicResultDataが_blogic.Execute()の戻り値のBLogicResult.ResultDataと同一
            Assert.IsInstanceOfType(actual.BLogicResultData, typeof(Coordinator_DataSetStub04));

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.BLogicName が null の場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest38()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            eventController.BLogicName = null;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            ExecutionResult actual = target.Execute(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
            Assert.IsNull(Coordinator_BLogicFactoryStub01._blogicName);
            Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
            Assert.AreEqual(_blogicResultTypeName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

            // _blogic.Execute() が呼び出されており、正しい引数がわたっていること
            Coordinator_IBLogicImpl01 blogicImpl = (Coordinator_IBLogicImpl01)accessor._blogic;
            Assert.IsTrue(blogicImpl.IsCalledExecute);
            Assert.AreEqual(1, blogicImpl.ExecuteBLogicParam.Items.Count);
            Assert.AreEqual(_requestName, blogicImpl.ExecuteBLogicParam.Items[EventController.REQUEST_NAME]);
            Assert.IsInstanceOfType(blogicImpl.ExecuteBLogicParam.ParamData, _blogicParamType);

            // _converter.Reverseメソッドが呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledReverse);

            // eventController.ReportBLogicFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportBLogicFailed);

            // eventController.Itemsが変更されていないこと
            Assert.AreEqual(0, eventController.Items.Count);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 戻り値のexecutionResult.Errorsが_blogic.Execute()の戻り値のBlogicResult.Errorsと要素が同一
            Assert.AreEqual(0, actual.Errors.Count);

            // 戻り値のexecutionResult.ResultStringが_blogic.Execute()の戻り値のBLogicResult.ResultStringと同一
            Assert.AreEqual(BLogicResult.SUCCESS, actual.ResultString);

            // 戻り値のexecutionResult.BLogicResultDataが_blogic.Execute()の戻り値のBLogicResult.ResultDataと同一
            Assert.IsInstanceOfType(actual.BLogicResultData, typeof(Coordinator_DataSetStub04));

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.BLogicName が空文字列の場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest39()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            eventController.BLogicName = string.Empty;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            ExecutionResult actual = target.Execute(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
            Assert.AreEqual(string.Empty, Coordinator_BLogicFactoryStub01._blogicName);
            Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
            Assert.AreEqual(_blogicResultTypeName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

            // _blogic.Execute() が呼び出されており、正しい引数がわたっていること
            Coordinator_IBLogicImpl01 blogicImpl = (Coordinator_IBLogicImpl01)accessor._blogic;
            Assert.IsTrue(blogicImpl.IsCalledExecute);
            Assert.AreEqual(1, blogicImpl.ExecuteBLogicParam.Items.Count);
            Assert.AreEqual(_requestName, blogicImpl.ExecuteBLogicParam.Items[EventController.REQUEST_NAME]);
            Assert.IsInstanceOfType(blogicImpl.ExecuteBLogicParam.ParamData, _blogicParamType);

            // _converter.Reverseメソッドが呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledReverse);

            // eventController.ReportBLogicFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportBLogicFailed);

            // eventController.Itemsが変更されていないこと
            Assert.AreEqual(0, eventController.Items.Count);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 戻り値のexecutionResult.Errorsが_blogic.Execute()の戻り値のBlogicResult.Errorsと要素が同一
            Assert.AreEqual(0, actual.Errors.Count);

            // 戻り値のexecutionResult.ResultStringが_blogic.Execute()の戻り値のBLogicResult.ResultStringと同一
            Assert.AreEqual(BLogicResult.SUCCESS, actual.ResultString);

            // 戻り値のexecutionResult.BLogicResultDataが_blogic.Execute()の戻り値のBLogicResult.ResultDataと同一
            Assert.IsInstanceOfType(actual.BLogicResultData, typeof(Coordinator_DataSetStub04));

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// ビジネスロジッククラスが IProgressChengedEventInvoker を実装している場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest40()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl02 blogic = new Coordinator_IBLogicImpl02();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            ExecutionResult actual = target.Execute(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
            Assert.AreEqual(_blogicName, Coordinator_BLogicFactoryStub01._blogicName);
            Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
            Assert.AreEqual(_blogicResultTypeName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

            // BLogicFactory.CreateBLogic()で生成したIBLogic実装クラスのインスタンスのProgressChangedイベントに
            // DoExecuteProgressChanged()が登録されていること
            Assert.IsTrue(blogic.IsNotNullProgressChanged);

            // _blogic.Execute() が呼び出されており、正しい引数がわたっていること
            Coordinator_IBLogicImpl02 blogicImpl = (Coordinator_IBLogicImpl02)accessor._blogic;
            Assert.IsTrue(blogicImpl.IsCalledExecute);
            Assert.AreEqual(1, blogicImpl.ExecuteBLogicParam.Items.Count);
            Assert.AreEqual(_requestName, blogicImpl.ExecuteBLogicParam.Items[EventController.REQUEST_NAME]);
            Assert.IsInstanceOfType(blogicImpl.ExecuteBLogicParam.ParamData, _blogicParamType);

            // _converter.Reverseメソッドが呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledReverse);

            // eventController.ReportBLogicFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportBLogicFailed);

            // eventController.Itemsが変更されていないこと
            Assert.AreEqual(0, eventController.Items.Count);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 戻り値のexecutionResult.Errorsが_blogic.Execute()の戻り値のBlogicResult.Errorsと要素が同一
            Assert.AreEqual(0, actual.Errors.Count);

            // 戻り値のexecutionResult.ResultStringが_blogic.Execute()の戻り値のBLogicResult.ResultStringと同一
            Assert.AreEqual(BLogicResult.SUCCESS, actual.ResultString);

            // 戻り値のexecutionResult.BLogicResultDataが_blogic.Execute()の戻り値のBLogicResult.ResultDataと同一
            Assert.IsInstanceOfType(actual.BLogicResultData, typeof(Coordinator_DataSetStub04));

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// ビジネスロジック実行中に例外が発生した場合、
        /// スローされた例外がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest42()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl03 blogic = new Coordinator_IBLogicImpl03();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            try
            {
                // 実施
                target.Execute(eventController);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (Coordinator_ExceptionStub01)
            {
                // 確認
                TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

                // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
                Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

                // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
                Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

                // _validator.ValidationFilePathが初期値であること
                Assert.IsNull(accessor._validator.ValidationFilePath);

                // _validator.RuleSetが初期値であること
                Assert.IsNull(accessor._validator.RuleSet);

                // _validator.Validate() が呼び出されないこと
                Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

                // eventController.ReportCustomValidating() が呼び出され、引数で渡された
                // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
                Assert.IsTrue(eventController.IsCalledReportCustomValidating);
                Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

                // eventController.ReportValidationFailed() が呼び出されないこと
                Assert.IsFalse(eventController.IsCalledReportValidationFailed);

                // _converter.Convert() が呼び出されていないこと
                Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

                // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
                // 引数 executionResultの各プロパティが初期値
                // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
                Assert.IsTrue(eventController.IsCalledReportPreprocessed);
                Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
                Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
                Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
                Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

                // eventController.ReportPreprocessedFailed() が呼び出されていないこと
                Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

                // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
                Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
                Assert.AreEqual(_blogicName, Coordinator_BLogicFactoryStub01._blogicName);
                Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
                Assert.AreEqual(_blogicResultTypeName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

                // _blogic.Execute() が呼び出されており、正しい引数がわたっていること
                Coordinator_IBLogicImpl03 blogicImpl = (Coordinator_IBLogicImpl03)accessor._blogic;
                Assert.IsTrue(blogicImpl.IsCalledExecute);
                Assert.AreEqual(1, blogicImpl.ExecuteBLogicParam.Items.Count);
                Assert.AreEqual(_requestName, blogicImpl.ExecuteBLogicParam.Items[EventController.REQUEST_NAME]);
                Assert.IsInstanceOfType(blogicImpl.ExecuteBLogicParam.ParamData, _blogicParamType);

                // _converter.Reverseメソッドが呼び出されていないこと
                Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledReverse);

                // eventController.ReportBLogicFailed() が呼び出されないこと
                Assert.IsFalse(eventController.IsCalledReportBLogicFailed);

                // eventController.Itemsが変更されていないこと
                Assert.AreEqual(0, eventController.Items.Count);

                // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
                Assert.AreEqual(0, eventController.ErrorPaths.Count);
            }
            catch (Exception e)
            {
                Assert.Fail("予期せぬエラーです。" + e.ToString());
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(
                typeof(Coordinator), typeof(Coordinator_ExceptionStub01), "E_BLOGIC_EXECUTE_EXCEPTION"));
        }

        /// <summary>
        /// Execute (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// ビジネスロジック実行後の戻り値が null の場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest43()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl04 blogic = new Coordinator_IBLogicImpl04();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            try
            {
                // 実施
                target.Execute(eventController);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // 確認
                Assert.AreEqual(MessageManager.GetMessage("E_BLOGIC_NULL_RESULT"), e.Message);

                TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

                // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
                Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

                // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
                Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

                // _validator.ValidationFilePathが初期値であること
                Assert.IsNull(accessor._validator.ValidationFilePath);

                // _validator.RuleSetが初期値であること
                Assert.IsNull(accessor._validator.RuleSet);

                // _validator.Validate() が呼び出されないこと
                Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

                // eventController.ReportCustomValidating() が呼び出され、引数で渡された
                // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
                Assert.IsTrue(eventController.IsCalledReportCustomValidating);
                Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

                // eventController.ReportValidationFailed() が呼び出されないこと
                Assert.IsFalse(eventController.IsCalledReportValidationFailed);

                // _converter.Convert() が呼び出されていないこと
                Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

                // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
                // 引数 executionResultの各プロパティが初期値
                // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
                Assert.IsTrue(eventController.IsCalledReportPreprocessed);
                Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
                Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
                Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
                Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

                // eventController.ReportPreprocessedFailed() が呼び出されていないこと
                Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

                // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
                Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
                Assert.AreEqual(_blogicName, Coordinator_BLogicFactoryStub01._blogicName);
                Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
                Assert.AreEqual(_blogicResultTypeName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

                // _blogic.Execute() が呼び出されており、正しい引数がわたっていること
                Coordinator_IBLogicImpl04 blogicImpl = (Coordinator_IBLogicImpl04)accessor._blogic;
                Assert.IsTrue(blogicImpl.IsCalledExecute);
                Assert.AreEqual(1, blogicImpl.ExecuteBLogicParam.Items.Count);
                Assert.AreEqual(_requestName, blogicImpl.ExecuteBLogicParam.Items[EventController.REQUEST_NAME]);
                Assert.IsInstanceOfType(blogicImpl.ExecuteBLogicParam.ParamData, _blogicParamType);

                // _converter.Reverseメソッドが呼び出されていないこと
                Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledReverse);

                // eventController.ReportBLogicFailed() が呼び出されないこと
                Assert.IsFalse(eventController.IsCalledReportBLogicFailed);

                // eventController.Itemsが変更されていないこと
                Assert.AreEqual(0, eventController.Items.Count);

                // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
                Assert.AreEqual(0, eventController.ErrorPaths.Count);
            }
            catch (Exception e)
            {
                Assert.Fail("予期せぬエラーです。" + e.ToString());
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(
                typeof(Coordinator), typeof(TerasolunaException), "E_BLOGIC_NULL_RESULT"));
        }

        /// <summary>
        /// Execute (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.ConvertId が null・空文字列でなく、
        /// ビジネスロジック実行後の blogicResult.ResultString が Success、
        /// blogicResult.BLogicResultData が null の場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest44()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            string convertId = "convertId";
            eventController.ConvertId = convertId;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl05 blogic = new Coordinator_IBLogicImpl05();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            try
            {
                // 実施
                target.Execute(eventController);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // 確認
                Assert.AreEqual(MessageManager.GetMessage("E_BLOGIC_RESULT_NULL_DATASET", "ConvertId"), e.Message);

                TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

                // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
                Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

                // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
                Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

                // _validator.ValidationFilePathが初期値であること
                Assert.IsNull(accessor._validator.ValidationFilePath);

                // _validator.RuleSetが初期値であること
                Assert.IsNull(accessor._validator.RuleSet);

                // _validator.Validate() が呼び出されないこと
                Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

                // eventController.ReportCustomValidating() が呼び出され、引数で渡された
                // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
                Assert.IsTrue(eventController.IsCalledReportCustomValidating);
                Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

                // eventController.ReportValidationFailed() が呼び出されないこと
                Assert.IsFalse(eventController.IsCalledReportValidationFailed);

                // _converter.Convert() が呼び出されており、正しい引数がわたっていること
                Coordinator_IConverterImpl01 converter = (Coordinator_IConverterImpl01)accessor._converter;
                Assert.IsTrue(converter.IsCalledConvert);
                Assert.AreEqual(convertId, converter.ConvertId);
                Assert.AreSame(eventController.ViewData, converter.ConvertSrcData);
                Assert.IsInstanceOfType(converter.ConvertDestData, _blogicParamType);

                // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
                // 引数 executionResultの各プロパティが初期値
                // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
                Assert.IsTrue(eventController.IsCalledReportPreprocessed);
                Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
                Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
                Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
                Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

                // eventController.ReportPreprocessedFailed() が呼び出されていないこと
                Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

                // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
                Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
                Assert.AreEqual(_blogicName, Coordinator_BLogicFactoryStub01._blogicName);
                Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
                Assert.AreEqual(_blogicResultTypeName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

                // _blogic.Execute() が呼び出されており、正しい引数がわたっていること
                Coordinator_IBLogicImpl05 blogicImpl = (Coordinator_IBLogicImpl05)accessor._blogic;
                Assert.IsTrue(blogicImpl.IsCalledExecute);
                Assert.AreEqual(1, blogicImpl.ExecuteBLogicParam.Items.Count);
                Assert.AreEqual(_requestName, blogicImpl.ExecuteBLogicParam.Items[EventController.REQUEST_NAME]);
                Assert.IsInstanceOfType(blogicImpl.ExecuteBLogicParam.ParamData, _blogicParamType);

                // _converter.Reverseメソッドが呼び出されていないこと
                Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledReverse);

                // eventController.ReportBLogicFailed() が呼び出されないこと
                Assert.IsFalse(eventController.IsCalledReportBLogicFailed);

                // eventController.Itemsが変更されていないこと
                Assert.AreEqual(0, eventController.Items.Count);

                // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
                Assert.AreEqual(0, eventController.ErrorPaths.Count);
            }
            catch (Exception e)
            {
                Assert.Fail("予期せぬエラーです。" + e.ToString());
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(
                typeof(Coordinator), typeof(TerasolunaException), "E_BLOGIC_RESULT_NULL_DATASET", "ConvertId"));
        }

        /// <summary>
        /// Execute (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// ビジネスロジック実行後の blogicResult.ResultString が Success、
        /// blogicResult.BLogicResultData が null の場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest45()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl05 blogic = new Coordinator_IBLogicImpl05();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            ExecutionResult actual = target.Execute(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
            Assert.AreEqual(_blogicName, Coordinator_BLogicFactoryStub01._blogicName);
            Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
            Assert.AreEqual(_blogicResultTypeName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

            // _blogic.Execute() が呼び出されており、正しい引数がわたっていること
            Coordinator_IBLogicImpl05 blogicImpl = (Coordinator_IBLogicImpl05)accessor._blogic;
            Assert.IsTrue(blogicImpl.IsCalledExecute);
            Assert.AreEqual(1, blogicImpl.ExecuteBLogicParam.Items.Count);
            Assert.AreEqual(_requestName, blogicImpl.ExecuteBLogicParam.Items[EventController.REQUEST_NAME]);
            Assert.IsInstanceOfType(blogicImpl.ExecuteBLogicParam.ParamData, _blogicParamType);

            // _converter.Reverseメソッドが呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledReverse);

            // eventController.ReportBLogicFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportBLogicFailed);

            // eventController.Itemsが変更されていないこと
            Assert.AreEqual(0, eventController.Items.Count);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 戻り値のexecutionResult.Errorsが_blogic.Execute()の戻り値のBlogicResult.Errorsと要素が同一
            Assert.AreEqual(0, actual.Errors.Count);

            // 戻り値のexecutionResult.ResultStringが_blogic.Execute()の戻り値のBLogicResult.ResultStringと同一
            Assert.AreEqual(BLogicResult.SUCCESS, actual.ResultString);

            // 戻り値のexecutionResult.BLogicResultDataが_blogic.Execute()の戻り値のBLogicResult.ResultDataと同一
            Assert.IsNull(actual.BLogicResultData);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.ErrorHandler が null でなく、
        /// ビジネスロジック実行後の blogicResult.ResultString が Success 以外の場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest46()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            Coordinator_IErrorHandlerImpl01 errorHandler = new Coordinator_IErrorHandlerImpl01();
            eventController.ErrorHandler = errorHandler;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl06 blogic = new Coordinator_IBLogicImpl06();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            ExecutionResult actual = target.Execute(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
            Assert.AreEqual(_blogicName, Coordinator_BLogicFactoryStub01._blogicName);
            Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
            Assert.AreEqual(_blogicResultTypeName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

            // _blogic.Execute() が呼び出されており、正しい引数がわたっていること
            Coordinator_IBLogicImpl06 blogicImpl = (Coordinator_IBLogicImpl06)accessor._blogic;
            Assert.IsTrue(blogicImpl.IsCalledExecute);
            Assert.AreEqual(1, blogicImpl.ExecuteBLogicParam.Items.Count);
            Assert.AreEqual(_requestName, blogicImpl.ExecuteBLogicParam.Items[EventController.REQUEST_NAME]);
            Assert.IsInstanceOfType(blogicImpl.ExecuteBLogicParam.ParamData, _blogicParamType);

            // _converter.Reverseメソッドが呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledReverse);

            // eventController.ErrorHandler.HandleError()が呼ばれ引数が正しいこと
            Assert.IsTrue(errorHandler.IsCalledHandleError);
            Assert.AreEqual("FAILED", errorHandler.HandleErrorResultString);
            Assert.AreEqual(0, errorHandler.HandleErrorMessages.Count);
            Assert.IsInstanceOfType(errorHandler.HandleErrorView, _viewDataType);

            // eventController.ReportBLogicFailedメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(eventController.IsCalledReportBLogicFailed);
            Assert.AreEqual(blogic.ExecuteBLogicResult.Errors.Count, eventController.ReportBLogicFailedExecutionResult.Errors.Count);
            Assert.AreEqual(blogic.ExecuteBLogicResult.ResultString, eventController.ReportBLogicFailedExecutionResult.ResultString);
            Assert.IsInstanceOfType(eventController.ReportBLogicFailedExecutionResult.BLogicResultData, typeof(Coordinator_DataSetStub04));

            // eventController.Itemsが変更されていないこと
            Assert.AreEqual(0, eventController.Items.Count);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 戻り値のexecutionResult.Errorsが_blogic.Execute()の戻り値のBlogicResult.Errorsと要素が同一
            Assert.AreEqual(0, actual.Errors.Count);

            // 戻り値のexecutionResult.ResultStringが_blogic.Execute()の戻り値のBLogicResult.ResultStringと同一
            Assert.AreEqual("FAILED", actual.ResultString);

            // 戻り値のexecutionResult.BLogicResultDataが_blogic.Execute()の戻り値のBLogicResult.ResultDataと同一
            Assert.IsInstanceOfType(actual.BLogicResultData, typeof(Coordinator_DataSetStub04));

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }


        /// <summary>
        /// Execute (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.ErrorHandler が null であり、
        /// ビジネスロジック実行後の blogicResult.ResultString が Success 以外の場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest47()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl06 blogic = new Coordinator_IBLogicImpl06();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            ExecutionResult actual = target.Execute(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
            Assert.AreEqual(_blogicName, Coordinator_BLogicFactoryStub01._blogicName);
            Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
            Assert.AreEqual(_blogicResultTypeName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

            // _blogic.Execute() が呼び出されており、正しい引数がわたっていること
            Coordinator_IBLogicImpl06 blogicImpl = (Coordinator_IBLogicImpl06)accessor._blogic;
            Assert.IsTrue(blogicImpl.IsCalledExecute);
            Assert.AreEqual(1, blogicImpl.ExecuteBLogicParam.Items.Count);
            Assert.AreEqual(_requestName, blogicImpl.ExecuteBLogicParam.Items[EventController.REQUEST_NAME]);
            Assert.IsInstanceOfType(blogicImpl.ExecuteBLogicParam.ParamData, _blogicParamType);

            // _converter.Reverseメソッドが呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledReverse);

            // eventController.ReportBLogicFailedメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(eventController.IsCalledReportBLogicFailed);
            Assert.AreEqual(blogic.ExecuteBLogicResult.Errors.Count, eventController.ReportBLogicFailedExecutionResult.Errors.Count);
            Assert.AreEqual(blogic.ExecuteBLogicResult.ResultString, eventController.ReportBLogicFailedExecutionResult.ResultString);
            Assert.IsInstanceOfType(eventController.ReportBLogicFailedExecutionResult.BLogicResultData, typeof(Coordinator_DataSetStub04));

            // eventController.Itemsが変更されていないこと
            Assert.AreEqual(0, eventController.Items.Count);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 戻り値のexecutionResult.Errorsが_blogic.Execute()の戻り値のBlogicResult.Errorsと要素が同一
            Assert.AreEqual(0, actual.Errors.Count);

            // 戻り値のexecutionResult.ResultStringが_blogic.Execute()の戻り値のBLogicResult.ResultStringと同一
            Assert.AreEqual("FAILED", actual.ResultString);

            // 戻り値のexecutionResult.BLogicResultDataが_blogic.Execute()の戻り値のBLogicResult.ResultDataと同一
            Assert.IsInstanceOfType(actual.BLogicResultData, typeof(Coordinator_DataSetStub04));

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// ビジネスロジック実行後の blogicResult.Items の要素数が 1 の場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest48()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            string key = "key01";
            string item = "Item01";
            eventController.Items.Add(key, item);

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl07 blogic = new Coordinator_IBLogicImpl07();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            ExecutionResult actual = target.Execute(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
            Assert.AreEqual(_blogicName, Coordinator_BLogicFactoryStub01._blogicName);
            Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
            Assert.AreEqual(_blogicResultTypeName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

            // _blogic.Execute() が呼び出されており、正しい引数がわたっていること
            Coordinator_IBLogicImpl07 blogicImpl = (Coordinator_IBLogicImpl07)accessor._blogic;
            Assert.IsTrue(blogicImpl.IsCalledExecute);
            Assert.AreEqual(2, blogicImpl.ExecuteBLogicParam.Items.Count);
            Assert.AreEqual(_requestName, blogicImpl.ExecuteBLogicParam.Items[EventController.REQUEST_NAME]);
            Assert.AreEqual(item, blogicImpl.ExecuteBLogicParam.Items[key]);
            Assert.IsInstanceOfType(blogicImpl.ExecuteBLogicParam.ParamData, _blogicParamType);

            // _converter.Reverseメソッドが呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledReverse);

            // eventController.ReportBLogicFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportBLogicFailed);

            // eventController.ItemsがビジネスロジックのItemsで上書きされていること
            Assert.AreEqual(1, eventController.Items.Count);
            Assert.AreEqual("value01", eventController.Items["key01"]);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 戻り値のexecutionResult.Errorsが_blogic.Execute()の戻り値のBlogicResult.Errorsと要素が同一
            Assert.AreEqual(0, actual.Errors.Count);

            // 戻り値のexecutionResult.ResultStringが_blogic.Execute()の戻り値のBLogicResult.ResultStringと同一
            Assert.AreEqual(BLogicResult.SUCCESS, actual.ResultString);

            // 戻り値のexecutionResult.BLogicResultDataが_blogic.Execute()の戻り値のBLogicResult.ResultDataと同一
            Assert.IsInstanceOfType(actual.BLogicResultData, typeof(Coordinator_DataSetStub04));

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// ビジネスロジック実行後の blogicResult.Items の要素数が 1 の場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest49()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            string key = "key001";
            string item = "Item001";
            eventController.Items.Add(key, item);

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl08 blogic = new Coordinator_IBLogicImpl08();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            ExecutionResult actual = target.Execute(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
            Assert.AreEqual(_blogicName, Coordinator_BLogicFactoryStub01._blogicName);
            Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
            Assert.AreEqual(_blogicResultTypeName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

            // _blogic.Execute() が呼び出されており、正しい引数がわたっていること
            Coordinator_IBLogicImpl08 blogicImpl = (Coordinator_IBLogicImpl08)accessor._blogic;
            Assert.IsTrue(blogicImpl.IsCalledExecute);
            Assert.AreEqual(2, blogicImpl.ExecuteBLogicParam.Items.Count);
            Assert.AreEqual(_requestName, blogicImpl.ExecuteBLogicParam.Items[EventController.REQUEST_NAME]);
            Assert.AreEqual(item, blogicImpl.ExecuteBLogicParam.Items[key]);
            Assert.IsInstanceOfType(blogicImpl.ExecuteBLogicParam.ParamData, _blogicParamType);

            // _converter.Reverseメソッドが呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledReverse);

            // eventController.ReportBLogicFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportBLogicFailed);

            // eventController.ItemsがビジネスロジックのItemsで上書きされていること
            Assert.AreEqual(4, eventController.Items.Count);
            Assert.AreEqual(item, eventController.Items[key]);
            Assert.AreEqual("value01", eventController.Items["key01"]);
            Assert.AreEqual("value02", eventController.Items["key02"]);
            Assert.AreEqual("value03", eventController.Items["key03"]);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 戻り値のexecutionResult.Errorsが_blogic.Execute()の戻り値のBlogicResult.Errorsと要素が同一
            Assert.AreEqual(0, actual.Errors.Count);

            // 戻り値のexecutionResult.ResultStringが_blogic.Execute()の戻り値のBLogicResult.ResultStringと同一
            Assert.AreEqual(BLogicResult.SUCCESS, actual.ResultString);

            // 戻り値のexecutionResult.BLogicResultDataが_blogic.Execute()の戻り値のBLogicResult.ResultDataと同一
            Assert.IsInstanceOfType(actual.BLogicResultData, typeof(Coordinator_DataSetStub04));

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.RequestName が null の場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest50()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            eventController.RequestName = null;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            ExecutionResult actual = target.Execute(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
            Assert.AreEqual(_blogicName, Coordinator_BLogicFactoryStub01._blogicName);
            Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
            Assert.AreEqual(_blogicResultTypeName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

            // _blogic.Execute() が呼び出されており、正しい引数がわたっていること
            Coordinator_IBLogicImpl01 blogicImpl = (Coordinator_IBLogicImpl01)accessor._blogic;
            Assert.IsTrue(blogicImpl.IsCalledExecute);
            Assert.AreEqual(0, blogicImpl.ExecuteBLogicParam.Items.Count);

            // _converter.Reverseメソッドが呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledReverse);

            // eventController.ReportBLogicFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportBLogicFailed);

            // eventController.Itemsが変更されていないこと
            Assert.AreEqual(0, eventController.Items.Count);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 戻り値のexecutionResult.Errorsが_blogic.Execute()の戻り値のBlogicResult.Errorsと要素が同一
            Assert.AreEqual(0, actual.Errors.Count);

            // 戻り値のexecutionResult.ResultStringが_blogic.Execute()の戻り値のBLogicResult.ResultStringと同一
            Assert.AreEqual(BLogicResult.SUCCESS, actual.ResultString);

            // 戻り値のexecutionResult.BLogicResultDataが_blogic.Execute()の戻り値のBLogicResult.ResultDataと同一
            Assert.IsInstanceOfType(actual.BLogicResultData, typeof(Coordinator_DataSetStub04));

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.ErrorHandler が null でなく、
        /// 前処理完了イベントでのエラー数が 0 で、resultString が任意の文字列の場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest51()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub08 eventController = new Coordinator_EventControllerStub08();
            Coordinator_IErrorHandlerImpl01 errorHandler = new Coordinator_IErrorHandlerImpl01();
            eventController.ErrorHandler = errorHandler;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            ExecutionResult actual = target.Execute(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ErrorHandler.HandleError()が呼ばれ引数が正しいこと
            Assert.IsTrue(errorHandler.IsCalledHandleError);
            Assert.AreEqual("terasoluna", errorHandler.HandleErrorResultString);
            Assert.AreEqual(0, errorHandler.HandleErrorMessages.Count);
            Assert.IsInstanceOfType(errorHandler.HandleErrorView, _viewDataType);

            // eventController.ReportPreprocessedFailedメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(eventController.IsCalledReportPreprocessedFailed);
            Assert.AreEqual(0, eventController.ReportPreprocessedFailedExecutionResult.Errors.Count);
            Assert.AreEqual("terasoluna", eventController.ReportPreprocessedFailedExecutionResult.ResultString);
            Assert.IsNull(eventController.ReportPreprocessedFailedExecutionResult.BLogicResultData);

            // BLogicFactory.CreateBLogicメソッドが呼び出されないこと
            Assert.IsFalse(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);

            // _converter.Reverseメソッドが呼び出されないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledReverse);

            // eventController.ReportBLogicFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportBLogicFailed);

            // eventController.Itemsが変更されていないこと
            Assert.AreEqual(0, eventController.Items.Count);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 戻り値のexecutionResult.ErrorsとeventController.ReportPreprocessed()で追加されたexecutionResult.Errorsの中身が同じ
            Assert.AreEqual(0, actual.Errors.Count);

            // 戻り値のexecutionResult.ResultStringが"terasoluna"
            Assert.AreEqual("terasoluna", actual.ResultString);

            // 戻り値のexecutionResult.BLogicResultDataがnull
            Assert.IsNull(actual.BLogicResultData);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        ///<summary>
        /// CreateCoordinator () のテスト
        ///</summary>
        ///<remarks>
        /// AppSettingsにCoordinatorTypeNameが設定されていない場合
        /// デフォルトのCoordinatorインスタンスが生成されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void CreateCoordinatorTest01()
        {
            // 実行
            Coordinator actual = Coordinator.CreateCoordinator();

            // 確認
            Assert.IsInstanceOfType(actual, typeof(Coordinator));
            Assert.AreEqual(typeof(Coordinator), TERASOLUNA_Fw_Client_CoordinatorAccessor._coordinatorType);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        ///<summary>
        /// CreateCoordinator () のテスト
        ///</summary>
        ///<remarks>
        /// AppSettingsのCoordinatorTypeNameの値がCoordinator派生クラスの場合
        /// Coordinator型のインスタンスが返却されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void CreateCoordinatorTest02()
        {
            // 準備
            ConfigurationManager.AppSettings["CoordinatorTypeName"] = typeof(Coordinator_Stub01).AssemblyQualifiedName;

            // 実行
            Coordinator actual = Coordinator.CreateCoordinator();

            // 確認
            Assert.IsInstanceOfType(actual, typeof(Coordinator_Stub01));
            Assert.AreEqual(typeof(Coordinator_Stub01), TERASOLUNA_Fw_Client_CoordinatorAccessor._coordinatorType);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        ///<summary>
        /// CreateCoordinator () のテスト
        ///</summary>
        ///<remarks>
        /// AppSettingsのCoordinatorTypeNameの値がCoordinator派生クラスでない場合
        /// TerasolunaExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void CreateCoordinatorTest03()
        {
            // 準備
            ConfigurationManager.AppSettings["CoordinatorTypeName"] = typeof(Coordinator_Stub02).AssemblyQualifiedName;

            // 実行
            Coordinator actual;
            try
            {
                //実行
                actual = Coordinator.CreateCoordinator();
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                //成功
                Assert.AreEqual(MessageManager.GetMessage("E_INVALID_CAST_TYPE_FROM_GENERICPARAM",
                    typeof(Coordinator_Stub02).AssemblyQualifiedName, typeof(Coordinator).AssemblyQualifiedName), e.Message);
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        ///<summary>
        /// CreateCoordinator () のテスト
        ///</summary>
        ///<remarks>
        /// _corrdinatorTypeにキャッシュされている場合、キャッシュされている
        /// Coordinator型のインスタンスが返却されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void CreateCoordinatorTest04()
        {
            // 準備
            TERASOLUNA_Fw_Client_CoordinatorAccessor._coordinatorType = typeof(Coordinator_Stub01);

            // 実行
            Coordinator actual = Coordinator.CreateCoordinator();

            // 確認
            Assert.IsInstanceOfType(actual, typeof(Coordinator_Stub01));
            Assert.AreEqual(typeof(Coordinator_Stub01), TERASOLUNA_Fw_Client_CoordinatorAccessor._coordinatorType);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        ///<summary>
        /// ExecuteAsyncCancel () のテスト
        ///</summary>
        ///<remarks>
        /// _isBusyフィールドがtrue、_blogicがnullの場合、
        /// InvalidOperationExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ExecuteAsyncCancelTest01()
        {
            Coordinator target = Coordinator.CreateCoordinator();
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);
            accessor._blogic = null;
            accessor._isBusy = true;

            try
            {
                //実行
                target.ExecuteAsyncCancel();
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (InvalidOperationException e)
            {
                //成功
                Assert.AreEqual(MessageManager.GetMessage("E_BLOGIC_NOT_EXECUTEASYNC_OPERATED"), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(
               typeof(Coordinator), typeof(InvalidOperationException), "E_BLOGIC_NOT_EXECUTEASYNC_OPERATED"));
        }

        ///<summary>
        /// ExecuteAsyncCancel () のテスト
        ///</summary>
        ///<remarks>
        /// _isBusyフィールドがtrue、_blogicがnot null(ICancelable実装クラス)の場合、
        /// ビジネスロジックのCancelメソッドが呼ばれることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ExecuteAsyncCancelTest02()
        {
            Coordinator target = Coordinator.CreateCoordinator();
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);
            Coordinator_IBLogicImpl09 blogic = new Coordinator_IBLogicImpl09();
            accessor._blogic = blogic;
            accessor._isBusy = true;

            //実行
            target.ExecuteAsyncCancel();

            // 確認
            Assert.IsTrue(blogic.Cancelled);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        ///<summary>
        /// ExecuteAsyncCancel () のテスト
        ///</summary>
        ///<remarks>
        /// _isBusyフィールドがtrue、_blogicがnot null(ICancelableを実装していないクラス)の場合、
        /// InvalidOperationExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ExecuteAsyncCancelTest03()
        {
            Coordinator target = Coordinator.CreateCoordinator();
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            accessor._blogic = blogic;
            accessor._isBusy = true;

            try
            {
                //実行
                target.ExecuteAsyncCancel();
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (InvalidOperationException e)
            {
                //成功
                Assert.AreEqual(MessageManager.GetMessage("E_BLOGIC_NOT_IMPLEMENTS_ICANCELABLE"), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(
               typeof(Coordinator), typeof(InvalidOperationException), "E_BLOGIC_NOT_IMPLEMENTS_ICANCELABLE"));
        }

        ///<summary>
        /// ExecuteAsyncCancel () のテスト
        ///</summary>
        ///<remarks>
        /// _isBusyフィールドがfalse、_blogicがnot null(ICancelable実装クラス)の場合、
        /// InvalidOperationExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ExecuteAsyncCancelTest04()
        {
            Coordinator target = Coordinator.CreateCoordinator();
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);
            Coordinator_IBLogicImpl09 blogic = new Coordinator_IBLogicImpl09();
            accessor._blogic = blogic;
            accessor._isBusy = false;

            try
            {
                //実行
                target.ExecuteAsyncCancel();
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (InvalidOperationException e)
            {
                //成功
                Assert.AreEqual(MessageManager.GetMessage("E_BLOGIC_NOT_EXECUTEASYNC_OPERATED"), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(
               typeof(Coordinator), typeof(InvalidOperationException), "E_BLOGIC_NOT_EXECUTEASYNC_OPERATED"));
        }

        ///<summary>
        /// ExecuteAsyncProgressChangedCallback (object) のテスト
        ///</summary>
        ///<remarks>
        /// 引数にExecuteProgressChangedEventArgsクラスのインスタンスを指定した場合、
        /// EventControllerのReportExecuteProgressChangedメソッドが実行されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ExecuteAsyncProgressChangedCallbackTest01()
        {
            // 準備
            Coordinator target = Coordinator.CreateCoordinator();
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);
            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            accessor._eventController = eventController;

            int percent = 80;
            ExecuteProgressChangedEventArgs obj = new ExecuteProgressChangedEventArgs(percent);

            //実行
            accessor.ExecuteAsyncProgressChangedCallback(obj);

            //検証
            Assert.IsTrue(eventController.IsCalledReportExecuteProgressChanged);
            Assert.AreEqual(percent, eventController.ReportExecuteProgressChangedParamPercentage);
            Assert.AreEqual(0, eventController.ReportExecuteProgressChangedItems.Count);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        ///<summary>
        /// DoExecuteProgressChanged (object, ExecuteProgressChangedEventArgs) のテスト
        ///</summary>
        ///<remarks>
        /// senderがnullの場合、ArgumentNullExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void DoExecuteProgressChangedTest01()
        {
            // 準備
            Coordinator target = Coordinator.CreateCoordinator();
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);
            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            accessor._eventController = eventController;
            accessor._asyncOpe = null;

            int percent = 80;
            ExecuteProgressChangedEventArgs e = new ExecuteProgressChangedEventArgs(percent);
            object sender = null;

            try
            {
                //実行
                accessor.DoExecuteProgressChanged(sender, e);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException ex)
            {
                Assert.IsInstanceOfType(ex, typeof(ArgumentNullException));
                //ArgumentNullException innerException = (ArgumentNullException)ex.InnerException;
                Assert.AreEqual("sender", ex.ParamName);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(
               typeof(Coordinator), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "sender"));
        }

        ///<summary>
        /// DoExecuteProgressChanged (object, ExecuteProgressChangedEventArgs) のテスト
        ///</summary>
        ///<remarks>
        /// 引数が共にnullでない場合、
        /// EventControllerのReportExecuteProgressChangedメソッドが実行されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void DoExecuteProgressChangedTest02()
        {
            // 準備
            Coordinator target = Coordinator.CreateCoordinator();
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);
            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            accessor._eventController = eventController;
            accessor._asyncOpe = null;

            int percent = 80;
            ExecuteProgressChangedEventArgs e = new ExecuteProgressChangedEventArgs(percent);
            object sender = new object();

            //実行
            accessor.DoExecuteProgressChanged(sender, e);

            //検証
            Assert.IsTrue(eventController.IsCalledReportExecuteProgressChanged);
            Assert.AreEqual(percent, eventController.ReportExecuteProgressChangedParamPercentage);
            Assert.AreEqual(0, eventController.ReportExecuteProgressChangedItems.Count);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        ///<summary>
        /// DoExecuteProgressChanged (object, ExecuteProgressChangedEventArgs) のテスト
        ///</summary>
        ///<remarks>
        /// ExecuteProgressChangedEventArgsがnullの場合、ArgumentNullExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void DoExecuteProgressChangedTest03()
        {
            // 準備
            Coordinator target = Coordinator.CreateCoordinator();
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);
            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            accessor._eventController = eventController;
            accessor._asyncOpe = null;

            ExecuteProgressChangedEventArgs e = null;
            object sender = new object();

            try
            {
                //実行
                accessor.DoExecuteProgressChanged(sender, e);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException ex)
            {
                Assert.IsInstanceOfType(ex, typeof(ArgumentNullException));
                //ArgumentNullException innerException = (ArgumentNullException)ex.InnerException;
                Assert.AreEqual("e", ex.ParamName);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(
               typeof(Coordinator), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "e"));
        }

        /// <summary>
        /// Constructor() のテスト
        /// </summary>
        /// <remarks>
        /// 正常にインスタンス化され、デリゲート、コールバックデリゲートが初期化されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ConstructorTest01()
        {
            //実行
            Coordinator target = new Coordinator();

            //検証
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);
            Assert.IsNotNull(accessor._doExecuteAsyncDelegate);
            Assert.IsNotNull(accessor._executeAsyncCompletedCallback);
            Assert.IsNotNull(accessor._executeAsyncProgressChangedCallback);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// ExecuteAsync (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController が null の場合、ArgumentNullException
        /// がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteAsyncTest01()
        {
            Coordinator target = new Coordinator();

            EventController eventController = null;

            try
            {
                // 実施
                target.ExecuteAsync(eventController);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // 成功
                Assert.AreEqual("eventController", e.ParamName);
            }
            catch (Exception e)
            {
                Assert.Fail("予期せぬエラーです。" + e.ToString());
            }

            // 非同期処理が実行されていないこと
            Assert.IsFalse(target.IsBusy);

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(
                typeof(Coordinator), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "eventController"));
        }

        /// <summary>
        /// ExecuteAsync (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController が null でない場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteAsyncTest02()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            target.ExecuteAsync(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // eventController.ReportExecuteCompleted() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportExecuteCompleted);

            // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
            Assert.AreEqual(_blogicName, Coordinator_BLogicFactoryStub01._blogicName);
            Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
            Assert.AreEqual(_blogicResultTypeName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 非同期処理が実行中であること
            Assert.IsTrue(target.IsBusy);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// ExecuteAsync (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// eventController.ErrorHandler が null でない場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteAsyncTest03()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            Coordinator_IErrorHandlerImpl01 errorHandler = new Coordinator_IErrorHandlerImpl01();
            eventController.ErrorHandler = errorHandler;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            target.ExecuteAsync(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // eventController.ErrorHandlerのClearErrorメソッドが呼び出され、正しい引数がわたっていること
            Assert.IsTrue(errorHandler.IsCalledClearError);
            Assert.AreSame(eventController.ViewData, errorHandler.ClearErrorView);
            Assert.IsInstanceOfType(errorHandler.ClearErrorView, _viewDataType);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ErrorHandler.HandleError() が呼ばれないこと
            Assert.IsFalse(errorHandler.IsCalledHandleError);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // eventController.ReportExecuteCompleted() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportExecuteCompleted);

            // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
            Assert.AreEqual(_blogicName, Coordinator_BLogicFactoryStub01._blogicName);
            Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
            Assert.AreEqual(_blogicResultTypeName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 非同期処理が実行中であること
            Assert.IsTrue(target.IsBusy);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// ExecuteAsync (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// eventController.ErrorHandler が null でなく、
        /// eventController.ErrorPaths のリスト数が 1 の場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteAsyncTest04()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            Coordinator_IErrorHandlerImpl01 errorHandler = new Coordinator_IErrorHandlerImpl01();
            eventController.ErrorHandler = errorHandler;
            string errorPath01 = "errorPath01";
            eventController.ErrorPaths.Add(errorPath01);

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            target.ExecuteAsync(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // eventController.ErrorHandlerのClearErrorメソッドが呼び出され、正しい引数がわたっていること
            Assert.IsTrue(errorHandler.IsCalledClearError);
            Assert.AreSame(eventController.ViewData, errorHandler.ClearErrorView);
            Assert.IsInstanceOfType(errorHandler.ClearErrorView, _viewDataType);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ErrorHandler.HandleError() が呼ばれないこと
            Assert.IsFalse(errorHandler.IsCalledHandleError);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // eventController.ReportExecuteCompleted() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportExecuteCompleted);

            // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
            Assert.AreEqual(_blogicName, Coordinator_BLogicFactoryStub01._blogicName);
            Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
            Assert.AreEqual(_blogicResultTypeName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 非同期処理が実行中であること
            Assert.IsTrue(target.IsBusy);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// ExecuteAsync (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// eventController.ErrorHandler が null でなく、
        /// eventController.ErrorPaths のリスト数が 3 の場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteAsyncTest05()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            Coordinator_IErrorHandlerImpl01 errorHandler = new Coordinator_IErrorHandlerImpl01();
            eventController.ErrorHandler = errorHandler;
            string errorPath01 = "errorPath01";
            string errorPath02 = "errorPath02";
            string errorPath03 = "errorPath03";
            eventController.ErrorPaths.Add(errorPath01);
            eventController.ErrorPaths.Add(errorPath02);
            eventController.ErrorPaths.Add(errorPath03);

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            target.ExecuteAsync(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // eventController.ErrorHandlerのClearErrorメソッドが呼び出され、正しい引数がわたっていること
            Assert.IsTrue(errorHandler.IsCalledClearError);
            Assert.AreSame(eventController.ViewData, errorHandler.ClearErrorView);
            Assert.IsInstanceOfType(errorHandler.ClearErrorView, _viewDataType);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ErrorHandler.HandleError() が呼ばれないこと
            Assert.IsFalse(errorHandler.IsCalledHandleError);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // eventController.ReportExecuteCompleted() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportExecuteCompleted);

            // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
            Assert.AreEqual(_blogicName, Coordinator_BLogicFactoryStub01._blogicName);
            Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
            Assert.AreEqual(_blogicResultTypeName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 非同期処理が実行中であること
            Assert.IsTrue(target.IsBusy);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// ExecuteAsync (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// IValidator 実装クラスのインスタンスを生成できない場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteAsyncTest06()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            ConfigurationManager.AppSettings["ValidatorTypeName"] = typeof(Coordinator_IValidatorImpl02).AssemblyQualifiedName;

            try
            {
                // 実施
                target.ExecuteAsync(eventController);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // 成功
                Assert.AreEqual(MessageManager.GetMessage("E_MEMBER_ACCESS_EXCEPTION", typeof(Coordinator_IValidatorImpl02).AssemblyQualifiedName), e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("予期せぬエラーです。" + e.ToString());
            }

            // 非同期処理が実行されていないこと
            Assert.IsFalse(target.IsBusy);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// ExecuteAsync (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// IConverter 実装クラスのインスタンスを生成できない場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteAsyncTest07()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            ConfigurationManager.AppSettings["ConverterTypeName"] = typeof(Coordinator_IConverterImpl02).AssemblyQualifiedName;

            try
            {
                // 実施
                target.ExecuteAsync(eventController);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // 成功
                Assert.AreEqual(MessageManager.GetMessage("E_MEMBER_ACCESS_EXCEPTION", typeof(Coordinator_IConverterImpl02).AssemblyQualifiedName), e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("予期せぬエラーです。" + e.ToString());
            }

            // 非同期処理が実行されていないこと
            Assert.IsFalse(target.IsBusy);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// ExecuteAsync (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.ValidationFilePath が空文字列の場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteAsyncTest08()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            eventController.ValidationFilePath = string.Empty;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            target.ExecuteAsync(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // eventController.ReportExecuteCompleted() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportExecuteCompleted);

            // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
            Assert.AreEqual(_blogicName, Coordinator_BLogicFactoryStub01._blogicName);
            Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
            Assert.AreEqual(_blogicResultTypeName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 非同期処理が実行中であること
            Assert.IsTrue(target.IsBusy);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// ExecuteAsync (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.ValidationFilePath が null・空文字列でなく、
        /// eventController.ViewData が null の場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteAsyncTest09()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            string validationFilePath = "validationFilePath";
            eventController.ValidationFilePath = validationFilePath;
            eventController.ViewData = null;

            try
            {
                // 実施
                target.ExecuteAsync(eventController);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // 成功
                Assert.AreEqual(MessageManager.GetMessage("E_BLOGIC_NULL_DATASET", "ValidationFilePath", "ViewData"), e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("予期せぬエラーです。" + e.ToString());
            }

            // 非同期処理が実行されていないこと
            Assert.IsFalse(target.IsBusy);

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(Coordinator), typeof(TerasolunaException), "E_BLOGIC_NULL_DATASET", "ValidationFilePath", "ViewData"));
        }

        /// <summary>
        /// ExecuteAsync (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.ValidationFilePath が null・空文字列でなく、
        /// eventController.ViewData が null でない場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteAsyncTest10()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            string validationFilePath = "validationFilePath";
            eventController.ValidationFilePath = validationFilePath;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            target.ExecuteAsync(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // validator.ValidationFilePathにeventController.ValidationFilePathの値が設定されていること
            Assert.AreEqual(validationFilePath, accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validateメソッドが呼び出され、正しい引数がわたっていること
            Coordinator_IValidatorImpl01 validatorImpl = (Coordinator_IValidatorImpl01)accessor._validator;
            Assert.IsTrue(validatorImpl.IsCalledValidate);
            Assert.AreSame(eventController.ViewData, validatorImpl.ValidateDataSet);
            Assert.IsInstanceOfType(validatorImpl.ValidateDataSet, _viewDataType);

            // eventController.ReportCustomValidatingメソッドが呼び出され引数が_validator.Validate()の戻り値と等しいこと
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // eventController.ReportExecuteCompleted() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportExecuteCompleted);

            // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
            Assert.AreEqual(_blogicName, Coordinator_BLogicFactoryStub01._blogicName);
            Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
            Assert.AreEqual(_blogicResultTypeName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 非同期処理が実行中であること
            Assert.IsTrue(target.IsBusy);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// ExecuteAsync (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 入力値検証機能による単項目チェックにおいてエラー数が 1 の場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteAsyncTest11()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            string validationFilePath = "validationFilePath";
            eventController.ValidationFilePath = validationFilePath;

            ConfigurationManager.AppSettings["ValidatorTypeName"] = typeof(Coordinator_IValidatorImpl03).AssemblyQualifiedName;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            target.ExecuteAsync(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl03));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // validator.ValidationFilePathにeventController.ValidationFilePathの値が設定されていること
            Assert.AreEqual(validationFilePath, accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validateメソッドが呼び出され、正しい引数がわたっていること
            Coordinator_IValidatorImpl03 validatorImpl = (Coordinator_IValidatorImpl03)accessor._validator;
            Assert.IsTrue(validatorImpl.IsCalledValidate);
            Assert.AreSame(eventController.ViewData, validatorImpl.ValidateDataSet);
            Assert.IsInstanceOfType(validatorImpl.ValidateDataSet, _viewDataType);

            // eventController.ReportCustomValidatingメソッドが呼び出され引数が_validator.Validate()の戻り値と等しいこと
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(1, eventController.ReportCustomValidatingValidationResult.Errors.Count);
            Assert.AreSame(validatorImpl.ValidateValidationMessageInfo01, eventController.ReportCustomValidatingValidationResult.Errors[0]);

            // eventController.ErrorPathsにvalidationResult.Errorsの各ValidationMessageInfoのパスが追加されていること
            Assert.AreEqual(1, eventController.ErrorPaths.Count);
            Assert.AreEqual("errorPath01", eventController.ErrorPaths[0]);

            // eventController.ReportValidationFailedメソッドが呼び出され、正しい引数がわたってきていること
            Assert.IsTrue(eventController.IsCalledReportValidationFailed);
            Assert.AreEqual(1, eventController.ReportValidationFailedExecutionResult.Errors.Count);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Key, eventController.ReportValidationFailedExecutionResult.Errors[0].Key);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Message, eventController.ReportValidationFailedExecutionResult.Errors[0].Message);
            Assert.AreEqual(Coordinator.VALIDATION_FAILED, eventController.ReportValidationFailedExecutionResult.ResultString);
            Assert.IsNull(eventController.ReportValidationFailedExecutionResult.BLogicResultData);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessed);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // eventController.ReportExecuteCompleted() が呼び出され、正しい引数がわたっていること
            Assert.IsTrue(eventController.IsCalledReportExecuteCompleted);
            Assert.AreEqual(Coordinator.VALIDATION_FAILED, eventController.ReportExecuteCompletedExecutionResult.ResultString);
            Assert.IsNull(eventController.ReportExecuteCompletedExecutionResult.BLogicResultData);
            Assert.AreEqual(1, eventController.ReportExecuteCompletedExecutionResult.Errors.Count);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Key, eventController.ReportExecuteCompletedExecutionResult.Errors[0].Key);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Message, eventController.ReportExecuteCompletedExecutionResult.Errors[0].Message);
            Assert.IsNull(eventController.ReportExecuteCompletedError);
            Assert.IsFalse(eventController.ReportExecuteCompletedCancelled);

            // BLogicFactory.CreateBLogic() が呼び出されていないこと
            Assert.IsFalse(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);

            // 非同期処理が実行されていないこと
            Assert.IsFalse(target.IsBusy);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// ExecuteAsync (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 入力値検証機能による単項目チェックにおいてエラー数が 3 の場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteAsyncTest12()
        {
            // 準備
            Coordinator target = new Coordinator();
            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();

            string validationFilePath = "validationFilePath";
            eventController.ValidationFilePath = validationFilePath;

            ConfigurationManager.AppSettings["ValidatorTypeName"] = typeof(Coordinator_IValidatorImpl04).AssemblyQualifiedName;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            target.ExecuteAsync(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl04));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // validator.ValidationFilePathにeventController.ValidationFilePathの値が設定されていること
            Assert.AreEqual(validationFilePath, accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validateメソッドが呼び出され、正しい引数がわたっていること
            Coordinator_IValidatorImpl04 validatorImpl = (Coordinator_IValidatorImpl04)accessor._validator;
            Assert.IsTrue(validatorImpl.IsCalledValidate);
            Assert.AreSame(eventController.ViewData, validatorImpl.ValidateDataSet);
            Assert.IsInstanceOfType(validatorImpl.ValidateDataSet, _viewDataType);

            // eventController.ReportCustomValidatingメソッドが呼び出され引数が_validator.Validate()の戻り値と等しいこと
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(3, eventController.ReportCustomValidatingValidationResult.Errors.Count);
            Assert.AreSame(validatorImpl.ValidateValidationMessageInfo01, eventController.ReportCustomValidatingValidationResult.Errors[0]);
            Assert.AreSame(validatorImpl.ValidateValidationMessageInfo02, eventController.ReportCustomValidatingValidationResult.Errors[1]);
            Assert.AreSame(validatorImpl.ValidateValidationMessageInfo03, eventController.ReportCustomValidatingValidationResult.Errors[2]);

            // eventController.ErrorPathsにvalidationResult.Errorsの各ValidationMessageInfoのパスが追加されていること
            Assert.AreEqual(3, eventController.ErrorPaths.Count);
            Assert.AreEqual("errorPath01", eventController.ErrorPaths[0]);
            Assert.AreEqual("errorPath02", eventController.ErrorPaths[1]);
            Assert.AreEqual("errorPath03", eventController.ErrorPaths[2]);

            // eventController.ReportValidationFailedメソッドが呼び出され、正しい引数がわたってきていること
            Assert.IsTrue(eventController.IsCalledReportValidationFailed);
            Assert.AreEqual(3, eventController.ReportValidationFailedExecutionResult.Errors.Count);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Key, eventController.ReportValidationFailedExecutionResult.Errors[0].Key);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo02.Key, eventController.ReportValidationFailedExecutionResult.Errors[1].Key);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo03.Key, eventController.ReportValidationFailedExecutionResult.Errors[2].Key);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Message, eventController.ReportValidationFailedExecutionResult.Errors[0].Message);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo02.Message, eventController.ReportValidationFailedExecutionResult.Errors[1].Message);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo03.Message, eventController.ReportValidationFailedExecutionResult.Errors[2].Message);

            Assert.AreEqual(Coordinator.VALIDATION_FAILED, eventController.ReportValidationFailedExecutionResult.ResultString);
            Assert.IsNull(eventController.ReportValidationFailedExecutionResult.BLogicResultData);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessed);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // eventController.ReportExecuteCompleted() が呼び出され、正しい引数がわたっていること
            Assert.IsTrue(eventController.IsCalledReportExecuteCompleted);
            Assert.AreEqual(Coordinator.VALIDATION_FAILED, eventController.ReportExecuteCompletedExecutionResult.ResultString);
            Assert.IsNull(eventController.ReportExecuteCompletedExecutionResult.BLogicResultData);
            Assert.AreEqual(3, eventController.ReportExecuteCompletedExecutionResult.Errors.Count);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Key, eventController.ReportExecuteCompletedExecutionResult.Errors[0].Key);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo02.Key, eventController.ReportExecuteCompletedExecutionResult.Errors[1].Key);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo03.Key, eventController.ReportExecuteCompletedExecutionResult.Errors[2].Key);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Message, eventController.ReportExecuteCompletedExecutionResult.Errors[0].Message);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo02.Message, eventController.ReportExecuteCompletedExecutionResult.Errors[1].Message);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo03.Message, eventController.ReportExecuteCompletedExecutionResult.Errors[2].Message);           
            Assert.IsNull(eventController.ReportExecuteCompletedError);
            Assert.IsFalse(eventController.ReportExecuteCompletedCancelled);

            // BLogicFactory.CreateBLogic() が呼び出されていないこと
            Assert.IsFalse(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
                      
            // 非同期処理が実行されていないこと
            Assert.IsFalse(target.IsBusy);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// ExecuteAsync (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.EventHander が null でなく、
        /// 入力値検証機能による単項目チェックにおいてエラー数が 1 の場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteAsyncTest13()
        {
            // 準備
            Coordinator target = new Coordinator();
            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();

            string validationFilePath = "validationFilePath";
            eventController.ValidationFilePath = validationFilePath;

            Coordinator_IErrorHandlerImpl01 errorHandler = new Coordinator_IErrorHandlerImpl01();
            eventController.ErrorHandler = errorHandler;

            ConfigurationManager.AppSettings["ValidatorTypeName"] = typeof(Coordinator_IValidatorImpl03).AssemblyQualifiedName;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            target.ExecuteAsync(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // eventController.ErrorHandlerのClearErrorメソッドが呼び出され、正しい引数がわたっていること
            Assert.IsTrue(errorHandler.IsCalledClearError);
            Assert.AreSame(eventController.ViewData, errorHandler.ClearErrorView);
            Assert.IsInstanceOfType(errorHandler.ClearErrorView, _viewDataType);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl03));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // validator.ValidationFilePathにeventController.ValidationFilePathの値が設定されていること
            Assert.AreEqual(validationFilePath, accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validateメソッドが呼び出され、正しい引数がわたっていること
            Coordinator_IValidatorImpl03 validatorImpl = (Coordinator_IValidatorImpl03)accessor._validator;
            Assert.IsTrue(validatorImpl.IsCalledValidate);
            Assert.AreSame(eventController.ViewData, validatorImpl.ValidateDataSet);
            Assert.IsInstanceOfType(validatorImpl.ValidateDataSet, _viewDataType);

            // eventController.ReportCustomValidatingメソッドが呼び出され引数が_validator.Validate()の戻り値と等しいこと
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(1, eventController.ReportCustomValidatingValidationResult.Errors.Count);
            Assert.AreSame(validatorImpl.ValidateValidationMessageInfo01, eventController.ReportCustomValidatingValidationResult.Errors[0]);

            // eventController.ErrorHandler.HandleError()が呼ばれ引数が正しいこと
            Assert.IsTrue(errorHandler.IsCalledHandleError);
            Assert.AreEqual(Coordinator.VALIDATION_FAILED, errorHandler.HandleErrorResultString);
            Assert.AreEqual(1, errorHandler.HandleErrorMessages.Count);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Key, errorHandler.HandleErrorMessages[0].Key);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Message, errorHandler.HandleErrorMessages[0].Message);
            Assert.AreSame(eventController.ViewData, errorHandler.HandleErrorView);
            Assert.IsInstanceOfType(errorHandler.HandleErrorView, _viewDataType);

            // eventController.ErrorPathsにvalidationResult.Errorsの各ValidationMessageInfoのパスが追加されていること
            Assert.AreEqual(1, eventController.ErrorPaths.Count);
            Assert.AreEqual("errorPath01", eventController.ErrorPaths[0]);

            // eventController.ReportValidationFailedメソッドが呼び出され、正しい引数がわたってきていること
            Assert.IsTrue(eventController.IsCalledReportValidationFailed);
            Assert.AreEqual(1, eventController.ReportValidationFailedExecutionResult.Errors.Count);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Key, eventController.ReportValidationFailedExecutionResult.Errors[0].Key);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Message, eventController.ReportValidationFailedExecutionResult.Errors[0].Message);
            Assert.AreEqual(Coordinator.VALIDATION_FAILED, eventController.ReportValidationFailedExecutionResult.ResultString);
            Assert.IsNull(eventController.ReportValidationFailedExecutionResult.BLogicResultData);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessed);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // eventController.ReportExecuteCompleted() が呼び出され、正しい引数がわたっていること
            Assert.IsTrue(eventController.IsCalledReportExecuteCompleted);
            Assert.AreEqual(Coordinator.VALIDATION_FAILED, eventController.ReportExecuteCompletedExecutionResult.ResultString);
            Assert.IsNull(eventController.ReportExecuteCompletedExecutionResult.BLogicResultData);
            Assert.AreEqual(1, eventController.ReportExecuteCompletedExecutionResult.Errors.Count);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Key, eventController.ReportExecuteCompletedExecutionResult.Errors[0].Key);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Message, eventController.ReportExecuteCompletedExecutionResult.Errors[0].Message);
            Assert.IsNull(eventController.ReportExecuteCompletedError);
            Assert.IsFalse(eventController.ReportExecuteCompletedCancelled);

            // BLogicFactory.CreateBLogic() が呼び出されていないこと
            Assert.IsFalse(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);

            // 非同期処理が実行されていないこと
            Assert.IsFalse(target.IsBusy);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// ExecuteAsync (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 入力値検証機能による単項目チェックにおいてエラー数が 1 であり
        /// カスタムチェックイベントによる入力値検証においてエラー数が 1 の場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteAsyncTest14()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub02 eventController = new Coordinator_EventControllerStub02();
            string validationFilePath = "validationFilePath";
            eventController.ValidationFilePath = validationFilePath;

            ConfigurationManager.AppSettings["ValidatorTypeName"] = typeof(Coordinator_IValidatorImpl03).AssemblyQualifiedName;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            target.ExecuteAsync(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl03));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // validator.ValidationFilePathにeventController.ValidationFilePathの値が設定されていること
            Assert.AreEqual(validationFilePath, accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validateメソッドが呼び出され、正しい引数がわたっていること
            Coordinator_IValidatorImpl03 validatorImpl = (Coordinator_IValidatorImpl03)accessor._validator;
            Assert.IsTrue(validatorImpl.IsCalledValidate);
            Assert.AreSame(eventController.ViewData, validatorImpl.ValidateDataSet);
            Assert.IsInstanceOfType(validatorImpl.ValidateDataSet, _viewDataType);

            // eventController.ReportCustomValidatingメソッドが呼び出され引数が_validator.Validate()の戻り値と等しいこと
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(1, eventController.ReportCustomValidatingValidationResult.Errors.Count);
            Assert.AreSame(validatorImpl.ValidateValidationMessageInfo01, eventController.ReportCustomValidatingValidationResult.Errors[0]);

            // eventController.ErrorPathsにvalidationResult.Errorsの各ValidationMessageInfoのパスが追加されていること
            Assert.AreEqual(2, eventController.ErrorPaths.Count);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.ErrorPath, eventController.ErrorPaths[0]);
            Assert.AreEqual(((ValidationMessageInfo)eventController.ReportCustomValidatingMessageInfo01).ErrorPath, eventController.ErrorPaths[1]);

            // eventController.ReportValidationFailedメソッドが呼び出され、正しい引数がわたってきていること
            Assert.IsTrue(eventController.IsCalledReportValidationFailed);
            Assert.AreEqual(2, eventController.ReportValidationFailedExecutionResult.Errors.Count);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Key, eventController.ReportValidationFailedExecutionResult.Errors[0].Key);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Message, eventController.ReportValidationFailedExecutionResult.Errors[0].Message);
            Assert.AreEqual(eventController.ReportCustomValidatingMessageInfo01.Key, eventController.ReportValidationFailedExecutionResult.Errors[1].Key);
            Assert.AreEqual(eventController.ReportCustomValidatingMessageInfo01.Message, eventController.ReportValidationFailedExecutionResult.Errors[1].Message);
            Assert.AreEqual(Coordinator.VALIDATION_FAILED, eventController.ReportValidationFailedExecutionResult.ResultString);
            Assert.IsNull(eventController.ReportValidationFailedExecutionResult.BLogicResultData);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessed);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // eventController.ReportExecuteCompleted() が呼び出され、正しい引数がわたっていること
            Assert.IsTrue(eventController.IsCalledReportExecuteCompleted);
            Assert.AreEqual(Coordinator.VALIDATION_FAILED, eventController.ReportExecuteCompletedExecutionResult.ResultString);
            Assert.IsNull(eventController.ReportExecuteCompletedExecutionResult.BLogicResultData);
            Assert.AreEqual(2, eventController.ReportExecuteCompletedExecutionResult.Errors.Count);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Key, eventController.ReportExecuteCompletedExecutionResult.Errors[0].Key);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Message, eventController.ReportExecuteCompletedExecutionResult.Errors[0].Message);
            Assert.AreEqual(eventController.ReportCustomValidatingMessageInfo01.Key, eventController.ReportExecuteCompletedExecutionResult.Errors[1].Key);
            Assert.AreEqual(eventController.ReportCustomValidatingMessageInfo01.Message, eventController.ReportExecuteCompletedExecutionResult.Errors[1].Message);
            Assert.IsNull(eventController.ReportExecuteCompletedError);
            Assert.IsFalse(eventController.ReportExecuteCompletedCancelled);

            // BLogicFactory.CreateBLogic() が呼び出されていないこと
            Assert.IsFalse(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);

            // 非同期処理が実行されていないこと
            Assert.IsFalse(target.IsBusy);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// ExecuteAsync (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 入力値検証機能による単項目チェックにおいてエラー数が 1 であり
        /// カスタムチェックイベントによる入力値検証においてエラー数が 3 の場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteAsyncTest15()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub03 eventController = new Coordinator_EventControllerStub03();
            string validationFilePath = "validationFilePath";
            eventController.ValidationFilePath = validationFilePath;

            ConfigurationManager.AppSettings["ValidatorTypeName"] = typeof(Coordinator_IValidatorImpl03).AssemblyQualifiedName;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            target.ExecuteAsync(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl03));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // validator.ValidationFilePathにeventController.ValidationFilePathの値が設定されていること
            Assert.AreEqual(validationFilePath, accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validateメソッドが呼び出され、正しい引数がわたっていること
            Coordinator_IValidatorImpl03 validatorImpl = (Coordinator_IValidatorImpl03)accessor._validator;
            Assert.IsTrue(validatorImpl.IsCalledValidate);
            Assert.AreSame(eventController.ViewData, validatorImpl.ValidateDataSet);
            Assert.IsInstanceOfType(validatorImpl.ValidateDataSet, _viewDataType);

            // eventController.ReportCustomValidatingメソッドが呼び出され引数が_validator.Validate()の戻り値と等しいこと
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(1, eventController.ReportCustomValidatingValidationResult.Errors.Count);
            Assert.AreSame(validatorImpl.ValidateValidationMessageInfo01, eventController.ReportCustomValidatingValidationResult.Errors[0]);

            // eventController.ErrorPathsにvalidationResult.Errorsの各ValidationMessageInfoのパスが追加されていること
            Assert.AreEqual(4, eventController.ErrorPaths.Count);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.ErrorPath, eventController.ErrorPaths[0]);
            Assert.AreEqual(((ValidationMessageInfo)eventController.ReportCustomValidatingMessageInfo01).ErrorPath, eventController.ErrorPaths[1]);
            Assert.AreEqual(((ValidationMessageInfo)eventController.ReportCustomValidatingMessageInfo02).ErrorPath, eventController.ErrorPaths[2]);
            Assert.AreEqual(((ValidationMessageInfo)eventController.ReportCustomValidatingMessageInfo03).ErrorPath, eventController.ErrorPaths[3]);

            // eventController.ReportValidationFailedメソッドが呼び出され、正しい引数がわたってきていること
            Assert.IsTrue(eventController.IsCalledReportValidationFailed);
            Assert.AreEqual(4, eventController.ReportValidationFailedExecutionResult.Errors.Count);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Key, eventController.ReportValidationFailedExecutionResult.Errors[0].Key);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Message, eventController.ReportValidationFailedExecutionResult.Errors[0].Message);
            Assert.AreEqual(eventController.ReportCustomValidatingMessageInfo01.Key, eventController.ReportValidationFailedExecutionResult.Errors[1].Key);
            Assert.AreEqual(eventController.ReportCustomValidatingMessageInfo01.Message, eventController.ReportValidationFailedExecutionResult.Errors[1].Message);
            Assert.AreEqual(eventController.ReportCustomValidatingMessageInfo02.Key, eventController.ReportValidationFailedExecutionResult.Errors[2].Key);
            Assert.AreEqual(eventController.ReportCustomValidatingMessageInfo02.Message, eventController.ReportValidationFailedExecutionResult.Errors[2].Message);
            Assert.AreEqual(eventController.ReportCustomValidatingMessageInfo03.Key, eventController.ReportValidationFailedExecutionResult.Errors[3].Key);
            Assert.AreEqual(eventController.ReportCustomValidatingMessageInfo03.Message, eventController.ReportValidationFailedExecutionResult.Errors[3].Message);
            Assert.AreEqual(Coordinator.VALIDATION_FAILED, eventController.ReportValidationFailedExecutionResult.ResultString);
            Assert.IsNull(eventController.ReportValidationFailedExecutionResult.BLogicResultData);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessed);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // eventController.ReportExecuteCompleted() が呼び出され、正しい引数がわたっていること
            Assert.IsTrue(eventController.IsCalledReportExecuteCompleted);
            Assert.AreEqual(Coordinator.VALIDATION_FAILED, eventController.ReportExecuteCompletedExecutionResult.ResultString);
            Assert.IsNull(eventController.ReportExecuteCompletedExecutionResult.BLogicResultData);
            Assert.AreEqual(4, eventController.ReportExecuteCompletedExecutionResult.Errors.Count);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Key, eventController.ReportExecuteCompletedExecutionResult.Errors[0].Key);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Message, eventController.ReportExecuteCompletedExecutionResult.Errors[0].Message);
            Assert.AreEqual(eventController.ReportCustomValidatingMessageInfo01.Key, eventController.ReportExecuteCompletedExecutionResult.Errors[1].Key);
            Assert.AreEqual(eventController.ReportCustomValidatingMessageInfo01.Message, eventController.ReportExecuteCompletedExecutionResult.Errors[1].Message);
            Assert.AreEqual(eventController.ReportCustomValidatingMessageInfo02.Key, eventController.ReportExecuteCompletedExecutionResult.Errors[2].Key);
            Assert.AreEqual(eventController.ReportCustomValidatingMessageInfo02.Message, eventController.ReportExecuteCompletedExecutionResult.Errors[2].Message);
            Assert.AreEqual(eventController.ReportCustomValidatingMessageInfo03.Key, eventController.ReportExecuteCompletedExecutionResult.Errors[3].Key);
            Assert.AreEqual(eventController.ReportCustomValidatingMessageInfo03.Message, eventController.ReportExecuteCompletedExecutionResult.Errors[3].Message);
            Assert.IsNull(eventController.ReportExecuteCompletedError);
            Assert.IsFalse(eventController.ReportExecuteCompletedCancelled);

            // BLogicFactory.CreateBLogic() が呼び出されていないこと
            Assert.IsFalse(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);

            // 非同期処理が実行されていないこと
            Assert.IsFalse(target.IsBusy);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// ExecuteAsync (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.RuleSet が空文字であり、
        /// 入力値検証機能による単項目チェックにおいてエラー数が 1 の場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteAsyncTest16()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            string validationFilePath = "validationFilePath";
            eventController.ValidationFilePath = validationFilePath;
            eventController.RuleSet = string.Empty;

            ConfigurationManager.AppSettings["ValidatorTypeName"] = typeof(Coordinator_IValidatorImpl03).AssemblyQualifiedName;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            target.ExecuteAsync(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl03));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // validator.ValidationFilePathにeventController.ValidationFilePathの値が設定されていること
            Assert.AreEqual(validationFilePath, accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validateメソッドが呼び出され、正しい引数がわたっていること
            Coordinator_IValidatorImpl03 validatorImpl = (Coordinator_IValidatorImpl03)accessor._validator;
            Assert.IsTrue(validatorImpl.IsCalledValidate);
            Assert.AreSame(eventController.ViewData, validatorImpl.ValidateDataSet);
            Assert.IsInstanceOfType(validatorImpl.ValidateDataSet, _viewDataType);

            // eventController.ReportCustomValidatingメソッドが呼び出され引数が_validator.Validate()の戻り値と等しいこと
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(1, eventController.ReportCustomValidatingValidationResult.Errors.Count);
            Assert.AreSame(validatorImpl.ValidateValidationMessageInfo01, eventController.ReportCustomValidatingValidationResult.Errors[0]);

            // eventController.ErrorPathsにvalidationResult.Errorsの各ValidationMessageInfoのパスが追加されていること
            Assert.AreEqual(1, eventController.ErrorPaths.Count);
            Assert.AreEqual("errorPath01", eventController.ErrorPaths[0]);

            // eventController.ReportValidationFailedメソッドが呼び出され、正しい引数がわたってきていること
            Assert.IsTrue(eventController.IsCalledReportValidationFailed);
            Assert.AreEqual(1, eventController.ReportValidationFailedExecutionResult.Errors.Count);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Key, eventController.ReportValidationFailedExecutionResult.Errors[0].Key);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Message, eventController.ReportValidationFailedExecutionResult.Errors[0].Message);
            Assert.AreEqual(Coordinator.VALIDATION_FAILED, eventController.ReportValidationFailedExecutionResult.ResultString);
            Assert.IsNull(eventController.ReportValidationFailedExecutionResult.BLogicResultData);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessed);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // eventController.ReportExecuteCompleted() が呼び出され、正しい引数がわたっていること
            Assert.IsTrue(eventController.IsCalledReportExecuteCompleted);
            Assert.AreEqual(Coordinator.VALIDATION_FAILED, eventController.ReportExecuteCompletedExecutionResult.ResultString);
            Assert.IsNull(eventController.ReportExecuteCompletedExecutionResult.BLogicResultData);
            Assert.AreEqual(1, eventController.ReportExecuteCompletedExecutionResult.Errors.Count);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Key, eventController.ReportExecuteCompletedExecutionResult.Errors[0].Key);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Message, eventController.ReportExecuteCompletedExecutionResult.Errors[0].Message);
            Assert.IsNull(eventController.ReportExecuteCompletedError);
            Assert.IsFalse(eventController.ReportExecuteCompletedCancelled);

            // BLogicFactory.CreateBLogic() が呼び出されていないこと
            Assert.IsFalse(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);

            // 非同期処理が実行されていないこと
            Assert.IsFalse(target.IsBusy);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// ExecuteAsync (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.RuleSet が null・空文字でなく、
        /// 入力値検証機能による単項目チェックにおいてエラー数が 1 の場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteAsyncTest17()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            string validationFilePath = "validationFilePath";
            eventController.ValidationFilePath = validationFilePath;
            string ruleSet = "RuleSet";
            eventController.RuleSet = ruleSet;

            ConfigurationManager.AppSettings["ValidatorTypeName"] = typeof(Coordinator_IValidatorImpl03).AssemblyQualifiedName;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            target.ExecuteAsync(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl03));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // validator.ValidationFilePathにeventController.ValidationFilePathの値が設定されていること
            Assert.AreEqual(validationFilePath, accessor._validator.ValidationFilePath);

            // _validator.RuleSetにeventController.RuleSetの値が設定されていること
            Assert.AreEqual(ruleSet, accessor._validator.RuleSet);

            // _validator.Validateメソッドが呼び出され、正しい引数がわたっていること
            Coordinator_IValidatorImpl03 validatorImpl = (Coordinator_IValidatorImpl03)accessor._validator;
            Assert.IsTrue(validatorImpl.IsCalledValidate);
            Assert.AreSame(eventController.ViewData, validatorImpl.ValidateDataSet);
            Assert.IsInstanceOfType(validatorImpl.ValidateDataSet, _viewDataType);

            // eventController.ReportCustomValidatingメソッドが呼び出され引数が_validator.Validate()の戻り値と等しいこと
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(1, eventController.ReportCustomValidatingValidationResult.Errors.Count);
            Assert.AreSame(validatorImpl.ValidateValidationMessageInfo01, eventController.ReportCustomValidatingValidationResult.Errors[0]);

            // eventController.ErrorPathsにvalidationResult.Errorsの各ValidationMessageInfoのパスが追加されていること
            Assert.AreEqual(1, eventController.ErrorPaths.Count);
            Assert.AreEqual("errorPath01", eventController.ErrorPaths[0]);

            // eventController.ReportValidationFailedメソッドが呼び出され、正しい引数がわたってきていること
            Assert.IsTrue(eventController.IsCalledReportValidationFailed);
            Assert.AreEqual(1, eventController.ReportValidationFailedExecutionResult.Errors.Count);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Key, eventController.ReportValidationFailedExecutionResult.Errors[0].Key);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Message, eventController.ReportValidationFailedExecutionResult.Errors[0].Message);
            Assert.AreEqual(Coordinator.VALIDATION_FAILED, eventController.ReportValidationFailedExecutionResult.ResultString);
            Assert.IsNull(eventController.ReportValidationFailedExecutionResult.BLogicResultData);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessed);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // eventController.ReportExecuteCompleted() が呼び出され、正しい引数がわたっていること
            Assert.IsTrue(eventController.IsCalledReportExecuteCompleted);
            Assert.AreEqual(Coordinator.VALIDATION_FAILED, eventController.ReportExecuteCompletedExecutionResult.ResultString);
            Assert.IsNull(eventController.ReportExecuteCompletedExecutionResult.BLogicResultData);
            Assert.AreEqual(1, eventController.ReportExecuteCompletedExecutionResult.Errors.Count);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Key, eventController.ReportExecuteCompletedExecutionResult.Errors[0].Key);
            Assert.AreEqual(validatorImpl.ValidateValidationMessageInfo01.Message, eventController.ReportExecuteCompletedExecutionResult.Errors[0].Message);
            Assert.IsNull(eventController.ReportExecuteCompletedError);
            Assert.IsFalse(eventController.ReportExecuteCompletedCancelled);

            // BLogicFactory.CreateBLogic() が呼び出されていないこと
            Assert.IsFalse(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);

            // 非同期処理が実行されていないこと
            Assert.IsFalse(target.IsBusy);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// ExecuteAsync (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 入力値検証機能による単項目チェックにおいてエラー数が 1 、
        /// カスタムチェックイベントによる入力値検証においてエラー数が 1 であり、
        /// かつ、エラーの型が ValidationMessageInfo でない場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteAsyncTest18()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub04 eventController = new Coordinator_EventControllerStub04();
            string validationFilePath = "validationFilePath";
            eventController.ValidationFilePath = validationFilePath;

            ConfigurationManager.AppSettings["ValidatorTypeName"] = typeof(Coordinator_IValidatorImpl05).AssemblyQualifiedName;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            target.ExecuteAsync(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl05));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // validator.ValidationFilePathにeventController.ValidationFilePathの値が設定されていること
            Assert.AreEqual(validationFilePath, accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validateメソッドが呼び出され、正しい引数がわたっていること
            Coordinator_IValidatorImpl05 validatorImpl = (Coordinator_IValidatorImpl05)accessor._validator;
            Assert.IsTrue(validatorImpl.IsCalledValidate);
            Assert.AreSame(eventController.ViewData, validatorImpl.ValidateDataSet);
            Assert.IsInstanceOfType(validatorImpl.ValidateDataSet, _viewDataType);

            // eventController.ReportCustomValidatingメソッドが呼び出され引数が_validator.Validate()の戻り値と等しいこと
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(1, eventController.ReportCustomValidatingValidationResult.Errors.Count);
            Assert.AreSame(validatorImpl.ValidateMessageInfo01, eventController.ReportCustomValidatingValidationResult.Errors[0]);

            // eventController.ReportValidationFailedメソッドが呼び出され、正しい引数がわたってきていること
            Assert.IsTrue(eventController.IsCalledReportValidationFailed);
            Assert.AreEqual(2, eventController.ReportValidationFailedExecutionResult.Errors.Count);
            Assert.AreEqual(validatorImpl.ValidateMessageInfo01.Key, eventController.ReportValidationFailedExecutionResult.Errors[0].Key);
            Assert.AreEqual(validatorImpl.ValidateMessageInfo01.Message, eventController.ReportValidationFailedExecutionResult.Errors[0].Message);
            Assert.AreEqual(eventController.ReportCustomValidatingMessageInfo01.Key, eventController.ReportValidationFailedExecutionResult.Errors[1].Key);
            Assert.AreEqual(eventController.ReportCustomValidatingMessageInfo01.Message, eventController.ReportValidationFailedExecutionResult.Errors[1].Message);
            Assert.AreEqual(Coordinator.VALIDATION_FAILED, eventController.ReportValidationFailedExecutionResult.ResultString);
            Assert.IsNull(eventController.ReportValidationFailedExecutionResult.BLogicResultData);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessed);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // eventController.ReportExecuteCompleted() が呼び出され、正しい引数がわたっていること
            Assert.IsTrue(eventController.IsCalledReportExecuteCompleted);
            Assert.AreEqual(Coordinator.VALIDATION_FAILED, eventController.ReportExecuteCompletedExecutionResult.ResultString);
            Assert.IsNull(eventController.ReportExecuteCompletedExecutionResult.BLogicResultData);
            Assert.AreEqual(2, eventController.ReportExecuteCompletedExecutionResult.Errors.Count);
            Assert.AreEqual(validatorImpl.ValidateMessageInfo01.Key, eventController.ReportExecuteCompletedExecutionResult.Errors[0].Key);
            Assert.AreEqual(validatorImpl.ValidateMessageInfo01.Message, eventController.ReportExecuteCompletedExecutionResult.Errors[0].Message);
            Assert.AreEqual(eventController.ReportCustomValidatingMessageInfo01.Key, eventController.ReportExecuteCompletedExecutionResult.Errors[1].Key);
            Assert.AreEqual(eventController.ReportCustomValidatingMessageInfo01.Message, eventController.ReportExecuteCompletedExecutionResult.Errors[1].Message);
            Assert.IsNull(eventController.ReportExecuteCompletedError);
            Assert.IsFalse(eventController.ReportExecuteCompletedCancelled);

            // BLogicFactory.CreateBLogic() が呼び出されていないこと
            Assert.IsFalse(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 非同期処理が実行されていないこと
            Assert.IsFalse(target.IsBusy);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// ExecuteAsync (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.BLogicParamTypeName が null であり、
        /// eventController.ViewData が null の場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteAsyncTest19()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            eventController.BLogicParamTypeName = null;
            eventController.ViewData = null;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            target.ExecuteAsync(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataの型がDataSet
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, typeof(DataSet));

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // eventController.ReportExecuteCompleted() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportExecuteCompleted);

            // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
            Assert.AreEqual(_blogicName, Coordinator_BLogicFactoryStub01._blogicName);
            Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
            Assert.AreEqual(_blogicResultTypeName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 非同期処理が実行中であること
            Assert.IsTrue(target.IsBusy);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// ExecuteAsync (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.BLogicParamTypeName が null であり、
        /// eventController.ViewData が null でない場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteAsyncTest20()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            eventController.BLogicParamTypeName = null;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            target.ExecuteAsync(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataの型がViewDataで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _viewDataType);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // eventController.ReportExecuteCompleted() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportExecuteCompleted);

            // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
            Assert.AreEqual(_blogicName, Coordinator_BLogicFactoryStub01._blogicName);
            Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
            Assert.AreEqual(_blogicResultTypeName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 非同期処理が実行中であること
            Assert.IsTrue(target.IsBusy);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// ExecuteAsync (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.BLogicParamTypeName が 空文字 であり、
        /// eventController.ViewData が null の場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteAsyncTest21()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            eventController.BLogicParamTypeName = string.Empty;
            eventController.ViewData = null;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            target.ExecuteAsync(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataの型がDataSet
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, typeof(DataSet));

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // eventController.ReportExecuteCompleted() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportExecuteCompleted);

            // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
            Assert.AreEqual(_blogicName, Coordinator_BLogicFactoryStub01._blogicName);
            Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
            Assert.AreEqual(_blogicResultTypeName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 非同期処理が実行中であること
            Assert.IsTrue(target.IsBusy);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// ExecuteAsync (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// ビジネスロジック入力 DataSet のインスタンスを生成できない場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteAsyncTest22()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            string blogicParamenTypeName = typeof(Coordinator_DataSetStub05).AssemblyQualifiedName;
            eventController.BLogicParamTypeName = blogicParamenTypeName;

            try
            {
                // 実施
                target.ExecuteAsync(eventController);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // 成功
                Assert.AreEqual(MessageManager.GetMessage("E_INVALID_CAST_TYPE_FROM_GENERICPARAM", blogicParamenTypeName, typeof(DataSet).AssemblyQualifiedName), e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("予期せぬエラーです。" + e.ToString());
            }

            // 非同期処理が実行されていないこと
            Assert.IsFalse(target.IsBusy);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// ExecuteAsync (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.Items の要素数が 1 で EventController.REQUEST_NAME がキーに存在している場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteAsyncTest23()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            string item = "Item01";
            eventController.Items.Add(EventController.REQUEST_NAME, item);

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            target.ExecuteAsync(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // eventController.ReportExecuteCompleted() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportExecuteCompleted);

            // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
            Assert.AreEqual(_blogicName, Coordinator_BLogicFactoryStub01._blogicName);
            Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
            Assert.AreEqual(_blogicResultTypeName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 非同期処理が実行中であること
            Assert.IsTrue(target.IsBusy);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// ExecuteAsync (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.Items の要素数が 1 で EventController.REQUEST_NAME がキーに存在していない場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteAsyncTest24()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            string key = "key01";
            string item = "Item01";
            eventController.Items.Add(key, item);

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            target.ExecuteAsync(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // eventController.ReportExecuteCompleted() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportExecuteCompleted);

            // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
            Assert.AreEqual(_blogicName, Coordinator_BLogicFactoryStub01._blogicName);
            Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
            Assert.AreEqual(_blogicResultTypeName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 非同期処理が実行中であること
            Assert.IsTrue(target.IsBusy);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// ExecuteAsync (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.Items の要素数が 3 で EventController.REQUEST_NAME がキーに存在している場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteAsyncTest25()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            string key01 = "key01";
            string key02 = "key02";
            string item01 = "Item01";
            string item02 = "Item02";
            string item03 = "Item03";
            eventController.Items.Add(key01, item01);
            eventController.Items.Add(key02, item02);
            eventController.Items.Add(EventController.REQUEST_NAME, item03);

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            target.ExecuteAsync(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // eventController.ReportExecuteCompleted() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportExecuteCompleted);

            // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
            Assert.AreEqual(_blogicName, Coordinator_BLogicFactoryStub01._blogicName);
            Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
            Assert.AreEqual(_blogicResultTypeName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 非同期処理が実行中であること
            Assert.IsTrue(target.IsBusy);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// ExecuteAsync (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.Items の要素数が 3 で EventController.REQUEST_NAME がキーに存在していない場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteAsyncTest26()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            string key01 = "key01";
            string key02 = "key02";
            string key03 = "key03";
            string item01 = "Item01";
            string item02 = "Item02";
            string item03 = "Item03";
            eventController.Items.Add(key01, item01);
            eventController.Items.Add(key02, item02);
            eventController.Items.Add(key03, item03);

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            target.ExecuteAsync(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // eventController.ReportExecuteCompleted() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportExecuteCompleted);

            // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
            Assert.AreEqual(_blogicName, Coordinator_BLogicFactoryStub01._blogicName);
            Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
            Assert.AreEqual(_blogicResultTypeName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 非同期処理が実行中であること
            Assert.IsTrue(target.IsBusy);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// ExecuteAsync (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.ConvertId が空文字列の場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteAsyncTest27()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            eventController.ConvertId = string.Empty;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            target.ExecuteAsync(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // eventController.ReportExecuteCompleted() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportExecuteCompleted);

            // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
            Assert.AreEqual(_blogicName, Coordinator_BLogicFactoryStub01._blogicName);
            Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
            Assert.AreEqual(_blogicResultTypeName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 非同期処理が実行中であること
            Assert.IsTrue(target.IsBusy);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// ExecuteAsync (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.ConvertId が null・空文字列でなく
        /// eventController.ViewData が null の場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteAsyncTest28()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            string convertId = "convertId";
            eventController.ConvertId = convertId;
            eventController.ViewData = null;

            try
            {
                // 実施
                target.ExecuteAsync(eventController);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // 成功
                Assert.AreEqual(MessageManager.GetMessage("E_BLOGIC_NULL_DATASET", "ConvertId", "ViewData"), e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("予期せぬエラーです。" + e.ToString());
            }

            // 非同期処理が実行されていないこと
            Assert.IsFalse(target.IsBusy);

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(
                typeof(Coordinator), typeof(TerasolunaException), "E_BLOGIC_NULL_DATASET", "ConvertId", "ViewData"));
        }

        /// <summary>
        /// ExecuteAsync (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.ConvertId が null・空文字列でなく
        /// eventController.ViewData が null でない場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteAsyncTest30()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            string convertId = "convertId";
            eventController.ConvertId = convertId;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            target.ExecuteAsync(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されており、正しい引数がわたっていること
            Coordinator_IConverterImpl01 converter = (Coordinator_IConverterImpl01)accessor._converter;
            Assert.IsTrue(converter.IsCalledConvert);
            Assert.AreEqual(convertId, converter.ConvertId);
            Assert.AreSame(eventController.ViewData, converter.ConvertSrcData);
            Assert.IsInstanceOfType(converter.ConvertDestData, _blogicParamType);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // eventController.ReportExecuteCompleted() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportExecuteCompleted);

            // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
            Assert.AreEqual(_blogicName, Coordinator_BLogicFactoryStub01._blogicName);
            Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
            Assert.AreEqual(_blogicResultTypeName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 非同期処理が実行中であること
            Assert.IsTrue(target.IsBusy);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// ExecuteAsync (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 前処理完了イベントでのエラー数が 1 でエラーの型が ValidationMessageInfo の場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteAsyncTest31()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub05 eventController = new Coordinator_EventControllerStub05();

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            target.ExecuteAsync(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ErrorPathsにexecutionResult.Errorsの各ValidationMessageInfoのパスが追加されていること
            Assert.AreEqual(1, eventController.ErrorPaths.Count);
            Assert.AreEqual(((ValidationMessageInfo)eventController.ReportPreprocessedMessageInfo01).ErrorPath, eventController.ErrorPaths[0]);

            // eventController.ReportPreprocessedFailedメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(eventController.IsCalledReportPreprocessedFailed);
            Assert.AreEqual(1, eventController.ReportPreprocessedFailedExecutionResult.Errors.Count);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo01.Key, eventController.ReportPreprocessedFailedExecutionResult.Errors[0].Key);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo01.Message, eventController.ReportPreprocessedFailedExecutionResult.Errors[0].Message);
            Assert.AreEqual(Coordinator.PREPROCESSED_FAILED, eventController.ReportPreprocessedFailedExecutionResult.ResultString);
            Assert.IsNull(eventController.ReportPreprocessedFailedExecutionResult.BLogicResultData);

            // eventController.ReportExecuteCompleted() が呼び出され、正しい引数がわたっていること
            Assert.IsTrue(eventController.IsCalledReportExecuteCompleted);
            Assert.AreEqual(Coordinator.PREPROCESSED_FAILED, eventController.ReportExecuteCompletedExecutionResult.ResultString);
            Assert.IsNull(eventController.ReportExecuteCompletedExecutionResult.BLogicResultData);
            Assert.AreEqual(1, eventController.ReportExecuteCompletedExecutionResult.Errors.Count);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo01.Key, eventController.ReportExecuteCompletedExecutionResult.Errors[0].Key);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo01.Message, eventController.ReportExecuteCompletedExecutionResult.Errors[0].Message);
            Assert.IsNull(eventController.ReportExecuteCompletedError);
            Assert.IsFalse(eventController.ReportExecuteCompletedCancelled);

            // BLogicFactory.CreateBLogicメソッドが呼び出されないこと
            Assert.IsFalse(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);

            // 非同期処理が実行されていないこと
            Assert.IsFalse(target.IsBusy);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// ExecuteAsync (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.ErrorHandler が null でなく、
        /// 前処理完了イベントでのエラー数が 1 でエラーの型が ValidationMessageInfo でない場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteAsyncTest32()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub06 eventController = new Coordinator_EventControllerStub06();
            Coordinator_IErrorHandlerImpl01 errorHandler = new Coordinator_IErrorHandlerImpl01();
            eventController.ErrorHandler = errorHandler;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            target.ExecuteAsync(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ErrorHandler.HandleError()が呼ばれ引数が正しいこと
            Assert.IsTrue(errorHandler.IsCalledHandleError);
            Assert.AreEqual(Coordinator.PREPROCESSED_FAILED, errorHandler.HandleErrorResultString);
            Assert.AreEqual(1, errorHandler.HandleErrorMessages.Count);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo01.Key, errorHandler.HandleErrorMessages[0].Key);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo01.Message, errorHandler.HandleErrorMessages[0].Message);
            Assert.IsInstanceOfType(errorHandler.HandleErrorView, _viewDataType);

            // eventController.ReportPreprocessedFailedメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(eventController.IsCalledReportPreprocessedFailed);
            Assert.AreEqual(1, eventController.ReportPreprocessedFailedExecutionResult.Errors.Count);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo01.Key, eventController.ReportPreprocessedFailedExecutionResult.Errors[0].Key);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo01.Message, eventController.ReportPreprocessedFailedExecutionResult.Errors[0].Message);
            Assert.AreEqual(Coordinator.PREPROCESSED_FAILED, eventController.ReportPreprocessedFailedExecutionResult.ResultString);
            Assert.IsNull(eventController.ReportPreprocessedFailedExecutionResult.BLogicResultData);

            // eventController.ReportExecuteCompleted() が呼び出され、正しい引数がわたっていること
            Assert.IsTrue(eventController.IsCalledReportExecuteCompleted);
            Assert.AreEqual(Coordinator.PREPROCESSED_FAILED, eventController.ReportExecuteCompletedExecutionResult.ResultString);
            Assert.IsNull(eventController.ReportExecuteCompletedExecutionResult.BLogicResultData);
            Assert.AreEqual(1, eventController.ReportExecuteCompletedExecutionResult.Errors.Count);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo01.Key, eventController.ReportExecuteCompletedExecutionResult.Errors[0].Key);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo01.Message, eventController.ReportExecuteCompletedExecutionResult.Errors[0].Message);
            Assert.IsNull(eventController.ReportExecuteCompletedError);
            Assert.IsFalse(eventController.ReportExecuteCompletedCancelled);

            // BLogicFactory.CreateBLogicメソッドが呼び出されないこと
            Assert.IsFalse(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 非同期処理が実行されていないこと
            Assert.IsFalse(target.IsBusy);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// ExecuteAsync (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.ErrorHandler が null でなく、
        /// 前処理完了イベントでのエラー数が 3 でエラーの型が ValidationMessageInfo でない場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteAsyncTest33()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub07 eventController = new Coordinator_EventControllerStub07();
            Coordinator_IErrorHandlerImpl01 errorHandler = new Coordinator_IErrorHandlerImpl01();
            eventController.ErrorHandler = errorHandler;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            target.ExecuteAsync(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ErrorHandler.HandleError()が呼ばれ引数が正しいこと
            Assert.IsTrue(errorHandler.IsCalledHandleError);
            Assert.AreEqual(Coordinator.PREPROCESSED_FAILED, errorHandler.HandleErrorResultString);
            Assert.AreEqual(3, errorHandler.HandleErrorMessages.Count);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo01.Key, errorHandler.HandleErrorMessages[0].Key);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo02.Key, errorHandler.HandleErrorMessages[1].Key);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo03.Key, errorHandler.HandleErrorMessages[2].Key);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo01.Message, errorHandler.HandleErrorMessages[0].Message);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo02.Message, errorHandler.HandleErrorMessages[1].Message);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo03.Message, errorHandler.HandleErrorMessages[2].Message);
            Assert.IsInstanceOfType(errorHandler.HandleErrorView, _viewDataType);

            // eventController.ReportPreprocessedFailedメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(eventController.IsCalledReportPreprocessedFailed);
            Assert.AreEqual(3, eventController.ReportPreprocessedFailedExecutionResult.Errors.Count);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo01.Key, eventController.ReportPreprocessedFailedExecutionResult.Errors[0].Key);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo02.Key, eventController.ReportPreprocessedFailedExecutionResult.Errors[1].Key);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo03.Key, eventController.ReportPreprocessedFailedExecutionResult.Errors[2].Key);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo01.Message, eventController.ReportPreprocessedFailedExecutionResult.Errors[0].Message);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo02.Message, eventController.ReportPreprocessedFailedExecutionResult.Errors[1].Message);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo03.Message, eventController.ReportPreprocessedFailedExecutionResult.Errors[2].Message);
            Assert.AreEqual(Coordinator.PREPROCESSED_FAILED, eventController.ReportPreprocessedFailedExecutionResult.ResultString);
            Assert.IsNull(eventController.ReportPreprocessedFailedExecutionResult.BLogicResultData);

            // eventController.ReportExecuteCompleted() が呼び出され、正しい引数がわたっていること
            Assert.IsTrue(eventController.IsCalledReportExecuteCompleted);
            Assert.AreEqual(Coordinator.PREPROCESSED_FAILED, eventController.ReportExecuteCompletedExecutionResult.ResultString);
            Assert.IsNull(eventController.ReportExecuteCompletedExecutionResult.BLogicResultData);
            Assert.AreEqual(3, eventController.ReportExecuteCompletedExecutionResult.Errors.Count);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo01.Key, eventController.ReportExecuteCompletedExecutionResult.Errors[0].Key);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo02.Key, eventController.ReportExecuteCompletedExecutionResult.Errors[1].Key);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo03.Key, eventController.ReportExecuteCompletedExecutionResult.Errors[2].Key);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo01.Message, eventController.ReportExecuteCompletedExecutionResult.Errors[0].Message);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo02.Message, eventController.ReportExecuteCompletedExecutionResult.Errors[1].Message);
            Assert.AreEqual(eventController.ReportPreprocessedMessageInfo03.Message, eventController.ReportExecuteCompletedExecutionResult.Errors[2].Message);
            Assert.IsNull(eventController.ReportExecuteCompletedError);
            Assert.IsFalse(eventController.ReportExecuteCompletedCancelled);

            // BLogicFactory.CreateBLogicメソッドが呼び出されないこと
            Assert.IsFalse(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 非同期処理が実行されていないこと
            Assert.IsFalse(target.IsBusy);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// ExecuteAsync (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.BLogicResultTypeName が null であり、
        /// eventController.ViewData が null の場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteAsyncTest34()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            eventController.BLogicResultTypeName = null;
            eventController.ViewData = null;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            target.ExecuteAsync(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // eventController.ReportExecuteCompleted() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportExecuteCompleted);

            // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
            Assert.AreEqual(_blogicName, Coordinator_BLogicFactoryStub01._blogicName);
            Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
            Assert.AreEqual(typeof(DataSet).AssemblyQualifiedName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 非同期処理が実行中であること
            Assert.IsTrue(target.IsBusy);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// ExecuteAsync (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.BLogicResultTypeName が null であり、
        /// eventController.ViewData が null でない場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteAsyncTest35()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            eventController.BLogicResultTypeName = null;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            target.ExecuteAsync(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // eventController.ReportExecuteCompleted() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportExecuteCompleted);

            // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
            Assert.AreEqual(_blogicName, Coordinator_BLogicFactoryStub01._blogicName);
            Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
            Assert.AreEqual(_viewDataType.AssemblyQualifiedName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 非同期処理が実行中であること
            Assert.IsTrue(target.IsBusy);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// ExecuteAsync (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.BLogicResultTypeName が空文字列であり、
        /// eventController.ViewData が null の場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteAsyncTest36()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            eventController.BLogicResultTypeName = string.Empty;
            eventController.ViewData = null;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            target.ExecuteAsync(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // eventController.ReportExecuteCompleted() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportExecuteCompleted);

            // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
            Assert.AreEqual(_blogicName, Coordinator_BLogicFactoryStub01._blogicName);
            Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
            Assert.AreEqual(typeof(DataSet).AssemblyQualifiedName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 非同期処理が実行中であること
            Assert.IsTrue(target.IsBusy);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// ExecuteAsync (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.BLogicResultTypeName が空文字列であり、
        /// eventController.ViewData が null でない場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteAsyncTest37()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            eventController.BLogicResultTypeName = string.Empty;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            target.ExecuteAsync(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // eventController.ReportExecuteCompleted() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportExecuteCompleted);

            // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
            Assert.AreEqual(_blogicName, Coordinator_BLogicFactoryStub01._blogicName);
            Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
            Assert.AreEqual(_viewDataType.AssemblyQualifiedName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 非同期処理が実行中であること
            Assert.IsTrue(target.IsBusy);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// ExecuteAsync (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.BLogicName が null の場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteAsyncTest38()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            eventController.BLogicName = null;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            target.ExecuteAsync(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // eventController.ReportExecuteCompleted() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportExecuteCompleted);

            // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
            Assert.IsNull(Coordinator_BLogicFactoryStub01._blogicName);
            Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
            Assert.AreEqual(_blogicResultTypeName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 非同期処理が実行中であること
            Assert.IsTrue(target.IsBusy);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// ExecuteAsync (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.BLogicName が空文字列の場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteAsyncTest39()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            eventController.BLogicName = string.Empty;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            target.ExecuteAsync(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // eventController.ReportExecuteCompleted() が呼び出されていないこと

            Assert.IsFalse(eventController.IsCalledReportExecuteCompleted);
            // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
            Assert.AreEqual(string.Empty, Coordinator_BLogicFactoryStub01._blogicName);
            Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
            Assert.AreEqual(_blogicResultTypeName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 非同期処理が実行中であること
            Assert.IsTrue(target.IsBusy);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// ExecuteAsync (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// ビジネスロジッククラスが IProgressChengedEventInvoker を実装している場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteAsyncTest40()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl02 blogic = new Coordinator_IBLogicImpl02();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            target.ExecuteAsync(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // eventController.ReportExecuteCompleted() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportExecuteCompleted);

            // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
            Assert.AreEqual(_blogicName, Coordinator_BLogicFactoryStub01._blogicName);
            Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
            Assert.AreEqual(_blogicResultTypeName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

            // BLogicFactory.CreateBLogic()で生成したIBLogic実装クラスのインスタンスのProgressChangedイベントに
            // DoExecuteAsyncProgressChanged()が登録されていること
            Assert.IsTrue(blogic.IsNotNullProgressChanged);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 非同期処理が実行中であること
            Assert.IsTrue(target.IsBusy);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// ExecuteAsync (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.RequestName が null の場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteAsyncTest42()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            eventController.RequestName = null;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            target.ExecuteAsync(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ReportPreprocessedFailed() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportPreprocessedFailed);

            // eventController.ReportExecuteCompleted() が呼び出されていないこと
            Assert.IsFalse(eventController.IsCalledReportExecuteCompleted);

            // BLogicFactory.CreateBLogicメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);
            Assert.AreEqual(_blogicName, Coordinator_BLogicFactoryStub01._blogicName);
            Assert.AreEqual(1, Coordinator_BLogicFactoryStub01._genericTypeNames.Length);
            Assert.AreEqual(_blogicResultTypeName, Coordinator_BLogicFactoryStub01._genericTypeNames[0]);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // 非同期処理が実行中であること
            Assert.IsTrue(target.IsBusy);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// ExecuteAsync (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// フィールドの _isBusy が true の場合、
        /// InvalidOperationException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteAsyncTest43()
        {
            Coordinator target = new Coordinator();
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);
            accessor._isBusy = true;
            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;     

            try
            {
                // 実施
                target.ExecuteAsync(eventController);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (InvalidOperationException e)
            {
                // 成功
                Assert.AreEqual(MessageManager.GetMessage("E_BLOGIC_EXECUTEASYNC_OPERATED"), e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("予期せぬエラーです。" + e.ToString());
            }

            // 非同期処理が実行されていないこと
            Assert.IsTrue(target.IsBusy);

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(
                typeof(Coordinator), typeof(InvalidOperationException), "E_BLOGIC_EXECUTEASYNC_OPERATED"));
        }

        /// <summary>
        /// ExecuteAsync (EventController) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の eventController.ErrorHandler が null でなく、
        /// 前処理完了イベントでのエラー数が 0 で、resultString が任意の文字列の場合、
        /// 正常に実行されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExecuteAsyncTest44()
        {
            // 準備
            Coordinator target = new Coordinator();

            Coordinator_EventControllerStub08 eventController = new Coordinator_EventControllerStub08();
            Coordinator_IErrorHandlerImpl01 errorHandler = new Coordinator_IErrorHandlerImpl01();
            eventController.ErrorHandler = errorHandler;

            Coordinator_BLogicFactoryStub01.Initialize();
            Coordinator_IBLogicImpl01 blogic = new Coordinator_IBLogicImpl01();
            Coordinator_BLogicFactoryStub01._blogic = blogic;

            // 実施
            target.ExecuteAsync(eventController);

            // 確認
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);

            // _validatorフィールドに生成されたIValidator実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._validator, typeof(Coordinator_IValidatorImpl01));

            // _converterフィールドに生成されたIConverter実装クラスのインスタンスが設定されていること
            Assert.IsInstanceOfType(accessor._converter, typeof(Coordinator_IConverterImpl01));

            // _validator.ValidationFilePathが初期値であること
            Assert.IsNull(accessor._validator.ValidationFilePath);

            // _validator.RuleSetが初期値であること
            Assert.IsNull(accessor._validator.RuleSet);

            // _validator.Validate() が呼び出されないこと
            Assert.IsFalse(((Coordinator_IValidatorImpl01)accessor._validator).IsCalledValidate);

            // eventController.ReportCustomValidating() が呼び出され、引数で渡された
            // ValidationResultクラスのインスタンスのErrorsプロパティが初期値であること
            Assert.IsTrue(eventController.IsCalledReportCustomValidating);
            Assert.AreEqual(0, eventController.ReportCustomValidatingValidationResult.Errors.Count);

            // eventController.ReportValidationFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportValidationFailed);

            // _converter.Convert() が呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledConvert);

            // eventController.ReportPreprocessed() が呼び出されており、正しい引数がわたっていること
            // 引数 executionResultの各プロパティが初期値
            // 引数 blogicParamDataがBLogicParamTypeNameで指定した型
            Assert.IsTrue(eventController.IsCalledReportPreprocessed);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.ResultString);
            Assert.AreEqual(0, eventController.ReportPreprocessedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportPreprocessedExecutionResult.BLogicResultData);
            Assert.IsInstanceOfType(eventController.ReportPreprocessedBlogicParamData, _blogicParamType);

            // eventController.ErrorPathsにexecutionResult.Errorsの各ValidationMessageInfoのパスが追加されていること
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // eventController.ReportPreprocessedFailedメソッドが呼び出されており、正しい引数がわたっていること
            Assert.IsTrue(eventController.IsCalledReportPreprocessedFailed);
            Assert.AreEqual(0, eventController.ReportPreprocessedFailedExecutionResult.Errors.Count);
            Assert.AreEqual("terasoluna", eventController.ReportPreprocessedFailedExecutionResult.ResultString);
            Assert.IsNull(eventController.ReportPreprocessedFailedExecutionResult.BLogicResultData);

            // eventController.ReportExecuteCompleted() が呼び出され、正しい引数がわたっていること
            Assert.IsTrue(eventController.IsCalledReportExecuteCompleted);
            Assert.AreEqual("terasoluna", eventController.ReportExecuteCompletedExecutionResult.ResultString);
            Assert.IsNull(eventController.ReportExecuteCompletedExecutionResult.BLogicResultData);
            Assert.AreEqual(0, eventController.ReportExecuteCompletedExecutionResult.Errors.Count);
            Assert.IsNull(eventController.ReportExecuteCompletedError);
            Assert.IsFalse(eventController.ReportExecuteCompletedCancelled);

            // BLogicFactory.CreateBLogicメソッドが呼び出されないこと
            Assert.IsFalse(Coordinator_BLogicFactoryStub01._isCalledCreateInstance);

            // 非同期処理が実行されていないこと
            Assert.IsFalse(target.IsBusy);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        ///<summary>
        /// ExecuteAsyncCompletedCallback (object) のテスト
        ///</summary>
        ///<remarks>
        /// BLogicCompletedEventArgs.BLogicResult が null、
        /// BLogicCompletedEventArgs.Error が not null の場合、
        /// 正常に処理が実行されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ExecuteAsyncCompletedCallbackTest01()
        {
            // 準備
            ExecutionResult executionResult = new ExecutionResult();
            BLogicResult blogicResult = null;
            Exception error = new Coordinator_ExceptionStub01();
            bool cancelled = false;
            BLogicCompletedEventArgs obj = new BLogicCompletedEventArgs(executionResult, blogicResult, error, cancelled);

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            eventController.ConvertId = "convertId";
            eventController.Items.Add("key01", "value01");
            eventController.ErrorHandler = new Coordinator_IErrorHandlerImpl01();

            Coordinator target = new Coordinator();
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);
            accessor._eventController = eventController;
            accessor._converter = new Coordinator_IConverterImpl01();
            accessor._isBusy = true;

            // 実施
            accessor.ExecuteAsyncCompletedCallback(obj);

            // 確認
            // _converter.Reverseメソッドが呼び出されていないこと
            Assert.IsFalse(((Coordinator_IConverterImpl01)accessor._converter).IsCalledReverse);

            // eventController.ErrorHandler.HandleError()が呼ばれていないこと
            Assert.IsFalse(((Coordinator_IErrorHandlerImpl01)accessor._eventController.ErrorHandler).IsCalledHandleError);

            // eventController.ReportBLogicFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportBLogicFailed);

            // eventController.Itemsが変更されていないこと
            Assert.AreEqual(1, eventController.Items.Count);
            Assert.AreEqual("value01", eventController.Items["key01"]);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // eventController.ReportExecuteCompletedに正しい引数が渡されること
            Assert.IsTrue(eventController.IsCalledReportExecuteCompleted);
            Assert.IsNull(eventController.ReportExecuteCompletedExecutionResult);
            Assert.AreSame(error, eventController.ReportExecuteCompletedError);
            Assert.IsFalse(eventController.ReportExecuteCompletedCancelled);

            // _isBusyがfalseになっていること
            Assert.IsFalse(accessor._isBusy);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        ///<summary>
        /// ExecuteAsyncCompletedCallback (object) のテスト
        ///</summary>
        ///<remarks>
        /// BLogicCompletedEventArgs.ResultString が success で、その他の値が正常値の場合、
        /// 正常に処理が実行されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ExecuteAsyncCompletedCallbackTest02()
        {
            // 準備
            ExecutionResult executionResult = new ExecutionResult();
            BLogicResult blogicResult = new BLogicResult();
            blogicResult.ResultString = "success";
            blogicResult.ResultData = new Coordinator_DataSetStub04();
            MessageInfo messageInfo = new MessageInfo("terasoluna");
            blogicResult.Errors.Add(messageInfo);
            blogicResult.Items.Add("key001", "value001");
            Exception error = null;
            bool cancelled = false;
            BLogicCompletedEventArgs obj = new BLogicCompletedEventArgs(executionResult, blogicResult, error, cancelled);

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            eventController.ConvertId = "convertId";
            eventController.Items.Add("key01", "value01");
            eventController.ErrorHandler = new Coordinator_IErrorHandlerImpl01();

            Coordinator target = new Coordinator();
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);
            accessor._eventController = eventController;
            accessor._converter = new Coordinator_IConverterImpl01();
            accessor._isBusy = true;

            // 実施
            accessor.ExecuteAsyncCompletedCallback(obj);

            // 確認
            // _converter.Reverseメソッドが呼び出されていること
            Coordinator_IConverterImpl01 converter = (Coordinator_IConverterImpl01)accessor._converter;
            Assert.IsTrue(converter.IsCalledReverse);
            Assert.AreEqual("convertId", converter.ReverseId);
            Assert.AreSame(eventController.ViewData, converter.ReverseSrcData);
            Assert.IsInstanceOfType(converter.ReverseDestData, typeof(Coordinator_DataSetStub04));

            // eventController.ErrorHandler.HandleError()が呼ばれていないこと
            Assert.IsFalse(((Coordinator_IErrorHandlerImpl01)accessor._eventController.ErrorHandler).IsCalledHandleError);

            // eventController.ReportBLogicFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportBLogicFailed);

            // eventController.Itemsが更新されていること
            Assert.AreEqual(2, eventController.Items.Count);
            Assert.AreEqual("value01", eventController.Items["key01"]);
            Assert.AreEqual("value001", eventController.Items["key001"]);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // eventController.ReportExecuteCompletedに正しい引数が渡されること
            Assert.IsTrue(eventController.IsCalledReportExecuteCompleted);
            Assert.AreSame(blogicResult.ResultData, eventController.ReportExecuteCompletedExecutionResult.BLogicResultData);
            Assert.AreEqual(1, eventController.ReportExecuteCompletedExecutionResult.Errors.Count);
            Assert.AreSame(messageInfo, eventController.ReportExecuteCompletedExecutionResult.Errors[0]);
            Assert.AreEqual("success", eventController.ReportExecuteCompletedExecutionResult.ResultString);
            Assert.IsNull(eventController.ReportExecuteCompletedError);
            Assert.IsFalse(eventController.ReportExecuteCompletedCancelled);

            // _isBusyがfalseになっていること
            Assert.IsFalse(accessor._isBusy);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        ///<summary>
        /// ExecuteAsyncCompletedCallback (object) のテスト
        ///</summary>
        ///<remarks>
        /// BLogicCompletedEventArgs.ResultString が success 以外、
        /// EventController.ErrorHandler が null で、その他の値が正常値の場合、
        /// 正常に処理が実行されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ExecuteAsyncCompletedCallbackTest03()
        {
            // 準備
            ExecutionResult executionResult = new ExecutionResult();
            BLogicResult blogicResult = new BLogicResult();
            blogicResult.ResultString = "failure";
            blogicResult.ResultData = new Coordinator_DataSetStub04();
            MessageInfo messageInfo = new MessageInfo("terasoluna");
            blogicResult.Errors.Add(messageInfo);
            blogicResult.Items.Add("key001", "value001");
            Exception error = null;
            bool cancelled = false;
            BLogicCompletedEventArgs obj = new BLogicCompletedEventArgs(executionResult, blogicResult, error, cancelled);

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            eventController.ConvertId = "convertId";
            eventController.Items.Add("key01", "value01");

            Coordinator target = new Coordinator();
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);
            accessor._eventController = eventController;
            accessor._converter = new Coordinator_IConverterImpl01();
            accessor._isBusy = true;

            // 実施
            accessor.ExecuteAsyncCompletedCallback(obj);

            // 確認
            // _converter.Reverseメソッドが呼び出されていないこと
            Coordinator_IConverterImpl01 converter = (Coordinator_IConverterImpl01)accessor._converter;
            Assert.IsFalse(converter.IsCalledReverse);

            // eventController.ReportBLogicFailed() が呼び出されること
            Assert.IsTrue(eventController.IsCalledReportBLogicFailed);
            Assert.IsInstanceOfType(eventController.ReportBLogicFailedExecutionResult.BLogicResultData, typeof(Coordinator_DataSetStub04));
            Assert.AreEqual(1, eventController.ReportBLogicFailedExecutionResult.Errors.Count);
            Assert.AreEqual("terasoluna", eventController.ReportBLogicFailedExecutionResult.Errors[0].Message);
            Assert.AreEqual("failure", eventController.ReportBLogicFailedExecutionResult.ResultString);

            // eventController.Itemsが更新されていること
            Assert.AreEqual(2, eventController.Items.Count);
            Assert.AreEqual("value01", eventController.Items["key01"]);
            Assert.AreEqual("value001", eventController.Items["key001"]);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // eventController.ReportExecuteCompletedに正しい引数が渡されること
            Assert.IsTrue(eventController.IsCalledReportExecuteCompleted);
            Assert.AreSame(blogicResult.ResultData, eventController.ReportExecuteCompletedExecutionResult.BLogicResultData);
            Assert.AreEqual(1, eventController.ReportExecuteCompletedExecutionResult.Errors.Count);
            Assert.AreSame(messageInfo, eventController.ReportExecuteCompletedExecutionResult.Errors[0]);
            Assert.AreEqual("failure", eventController.ReportExecuteCompletedExecutionResult.ResultString);
            Assert.IsNull(eventController.ReportExecuteCompletedError);
            Assert.IsFalse(eventController.ReportExecuteCompletedCancelled);

            // _isBusyがfalseになっていること
            Assert.IsFalse(accessor._isBusy);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        ///<summary>
        /// ExecuteAsyncCompletedCallback (object) のテスト
        ///</summary>
        ///<remarks>
        /// BLogicCompletedEventArgs.ResultString が success 以外で、その他の値が正常値の場合、
        /// 正常に処理が実行されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ExecuteAsyncCompletedCallbackTest04()
        {
            // 準備
            ExecutionResult executionResult = new ExecutionResult();
            BLogicResult blogicResult = new BLogicResult();
            blogicResult.ResultString = "failure";
            blogicResult.ResultData = new Coordinator_DataSetStub04();
            MessageInfo messageInfo = new MessageInfo("terasoluna");
            blogicResult.Errors.Add(messageInfo);
            blogicResult.Items.Add("key001", "value001");
            Exception error = null;
            bool cancelled = false;
            BLogicCompletedEventArgs obj = new BLogicCompletedEventArgs(executionResult, blogicResult, error, cancelled);

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            eventController.ConvertId = "convertId";
            eventController.Items.Add("key01", "value01");
            Coordinator_IErrorHandlerImpl01 errorHandler = new Coordinator_IErrorHandlerImpl01();
            eventController.ErrorHandler = errorHandler;

            Coordinator target = new Coordinator();
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);
            accessor._eventController = eventController;
            accessor._converter = new Coordinator_IConverterImpl01();
            accessor._isBusy = true;

            // 実施
            accessor.ExecuteAsyncCompletedCallback(obj);

            // 確認
            // _converter.Reverseメソッドが呼び出されていないこと
            Coordinator_IConverterImpl01 converter = (Coordinator_IConverterImpl01)accessor._converter;
            Assert.IsFalse(converter.IsCalledReverse);

            // eventController.ErrorHandler.HandleError()が呼ばれていること
            Assert.IsTrue(errorHandler.IsCalledHandleError);
            Assert.AreEqual(1, errorHandler.HandleErrorMessages.Count);
            Assert.AreSame(messageInfo, errorHandler.HandleErrorMessages[0]);
            Assert.AreEqual("failure", errorHandler.HandleErrorResultString);
            Assert.AreEqual(eventController.ViewData , errorHandler.HandleErrorView);

            // eventController.ReportBLogicFailed() が呼び出されること
            Assert.IsTrue(eventController.IsCalledReportBLogicFailed);
            Assert.IsInstanceOfType(eventController.ReportBLogicFailedExecutionResult.BLogicResultData, typeof(Coordinator_DataSetStub04));
            Assert.AreEqual(1, eventController.ReportBLogicFailedExecutionResult.Errors.Count);
            Assert.AreEqual("terasoluna", eventController.ReportBLogicFailedExecutionResult.Errors[0].Message);
            Assert.AreEqual("failure", eventController.ReportBLogicFailedExecutionResult.ResultString);

            // eventController.Itemsが更新されていること
            Assert.AreEqual(2, eventController.Items.Count);
            Assert.AreEqual("value01", eventController.Items["key01"]);
            Assert.AreEqual("value001", eventController.Items["key001"]);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // eventController.ReportExecuteCompletedに正しい引数が渡されること
            Assert.IsTrue(eventController.IsCalledReportExecuteCompleted);
            Assert.AreSame(blogicResult.ResultData, eventController.ReportExecuteCompletedExecutionResult.BLogicResultData);
            Assert.AreEqual(1, eventController.ReportExecuteCompletedExecutionResult.Errors.Count);
            Assert.AreSame(messageInfo, eventController.ReportExecuteCompletedExecutionResult.Errors[0]);
            Assert.AreEqual("failure", eventController.ReportExecuteCompletedExecutionResult.ResultString);
            Assert.IsNull(eventController.ReportExecuteCompletedError);
            Assert.IsFalse(eventController.ReportExecuteCompletedCancelled);

            // _isBusyがfalseになっていること
            Assert.IsFalse(accessor._isBusy);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        ///<summary>
        /// ExecuteAsyncCompletedCallback (object) のテスト
        ///</summary>
        ///<remarks>
        /// BLogicCompletedEventArgs.ResultData が null、
        /// BLogicCompletedEventArgs.ConvertId が空文字列でない、
        /// その他の値が正常値の場合、正常に処理が実行されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ExecuteAsyncCompletedCallbackTest05()
        {
            // 準備
            ExecutionResult executionResult = new ExecutionResult();
            BLogicResult blogicResult = new BLogicResult();
            blogicResult.ResultString = "success";
            blogicResult.ResultData = null;
            MessageInfo messageInfo = new MessageInfo("terasoluna");
            blogicResult.Errors.Add(messageInfo);
            blogicResult.Items.Add("key001", "value001");
            Exception error = null;
            bool cancelled = false;
            BLogicCompletedEventArgs obj = new BLogicCompletedEventArgs(executionResult, blogicResult, error, cancelled);

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            eventController.ConvertId = "convertId";
            eventController.Items.Add("key01", "value01");
            eventController.ErrorHandler = new Coordinator_IErrorHandlerImpl01();

            Coordinator target = new Coordinator();
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);
            accessor._eventController = eventController;
            accessor._converter = new Coordinator_IConverterImpl01();
            accessor._isBusy = true;

            try
            {
                // 実施
                accessor.ExecuteAsyncCompletedCallback(obj);
            }
            catch (TerasolunaException e)
            {
                // 確認
                Assert.IsInstanceOfType(e, typeof(TerasolunaException));
                Assert.AreEqual(MessageManager.GetMessage("E_BLOGIC_RESULT_NULL_DATASET"), e.Message);

                // _converter.Reverseメソッドが呼び出されていないこと
                Coordinator_IConverterImpl01 converter = (Coordinator_IConverterImpl01)accessor._converter;
                Assert.IsFalse(converter.IsCalledReverse);

                // eventController.ErrorHandler.HandleError()が呼ばれていないこと
                Assert.IsFalse(((Coordinator_IErrorHandlerImpl01)accessor._eventController.ErrorHandler).IsCalledHandleError);

                // eventController.ReportBLogicFailed() が呼び出されないこと
                Assert.IsFalse(eventController.IsCalledReportBLogicFailed);

                // eventController.Itemsが更新されていること
                Assert.AreEqual(2, eventController.Items.Count);
                Assert.AreEqual("value01", eventController.Items["key01"]);
                Assert.AreEqual("value001", eventController.Items["key001"]);

                // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
                Assert.AreEqual(0, eventController.ErrorPaths.Count);

                // eventController.ReportExecuteCompletedに正しい引数が渡されること
                Assert.IsFalse(eventController.IsCalledReportExecuteCompleted);

                // _isBusyがfalseになっていること
                Assert.IsFalse(accessor._isBusy);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(
                typeof(Coordinator), typeof(TerasolunaException), "E_BLOGIC_RESULT_NULL_DATASET"));
        }

        ///<summary>
        /// ExecuteAsyncCompletedCallback (object) のテスト
        ///</summary>
        ///<remarks>
        /// BLogicCompletedEventArgs.ResultData が null、
        /// BLogicCompletedEventArgs.ConvertId が null、
        /// その他の値が正常値の場合、正常に処理が実行されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ExecuteAsyncCompletedCallbackTest06()
        {
            // 準備
            ExecutionResult executionResult = new ExecutionResult();
            BLogicResult blogicResult = new BLogicResult();
            blogicResult.ResultString = "success";
            blogicResult.ResultData = null;
            MessageInfo messageInfo = new MessageInfo("terasoluna");
            blogicResult.Errors.Add(messageInfo);
            blogicResult.Items.Add("key001", "value001");
            Exception error = null;
            bool cancelled = false;
            BLogicCompletedEventArgs obj = new BLogicCompletedEventArgs(executionResult, blogicResult, error, cancelled);

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            eventController.ConvertId = null;
            eventController.Items.Add("key01", "value01");
            eventController.ErrorHandler = new Coordinator_IErrorHandlerImpl01();

            Coordinator target = new Coordinator();
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);
            accessor._eventController = eventController;
            accessor._converter = new Coordinator_IConverterImpl01();
            accessor._isBusy = true;

            // 実施
            accessor.ExecuteAsyncCompletedCallback(obj);

            // 確認
            // _converter.Reverseメソッドが呼び出されていないこと
            Coordinator_IConverterImpl01 converter = (Coordinator_IConverterImpl01)accessor._converter;
            Assert.IsFalse(converter.IsCalledReverse);

            // eventController.ErrorHandler.HandleError()が呼ばれていないこと
            Assert.IsFalse(((Coordinator_IErrorHandlerImpl01)accessor._eventController.ErrorHandler).IsCalledHandleError);

            // eventController.ReportBLogicFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportBLogicFailed);

            // eventController.Itemsが更新されていること
            Assert.AreEqual(2, eventController.Items.Count);
            Assert.AreEqual("value01", eventController.Items["key01"]);
            Assert.AreEqual("value001", eventController.Items["key001"]);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // eventController.ReportExecuteCompletedに正しい引数が渡されること
            Assert.IsTrue(eventController.IsCalledReportExecuteCompleted);
            Assert.AreSame(blogicResult.ResultData, eventController.ReportExecuteCompletedExecutionResult.BLogicResultData);
            Assert.AreEqual(1, eventController.ReportExecuteCompletedExecutionResult.Errors.Count);
            Assert.AreSame(messageInfo, eventController.ReportExecuteCompletedExecutionResult.Errors[0]);
            Assert.AreEqual("success", eventController.ReportExecuteCompletedExecutionResult.ResultString);
            Assert.IsNull(eventController.ReportExecuteCompletedError);
            Assert.IsFalse(eventController.ReportExecuteCompletedCancelled);

            // _isBusyがfalseになっていること
            Assert.IsFalse(accessor._isBusy);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        ///<summary>
        /// ExecuteAsyncCompletedCallback (object) のテスト
        ///</summary>
        ///<remarks>
        /// BLogicCompletedEventArgs.ResultData が null、
        /// BLogicCompletedEventArgs.ConvertId が 空文字列、
        /// その他の値が正常値の場合、正常に処理が実行されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ExecuteAsyncCompletedCallbackTest07()
        {
            // 準備
            ExecutionResult executionResult = new ExecutionResult();
            BLogicResult blogicResult = new BLogicResult();
            blogicResult.ResultString = "success";
            blogicResult.ResultData = null;
            MessageInfo messageInfo = new MessageInfo("terasoluna");
            blogicResult.Errors.Add(messageInfo);
            blogicResult.Items.Add("key001", "value001");
            Exception error = null;
            bool cancelled = false;
            BLogicCompletedEventArgs obj = new BLogicCompletedEventArgs(executionResult, blogicResult, error, cancelled);

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            eventController.ConvertId = string.Empty;
            eventController.Items.Add("key01", "value01");
            eventController.ErrorHandler = new Coordinator_IErrorHandlerImpl01();

            Coordinator target = new Coordinator();
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);
            accessor._eventController = eventController;
            accessor._converter = new Coordinator_IConverterImpl01();
            accessor._isBusy = true;

            // 実施
            accessor.ExecuteAsyncCompletedCallback(obj);

            // 確認
            // _converter.Reverseメソッドが呼び出されていないこと
            Coordinator_IConverterImpl01 converter = (Coordinator_IConverterImpl01)accessor._converter;
            Assert.IsFalse(converter.IsCalledReverse);

            // eventController.ErrorHandler.HandleError()が呼ばれていないこと
            Assert.IsFalse(((Coordinator_IErrorHandlerImpl01)accessor._eventController.ErrorHandler).IsCalledHandleError);

            // eventController.ReportBLogicFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportBLogicFailed);

            // eventController.Itemsが更新されていること
            Assert.AreEqual(2, eventController.Items.Count);
            Assert.AreEqual("value01", eventController.Items["key01"]);
            Assert.AreEqual("value001", eventController.Items["key001"]);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // eventController.ReportExecuteCompletedに正しい引数が渡されること
            Assert.IsTrue(eventController.IsCalledReportExecuteCompleted);
            Assert.AreSame(blogicResult.ResultData, eventController.ReportExecuteCompletedExecutionResult.BLogicResultData);
            Assert.AreEqual(1, eventController.ReportExecuteCompletedExecutionResult.Errors.Count);
            Assert.AreSame(messageInfo, eventController.ReportExecuteCompletedExecutionResult.Errors[0]);
            Assert.AreEqual("success", eventController.ReportExecuteCompletedExecutionResult.ResultString);
            Assert.IsNull(eventController.ReportExecuteCompletedError);
            Assert.IsFalse(eventController.ReportExecuteCompletedCancelled);

            // _isBusyがfalseになっていること
            Assert.IsFalse(accessor._isBusy);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        ///<summary>
        /// ExecuteAsyncCompletedCallback (object) のテスト
        ///</summary>
        ///<remarks>
        /// BLogicCompletedEventArgs.Cancelled が true で、その他の値が正常値の場合、
        /// 正常に処理が実行されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ExecuteAsyncCompletedCallbackTest08()
        {
            // 準備
            ExecutionResult executionResult = new ExecutionResult();
            BLogicResult blogicResult = new BLogicResult();
            blogicResult.ResultString = "success";
            blogicResult.ResultData = new Coordinator_DataSetStub04();
            MessageInfo messageInfo = new MessageInfo("terasoluna");
            blogicResult.Errors.Add(messageInfo);
            blogicResult.Items.Add("key001", "value001");
            Exception error = null;
            bool cancelled = true;
            BLogicCompletedEventArgs obj = new BLogicCompletedEventArgs(executionResult, blogicResult, error, cancelled);

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            eventController.ConvertId = "convertId";
            eventController.Items.Add("key01", "value01");
            eventController.ErrorHandler = new Coordinator_IErrorHandlerImpl01();

            Coordinator target = new Coordinator();
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);
            accessor._eventController = eventController;
            accessor._converter = new Coordinator_IConverterImpl01();
            accessor._isBusy = true;

            // 実施
            accessor.ExecuteAsyncCompletedCallback(obj);

            // 確認
            // _converter.Reverseメソッドが呼び出されていないこと
            Coordinator_IConverterImpl01 converter = (Coordinator_IConverterImpl01)accessor._converter;
            Assert.IsFalse(converter.IsCalledReverse);

            // eventController.ErrorHandler.HandleError()が呼ばれていないこと
            Assert.IsFalse(((Coordinator_IErrorHandlerImpl01)accessor._eventController.ErrorHandler).IsCalledHandleError);

            // eventController.ReportBLogicFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportBLogicFailed);

            // eventController.Itemsが変更されていないこと
            Assert.AreEqual(1, eventController.Items.Count);
            Assert.AreEqual("value01", eventController.Items["key01"]);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // eventController.ReportExecuteCompletedに正しい引数が渡されること
            Assert.IsTrue(eventController.IsCalledReportExecuteCompleted);
            Assert.IsNull(eventController.ReportExecuteCompletedExecutionResult);
            Assert.IsNull(eventController.ReportExecuteCompletedError);
            Assert.IsTrue(eventController.ReportExecuteCompletedCancelled);

            // _isBusyがfalseになっていること
            Assert.IsFalse(accessor._isBusy);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        ///<summary>
        /// ExecuteAsyncCompletedCallback (object) のテスト
        ///</summary>
        ///<remarks>
        /// BLogicCompletedEventArgs.ResultData が not null、
        /// BLogicCompletedEventArgs.ConvertId が null、
        /// その他の値が正常値の場合、正常に処理が実行されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ExecuteAsyncCompletedCallbackTest09()
        {
            // 準備
            ExecutionResult executionResult = new ExecutionResult();
            BLogicResult blogicResult = new BLogicResult();
            blogicResult.ResultString = "success";
            blogicResult.ResultData = new Coordinator_DataSetStub04();
            MessageInfo messageInfo = new MessageInfo("terasoluna");
            blogicResult.Errors.Add(messageInfo);
            blogicResult.Items.Add("key001", "value001");
            Exception error = null;
            bool cancelled = false;
            BLogicCompletedEventArgs obj = new BLogicCompletedEventArgs(executionResult, blogicResult, error, cancelled);

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            eventController.ConvertId = null;
            eventController.Items.Add("key01", "value01");
            eventController.ErrorHandler = new Coordinator_IErrorHandlerImpl01();

            Coordinator target = new Coordinator();
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);
            accessor._eventController = eventController;
            accessor._converter = new Coordinator_IConverterImpl01();
            accessor._isBusy = true;

            // 実施
            accessor.ExecuteAsyncCompletedCallback(obj);

            // 確認
            // _converter.Reverseメソッドが呼び出されていないこと
            Coordinator_IConverterImpl01 converter = (Coordinator_IConverterImpl01)accessor._converter;
            Assert.IsFalse(converter.IsCalledReverse);

            // eventController.ErrorHandler.HandleError()が呼ばれていないこと
            Assert.IsFalse(((Coordinator_IErrorHandlerImpl01)accessor._eventController.ErrorHandler).IsCalledHandleError);

            // eventController.ReportBLogicFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportBLogicFailed);

            // eventController.Itemsが更新されていること
            Assert.AreEqual(2, eventController.Items.Count);
            Assert.AreEqual("value01", eventController.Items["key01"]);
            Assert.AreEqual("value001", eventController.Items["key001"]);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // eventController.ReportExecuteCompletedに正しい引数が渡されること
            Assert.IsTrue(eventController.IsCalledReportExecuteCompleted);
            Assert.AreSame(blogicResult.ResultData, eventController.ReportExecuteCompletedExecutionResult.BLogicResultData);
            Assert.AreEqual(1, eventController.ReportExecuteCompletedExecutionResult.Errors.Count);
            Assert.AreSame(messageInfo, eventController.ReportExecuteCompletedExecutionResult.Errors[0]);
            Assert.AreEqual("success", eventController.ReportExecuteCompletedExecutionResult.ResultString);
            Assert.IsNull(eventController.ReportExecuteCompletedError);
            Assert.IsFalse(eventController.ReportExecuteCompletedCancelled);

            // _isBusyがfalseになっていること
            Assert.IsFalse(accessor._isBusy);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        ///<summary>
        /// ExecuteAsyncCompletedCallback (object) のテスト
        ///</summary>
        ///<remarks>
        /// BLogicCompletedEventArgs.ResultData が not null、
        /// BLogicCompletedEventArgs.ConvertId が 空文字列、
        /// その他の値が正常値の場合、正常に処理が実行されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ExecuteAsyncCompletedCallbackTest10()
        {
            // 準備
            ExecutionResult executionResult = new ExecutionResult();
            BLogicResult blogicResult = new BLogicResult();
            blogicResult.ResultString = "success";
            blogicResult.ResultData = new Coordinator_DataSetStub04();
            MessageInfo messageInfo = new MessageInfo("terasoluna");
            blogicResult.Errors.Add(messageInfo);
            Exception error = null;
            bool cancelled = false;
            BLogicCompletedEventArgs obj = new BLogicCompletedEventArgs(executionResult, blogicResult, error, cancelled);

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            eventController.ConvertId = string.Empty;
            eventController.ErrorHandler = new Coordinator_IErrorHandlerImpl01();

            Coordinator target = new Coordinator();
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);
            accessor._eventController = eventController;
            accessor._converter = new Coordinator_IConverterImpl01();
            accessor._isBusy = true;

            // 実施
            accessor.ExecuteAsyncCompletedCallback(obj);

            // 確認
            // _converter.Reverseメソッドが呼び出されていないこと
            Coordinator_IConverterImpl01 converter = (Coordinator_IConverterImpl01)accessor._converter;
            Assert.IsFalse(converter.IsCalledReverse);

            // eventController.ErrorHandler.HandleError()が呼ばれていないこと
            Assert.IsFalse(((Coordinator_IErrorHandlerImpl01)accessor._eventController.ErrorHandler).IsCalledHandleError);

            // eventController.ReportBLogicFailed() が呼び出されないこと
            Assert.IsFalse(eventController.IsCalledReportBLogicFailed);

            // eventController.Itemsが変更されていないこと
            Assert.AreEqual(0, eventController.Items.Count);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(0, eventController.ErrorPaths.Count);

            // eventController.ReportExecuteCompletedに正しい引数が渡されること
            Assert.IsTrue(eventController.IsCalledReportExecuteCompleted);
            Assert.AreSame(blogicResult.ResultData, eventController.ReportExecuteCompletedExecutionResult.BLogicResultData);
            Assert.AreEqual(1, eventController.ReportExecuteCompletedExecutionResult.Errors.Count);
            Assert.AreSame(messageInfo, eventController.ReportExecuteCompletedExecutionResult.Errors[0]);
            Assert.AreEqual("success", eventController.ReportExecuteCompletedExecutionResult.ResultString);
            Assert.IsNull(eventController.ReportExecuteCompletedError);
            Assert.IsFalse(eventController.ReportExecuteCompletedCancelled);

            // _isBusyがfalseになっていること
            Assert.IsFalse(accessor._isBusy);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        ///<summary>
        /// ExecuteAsyncCompletedCallback (object) のテスト
        ///</summary>
        ///<remarks>
        /// BLogicCompletedEventArgs.ResultString が failure、
        /// BLogicResult.Errors に ValidationMessageInfo、その他の値が正常値の場合、
        /// 正常に処理が実行されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ExecuteAsyncCompletedCallbackTest11()
        {
            // 準備
            ExecutionResult executionResult = new ExecutionResult();
            BLogicResult blogicResult = new BLogicResult();
            blogicResult.ResultString = "failure";
            blogicResult.ResultData = new Coordinator_DataSetStub04();
            ValidationMessageInfo messageInfo = new ValidationMessageInfo("key0001", "message0001", "errorPath0001");
            blogicResult.Errors.Add(messageInfo);
            blogicResult.Items.Add("key001", "value001");
            Exception error = null;
            bool cancelled = false;
            BLogicCompletedEventArgs obj = new BLogicCompletedEventArgs(executionResult, blogicResult, error, cancelled);

            Coordinator_EventControllerStub01 eventController = new Coordinator_EventControllerStub01();
            eventController.ConvertId = "convertId";
            eventController.Items.Add("key01", "value01");
            Coordinator_IErrorHandlerImpl01 errorHandler = new Coordinator_IErrorHandlerImpl01();
            eventController.ErrorHandler = errorHandler;

            Coordinator target = new Coordinator();
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);
            accessor._eventController = eventController;
            accessor._converter = new Coordinator_IConverterImpl01();
            accessor._isBusy = true;

            // 実施
            accessor.ExecuteAsyncCompletedCallback(obj);

            // 確認
            // _converter.Reverseメソッドが呼び出されていないこと
            Coordinator_IConverterImpl01 converter = (Coordinator_IConverterImpl01)accessor._converter;
            Assert.IsFalse(converter.IsCalledReverse);

            // eventController.ErrorHandler.HandleError()が呼ばれていること
            Assert.IsTrue(errorHandler.IsCalledHandleError);
            Assert.AreEqual(1, errorHandler.HandleErrorMessages.Count);
            Assert.AreSame(messageInfo, errorHandler.HandleErrorMessages[0]);
            Assert.AreEqual("failure", errorHandler.HandleErrorResultString);
            Assert.AreEqual(eventController.ViewData, errorHandler.HandleErrorView);

            // eventController.ReportBLogicFailed() が呼び出されること
            Assert.IsTrue(eventController.IsCalledReportBLogicFailed);
            Assert.IsInstanceOfType(eventController.ReportBLogicFailedExecutionResult.BLogicResultData, typeof(Coordinator_DataSetStub04));
            Assert.AreEqual(1, eventController.ReportBLogicFailedExecutionResult.Errors.Count);
            Assert.AreEqual("message0001", eventController.ReportBLogicFailedExecutionResult.Errors[0].Message);
            Assert.AreEqual("failure", eventController.ReportBLogicFailedExecutionResult.ResultString);

            // eventController.Itemsが更新されていること
            Assert.AreEqual(2, eventController.Items.Count);
            Assert.AreEqual("value01", eventController.Items["key01"]);
            Assert.AreEqual("value001", eventController.Items["key001"]);

            // eventController.ErrorPathsプロパティのリスト数・値が正しいこと
            Assert.AreEqual(1, eventController.ErrorPaths.Count);
            Assert.AreEqual("errorPath0001", eventController.ErrorPaths[0]);

            // eventController.ReportExecuteCompletedに正しい引数が渡されること
            Assert.IsTrue(eventController.IsCalledReportExecuteCompleted);
            Assert.AreSame(blogicResult.ResultData, eventController.ReportExecuteCompletedExecutionResult.BLogicResultData);
            Assert.AreEqual(1, eventController.ReportExecuteCompletedExecutionResult.Errors.Count);
            Assert.AreSame(messageInfo, eventController.ReportExecuteCompletedExecutionResult.Errors[0]);
            Assert.AreEqual("failure", eventController.ReportExecuteCompletedExecutionResult.ResultString);
            Assert.IsNull(eventController.ReportExecuteCompletedError);
            Assert.IsFalse(eventController.ReportExecuteCompletedCancelled);

            // _isBusyがfalseになっていること
            Assert.IsFalse(accessor._isBusy);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(Coordinator), LogLevel.INFO));
        }

        /// <summary>
        /// CoordinatorType プロパティのテスト
        /// </summary>
        [TestMethod()]
        public void CoordinatorType01()
        {
            // 準備
            Coordinator target = new Coordinator();
            Type expected = typeof(Object);
            PropertyInfo property = typeof(Coordinator).GetProperty("CoordinatorType", BindingFlags.NonPublic | BindingFlags.Static);


            // 実施
            property.SetValue(target, expected, null);
            Type actual = property.GetValue(target, null) as Type;

            // 確認
            Assert.AreEqual(expected, actual);
        }

        /// <summary>
        /// EventController プロパティのテスト
        /// </summary>
        [TestMethod()]
        public void EventController01()
        {
            // 準備
            Coordinator target = new Coordinator();
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);
            EventController expected = new EventController();

            // 実施
            accessor.EventController = expected;
            EventController actual = accessor.EventController;

            // 確認
            Assert.AreEqual(expected, actual);
        }

        /// <summary>
        /// Validator プロパティのテスト
        /// </summary>
        [TestMethod()]
        public void Validator01()
        {
            // 準備
            Coordinator target = new Coordinator();
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);
            IValidator expected = new Coordinator_IValidatorImpl01();

            // 実施
            accessor.Validator = expected;
            IValidator actual = accessor.Validator;

            // 確認
            Assert.AreEqual(expected, actual);
        }

        /// <summary>
        /// Converter プロパティのテスト
        /// </summary>
        [TestMethod()]
        public void Converter01()
        {
            // 準備
            Coordinator target = new Coordinator();
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);
            IConverter expected = new Coordinator_IConverterImpl01();

            // 実施
            accessor.Converter = expected;
            IConverter actual = accessor.Converter;

            // 確認
            Assert.AreEqual(expected, actual);
        }

        /// <summary>
        /// AsyncOpe プロパティのテスト
        /// </summary>
        [TestMethod()]
        public void AsyncOpe01()
        {
            // 準備
            Coordinator target = new Coordinator();
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);
            AsyncOperation expected = AsyncOperationManager.CreateOperation(1);

            // 実施
            accessor.AsyncOpe = expected;
            AsyncOperation actual = accessor.AsyncOpe;

            // 確認
            Assert.AreEqual(expected, actual);
        }

        /// <summary>
        /// BLogic プロパティのテスト
        /// </summary>
        [TestMethod()]
        public void BLogic01()
        {
            // 準備
            Coordinator target = new Coordinator();
            TERASOLUNA_Fw_Client_CoordinatorAccessor accessor = new TERASOLUNA_Fw_Client_CoordinatorAccessor(target);
            IBLogic expected = new Coordinator_IBLogicImpl01();

            // 実施
            accessor.BLogic = expected;
            IBLogic actual = accessor.BLogic;

            // 確認
            Assert.AreEqual(expected, actual);
        }

        /// <summary>
        /// IsBusy プロパティのテスト
        /// </summary>
        [TestMethod()]
        public void IsBusy01()
        {
            // 準備
            Coordinator target = new Coordinator();
            PropertyInfo property = typeof(Coordinator).GetProperty("IsBusy");
            
            // 実施
            property.SetValue(target, true, null);
            bool actual = target.IsBusy;

            // 確認
            Assert.IsTrue(actual);
        }
    }
}
