﻿// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Text;
using System.Collections.Generic;
using TERASOLUNA.Fw.Client.Communication;
using System.IO;
using TERASOLUNA.Fw.Utlib.Logging;
using TERASOLUNA.Fw.Common.Logging;
using System.Runtime.Serialization;
using System.Runtime.Serialization.Formatters.Binary;
using TERASOLUNA.Fw.Common;
namespace TERASOLUNA.Fw.ClientUT.Communication
{
    /// <summary>
    /// TERASOLUNA.Fw.Client.Communication.ServerException のテスト クラスです。すべての
    /// TERASOLUNA.Fw.Client.Communication.ServerException 単体テストをここに含めます
    /// </summary>
    [TestClass()]
    public class ServerExceptionTest
    {
        private TestContext testContextInstance;

        ///<summary>
        /// 現在のテストの実行についての情報および機能を
        /// 提供するテスト コンテキストを取得または設定します。
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }
        #region 追加のテスト属性
        // 
        //テストを作成するときに、次の追加属性を使用することができます:
        //
        //クラスの最初のテストを実行する前にコードを実行するには、ClassInitialize を使用
        //
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //クラスのすべてのテストを実行した後にコードを実行するには、ClassCleanup を使用
        //
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //各テストを実行する前にコードを実行するには、TestInitialize を使用
        [TestInitialize()]
        public void MyTestInitialize()
        {
            LogUTUtil.Clear();
        }
        //
        //各テストを実行した後にコードを実行するには、TestCleanup を使用
        //
        //[TestCleanup()]
        //public void MyTestCleanup()
        //{
        //}
        //
        #endregion

        /// <summary>
        /// ServerException () のテスト
        /// </summary>
        /// <remarks>
        /// 引数無しのコンストラクタを呼び出した場合、インスタンスが返却されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ConstructorTest01()
        {
            //実行
            ServerException target = new ServerException();

            //検証
            Assert.IsNotNull(target);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ServerException), LogLevel.INFO));
        }

        /// <summary>
        /// ServerException (string) のテスト
        /// </summary>
        /// <remarks>
        /// エラーメッセージを引数とするコンストラクタを呼び出した場合、
        /// インスタンスが返却され取得したメッセージが正常であることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ConstructorStringTest01()
        {
            //準備
            string expectedMessage = "TERASOLUNA";

            //実行
            ServerException target = new ServerException(expectedMessage);

            //検証
            Assert.IsNotNull(target);
            Assert.AreEqual(expectedMessage, target.Message);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ServerException), LogLevel.INFO));
        }

        /// <summary>
        /// ServerException (string, Exception) のテスト
        /// </summary>
        /// <remarks>
        /// エラーメッセージと内部例外を引数とするコンストラクタを呼び出した場合、
        /// インスタンスが返却され取得したメッセージと例外が正常であることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ConstructorStringExceptionTest01()
        {
            //準備
            string expectedMessage = "TERASOLUNA";
            Exception expectedInnerExpection = new NullReferenceException();

            //実行
            ServerException target = new ServerException(expectedMessage, expectedInnerExpection);

            //検証
            Assert.IsNotNull(target);
            Assert.AreEqual(expectedMessage, target.Message);
            Assert.IsNotNull(target.InnerException);
            Assert.AreSame(expectedInnerExpection, target.InnerException);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ServerException), LogLevel.INFO));
        }

        /// <summary>
        /// ServerException (string) のテスト
        /// </summary>
        /// <remarks>
        /// エラーメッセージとエラータイプを引数とするコンストラクタを呼び出した場合、
        /// インスタンスが正常であることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ConstructorStringStringTest01()
        {
            //準備
            string expectedMessage = "TERASOLUNA01";
            string expectedErrorType = "TERASOLUNA02";

            //実行
            ServerException target = new ServerException(expectedMessage, expectedErrorType);

            //検証
            Assert.IsNotNull(target);
            Assert.AreEqual(expectedMessage, target.Message);
            Assert.AreEqual(expectedErrorType, target.ErrorType);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ServerException), LogLevel.INFO));
        }

        /// <summary>
        /// ServerException (string) のテスト
        /// </summary>
        /// <remarks>
        /// エラーメッセージとエラータイプ、エラー情報リストを引数とするコンストラクタを呼び出した場合、
        /// インスタンスが正常であることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ConstructorStringStringListTest01()
        {
            //準備
            string expectedMessage = "TERASOLUNA01";
            string expectedErrorType = "TERASOLUNA02";
            IList<MessageInfo> expectedErrors = new List<MessageInfo>();
            expectedErrors.Add(new MessageInfo("key01", "value01"));

            //実行
            ServerException target = new ServerException(expectedMessage, expectedErrorType, expectedErrors);

            //検証
            Assert.IsNotNull(target);
            Assert.AreEqual(expectedMessage, target.Message);
            Assert.AreEqual(expectedErrorType, target.ErrorType);
            Assert.AreSame(expectedErrors, target.Errors);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ServerException), LogLevel.INFO));
        }

        /// <summary>
        /// ServerException (SerializationInfo, StreamingContext) のテスト
        /// </summary>
        /// <remarks>
        /// シリアル化されたオブジェクトと転送元または転送先に関するコンテキスト情報を引数とするコンストラクタを呼び出した場合、
        /// インスタンスが正常であることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ConstructorSerializationInfoStreamingContextTest01()
        {
            string expectedMessage = "TERASOLUNA";
            ServerException newException = new ServerException(expectedMessage);
            ServerException deserializeException = null;

            // シリアライズ用に利用
            using (MemoryStream stream = new MemoryStream())
            {
                // 例外をシリアライズ
                IFormatter formatter = new BinaryFormatter();
                formatter.Serialize(stream, newException);

                // シリアライズした例外をデシリアライズ
                stream.Position = 0;
                deserializeException = (ServerException)formatter.Deserialize(stream);
            }

            //確認
            Assert.IsNotNull(newException);
            Assert.IsNotNull(deserializeException);
            Assert.AreEqual(expectedMessage, deserializeException.Message);

            //ログ
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ServerException), LogLevel.INFO));
        }
    }
}
