// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Text;
using System.Collections.Generic;
using System.Net;
using TERASOLUNA.Fw.Client.Communication;
using TERASOLUNA.Fw.Utlib.Logging;
using System.Data;
using System.Configuration;
using TERASOLUNA.Fw.ClientUT.Communication.MultipartSenderData;
using TERASOLUNA.Fw.Client;
using TERASOLUNA.Fw.Common.Logging;
using TERASOLUNA.Fw.Common;
using System.IO;
using System.Reflection;
using System.Collections;
namespace TERASOLUNA.Fw.ClientUT.Communication
{
    /// <summary>
    ///TERASOLUNA.Fw.Client.Communication.Sender.MultipartSender&lt;T&gt; のテスト クラスです。すべての
    ///TERASOLUNA.Fw.Client.Communication.Sender.MultipartSender&lt;T&gt; 単体テストをここに含めます
    ///</summary>
    [TestClass()]
    public class MultipartSenderTest
    {
        private TestContext testContextInstance;

        ///<summary>
        /// 正常な通信先を表すURL文字列です。
        ///</summary>
        private static string MULTIPART_TEST_SERVER_URL = ConfigurationManager.AppSettings["MultipartTestServerURL"];

        ///<summary>
        /// 存在しないサーバを表すURL文字列です。
        ///</summary>
        private static string TEST_NO_SERVER_URL = ConfigurationManager.AppSettings["TestNoServerUrl"];

        /// <summary>
        ///現在のテストの実行についての情報および機能を
        ///提供するテスト コンテキストを取得または設定します。
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }
        #region 追加のテスト属性
        // 
        //テストを作成するときに、次の追加属性を使用することができます:
        //
        //クラスの最初のテストを実行する前にコードを実行するには、ClassInitialize を使用
        //
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //クラスのすべてのテストを実行した後にコードを実行するには、ClassCleanup を使用
        //
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //各テストを実行する前にコードを実行するには、TestInitialize を使用
        [TestInitialize()]
        public void MyTestInitialize()
        {
            LogUTUtil.Clear();
        }
        //
        //各テストを実行した後にコードを実行するには、TestCleanup を使用
        //
        //[TestCleanup()]
        //public void MyTestCleanup()
        //{
        //}
        //
        #endregion

        /// <summary>
        /// Encoding のテスト
        /// </summary>
        /// <remarks>
        /// Encoding に null を設定した場合、
        /// UTF8が取得できることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void EncodingTest01()
        {
            // 準備
            MultipartSender target = new MultipartSender();
            Encoding value = null;
            TERASOLUNA_Fw_Client_Communication_MultipartSenderAccessor accessor = new TERASOLUNA_Fw_Client_Communication_MultipartSenderAccessor(target);
            accessor._encoding = null;

            // 実施
            target.Encoding = value;
            Encoding actualEncoding = target.Encoding;

            // 結果
            Assert.AreEqual(Encoding.UTF8, actualEncoding);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartSender), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<IList<MultipartElement>>), LogLevel.INFO));
        }

        /// <summary>
        /// Encoding のテスト
        /// </summary>
        /// <remarks>
        /// Encoding に ASCII を設定した場合、
        /// ASCII が取得できることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void EncodingTest02()
        {
            // 準備
            MultipartSender target = new MultipartSender();
            Encoding value = Encoding.ASCII;
            TERASOLUNA_Fw_Client_Communication_MultipartSenderAccessor accessor = new TERASOLUNA_Fw_Client_Communication_MultipartSenderAccessor(target);
            accessor._encoding = null;

            // 実施
            target.Encoding = value;
            Encoding actualEncoding = target.Encoding;

            // 結果
            Assert.AreEqual(Encoding.ASCII, actualEncoding);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartSender), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<IList<MultipartElement>>), LogLevel.INFO));
        }

        /// <summary>
        /// Send (HttpWebRequest, T, IDictionary&lt;string,string&gt;, IProgressChangeReporter) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 request が null の場合、
        /// ArgumentNullexception をスローすることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void SendTest01()
        {
            // 準備
            MultipartSender target = new MultipartSender();
            HttpWebRequest request = null;
            IList<MultipartElement> paramData = new List<MultipartElement>();
            MultipartValueElement element01 = new MultipartValueElement("terasolunaText01");
            element01.Value = "TERASOLUNA01";
            paramData.Add(element01);
            IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add("RequestName", "multipartTest01");
            IProgressChangeReporter reporter = null;
            target.BufferSize = 8192;
            target.CheckBoundary = true;

            // 実施
            try
            {
                target.Send(request, paramData, headerCollection, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // 確認
                Assert.AreEqual("request", e.ParamName);
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartSender), LogLevel.INFO));
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(SenderBase<IList<MultipartElement>>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(SenderBase<IList<MultipartElement>>), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "request"));
        }

        /// <summary>
        /// Send (HttpWebRequest, T, IDictionary&lt;string,string&gt;, IProgressChangeReporter) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 request がキャンセルされている場合、
        /// ArgumentNullException をスローすることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void SendTest02()
        {
            // 準備
            MultipartSender target = new MultipartSender();
            HttpWebRequest request = HttpWebRequest.Create(TEST_NO_SERVER_URL) as HttpWebRequest;
            IList<MultipartElement> paramData = new List<MultipartElement>();
            MultipartValueElement element01 = new MultipartValueElement("terasolunaText01");
            element01.Value = "TERASOLUNA01";
            paramData.Add(element01);
            IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add("RequestName", "multipartTest01");
            IProgressChangeReporter reporter = null;
            target.BufferSize = 8192;
            target.CheckBoundary = true;

            try
            {
                // 実施
                target.Send(request, paramData, headerCollection, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (WebException)
            {
                // 発生していることの確認
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartSender), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<IList<MultipartElement>>), LogLevel.INFO));
        }

        /// <summary>
        /// Send (HttpWebRequest, T, IDictionary&lt;string,string&gt;, IProgressChangeReporter) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 request の送信先のサーバが存在する場合、
        /// 正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void SendTest03()
        {
            // 準備
            MultipartSender target = new MultipartSender();
            HttpWebRequest request = HttpWebRequest.Create(MULTIPART_TEST_SERVER_URL) as HttpWebRequest;
            IList<MultipartElement> paramData = new List<MultipartElement>();
            MultipartValueElement element01 = new MultipartValueElement("terasolunaText01");
            element01.Value = "TERASOLUNA01";
            paramData.Add(element01);
            IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add("RequestName", "multipartTest01");
            IProgressChangeReporter reporter = null;
            target.BufferSize = 8192;
            target.CheckBoundary = true;

            // 実施
            target.Send(request, paramData, headerCollection, reporter);
            Assert.IsTrue(CheckRequestLogData(request, paramData));
            WebResponse response = request.GetResponse();

            // 通信結果の確認
            DataSet resultData = new DataSet();
            resultData.ReadXml(response.GetResponseStream());
            Assert.AreEqual("true", resultData.Tables["resultBean"].Rows[0]["result"]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartSender), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<IList<MultipartElement>>), LogLevel.INFO));
        }

        /// <summary>
        /// Send (HttpWebRequest, T, IDictionary&lt;string,string&gt;, IProgressChangeReporter) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 paramData が null の場合、
        /// ArgumentNullException をスローすることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void SendTest04()
        {
            // 準備
            MultipartSender target = new MultipartSender();
            HttpWebRequest request = HttpWebRequest.Create(MULTIPART_TEST_SERVER_URL) as HttpWebRequest;
            IList<MultipartElement> paramData = null;
            IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add("RequestName", "multipartTest01");
            IProgressChangeReporter reporter = null;
            target.BufferSize = 8192;
            target.CheckBoundary = true;

            try
            {
                // 実施
                target.Send(request, paramData, headerCollection, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                Assert.AreEqual("paramData", e.ParamName);
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartSender), LogLevel.INFO));
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(SenderBase<IList<MultipartElement>>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(SenderBase<IList<MultipartElement>>), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "paramData"));
        }

        /// <summary>
        /// Send (HttpWebRequest, T, IDictionary&lt;string,string&gt;, IProgressChangeReporter) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 paramData の要素数が 0 個の場合、
        /// ArgumentException をスローすることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void SendTest05()
        {
            // 準備
            MultipartSender target = new MultipartSender();
            HttpWebRequest request = HttpWebRequest.Create(MULTIPART_TEST_SERVER_URL) as HttpWebRequest;
            IList<MultipartElement> paramData = new List<MultipartElement>();
            IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add("RequestName", "multipartTest01");
            IProgressChangeReporter reporter = null;
            target.BufferSize = 8192;
            target.CheckBoundary = true;

            try
            {
                // 実施
                target.Send(request, paramData, headerCollection, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_EMPTY_ILIST", "paramData"), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(MultipartSender), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<IList<MultipartElement>>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(MultipartSender), typeof(ArgumentException), "E_EMPTY_ILIST", "paramData"));
        }

        /// <summary>
        /// Send (HttpWebRequest, T, IDictionary&lt;string,string&gt;, IProgressChangeReporter) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 paramData の要素数が 3 個でありマルチパート名の重複が無い場合、
        /// 正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void SendTest06()
        {
            // 準備
            MultipartSender target = new MultipartSender();
            HttpWebRequest request = HttpWebRequest.Create(MULTIPART_TEST_SERVER_URL) as HttpWebRequest;
            IList<MultipartElement> paramData = new List<MultipartElement>();
            MultipartValueElement element01 = new MultipartValueElement("terasolunaText01");
            element01.Value = "TERASOLUNA01";
            MultipartValueElement element02 = new MultipartValueElement("terasolunaText02");
            element02.Value = "TERASOLUNA02";
            MultipartValueElement element03 = new MultipartValueElement("terasolunaText03");
            element03.Value = "TERASOLUNA03";
            paramData.Add(element01);
            paramData.Add(element02);
            paramData.Add(element03);
            IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add("RequestName", "multipartTest02");
            IProgressChangeReporter reporter = null;
            target.BufferSize = 8192;
            target.CheckBoundary = true;

            // 実施
            target.Send(request, paramData, headerCollection, reporter);
            Assert.IsTrue(CheckRequestLogData(request, paramData));
            WebResponse response = request.GetResponse();

            // 通信結果の確認
            DataSet resultData = new DataSet();
            resultData.ReadXml(response.GetResponseStream());
            Assert.AreEqual("true", resultData.Tables["resultBean"].Rows[0]["result"]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartSender), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<IList<MultipartElement>>), LogLevel.INFO));
        }

        /// <summary>
        /// Send (HttpWebRequest, T, IDictionary&lt;string,string&gt;, IProgressChangeReporter) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 paramData の要素数が 3 個であり、マルチパート名の重複が有る場合、
        /// ArgumentException をスローすることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void SendTest07()
        {
            // 準備
            MultipartSender target = new MultipartSender();
            HttpWebRequest request = HttpWebRequest.Create(MULTIPART_TEST_SERVER_URL) as HttpWebRequest;
            IList<MultipartElement> paramData = new List<MultipartElement>();
            MultipartValueElement element01 = new MultipartValueElement("terasolunaText01");
            element01.Value = "TERASOLUNA01";
            MultipartValueElement element02 = new MultipartValueElement("terasolunaText02");
            element02.Value = "TERASOLUNA02";
            MultipartValueElement element03 = new MultipartValueElement("terasolunaText02");
            element03.Value = "TERASOLUNA03";
            paramData.Add(element01);
            paramData.Add(element02);
            paramData.Add(element03);
            IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add("RequestName", "multipartTest02");
            IProgressChangeReporter reporter = null;
            target.BufferSize = 8192;
            target.CheckBoundary = true;

            try
            {
                // 実施
                target.Send(request, paramData, headerCollection, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_MULTIPART_NAME_REPEATED", "terasolunaText02"), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(MultipartSender), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<IList<MultipartElement>>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(MultipartSender), typeof(ArgumentException), "E_MULTIPART_NAME_REPEATED", "terasolunaText02"));
        }

        /// <summary>
        /// Send (HttpWebRequest, T, IDictionary&lt;string,string&gt;, IProgressChangeReporter) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 paramData の要素数が 1 個であり、送信するMultipartFileElementに設定したパスが存在するファイルパスの場合、
        /// 正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Communication\MultipartSenderData\MultipartSender_SendFile01.bmp")]
        public void SendTest08()
        {
            // 準備
            MultipartSender target = new MultipartSender();
            HttpWebRequest request = HttpWebRequest.Create(MULTIPART_TEST_SERVER_URL) as HttpWebRequest;
            IList<MultipartElement> paramData = new List<MultipartElement>();
            MultipartFileElement element01 = new MultipartFileElement("terasolunaFile01");
            element01.UploadFilePath = "MultipartSender_SendFile01.bmp";
            paramData.Add(element01);
            IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add("RequestName", "multipartTest03");
            IProgressChangeReporter reporter = null;
            target.BufferSize = 8192;
            target.CheckBoundary = true;

            // 実施
            target.Send(request, paramData, headerCollection, reporter);
            Assert.IsTrue(CheckRequestLogData(request, paramData));
            WebResponse response = request.GetResponse();

            // 通信結果の確認
            DataSet resultData = new DataSet();
            resultData.ReadXml(response.GetResponseStream());
            Assert.AreEqual("true", resultData.Tables["resultBean"].Rows[0]["result"]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartSender), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<IList<MultipartElement>>), LogLevel.INFO));
        }

        /// <summary>
        /// Send (HttpWebRequest, T, IDictionary&lt;string,string&gt;, IProgressChangeReporter) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 paramData の要素数が 1 個であり、送信するMultipartValueElementが null の場合、
        /// ArgumentNullException をスローすることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void SendTest09()
        {
            // 準備
            MultipartSender target = new MultipartSender();
            HttpWebRequest request = HttpWebRequest.Create(MULTIPART_TEST_SERVER_URL) as HttpWebRequest;
            IList<MultipartElement> paramData = new List<MultipartElement>();
            MultipartValueElement element01 = null;
            paramData.Add(element01);
            IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add("RequestName", "multipartTest01");
            IProgressChangeReporter reporter = null;
            target.BufferSize = 8192;
            target.CheckBoundary = true;

            // 実施
            try
            {
                target.Send(request, paramData, headerCollection, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_NULL_PARAMETER", typeof(MultipartElement).Name), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(MultipartSender), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<IList<MultipartElement>>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(MultipartSender), typeof(ArgumentException), "E_NULL_PARAMETER", typeof(MultipartElement).Name));
        }

        /// <summary>
        /// Send (HttpWebRequest, T, IDictionary&lt;string,string&gt;, IProgressChangeReporter) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 paramData の要素数が 1 個であり、送信するMultipartFileElementが null の場合、
        /// ArgumentNullException をスローすることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void SendTest10()
        {
            // 準備
            MultipartSender target = new MultipartSender();
            HttpWebRequest request = HttpWebRequest.Create(MULTIPART_TEST_SERVER_URL) as HttpWebRequest;
            IList<MultipartElement> paramData = new List<MultipartElement>();
            MultipartFileElement element01 = null;
            paramData.Add(element01);
            IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add("RequestName", "multipartTest01");
            IProgressChangeReporter reporter = null;
            target.BufferSize = 8192;
            target.CheckBoundary = true;

            // 実施
            try
            {
                target.Send(request, paramData, headerCollection, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<IList<MultipartElement>>), LogLevel.INFO));
                Assert.AreEqual(MessageManager.GetMessage("E_NULL_PARAMETER", typeof(MultipartElement).Name), e.Message);
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<IList<MultipartElement>>), LogLevel.INFO));
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(MultipartSender), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(MultipartSender), typeof(ArgumentException), "E_NULL_PARAMETER", typeof(MultipartElement).Name));
        }

        /// <summary>
        /// Send (HttpWebRequest, T, IDictionary&lt;string,string&gt;, IProgressChangeReporter) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 paramData の要素数が 1 個であり、送信するMultipartFileElementに設定したパスが存在しないファイルパスの場合、
        /// FileNotFoundException をスローすることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void SendTest11()
        {
            // 準備
            MultipartSender target = new MultipartSender();
            HttpWebRequest request = HttpWebRequest.Create(MULTIPART_TEST_SERVER_URL) as HttpWebRequest;
            IList<MultipartElement> paramData = new List<MultipartElement>();
            string fileName = "NotExist.bmp";
            MultipartFileElement element01 = new MultipartFileElement("terasolunaFile01");
            element01.UploadFilePath = fileName;
            paramData.Add(element01);
            IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add("RequestName", "multipartTest01");
            IProgressChangeReporter reporter = null;
            target.BufferSize = 8192;
            target.CheckBoundary = true;

            // 実施
            try
            {
                target.Send(request, paramData, headerCollection, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (FileNotFoundException e)
            {
                Assert.AreEqual(fileName, e.FileName);
                Assert.AreEqual(MessageManager.GetMessage("E_COMMUNICATION_FILE_NOT_FOUND", fileName), e.Message);
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartSender), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<IList<MultipartElement>>), LogLevel.INFO));
        }

        /// <summary>
        /// Send (HttpWebRequest, T, IDictionary&lt;string,string&gt;, IProgressChangeReporter) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 paramData の要素数が 3 個であり、
        /// 送信するクラスはすべて MultipartFileElement の場合、
        /// 正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Communication\MultipartSenderData\MultipartSender_SendFile01.bmp")]
        [DeploymentItem(@"Communication\MultipartSenderData\MultipartSender_SendFile02.bmp")]
        [DeploymentItem(@"Communication\MultipartSenderData\MultipartSender_SendFile03.bmp")]
        public void SendTest12()
        {
            // 準備
            MultipartSender target = new MultipartSender();
            HttpWebRequest request = HttpWebRequest.Create(MULTIPART_TEST_SERVER_URL) as HttpWebRequest;
            IList<MultipartElement> paramData = new List<MultipartElement>();
            MultipartFileElement element01 = new MultipartFileElement("terasolunaFile01");
            MultipartFileElement element02 = new MultipartFileElement("terasolunaFile02");
            MultipartFileElement element03 = new MultipartFileElement("terasolunaFile03");
            element01.UploadFilePath = "MultipartSender_SendFile01.bmp";
            element02.UploadFilePath = "MultipartSender_SendFile02.bmp";
            element03.UploadFilePath = "MultipartSender_SendFile03.bmp";
            paramData.Add(element01);
            paramData.Add(element02);
            paramData.Add(element03);
            IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add("RequestName", "multipartTest04");
            IProgressChangeReporter reporter = null;
            target.BufferSize = 8192;
            target.CheckBoundary = true;

            // 実施
            target.Send(request, paramData, headerCollection, reporter);
            Assert.IsTrue(CheckRequestLogData(request, paramData));
            WebResponse response = request.GetResponse();

            // 通信結果の確認
            DataSet resultData = new DataSet();
            resultData.ReadXml(response.GetResponseStream());
            Assert.AreEqual("true", resultData.Tables["resultBean"].Rows[0]["result"]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartSender), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<IList<MultipartElement>>), LogLevel.INFO));
        }

        /// <summary>
        /// Send (HttpWebRequest, T, IDictionary&lt;string,string&gt;, IProgressChangeReporter) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 paramData の要素数が 3 個であり、
        /// 送信するクラスが MultipartValueElement 及び MultipartFileElement が混在する場合、
        /// 正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Communication\MultipartSenderData\MultipartSender_SendFile01.bmp")]
        public void SendTest13()
        {
            // 準備
            MultipartSender target = new MultipartSender();
            HttpWebRequest request = HttpWebRequest.Create(MULTIPART_TEST_SERVER_URL) as HttpWebRequest;
            IList<MultipartElement> paramData = new List<MultipartElement>();
            MultipartValueElement element01 = new MultipartValueElement("terasolunaText01");
            MultipartValueElement element02 = new MultipartValueElement("terasolunaText02");
            MultipartFileElement element03 = new MultipartFileElement("terasolunaFile01");
            element01.Value = "TERASOLUNA01";
            element02.Value = "TERASOLUNA02";
            element03.UploadFilePath = "MultipartSender_SendFile01.bmp";
            paramData.Add(element01);
            paramData.Add(element02);
            paramData.Add(element03);
            IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add("RequestName", "multipartTest05");
            IProgressChangeReporter reporter = null;
            target.BufferSize = 8192;
            target.CheckBoundary = true;

            // 実施
            target.Send(request, paramData, headerCollection, reporter);
            Assert.IsTrue(CheckRequestLogData(request, paramData));
            WebResponse response = request.GetResponse();

            // 通信結果の確認
            DataSet resultData = new DataSet();
            resultData.ReadXml(response.GetResponseStream());
            Assert.AreEqual("true", resultData.Tables["resultBean"].Rows[0]["result"]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartSender), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<IList<MultipartElement>>), LogLevel.INFO));
        }

        /// <summary>
        /// Send (HttpWebRequest, T, IDictionary&lt;string,string&gt;, IProgressChangeReporter) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 headerCollection が null の場合、
        /// ArgumentNullException をスローすることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Communication\MultipartSenderData\MultipartSender_SendFile01.bmp")]
        public void SendTest14()
        {
            // 準備
            MultipartSender target = new MultipartSender();
            HttpWebRequest request = HttpWebRequest.Create(MULTIPART_TEST_SERVER_URL) as HttpWebRequest;
            IList<MultipartElement> paramData = new List<MultipartElement>();
            MultipartValueElement element01 = new MultipartValueElement("terasolunaText01");
            element01.Value = "TERASOLUNA01";
            paramData.Add(element01);
            IDictionary<string, string> headerCollection = null;
            IProgressChangeReporter reporter = null;
            target.BufferSize = 8192;
            target.CheckBoundary = true;

            // 実施
            try
            {
                target.Send(request, paramData, headerCollection, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                Assert.AreEqual("headerCollection", e.ParamName);
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartSender), LogLevel.INFO));
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(SenderBase<IList<MultipartElement>>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(SenderBase<IList<MultipartElement>>), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "headerCollection"));
        }

        /// <summary>
        /// Send (HttpWebRequest, T, IDictionary&lt;string,string&gt;, IProgressChangeReporter) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 headerCollection の要素数が 0 個の場合、
        /// 正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Communication\MultipartSenderData\MultipartSender_SendFile01.bmp")]
        public void SendTest15()
        {
            // 準備
            MultipartSender target = new MultipartSender();
            HttpWebRequest request = HttpWebRequest.Create(MULTIPART_TEST_SERVER_URL) as HttpWebRequest;
            IList<MultipartElement> paramData = new List<MultipartElement>();
            MultipartValueElement element01 = new MultipartValueElement("terasolunaText01");
            element01.Value = "TERASOLUNA01";
            paramData.Add(element01);
            IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            IProgressChangeReporter reporter = null;
            target.BufferSize = 8192;
            target.CheckBoundary = true;

            // 実施
            target.Send(request, paramData, headerCollection, reporter);
            Assert.IsTrue(CheckRequestLogData(request, paramData));
            WebResponse response = request.GetResponse();

            // 通信結果の確認
            DataSet resultData = new DataSet();
            resultData.ReadXml(response.GetResponseStream());
            Assert.AreEqual("errors.8004C003", resultData.Tables["error"].Rows[0]["error-code"]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartSender), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<IList<MultipartElement>>), LogLevel.INFO));
        }

        /// <summary>
        /// Send (HttpWebRequest, T, IDictionary&lt;string,string&gt;, IProgressChangeReporter) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 headerCollection の要素数が 3 個であり、要素の key 値に空文字列が含まれる場合、
        /// TerasolunaException をスローすることとを確認するテスト。
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Communication\MultipartSenderData\MultipartSender_SendFile01.bmp")]
        public void SendTest16()
        {
            // 準備
            MultipartSender target = new MultipartSender();
            HttpWebRequest request = HttpWebRequest.Create(MULTIPART_TEST_SERVER_URL) as HttpWebRequest;
            IList<MultipartElement> paramData = new List<MultipartElement>();
            MultipartValueElement element01 = new MultipartValueElement("terasolunaText01");
            element01.Value = "TERASOLUNA01";
            paramData.Add(element01);
            IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add("RequestName", "multipartTest06");
            headerCollection.Add("", "terasolunaValue");
            headerCollection.Add("terasoluna", "terasolunaValue");
            IProgressChangeReporter reporter = null;
            target.BufferSize = 8192;
            target.CheckBoundary = true;

            // 実施
            try
            {
                target.Send(request, paramData, headerCollection, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // 成功
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartSender), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<IList<MultipartElement>>), LogLevel.INFO));
        }

        /// <summary>
        /// Send (HttpWebRequest, T, IDictionary&lt;string,string&gt;, IProgressChangeReporter) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 headerCollection の要素数が 3 個であり、要素の key 値に Content-Type が含まれる場合、
        /// headerCollection の Content-Type が優先されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Communication\MultipartSenderData\MultipartSender_SendFile01.bmp")]
        public void SendTest17()
        {
            // 準備
            MultipartSender target = new MultipartSender();
            HttpWebRequest request = HttpWebRequest.Create(MULTIPART_TEST_SERVER_URL) as HttpWebRequest;
            IList<MultipartElement> paramData = new List<MultipartElement>();
            MultipartValueElement element01 = new MultipartValueElement("terasolunaText01");
            element01.Value = "TERASOLUNA01";
            paramData.Add(element01);
            IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add("RequestName", "multipartTest09");
            headerCollection.Add("Content-Type", "text/xml");
            headerCollection.Add("terasoluna", "terasolunaValue");
            IProgressChangeReporter reporter = null;
            target.BufferSize = 8192;
            target.CheckBoundary = true;

            try
            {
                // 実施
                target.Send(request, paramData, headerCollection, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // 成功
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                Assert.AreEqual("このヘッダーは、適切なプロパティを使用して変更しなければなりません。\r\nパラメータ名: name", e.Message);
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartSender), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<IList<MultipartElement>>), LogLevel.INFO));
        }

        /// <summary>
        /// Send (HttpWebRequest, T, IDictionary&lt;string,string&gt;, IProgressChangeReporter) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 headerCollection の要素数が 3 個であり、要素の value 値に null が含まれる場合、
        /// 正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Communication\MultipartSenderData\MultipartSender_SendFile01.bmp")]
        public void SendTest18()
        {
            // 準備
            MultipartSender target = new MultipartSender();
            HttpWebRequest request = HttpWebRequest.Create(MULTIPART_TEST_SERVER_URL) as HttpWebRequest;
            IList<MultipartElement> paramData = new List<MultipartElement>();
            MultipartValueElement element01 = new MultipartValueElement("terasolunaText01");
            element01.Value = "TERASOLUNA01";
            paramData.Add(element01);
            IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add("RequestName", "multipartTest08");
            headerCollection.Add("terasoluna01", null);
            headerCollection.Add("terasoluna02", "terasolunaValue");
            IProgressChangeReporter reporter = null;
            target.BufferSize = 8192;
            target.CheckBoundary = true;

            // 実施
            target.Send(request, paramData, headerCollection, reporter);
            Assert.IsTrue(CheckRequestLogData(request, paramData));
            WebResponse response = request.GetResponse();

            // 通信結果の確認
            DataSet resultData = new DataSet();
            resultData.ReadXml(response.GetResponseStream());
            Assert.AreEqual("true", resultData.Tables["resultBean"].Rows[0]["result"]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartSender), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<IList<MultipartElement>>), LogLevel.INFO));
        }

        /// <summary>
        /// Send (HttpWebRequest, T, IDictionary&lt;string,string&gt;, IProgressChangeReporter) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 headerCollection の要素数が 3 個であり、要素の value 値に空文字列が含まれる場合、
        /// 正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Communication\MultipartSenderData\MultipartSender_SendFile01.bmp")]
        public void SendTest19()
        {
            // 準備
            MultipartSender target = new MultipartSender();
            HttpWebRequest request = HttpWebRequest.Create(MULTIPART_TEST_SERVER_URL) as HttpWebRequest;
            IList<MultipartElement> paramData = new List<MultipartElement>();
            MultipartValueElement element01 = new MultipartValueElement("terasolunaText01");
            element01.Value = "TERASOLUNA01";
            paramData.Add(element01);
            IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add("RequestName", "multipartTest08");
            headerCollection.Add("terasoluna01", "");
            headerCollection.Add("terasoluna02", "terasolunaValue");
            IProgressChangeReporter reporter = null;
            target.BufferSize = 8192;
            target.CheckBoundary = true;

            // 実施
            target.Send(request, paramData, headerCollection, reporter);
            Assert.IsTrue(CheckRequestLogData(request, paramData));
            WebResponse response = request.GetResponse();

            // 通信結果の確認
            DataSet resultData = new DataSet();
            resultData.ReadXml(response.GetResponseStream());
            Assert.AreEqual("true", resultData.Tables["resultBean"].Rows[0]["result"]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartSender), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<IList<MultipartElement>>), LogLevel.INFO));
        }

        /// <summary>
        /// Send (HttpWebRequest, T, IDictionary&lt;string,string&gt;, IProgressChangeReporter) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 headerCollection の要素数が 3 個であり、要素の value 値に null ・空文字列が含まない場合、
        /// 正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Communication\MultipartSenderData\MultipartSender_SendFile01.bmp")]
        public void SendTest20()
        {
            // 準備
            MultipartSender target = new MultipartSender();
            HttpWebRequest request = HttpWebRequest.Create(MULTIPART_TEST_SERVER_URL) as HttpWebRequest;
            IList<MultipartElement> paramData = new List<MultipartElement>();
            MultipartValueElement element01 = new MultipartValueElement("terasolunaText01");
            element01.Value = "TERASOLUNA01";
            paramData.Add(element01);
            IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add("RequestName", "multipartTest07");
            headerCollection.Add("terasoluna01", "terasolunaValue01");
            headerCollection.Add("terasoluna02", "terasolunaValue02");
            IProgressChangeReporter reporter = null;
            target.BufferSize = 8192;
            target.CheckBoundary = true;

            // 実施
            target.Send(request, paramData, headerCollection, reporter);
            Assert.IsTrue(CheckRequestLogData(request, paramData));
            WebResponse response = request.GetResponse();

            // 通信結果の確認
            DataSet resultData = new DataSet();
            resultData.ReadXml(response.GetResponseStream());
            Assert.AreEqual("true", resultData.Tables["resultBean"].Rows[0]["result"]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartSender), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<IList<MultipartElement>>), LogLevel.INFO));
        }

        /// <summary>
        /// Send (HttpWebRequest, T, IDictionary&lt;string,string&gt;, IProgressChangeReporter) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 reporter が not null の場合、
        /// reporterのイベントが実行され、正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Communication\MultipartSenderData\MultipartSender_SendFile01.bmp")]
        public void SendTest21()
        {
            // 準備
            MultipartSender target = new MultipartSender();
            HttpWebRequest request = HttpWebRequest.Create(MULTIPART_TEST_SERVER_URL) as HttpWebRequest;
            IList<MultipartElement> paramData = new List<MultipartElement>();
            MultipartValueElement element01 = new MultipartValueElement("terasolunaText01");
            element01.Value = "TERASOLUNA01";
            paramData.Add(element01);
            IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add("RequestName", "multipartTest01");
            MultipartSender_IProgressChangeReporterStub01 reporter = new MultipartSender_IProgressChangeReporterStub01();
            target.BufferSize = 8192;
            target.CheckBoundary = true;

            // 実施
            target.Send(request, paramData, headerCollection, reporter);
            Assert.IsTrue(CheckRequestLogData(request, paramData));
            WebResponse response = request.GetResponse();

            // 通信結果の確認
            DataSet resultData = new DataSet();
            resultData.ReadXml(response.GetResponseStream());
            Assert.AreEqual("true", resultData.Tables["resultBean"].Rows[0]["result"]);
            Assert.AreEqual(50, reporter._eventArgs.ProgressPercentage);
            Assert.AreEqual(2, reporter._count);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartSender), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<IList<MultipartElement>>), LogLevel.INFO));
        }

        /// <summary>
        /// Send (HttpWebRequest, T, IDictionary&lt;string,string&gt;, IProgressChangeReporter) のテスト
        /// </summary>
        /// <remarks>
        /// プロパティ BufferSize が 1 バイトの場合、
        /// 正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Communication\MultipartSenderData\MultipartSender_SendFile01.bmp")]
        public void SendTest22()
        {
            // 準備
            MultipartSender target = new MultipartSender();
            HttpWebRequest request = HttpWebRequest.Create(MULTIPART_TEST_SERVER_URL) as HttpWebRequest;
            IList<MultipartElement> paramData = new List<MultipartElement>();
            MultipartValueElement element01 = new MultipartValueElement("terasolunaText01");
            element01.Value = "TERASOLUNA01";
            paramData.Add(element01);
            IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add("RequestName", "multipartTest01");
            MultipartSender_IProgressChangeReporterStub01 reporter = new MultipartSender_IProgressChangeReporterStub01();
            target.BufferSize = 1;
            target.CheckBoundary = true;

            // 実施
            target.Send(request, paramData, headerCollection, reporter);
            Assert.IsTrue(CheckRequestLogData(request, paramData));
            WebResponse response = request.GetResponse();

            // 通信結果の確認
            DataSet resultData = new DataSet();
            resultData.ReadXml(response.GetResponseStream());
            Assert.AreEqual("true", resultData.Tables["resultBean"].Rows[0]["result"]);
            Assert.AreEqual(50, reporter._eventArgs.ProgressPercentage);
            Assert.AreEqual(13, reporter._count);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartSender), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<IList<MultipartElement>>), LogLevel.INFO));
        }

        /// <summary>
        /// Send (HttpWebRequest, T, IDictionary&lt;string,string&gt;, IProgressChangeReporter) のテスト
        /// </summary>
        /// <remarks>
        /// プロパティ CheckBoundary が false の場合、
        /// 正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Communication\MultipartSenderData\MultipartSender_SendFile01.bmp")]
        public void SendTest23()
        {
            // 準備
            MultipartSender target = new MultipartSender();
            HttpWebRequest request = HttpWebRequest.Create(MULTIPART_TEST_SERVER_URL) as HttpWebRequest;
            IList<MultipartElement> paramData = new List<MultipartElement>();
            MultipartValueElement element01 = new MultipartValueElement("terasolunaText01");
            element01.Value = "TERASOLUNA01";
            paramData.Add(element01);
            IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add("RequestName", "multipartTest01");
            IProgressChangeReporter reporter = null;
            target.BufferSize = 8192;
            target.CheckBoundary = false;

            // 実施
            target.Send(request, paramData, headerCollection, reporter);
            Assert.IsTrue(CheckRequestLogData(request, paramData));
            WebResponse response = request.GetResponse();

            // 通信結果の確認
            DataSet resultData = new DataSet();
            resultData.ReadXml(response.GetResponseStream());
            Assert.AreEqual("true", resultData.Tables["resultBean"].Rows[0]["result"]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartSender), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<IList<MultipartElement>>), LogLevel.INFO));
        }

        /// <summary>
        /// Send (HttpWebRequest, T, IDictionary&lt;string,string&gt;, IProgressChangeReporter) のテスト
        /// </summary>
        /// <remarks>
        /// 必ずバウンダリ文字列を再作成する場合、正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void SendTest24()
        {
            // 準備
            MultipartSender target = new MultipartSenderStub01();
            HttpWebRequest request = HttpWebRequest.Create(MULTIPART_TEST_SERVER_URL) as HttpWebRequest;
            IList<MultipartElement> paramData = new List<MultipartElement>();
            MultipartValueElement element01 = new MultipartValueElement("terasolunaText01");
            element01.Value = "TERASOLUNA01";
            paramData.Add(element01);
            IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add("RequestName", "multipartTest01");
            IProgressChangeReporter reporter = null;
            target.BufferSize = 8192;
            target.CheckBoundary = true;

            // 実施
            target.Send(request, paramData, headerCollection, reporter);
            Assert.IsTrue(CheckRequestLogData(request, paramData));
            WebResponse response = request.GetResponse();

            // 通信結果の確認
            DataSet resultData = new DataSet();
            resultData.ReadXml(response.GetResponseStream());
            Assert.AreEqual("true", resultData.Tables["resultBean"].Rows[0]["result"]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartSender), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<IList<MultipartElement>>), LogLevel.INFO));
        }

        /// <summary>
        /// Send (HttpWebRequest, T, IDictionary&lt;string,string&gt;, IProgressChangeReporter) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 request がキャンセルされている場合、
        /// ArgumentNullException をスローすることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void SendTest25()
        {
            // 準備
            MultipartSender target = new MultipartSender();
            HttpWebRequest request = HttpWebRequest.Create(MULTIPART_TEST_SERVER_URL) as HttpWebRequest;
            IList<MultipartElement> paramData = new List<MultipartElement>();
            MultipartValueElement element01 = new MultipartValueElement("terasolunaText01");
            element01.Value = "TERASOLUNA01";
            paramData.Add(element01);
            IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            IProgressChangeReporter reporter = null;
            target.BufferSize = 8192;
            target.CheckBoundary = true;

            // リクエストのキャンセル
            request.Abort();

            try
            {
                // 実施
                target.Send(request, paramData, headerCollection, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (WebException)
            {
                // 発生していることの確認
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartSender), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<IList<MultipartElement>>), LogLevel.INFO));
        }

        /// <summary>
        /// CalcSendPercentage (int, int, long, long) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 max が 0 の場合、
        /// ArgumentOutOfRangeException をスローすることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CalcSendPercentageTest01()
        {
            // 準備
            int current = 100;
            int max = 0;
            int sendDataSize = 100;
            int length = 100;
            MultipartSender target = new MultipartSender();
            TERASOLUNA_Fw_Client_Communication_MultipartSenderAccessor accessor = new TERASOLUNA_Fw_Client_Communication_MultipartSenderAccessor(target);

            try
            {
                // 実施
                accessor.CalcSendPercentage(current, max, sendDataSize, length);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentOutOfRangeException e)
            {
                //ArgumentOutOfRangeException exception = e.InnerException as ArgumentOutOfRangeException;
                Assert.AreEqual("指定された引数は、有効な値の範囲内にありません。", e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(MultipartSender), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckError(typeof(MultipartSender), typeof(ArgumentOutOfRangeException), "指定された引数は、有効な値の範囲内にありません。"));
        }

        /// <summary>
        /// CalcSendPercentage (int, int, long, long) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 length が 0 の場合、
        /// ArgumentOutOfRangeException をスローすることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CalcSendPercentageTest02()
        {
            // 準備
            int current = 100;
            int max = 100;
            int sendDataSize = 100;
            int length = 0;
            MultipartSender target = new MultipartSender();
            TERASOLUNA_Fw_Client_Communication_MultipartSenderAccessor accessor = new TERASOLUNA_Fw_Client_Communication_MultipartSenderAccessor(target);

            try
            {
                // 実施
                accessor.CalcSendPercentage(current, max, sendDataSize, length);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentOutOfRangeException e)
            {
                //ArgumentOutOfRangeException exception = e.InnerException as ArgumentOutOfRangeException;
                Assert.AreEqual("指定された引数は、有効な値の範囲内にありません。", e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(MultipartSender), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckError(typeof(MultipartSender), typeof(ArgumentOutOfRangeException), "指定された引数は、有効な値の範囲内にありません。"));
        }

        /// <summary>
        /// ExistsBoundaryKey (IList&lt;MultipartElement&gt;, string) のテスト
        /// </summary>
        /// <remarks>
        /// マルチパート名とバウンダリーが同一の場合、
        /// true を取得することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExistsBoundaryKeyTest01()
        {
            // 準備
            MultipartSender target = new MultipartSender();
            IList<MultipartElement> elements = new List<MultipartElement>();
            MultipartValueElement element01 = new MultipartValueElement("aab");
            element01.Value = "value";
            elements.Add(element01);
            TERASOLUNA_Fw_Client_Communication_MultipartSenderAccessor accessor = new TERASOLUNA_Fw_Client_Communication_MultipartSenderAccessor(target);
            string boundaryKey = "aab";

            // 実施
            bool actual = accessor.ExistsBoundaryKey(elements, boundaryKey);

            // 確認
            Assert.IsTrue(actual);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartSender), LogLevel.INFO));
        }

        /// <summary>
        /// ExistsBoundaryKey (IList&lt;MultipartElement&gt;, string) のテスト
        /// </summary>
        /// <remarks>
        /// マルチパート送信するテキストの内容とバウンダリーが同一の場合、
        /// true を取得することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExistsBoundaryKeyTest02()
        {
            // 準備
            MultipartSender target = new MultipartSender();
            IList<MultipartElement> list = new List<MultipartElement>();
            MultipartValueElement element01 = new MultipartValueElement("key");
            element01.Value = "aab";
            list.Add(element01);
            TERASOLUNA_Fw_Client_Communication_MultipartSenderAccessor accessor = new TERASOLUNA_Fw_Client_Communication_MultipartSenderAccessor(target);
            string boundaryKey = "aab";

            // 実施
            bool actual = accessor.ExistsBoundaryKey(list, boundaryKey);

            // 確認
            Assert.IsTrue(actual);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartSender), LogLevel.INFO));
        }

        /// <summary>
        /// ExistsBoundaryKey (IList&lt;MultipartElement&gt;, string) のテスト
        /// </summary>
        /// <remarks>
        /// マルチパート送信するテキストの内容とバウンダリーが同一の場合、
        /// true を取得することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExistsBoundaryKeyTest03()
        {
            // 準備
            MultipartSender target = new MultipartSender();
            IList<MultipartElement> list = new List<MultipartElement>();
            MultipartValueElement element01 = new MultipartValueElement("key");
            element01.Value = "aaabcd";
            list.Add(element01);
            TERASOLUNA_Fw_Client_Communication_MultipartSenderAccessor accessor = new TERASOLUNA_Fw_Client_Communication_MultipartSenderAccessor(target);
            string boundaryKey = "aab";

            // 実施
            bool actual = accessor.ExistsBoundaryKey(list, boundaryKey);

            // 確認
            Assert.IsTrue(actual);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartSender), LogLevel.INFO));
        }

        /// <summary>
        /// ExistsBoundaryKey (IList&lt;MultipartElement&gt;, string) のテスト
        /// </summary>
        /// <remarks>
        /// マルチパート送信するテキストの内容とバウンダリーが異なるの場合、
        /// false を取得することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExistsBoundaryKeyTest04()
        {
            // 準備
            MultipartSender target = new MultipartSender();
            IList<MultipartElement> list = new List<MultipartElement>();
            MultipartValueElement element01 = new MultipartValueElement("key");
            element01.Value = "vabluabe";
            list.Add(element01);
            TERASOLUNA_Fw_Client_Communication_MultipartSenderAccessor accessor = new TERASOLUNA_Fw_Client_Communication_MultipartSenderAccessor(target);
            string boundaryKey = "abc";

            // 実施
            bool actual = accessor.ExistsBoundaryKey(list, boundaryKey);

            // 確認
            Assert.IsFalse(actual);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartSender), LogLevel.INFO));
        }

        /// <summary>
        /// ExistsBytesFromStream(Stream,byte[]) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 target に 空設定した場合、
        /// 規定値のエンコードを取得することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExistsBytesFromStreamTest01()
        {
            // 準備
            Stream stream = null;
            byte[] target = new byte[5];

            // 実施
            bool actual = TERASOLUNA_Fw_Client_Communication_MultipartSenderAccessor.ExistsBytesFromStream(stream, target);

            // 確認
            Assert.IsFalse(actual);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartSender), LogLevel.INFO));
        }

        /// <summary>
        /// ExistsBytesFromStream(Stream,byte[]) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 target に 空設定した場合、
        /// 規定値のエンコードを取得することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ExistsBytesFromStreamTest02()
        {
            // 準備
            Stream stream = new MemoryStream();
            byte[] target = new byte[0];

            // 実施
            bool actual = TERASOLUNA_Fw_Client_Communication_MultipartSenderAccessor.ExistsBytesFromStream(stream, target);

            // 確認
            Assert.IsFalse(actual);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartSender), LogLevel.INFO));
        }

        /// <summary>
        /// 電文ログの出力チェックを行い、成功時にtrueを返却するメソッド。
        /// </summary>
        /// <param name="request">リクエスト</param>
        /// <param name="paramData">データセット</param>
        /// <returns>成功の場合true</returns>
        private bool CheckRequestLogData(HttpWebRequest request, IList<MultipartElement> paramData)
        {
            if (!LogUTUtil.CheckTraceUseMessageId(typeof(MultipartSender), "T_REQUEST_SEND_ADDRESS", request.Address.ToString()))
            {
                return false;
            }

            StringBuilder requestHeaders = new StringBuilder();
            requestHeaders.AppendLine(MessageManager.GetMessage("T_REQUEST_SEND_HEADER"));
            foreach (string key in request.Headers.AllKeys)
            {
                requestHeaders.AppendLine(MessageManager.GetMessage("T_DICTIONARY_KEY_VALUE", key, request.Headers[key]));
            }

            if (!LogUTUtil.CheckTrace(typeof(MultipartSender), requestHeaders.ToString().Trim()))
            {
                return false;
            }

            // HTTPボディ
            StringBuilder requestBody = new StringBuilder();
            requestBody.AppendLine(MessageManager.GetMessage("T_REQUEST_SEND_BODY"));
            foreach (MultipartElement element in paramData)
            {
                if (element.GetType().Name.Equals(typeof(MultipartValueElement).Name))
                {
                    MultipartValueElement valueElement = element as MultipartValueElement;
                    requestBody.AppendLine(MessageManager.GetMessage("T_MULTIPART_VALUE", valueElement.Name, valueElement.Value));
                }
                else
                {
                    MultipartFileElement fileElement = element as MultipartFileElement;
                    requestBody.AppendLine(MessageManager.GetMessage("T_MULTIPART_FILE", fileElement.Name, fileElement.UploadFilePath));
                }
            }

            if (!LogUTUtil.CheckTrace(typeof(MultipartSender), requestBody.ToString().Trim()))
            {
                return false;
            }

            return true;
        }
    }

}
