﻿// <著作権>
// Copyright (c) 2007, NTT DATA Corporation.

using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Text;
using System.Collections.Generic;
using System.IO;
using NTTDATA.TERASOLUNA.FW.Web;
using NTTDATA.TERASOLUNA.FW.Web.Controller.Multipart;

namespace MultipartUploadUT
{
    ///<summary>
    /// MultipartFileItem のテストクラスです。
    ///</summary>
    ///<author>
    /// 串野亮
    ///</author>
    [TestClass()]
    public class MultipartFileItemTest
    {

        private TestContext testContextInstance;

        // ボディファイル(終端文字列有り)
        private static readonly string FILE_PATH_SUCCESS 
            = @"MultipartFileItem.txt";

        // ボディファイル(終端文字列無し)
        private static readonly string FILE_PATH_FAIL 
            = @"MultipartFileItem_error.txt";

        // boundary[文字列 : boundary]
        private static readonly string BOUNDARY = "boundary";

        // エンコード[UTF-8]
        private static readonly Encoding ENCODING = new UTF8Encoding();

        ///<summary>
        /// 現在のテストの実行についての情報および機能を
        /// 提供するテスト コンテキストを取得または設定します。
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }

        #region 追加のテスト属性
        // 
        //テストを作成するときに、次の追加属性を使用することができます:
        //
        //クラスの最初のテストを実行する前にコードを実行するには、ClassInitialize を使用
        //
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //クラスのすべてのテストを実行した後にコードを実行するには、ClassCleanup を使用
        //
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //各テストを実行する前にコードを実行するには、TestInitialize を使用
        //
        //[TestInitialize()]
        //public void MyTestInitialize()
        //{
        //}
        //
        //各テストを実行した後にコードを実行するには、TestCleanup を使用
        //
        //[TestCleanup()]
        //public void MyTestCleanup()
        //{
        //}
        //
        #endregion

        ///<summary>
        /// MultipartFileItem (string, Stream, IDictionary&lt;string,string&gt;, Encoding) のテスト
        ///</summary>
        ///<remarks>
        /// 引数が全て正常な値の時、正常に動作することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartFileItem.txt")]
        public void ConstructorTest01()
        {
            using (Stream partStream = new FileStream(FILE_PATH_SUCCESS, FileMode.Open, FileAccess.Read))
            {
                // 準備
                string fileName = "ConstructorTest01.txt";
                IDictionary<string, string> headerList 
                    = new Dictionary<string, string>();
                headerList.Add(
                    "CONTENT-DISPOSITION",
                    "form-data; name=file1;filename=" + fileName);

                // 実行
                MultipartFileItem target =
                    new MultipartFileItem(
                        BOUNDARY, partStream, headerList, ENCODING);

                // 検証
                NTTDATA_TERASOLUNA_FW_Web_Multipart_MultipartFileItemAccessor accessor
                    = new NTTDATA_TERASOLUNA_FW_Web_Multipart_MultipartFileItemAccessor(target);
                Assert.AreEqual(fileName, accessor._filename);
                Assert.IsTrue(File.Exists(accessor._temporaryFilePath));
                Assert.IsNotNull(accessor._outputStream);
            }
        }

        ///<summary>
        /// MultipartFileItem (string, Stream, IDictionary&lt;string,string&gt;, Encoding) のテスト
        ///</summary>
        ///<remarks>
        /// 引数partStreamがnullの場合、
        /// 例外(ArgumentNullException)が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartFileItem.txt")]
        public void ConstructorTest02()
        {
            // 準備
            Stream partStream = null;
            IDictionary<string, string> headerList
                = new Dictionary<string, string>();
            headerList.Add(
                "CONTENT-DISPOSITION", 
                "form-data; name=file1;filename=ConstructorTest01.txt");

            try
            {
                // 失敗
                new MultipartFileItem(BOUNDARY, partStream, headerList, ENCODING);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException)
            {
                // 成功
            }
            catch (Exception ex)
            {
                // 失敗
                Assert.Fail("予期せぬエラーです。" + ex.ToString());
            }
        }

        ///<summary>
        /// MultipartFileItem (string, Stream, IDictionary&lt;string,string&gt;, Encoding) のテスト
        ///</summary>
        ///<remarks>
        /// 引数partStreamのファイルの区切り文字が存在しない場合、
        /// 例外(InvalidRequestException)が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartFileItem_error.txt")]
        public void ConstructorTest03()
        {
            using (Stream partStream = new FileStream(FILE_PATH_FAIL, FileMode.Open,FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList
                    = new Dictionary<string, string>();
                headerList.Add(
                    "CONTENT-DISPOSITION", 
                    "form-data; name=file1;filename=constructorTest02.txt");

                try
                {
                    // 失敗
                    new MultipartFileItem(
                        BOUNDARY, partStream, headerList, ENCODING);
                    Assert.Fail("到達してはいけないコードです。");
                }
                catch (InvalidRequestException)
                {
                    // 成功
                }
                catch (Exception ex)
                {
                    // 失敗
                    Assert.Fail("予期せぬエラーです。" + ex.ToString());
                }
            }
        }

        ///<summary>
        /// MultipartFileItem (string, Stream, IDictionary&lt;string,string&gt;, Encoding) のテスト
        ///</summary>
        ///<remarks>
        /// 引数headerListに"Content-Disposition"が存在しない場合、
        /// 例外(InvalidRequestException)が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartFileItem.txt")]
        public void ConstructorTest04()
        {
            using (Stream partStream = new FileStream(FILE_PATH_SUCCESS, FileMode.Open, FileAccess.Read))
            {
                try
                {
                    // 準備
                    IDictionary<string, string> headerList
                        = new Dictionary<string, string>();

                    // 失敗
                    new MultipartFileItem(
                        BOUNDARY, partStream, headerList, ENCODING);
                    Assert.Fail("到達してはいけないコードです。");
                }
                catch (InvalidRequestException)
                {
                    // 成功
                }
                catch (Exception ex)
                {
                    // 失敗
                    Assert.Fail("予期せぬエラーです。" + ex.ToString());
                }
            }
        }

        ///<summary>
        /// MultipartFileItem (string, Stream, IDictionary&lt;string,string&gt;, Encoding) のテスト
        ///</summary>
        ///<remarks>
        /// 引数headerListに"Content-Disposition"(大文字)が存在し,
        /// filenameが存在しない場合、
        /// 例外(InvalidRequestException)が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartFileItem.txt")]
        public void ConstructorTest05()
        {

            using (Stream partStream = new FileStream(FILE_PATH_SUCCESS, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList
                    = new Dictionary<string, string>();
                headerList.Add("CONTENT-DISPOSITION", "form-data; name=file1");

                try
                {
                    // 失敗
                    new MultipartFileItem(
                        BOUNDARY, partStream, headerList, ENCODING);
                    Assert.Fail("到達してはいけないコードです。");
                }
                catch (InvalidRequestException)
                {
                    // 成功
                }
                catch (Exception ex)
                {
                    // 失敗
                    Assert.Fail("予期せぬエラーです。" + ex.ToString());
                }
            }
        }

        ///<summary>
        /// MultipartFileItem (string, Stream, IDictionary&lt;string,string&gt;, Encoding) のテスト
        ///</summary>
        ///<remarks>
        /// 引数headerListに"Content-Disposition"(大文字)が存在し、
        /// "filename"(小文字 + 大文字)が存在する場合、
        /// 正常に動作していることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartFileItem.txt")]
        public void ConstructorTest06()
        {
            using (Stream partStream = new FileStream(FILE_PATH_SUCCESS, FileMode.Open, FileAccess.Read))
            {
                // 準備
                string fileName = "ConstructorTest07.txt";
                IDictionary<string, string> headerList
                    = new Dictionary<string, string>();
                headerList.Add(
                    "CONTENT-DISPOSITION",
                    "form-data; name=file1;FileName=" + fileName);

                // 実行
                MultipartFileItem target
                    = new MultipartFileItem(
                        BOUNDARY, partStream, headerList, ENCODING);

                // 検証
                NTTDATA_TERASOLUNA_FW_Web_Multipart_MultipartFileItemAccessor accessor
                    = new NTTDATA_TERASOLUNA_FW_Web_Multipart_MultipartFileItemAccessor(target);
                Assert.AreEqual(fileName, accessor._filename);
                Assert.IsTrue(File.Exists(accessor._temporaryFilePath));
                Assert.IsNotNull(accessor._outputStream);
            }
        }

        ///<summary>
        /// MultipartFileItem (string, Stream, IDictionary&lt;string,string&gt;, Encoding) のテスト
        ///</summary>
        ///<remarks>
        /// 引数headerListに"Content-Disposition"(大文字)が存在し、
        /// "filename"(大文字)が1個存在する場合、
        /// 正常に動作していることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartFileItem.txt")]
        public void ConstructorTest07()
        {
            using (Stream partStream = new FileStream(FILE_PATH_SUCCESS, FileMode.Open, FileAccess.Read))
            {
                // 準備
                string fileName = "ConstructorTest07.txt";
                IDictionary<string, string> headerList
                    = new Dictionary<string, string>();
                headerList.Add(
                    "CONTENT-DISPOSITION",
                    "form-data; name=file1;FILENAME=" + fileName);

                // 実行
                MultipartFileItem target
                    = new MultipartFileItem(
                        BOUNDARY, partStream, headerList, ENCODING);

                // 検証
                NTTDATA_TERASOLUNA_FW_Web_Multipart_MultipartFileItemAccessor accessor
                    = new NTTDATA_TERASOLUNA_FW_Web_Multipart_MultipartFileItemAccessor(target);
                Assert.AreEqual(fileName, accessor._filename);
                Assert.IsTrue(File.Exists(accessor._temporaryFilePath));
                Assert.IsNotNull(accessor._outputStream);
            }
        }


        ///<summary>
        /// MultipartFileItem (string, Stream, IDictionary&lt;string,string&gt;, Encoding) のテスト
        ///</summary>
        ///<remarks>
        /// 引数headerListに"Content-Disposition"(大文字)が存在し、
        /// "filename"を含む別の文字列が存在する場合、
        /// 例外(InvalidRequestException)が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartFileItem.txt")]
        public void ConstructorTest08()
        {

            using (Stream partStream = new FileStream(FILE_PATH_SUCCESS, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList
                    = new Dictionary<string, string>();
                headerList.Add(
                    "CONTENT-DISPOSITION",
                    "form-data; name=file1;filenames=constructorTest06.txt");

                try
                {
                    // 失敗
                    new MultipartFileItem(
                        BOUNDARY, partStream, headerList, ENCODING);
                    Assert.Fail("到達してはいけないコードです。");
                }
                catch (InvalidRequestException)
                {
                    // 成功
                }
                catch (Exception ex)
                {
                    // 失敗
                    Assert.Fail("予期せぬエラーです。" + ex.ToString());
                }
            }
        }

        ///<summary>
        /// MultipartFileItem (string, Stream, IDictionary&lt;string,string&gt;, Encoding) のテスト
        ///</summary>
        ///<remarks>
        /// 引数headerListに"Content-Disposition"が存在し、"filename"が2個存在する場合、
        /// 例外(InvalidRequestException)が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartFileItem.txt")]
        public void ConstructorTest09()
        {
            using (Stream partStream = new FileStream(FILE_PATH_SUCCESS, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList
                    = new Dictionary<string, string>();
                headerList.Add(
                    "CONTENT-DISPOSITION",
                    "form-data; name=file1;filename=test01.txt;filename=test02.txt");

                try
                {
                    // 失敗
                    new MultipartFileItem(
                        BOUNDARY, partStream, headerList, ENCODING);
                    Assert.Fail("到達してはいけないコードです。");
                }
                catch (InvalidRequestException)
                {
                    // 成功
                }
                catch (Exception ex)
                {
                    // 失敗
                    Assert.Fail("予期せぬエラーです。" + ex.ToString());
                }
            }
        }


        ///<summary>
        /// MultipartFileItem (string, Stream, IDictionary&lt;string,string&gt;, Encoding) のテスト
        ///</summary>
        ///<remarks>
        /// 引数headerListに"Content-Disposition"を含む別の文字列が存在する場合、
        /// 例外(InvalidRequestException)が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartFileItem.txt")]
        public void ConstructorTest10()
        {

            using (Stream partStream = new FileStream(FILE_PATH_SUCCESS, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList
                    = new Dictionary<string, string>();
                headerList.Add(
                    "CONTENT-DISPOSITIONS",
                    "form-data; name=file1;filename=constructorTest08.txt");

                try
                {
                    // 失敗
                    new MultipartFileItem(
                        BOUNDARY, partStream, headerList, ENCODING);
                    Assert.Fail("到達してはいけないコードです。");
                }
                catch (InvalidRequestException)
                {
                    // 成功
                }
                catch (Exception ex)
                {
                    // 失敗
                    Assert.Fail("予期せぬエラーです。" + ex.ToString());
                }
            }
        }

        ///<summary>
        /// Filename のテスト
        ///</summary>
        ///<remarks>
        /// Filenameでnullを取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartFileItem.txt")]
        public void FilenameTest01()
        {
            using (Stream partStream = new FileStream(FILE_PATH_SUCCESS, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList
                    = new Dictionary<string, string>();
                headerList.Add(
                    "CONTENT-DISPOSITION", "form-data; name=file1;filename=text.txt");
                MultipartFileItem target
                    = new MultipartFileItem(BOUNDARY, partStream, headerList, ENCODING);
                NTTDATA_TERASOLUNA_FW_Web_Multipart_MultipartFileItemAccessor accesssor =
                    new NTTDATA_TERASOLUNA_FW_Web_Multipart_MultipartFileItemAccessor(target);
                accesssor._filename = null;

                // 実行及び検証
                Assert.IsNull(target.Filename);
            }
        }

        ///<summary>
        /// Filename のテスト
        ///</summary>
        ///<remarks>
        /// Filenameで空文字を取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartFileItem.txt")]
        public void FilenameTest02()
        {
            using (Stream partStream = new FileStream(FILE_PATH_SUCCESS, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList
                    = new Dictionary<string, string>();
                headerList.Add(
                    "CONTENT-DISPOSITION", "form-data; name=file1;filename=text.txt");
                MultipartFileItem target
                    = new MultipartFileItem(BOUNDARY, partStream, headerList, ENCODING);
                NTTDATA_TERASOLUNA_FW_Web_Multipart_MultipartFileItemAccessor accesssor =
                    new NTTDATA_TERASOLUNA_FW_Web_Multipart_MultipartFileItemAccessor(target);
                accesssor._filename = "";

                // 実行及び検証
                Assert.AreEqual("", target.Filename);
            }
        }

        ///<summary>
        /// Filename のテスト
        ///</summary>
        ///<remarks>
        /// Filenameで"abcde"を取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartFileItem.txt")]
        public void FilenameTest03()
        {
            using (Stream partStream = new FileStream(FILE_PATH_SUCCESS, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList
                    = new Dictionary<string, string>();
                headerList.Add(
                    "CONTENT-DISPOSITION", "form-data; name=file1;filename=text.txt");
                MultipartFileItem target
                    = new MultipartFileItem(BOUNDARY, partStream, headerList, ENCODING);
                NTTDATA_TERASOLUNA_FW_Web_Multipart_MultipartFileItemAccessor accesssor =
                    new NTTDATA_TERASOLUNA_FW_Web_Multipart_MultipartFileItemAccessor(target);
                accesssor._filename = "abcde";

                // 実行及び検証
                Assert.AreEqual("abcde", target.Filename);
            }
        }

        ///<summary>
        /// Filename のテスト
        ///</summary>
        ///<remarks>
        /// Filenameで"ＡＢＣＤＥ"を取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartFileItem.txt")]
        public void FilenameTest04()
        {
            using (Stream partStream = new FileStream(FILE_PATH_SUCCESS, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList
                    = new Dictionary<string, string>();
                headerList.Add(
                    "CONTENT-DISPOSITION", "form-data; name=file1;filename=text.txt");
                MultipartFileItem target
                    = new MultipartFileItem(BOUNDARY, partStream, headerList, ENCODING);
                NTTDATA_TERASOLUNA_FW_Web_Multipart_MultipartFileItemAccessor accesssor =
                    new NTTDATA_TERASOLUNA_FW_Web_Multipart_MultipartFileItemAccessor(target);
                accesssor._filename = "ＡＢＣＤＥ";

                // 実行及び検証定
                Assert.AreEqual("ＡＢＣＤＥ", target.Filename);
            }
        }

        ///<summary>
        /// TemporaryFilePath のテスト
        ///</summary>
        ///<remarks>
        /// TemporaryFilePathでnullを取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartFileItem.txt")]
        public void TemporaryFilePathTest01()
        {
            using (Stream partStream = new FileStream(FILE_PATH_SUCCESS, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList
                    = new Dictionary<string, string>();
                headerList.Add(
                    "CONTENT-DISPOSITION", "form-data; name=file1;filename=text.txt");
                MultipartFileItem target
                    = new MultipartFileItem(BOUNDARY, partStream, headerList, ENCODING);
                NTTDATA_TERASOLUNA_FW_Web_Multipart_MultipartFileItemAccessor accessor
                    = new NTTDATA_TERASOLUNA_FW_Web_Multipart_MultipartFileItemAccessor(target);
                string temporaryFilePath = accessor._temporaryFilePath;
                accessor._temporaryFilePath = null;

                // 実行及び検証
                Assert.IsNull(accessor.TemporaryFilePath);

                accessor._temporaryFilePath = temporaryFilePath;
            }
        }

        ///<summary>
        /// TemporaryFilePath のテスト
        ///</summary>
        ///<remarks>
        /// TemporaryFilePathで空文字を取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartFileItem.txt")]
        public void TemporaryFilePathTest02()
        {

            using (Stream partStream = new FileStream(FILE_PATH_SUCCESS, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList
                    = new Dictionary<string, string>();
                headerList.Add(
                    "CONTENT-DISPOSITION", "form-data; name=file1;filename=text.txt");
                MultipartFileItem target
                    = new MultipartFileItem(BOUNDARY, partStream, headerList, ENCODING);
                NTTDATA_TERASOLUNA_FW_Web_Multipart_MultipartFileItemAccessor accessor
                    = new NTTDATA_TERASOLUNA_FW_Web_Multipart_MultipartFileItemAccessor(target);
                string temporaryFilePath = accessor._temporaryFilePath;
                accessor._temporaryFilePath = "";

                // 実行及び検証
                Assert.AreEqual("", accessor.TemporaryFilePath);

                accessor._temporaryFilePath = temporaryFilePath;
            }

        }

        ///<summary>
        /// TemporaryFilePath のテスト
        ///</summary>
        ///<remarks>
        /// TemporaryFilePathで半角文字列"abcde"を取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartFileItem.txt")]
        public void TemporaryFilePathTest03()
        {
            using (Stream partStream = new FileStream(FILE_PATH_SUCCESS, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList
                    = new Dictionary<string, string>();
                headerList.Add(
                    "CONTENT-DISPOSITION", "form-data; name=file1;filename=text.txt");
                MultipartFileItem target
                    = new MultipartFileItem(BOUNDARY, partStream, headerList, ENCODING);
                NTTDATA_TERASOLUNA_FW_Web_Multipart_MultipartFileItemAccessor accessor
                    = new NTTDATA_TERASOLUNA_FW_Web_Multipart_MultipartFileItemAccessor(target);
                string temporaryFilePath = accessor._temporaryFilePath;
                accessor._temporaryFilePath = "abcde";

                // 実行及び検証
                Assert.AreEqual("abcde", accessor.TemporaryFilePath);
                accessor._temporaryFilePath = temporaryFilePath;
            }
        }

        ///<summary>
        /// TemporaryFilePath のテスト
        ///</summary>
        ///<remarks>
        /// TemporaryFilePathで半角文字列"ＡＢＣＤＥ"を取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartFileItem.txt")]
        public void TemporaryFilePathTest04()
        {
            using (Stream partStream = new FileStream(FILE_PATH_SUCCESS, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList
                    = new Dictionary<string, string>();
                headerList.Add(
                    "CONTENT-DISPOSITION", "form-data; name=file1;filename=text.txt");
                MultipartFileItem target
                    = new MultipartFileItem(BOUNDARY, partStream, headerList, ENCODING);
                NTTDATA_TERASOLUNA_FW_Web_Multipart_MultipartFileItemAccessor accessor
                    = new NTTDATA_TERASOLUNA_FW_Web_Multipart_MultipartFileItemAccessor(target);
                string temporaryFilePath = accessor._temporaryFilePath;
                accessor._temporaryFilePath = "ＡＢＣＤＥ";

                // 実行及び検証
                Assert.AreEqual("ＡＢＣＤＥ", accessor.TemporaryFilePath);
                accessor._temporaryFilePath = temporaryFilePath;
            }
        }

        ///<summary>
        /// IsText のテスト
        ///</summary>
        ///<remarks>
        /// IsTextでfalseを取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartFileItem.txt")]
        public void IsTextTest01()
        {
            using (Stream partStream = new FileStream(FILE_PATH_SUCCESS, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList
                    = new Dictionary<string, string>();
                headerList.Add(
                    "CONTENT-DISPOSITION", "form-data; name=file1;filename=text.txt");
                MultipartFileItem target
                    = new MultipartFileItem(BOUNDARY, partStream, headerList, ENCODING);

                // 実行及び検証
                Assert.IsFalse(target.IsText);
            }
        }

        ///<summary>
        /// OutputStream のテスト
        ///</summary>
        ///<remarks>
        /// OutputStreamでnullを取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartFileItem.txt")]
        public void OutputStreamTest01()
        {
            using (Stream partStream = new FileStream(FILE_PATH_SUCCESS, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList
                    = new Dictionary<string, string>();
                headerList.Add(
                    "CONTENT-DISPOSITION", "form-data; name=file1;filename=text.txt");
                MultipartFileItem target
                    = new MultipartFileItem(BOUNDARY, partStream, headerList, ENCODING);
                NTTDATA_TERASOLUNA_FW_Web_Multipart_MultipartFileItemAccessor accessor
                    = new NTTDATA_TERASOLUNA_FW_Web_Multipart_MultipartFileItemAccessor(target);
                Stream outputStream = accessor._outputStream;
                accessor._outputStream = null;

                // 実行及び検証
                Assert.IsNull(target.OutputStream);
                accessor._outputStream = outputStream;
            }
        }

        ///<summary>
        /// OutputStream のテスト
        ///</summary>
        ///<remarks>
        /// OutputStreamに任意の値を設定し、
        /// OutputStreamで任意の値を取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartFileItem.txt")]
        public void OutputStreamTest02()
        {
            using (Stream partStream = new FileStream(FILE_PATH_SUCCESS, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList
                    = new Dictionary<string, string>();
                headerList.Add(
                    "CONTENT-DISPOSITION", "form-data; name=file1;filename=text.txt");
                MultipartFileItem target
                    = new MultipartFileItem(BOUNDARY, partStream, headerList, ENCODING);
                NTTDATA_TERASOLUNA_FW_Web_Multipart_MultipartFileItemAccessor accessor
                    = new NTTDATA_TERASOLUNA_FW_Web_Multipart_MultipartFileItemAccessor(target);
                Stream outputStream = accessor._outputStream;
                using (Stream testStream = new MemoryStream())
                {
                    accessor._outputStream = testStream;

                    // 実行及び検証
                    Assert.AreSame(testStream, target.OutputStream);
                }
                accessor._outputStream = outputStream;
            }
        }

        ///<summary>
        /// Disposed のテスト
        ///</summary>
        ///<remarks>
        /// Disposedでfalseを設定・取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartFileItem.txt")]
        public void DisposedTest01()
        {
            using (Stream partStream = new FileStream(FILE_PATH_SUCCESS, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList
                    = new Dictionary<string, string>();
                headerList.Add(
                    "CONTENT-DISPOSITION", "form-data; name=file1;filename=text.txt");
                MultipartFileItem target
                    = new MultipartFileItem(BOUNDARY, partStream, headerList, ENCODING);
                NTTDATA_TERASOLUNA_FW_Web_Multipart_MultipartFileItemAccessor accessor
                    = new NTTDATA_TERASOLUNA_FW_Web_Multipart_MultipartFileItemAccessor(target);
                accessor.Disposed = false;

                // 実行及び検証
                Assert.IsFalse(accessor.Disposed);
            }
        }

        ///<summary>
        /// Disposed のテスト
        ///</summary>
        ///<remarks>
        /// Disposedでtrueを設定・取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartFileItem.txt")]
        public void DisposedTest02()
        {
            using (Stream partStream = new FileStream(FILE_PATH_SUCCESS, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList
                    = new Dictionary<string, string>();
                headerList.Add(
                    "CONTENT-DISPOSITION", "form-data; name=file1;filename=text.txt");
                MultipartFileItem target
                    = new MultipartFileItem(BOUNDARY, partStream, headerList, ENCODING);
                NTTDATA_TERASOLUNA_FW_Web_Multipart_MultipartFileItemAccessor accessor
                    = new NTTDATA_TERASOLUNA_FW_Web_Multipart_MultipartFileItemAccessor(target);
                accessor.Disposed = true;

                // 実行及び検証
                Assert.IsTrue(accessor.Disposed);
            }
        }

        ///<summary>
        /// Dispose のテスト
        ///</summary>
        ///<remarks>
        /// 一時ファイルパスに存在しないパスが設定している場合、
        /// 例外(FileNotFoundException)が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartFileItem.txt")]
        public void DisposeTest01()
        {
            using (Stream partStream = new FileStream(FILE_PATH_SUCCESS, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList
                    = new Dictionary<string, string>();
                headerList.Add(
                    "CONTENT-DISPOSITION", "form-data; name=file1;filename=text.txt");
                MultipartFileItem target
                    = new MultipartFileItem(BOUNDARY, partStream, headerList, ENCODING);
                NTTDATA_TERASOLUNA_FW_Web_Multipart_MultipartFileItemAccessor accessor
                    = new NTTDATA_TERASOLUNA_FW_Web_Multipart_MultipartFileItemAccessor(target);
                string temporaryFilePath = accessor._temporaryFilePath;
                accessor._temporaryFilePath = @"NotFile.txt";

                // 実行
                target.Dispose();

                // 検証
                Assert.IsTrue(accessor.Disposed);
            }
        }

        ///<summary>
        /// Dispose のテスト
        ///</summary>
        ///<remarks>
        /// 出力ストリームがオープンされている場合、
        /// 正常に動作することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartFileItem.txt")]
        public void DisposeTest02()
        {
            using (Stream partStream = new FileStream(FILE_PATH_SUCCESS, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList
                    = new Dictionary<string, string>();
                headerList.Add(
                    "CONTENT-DISPOSITION", "form-data; name=file1;filename=text.txt");
                MultipartFileItem target
                    = new MultipartFileItem(BOUNDARY, partStream, headerList, ENCODING);
                NTTDATA_TERASOLUNA_FW_Web_Multipart_MultipartFileItemAccessor accessor
                    = new NTTDATA_TERASOLUNA_FW_Web_Multipart_MultipartFileItemAccessor(target);

                // 実行
                target.Dispose();

                // 検証
                Assert.IsFalse(File.Exists(accessor._temporaryFilePath));
                Assert.IsTrue(accessor.Disposed);
            }
        }

        ///<summary>
        /// Dispose のテスト
        ///</summary>
        ///<remarks>
        /// 出力ストリームがオープンされていない場合、
        /// 正常に動作することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartFileItem.txt")]
        public void DisposeTest03()
        {
            using (Stream partStream = new FileStream(FILE_PATH_SUCCESS, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList
                    = new Dictionary<string, string>();
                headerList.Add(
                    "CONTENT-DISPOSITION", "form-data; name=file1;filename=text.txt");
                MultipartFileItem target
                    = new MultipartFileItem(BOUNDARY, partStream, headerList, ENCODING);
                NTTDATA_TERASOLUNA_FW_Web_Multipart_MultipartFileItemAccessor accessor
                    = new NTTDATA_TERASOLUNA_FW_Web_Multipart_MultipartFileItemAccessor(target);
                accessor._outputStream.Close();

                // 実行
                target.Dispose();

                // 検証
                Assert.IsFalse(File.Exists(accessor._temporaryFilePath));
                Assert.IsTrue(accessor.Disposed);
            }
        }

        ///<summary>
        /// Dispose のテスト
        ///</summary>
        ///<remarks>
        /// Disposeされた後にもう一度Disposeされた場合、
        /// 正常に動作することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartFileItem.txt")]
        public void DisposeTest04()
        {
            using (Stream partStream = new FileStream(FILE_PATH_SUCCESS, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList
                    = new Dictionary<string, string>();
                headerList.Add(
                    "CONTENT-DISPOSITION", "form-data; name=file1;filename=text.txt");
                MultipartFileItem target
                    = new MultipartFileItem(BOUNDARY, partStream, headerList, ENCODING);
                NTTDATA_TERASOLUNA_FW_Web_Multipart_MultipartFileItemAccessor accessor
                    = new NTTDATA_TERASOLUNA_FW_Web_Multipart_MultipartFileItemAccessor(target);
                accessor.Disposed = true;

                // 実行
                target.Dispose();

                // 検証
                Assert.IsTrue(File.Exists(accessor._temporaryFilePath));
                Assert.IsTrue(accessor.Disposed);

                // 一時ファイルの削除
                accessor.Disposed = false;
                target.Dispose();
            }
        }
    }
}
