// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Text;
using System.Collections.Generic;
using System.Xml;
using System.IO;
using System.Data;
using System.Web;
using TERASOLUNA.Fw.Common.Validation;
using TERASOLUNA.Fw.Web.Controller;
using TERASOLUNA.Fw.Common;
using TERASOLUNA.Fw.WebUT;
using TERASOLUNA.Fw.WebUT.Controller;
using TERASOLUNA.Fw.Utlib.Logging;
using TERASOLUNA.Fw.Common.Logging;
using System.Reflection;
namespace TERASOLUNA.Fw.WebUT.Controller
{
    /// <summary>
    ///TERASOLUNA.Fw.Web.Controller.BLogicRequestController のテスト クラスです。すべての
    ///TERASOLUNA.Fw.Web.Controller.BLogicRequestController 単体テストをここに含めます
    ///</summary>
    
    [TestClass()]
    public class BLogicRequestControllerTest
    {


        private TestContext testContextInstance;

        /// <summary>
        ///現在のテストの実行についての情報および機能を
        ///提供するテスト コンテキストを取得または設定します。
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }
        #region 追加のテスト属性
        // 
        //テストを作成するときに、次の追加属性を使用することができます:
        //
        //クラスの最初のテストを実行する前にコードを実行するには、ClassInitialize を使用
        //
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //クラスのすべてのテストを実行した後にコードを実行するには、ClassCleanup を使用
        //
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //各テストを実行する前にコードを実行するには、TestInitialize を使用
        //
        [TestInitialize()]
        public void MyTestInitialize()
        {
            LogUTUtil.Clear();
        }
        //
        //各テストを実行した後にコードを実行するには、TestCleanup を使用
        //
        //[TestCleanup()]
        //public void MyTestCleanup()
        //{
        //}
        //
        #endregion

        /// <summary>
        /// IsReusable のテスト
        /// </summary>
        /// <TestTarget>IsReusable</TestTarget>
        /// <TestCondition></TestCondition>
        /// <TestAssertion>trueが返却されること</TestAssertion>
        [TestMethod()]
        public void TestIsReusable01()
        {
            BLogicRequestController target = new BLogicRequestController();

            bool val = true;

            Assert.AreEqual(val, target.IsReusable);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(BLogicRequestController), LogLevel.INFO));
        }

        /// <summary>
        /// ValidateContentType (string, string) のテスト
        /// </summary>
        /// <TestTarget>ValidateContentType</TestTarget>
        /// <TestCondition>
        /// requestContentTypeに設定されたメディアタイプ値が正常であること
        /// requestContentTypeにcharsetが設定されていないこと
        /// requestContentTypeのメディアタイプがexpectedMediaTypeと一致していること
        /// </TestCondition>
        /// <TestAssertion>InvalidRequestExceptionが発生すること</TestAssertion>
        [TestMethod()]
        public void TestValidateContentType01()
        {
            BLogicRequestController target = new BLogicRequestController();

            TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_BLogicRequestControllerAccessor accessor = new TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_BLogicRequestControllerAccessor(target);

            string requestContentType = "text/xml";

            string expectedMediaType = "text/xml";

            Type expectedType = typeof(InvalidRequestException);

            try
            {
                accessor.ValidateContentType(requestContentType, expectedMediaType);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TargetInvocationException ex)
            {
                InvalidRequestException re = ex.InnerException as InvalidRequestException;
                Assert.AreEqual(typeof(InvalidRequestException).Name, re.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_INVALID_CONTENTTYPE_CHARSET_UTF8", ""), re.Message);

            }
            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(BLogicRequestController), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(
                    typeof(BLogicRequestController), typeof(InvalidRequestException), "E_INVALID_CONTENTTYPE_CHARSET_UTF8", ""));
        }

        /// <summary>
        /// ValidateContentType (string, string) のテスト
        /// </summary>
        /// <TestTarget>ValidateContentType</TestTarget>
        /// <TestCondition>
        /// requestContentTypeに設定された値が正常であること
        /// requestContentTypeにcharsetがutf-8で設定されていること
        /// requestContentTypeのメディアタイプがexpectedMediaTypeと一致していること
        /// </TestCondition>
        /// <TestAssertion>例外が発生しないこと</TestAssertion>
        [TestMethod()]
        public void TestValidateContentType02()
        {
            BLogicRequestController target = new BLogicRequestController();

            TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_BLogicRequestControllerAccessor accessor = new TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_BLogicRequestControllerAccessor(target);

            string requestContentType = "text/xml; charset=utf-8";

            string expectedMediaType = "text/xml";

            accessor.ValidateContentType(requestContentType, expectedMediaType);
            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(BLogicRequestController), LogLevel.INFO));
        }

        /// <summary>
        /// ValidateContentType (string, string) のテスト
        /// </summary>
        /// <TestTarget>ValidateContentType</TestTarget>
        /// <TestCondition>
        /// requestContentTypeに設定された値が正常であること
        /// requestContentTypeにcharsetがutf-8で設定されていること
        /// requestContentTypeのメディアタイプがexpectedMediaTypeと大小文字の違いだけで一致していること
        /// </TestCondition>
        /// <TestAssertion>例外が発生しないこと</TestAssertion>
        [TestMethod()]
        public void TestValidateContentType03()
        {
            BLogicRequestController target = new BLogicRequestController();

            TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_BLogicRequestControllerAccessor accessor = new TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_BLogicRequestControllerAccessor(target);

            string requestContentType = "Text/Xml; charset=utf-8";

            string expectedMediaType = "text/xml";

            accessor.ValidateContentType(requestContentType, expectedMediaType);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(BLogicRequestController), LogLevel.INFO));
        }

        /// <summary>
        /// ValidateContentType (string, string) のテスト
        /// </summary>
        /// <TestTarget>ValidateContentType</TestTarget>
        /// <TestCondition>
        /// requestContentTypeに設定された値のフォーマットが不正であること
        /// </TestCondition>
        /// <TestAssertion>InvalidRequestExceptionが発生すること</TestAssertion>
        [TestMethod()]
        public void TestValidateContentType04()
        {
            BLogicRequestController target = new BLogicRequestController();

            TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_BLogicRequestControllerAccessor accessor = new TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_BLogicRequestControllerAccessor(target);

            string requestContentType = "text/xml charset=utf-8";

            string expectedMediaType = "text/xml";

            Type expectedType = typeof(InvalidRequestException);

            try
            {
                accessor.ValidateContentType(requestContentType, expectedMediaType);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TargetInvocationException ex)
            {
                InvalidRequestException re = ex.InnerException as InvalidRequestException;
                Assert.AreEqual(typeof(InvalidRequestException).Name, re.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_INVALID_CONTENTTYPE_FORMAT", requestContentType), re.Message);

            }
            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(BLogicRequestController), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(
                    typeof(BLogicRequestController), typeof(InvalidRequestException), "E_INVALID_CONTENTTYPE_FORMAT", requestContentType));
        }

        /// <summary>
        /// ValidateContentType (string, string) のテスト
        /// </summary>
        /// <TestTarget>ValidateContentType</TestTarget>
        /// <TestCondition>
        /// requestContentTypeに設定された値が正常であること
        /// requestContentTypeのメディアタイプがexpectedMediaTypeが一致していない
        /// </TestCondition>
        /// <TestAssertion>InvalidRequestExceptionが発生すること</TestAssertion>
        [TestMethod()]
        public void TestValidateContentType05()
        {
            BLogicRequestController target = new BLogicRequestController();

            TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_BLogicRequestControllerAccessor accessor = new TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_BLogicRequestControllerAccessor(target);

            string requestContentType = "image/gif";

            string expectedMediaType = "text/xml";

            Type expectedType = typeof(InvalidRequestException);

            try
            {
                accessor.ValidateContentType(requestContentType, expectedMediaType);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TargetInvocationException ex)
            {
                InvalidRequestException re = ex.InnerException as InvalidRequestException;
                Assert.AreEqual(typeof(InvalidRequestException).Name, re.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_INVALID_CONTENTTYPE_MEDIATYPE", requestContentType, expectedMediaType), re.Message);

            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(BLogicRequestController), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(
                    typeof(BLogicRequestController), typeof(InvalidRequestException), "E_INVALID_CONTENTTYPE_MEDIATYPE", requestContentType, expectedMediaType));

        }

        /// <summary>
        /// ValidateContentType (string, string) のテスト
        /// </summary>
        /// <TestTarget>ValidateContentType</TestTarget>
        /// <TestCondition>
        /// requestContentTypeに設定された値が空文字であること
        /// </TestCondition>
        /// <TestAssertion>InvalidRequestExceptionが発生すること</TestAssertion>
        [TestMethod()]
        public void TestValidateContentType06()
        {
            BLogicRequestController target = new BLogicRequestController();

            TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_BLogicRequestControllerAccessor accessor = new TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_BLogicRequestControllerAccessor(target);

            string requestContentType = "";

            string expectedMediaType = "text/xml";

            Type expectedType = typeof(InvalidRequestException);

            try
            {
                accessor.ValidateContentType(requestContentType, expectedMediaType);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TargetInvocationException ex)
            {
                InvalidRequestException re = ex.InnerException as InvalidRequestException;
                Assert.AreEqual(typeof(InvalidRequestException).Name, re.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_REQUIRED_HEADER_EMPTY", "content-type"), re.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(BLogicRequestController), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(
                    typeof(BLogicRequestController), typeof(InvalidRequestException), "E_REQUIRED_HEADER_EMPTY", "content-type"));
        }

        /// <summary>
        /// ValidateContentType (string, string) のテスト
        /// </summary>
        /// <TestTarget>ValidateContentType</TestTarget>
        /// <TestCondition>
        /// requestContentTypeに設定された値がnullであること
        /// </TestCondition>
        /// <TestAssertion>InvalidRequestExceptionが発生すること</TestAssertion>
        [TestMethod()]
        public void TestValidateContentType07()
        {
            BLogicRequestController target = new BLogicRequestController();

            TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_BLogicRequestControllerAccessor accessor = new TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_BLogicRequestControllerAccessor(target);

            string requestContentType = null;

            string expectedMediaType = "text/xml";

            Type expectedType = typeof(InvalidRequestException);

            try
            {
                accessor.ValidateContentType(requestContentType, expectedMediaType);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TargetInvocationException ex)
            {
                InvalidRequestException re = ex.InnerException as InvalidRequestException;
                Assert.AreEqual(typeof(InvalidRequestException).Name, re.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_REQUIRED_HEADER_EMPTY", "content-type"), re.Message);

            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(BLogicRequestController), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(
                    typeof(BLogicRequestController), typeof(InvalidRequestException), "E_REQUIRED_HEADER_EMPTY", "content-type"));
        }

        /// <summary>
        /// ValidateContentType (string, string) のテスト
        /// </summary>
        /// <TestTarget>ValidateContentType</TestTarget>
        /// <TestCondition>
        /// requestContentTypeに正常な値が設定されていること
        /// requestContentTypeにcharsetがutf-8で設定されていること
        /// expectedMediaTypeに設定された値が空文字であること
        /// </TestCondition>
        /// <TestAssertion>InvalidRequestExceptionが発生すること</TestAssertion>
        [TestMethod()]
        public void TestValidateContentType08()
        {
            BLogicRequestController target = new BLogicRequestController();

            TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_BLogicRequestControllerAccessor accessor = new TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_BLogicRequestControllerAccessor(target);

            string requestContentType = "text/xml; charset=utf-8";

            string expectedMediaType = "";

            Type expectedType = typeof(InvalidRequestException);

            try
            {
                accessor.ValidateContentType(requestContentType, expectedMediaType);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TargetInvocationException ex)
            {
                InvalidRequestException re = ex.InnerException as InvalidRequestException;
                Assert.AreEqual(typeof(InvalidRequestException).Name, re.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_INVALID_CONTENTTYPE_MEDIATYPE", requestContentType, expectedMediaType), re.Message);

            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(BLogicRequestController), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(
                    typeof(BLogicRequestController), typeof(InvalidRequestException), "E_INVALID_CONTENTTYPE_MEDIATYPE", requestContentType, expectedMediaType));
        }

        /// <summary>
        /// ValidateContentType (string, string) のテスト
        /// </summary>
        /// <TestTarget>ValidateContentType</TestTarget>
        /// <TestCondition>
        /// expectedMediaTypeに設定された値がnullであること
        /// </TestCondition>
        /// <TestAssertion>NullReferenceExceptionが発生すること</TestAssertion>
        [TestMethod()]
        public void TestValidateContentType09()
        {
            BLogicRequestController target = new BLogicRequestController();

            TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_BLogicRequestControllerAccessor accessor = new TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_BLogicRequestControllerAccessor(target);

            string requestContentType = "text/xml; charset=utf-8";

            string expectedMediaType = null;

            Type expectedType = typeof(NullReferenceException);

            try
            {
                accessor.ValidateContentType(requestContentType, expectedMediaType);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TargetInvocationException ex)
            {
                NullReferenceException re = ex.InnerException as NullReferenceException;
                Assert.AreEqual(typeof(NullReferenceException).Name, re.GetType().Name);
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(BLogicRequestController), LogLevel.INFO));
            
        }

        /// <summary>
        /// ValidateContentType (string, string) のテスト
        /// </summary>
        /// <TestTarget>ValidateContentType</TestTarget>
        /// <TestCondition>
        /// requestContentTypeに設定されたメディアタイプ値が正常であること
        /// requestContentTypeにcharsetとしてutf-8以外が設定されている。
        /// requestContentTypeのメディアタイプがexpectedMediaTypeと一致していること
        /// </TestCondition>
        /// <TestAssertion>InvalidRequestExceptionが発生すること</TestAssertion>
        [TestMethod()]
        public void TestValidateContentType10()
        {
            BLogicRequestController target = new BLogicRequestController();

            TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_BLogicRequestControllerAccessor accessor = new TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_BLogicRequestControllerAccessor(target);

            string requestContentType = "text/xml; charset=utf-7";

            string expectedMediaType = "text/xml";

            Type expectedType = typeof(InvalidRequestException);

            try
            {
                accessor.ValidateContentType(requestContentType, expectedMediaType);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TargetInvocationException ex)
            {
                InvalidRequestException re = ex.InnerException as InvalidRequestException;
                Assert.AreEqual(typeof(InvalidRequestException).Name, re.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_INVALID_CONTENTTYPE_CHARSET_UTF8", "utf-7"), re.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(BLogicRequestController), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(BLogicRequestController), typeof(InvalidRequestException), "E_INVALID_CONTENTTYPE_CHARSET_UTF8", "utf-7"));
        }

        /// <summary>
        /// CreateInputDataSet (Type, Stream) のテスト
        /// </summary>
        /// <TestTarget>CreateInputDataSet</TestTarget>
        /// <TestCondition>
        /// dataSetTypeに型付DataSetを指定すること
        /// inputStreamに格納されているデータがdataSetTypeに指定された
        /// DataSetのスキーマと一致していること
        /// </TestCondition>
        /// <TestAssertion>
        /// dataSetTypeに指定した型で生成されること
        /// 生成されたDataSetがinputStreamで指定したデータの値で構成されていること
        /// </TestAssertion>
        [TestMethod()]
        public void TestCreateInputDataSet01()
        {
            BLogicRequestController target = new BLogicRequestController();

            TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_BLogicRequestControllerAccessor accessor
                = new TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_BLogicRequestControllerAccessor(target);

            Type dataSetType = typeof(TestDS_BLogicRequestController001);

            // 入力値となるストリームを初期化
            Stream inputStream = new MemoryStream();
            TestDS_BLogicRequestController001 expectedDs = new TestDS_BLogicRequestController001();

            expectedDs.TestTable01.AddTestTable01Row("aaa", 1);
            expectedDs.TestTable01.AddTestTable01Row("あああ", 2);
            expectedDs.TestTable02.AddTestTable02Row(true, 1.1);
            expectedDs.WriteXml(inputStream);
            inputStream.Position = 0;

            Type expectedType = typeof(TestDS_BLogicRequestController001);
            DataSet actual;

            actual = accessor.CreateInputDataSet(dataSetType, inputStream);

            // 想定通りのデータセットの型で生成されたか確認
            Assert.IsInstanceOfType(actual, expectedType);
            // DataSetの内容の確認
            Assert.AreEqual(expectedDs.GetXml(), actual.GetXml());

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(BLogicRequestController), LogLevel.INFO));
        }

        /// <summary>
        /// CreateInputDataSet (Type, Stream) のテスト
        /// </summary>
        /// <TestTarget>CreateInputDataSet</TestTarget>
        /// <TestCondition>
        /// dataSetTypeに型なしのDataSetを指定すること
        /// inputStreamに格納されているデータがXMLフォーマットであること
        /// </TestCondition>
        /// <TestAssertion>
        /// dataSetTypeに指定した型で生成されること
        /// 生成されたDataSetがinputStreamで指定したデータの値で構成されていること
        /// </TestAssertion>
        [TestMethod()]
        public void TestCreateInputDataSet02()
        {
            BLogicRequestController target = new BLogicRequestController();

            TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_BLogicRequestControllerAccessor accessor
                = new TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_BLogicRequestControllerAccessor(target);

            Type dataSetType = typeof(DataSet);

            // 入力値となるストリームを初期化
            Stream inputStream = new MemoryStream();
            DataSet expectedDs = new DataSet();

            DataTable dt = new DataTable("TestTable01");
            dt.Columns.Add("TestColumn01");
            dt.Columns.Add("TestColumn02");
            dt.Rows.Add("aaa", 1);
            dt.Rows.Add("あああ", 2);
            expectedDs.Tables.Add(dt);
            dt = new DataTable("TestTable02");
            dt.Columns.Add("TestColumn01");
            dt.Columns.Add("TestColumn02");
            dt.Rows.Add(true, 1.1);
            expectedDs.Tables.Add(dt);

            expectedDs.WriteXml(inputStream);
            inputStream.Position = 0;

            Type expectedType = typeof(DataSet);
            DataSet actual;

            actual = accessor.CreateInputDataSet(dataSetType, inputStream);

            // 想定通りのデータセットの型で生成されたか確認
            Assert.IsInstanceOfType(actual, expectedType);
            // DataSetの内容の確認
            Assert.AreEqual(expectedDs.GetXml(), actual.GetXml());

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(BLogicRequestController), LogLevel.INFO));
        }

        /// <summary>
        /// CreateInputDataSet (Type, Stream) のテスト
        /// </summary>
        /// <TestTarget>CreateInputDataSet</TestTarget>
        /// <TestCondition>
        /// dataSetTypeにDataSet、DataSetのサブクラスでない値を指定すること
        /// </TestCondition>
        /// <TestAssertion>TerasolunaExceptionがスローされること</TestAssertion>
        [TestMethod()]
        public void TestCreateInputDataSet03()
        {
            BLogicRequestController target = new BLogicRequestController();

            TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_BLogicRequestControllerAccessor accessor
                = new TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_BLogicRequestControllerAccessor(target);

            Type dataSetType = typeof(Byte);

            // 入力値となるストリームを初期化
            Stream inputStream = new MemoryStream();
            TestDS_BLogicRequestController001 expectedDs = new TestDS_BLogicRequestController001();

            expectedDs.TestTable01.AddTestTable01Row("aaa", 1);
            expectedDs.TestTable01.AddTestTable01Row("あああ", 2);
            expectedDs.TestTable02.AddTestTable02Row(true, 1.1);
            expectedDs.WriteXml(inputStream);
            inputStream.Position = 0;

            Type expectedType = typeof(TerasolunaException);
            DataSet actual;

            try
            {
                actual = accessor.CreateInputDataSet(dataSetType, inputStream);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TargetInvocationException ex)
            {
                TerasolunaException te = ex.InnerException as TerasolunaException;
                Assert.AreEqual(typeof(TerasolunaException).Name, te.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_INVALID_BASE_CLASS", dataSetType.FullName, typeof(DataSet).FullName), te.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(BLogicRequestController), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(
                    typeof(BLogicRequestController), typeof(TerasolunaException), "E_INVALID_BASE_CLASS", dataSetType.FullName, typeof(DataSet).FullName));
        }

        /// <summary>
        /// CreateInputDataSet (Type, Stream) のテスト
        /// </summary>
        /// <TestTarget>CreateInputDataSet</TestTarget>
        /// <TestCondition>
        /// dataSetTypeにデフォルトコンストラクタの存在しないDataSetのサブクラスのタイプを指定すること
        /// </TestCondition>
        /// <TestAssertion>TerasolunaExceptionがスローされること</TestAssertion>
        [TestMethod()]
        public void TestCreateInputDataSet04()
        {
            BLogicRequestController target = new BLogicRequestController();

            TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_BLogicRequestControllerAccessor accessor
                = new TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_BLogicRequestControllerAccessor(target);

            Type dataSetType = typeof(TestDS_BLogicRequestController002);

            // 入力値となるストリームを初期化
            Stream inputStream = new MemoryStream();
            DataSet expectedDs = new DataSet();

            DataTable dt = new DataTable("TestTable01");
            dt.Columns.Add("TestColumn01");
            dt.Columns.Add("TestColumn02");
            dt.Rows.Add("aaa", 1);
            dt.Rows.Add("あああ", 2);
            expectedDs.Tables.Add(dt);
            dt = new DataTable("TestTable02");
            dt.Columns.Add("TestColumn01");
            dt.Columns.Add("TestColumn02");
            dt.Rows.Add(true, 1.1);
            expectedDs.Tables.Add(dt);

            expectedDs.WriteXml(inputStream);
            inputStream.Position = 0;

            Type expectedType = typeof(TerasolunaException);
            DataSet actual;

            try
            {
                actual = accessor.CreateInputDataSet(dataSetType, inputStream);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TargetInvocationException ex)
            {
                TerasolunaException te = ex.InnerException as TerasolunaException;
                Assert.AreEqual(typeof(TerasolunaException).Name, te.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_INVALID_CONSTRUCTOR", dataSetType.FullName), te.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(BLogicRequestController), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(
                    typeof(BLogicRequestController), typeof(TerasolunaException), "E_INVALID_CONSTRUCTOR", dataSetType.FullName));
        }

        /// <summary>
        /// CreateInputDataSet (Type, Stream) のテスト
        /// </summary>
        /// <TestTarget>CreateInputDataSet</TestTarget>
        /// <TestCondition>
        /// dataSetTypeにデフォルトコンストラクタで例外が発生する
        /// DataSetのサブクラスのタイプを指定すること
        /// </TestCondition>
        /// <TestAssertion>TerasolunaExceptionがスローされること</TestAssertion>
        [TestMethod()]
        public void TestCreateInputDataSet05()
        {
            BLogicRequestController target = new BLogicRequestController();

            TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_BLogicRequestControllerAccessor accessor
                = new TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_BLogicRequestControllerAccessor(target);

            Type dataSetType = typeof(TestDS_BLogicRequestController003);

            // 入力値となるストリームを初期化
            Stream inputStream = new MemoryStream();
            DataSet expectedDs = new DataSet();

            DataTable dt = new DataTable("TestTable01");
            dt.Columns.Add("TestColumn01");
            dt.Columns.Add("TestColumn02");
            dt.Rows.Add("aaa", 1);
            dt.Rows.Add("あああ", 2);
            expectedDs.Tables.Add(dt);
            dt = new DataTable("TestTable02");
            dt.Columns.Add("TestColumn01");
            dt.Columns.Add("TestColumn02");
            dt.Rows.Add(true, 1.1);
            expectedDs.Tables.Add(dt);

            expectedDs.WriteXml(inputStream);
            inputStream.Position = 0;

            Type expectedType = typeof(TerasolunaException);
            DataSet actual;

            try
            {
                actual = accessor.CreateInputDataSet(dataSetType, inputStream);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TargetInvocationException ex)
            {
                TerasolunaException te = ex.InnerException as TerasolunaException;
                Assert.AreEqual(typeof(TerasolunaException).Name, te.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_INVALID_CONSTRUCTOR", dataSetType.FullName), te.Message);
            }
            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(BLogicRequestController), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(
                    typeof(BLogicRequestController), typeof(TerasolunaException), "E_INVALID_CONSTRUCTOR", dataSetType.FullName));
        }

        /// <summary>
        /// CreateInputDataSet (Type, Stream) のテスト
        /// </summary>
        /// <TestTarget>CreateInputDataSet</TestTarget>
        /// <TestCondition>
        /// dataSetTypeにnullを指定すること
        /// </TestCondition>
        /// <TestAssertion>ArgumentNullExceptionがスローされること</TestAssertion>
        [TestMethod()]
        public void TestCreateInputDataSet06()
        {
            BLogicRequestController target = new BLogicRequestController();

            TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_BLogicRequestControllerAccessor accessor
                = new TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_BLogicRequestControllerAccessor(target);

            Type dataSetType = null;

            // 入力値となるストリームを初期化
            Stream inputStream = new MemoryStream();
            TestDS_BLogicRequestController001 expectedDs = new TestDS_BLogicRequestController001();

            expectedDs.TestTable01.AddTestTable01Row("aaa", 1);
            expectedDs.TestTable01.AddTestTable01Row("あああ", 2);
            expectedDs.TestTable02.AddTestTable02Row(true, 1.1);
            expectedDs.WriteXml(inputStream);
            inputStream.Position = 0;

            Type expectedType = typeof(ArgumentNullException);
            DataSet actual;

            try
            {
                actual = accessor.CreateInputDataSet(dataSetType, inputStream);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TargetInvocationException ex)
            {
                ArgumentNullException ae = ex.InnerException as ArgumentNullException;
                Assert.AreEqual(typeof(ArgumentNullException).Name, ae.GetType().Name);
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(BLogicRequestController), LogLevel.INFO));
        }

        /// <summary>
        /// CreateInputDataSet (Type, Stream) のテスト
        /// </summary>
        /// <TestTarget>CreateInputDataSet</TestTarget>
        /// <TestCondition>
        /// inputStreamにXML形式以外の値を設定すること
        /// </TestCondition>
        /// <TestAssertion>InvalidRequestExceptionがスローされること</TestAssertion>
        [TestMethod()]
        public void TestCreateInputDataSet07()
        {
            BLogicRequestController target = new BLogicRequestController();

            TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_BLogicRequestControllerAccessor accessor
                = new TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_BLogicRequestControllerAccessor(target);

            Type dataSetType = typeof(TestDS_BLogicRequestController001);

            // 入力値となるストリームを初期化
            Stream inputStream = new MemoryStream();
            string inputStr = "aaaaaa";
            byte[] byteArray = Encoding.UTF8.GetBytes(inputStr);
            inputStream.Write(byteArray, 0, byteArray.Length);
            inputStream.Position = 0;

            Type expectedType = typeof(InvalidRequestException);
            DataSet actual;

            string exceptionMessage = null;

            try
            {
                actual = accessor.CreateInputDataSet(dataSetType, inputStream);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TargetInvocationException ex)
            {
                InvalidRequestException exception = ex.InnerException as InvalidRequestException;
                XmlException fe = exception.InnerException as XmlException;
                exceptionMessage = fe.Message;
                Assert.AreEqual(typeof(InvalidRequestException).Name, exception.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_INVALID_REQUEST") + exceptionMessage, exception.Message);
            }
            
            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(BLogicRequestController), LogLevel.INFO));
            string logMessage = MessageManager.GetMessage("E_INVALID_REQUEST") + exceptionMessage;
            Assert.IsTrue(LogUTUtil.CheckError(typeof(BLogicRequestController), typeof(InvalidRequestException),logMessage));
        }

        /// <summary>
        /// CreateInputDataSet (Type, Stream) のテスト
        /// </summary>
        /// <TestTarget>CreateInputDataSet</TestTarget>
        /// <TestCondition>
        /// inputStreamにUTF-8以外でエンコーディングした値を設定すること
        /// </TestCondition>
        /// <TestAssertion>InvalidRequestExceptionがスローされること</TestAssertion>
        [TestMethod()]
        public void TestCreateInputDataSet08()
        {
            BLogicRequestController target = new BLogicRequestController();

            TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_BLogicRequestControllerAccessor accessor
                = new TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_BLogicRequestControllerAccessor(target);

            Type dataSetType = typeof(TestDS_BLogicRequestController001);

            // 入力値となるストリームを初期化
            Stream inputStream = new MemoryStream();
            TestDS_BLogicRequestController001 expectedDs = new TestDS_BLogicRequestController001();

            expectedDs.TestTable01.AddTestTable01Row("aaa", 1);
            expectedDs.TestTable01.AddTestTable01Row("あああ", 2);
            expectedDs.TestTable02.AddTestTable02Row(true, 1.1);
            string inputStr = expectedDs.GetXml();
            byte[] byteArray = Encoding.UTF7.GetBytes(inputStr);
            inputStream.Write(byteArray, 0, byteArray.Length);
            inputStream.Position = 0;

            Type expectedType = typeof(InvalidRequestException);
            DataSet actual;

            string exceptionMessage = null;

            try
            {
                actual = accessor.CreateInputDataSet(dataSetType, inputStream);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TargetInvocationException ex)
            {
                InvalidRequestException exception = ex.InnerException as InvalidRequestException;
                XmlException fe = exception.InnerException as XmlException;
                exceptionMessage = fe.Message;
                Assert.AreEqual(typeof(InvalidRequestException).Name, exception.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_INVALID_REQUEST")+exceptionMessage, exception.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(BLogicRequestController), LogLevel.INFO));
            string logMessage = MessageManager.GetMessage("E_INVALID_REQUEST")+exceptionMessage;
            Assert.IsTrue(LogUTUtil.CheckError(typeof(BLogicRequestController), typeof(InvalidRequestException), logMessage));
        }

        /// <summary>
        /// CreateInputDataSet (Type, Stream) のテスト
        /// </summary>
        /// <TestTarget>CreateInputDataSet</TestTarget>
        /// <TestCondition>
        /// inputStreamに対象となるDataSetのスキーマに合致しない値を設定すること
        /// </TestCondition>
        /// <TestAssertion>InvalidRequestExceptionがスローされること</TestAssertion>
        [TestMethod()]
        public void TestCreateInputDataSet09()
        {
            BLogicRequestController target = new BLogicRequestController();

            TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_BLogicRequestControllerAccessor accessor
                = new TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_BLogicRequestControllerAccessor(target);

            Type dataSetType = typeof(TestDS_BLogicRequestController001);

            // 入力値となるストリームを初期化
            Stream inputStream = new MemoryStream();
            TestDS_BLogicRequestController001 ds = new TestDS_BLogicRequestController001();
            DataSet expectedDs = new DataSet("TestDS_BLogicRequestController001");
            expectedDs.Namespace = ds.Namespace;
            DataTable dt = new DataTable("TestTable01");
            dt.Columns.Add("TestColumn01");
            dt.Columns.Add("TestColumn02");
            dt.Rows.Add("aaa", 1.1);
            dt.Rows.Add("あああ", 2);
            expectedDs.Tables.Add(dt);
            dt = new DataTable("TestTable02");
            dt.Columns.Add("TestColumn01");
            dt.Columns.Add("TestColumn02");
            dt.Rows.Add(true, 1.1);
            expectedDs.Tables.Add(dt);

            expectedDs.WriteXml(inputStream);
            inputStream.Position = 0;

            Type expectedType = typeof(InvalidRequestException);
            DataSet actual;
            string exceptionMessage = null;

            try
            {
                actual = accessor.CreateInputDataSet(dataSetType, inputStream);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TargetInvocationException ex)
            {
                InvalidRequestException exception = ex.InnerException as InvalidRequestException;
                FormatException fe = exception.InnerException as FormatException;
                exceptionMessage = fe.Message;
                Assert.AreEqual(typeof(InvalidRequestException).Name, exception.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_INVALID_REQUEST") + exceptionMessage, exception.Message);
             }
              
            // ログの確認
            
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(BLogicRequestController), LogLevel.INFO));
            string logMessage = MessageManager.GetMessage("E_INVALID_REQUEST") + exceptionMessage;
            Assert.IsTrue(LogUTUtil.CheckError(typeof(BLogicRequestController), typeof(InvalidRequestException), logMessage));

        }

        /// <summary>
        /// CreateInputDataSet (Type, Stream) のテスト
        /// </summary>
        /// <TestTarget>CreateInputDataSet</TestTarget>
        /// <TestCondition>
        /// inputStreamにnullを設定すること
        /// </TestCondition>
        /// <TestAssertion>値が設定されていないDataSetが生成されること</TestAssertion>
        [TestMethod()]
        public void TestCreateInputDataSet10()
        {
            BLogicRequestController target = new BLogicRequestController();

            TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_BLogicRequestControllerAccessor accessor
                = new TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_BLogicRequestControllerAccessor(target);

            Type dataSetType = typeof(TestDS_BLogicRequestController001);

            // 入力値となるストリームを初期化
            Stream inputStream = null;

            Type expectedType = typeof(TestDS_BLogicRequestController001);
            DataSet expectedDs = new TestDS_BLogicRequestController001();
            DataSet actual;

            actual = accessor.CreateInputDataSet(dataSetType, inputStream);

            // DataSetの型の確認
            Assert.IsInstanceOfType(actual, expectedType);
            // DataSetの内容の確認
            Assert.AreEqual(expectedDs.GetXml(), actual.GetXml());

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(BLogicRequestController), LogLevel.INFO));
        }

        /// <summary>
        /// CreateInitializeErrorDocument () のテスト
        /// </summary>
        /// <TestTarget>CreateInitializeErrorDocument</TestTarget>
        /// <TestCondition></TestCondition>
        /// <TestAssertion><![CDATA[<errors></errors>]]>であるXmlDocumentが取得できること</TestAssertion>
        [TestMethod()]
        public void TestCreateInitializeErrorDocument01()
        {
            BLogicRequestController target = new BLogicRequestController();

            TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_BLogicRequestControllerAccessor accessor
                = new TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Controller_BLogicRequestControllerAccessor(target);

            Type expectedType = typeof(XmlDocument);
            string expectedContent = "<errors></errors>";
            XmlDocument actual;

            actual = accessor.CreateInitializeErrorDocument();

            Assert.IsInstanceOfType(actual, expectedType);
            Assert.AreEqual(expectedContent, actual.OuterXml);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(BLogicRequestController), LogLevel.INFO));
        }
    }
}
