// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Text;
using System.Collections.Generic;
using System.Net;
using System.Data;
using TERASOLUNA.Fw.Client.Communication;
using TERASOLUNA.Fw.Client;
using TERASOLUNA.Fw.Utlib.Logging;
using TERASOLUNA.Fw.Common.Logging;
using System.Xml;
using System.IO;
using TERASOLUNA.Fw.Common;
using TERASOLUNA.Fw.ClientUT.Communication.DataSetXmlReceiverData;
using TERASOLUNA.Fw.ClientUT.Communication.DataSetXmlSenderData;
using System.Net.Security;
using System.Security.Cryptography.X509Certificates;
namespace TERASOLUNA.Fw.ClientUT.Communication
{
    /// <summary>
    ///TERASOLUNA.Fw.Client.Communication.Sender.DataSetXmlSender のテスト クラスです。すべての
    ///TERASOLUNA.Fw.Client.Communication.Sender.DataSetXmlSender 単体テストをここに含めます
    ///</summary>
    [TestClass()]
    public class DataSetXmlSenderTest
    {

        private TestContext testContextInstance;

        ///<summary>
        /// 正常な通信先を表すURL文字列です。
        ///</summary>
        ///<remarks>
        ///  <para>
        ///   この定数の値は "http://localhost/TestServer/Index.aspx" です。
        ///  </para>
        ///</remarks>
        private static readonly string TEST_NORMAL_URL = "https://10.68.100.216/HttpsTestServer/Index.aspx";

        ///<summary>
        /// 存在しないサーバを表すURL文字列です。
        ///</summary>
        ///<remarks>
        ///  <para>
        ///   この定数の値は "http://errorserver/TestServer/Index.aspx" です。
        ///  </para>
        ///</remarks>
        private static readonly string TEST_NO_SERVER_URL = "http://errorserver/DummyServer/Index.aspx";

        /// <summary>
        /// 通信相手に通知するリクエスト名のキー値です。
        /// </summary>
        /// <remarks>この定数の値は "RequestName" です。</remarks>
        private static readonly string REQUEST_NAME = "RequestName";

        /// <summary>
        ///現在のテストの実行についての情報および機能を
        ///提供するテスト コンテキストを取得または設定します。
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }
        #region 追加のテスト属性
        // 
        //テストを作成するときに、次の追加属性を使用することができます:
        //
        //クラスの最初のテストを実行する前にコードを実行するには、ClassInitialize を使用
        //
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //クラスのすべてのテストを実行した後にコードを実行するには、ClassCleanup を使用
        //
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //各テストを実行する前にコードを実行するには、TestInitialize を使用
        //
        [TestInitialize()]
        public void MyTestInitialize()
        {
            LogUTUtil.Clear();
            ServicePointManager.ServerCertificateValidationCallback = new RemoteCertificateValidationCallback(OnRemoteCertificateValidationCallback);
        }
        //
        //各テストを実行した後にコードを実行するには、TestCleanup を使用
        //
        //[TestCleanup()]
        //public void MyTestCleanup()
        //{
        //}
        //
        #endregion


        /// <summary>
        ///ContentType のテスト
        ///</summary>
        [TestMethod()]
        public void ContentTypeTest()
        {
            DataSetXmlSender target = new DataSetXmlSender();

            // 定数CONTENT_TYPEを変更するので退避しておく
            string bakContentType = TERASOLUNA_Fw_Client_Communication_DataSetXmlSenderAccessor.CONTENT_TYPE;

            TERASOLUNA_Fw_Client_Communication_DataSetXmlSenderAccessor accessor = new TERASOLUNA_Fw_Client_Communication_DataSetXmlSenderAccessor(target);

            string val = "TERASOLUNA";

            TERASOLUNA_Fw_Client_Communication_DataSetXmlSenderAccessor.CONTENT_TYPE = val;

            Assert.AreEqual(val, target.ContentType);

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<DataSet>), LogLevel.INFO));

            // 定数CONTENT_TYPEをバックアップから復元する
            TERASOLUNA_Fw_Client_Communication_DataSetXmlSenderAccessor.CONTENT_TYPE = bakContentType;
        }
        
        /// <summary>
        ///XmlWriteMode のテスト
        ///</summary>
        [TestMethod()]
        public void XmlWriteModeTest()
        {
            //準備
            DataSetXmlSender target = new DataSetXmlSender();
            XmlWriteMode val = XmlWriteMode.WriteSchema;

            //実行
            target.WriteMode = val;

            //検証
            Assert.AreEqual(val, target.WriteMode);

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlSender), LogLevel.INFO));
        }

        /// <summary>
        ///Send (HttpWebRequest, DataSet, IDictionary&lt;string,string&gt;, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// requestがnullの場合、ArgumentNullExceptionがスローされることの確認。
        ///</remarks>
        [TestMethod()]
        public void SendTest01()
        {
            DataSetXmlSender target = new DataSetXmlSender();

            HttpWebRequest request = null;

            // データセット
            DataSetXmlSenderDataSet01 paramData = new DataSetXmlSenderDataSet01();
            DataSetXmlSenderDataSet01.DataTable1Row row = paramData.DataTable1.NewDataTable1Row();
            row.Column1 = "テラソルナ";
            paramData.DataTable1.AddDataTable1Row(row);            

            // ヘッダ
            System.Collections.Generic.IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add(REQUEST_NAME, "DataSetXmlSender_Success01");

            // レポーター
            DataSetXmlSender_IProgressChangeReporterStub01 reporter = new DataSetXmlSender_IProgressChangeReporterStub01();

            // プロパティ
            target.BufferSize = 8192;

            try
            {
                // 実行
                target.Send(request, paramData, headerCollection, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // メッセージ確認
                Assert.AreEqual("request", e.ParamName);
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(SenderBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlSender), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(SenderBase<DataSet>), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "request"));
        }

        /// <summary>
        ///Send (HttpWebRequest, DataSet, IDictionary&lt;string,string&gt;, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// リクエストがキャンセルされた場合、WebExceptionがスローされることの確認。
        ///</remarks>
        [TestMethod()]
        public void SendTest02()
        {
            DataSetXmlSender target = new DataSetXmlSender();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // キャンセルする
            request.Abort();

            // データセット
            DataSetXmlSenderDataSet01 paramData = new DataSetXmlSenderDataSet01();
            DataSetXmlSenderDataSet01.DataTable1Row row = paramData.DataTable1.NewDataTable1Row();
            row.Column1 = "テラソルナ";
            paramData.DataTable1.AddDataTable1Row(row);

            // ヘッダ
            System.Collections.Generic.IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add(REQUEST_NAME, "DataSetXmlSender_Success01");

            // レポーター
            DataSetXmlSender_IProgressChangeReporterStub01 reporter = new DataSetXmlSender_IProgressChangeReporterStub01();

            // プロパティ
            target.BufferSize = 8192;

            try
            {
                // 実行
                target.Send(request, paramData, headerCollection, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (WebException)
            {
                // 成功
            }

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlSender), LogLevel.INFO));
        }

        /// <summary>
        ///Send (HttpWebRequest, DataSet, IDictionary&lt;string,string&gt;, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        ///paramDataがnullの場合、ArgumentNullExceptionがスローされることの確認。
        ///</remarks>
        [TestMethod()]
        public void SendTest03()
        {
            DataSetXmlSender target = new DataSetXmlSender();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // データセット
            DataSetXmlSenderDataSet01 paramData = null;

            // ヘッダ
            System.Collections.Generic.IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add(REQUEST_NAME, "DataSetXmlSender_Success01");

            // レポーター
            DataSetXmlSender_IProgressChangeReporterStub01 reporter = new DataSetXmlSender_IProgressChangeReporterStub01();

            // プロパティ
            target.BufferSize = 8192;
            try
            {
                // 実行
                target.Send(request, paramData, headerCollection, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // メッセージ確認
                Assert.AreEqual("paramData", e.ParamName);
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(SenderBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlSender), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(SenderBase<DataSet>), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "paramData"));
        }

        /// <summary>
        ///Send (HttpWebRequest, DataSet, IDictionary&lt;string,string&gt;, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        ///型なしデータセットの値が正しくサーバに送信されることの確認。
        ///名前空間：あり
        ///DataTableの要素数：1
        ///DataColumnの要素数：1
        ///headerCollectionの要素数：1
        ///reporter：not null
        ///BufferSize：3
        ///</remarks>
        [TestMethod()]
        public void SendTest04()
        {
            DataSetXmlSender target = new DataSetXmlSender();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // データセット
            DataSet dataSet = new DataSet("DataSetXmlSenderDataSet01");
            DataTable dataTable = new DataTable("DataTable1");
            DataColumn dataColumn = new DataColumn("Column1");
            dataTable.Columns.Add(dataColumn);
            dataSet.Tables.Add(dataTable);
            DataRow row = dataTable.NewRow();
            row["Column1"] = "テラソルナ01";
            dataTable.Rows.Add(row);

            // ヘッダ
            System.Collections.Generic.IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add(REQUEST_NAME, "DataSetXmlSender_Success01");

            // レポーター
            DataSetXmlSender_IProgressChangeReporterStub01 reporter = new DataSetXmlSender_IProgressChangeReporterStub01();

            // プロパティ
            target.BufferSize = 8192;

            // 実行
            target.Send(request, dataSet, headerCollection, reporter);

            // 確認
            WebResponse response = request.GetResponse();

            // 成功レスポンスが返却されることの確認
            DataSet resultData = new DataSet();
            resultData.ReadXml(response.GetResponseStream());
            Assert.AreEqual("true", resultData.Tables["SuccessTable"].Rows[0]["SuccessValue"]);

            // イベント実行確認
            Assert.IsTrue(reporter._reportProgressChanged);

            // 電文ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(3, LogUTUtil.CountLogs(typeof(DataSetXmlSender), LogLevel.TRACE));
            Assert.IsTrue(CheckRequestLogData(request, dataSet));
            
        }

        /// <summary>
        ///Send (HttpWebRequest, DataSet, IDictionary&lt;string,string&gt;, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        ///型付きデータセットの値が正しくサーバに送信されることの確認。
        ///名前空間：あり
        ///DataTableの要素数：1
        ///DataColumnの要素数：1
        ///headerCollectionの要素数：1
        ///reporter：not null
        ///BufferSize：3
        ///</remarks>
        [TestMethod()]
        public void SendTest05()
        {
            DataSetXmlSender target = new DataSetXmlSender();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // データセット
            DataSetXmlSenderDataSet01 paramData = new DataSetXmlSenderDataSet01();
            DataSetXmlSenderDataSet01.DataTable1Row row = paramData.DataTable1.NewDataTable1Row();
            row.Column1 = "テラソルナ01";
            paramData.DataTable1.AddDataTable1Row(row);

            // ヘッダ
            System.Collections.Generic.IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add(REQUEST_NAME, "DataSetXmlSender_Success02");

            // レポーター
            DataSetXmlSender_IProgressChangeReporterStub01 reporter = new DataSetXmlSender_IProgressChangeReporterStub01();

            // プロパティ
            target.BufferSize = 8192;

            // 実行
            target.Send(request, paramData, headerCollection, reporter);
            WebResponse response = request.GetResponse();

            // 成功レスポンスが返却されることの確認
            DataSet resultData = new DataSet();
            resultData.ReadXml(response.GetResponseStream());
            Assert.AreEqual("true", resultData.Tables["SuccessTable"].Rows[0]["SuccessValue"]);

            // イベント実行確認
            Assert.IsTrue(reporter._reportProgressChanged);

            // 電文ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(3, LogUTUtil.CountLogs(typeof(DataSetXmlSender), LogLevel.TRACE));
            Assert.IsTrue(CheckRequestLogData(request, paramData));
        }

        /// <summary>
        ///Send (HttpWebRequest, DataSet, IDictionary&lt;string,string&gt;, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        ///型付きデータセットの値が正しくサーバに送信されることの確認。
        ///名前空間：なし
        ///DataTableの要素数：1
        ///DataColumnの要素数：1
        ///headerCollectionの要素数：1
        ///reporter：not null
        ///BufferSize：3
        ///</remarks>
        [TestMethod()]
        public void SendTest06()
        {
            DataSetXmlSender target = new DataSetXmlSender();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // データセット
            DataSetXmlSenderDataSet02 paramData = new DataSetXmlSenderDataSet02();
            DataSetXmlSenderDataSet02.DataTable1Row row = paramData.DataTable1.NewDataTable1Row();
            row.Column1 = "テラソルナ01";
            paramData.DataTable1.AddDataTable1Row(row);

            // ヘッダ
            System.Collections.Generic.IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add(REQUEST_NAME, "DataSetXmlSender_Success03");

            // レポーター
            DataSetXmlSender_IProgressChangeReporterStub01 reporter = new DataSetXmlSender_IProgressChangeReporterStub01();

            // プロパティ
            target.BufferSize = 8192;

            // 実行
            target.Send(request, paramData, headerCollection, reporter);
            WebResponse response = request.GetResponse();

            // 成功レスポンスが返却されることの確認
            DataSet resultData = new DataSet();
            resultData.ReadXml(response.GetResponseStream());
            Assert.AreEqual("true", resultData.Tables["SuccessTable"].Rows[0]["SuccessValue"]);

            // イベント実行確認
            Assert.IsTrue(reporter._reportProgressChanged);

            // 電文ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(3, LogUTUtil.CountLogs(typeof(DataSetXmlSender), LogLevel.TRACE));
            Assert.IsTrue(CheckRequestLogData(request, paramData));
        }

        /// <summary>
        ///Send (HttpWebRequest, DataSet, IDictionary&lt;string,string&gt;, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        ///型付きデータセットの値が正しくサーバに送信されることの確認。
        ///名前空間：あり
        ///DataTableの要素数：0
        ///headerCollectionの要素数：1
        ///reporter：not null
        ///BufferSize：3
        ///</remarks>
        [TestMethod()]
        public void SendTest07()
        {
            DataSetXmlSender target = new DataSetXmlSender();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // データセット
            DataSetXmlSenderDataSet01 paramData = new DataSetXmlSenderDataSet01();

            // ヘッダ
            System.Collections.Generic.IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add(REQUEST_NAME, "DataSetXmlSender_Success04");

            // レポーター
            DataSetXmlSender_IProgressChangeReporterStub01 reporter = new DataSetXmlSender_IProgressChangeReporterStub01();

            // プロパティ
            target.BufferSize = 8192;
            
            // 実行
            target.Send(request, paramData, headerCollection, reporter);
            WebResponse response = request.GetResponse();

            // 成功レスポンスが返却されることの確認
            DataSet resultData = new DataSet();
            resultData.ReadXml(response.GetResponseStream());
            Assert.AreEqual("true", resultData.Tables["SuccessTable"].Rows[0]["SuccessValue"]);

            // イベント実行確認
            Assert.IsTrue(reporter._reportProgressChanged);

            // 電文ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(3, LogUTUtil.CountLogs(typeof(DataSetXmlSender), LogLevel.TRACE));
            Assert.IsTrue(CheckRequestLogData(request, paramData));
        }

        /// <summary>
        ///Send (HttpWebRequest, DataSet, IDictionary&lt;string,string&gt;, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        ///型付きデータセットの値が正しくサーバに送信されることの確認。
        ///名前空間：あり
        ///DataTableの要素数：3
        ///DataColumnの要素数：1
        ///headerCollectionの要素数：1
        ///reporter：not null
        ///BufferSize：3
        ///</remarks>
        [TestMethod()]
        public void SendTest08()
        {
            DataSetXmlSender target = new DataSetXmlSender();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // データセット
            DataSetXmlSenderDataSet01 paramData = new DataSetXmlSenderDataSet01();
            DataSetXmlSenderDataSet01.DataTable1Row row1 = paramData.DataTable1.NewDataTable1Row();
            row1.Column1 = "テラソルナ01";
            paramData.DataTable1.AddDataTable1Row(row1);
            DataSetXmlSenderDataSet01.DataTable2Row row2 = paramData.DataTable2.NewDataTable2Row();
            row2.Column2 = "テラソルナ02";
            paramData.DataTable2.AddDataTable2Row(row2);
            DataSetXmlSenderDataSet01.DataTable3Row row3 = paramData.DataTable3.NewDataTable3Row();
            row3.Column3 = "テラソルナ03";
            paramData.DataTable3.AddDataTable3Row(row3);

            // ヘッダ
            System.Collections.Generic.IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add(REQUEST_NAME, "DataSetXmlSender_Success15");

            // レポーター
            DataSetXmlSender_IProgressChangeReporterStub01 reporter = new DataSetXmlSender_IProgressChangeReporterStub01();

            // プロパティ
            target.BufferSize = 8192;

            // 実行
            target.Send(request, paramData, headerCollection, reporter);
            WebResponse response = request.GetResponse();

            // 成功レスポンスが返却されることの確認
            DataSet resultData = new DataSet();
            resultData.ReadXml(response.GetResponseStream());
            Assert.AreEqual("true", resultData.Tables["SuccessTable"].Rows[0]["SuccessValue"]);

            // イベント実行確認
            Assert.IsTrue(reporter._reportProgressChanged);

            // 電文ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(3, LogUTUtil.CountLogs(typeof(DataSetXmlSender), LogLevel.TRACE));
            Assert.IsTrue(CheckRequestLogData(request, paramData));
        }

        /// <summary>
        ///Send (HttpWebRequest, DataSet, IDictionary&lt;string,string&gt;, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        ///型付きデータセットの値が正しくサーバに送信されることの確認。
        ///名前空間：あり
        ///DataTableの要素数：1
        ///DataColumnの要素数：0
        ///headerCollectionの要素数：1
        ///reporter：not null
        ///BufferSize：3
        ///</remarks>
        [TestMethod()]
        public void SendTest09()
        {
            DataSetXmlSender target = new DataSetXmlSender();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // データセット
            DataSetXmlSenderDataSet01 paramData = new DataSetXmlSenderDataSet01();
            DataSetXmlSenderDataSet01.DataTable1Row row = paramData.DataTable1.NewDataTable1Row();
            paramData.DataTable1.AddDataTable1Row(row);

            // ヘッダ
            System.Collections.Generic.IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add(REQUEST_NAME, "DataSetXmlSender_Success06");

            // レポーター
            DataSetXmlSender_IProgressChangeReporterStub01 reporter = new DataSetXmlSender_IProgressChangeReporterStub01();

            // プロパティ
            target.BufferSize = 8192;

            // 実行
            target.Send(request, paramData, headerCollection, reporter);
            WebResponse response = request.GetResponse();

            // 成功レスポンスが返却されることの確認
            DataSet resultData = new DataSet();
            resultData.ReadXml(response.GetResponseStream());
            Assert.AreEqual("true", resultData.Tables["SuccessTable"].Rows[0]["SuccessValue"]);

            // イベント実行確認
            Assert.IsTrue(reporter._reportProgressChanged);

            // 電文ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(3, LogUTUtil.CountLogs(typeof(DataSetXmlSender), LogLevel.TRACE));
            Assert.IsTrue(CheckRequestLogData(request, paramData));
        }

        /// <summary>
        ///Send (HttpWebRequest, DataSet, IDictionary&lt;string,string&gt;, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        ///型付きデータセットの値が正しくサーバに送信されることの確認。
        ///名前空間：あり
        ///DataTableの要素数：1
        ///DataColumnの要素数：3
        ///headerCollectionの要素数：1
        ///reporter：not null
        ///BufferSize：3
        ///</remarks>
        [TestMethod()]
        public void SendTest10()
        {
            DataSetXmlSender target = new DataSetXmlSender();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // データセット
            DataSetXmlSenderDataSet01 paramData = new DataSetXmlSenderDataSet01();
            DataSetXmlSenderDataSet01.DataTable1Row row = paramData.DataTable1.NewDataTable1Row();
            row.Column1 = "テラソルナ01";
            row.Column2 = "テラソルナ02";
            row.Column3 = "テラソルナ03";
            paramData.DataTable1.AddDataTable1Row(row);

            // ヘッダ
            System.Collections.Generic.IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add(REQUEST_NAME, "DataSetXmlSender_Success07");

            // レポーター
            DataSetXmlSender_IProgressChangeReporterStub01 reporter = new DataSetXmlSender_IProgressChangeReporterStub01();

            // プロパティ
            target.BufferSize = 8192;

            // 実行
            target.Send(request, paramData, headerCollection, reporter);
            WebResponse response = request.GetResponse();

            // 成功レスポンスが返却されることの確認
            DataSet resultData = new DataSet();
            resultData.ReadXml(response.GetResponseStream());
            Assert.AreEqual("true", resultData.Tables["SuccessTable"].Rows[0]["SuccessValue"]);

            // イベント実行確認
            Assert.IsTrue(reporter._reportProgressChanged);

            // 電文ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(3, LogUTUtil.CountLogs(typeof(DataSetXmlSender), LogLevel.TRACE));
            Assert.IsTrue(CheckRequestLogData(request, paramData));
        }

        /// <summary>
        ///Send (HttpWebRequest, DataSet, IDictionary&lt;string,string&gt;, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        ///型付きデータセットの値が正しくサーバに送信されることの確認。
        ///名前空間：あり
        ///DataTableの要素数：1
        ///DataColumnの要素数：1
        ///DataColumnの内容：DBNUllが格納される
        ///headerCollectionの要素数：1
        ///reporter：not null
        ///BufferSize：3
        ///</remarks>
        [TestMethod()]
        public void SendTest11()
        {
            DataSetXmlSender target = new DataSetXmlSender();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // データセット
            DataSetXmlSenderDataSet01 paramData = new DataSetXmlSenderDataSet01();
            DataSetXmlSenderDataSet01.DataTable1Row row = paramData.DataTable1.NewDataTable1Row();
            row.Column1 = null;
            paramData.DataTable1.AddDataTable1Row(row);

            // ヘッダ
            System.Collections.Generic.IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add(REQUEST_NAME, "DataSetXmlSender_Success08");

            // レポーター
            DataSetXmlSender_IProgressChangeReporterStub01 reporter = new DataSetXmlSender_IProgressChangeReporterStub01();

            // プロパティ
            target.BufferSize = 8192;

            // 実行
            target.Send(request, paramData, headerCollection, reporter);
            WebResponse response = request.GetResponse();

            // 成功レスポンスが返却されることの確認
            DataSet resultData = new DataSet();
            resultData.ReadXml(response.GetResponseStream());
            Assert.AreEqual("true", resultData.Tables["SuccessTable"].Rows[0]["SuccessValue"]);

            // イベント実行確認
            Assert.IsTrue(reporter._reportProgressChanged);

            // 電文ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(3, LogUTUtil.CountLogs(typeof(DataSetXmlSender), LogLevel.TRACE));
            Assert.IsTrue(CheckRequestLogData(request, paramData));
        }


        /// <summary>
        ///Send (HttpWebRequest, DataSet, IDictionary&lt;string,string&gt;, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        ///型付きデータセットの値が正しくサーバに送信されることの確認。
        ///名前空間：あり
        ///DataTableの要素数：1
        ///DataColumnの要素数：1
        ///DataColumnの内容：空文字列が格納される
        ///headerCollectionの要素数：1
        ///reporter：not null
        ///BufferSize：3
        ///</remarks>
        [TestMethod()]
        public void SendTest12()
        {
            DataSetXmlSender target = new DataSetXmlSender();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // データセット
            DataSetXmlSenderDataSet01 paramData = new DataSetXmlSenderDataSet01();
            DataSetXmlSenderDataSet01.DataTable1Row row = paramData.DataTable1.NewDataTable1Row();
            row.Column1 = "";
            paramData.DataTable1.AddDataTable1Row(row);

            // ヘッダ
            System.Collections.Generic.IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add(REQUEST_NAME, "DataSetXmlSender_Success09");

            // レポーター
            DataSetXmlSender_IProgressChangeReporterStub01 reporter = new DataSetXmlSender_IProgressChangeReporterStub01();

            // プロパティ
            target.BufferSize = 8192;

            // 実行
            target.Send(request, paramData, headerCollection, reporter);
            WebResponse response = request.GetResponse();

            // 成功レスポンスが返却されることの確認
            DataSet resultData = new DataSet();
            resultData.ReadXml(response.GetResponseStream());
            Assert.AreEqual("true", resultData.Tables["SuccessTable"].Rows[0]["SuccessValue"]);

            // イベント実行確認
            Assert.IsTrue(reporter._reportProgressChanged);

            // 電文ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(3, LogUTUtil.CountLogs(typeof(DataSetXmlSender), LogLevel.TRACE));
            Assert.IsTrue(CheckRequestLogData(request, paramData));
        }

        /// <summary>
        ///Send (HttpWebRequest, DataSet, IDictionary&lt;string,string&gt;, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        ///型付きデータセットの値が正しくサーバに送信されることの確認。
        ///名前空間：あり
        ///DataTableの要素数：1
        ///DataColumnの要素数：1
        ///headerCollection：null
        ///reporter：not null
        ///BufferSize：3
        ///</remarks>
        [TestMethod()]
        public void SendTest13()
        {
            DataSetXmlSender target = new DataSetXmlSender();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // データセット
            DataSetXmlSenderDataSet01 paramData = new DataSetXmlSenderDataSet01();
            DataSetXmlSenderDataSet01.DataTable1Row row = paramData.DataTable1.NewDataTable1Row();
            row.Column1 = "テラソルナ01";
            paramData.DataTable1.AddDataTable1Row(row);

            // ヘッダ
            System.Collections.Generic.IDictionary<string, string> headerCollection = null;

            // レポーター
            DataSetXmlSender_IProgressChangeReporterStub01 reporter = new DataSetXmlSender_IProgressChangeReporterStub01();

            // プロパティ
            target.BufferSize = 8192;

            try
            {
                // 実行
                target.Send(request, paramData, headerCollection, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // メッセージ確認
                Assert.AreEqual("headerCollection", e.ParamName);
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(SenderBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlSender), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(SenderBase<DataSet>), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "headerCollection"));
        }

        /// <summary>
        ///Send (HttpWebRequest, DataSet, IDictionary&lt;string,string&gt;, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        ///型付きデータセットの値が正しくサーバに送信されることの確認。
        ///名前空間：あり
        ///DataTableの要素数：1
        ///DataColumnの要素数：1
        ///headerCollectionの要素数：0
        ///reporter：not null
        ///BufferSize：3
        ///</remarks>
        [TestMethod()]
        public void SendTest14()
        {
            DataSetXmlSender target = new DataSetXmlSender();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // データセット
            DataSetXmlSenderDataSet01 paramData = new DataSetXmlSenderDataSet01();
            DataSetXmlSenderDataSet01.DataTable1Row row = paramData.DataTable1.NewDataTable1Row();
            row.Column1 = "テラソルナ01";
            paramData.DataTable1.AddDataTable1Row(row);

            // ヘッダ
            System.Collections.Generic.IDictionary<string, string> headerCollection = new Dictionary<string, string>();

            // レポーター
            DataSetXmlSender_IProgressChangeReporterStub01 reporter = new DataSetXmlSender_IProgressChangeReporterStub01();

            // プロパティ
            target.BufferSize = 8192;

            // 実行
            target.Send(request, paramData, headerCollection, reporter);
            WebResponse response = request.GetResponse();

            // 成功レスポンスが返却されることの確認
            DataSet resultData = new DataSet();
            resultData.ReadXml(response.GetResponseStream());
            Assert.AreEqual("ERROR", resultData.Tables["error"].Rows[0]["error-code"] as string);
            Assert.AreEqual("リクエストヘッダにリクエスト名が設定されていません。", resultData.Tables["error"].Rows[0]["error-message"] as string);

            // イベント実行確認
            Assert.IsTrue(reporter._reportProgressChanged);

            // 電文ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(3, LogUTUtil.CountLogs(typeof(DataSetXmlSender), LogLevel.TRACE));
            Assert.IsTrue(CheckRequestLogData(request, paramData));
        }

        /// <summary>
        ///Send (HttpWebRequest, DataSet, IDictionary&lt;string,string&gt;, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        ///型付きデータセットの値が正しくサーバに送信されることの確認。
        ///名前空間：あり
        ///DataTableの要素数：1
        ///DataColumnの要素数：1
        ///headerCollectionの要素数：3
        ///reporter：not null
        ///BufferSize：3
        ///</remarks>
        [TestMethod()]
        public void SendTest15()
        {
            DataSetXmlSender target = new DataSetXmlSender();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // データセット
            DataSetXmlSenderDataSet01 paramData = new DataSetXmlSenderDataSet01();
            DataSetXmlSenderDataSet01.DataTable1Row row = paramData.DataTable1.NewDataTable1Row();
            row.Column1 = "テラソルナ01";
            paramData.DataTable1.AddDataTable1Row(row);

            // ヘッダ
            System.Collections.Generic.IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add(REQUEST_NAME, "DataSetXmlSender_Success10");
            headerCollection.Add("KEY01", "VALUE01");
            headerCollection.Add("KEY02", "VALUE02");

            // レポーター
            DataSetXmlSender_IProgressChangeReporterStub01 reporter = new DataSetXmlSender_IProgressChangeReporterStub01();

            // プロパティ
            target.BufferSize = 8192;

            // 実行
            target.Send(request, paramData, headerCollection, reporter);
            WebResponse response = request.GetResponse();

            // 成功レスポンスが返却されることの確認
            DataSet resultData = new DataSet();
            resultData.ReadXml(response.GetResponseStream());
            Assert.AreEqual("true", resultData.Tables["SuccessTable"].Rows[0]["SuccessValue"]);

            // イベント実行確認
            Assert.IsTrue(reporter._reportProgressChanged);

            // 電文ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(3, LogUTUtil.CountLogs(typeof(DataSetXmlSender), LogLevel.TRACE));
            Assert.IsTrue(CheckRequestLogData(request, paramData));
        }

        /// <summary>
        ///Send (HttpWebRequest, DataSet, IDictionary&lt;string,string&gt;, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        ///型付きデータセットの値が正しくサーバに送信されることの確認。
        ///名前空間：あり
        ///DataTableの要素数：1
        ///DataColumnの要素数：1
        ///headerCollectionの要素数：1
        ///headerCollectionのキー：空文字列
        ///reporter：not null
        ///BufferSize：3
        ///</remarks>
        [TestMethod()]
        public void SendTest16()
        {
            DataSetXmlSender target = new DataSetXmlSender();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // データセット
            DataSetXmlSenderDataSet01 paramData = new DataSetXmlSenderDataSet01();
            DataSetXmlSenderDataSet01.DataTable1Row row = paramData.DataTable1.NewDataTable1Row();
            row.Column1 = "テラソルナ01";
            paramData.DataTable1.AddDataTable1Row(row);

            // ヘッダ
            System.Collections.Generic.IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add(REQUEST_NAME, "DataSetXmlSender_Success02");
            headerCollection.Add("", "VALUE01");

            // レポーター
            DataSetXmlSender_IProgressChangeReporterStub01 reporter = new DataSetXmlSender_IProgressChangeReporterStub01();

            // プロパティ
            target.BufferSize = 8192;

            // 実行
            try
            {
                target.Send(request, paramData, headerCollection, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException)
            {
                // 成功
            }

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlSender), LogLevel.INFO));
        }

        /// <summary>
        ///Send (HttpWebRequest, DataSet, IDictionary&lt;string,string&gt;, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        ///型付きデータセットの値が正しくサーバに送信されることの確認。
        ///名前空間：あり
        ///DataTableの要素数：1
        ///DataColumnの要素数：1
        ///headerCollectionの要素数：1
        ///headerCollectionの値：null
        ///reporter：not null
        ///BufferSize：3
        ///</remarks>
        [TestMethod()]
        public void SendTest17()
        {
            DataSetXmlSender target = new DataSetXmlSender();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // データセット
            DataSetXmlSenderDataSet01 paramData = new DataSetXmlSenderDataSet01();
            DataSetXmlSenderDataSet01.DataTable1Row row = paramData.DataTable1.NewDataTable1Row();
            row.Column1 = "テラソルナ01";
            paramData.DataTable1.AddDataTable1Row(row);

            // ヘッダ
            System.Collections.Generic.IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add(REQUEST_NAME, "DataSetXmlSender_Success12");
            headerCollection.Add("KEY01", null);

            // レポーター
            DataSetXmlSender_IProgressChangeReporterStub01 reporter = new DataSetXmlSender_IProgressChangeReporterStub01();

            // プロパティ
            target.BufferSize = 8192;

            // 実行
            target.Send(request, paramData, headerCollection, reporter);
            WebResponse response = request.GetResponse();

            // 成功レスポンスが返却されることの確認
            DataSet resultData = new DataSet();
            resultData.ReadXml(response.GetResponseStream());
            Assert.AreEqual("true", resultData.Tables["SuccessTable"].Rows[0]["SuccessValue"]);

            // イベント実行確認
            Assert.IsTrue(reporter._reportProgressChanged);

            // 電文ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(3, LogUTUtil.CountLogs(typeof(DataSetXmlSender), LogLevel.TRACE));
            Assert.IsTrue(CheckRequestLogData(request, paramData));
        }

        /// <summary>
        ///Send (HttpWebRequest, DataSet, IDictionary&lt;string,string&gt;, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        ///型付きデータセットの値が正しくサーバに送信されることの確認。
        ///名前空間：あり
        ///DataTableの要素数：1
        ///DataColumnの要素数：1
        ///headerCollectionの要素数：1
        ///headerCollectionの値：空文字列
        ///reporter：not null
        ///BufferSize：3
        ///</remarks>
        [TestMethod()]
        public void SendTest18()
        {
            DataSetXmlSender target = new DataSetXmlSender();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // データセット
            DataSetXmlSenderDataSet01 paramData = new DataSetXmlSenderDataSet01();
            DataSetXmlSenderDataSet01.DataTable1Row row = paramData.DataTable1.NewDataTable1Row();
            row.Column1 = "テラソルナ01";
            paramData.DataTable1.AddDataTable1Row(row);

            // ヘッダ
            System.Collections.Generic.IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add(REQUEST_NAME, "DataSetXmlSender_Success13");
            headerCollection.Add("KEY01", "");

            // レポーター
            DataSetXmlSender_IProgressChangeReporterStub01 reporter = new DataSetXmlSender_IProgressChangeReporterStub01();

            // プロパティ
            target.BufferSize = 8192;

            // 実行
            target.Send(request, paramData, headerCollection, reporter);
            WebResponse response = request.GetResponse();

            // 成功レスポンスが返却されることの確認
            DataSet resultData = new DataSet();
            resultData.ReadXml(response.GetResponseStream());
            Assert.AreEqual("true", resultData.Tables["SuccessTable"].Rows[0]["SuccessValue"]);

            // イベント実行確認
            Assert.IsTrue(reporter._reportProgressChanged);

            // 電文ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(3, LogUTUtil.CountLogs(typeof(DataSetXmlSender), LogLevel.TRACE));
            Assert.IsTrue(CheckRequestLogData(request, paramData));
        }

        /// <summary>
        ///Send (HttpWebRequest, DataSet, IDictionary&lt;string,string&gt;, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        ///型付きデータセットの値が正しくサーバに送信されることの確認。
        ///名前空間：あり
        ///DataTableの要素数：1
        ///DataColumnの要素数：1
        ///headerCollectionの要素数：1
        ///headerCollectionのキー：空文字列
        ///reporter：null
        ///BufferSize：3
        ///</remarks>
        [TestMethod()]
        public void SendTest19()
        {
            DataSetXmlSender target = new DataSetXmlSender();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // データセット
            DataSetXmlSenderDataSet01 paramData = new DataSetXmlSenderDataSet01();
            DataSetXmlSenderDataSet01.DataTable1Row row = paramData.DataTable1.NewDataTable1Row();
            row.Column1 = "テラソルナ01";
            paramData.DataTable1.AddDataTable1Row(row);

            // ヘッダ
            System.Collections.Generic.IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add(REQUEST_NAME, "DataSetXmlSender_Success02");

            // プロパティ
            target.BufferSize = 8192;

            // 実行
            target.Send(request, paramData, headerCollection, null);
            WebResponse response = request.GetResponse();

            // 成功レスポンスが返却されることの確認
            DataSet resultData = new DataSet();
            resultData.ReadXml(response.GetResponseStream());
            Assert.AreEqual("true", resultData.Tables["SuccessTable"].Rows[0]["SuccessValue"]);

            // 電文ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(3, LogUTUtil.CountLogs(typeof(DataSetXmlSender), LogLevel.TRACE));
            Assert.IsTrue(CheckRequestLogData(request, paramData));
        }

        /// <summary>
        ///Send (HttpWebRequest, DataSet, IDictionary&lt;string,string&gt;, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        ///型付きデータセットの値が正しくサーバに送信されることの確認。
        ///名前空間：あり
        ///DataTableの要素数：1
        ///DataColumnの要素数：1
        ///headerCollectionの要素数：1
        ///headerCollectionのキー：空文字列
        ///reporter：not null
        ///BufferSize：1
        ///</remarks>
        [TestMethod()]
        public void SendTest20()
        {
            DataSetXmlSender target = new DataSetXmlSender();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // データセット
            DataSetXmlSenderDataSet01 paramData = new DataSetXmlSenderDataSet01();
            DataSetXmlSenderDataSet01.DataTable1Row row = paramData.DataTable1.NewDataTable1Row();
            row.Column1 = "テラソルナ01";
            paramData.DataTable1.AddDataTable1Row(row);

            // ヘッダ
            System.Collections.Generic.IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add(REQUEST_NAME, "DataSetXmlSender_Success02");

            // レポーター
            DataSetXmlSender_IProgressChangeReporterStub01 reporter = new DataSetXmlSender_IProgressChangeReporterStub01();

            // プロパティ
            target.BufferSize = 1;

            // 実行
            target.Send(request, paramData, headerCollection, reporter);
            WebResponse response = request.GetResponse();

            // 成功レスポンスが返却されることの確認
            DataSet resultData = new DataSet();
            resultData.ReadXml(response.GetResponseStream());
            Assert.AreEqual("true", resultData.Tables["SuccessTable"].Rows[0]["SuccessValue"]);

            // イベント実行確認
            Assert.IsTrue(reporter._reportProgressChanged);

            // 電文ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(3, LogUTUtil.CountLogs(typeof(DataSetXmlSender), LogLevel.TRACE));
            Assert.IsTrue(CheckRequestLogData(request, paramData));
        }

        /// <summary>
        ///Send (HttpWebRequest, DataSet, IDictionary&lt;string,string&gt;, IProgressChangeReporter) のテスト
        ///</summary>
        //[TestMethod()]
        //public void SendTest21()
        //{
        //    DataSetXmlSender target = new DataSetXmlSender();

        //    HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

        //    // データセット
        //    DataSetXmlSenderDataSet01 paramData = new DataSetXmlSenderDataSet01();
        //    DataSetXmlSenderDataSet01.DataTable1Row row = paramData.DataTable1.NewDataTable1Row();
        //    row.Column1 = "テラソルナ01";
        //    paramData.DataTable1.AddDataTable1Row(row);

        //    // ヘッダ
        //    System.Collections.Generic.IDictionary<string, string> headerCollection = new Dictionary<string, string>();
        //    headerCollection.Add(REQUEST_NAME, "DataSetXmlSender_Success14");

        //    // プロパティ
        //    target.BufferSize = 8192;
        //    target.WriteMode = XmlWriteMode.DiffGram;

        //    // 実行
        //    target.Send(request, paramData, headerCollection, this);
        //    WebResponse response = request.GetResponse();

        //    // 成功レスポンスが返却されることの確認
        //    Assert.IsTrue(GetSuccessResponse(response));

        //    // イベント実行確認
        //    Assert.AreEqual((GetDataSetXmlStreamLength(paramData, target.WriteMode) - 1) / target.BufferSize + 1, this.reportProgressChangedCount);

        //    // ログ確認
        //    Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<DataSet>), LogLevel.INFO));
        //}

        /// <summary>
        ///Send (HttpWebRequest, DataSet, IDictionary&lt;string,string&gt;, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        ///型付きデータセットの値が正しくサーバに送信されることの確認。
        ///名前空間：あり
        ///DataTableの要素数：1
        ///DataColumnの要素数：1
        ///headerCollectionの要素数：3
        ///headerCollectionの内容：「Method」と「Content-Type」をキーを持つ要素が存在する。
        ///reporter：not null
        ///BufferSize：3
        ///</remarks>
        [TestMethod()]
        public void SendTest22()
        {
            DataSetXmlSender target = new DataSetXmlSender();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // データセット
            DataSetXmlSenderDataSet01 paramData = new DataSetXmlSenderDataSet01();
            DataSetXmlSenderDataSet01.DataTable1Row row = paramData.DataTable1.NewDataTable1Row();
            row.Column1 = "テラソルナ01";
            paramData.DataTable1.AddDataTable1Row(row);

            // ヘッダ
            System.Collections.Generic.IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add(REQUEST_NAME, "DataSetXmlSender_Success02");
            headerCollection.Add("KEY01", "VALUE01");
            headerCollection.Add("Content-Type", "text/xml;charset=utf-8");

            // レポーター
            DataSetXmlSender_IProgressChangeReporterStub01 reporter = new DataSetXmlSender_IProgressChangeReporterStub01();

            // プロパティ
            target.BufferSize = 8192;

            // 実行
            try
            {
                target.Send(request, paramData, headerCollection, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException)
            {
                // 成功
            }

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlSender), LogLevel.INFO));
        }

        /// <summary>
        ///Send (HttpWebRequest, DataSet, IDictionary&lt;string,string&gt;, IProgressChangeReporter) のテスト
        ///</summary>
        [TestMethod()]
        public void SendTest23()
        {
            DataSetXmlSender target = new DataSetXmlSender();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NO_SERVER_URL);

            // データセット
            DataSetXmlSenderDataSet01 paramData = new DataSetXmlSenderDataSet01();
            DataSetXmlSenderDataSet01.DataTable1Row row = paramData.DataTable1.NewDataTable1Row();
            row.Column1 = "TERASOLUNA";

            // レポーター
            DataSetXmlSender_IProgressChangeReporterStub01 reporter = new DataSetXmlSender_IProgressChangeReporterStub01();

            // ヘッダ
            System.Collections.Generic.IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add(REQUEST_NAME, "DataSetXmlSender_Success01");

            try
            {
                // 実行
                target.Send(request, paramData, headerCollection, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (WebException)
            {
                // 成功
            }

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlSender), LogLevel.INFO));
        }


        /// <summary>
        ///Send (HttpWebRequest, DataSet, IDictionary&lt;string,string&gt;, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        ///型付きデータセットの値が正しくサーバに送信されることの確認。
        ///名前空間：あり
        ///DataTableの要素数：1
        ///DataTableの行数：3
        ///DataColumnの要素数：1
        ///headerCollectionの要素数：1
        ///reporter：not null
        ///BufferSize：3
        ///</remarks>
        [TestMethod()]
        public void SendTest24()
        {
            DataSetXmlSender target = new DataSetXmlSender();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // データセット
            DataSetXmlSenderDataSet01 paramData = new DataSetXmlSenderDataSet01();
            DataSetXmlSenderDataSet01.DataTable1Row row1 = paramData.DataTable1.NewDataTable1Row();
            row1.Column1 = "テラソルナ01";
            paramData.DataTable1.AddDataTable1Row(row1);
            DataSetXmlSenderDataSet01.DataTable1Row row2 = paramData.DataTable1.NewDataTable1Row();
            row2.Column2 = "テラソルナ02";
            paramData.DataTable1.AddDataTable1Row(row2);
            DataSetXmlSenderDataSet01.DataTable1Row row3 = paramData.DataTable1.NewDataTable1Row();
            row3.Column3 = "テラソルナ03";
            paramData.DataTable1.AddDataTable1Row(row3);

            // ヘッダ
            System.Collections.Generic.IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add(REQUEST_NAME, "DataSetXmlSender_Success05");

            // レポーター
            DataSetXmlSender_IProgressChangeReporterStub01 reporter = new DataSetXmlSender_IProgressChangeReporterStub01();

            // プロパティ
            target.BufferSize = 8192;

            // 実行
            target.Send(request, paramData, headerCollection, reporter);
            WebResponse response = request.GetResponse();

            // 成功レスポンスが返却されることの確認
            DataSet resultData = new DataSet();
            resultData.ReadXml(response.GetResponseStream());
            Assert.AreEqual("true", resultData.Tables["SuccessTable"].Rows[0]["SuccessValue"]);

            // イベント実行確認
            Assert.IsTrue(reporter._reportProgressChanged);

            // 電文ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(3, LogUTUtil.CountLogs(typeof(DataSetXmlSender), LogLevel.TRACE));
            Assert.IsTrue(CheckRequestLogData(request, paramData));
        }

        /// <summary>
        /// DataSetから生成されるXMLのサイズ(バイト)を返却する。
        /// </summary>
        /// <returns>XMLのバイトサイズ</returns>
        private int GetDataSetXmlStreamLength(DataSet dataSet, XmlWriteMode mode){
            MemoryStream memoryStream = new MemoryStream();
            dataSet.WriteXml(memoryStream, mode);
            return (int)memoryStream.Length;
        }

        /// <summary>
        /// 電文ログの出力チェックを行い、成功時にtrueを返却するメソッド。
        /// </summary>
        /// <param name="request">リクエスト</param>
        /// <param name="paramData">データセット</param>
        /// <returns>成功の場合true</returns>
        private bool CheckRequestLogData(HttpWebRequest request, DataSet paramData)
        {
            // URLのチェック
            if (!LogUTUtil.CheckTraceUseMessageId(typeof(DataSetXmlSender), "T_REQUEST_SEND_ADDRESS", request.Address.ToString()))
            {
                return false;
            }

            // HTTPヘッダのチェック
            StringBuilder requestHeaders = new StringBuilder();
            requestHeaders.AppendLine(MessageManager.GetMessage("T_REQUEST_SEND_HEADER"));
            foreach (string key in request.Headers.AllKeys)
            {
                requestHeaders.AppendLine(MessageManager.GetMessage("T_DICTIONARY_KEY_VALUE", key, request.Headers[key]));
            }

            if (!LogUTUtil.CheckTrace(typeof(DataSetXmlSender), requestHeaders.ToString().Trim()))
            {
                return false;
            }

            // HTTPボディのチェック
            StringBuilder requestBody = new StringBuilder();
            requestBody.AppendLine(MessageManager.GetMessage("T_REQUEST_SEND_BODY"));
            requestBody.Append(paramData.GetXml());

            if (!LogUTUtil.CheckTrace(typeof(DataSetXmlSender), requestBody.ToString()))
            {
                return false;
            }

            return true;
        }

        private bool OnRemoteCertificateValidationCallback(
            Object sender,
            X509Certificate certificate,
            X509Chain chain,
            SslPolicyErrors sslPolicyErrors)
        {
            // 信用したことにする
            return true;
        }


    }

}
