﻿// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using System;
using System.Configuration;
using TERASOLUNA.Fw.Common.Logging;

namespace TERASOLUNA.Fw.Client.Configuration.Conversion
{
    /// <summary>
    /// データセット変換設定ファイル内の clear-view 要素のコレクションを格納する構成要素を表します。
    /// </summary>
    /// <remarks>
    /// <see cref="ClearViewCollection"/> は、
    /// result 要素内の clear-view 要素のコレクションを表します。 
    /// </remarks>
    public class ClearViewCollection : ConfigurationElementCollection
    {
        /// <summary>
        /// <see cref="ILog"/> 実装クラスのインスタンスです。
        /// </summary>
        /// <remarks>
        /// ログ出力に利用します。
        /// </remarks>
        private static ILog _log = LogFactory.GetLogger(typeof(ClearViewCollection));

        /// <summary>
        /// clear-view 要素の要素名です。
        /// </summary>
        /// <remarks>
        /// この定数の値は "clear-view" です。
        /// </remarks>
        private static readonly string ELEMENT_NAME = "clear-view";

        /// <summary>
        /// <see cref="ClearViewCollection"/> クラスの新しいインスタンスを初期化します。
        /// </summary>
        /// <remarks>
        /// デフォルトコンストラクタです。
        /// </remarks>
        public ClearViewCollection()
        {
        }

        /// <summary>
        /// <see cref="ClearViewCollection"/> の型を取得します。
        /// </summary>
        /// <remarks>
        /// このコレクションの <see cref="ConfigurationElementCollectionType"/> は BasicMap です。
        /// </remarks>
        /// <value>
        /// このコレクションの <see cref="ConfigurationElementCollectionType"/> 。
        /// </value>
        public override ConfigurationElementCollectionType CollectionType
        {
            get
            {
                return ConfigurationElementCollectionType.BasicMap;
            }
        }

        /// <summary>
        /// 新しい clear-view 要素を作成します。
        /// </summary>
        /// <remarks>
        /// 新しい <see cref="ClearViewElement"/> クラスのインスタンスを作成します。
        /// </remarks>
        /// <returns>新しい clear-view 要素。</returns>
        protected override ConfigurationElement CreateNewElement()
        {
            return new ClearViewElement();
        }

        /// <summary>
        /// データセット変換設定ファイル内のこの要素のコレクションを識別するために使用する名前を取得します。
        /// </summary>
        /// <remarks>
        /// 要素名は clear-view です。
        /// </remarks>
        /// <value>
        /// この要素のコレクションの要素名。
        /// </value>
        protected override string ElementName
        {
            get
            {
                return ELEMENT_NAME;
            }
        }

        /// <summary>
        /// clear-view 要素の要素キーを取得します。
        /// </summary>
        /// <remarks>
        /// clear-view 要素には一意のキーがないので、table 属性の値を返却します。
        /// string 型を返します。
        /// </remarks>
        /// <param name="element">キーを返す <see cref="ClearViewElement"/> 。</param>
        /// <returns>
        /// 指定した <see cref="ClearViewElement"/> のキーとなる table 属性の値。
        /// </returns>
        /// <exception cref="System.ArgumentNullException">
        /// <paramref name="element"/> が null 参照です。
        /// </exception>
        /// <exception cref="System.ArgumentException">
        /// <paramref name="element"/> が
        /// <see cref="ClearViewElement"/> ではありません。
        /// </exception>
        protected override object GetElementKey(ConfigurationElement element)
        {
            if (element == null)
            {
                ArgumentNullException exception = new ArgumentNullException("element");
                if (_log.IsErrorEnabled)
                {
                    _log.Error(string.Format(
                        Client.Properties.Resources.E_NULL_ARGUMENT, "element"), exception);
                }
                throw exception;
            }

            ClearViewElement clearViewElement = element as ClearViewElement;
            if (clearViewElement == null)
            {
                string message = string.Format(Client.Properties.Resources.E_INVALID_CAST, element.GetType().FullName, typeof(ClearViewElement).FullName);
                ArgumentException exception = new ArgumentException(message);
                if (_log.IsErrorEnabled)
                {
                    _log.Error(message, exception);
                }
                throw exception;
            }

            return clearViewElement.TableName;
        }

        /// <summary>
        /// 指定したインデックス位置にある claer-view 要素を取得します。
        /// </summary>
        /// <param name="index">取得する <see cref="ClearViewElement"/> のインデックス位置。</param>
        /// <value>
        /// 指定したインデックス位置にある <see cref="ClearViewElement"/> 。
        /// </value>
        /// <remarks>
        /// <see cref="ConfigurationElementCollection.BaseGet(int)"/> メソッドを使用して、
        /// <see cref="ClearViewCollection"/> から <see cref="ClearViewElement"/> を取得します。
        /// </remarks>
        public ClearViewElement this[int index]
        {
            get
            {
                return (ClearViewElement)BaseGet(index);
            }
        }

        /// <summary>
        /// 指定した table 属性を持つ clear-view 要素を取得します。
        /// </summary>
        /// <remarks>
        /// <see cref="ConfigurationElementCollection.BaseGet(object)"/> メソッドを使用して、
        /// <see cref="ClearViewCollection"/> から <see cref="ClearViewElement"/> を取得します。
        /// <para>
        /// 指定したキーを持つ <see cref="ClearViewElement"/> がコレクションに存在しない場合、
        /// null 参照を返します。
        /// </para>
        /// </remarks>
        /// <param name="table">取得する clear-view 要素の table 属性。</param>
        /// <value>
        /// 指定した <paramref name="table"/> を持つ <see cref="ClearViewElement"/> 。
        /// </value>
        public new ClearViewElement this[string table]
        {
            get
            {
                return (ClearViewElement)BaseGet(table);
            }
        }

        /// <summary>
        /// <see cref="ClearViewCollection"/> に clear-view 要素を追加します。
        /// </summary>
        /// <param name="element">追加する <see cref="ClearViewElement"/> 。</param>
        /// <remarks>
        /// <see cref="ConfigurationElementCollection.BaseAdd(ConfigurationElement)"/> メソッドを使用して、
        /// コレクションに新しい <see cref="ClearViewElement"/> を追加します。
        /// </remarks>
        public virtual void Add(ClearViewElement element)
        {
            BaseAdd(element);
        }

        /// <summary>
        /// <see cref="ClearViewCollection"/> に clear-view 要素を追加します。
        /// </summary>
        /// <param name="element">追加する <see cref="ClearViewElement"/> 。</param>
        /// <remarks>
        /// 指定した <paramref name="element"/> が clear-view 要素コレクションに既に存在する場合、例外は発生しません。
        /// </remarks>
        protected override void BaseAdd(ConfigurationElement element)
        {
            BaseAdd(element, false);
        }

        /// <summary>
        /// <see cref="ClearViewCollection"/> が読み取り専用かどうかを示す値を取得します。 
        /// </summary>
        /// <returns>
        /// <see cref="ClearViewCollection"/> が読み取り専用の場合は true 。それ以外の場合は false 。
        /// </returns>
        /// <remarks>
        /// <see cref="ClearViewCollection"/> は読み取り専用ではありません。
        /// </remarks>
        public override bool IsReadOnly()
        {
            return false;
        }
    }
}
