/*
 * Copyright (c) 2007 NTT DATA Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package jp.terasoluna.fw.file.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.util.List;

import jp.terasoluna.fw.file.dao.FileException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * t@C@\NXB
 * <p>
 * ̃NX́ArWlXWbN璼ڗp邱Ƃ\łB<br>
 * FileUtilityNX͈ȉ̋@\ĂB
 * <ul>
 * <li>t@C̕ύXt@C̈ړ</li>
 * <li>t@C̃Rs[</li>
 * <li>t@C̍폜</li>
 * <li>t@Č</li>
 * </ul>
 * ȂAt@C@\ŎgppX͑΃pXA΃pX̗wB
 * </p>
 * 
 * 
 */
public class FileUtility {

    /**
     * VK쐬t@C̑݃`FbNB
     */
    private static boolean checkFileExist = false;

    /**
     * OCX^XB
     */
    private static Log log = LogFactory.getLog(FileUtility.class);
    
    /**
     * t@CRs[B
     * <p>
     * Rs[̃t@C̃pX󂯎ARs[̃pXɃt@CRs[B<br/>
     * Rs[Ƀt@C݂ꍇÃt@C폜At@C̃Rs[sB<br/>
     * Rs[̃pXɃt@C݂ȂꍇA񌟍OX[B<br/> Rs[̃Rs[ɎsꍇA񌟍OX[B
     * </p>
     * 
     * @param srcFile Rs[̃t@C̃pX
     * @param newFile Rs[̃t@C̃pX
     */
    public static void copyFile(String srcFile, String newFile) {
        
        checkAbsolutePath(srcFile);
        checkAbsolutePath(newFile);
        
        File srcFileObject = new File(srcFile);
        // Rs[̃pXɃt@C݂ȂꍇAG[𓊂ďIB
        if (!srcFileObject.exists()) {
            throw new FileException(srcFile + " is not exist.", srcFile);
        }

        File newFileObject = new File(newFile);
        // ړ̃t@C݂ꍇÃt@C폜B
        if (newFileObject.exists() && checkFileExist) {
            deleteFile(newFile);
        } else if (newFileObject.exists() && !checkFileExist) {
            throw new FileException(newFile + " is exist.", newFile);
        }

        FileOutputStream fos = null;
        FileChannel outputFileChannel = null;
        FileInputStream ios = null;
        FileChannel inputFileChannel = null;
        FileLock inputFileLock = null;
        FileLock outputFileLock = null;
        try {
            fos = new FileOutputStream(newFileObject, true);
            outputFileChannel = fos.getChannel();

            ios = new FileInputStream(srcFileObject);
            inputFileChannel = ios.getChannel();

            ByteBuffer buffer = ByteBuffer.allocate(1024);

            inputFileLock = inputFileChannel.lock(0L, Long.MAX_VALUE, true);
            outputFileLock = outputFileChannel.lock(0L, Long.MAX_VALUE, false);

            while (inputFileChannel.position() < inputFileChannel.size()) {
                buffer.clear();
                inputFileChannel.read(buffer);
                buffer.flip();
                outputFileChannel.write(buffer);
            }
        } catch (FileNotFoundException e) {
            throw new FileException(e);
        } catch (IOException e) {
            throw new FileException(e);
        } finally {
            try {
                if (inputFileLock != null) {
                    inputFileLock.release();
                }
                if (outputFileLock != null) {
                    outputFileLock.release();
                }

                if (ios != null) {
                    ios.close();
                }
                
                if (fos != null) {
                    fos.close();
                }
                
                if (outputFileChannel != null) {
                    outputFileChannel.close();
                }
                
                if (inputFileChannel != null) {
                    inputFileChannel.close();
                }
            } catch (IOException e) {
                log.error(e.getMessage(), e);
            }
        }
    }

    /**
     * t@C폜B
     * <p>
     * 폜t@C̃pX󂯎At@C폜B<br/> 폜t@C݂ȂꍇA񌟍OX[B<br/>
     * 폜ɎsꍇA񌟍OX[B
     * </p>
     * 
     * @param srcFile 폜t@C̃pX
     */
    public static void deleteFile(String srcFile) {
        
        checkAbsolutePath(srcFile);
        
        File srcFileObject = new File(srcFile);

        // ړƂ̃t@C݂ȂꍇAG[𓊂ďIB
        if (!srcFileObject.exists()) {
            throw new FileException(srcFile + " is not exist.", srcFile);
        }

        boolean result = srcFileObject.delete();

        if (!result) {
            throw new FileException("File control operation was failed.",
                    srcFile);
        }
    }

    /**
     * t@CB
     * <p>
     * t@C̃Xg󂯎At@CB<br/>
     * ĐV쐬t@C̃pXɁAJn܂łɃt@C݂ꍇÃt@C폜̂At@CB<br/>
     * t@CXgɊ܂܂t@C݂ȂꍇA񌟍OX[B<br/>
     * t@ČɎsꍇA񌟍OX[B
     * </p>
     * 
     * @param fileList t@C̃Xg
     * @param newFile Ăłt@C̃pX
     */
    public static void mergeFile(List<String> fileList, String newFile) {
        File newFileObject = new File(newFile);

        checkAbsolutePath(newFile);
        // ړ̃t@C݂ꍇÃt@C폜B
        if (newFileObject.exists() && checkFileExist) {
            deleteFile(newFile);
        } else if (newFileObject.exists() && !checkFileExist) {
            throw new FileException(newFile + " is exist.", newFile);
        }

        FileOutputStream fos = null;
        FileChannel outputFileChannel = null;
        FileLock outputFileLock = null;

        try {
            fos = new FileOutputStream(newFileObject, true);
            outputFileChannel = fos.getChannel();
            outputFileLock = outputFileChannel.lock(0L, Long.MAX_VALUE, false);
            
            File srcFileObject = null;
            
            for (String srcFile : fileList) {
                
                checkAbsolutePath(srcFile);
                
                srcFileObject = new File(srcFile);

                // }[W̃t@C݂ȂꍇAG[𓊂ďIB
                if (!srcFileObject.exists()) {
                    throw new FileException(srcFile + " is not exist.",
                            srcFile);
                }

                FileInputStream fis = null;
                FileChannel inputFileChannel = null;
                FileLock inputFileLock = null;

                try {
                    fis = new FileInputStream(srcFileObject);
                    inputFileChannel = fis.getChannel();
                    ByteBuffer buffer = ByteBuffer.allocate(1024);
    
                    inputFileLock =
                        inputFileChannel.lock(0L, Long.MAX_VALUE, true);
    
                    while (inputFileChannel.position()
                            < inputFileChannel.size()) {
                        buffer.clear();
                        inputFileChannel.read(buffer);
                        buffer.flip();
                        outputFileChannel.write(buffer);
                    }
                } finally {
                    if (inputFileLock != null) {
                        inputFileLock.release();
                    }
                    if (fis != null) {
                        fis.close();
                    }
                    if (inputFileChannel != null) {
                        inputFileChannel.close();
                    }
                }

            }
        } catch (FileNotFoundException e) {
            throw new FileException(e);
        } catch (IOException e) {
            throw new FileException(e);
        } finally {
            try {
                if (outputFileLock != null) {
                    outputFileLock.release();
                }
                
                if (fos != null) {
                    fos.close();
                }
          
                if (outputFileChannel != null) {
                    outputFileChannel.close();
                }
                
            } catch (IOException e) {
                log.error(e.getMessage(), e);
            }

        }
    }

    /**
     * t@C̕ύXt@C̈ړB
     * <p>
     * ړ̃t@C̃pX󂯎Aړ̃pXɃf[^ړB<br/>
     * ړ̃pXɃt@C݂ꍇÃt@C폜At@C̈ړsB<br/>
     * ړ̃t@C݂ȂꍇA񌟍OX[B<br/> t@C̈ړɎsꍇA񌟍OX[B
     * </p>
     * 
     * @param srcFile ړÕpX
     * @param newFile ړ̃pX
     */
    public static void renameFile(String srcFile, String newFile) {
        
        checkAbsolutePath(srcFile);
        checkAbsolutePath(newFile); 
        
        File srcFileObject = new File(srcFile);
        File newFileObject = new File(newFile);

        // ړƂ̃t@C݂ȂꍇAG[𓊂ďIB
        if (!srcFileObject.exists()) {
            throw new FileException(srcFile + " is not exist.", srcFile);
        }

        // ړ̃t@C݂ꍇÃt@C폜B
        if (newFileObject.exists() && checkFileExist) {
            deleteFile(newFile);
        } else if (newFileObject.exists() && !checkFileExist) {
            throw new FileException(newFile + " is exist.", newFile);
        }

        boolean result = true;
        result = srcFileObject.renameTo(newFileObject);

        if (!result) {
            throw new FileException("File control operation was failed.");
        }
    }

    /**
     * @return t@C݂ꍇ̏𔻕ʂtO
     */
    public static boolean isCheckFileExist() {
        return checkFileExist;
    }

    /**
     * t@C݂ꍇ̏𔻕ʂtOݒ肷B
     * @param checkFileExist t@C݂ꍇ̏𔻕ʂtO
     */
    public static void setCheckFileExist(boolean checkFileExist) {
        FileUtility.checkFileExist = checkFileExist;
    }

    /**
     * ̃pX΃pXł邱ƂmFB
     * <P>
     * ̃pX΃pXł邱ƂmFB΃pXłȂꍇAOX[B
     * </P>
     * @param filePath t@C̃pX
     */
    private static void checkAbsolutePath(String filePath) {
        File file = new File(filePath);
        if (!file.isAbsolute()) {
            throw new FileException("File path is not absolute.", filePath);
        }
    }
}
