﻿// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using System;
using System.Configuration;
using TERASOLUNA.Fw.Common;
using TERASOLUNA.Fw.Common.Logging;

namespace TERASOLUNA.Fw.Client.Configuration.Conversion
{
    /// <summary>
    /// データセット変換設定ファイルの設定情報を提供するクラスです。
    /// </summary>
    /// <remarks>
    /// アプリケーション構成ファイルに ConversionConfiguration セクションを
    /// 定義する必要があります。
    /// <see cref="ConversionConfigurationSection"/> を参照してください。
    /// </remarks>
    public abstract class ConversionConfiguration
    {
        /// <summary>
        /// <see cref="ILog"/> 実装クラスのインスタンスです。
        /// </summary>
        /// <remarks>
        /// ログ出力に利用します。
        /// </remarks>
        private static ILog _log = LogFactory.GetLogger(typeof(ConversionConfiguration));

        /// <summary>
        /// 指定した id を持つ convert 要素を取得します。
        /// </summary>
        /// <param name="id">取得する convert 要素の id 。</param>
        /// <returns><see cref="ConvertElement"/> 。</returns>
        /// <exception cref="ArgumentNullException">
        /// <paramref name="id"/> が null 参照です。
        /// </exception>
        /// <exception cref="ArgumentException">
        /// <paramref name="id"/> が空の文字列です。
        /// </exception>
        /// <exception cref="ConfigurationErrorsException">
        /// 以下のような場合に例外をスローします。
        /// <list type="bullet">
        /// <item>
        /// <description>
        /// アプリケーション構成ファイルが読み込めません。
        /// </description>
        /// </item>
        /// <item>
        /// <description>
        /// アプリケーション構成ファイルの設定情報が不正です。
        /// </description>
        /// </item>
        /// <item>
        /// <description>
        /// データセット変換設定ファイルが読み込めません。
        /// </description>
        /// </item>
        /// <item>
        /// <description>
        /// データセット変換設定ファイルの XML スキーマが不正です。
        /// </description>
        /// </item>
        /// </list>
        /// </exception>
        public static ConvertElement GetConvert(string id)
        {
            if (id == null)
            {
                ArgumentNullException exception = new ArgumentNullException("id");
                if (_log.IsErrorEnabled)
                {
                    _log.Error(string.Format(
                        Properties.Resources.E_NULL_ARGUMENT, "id"), exception);
                }
                throw exception;
            }
            else if (id.Length == 0)
            {
                string message = string.Format(Properties.Resources.E_EMPTY_STRING, "id");
                ArgumentException exception = new ArgumentException(message);
                if (_log.IsErrorEnabled)
                {
                    _log.Error(message, exception);
                }
                throw exception;
            }

            ConversionConfigurationSection conversionConfigSection = TerasolunaClientConfiguration.ConversionConfigSection;

            // idに対応する設定が無い場合はnullを返却する
            ConvertElement element = null;
            if (conversionConfigSection != null)
            {
                element = conversionConfigSection.ConvertCollection[id];
            }

            if (_log.IsTraceEnabled)
            {
                if (element != null)
                {
                    _log.Trace(string.Format(
                        Properties.Resources.T_CONVERSION_CONFIGURATION_LOAD_ELEMENT, id));
                }
                else
                {
                    _log.Trace(string.Format(
                        Properties.Resources.T_CONVERSION_CONFIGURATION_NOT_LOAD_ELEMENT, id));
                }
            }

            return element;
        }

        /// <summary>
        /// 指定した id を持つ convert 要素内に格納されている param 要素を取得します。
        /// </summary>
        /// <remarks>
        /// 指定した id を持つ convert 要素が存在しない場合は、<see cref="TerasolunaException"/> が発生します。
        /// </remarks>
        /// <param name="id">取得する param 要素を保持する convert 要素の id 。</param>
        /// <returns>param 要素。</returns>
        /// <exception cref="ArgumentNullException">id が null 参照です。</exception>
        /// <exception cref="ArgumentException">id が空の文字列です。</exception>
        /// <exception cref="TerasolunaException">id に対応する変換情報がデータセット変換設定ファイルにありません。</exception>
        public static ParamElement GetParam(string id)
        {
            if (id == null)
            {
                ArgumentNullException exception = new ArgumentNullException("id");
                if (_log.IsErrorEnabled)
                {
                    _log.Error(string.Format(
                        Properties.Resources.E_NULL_ARGUMENT, "id"), exception);
                }
                throw exception;
            }
            else if (id.Length == 0)
            {
                string message = string.Format(Properties.Resources.E_EMPTY_STRING, "id");
                ArgumentException exception = new ArgumentException(message);
                if (_log.IsErrorEnabled)
                {
                    _log.Error(message, exception);
                }
                throw exception;
            }

            ConversionConfigurationSection conversionConfigSection = TerasolunaClientConfiguration.ConversionConfigSection;

            ConvertElement element = null;
            if (conversionConfigSection != null)
            {
                element = conversionConfigSection.ConvertCollection[id];
            }

            if (element == null)
            {   // id に対応する設定が無い場合
                string message = string.Format(Properties.Resources.E_CONVERSION_TYPE_NOT_FOUND, id);
                TerasolunaException exception = new TerasolunaException(message);
                if (_log.IsErrorEnabled)
                {
                    _log.Error(message, exception);
                }
                throw exception;
            }

            if (_log.IsTraceEnabled)
            {
                _log.Trace(string.Format(
                    Properties.Resources.T_CONVERSION_CONFIGURATION_LOAD_PARAMS_ELEMENT, id));
            }

            return element.Param;
        }

        /// <summary>
        /// 指定した id を持つ convert 要素内に格納されている result 要素を取得します。
        /// </summary>
        /// <remarks>
        /// 指定した id を持つ convert 要素が存在しない場合は、<see cref="TerasolunaException"/> が発生します。
        /// </remarks>
        /// <param name="id">取得する result 要素を保持する convert 要素の id 。</param>
        /// <returns>result 要素。</returns>
        /// <exception cref="ArgumentNullException">id が null 参照です。</exception>
        /// <exception cref="ArgumentException">id が空の文字列です。</exception>
        /// <exception cref="TerasolunaException">id に対応する変換情報がデータセット変換設定ファイルにありません。</exception>
        public static ResultElement GetResult(string id)
        {
            if (id == null)
            {
                ArgumentNullException exception = new ArgumentNullException("id");
                if (_log.IsErrorEnabled)
                {
                    _log.Error(string.Format(
                        Properties.Resources.E_NULL_ARGUMENT, "id"), exception);
                }
                throw exception;
            }
            else if (id.Length == 0)
            {
                string message = string.Format(Properties.Resources.E_EMPTY_STRING, "id");
                ArgumentException exception = new ArgumentException(message);
                if (_log.IsErrorEnabled)
                {
                    _log.Error(message, exception);
                }
                throw exception;
            }

            ConversionConfigurationSection conversionConfigSection = TerasolunaClientConfiguration.ConversionConfigSection;

            ConvertElement element = null;
            if (conversionConfigSection != null)
            {
                element = conversionConfigSection.ConvertCollection[id];
            }

            if (element == null)
            {   // id に対応する設定が無い場合
                string message = string.Format(Properties.Resources.E_CONVERSION_TYPE_NOT_FOUND, id);
                TerasolunaException exception = new TerasolunaException(message);
                if (_log.IsErrorEnabled)
                {
                    _log.Error(message, exception);
                }
                throw exception;
            }

            if (_log.IsTraceEnabled)
            {
                _log.Trace(string.Format(
                    Properties.Resources.T_CONVERSION_CONFIGURATION_LOAD_RESULT_ELEMENT, id));
            }

            return element.Result;
        }
    }
}
