// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Text;
using System.Collections.Generic;
using TERASOLUNA.Fw.Client.Forms;
using System.Windows.Forms;
using System.Collections;
using System.Data;
using TERASOLUNA.Fw.Common;
using TERASOLUNA.Fw.Utlib.Logging;
using TERASOLUNA.Fw.Common.Logging;

namespace TERASOLUNA.Fw.ClientUT.Forms
{
    /// <summary>
    ///TERASOLUNA.Fw.Client.Forms.ForwardableFormCloseEventArgs のテスト クラスです。すべての
    ///TERASOLUNA.Fw.Client.Forms.ForwardableFormCloseEventArgs 単体テストをここに含めます
    ///</summary>
   
    [TestClass()]
    public class ForwardableFormCloseEventArgsTest
    {


        private TestContext testContextInstance;

        ///<summary>
        /// 現在のテストの実行についての情報および機能を
        /// 提供するテスト コンテキストを取得または設定します。
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }
        #region 追加のテスト属性
        // 
        //テストを作成するときに、次の追加属性を使用することができます:
        //
        //クラスの最初のテストを実行する前にコードを実行するには、ClassInitialize を使用
        //
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //クラスのすべてのテストを実行した後にコードを実行するには、ClassCleanup を使用
        //
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //各テストを実行する前にコードを実行するには、TestInitialize を使用
        //
        [TestInitialize()]
        public void MyTestInitialize()
        {
            LogUTUtil.Clear();
        }
        //
        //各テストを実行した後にコードを実行するには、TestCleanup を使用
        //
        //[TestCleanup()]
        //public void MyTestCleanup()
        //{
        //}
        //
        #endregion


        ///<summary>
        /// ForwardableFormCloseEventArgs (IForwardable, EventArgs) のテスト
        ///</summary>
        ///<remarks>
        /// forwardableにFormBaseインスタンス、innerEventArgsに有効な値を設定した場合に
        /// Items、ViewData、DialogResult、InnerEventArgsが正しく設定されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ConstructorTest01()
        {
            // 準備
            FormBase form = new FormBase();
            EventArgs expectedInnerEventArgs = new EventArgs();
            object expectedItems = form.Items;
            object expectedDataSet = form.ViewData;
            object expectedDialogResult = form.DialogResult;

            // 実行
            ForwardableFormCloseEventArgs target = new ForwardableFormCloseEventArgs(form, expectedInnerEventArgs);
            object actualItems = target.Items;
            object actualDataSet = target.ViewData;
            object actualDialogResult = target.DialogResult;
            object actualInnerEventArgs = target.InnerEventArgs;

            // 検証
            Assert.AreSame(expectedItems, actualItems);
            Assert.AreSame(expectedDataSet, actualDataSet);
            Assert.AreEqual(expectedDialogResult, actualDialogResult);
            Assert.AreSame(expectedInnerEventArgs, actualInnerEventArgs);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ForwardableFormCloseEventArgs), LogLevel.INFO));

        }

        ///<summary>
        /// ForwardableFormCloseEventArgs (IForwardable, EventArgs) のテスト
        ///</summary>
        ///<remarks>
        /// forwardableにFormBaseインスタンス、innerEventArgsにnullを設定した場合に
        /// Items、ViewData、DialogResultが正しく設定され、InnerEventArgsがnullを返却することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ConstructorTest02()
        {
            // 準備
            FormBase form = new FormBase();
            EventArgs expectedInnerEventArgs = null;
            object expectedItems = form.Items;
            object expectedDataSet = form.ViewData;
            object expectedDialogResult = form.DialogResult;

            // 実行
            ForwardableFormCloseEventArgs target = new ForwardableFormCloseEventArgs(form, expectedInnerEventArgs);
            object actualItems = target.Items;
            object actualDataSet = target.ViewData;
            object actualDialogResult = target.DialogResult;
            object actualInnerEventArgs = target.InnerEventArgs;

            // 検証
            Assert.AreSame(expectedItems, actualItems);
            Assert.AreSame(expectedDataSet, actualDataSet);
            Assert.AreEqual(expectedDialogResult, actualDialogResult);
            Assert.IsNull(actualInnerEventArgs);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ForwardableFormCloseEventArgs), LogLevel.INFO));

        }

        ///<summary>
        /// ForwardableFormCloseEventArgs (IForwardable, EventArgs) のテスト
        ///</summary>
        ///<remarks>
        /// forwardableにnull、innerEventArgsに有効な値を設定した場合に
        /// Items、ViewDataがnullを返却し、DialogResultがDialogResult.Noneを返却し、InnerEventArgsが正しく設定されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ConstructorTest03()
        {
            // 準備
            EventArgs expectedInnerEventArgs = new EventArgs();

            // 実行
            ForwardableFormCloseEventArgs target = new ForwardableFormCloseEventArgs(null, expectedInnerEventArgs);

            object actualItems = target.Items;
            object actualDataSet = target.ViewData;
            object actualDialogResult = target.DialogResult;
            object actualInnerEventArgs = target.InnerEventArgs;

            // 検証
            Assert.IsNull(actualItems);
            Assert.IsNull(actualDataSet);
            Assert.AreEqual(DialogResult.None, actualDialogResult);
            Assert.AreSame(expectedInnerEventArgs, actualInnerEventArgs);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ForwardableFormCloseEventArgs), LogLevel.INFO));

        }

        ///<summary>
        /// ForwardableFormCloseEventArgs (IForwardable, EventArgs) のテスト
        ///</summary>
        ///<remarks>
        /// forwardable、innerEventArgsにnullを設定した場合に
        /// Items、ViewData、InnerEventArgsがnullを返却し、DialogResultがDialogResult.Noneを返却することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ConstructorTest04()
        {
            // 準備
            FormBase form = null;
            EventArgs innerEventArgs = null;

            // 実行
            ForwardableFormCloseEventArgs target = new ForwardableFormCloseEventArgs(form, innerEventArgs);
            object actualItems = target.Items;
            object actualDataSet = target.ViewData;
            object actualDialogResult = target.DialogResult;
            object actualInnerEventArgs = target.InnerEventArgs;

            // 検証
            Assert.IsNull(actualItems);
            Assert.IsNull(actualDataSet);
            Assert.AreEqual(DialogResult.None, actualDialogResult);
            Assert.IsNull(actualInnerEventArgs);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ForwardableFormCloseEventArgs), LogLevel.INFO));

        }

        ///<summary>
        /// ForwardableFormCloseEventArgs (IForwardable, EventArgs) のテスト
        ///</summary>
        ///<remarks>
        /// forwardableにFormにキャストできないクラスを指定した場合にTerasolunaExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ConstructorTest05()
        {
            // 準備
            IForwardable form = new ForwardableFormCloseEventArgs_IForwardableStub01();
            EventArgs innerEventArgs = null;
            
            try
            {
                // 実行
                ForwardableFormCloseEventArgs target = new ForwardableFormCloseEventArgs(form, innerEventArgs);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // 検証
                Assert.AreEqual(MessageManager.GetMessage("E_INVALID_CAST", form.GetType().FullName, typeof(Form).FullName), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ForwardableFormCloseEventArgs), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ForwardableFormCloseEventArgs), typeof(TerasolunaException), "E_INVALID_CAST", form.GetType().FullName, typeof(Form).FullName));
        }

        ///<summary>
        /// InnerEventArgs のテスト
        ///</summary>
        ///<remarks>
        /// コンストラクタで設定したInnerEventArgsが正しく取り出せることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void InnerEventArgsTest01()
        {
            // 準備
            IForwardable forwardable = null;
            EventArgs expectedInnerEventArgs = new EventArgs();

            // 実行
            ForwardableFormCloseEventArgs target = new ForwardableFormCloseEventArgs(forwardable, expectedInnerEventArgs);
            object actual = target.InnerEventArgs;

            // 検証
            Assert.AreSame(expectedInnerEventArgs, actual);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ForwardableFormCloseEventArgs), LogLevel.INFO));
        }

        ///<summary>
        /// InnerEventArgs のテスト
        ///</summary>
        ///<remarks>
        /// コンストラクタでInnerEventArgsにnullを設定した場合、nullが返却されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void InnerEventArgsTest02()
        {
            // 準備
            IForwardable forwardable = null;
            EventArgs expectedInnerEventArgs = null;

            // 実行
            ForwardableFormCloseEventArgs target = new ForwardableFormCloseEventArgs(forwardable, expectedInnerEventArgs);
            object actual = target.InnerEventArgs;

            // 検証
            Assert.IsNull(actual);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ForwardableFormCloseEventArgs), LogLevel.INFO));
        }

        ///<summary>
        /// Items のテスト
        ///</summary>
        ///<remarks>
        /// forwardableのItemsに設定したIDictionayが正しく取り出せることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ItemsTest01()
        {
            // 準備
            FormBase form = new FormBase();
            form.Items.Add("key1", "value1");
            form.Items.Add("key2", "value2");
            form.Items.Add("key3", "value3");

            // 実行
            ForwardableFormCloseEventArgs target = new ForwardableFormCloseEventArgs(form, null);

            // 検証
            Assert.AreEqual(3, target.Items.Count);
            Assert.AreEqual("value1", target.Items["key1"]);
            Assert.AreEqual("value2", target.Items["key2"]);
            Assert.AreEqual("value3", target.Items["key3"]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ForwardableFormCloseEventArgs), LogLevel.INFO));
        }

        ///<summary>
        /// Items のテスト
        ///</summary>
        ///<remarks>
        /// forwardableのItemsに何も設定しない場合、空のディクショナリが返却されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ItemsTest02()
        {
            // 準備
            FormBase form = new FormBase();
            int expectedCount = 0;

            // 実行
            ForwardableFormCloseEventArgs target = new ForwardableFormCloseEventArgs(form, null);
            int actualCount = target.Items.Count;

            // 検証
            Assert.AreEqual(expectedCount, actualCount);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ForwardableFormCloseEventArgs), LogLevel.INFO));

        }

        ///<summary>
        /// Items のテスト
        ///</summary>
        ///<remarks>
        /// forwardableのItemsにひとつだけ値を設定した場合、値が正しく取り出せることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ItemsTest03()
        {
            // 準備
            FormBase form = new FormBase();
            int expectedCount = 1;
            object expectedValue = "value1";
            form.Items.Add("key1", expectedValue);
            ForwardableFormCloseEventArgs target = new ForwardableFormCloseEventArgs(form, null);

            // 実行
            int actualCount = target.Items.Count;
            object actualValue = target.Items["key1"];

            // 検証
            Assert.AreEqual(expectedCount, actualCount);
            Assert.AreSame(expectedValue, actualValue);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ForwardableFormCloseEventArgs), LogLevel.INFO));

        }

        ///<summary>
        /// ViewData のテスト
        ///</summary>
        ///<remarks>
        /// forwardableのViewDataにDataSetを設定した場合、正しくDataSetを取り出せることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ViewDataTest01()
        {
            // 準備
            FormBase form = new FormBase();
            DataSet expectedDataSet = new DataSet();
            form.ViewData = expectedDataSet;
            ForwardableFormCloseEventArgs target = new ForwardableFormCloseEventArgs(form, null);

            // 実行
            DataSet actualDataSet = target.ViewData;

            // 検証
            Assert.AreSame(expectedDataSet, actualDataSet);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ForwardableFormCloseEventArgs), LogLevel.INFO));
        }

        ///<summary>
        /// ViewData のテスト
        ///</summary>
        ///<remarks>
        /// forwardableのViewDataにnullを設定した場合、nullが返却されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ViewDataTest02()
        {
            // 準備
            FormBase form = new FormBase();
            DataSet expectedDataSet = new DataSet();
            form.ViewData = null;
            ForwardableFormCloseEventArgs target = new ForwardableFormCloseEventArgs(form, null);

            // 実行
            DataSet actualDataSet = target.ViewData;

            // 検証
            Assert.IsNull(actualDataSet);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ForwardableFormCloseEventArgs), LogLevel.INFO));

        }


        ///<summary>
        /// DialogResult のテスト
        ///</summary>
        ///<remarks>
        /// forwardableのDialogResultに何も設定しない場合、DialogResult.Noneが返却されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void DialogResultTest01()
        {
            // 準備
            FormBase form = new FormBase();
            ForwardableFormCloseEventArgs target = new ForwardableFormCloseEventArgs(form, null);

            // 実行
            DialogResult actualResult = target.DialogResult;

            // 検証
            Assert.AreEqual(DialogResult.None, actualResult);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ForwardableFormCloseEventArgs), LogLevel.INFO));

        }

        ///<summary>
        /// DialogResult のテスト
        ///</summary>
        ///<remarks>
        /// forwardableのDialogResultにDialogResult.Noneを設定した場合、DialogResult.Noneが返却されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void DialogResultTest02()
        {
            // 準備
            DialogResult expectedResult = DialogResult.None;
            FormBase form = new FormBase();
            form.DialogResult = DialogResult.None;
            ForwardableFormCloseEventArgs target = new ForwardableFormCloseEventArgs(form, null);

            // 実行
            DialogResult actualResult = target.DialogResult;

            // 検証
            Assert.AreEqual(expectedResult, actualResult);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ForwardableFormCloseEventArgs), LogLevel.INFO));

        }

        ///<summary>
        /// DialogResult のテスト
        ///</summary>
        ///<remarks>
        /// forwardableのDialogResultにDialogResult.Abortを設定した場合、DialogResult.Abortが返却されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void DialogResultTest03()
        {
            // 準備
            DialogResult expectedResult = DialogResult.Abort;
            FormBase form = new FormBase();
            form.DialogResult = expectedResult;
            ForwardableFormCloseEventArgs target = new ForwardableFormCloseEventArgs(form, null);

            // 実行
            DialogResult actualResult = target.DialogResult;

            // 検証
            Assert.AreEqual(expectedResult, actualResult);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ForwardableFormCloseEventArgs), LogLevel.INFO));

        }

        ///<summary>
        /// DialogResult のテスト
        ///</summary>
        ///<remarks>
        /// forwardableのDialogResultにDialogResult.Cancelを設定した場合、DialogResult.Cancelが返却されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void DialogResultTest04()
        {
            // 準備
            DialogResult expectedResult = DialogResult.Cancel;
            FormBase form = new FormBase();
            form.DialogResult = expectedResult;
            ForwardableFormCloseEventArgs target = new ForwardableFormCloseEventArgs(form, null);

            // 実行
            DialogResult actualResult = target.DialogResult;

            // 検証
            Assert.AreEqual(expectedResult, actualResult);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ForwardableFormCloseEventArgs), LogLevel.INFO));

        }

        ///<summary>
        /// DialogResult のテスト
        ///</summary>
        ///<remarks>
        /// forwardableのDialogResultにDialogResult.Ignoreを設定した場合、DialogResult.Ignoreが返却されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void DialogResultTest05()
        {
            // 準備
            DialogResult expectedResult = DialogResult.Ignore;
            FormBase form = new FormBase();
            form.DialogResult = expectedResult;
            ForwardableFormCloseEventArgs target = new ForwardableFormCloseEventArgs(form, null);

            // 実行
            DialogResult actualResult = target.DialogResult;

            // 検証
            Assert.AreEqual(expectedResult, actualResult);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ForwardableFormCloseEventArgs), LogLevel.INFO));

        }

        ///<summary>
        /// DialogResult のテスト
        ///</summary>
        ///<remarks>
        /// forwardableのDialogResultにDialogResult.Noを設定した場合、DialogResult.Noが返却されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void DialogResultTest06()
        {
            // 準備
            DialogResult expectedResult = DialogResult.No;
            FormBase form = new FormBase();
            form.DialogResult = expectedResult;
            ForwardableFormCloseEventArgs target = new ForwardableFormCloseEventArgs(form, null);

            // 実行
            DialogResult actualResult = target.DialogResult;

            // 検証
            Assert.AreEqual(expectedResult, actualResult);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ForwardableFormCloseEventArgs), LogLevel.INFO));

        }

        ///<summary>
        /// DialogResult のテスト
        ///</summary>
        ///<remarks>
        /// forwardableのDialogResultにDialogResult.OKを設定した場合、DialogResult.OKが返却されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void DialogResultTest07()
        {
            // 準備
            DialogResult expectedResult = DialogResult.OK;
            FormBase form = new FormBase();
            form.DialogResult = expectedResult;
            ForwardableFormCloseEventArgs target = new ForwardableFormCloseEventArgs(form, null);

            // 実行
            DialogResult actualResult = target.DialogResult;

            // 検証
            Assert.AreEqual(expectedResult, actualResult);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ForwardableFormCloseEventArgs), LogLevel.INFO));

        }

        ///<summary>
        /// DialogResult のテスト
        ///</summary>
        ///<remarks>
        /// forwardableのDialogResultにDialogResult.Retryを設定した場合、DialogResult.Retryが返却されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void DialogResultTest08()
        {
            // 準備
            DialogResult expectedResult = DialogResult.Retry;
            FormBase form = new FormBase();
            form.DialogResult = expectedResult;
            ForwardableFormCloseEventArgs target = new ForwardableFormCloseEventArgs(form, null);

            // 実行
            DialogResult actualResult = target.DialogResult;

            // 検証
            Assert.AreEqual(expectedResult, actualResult);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ForwardableFormCloseEventArgs), LogLevel.INFO));

        }

        ///<summary>
        ///DialogResult のテスト
        ///</summary>
        ///<remarks>
        /// forwardableのDialogResultにDialogResult.Yesを設定した場合、DialogResult.Yesが返却されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void DialogResultTest09()
        {
            // 準備
            DialogResult expectedResult = DialogResult.Yes;
            FormBase form = new FormBase();
            form.DialogResult = expectedResult;
            ForwardableFormCloseEventArgs target = new ForwardableFormCloseEventArgs(form, null);

            // 実行
            DialogResult actualResult = target.DialogResult;

            // 検証
            Assert.AreEqual(expectedResult, actualResult);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ForwardableFormCloseEventArgs), LogLevel.INFO));

        }

    }
    internal class ForwardableFormCloseEventArgs_IForwardableStub01 : IForwardable
    {

        #region IForwardable メンバ

        public IDictionary Items
        {
            get
            {
                return null;
            }
        }

        public IForwardable ParentForwardable
        {
            get
            {
                return null;
            }
            set
            {
            }
        }

        public DataSet ViewData
        {
            get
            {
                return null;
            }
        }

        public bool Init(IForwardable forwardable)
        {
            return true;
        }

        #endregion
    }
}
