﻿// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Text;
using System.Collections.Generic;
using TERASOLUNA.Fw.Client.Configuration.Conversion;
using System.Configuration;
using TERASOLUNA.Fw.Utlib.Logging;
using TERASOLUNA.Fw.Common.Logging;
using TERASOLUNA.Fw.Common;
namespace TERASOLUNA.Fw.ClientUT.Configuration.Conversion
{
    /// <summary>
    ///TERASOLUNA.Fw.Client.Configuration.Conversion.ConvertCollection のテスト クラスです。すべての
    ///TERASOLUNA.Fw.Client.Configuration.Conversion.ConvertCollection 単体テストをここに含めます
    ///</summary>
    [TestClass()]
    public class ConvertCollectionTest
    {
        private TestContext testContextInstance;

        ///<summary>
        /// 現在のテストの実行についての情報および機能を
        /// 提供するテスト コンテキストを取得または設定します。
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }
        #region 追加のテスト属性
        // 
        //テストを作成するときに、次の追加属性を使用することができます:
        //
        //クラスの最初のテストを実行する前にコードを実行するには、ClassInitialize を使用
        //
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //クラスのすべてのテストを実行した後にコードを実行するには、ClassCleanup を使用
        //
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //各テストを実行する前にコードを実行するには、TestInitialize を使用
        //
        [TestInitialize()]
        public void MyTestInitialize()
        {
            LogUTUtil.Clear();
        }
        //
        //各テストを実行した後にコードを実行するには、TestCleanup を使用
        //
        //[TestCleanup()]
        //public void MyTestCleanup()
        //{
        //}
        //
        #endregion

        ///<summary>
        /// ConvertCollection () のテスト
        ///</summary>
        ///<remarks>
        /// 引数無しのコンストラクタを呼び出した場合、インスタンスが返却されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ConstructorTest01()
        {
            //実行
            ConvertCollection target = new ConvertCollection();

            //検証
            Assert.IsNotNull(target);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ConvertCollection), LogLevel.INFO));
        }

        ///<summary>
        /// CreateNewElement () のテスト
        ///</summary>
        ///<remarks>
        /// 引数無しでメソッドを呼び出した場合、ConvertElementが返却されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void CreateNewElementTest01()
        {
            //準備
            ConvertCollectionImpl target = new ConvertCollectionImpl();

            //実行
            ConfigurationElement actualConvertElement = target.CreateNewElement();

            //検証
            Assert.IsNotNull(actualConvertElement);
            Assert.IsInstanceOfType(actualConvertElement, typeof(ConvertElement));

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ConvertCollection), LogLevel.INFO));
        }

        ///<summary>
        /// GetElementKey (ConfigurationElement) のテスト
        ///</summary>
        ///<remarks>
        /// Idを設定したConfigurationElementを引数とした場合、
        /// 例外が発生せずにConvertElementに設定したIdが返却されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void GetElementKeyTest01()
        {
            //準備
            ConvertCollectionImpl target = new ConvertCollectionImpl();
            string expectedId = "id0001";
            ParamElement paramElement = new ParamElement();
            ResultElement resultElement = new ResultElement();
            ConvertElement element = new ConvertElement(expectedId, paramElement, resultElement);

            //実行
            object actualId = target.GetElementKey(element);

            //検証
            Assert.IsInstanceOfType(actualId, typeof(string));
            Assert.AreEqual(expectedId, (string)actualId);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ConvertCollection), LogLevel.INFO));
        }

        ///<summary>
        /// GetElementKey (ConfigurationElement) のテスト
        ///</summary>
        ///<remarks>
        /// nullを引数とした場合、ArgumentNullExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void GetElementKeyTest02()
        {
            //準備
            ConvertCollectionImpl target = new ConvertCollectionImpl();
            ConvertElement element = null;
            try
            {
                //実行
                object actualId = target.GetElementKey(element);

                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // メッセージ確認
                Assert.AreEqual("element", e.ParamName);
            }
            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ConvertCollection), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ConvertCollection), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "element"));
        }

        ///<summary>
        /// GetElementKey (ConfigurationElement) のテスト
        ///</summary>
        ///<remarks>
        /// ConvertElementではないConfigurationElementのサブクラスを引数とした場合、
        /// ArgumentExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void GetElementKeyTest03()
        {
            //準備
            ConvertCollectionImpl target = new ConvertCollectionImpl();
            ColumnElement element = new ColumnElement();
            try
            {
                //実行
                object actualId = target.GetElementKey(element);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // 確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_INVALID_CAST", element.GetType().FullName, typeof(ConvertElement).FullName), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ConvertCollection), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ConvertCollection), typeof(ArgumentException), "E_INVALID_CAST", element.GetType().FullName, typeof(ConvertElement).FullName));

        }

        ///<summary>
        /// this [int] のテスト
        ///</summary>
        ///<remarks>
        /// ConvertElementを格納し、indexを指定して取得した場合、一致すること確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ThisIntTest01()
        {
            //準備
            ConvertCollection target = new ConvertCollection();
            ConvertElement expectedElement = new ConvertElement();
            target.Add(expectedElement);

            //実行
            ConvertElement actualElement = target[0];

            //検証
            Assert.AreEqual(expectedElement, actualElement);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ConvertCollection), LogLevel.INFO));
        }

        ///<summary>
        /// this [int] のテスト
        ///</summary>
        ///<remarks>
        /// 存在しないindexを指定して取得した場合、
        /// ConfigurationErrorsExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ThisIntTest02()
        {
            //準備
            ConvertCollection target = new ConvertCollection();
            ConvertElement expectedElement = new ConvertElement();
            target.Add(expectedElement);
            try
            {
                //実行
                ConvertElement actualElement = target[2];

                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ConfigurationErrorsException)
            {
                //成功
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ConvertCollection), LogLevel.INFO));
        }

        ///<summary>
        /// this [int] のテスト
        ///</summary>
        ///<remarks>
        /// ConvertElementを複数格納し、indexを指定して要素を取得できること確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ThisIntTest03()
        {
            //準備
            ConvertCollection target = new ConvertCollection();
            ConvertElement expectedElement1 = new ConvertElement("element1", null, null);
            ConvertElement expectedElement2 = new ConvertElement("element2", null, null);
            ConvertElement expectedElement3 = new ConvertElement("element3", null, null);
            target.Add(expectedElement1);
            target.Add(expectedElement2);
            target.Add(expectedElement3);
            int expectedCount = 3;

            //実行
            ConvertElement actualElement1 = target[0];
            ConvertElement actualElement2 = target[1];
            ConvertElement actualElement3 = target[2];

            //検証
            Assert.AreEqual(expectedCount, target.Count);
            Assert.AreSame(expectedElement1, actualElement1);
            Assert.AreSame(expectedElement2, actualElement2);
            Assert.AreSame(expectedElement3, actualElement3);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ConvertCollection), LogLevel.INFO));
        }


        ///<summary>
        /// this [string] のテスト
        ///</summary>
        ///<remarks>
        /// ConvertElementを格納し、Idを指定して取得した場合、一致すること確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ThisStringTest01()
        {
            //準備
            ConvertCollectionImpl target = new ConvertCollectionImpl();
            string expectedId = "id0001";
            ParamElement paramElement = new ParamElement();
            ResultElement resultElement = new ResultElement();
            ConvertElement element = new ConvertElement(expectedId, paramElement, resultElement);
            target.Add(element);

            //実行
            ConvertElement actualElement = target[expectedId];

            //検証
            Assert.AreEqual(element, actualElement);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ConvertCollection), LogLevel.INFO));
        }

        ///<summary>
        /// this [string] のテスト
        ///</summary>
        ///<remarks>
        /// ConvertElementを格納し、存在しないIdを指定して取得した場合、
        /// nullが返却されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ThisStringTest02()
        {
            //準備
            ConvertCollectionImpl target = new ConvertCollectionImpl();
            string expectedId = "id0001";
            ParamElement paramElement = new ParamElement();
            ResultElement resultElement = new ResultElement();
            ConvertElement element = new ConvertElement(expectedId, paramElement, resultElement);
            target.Add(element);

            //実行
            ConvertElement actualElement = target["id9999"];

            //検証
            Assert.IsNull(actualElement);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ConvertCollection), LogLevel.INFO));
        }

        ///<summary>
        /// ElementName () のテスト
        ///</summary>
        ///<remarks>
        /// 引数無しでメソッドを呼び出した場合、CONVERT_NAMEが返却されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ElementNameTest01()
        {
            //準備
            ConvertCollectionImpl target = new ConvertCollectionImpl();

            //実行
            string actualElementName = target.ElementName;

            //検証
            Assert.AreEqual("convert", actualElementName);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ConvertCollection), LogLevel.INFO));
        }

        ///<summary>
        /// ConvertCollectionを継承するクラスです。
        ///</summary>
        class ConvertCollectionImpl : ConvertCollection
        {
            ///<summary>
            /// 基底クラスの同名のメソッドを呼び出します。
            ///</summary>
            public new ConfigurationElement CreateNewElement()
            {
                return base.CreateNewElement();
            }

            ///<summary>
            /// 基底クラスの同名のメソッドを呼び出します。
            ///</summary>
            public new object GetElementKey(ConfigurationElement element)
            {
                return base.GetElementKey(element);
            }

            ///<summary>
            /// 基底クラスの同名のメソッドを呼び出します。
            ///</summary>
            public new string ElementName
            {
                get
                {
                    return base.ElementName;
                }
            }
        }
    }



}
