// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using System;
using System.Collections.Generic;
using System.IO;
using System.Net;
using System.Text;
using TERASOLUNA.Fw.Common.Logging;

namespace TERASOLUNA.Fw.Client.Communication
{
    /// <summary>
    /// }`p[gM@\񋟂NXłB
    /// </summary>
    public class MultipartSender : SenderBase<IList<MultipartElement>>
    {
        /// <summary>
        /// <see cref="ILog"/> NX̃CX^XłB
        /// </summary>
        /// <remarks>
        /// Oo͂ɗp܂B
        /// </remarks>
        private static ILog _log = LogFactory.GetLogger(typeof(MultipartSender));

        /// <summary>
        /// }`p[gf[^̋Eݒ肷ۂɎgp镶łB
        /// </summary>
        /// <remarks>
        /// <para>̒萔̒l "--{0}" łB</para>
        /// </remarks>
        protected static readonly string STARTLINE = "--{0}";

        /// <summary>
        /// Mf[^̏Iݒ肷ۂɎgp镶łB
        /// </summary>
        /// <remarks>
        /// <para>̒萔̒l "--{0}--" łB</para>
        /// </remarks>
        protected static readonly string ENDLINE = "--{0}--";

        /// <summary>
        /// NGXgwb_Content-Typeݒ肷ۂɎgp镶łB
        /// </summary>
        /// <remarks>
        /// <para>̒萔̒l "multipart/form-data; boundary={0}; charset={1}" łB</para>
        /// <para>ϐɁA؂蕶ݒ肵Ďgp܂B</para>
        /// </remarks>
        protected static readonly string CONTENT_TYPE_MULTIPART = "multipart/form-data; boundary={0}; charset={1}";

        /// <summary>
        /// ؂蕶񂪑MΏۂɑ݂ȂƂ`FbN邩ǂ̃tOłB
        /// </summary>
        /// <remarks>
        /// <para>ftHg̒l <c>true</c> łB</para>
        /// </remarks>
        private bool _checkBoundary = true;

        /// <summary>
        /// Abv[hΏۂ̃eLXgGR[h邽߂EncodingłB
        /// </summary>
        /// <remarks>
        /// <para>ftHg̒l "Encoding.UTF8" łB</para>
        /// </remarks>
        private Encoding _encoding = Encoding.UTF8;

        /// <summary>
        /// ؂蕶񂪑MΏۂɑ݂ȂƂ`FbN邩ǂ̃tO擾܂͐ݒ肵܂B
        /// </summary>
        /// <remarks>
        /// <para> <c>true</c> ̎ `FbN܂B</para>
        /// <para> <c>false</c> ̎ `FbN܂B</para>
        /// <para>l true łB</para>
        /// </remarks>
        public bool CheckBoundary
        {
            get
            {
                return _checkBoundary;
            }
            set
            {
                _checkBoundary = value;
            }
        }

        /// <summary>
        /// Abv[hΏۂ̃eLXgGR[h邽߂ <see cref="Encoding"/> 擾܂͐ݒ肵܂B
        /// </summary>
        /// <remarks>
        /// <para>l UTF-8 `łB</para>
        /// </remarks>
        public Encoding Encoding
        {
            get
            {
                return _encoding;
            }
            set
            {
                if (value == null)
                {
                    _encoding = Encoding.UTF8;
                }
                else
                {
                    _encoding = value;
                }
            }
        }

        /// <summary>
        /// Mɗp Content-Type wb_̒l擾܂B̃vpeB͕K <c>null</c> QƂԂ܂B
        /// </summary>
        /// <remarks>
        ///  <c>null</c> QƂԂ܂B<see cref="MultipartSender"/>  Content-Type wb_ɃoE_
        /// ܂񂾕 <see cref="SendRequest"/> \bhō쐬Đݒ肵܂B
        /// </remarks>
        public override string ContentType
        {
            get
            {
                return null;
            }
        }

        /// <summary>
        /// <see cref="MultipartSender"/> NX̐VCX^X܂B
        /// </summary>
        /// <remarks>
        /// ftHgRXgN^łB
        /// </remarks>
        public MultipartSender()
        {
        }

        /// <summary>
        /// <paramref name="request"/> œnꂽ <see cref="HttpWebRequest"/> pāA
        /// <paramref name="elements"/>  <paramref name="headerCollection"/> ̏
        /// M܂B
        /// </summary>
        /// <remarks>
        /// <para>
        /// M HTTP NGXg{fB <paramref name="elements"/>  <see cref="IList{MultipartElement}"/> 
        /// Ɋi[ꂽ <see cref="MultipartElement"/> ̃CX^Xꂼ HTTP wb_AHTTP {fB
        /// gݍ킹ƂăVACYAdȂ؂(oE_)ɂĕ̂łB
        /// </para>
        /// <para>
        /// }`p[g`ɂẮA RFC 1867 ɏ܂B
        /// </para>
        /// <para>
        /// HTTP NGXgwb_ɂ <paramref name="headerCollection"/> Ɋi[ꂽL[/l̏񂪑SĒǉ܂B
        /// <paramref name="headerCollection"/> Ɋi[L[/l͂ꂼ HTTP NGXgwb_̃L[/lƂėp
        /// \Ȍ`𖞑Kv܂B͋ḰARFC 2616 "request-header" Œ`Ă܂B
        /// </para>
        /// <para>
        /// <paramref name="reporter"/>  <c>null</c> QƂł͂ȂꍇA
        /// <see cref="SenderBase{TParam}.BufferSize"/> vpeBɐݒ肳ꂽobt@TCỸf[^iftHg8 KB jT[oɑM閈ɁA
        /// <see cref="IProgressChangeReporter.ReportProgressChanged"/> Cxg𔭐܂B
        /// </para>
        /// </remarks>
        /// <param name="request">MsNGXgIuWFNgB</param>
        /// <param name="paramData">Mf[^i[IuWFNgB</param>
        /// <param name="headerCollection">M HTTP wb_Ƃėpwb_̃RNVB</param>
        /// <param name="reporter">is󋵒ʒms <see cref="IProgressChangeReporter"/> CX^XB </param> 
        /// <exception cref="ArgumentException">
        /// ȉ̂悤ȏꍇɗOX[܂B
        /// <list type="bullet">
        /// <item>
        /// <paramref name="paramData"/> vf܂ł܂B
        /// </item>
        /// <item>
        /// <paramref name="paramData"/> Ɋ܂܂ <see cref="MultipartElement"/>  
        /// <see cref="MultipartElement.Name"/> ̒ldĂ܂B
        /// </item>
        /// </list>
        /// </exception>
        /// <exception cref="System.IO.FileNotFoundException">
        /// ʐMŕKvȃt@C܂B
        /// </exception>
        /// <exception cref="WebException">
        /// ʐMG[܂B܂́ANGXgLZ܂B
        /// </exception>
        protected override void SendRequest(HttpWebRequest request,
                                            IList<MultipartElement> paramData,
                                            IDictionary<string, string> headerCollection,
                                            IProgressChangeReporter reporter)
        {

            if (paramData.Count == 0)
            {
                string message = string.Format(Properties.Resources.E_EMPTY_ILIST, "paramData");
                ArgumentException exception = new ArgumentException(message);
                if (_log.IsErrorEnabled)
                {
                    _log.Error(message, exception);
                }
                throw exception;
            }

            // }`p[gvf̖Ȍd`FbN
            string[] tempList = new string[paramData.Count];
            int i = 0;
            foreach (MultipartElement element in paramData)
            {
                if (element == null)
                {
                    string message = string.Format(Properties.Resources.E_NULL_PARAMETER, typeof(MultipartElement).Name);
                    ArgumentException exception = new ArgumentException(message);
                    if (_log.IsErrorEnabled)
                    {
                        _log.Error(message, exception);
                    }
                    throw exception;
                }

                if (Array.IndexOf<string>(tempList, element.Name) != -1)
                {

                    ArgumentException exception = new ArgumentException(
                        string.Format(Properties.Resources.E_MULTIPART_NAME_REPEATED, element.Name));
                    if (_log.IsErrorEnabled)
                    {
                        _log.Error(exception.Message, exception);
                    }
                    throw exception;
                }
                tempList[i++] = element.Name;
            }

            // oE_
            string boundaryKey = CreateBoundary();
            if (CheckBoundary)
            {
                while (ExistsBoundaryKey(paramData, boundaryKey))
                {
                    boundaryKey = CreateBoundary();
                }
            }

            // Content-TypeɃoE_L[i[
            request.ContentType = string.Format(CONTENT_TYPE_MULTIPART, boundaryKey, this.Encoding.HeaderName);

            // oE_񂩂AoCgƂ̃TCY擾
            string boundaryString = string.Format(STARTLINE, boundaryKey);
            byte[] boundaryBytes = this.Encoding.GetBytes(boundaryString);
            int boundaryLength = boundaryBytes.Length;

            // oE_ɏI[q쐬AoCgƂ̃TCY擾
            string endBoundaryString = string.Format(ENDLINE, boundaryKey);
            byte[] endBoundaryBytes = this.Encoding.GetBytes(endBoundaryString);
            int endBoundaryBytesLength = endBoundaryBytes.Length;

            // sAoCgƂ̃TCY擾
            byte[] newLineBytes = this.Encoding.GetBytes("\r\n");
            int newLineLength = newLineBytes.Length;

            // Mɗpobt@TCYƁAobt@z쐬
            int localBufferSize = BufferSize;
            byte[] buffer = new byte[localBufferSize];

            // M
            using (Stream reqStream = request.GetRequestStream())
            {

                // M̃Oo
                if (_log.IsTraceEnabled)
                {
                    // URL
                    _log.Trace(string.Format(
                        Properties.Resources.T_REQUEST_SEND_ADDRESS, request.Address));

                    // HTTPwb_
                    StringBuilder requestHeaders = new StringBuilder();
                    requestHeaders.AppendLine(Properties.Resources.T_REQUEST_SEND_HEADER);
                    foreach (string key in request.Headers.AllKeys)
                    {
                        requestHeaders.AppendLine(string.Format(
                        Properties.Resources.T_DICTIONARY_KEY_VALUE, key, request.Headers[key]));
                    }
                    _log.Trace(requestHeaders.ToString().Trim());

                    StringBuilder requestBody = new StringBuilder();
                    requestBody.AppendLine(Properties.Resources.T_REQUEST_SEND_BODY);
                    foreach(MultipartElement element in paramData)
                    {
                        if (element.GetType().Name.Equals(typeof(MultipartValueElement).Name))
                        {
                            MultipartValueElement valueElement = element as MultipartValueElement;
                            requestBody.AppendLine(string.Format(Properties.Resources.T_MULTIPART_VALUE, valueElement.Name, valueElement.Value));
                        }
                        else
                        {
                            MultipartFileElement fileElement = element as MultipartFileElement;
                            requestBody.AppendLine(string.Format(Properties.Resources.T_MULTIPART_FILE, fileElement.Name, fileElement.UploadFilePath));
                        }
                    }
                    _log.Trace(requestBody.ToString().Trim());
                }

                int elementsSize = paramData.Count;
                for (int current = 0; current < elementsSize; current++)
                {
                    int percentage = 0; // Ŝ̐i
                    long sendSize = 0L; // X̃}`p[gf[^̑Mς݂̃oCg
                    long totalSize = 0L;

                    MultipartElement element = paramData[current];

                    // }`p[gf[^̃wb_̏
                    string headerString = element.CreateHeader();
                    byte[] headerBytes = this.Encoding.GetBytes(headerString);
                    long headerBytesLength = headerBytes.Length;

                    // ؂蕶𑗐M
                    reqStream.Write(boundaryBytes, 0, boundaryLength);
                    reqStream.Write(newLineBytes, 0, newLineLength);

                    // }`p[gf[^̃wb_𑗐M
                    reqStream.Write(headerBytes, 0, headerBytes.Length);
                    reqStream.Write(newLineBytes, 0, newLineLength);
                    reqStream.Write(newLineBytes, 0, newLineLength);

                    sendSize =
                        headerBytesLength + boundaryLength + newLineLength + newLineLength + newLineLength;

                    using (Stream streamBody = element.CreateBodyStream(this.Encoding))
                    {

                        // Mf[^ʂ̓{fBTCYɉāAwb_TCYƂ
                        totalSize = streamBody.Length + sendSize;

                        // Ō̃}`p[gf[^ł΁AI[qTCYvZɊ܂߂
                        if (current >= paramData.Count - 1)
                        {
                            totalSize += endBoundaryBytesLength + newLineLength + newLineLength;
                        }

                        int readSize = streamBody.Read(buffer, 0, localBufferSize);
                        while (readSize > 0)
                        {
                            reqStream.Write(buffer, 0, readSize);
                            sendSize += readSize;
                            if (reporter != null)
                            {
                                if (totalSize > 0)
                                {
                                    percentage =
                                        CalcSendPercentage(current, paramData.Count, sendSize, totalSize);
                                }
                                reporter.ReportProgressChanged(new ExecuteProgressChangedEventArgs(percentage));
                            }
                            readSize = streamBody.Read(buffer, 0, localBufferSize);
                        }
                    }

                    reqStream.Write(newLineBytes, 0, newLineLength);
                    sendSize += newLineLength;
                }

                // SẴ}`p[gf[^𑗂AI[qo
                reqStream.Write(endBoundaryBytes, 0, endBoundaryBytesLength);
                reqStream.Write(newLineBytes, 0, newLineLength);

                if (reporter != null)
                {
                    reporter.ReportProgressChanged(new ExecuteProgressChangedEventArgs(50));
                }
            }
        }

        /// <summary>
        /// }`p[gf[^M󋵂̐ivZ܂B
        /// </summary>
        /// <remarks>
        /// <para>}`p[gf[^̃Abv[h̐iłB</para>
        /// <para>MSĊƁAi 50% ɂȂ܂B</para>
        /// <para><paramref name="current"/> ̒l 2147483 𒴂邱ƂȂ悤ɂĂB ܂A
        /// <paramref name="max"/> ̒l 1073741824 𒴂邱ƂȂ悤ɂĂB</para>
        /// </remarks>
        /// <param name="current">ڂ̃}`p[gf[^\܂B( 0 ԖڂX^[g)</param>
        /// <param name="max">M}`p[gf[^̌łB</param>
        /// <param name="sendDataSize">MĂ}`p[gf[^̑Mς݂̃oCgłB</param>
        /// <param name="length">MĂ}`p[gf[^̑oCgłB</param>
        /// <exception cref="ArgumentOutOfRangeException">
        /// <paramref name="max"/> ܂ <paramref name="length"/>  0 łB
        /// </exception>
        /// <returns>
        /// M̐iB
        /// </returns>
        /// <exception cref="OverflowException">
        /// MTCỸp[Ze[W̌vZŃI[o[t[܂B
        /// </exception>
        protected virtual int CalcSendPercentage(int current, int max, long sendDataSize, long length)
        {
            if (max == 0 || length == 0)
            {
                ArgumentOutOfRangeException exception = new ArgumentOutOfRangeException();
                if (_log.IsErrorEnabled)
                {
                    _log.Error(exception.Message, exception);
                }
                throw exception;
            }

            int percentage = 0;
            checked
            {
                percentage = (int)(sendDataSize * 100L / length);
                percentage = (percentage + 100 * current) / (max * 2);
            }
            return percentage;
        }

        /// <summary>
        /// <paramref name="boundaryKey"/>  <paramref name="elements"/> ̃}`p[gf[^ɑ݂邩
        /// ǂmF܂B
        /// </summary>
        /// <remarks>
        /// <para><paramref name="elements"/> ̑SẴ}`p[gf[^̃wb_A{fB`FbN܂B
        /// </para>
        /// <para>݂ꍇɂ true A݂Ȃꍇɂ false Ԃ܂B</para>
        /// </remarks>
        /// <param name="elements"><see cref="MultipartElement"/> i[XgłB</param>
        /// <param name="boundaryKey">؂蕶łB</param>
        /// <exception cref="ArgumentNullException">
        /// <para><paramref name="boundaryKey"/>  <c>null</c> QƂłB</para>
        /// <para><paramref name="elements"/>  <c>null</c> QƂłB</para>
        /// </exception>
        /// <exception cref="ArgumentException">
        /// <para><paramref name="boundaryKey"/> 󕶎łB</para>
        /// <para><paramref name="elements"/>  0łB</para>
        /// </exception>
        /// <returns><paramref name="boundaryKey"/>  <paramref name="elements"/> ̃}`p[gf[^
        /// ݂ꍇ true A݂Ȃꍇɂ false B
        /// </returns>
        protected virtual bool ExistsBoundaryKey(IList<MultipartElement> elements, string boundaryKey)
        {
            byte[] boundaryBytes = this.Encoding.GetBytes(boundaryKey);
            bool result = false;

            foreach (MultipartElement m in elements)
            {
                // wb_mF܂B
                string multipartHeader = m.CreateHeader();
                if (multipartHeader.Contains(boundaryKey))
                {
                    result = true;
                    break;
                }
                else
                {
                    using (Stream multipartBodyStream = m.CreateBodyStream(this.Encoding))
                    {
                        if (ExistsBytesFromStream(multipartBodyStream, boundaryBytes))
                        {
                            result = true;
                            break;
                        }
                    }
                }
            }
            return result;
        }


        /// <summary>
        /// <paramref name="stream"/> ŗ^ꂽXg[A<paramref name="target"/> 
        /// Ɠ̃oCg񂪓ǂ݂邩mF܂B
        /// </summary>
        /// <remarks>
        /// <para>
        /// ̃\bh̓}`p[gf[^ɋ؂蕶񂪊܂܂邩ǂmF邽߂ɗp
        /// wp[\bhłB
        /// </para>
        /// <para>
        /// <paramref name="stream"/>̓V[N\ȃXg[łKv܂B
        /// </para>
        /// </remarks>
        /// <param name="stream">Ώۂ̃Xg[B</param>
        /// <param name="target">oCgB</param>
        /// <returns><paramref name="stream"/>  <paramref name="target"/> Ɠ̃oCg񂪓ǂݏoꂽ
        /// ꍇɂ true AXg[̍Ō܂œǂݏoȂ falseB</returns>
        protected static bool ExistsBytesFromStream(Stream stream, byte[] target)
        {
            if (stream == null)
            {
                return false;
            }

            if (target.Length <= 0 || stream.Length < target.Length)
            {
                return false;
            }

            bool[] checkFlags = new bool[target.Length + 1];
            for (int i = 0; i < checkFlags.Length; i++)
            {
                checkFlags[i] = false;
            }

            checkFlags[0] = true;
            bool result = false;

            byte[] buf = new byte[1];
            int nextTargetPosition = 0;
            // 1 byte擾
            while (stream.Read(buf, 0, 1) > 0)
            {
                int targetPosition = nextTargetPosition;
                nextTargetPosition = 0;
                for (int searchPosition = targetPosition; 0 <= searchPosition; searchPosition--)
                {
                    if (checkFlags[searchPosition])
                    {
                        // O̒lvꍇ
                        if (buf[0] == target[searchPosition])
                        {
                            // lvꍇA̒lΏۂƂ
                            checkFlags[searchPosition + 1] = true;
                            if (nextTargetPosition == 0)
                            {
                                // ΏۃJE^1グ
                                nextTargetPosition = searchPosition + 1;
                            }
                        }
                        else
                        {
                            // lvȂꍇA̒ľ߂
                            checkFlags[searchPosition + 1] = false;
                        }
                    }
                    else
                    {
                        // ΏۊȌꍇA̒lΏۂO
                        checkFlags[searchPosition + 1] = false;
                    }
                }

                // Ō̒l܂œBA[v𔲂
                if (nextTargetPosition == target.Length)
                {
                    result = true;
                    break;
                }
            }
            return result;
        }

        /// <summary>؂蕶𐶐܂B</summary>
        /// <remarks>
        /// O[oӎʎq(GUID)pāA36̃_ȕԂ܂B
        /// </remarks>
        /// <returns>
        /// ؂蕶ƂĎgp镶ԂB
        /// </returns>
        protected virtual string CreateBoundary()
        {
            // O[oӎʎq(GUID)pč쐬܂B
            return System.Guid.NewGuid().ToString();
        }
    }
}
