// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Text;
using System.Collections.Generic;
using TERASOLUNA.Fw.Client.Forms;
using TERASOLUNA.Fw.Utlib.Logging;
using TERASOLUNA.Fw.Common.Logging;
using TERASOLUNA.Fw.Common;
using TERASOLUNA.Fw.Client.Configuration.View;
using TERASOLUNA.Fw.ClientUT.Configuration;
using System.Configuration;

namespace TERASOLUNA.Fw.ClientUT.Forms
{
    /// <summary>
    ///TERASOLUNA.Fw.Client.Forms.ViewFactory のテスト クラスです。すべての
    ///TERASOLUNA.Fw.Client.Forms.ViewFactory 単体テストをここに含めます
    ///</summary>
    [TestClass()]
    public class ViewFactoryTest
    {
        ///<summary>
        /// 正常なviewIdを表す文字列です。
        ///</summary>
        private readonly string VIEW_ID = "view";

        ///<summary>
        /// 不正なviewIdを表す文字列です。
        ///</summary>
        private readonly string ILLEGAL_VIEW_ID = "illegalView";

        ///<summary>
        /// ViewFactoryを継承するクラスを表す文字列です。
        ///</summary>
        private readonly string FACTORY_TYPE = "TERASOLUNA.Fw.ClientUT.Forms.ViewFactory_ViewFactoryStub01,TERASOLUNA.Fw.ClientUT";

        ///<summary>
        /// ViewFactoryを継承しないクラスを表す文字列です。
        ///</summary>
        private readonly string NOT_FACTORY_TYPE = "TERASOLUNA.Fw.ClientUT.Forms.ViewFactory_Stub01,TERASOLUNA.Fw.ClientUT";

        ///<summary>
        /// テストデータ01を表す文字列です。
        ///</summary>
        private readonly string CONFIG_FILE_NAME_01 = "TestItem_ViewFactory01.config";

        ///<summary>
        /// テストデータ02を表す文字列です。
        ///</summary>
        private readonly string CONFIG_FILE_NAME_02 = "TestItem_ViewFactory02.config";

        ///<summary>
        /// テストデータ02を表す文字列です。
        ///</summary>
        private readonly string CONFIG_FILE_NAME_03 = "TestItem_ViewFactory03.config";

        private TestContext testContextInstance;

        /// <summary>
        ///現在のテストの実行についての情報および機能を
        ///提供するテスト コンテキストを取得または設定します。
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }
        #region 追加のテスト属性
        // 
        //テストを作成するときに、次の追加属性を使用することができます:
        //
        //クラスの最初のテストを実行する前にコードを実行するには、ClassInitialize を使用
        //
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //クラスのすべてのテストを実行した後にコードを実行するには、ClassCleanup を使用
        //
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //各テストを実行する前にコードを実行するには、TestInitialize を使用
        //
        [TestInitialize()]
        public void MyTestInitialize()
        {
            ConfigurationManager.AppSettings[TERASOLUNA_Fw_Client_Forms_ViewFactoryAccessor.VIEW_FACTORY_TYPENAME] = null;
            TERASOLUNA_Fw_Client_Forms_ViewFactoryAccessor._factory = null;
            TERASOLUNA_Fw_Client_Configuration_TerasolunaClientConfigurationAccessor.VIEW_CONFIGURATION_NAME = "viewConfiguration";
            TERASOLUNA_Fw_Client_Configuration_TerasolunaClientConfigurationAccessor._viewConfigSection = null;
            LogUTUtil.Clear();
        }
        //
        //各テストを実行した後にコードを実行するには、TestCleanup を使用
        //
        [TestCleanup()]
        public void MyTestCleanup()
        {
            ConfigurationManager.AppSettings[TERASOLUNA_Fw_Client_Forms_ViewFactoryAccessor.VIEW_FACTORY_TYPENAME] = null;
            TERASOLUNA_Fw_Client_Forms_ViewFactoryAccessor._factory = null;
            TERASOLUNA_Fw_Client_Configuration_TerasolunaClientConfigurationAccessor.VIEW_CONFIGURATION_NAME = "viewConfiguration";
            TERASOLUNA_Fw_Client_Configuration_TerasolunaClientConfigurationAccessor._viewConfigSection = null;
        }
        //
        #endregion

        ///<summary>
        /// ViewFactory () のテスト
        ///</summary>
        ///<remarks>
        /// 引数無しのコンストラクタを呼び出した場合、インスタンスが返却されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ConstructorTest01()
        {
            // 実行
            ViewFactory target = new ViewFactory();

            // 検証
            TERASOLUNA_Fw_Client_Forms_ViewFactoryAccessor accessor =
                 new TERASOLUNA_Fw_Client_Forms_ViewFactoryAccessor(target);
            Assert.IsNotNull(accessor.ForwardableTypes);
            Assert.AreEqual(0, accessor.ForwardableTypes.Count);
            Assert.IsNull(TERASOLUNA_Fw_Client_Forms_ViewFactoryAccessor._factory);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ViewFactory), LogLevel.INFO));
        }

        /// <summary>
        /// CreateView (string) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の viewId が nullの場合、
        /// ArgumentNullException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateViewTest01()
        {
            //準備
            string viewId = null;

            try
            {
                //実行
                IForwardable actual = ViewFactory.CreateView(viewId);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // 成功
                Assert.AreEqual("viewId", e.ParamName);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ViewFactory), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(
                typeof(ViewFactory), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "viewId"));
        }


        /// <summary>
        /// CreateView (string) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の viewId が空文字の場合、
        /// ArgumentNullException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateViewTest02()
        {
            //準備
            string viewId = string.Empty;

            try
            {
                //実行
                IForwardable actual = ViewFactory.CreateView(viewId);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // 確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_EMPTY_STRING", "viewId"), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ViewFactory), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(
                typeof(ViewFactory), typeof(ArgumentException), "E_EMPTY_STRING", "viewId"));
        }


        /// <summary>
        /// CreateView (string) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の viewId がnot nullの場合、
        /// 正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Forms\ViewFactoryTestData\TestItem_ViewFactory01.config")]
        [DeploymentItem(@"Forms\ViewFactoryTestData\TestItem_ViewFactoryExternal01.config")]
        public void CreateViewTest03()
        {
            //準備
            string viewId = VIEW_ID;

            ViewConfigurationSection configSection = ConfigTestHelper.GetViewConfigurationSectionFromCustomFile(CONFIG_FILE_NAME_01);
            configSection.CreateViewElements();
            TERASOLUNA_Fw_Client_Configuration_TerasolunaClientConfigurationAccessor._viewConfigSection = configSection;

            //実行
            IForwardable actual = ViewFactory.CreateView(viewId);

            // 確認
            Assert.IsInstanceOfType(actual, typeof(ViewFactory_IForwardableImpl01));
            Assert.IsInstanceOfType(TERASOLUNA_Fw_Client_Forms_ViewFactoryAccessor._factory, typeof(ViewFactory));
            TERASOLUNA_Fw_Client_Forms_ViewFactoryAccessor accessor = new TERASOLUNA_Fw_Client_Forms_ViewFactoryAccessor(TERASOLUNA_Fw_Client_Forms_ViewFactoryAccessor._factory);
            Assert.AreEqual(1, accessor.ForwardableTypes.Count);
            Assert.AreEqual(typeof(ViewFactory_IForwardableImpl01), accessor.ForwardableTypes[viewId]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ViewFactory), LogLevel.INFO));
        }


        /// <summary>
        /// CreateView (string) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の viewId がnot nullかつ、_factoryがnot nullの場合に
        /// 正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Forms\ViewFactoryTestData\TestItem_ViewFactory01.config")]
        [DeploymentItem(@"Forms\ViewFactoryTestData\TestItem_ViewFactoryExternal01.config")]
        public void CreateViewTest04()
        {
            //準備
            string viewId = VIEW_ID;

            ViewConfigurationSection configSection = ConfigTestHelper.GetViewConfigurationSectionFromCustomFile(CONFIG_FILE_NAME_01);
            configSection.CreateViewElements();
            TERASOLUNA_Fw_Client_Configuration_TerasolunaClientConfigurationAccessor._viewConfigSection = configSection;

            // _factory のキャッシュ
            ViewFactory expectedFactory = new ViewFactory();
            TERASOLUNA_Fw_Client_Forms_ViewFactoryAccessor._factory = expectedFactory;

            //実行
            IForwardable actual = ViewFactory.CreateView(viewId);

            // 確認
            Assert.IsInstanceOfType(actual, typeof(ViewFactory_IForwardableImpl01));
            Assert.IsInstanceOfType(TERASOLUNA_Fw_Client_Forms_ViewFactoryAccessor._factory, typeof(ViewFactory));
            Assert.AreSame(expectedFactory, TERASOLUNA_Fw_Client_Forms_ViewFactoryAccessor._factory);
            TERASOLUNA_Fw_Client_Forms_ViewFactoryAccessor accessor = new TERASOLUNA_Fw_Client_Forms_ViewFactoryAccessor(TERASOLUNA_Fw_Client_Forms_ViewFactoryAccessor._factory);
            Assert.AreEqual(1, accessor.ForwardableTypes.Count);
            Assert.AreEqual(typeof(ViewFactory_IForwardableImpl01), accessor.ForwardableTypes[viewId]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ViewFactory), LogLevel.INFO));
        }

        /// <summary>
        /// CreateView (string) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の viewId がnot nullかつ、_factoryがnullの場合に、
        /// _forwardableTypes のキーに blogicName と同じ文字列が存在する場合、
        /// 正常に実行することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Forms\ViewFactoryTestData\TestItem_ViewFactory01.config")]
        [DeploymentItem(@"Forms\ViewFactoryTestData\TestItem_ViewFactoryExternal01.config")]
        public void CreateViewTest05()
        {
            //準備
            string viewId = VIEW_ID;

            ViewConfigurationSection configSection = ConfigTestHelper.GetViewConfigurationSectionFromCustomFile(CONFIG_FILE_NAME_01);
            configSection.CreateViewElements();
            TERASOLUNA_Fw_Client_Configuration_TerasolunaClientConfigurationAccessor._viewConfigSection = configSection;

            ViewFactory expectedFactory = new ViewFactory();
            TERASOLUNA_Fw_Client_Forms_ViewFactoryAccessor accessor = new TERASOLUNA_Fw_Client_Forms_ViewFactoryAccessor(expectedFactory);
            Type type = typeof(ViewFactory_IForwardableImpl01);
            accessor.ForwardableTypes[viewId] = type;

            //実行
            IForwardable actual = ViewFactory.CreateView(viewId);

            // 確認
            Assert.IsInstanceOfType(actual, typeof(ViewFactory_IForwardableImpl01));
            Assert.IsInstanceOfType(TERASOLUNA_Fw_Client_Forms_ViewFactoryAccessor._factory, typeof(ViewFactory));
            Assert.AreSame(expectedFactory, accessor.Target);
            Assert.AreEqual(1, accessor.ForwardableTypes.Count);
            Assert.AreEqual(typeof(ViewFactory_IForwardableImpl01), accessor.ForwardableTypes[viewId]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ViewFactory), LogLevel.INFO));
        }


        /// <summary>
        /// CreateView (string) のテスト
        /// </summary>
        /// <remarks>
        /// 構成ファイルに設定する ViewFactoryTypeName が ViewFactory 派生クラスの
        /// のインスタンスを生成できない設定の場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Forms\ViewFactoryTestData\TestItem_ViewFactory01.config")]
        [DeploymentItem(@"Forms\ViewFactoryTestData\TestItem_ViewFactoryExternal01.config")]
        public void CreateViewTest06()
        {
            //準備
            string viewId = VIEW_ID;

            ConfigurationManager.AppSettings[TERASOLUNA_Fw_Client_Forms_ViewFactoryAccessor.VIEW_FACTORY_TYPENAME] = NOT_FACTORY_TYPE;

            ViewConfigurationSection configSection = ConfigTestHelper.GetViewConfigurationSectionFromCustomFile(CONFIG_FILE_NAME_01);
            configSection.CreateViewElements();
            TERASOLUNA_Fw_Client_Configuration_TerasolunaClientConfigurationAccessor._viewConfigSection = configSection;

            try
            {
                //実行
                IForwardable actual = ViewFactory.CreateView(viewId);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // 成功
                string exceptionMessage = MessageManager.GetMessage("E_INVALID_CAST_TYPE_FROM_GENERICPARAM",
                    typeof(ViewFactory_Stub01).AssemblyQualifiedName,
                    typeof(ViewFactory).AssemblyQualifiedName);
                Assert.AreEqual(exceptionMessage, e.Message);

            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ViewFactory), LogLevel.INFO));
        }

        /// <summary>
        /// CreateView (string) のテスト
        /// </summary>
        /// <remarks>
        /// 構成ファイルに設定する ViewFactoryTypeName が ViewFactory 派生クラスの
        /// のインスタンスを生成できる設定の場合、
        /// 正常に実行することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Forms\ViewFactoryTestData\TestItem_ViewFactory01.config")]
        [DeploymentItem(@"Forms\ViewFactoryTestData\TestItem_ViewFactoryExternal01.config")]
        public void CreateViewTest07()
        {
            //準備
            string viewId = VIEW_ID;

            ConfigurationManager.AppSettings[TERASOLUNA_Fw_Client_Forms_ViewFactoryAccessor.VIEW_FACTORY_TYPENAME] = FACTORY_TYPE;

            ViewConfigurationSection configSection = ConfigTestHelper.GetViewConfigurationSectionFromCustomFile(CONFIG_FILE_NAME_01);
            configSection.CreateViewElements();
            TERASOLUNA_Fw_Client_Configuration_TerasolunaClientConfigurationAccessor._viewConfigSection = configSection;

            //実行
            IForwardable actual = ViewFactory.CreateView(viewId);

            // 確認
            Assert.IsInstanceOfType(actual, typeof(ViewFactory_IForwardableImpl01));
            Assert.IsInstanceOfType(TERASOLUNA_Fw_Client_Forms_ViewFactoryAccessor._factory, typeof(ViewFactory_ViewFactoryStub01));
            TERASOLUNA_Fw_Client_Forms_ViewFactoryAccessor accessor = new TERASOLUNA_Fw_Client_Forms_ViewFactoryAccessor(TERASOLUNA_Fw_Client_Forms_ViewFactoryAccessor._factory);
            Assert.AreEqual(1, accessor.ForwardableTypes.Count);
            Assert.AreEqual(typeof(ViewFactory_IForwardableImpl01), accessor.ForwardableTypes[viewId]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ViewFactory), LogLevel.INFO));
        }


        /// <summary>
        /// CreateView (string) のテスト
        /// </summary>
        /// <remarks>
        /// 画面遷移設定ファイルを読み込むことができない場合、
        /// ConfigurationErrorsException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Forms\ViewFactoryTestData\TestItem_ViewFactory01.config")]
        [DeploymentItem(@"Forms\ViewFactoryTestData\TestItem_ViewFactoryExternal01.config")]
        public void CreateViewTest08()
        {
            //準備
            string viewId = VIEW_ID;

            TERASOLUNA_Fw_Client_Configuration_TerasolunaClientConfigurationAccessor.VIEW_CONFIGURATION_NAME = "dummyConfiguration";

            try
            {
                //実行
                IForwardable actual = ViewFactory.CreateView(viewId);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ConfigurationErrorsException)
            {
                // 成功
            }
           
            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ViewFactory), LogLevel.INFO));
        }


        /// <summary>
        /// CreateView (string) のテスト
        /// </summary>
        /// <remarks>
        /// 画面遷移設定ファイルに設定する view 要素の name 属性に、
        /// viewId と同じキーが存在しない場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Forms\ViewFactoryTestData\TestItem_ViewFactory01.config")]
        [DeploymentItem(@"Forms\ViewFactoryTestData\TestItem_ViewFactoryExternal01.config")]
        public void CreateViewTest09()
        {
            //準備
            string viewId = ILLEGAL_VIEW_ID;

            ViewConfigurationSection configSection = ConfigTestHelper.GetViewConfigurationSectionFromCustomFile(CONFIG_FILE_NAME_01);
            configSection.CreateViewElements();
            TERASOLUNA_Fw_Client_Configuration_TerasolunaClientConfigurationAccessor._viewConfigSection = configSection;

            try
            {
                //実行
                IForwardable actual = ViewFactory.CreateView(viewId);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // 成功
                string exceptionMessage = MessageManager.GetMessage("E_FORMS_FORM_TYPENAME_NOT_FOUND",viewId);
                Assert.AreEqual(exceptionMessage, e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ViewFactory), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(
                typeof(ViewFactory), typeof(TerasolunaException), "E_FORMS_FORM_TYPENAME_NOT_FOUND", viewId));

        }

        /// <summary>
        /// CreateView (string) のテスト
        /// </summary>
        /// <remarks>
        /// 画面遷移設定ファイルに設定する view 要素の name 属性に、
        /// viewId と同じキーが存在し、取得したタイプ名から型を解決できない場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Forms\ViewFactoryTestData\TestItem_ViewFactory02.config")]
        [DeploymentItem(@"Forms\ViewFactoryTestData\TestItem_ViewFactoryExternal02.config")]
        public void CreateViewTest10()
        {
            //準備
            string viewId = VIEW_ID;

            ViewConfigurationSection configSection = ConfigTestHelper.GetViewConfigurationSectionFromCustomFile(CONFIG_FILE_NAME_02);
            configSection.CreateViewElements();
            TERASOLUNA_Fw_Client_Configuration_TerasolunaClientConfigurationAccessor._viewConfigSection = configSection;

            try
            {
                //実行
                IForwardable actual = ViewFactory.CreateView(viewId);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // 成功
                string exceptionMessage = MessageManager.GetMessage("E_TYPE_LOAD_FAILED", "IllegalType");
                Assert.AreEqual(exceptionMessage, e.Message);
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ViewFactory), LogLevel.INFO));
          
        }


        /// <summary>
        /// CreateView (string) のテスト
        /// </summary>
        /// <remarks>
        /// 画面遷移設定ファイルに設定する view 要素の name 属性に、
        /// viewId と同じキーが存在し、取得したタイプ名から型を解決でき、
        /// IForwadable 実装クラスのインスタンスを生成できない場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Forms\ViewFactoryTestData\TestItem_ViewFactory03.config")]
        [DeploymentItem(@"Forms\ViewFactoryTestData\TestItem_ViewFactoryExternal03.config")]
        public void CreateViewTest11()
        {
            //準備
            string viewId = VIEW_ID;

            ViewConfigurationSection configSection = ConfigTestHelper.GetViewConfigurationSectionFromCustomFile(CONFIG_FILE_NAME_03);
            configSection.CreateViewElements();
            TERASOLUNA_Fw_Client_Configuration_TerasolunaClientConfigurationAccessor._viewConfigSection = configSection;

            try
            {
                //実行
                IForwardable actual = ViewFactory.CreateView(viewId);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // 成功
                string exceptionMessage = MessageManager.GetMessage("E_MEMBER_ACCESS_EXCEPTION",
                    typeof(ViewFactory_IForwardableImpl02).AssemblyQualifiedName);
                Assert.AreEqual(exceptionMessage, e.Message);
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ViewFactory), LogLevel.INFO));

        }
    }

    /// <summary>
    /// IForwardable を実装するスタブです。
    /// デフォルトコンストラクタが存在します。
    /// </summary>   
    public class ViewFactory_IForwardableImpl01 : IForwardable
    {
        public bool Init(IForwardable forwardableHost)
        {
            throw new Exception("ViewFactory_IForwardableImpl01");
        }

        public System.Collections.IDictionary Items
        {
            get
            {
                return null;
            }
        }

        public System.Data.DataSet ViewData
        {
            get
            {
                return null;
            }
        }
    }

    /// <summary>
    /// IForwardable を実装するスタブです。
    /// デフォルトコンストラクタが存在します。
    /// </summary>   
    public class ViewFactory_IForwardableImpl02 : IForwardable
    {
        public ViewFactory_IForwardableImpl02(string str)
        {
        }

        public bool Init(IForwardable forwardableHost)
        {
            throw new Exception("ViewFactory_IForwardableImpl01");
        }

        public System.Collections.IDictionary Items
        {
            get
            {
                return null;
            }
        }

        public System.Data.DataSet ViewData
        {
            get
            {
                return null;
            }
        }
    }


    /// <summary>
    /// ViewFactory を継承しないスタブです。
    /// デフォルトコンストラクタが存在します。
    /// </summary>
    public class ViewFactory_Stub01
    {
        public ViewFactory_Stub01()
        {
        }
    }

    /// <summary>
    /// ViewFactory を継承するスタブです。
    /// デフォルトコンストラクタが存在します。
    /// </summary>
    public class ViewFactory_ViewFactoryStub01 : ViewFactory
    {
        public ViewFactory_ViewFactoryStub01()
        {
        }
    }
}
