// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Text;
using System.Collections.Generic;
using System.Collections;
using System.Data;
using TERASOLUNA.Fw.Client.Forms;
using TERASOLUNA.Fw.Common.Validation;
using TERASOLUNA.Fw.Common;
using TERASOLUNA.Fw.Utlib.Logging;
using TERASOLUNA.Fw.Common.Logging;
using TERASOLUNA.Fw.ClientUT.Forms.FormBaseTestData;
using TERASOLUNA.Fw.Client;

namespace TERASOLUNA.Fw.ClientUT.Forms
{

    ///<summary>
    /// FormBase のテスト クラスです。
    ///</summary>
    [TestClass()]
    public class FormBaseTest
    {
        private TestContext testContextInstance;

        /// <summary>
        ///現在のテストの実行についての情報および機能を
        ///提供するテスト コンテキストを取得または設定します。
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }
        #region 追加のテスト属性
        // 
        //テストを作成するときに、次の追加属性を使用することができます:
        //
        //クラスの最初のテストを実行する前にコードを実行するには、ClassInitialize を使用
        //
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //クラスのすべてのテストを実行した後にコードを実行するには、ClassCleanup を使用
        //
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //各テストを実行する前にコードを実行するには、TestInitialize を使用
        //
        [TestInitialize()]
        public void MyTestInitialize()
        {
            LogUTUtil.Clear();
        }
        //
        //各テストを実行した後にコードを実行するには、TestCleanup を使用
        //
        //[TestCleanup()]
        //public void MyTestCleanup()
        //{
        //}
        //
        #endregion

        ///<summary>
        /// FormBase () のテスト
        ///</summary>
        ///<remarks>
        /// FormBaseのインスタンスがnullでないことを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ConstructorTest01()
        {
            FormBase target = new FormBase();

            Assert.IsNotNull(target);
            Assert.IsNull(target.ViewData);
            Assert.IsNotNull(target.Items);
            Assert.AreEqual(0, target.Items.Count);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(FormBase), LogLevel.INFO));
        }

        ///<summary>
        /// HandleError のテスト
        ///</summary>
        ///<remarks>
        /// resultStringがnullの場合、
        /// 例外ArgumentNullExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void HandleErrorTest01()
        {
            FormBaseEx target = new FormBaseEx();

            string resultString = null;
            IList<MessageInfo> messages = new List<MessageInfo>();
            object view = "terasoluna";

            try
            {
                // 実施
                target.HandleError(resultString, messages, view);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                Assert.AreEqual("resultString", e.ParamName);
            }

            // 確認
            Assert.IsFalse(target._handleValidationErrorCalled);

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(FormBase), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(FormBase), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "resultString"));
        }

        ///<summary>
        /// HandleError のテスト
        ///</summary>
        ///<remarks>
        /// resultStringが空文字の場合、
        /// 例外ArgumentExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void HandleErrorTest02()
        {
            FormBaseEx target = new FormBaseEx();

            string resultString = string.Empty;
            IList<MessageInfo> messages = new List<MessageInfo>();
            object view = "terasoluna";

            try
            {
                // 実施
                target.HandleError(resultString, messages, view);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_EMPTY_STRING", "resultString"), e.Message);
            }

            // 確認
            Assert.IsFalse(target._handleValidationErrorCalled);

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(FormBase), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(FormBase), typeof(ArgumentException), "E_EMPTY_STRING", "resultString"));
        }

        ///<summary>
        /// HandleError のテスト
        ///</summary>
        ///<remarks>
        /// resultStringがvalidationFailedの場合、かつviewがDataSetでない場合
        /// 例外TerasolunaExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void HandleErrorTest03()
        {
            FormBaseEx target = new FormBaseEx();

            string resultString = Coordinator.VALIDATION_FAILED;
            IList<MessageInfo> messages = new List<MessageInfo>();
            object view = "terasoluna";

            try
            {
                // 実施
                target.HandleError(resultString, messages, view);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_INVALID_CAST", view.GetType().FullName, typeof(DataSet).FullName), e.Message);
            }

            // 確認
            Assert.IsFalse(target._handleValidationErrorCalled);

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(FormBase), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(FormBase), typeof(TerasolunaException), "E_INVALID_CAST", view.GetType().FullName, typeof(DataSet).FullName));
        }

        ///<summary>
        /// HandleError のテスト
        ///</summary>
        ///<remarks>
        /// resultStringがvalidationFailedではなく、かつviewがDataSetでない場合
        /// 正常に終了することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void HandleErrorTest04()
        {
            FormBaseEx target = new FormBaseEx();

            string resultString = "TERASOLUNA";
            IList<MessageInfo> messages = new List<MessageInfo>();
            object view = "terasoluna";

            // 実施
            target.HandleError(resultString, messages, view);

            // 確認
            Assert.AreEqual("terasoluna", view);
            Assert.IsFalse(target._handleValidationErrorCalled);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(FormBase), LogLevel.INFO));
        }

        ///<summary>
        /// HandleError のテスト
        ///</summary>
        ///<remarks>
        /// messagesがnullの場合、
        /// 例外ArgumentExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void HandleErrorTest05()
        {
            FormBaseEx target = new FormBaseEx();

            string resultString = Coordinator.VALIDATION_FAILED;
            IList<MessageInfo> messages = null;
            object view = "terasoluna";

            try
            {
                // 実施
                target.HandleError(resultString, messages, view);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                Assert.AreEqual("messages", e.ParamName);
            }

            // 確認
            Assert.IsFalse(target._handleValidationErrorCalled);

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(FormBase), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(FormBase), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "messages"));
        }

        ///<summary>
        /// HandleError のテスト
        ///</summary>
        ///<remarks>
        /// resultStringがvalidationFailed、messagesの要素が0の場合、
        /// データセットにエラーが設定されないことを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void HandleErrorTest06()
        {
            FormBaseEx target = new FormBaseEx();

            string resultString = Coordinator.VALIDATION_FAILED;
            IList<MessageInfo> messages = new List<MessageInfo>();
            FormBaseDataSet01 view = new FormBaseDataSet01();
            view.DataTable1.AddDataTable1Row("row0col1", "row0,col2", "row0col3");
            view.DataTable1.AddDataTable1Row("row1col1", "row1,col2", "row1col3");
            view.DataTable1.AddDataTable1Row("row2col1", "row2,col2", "row2col3");
            view.AcceptChanges();

            // 実施
            target.HandleError(resultString, messages, view);

            // 確認
            Assert.IsFalse(view.HasErrors);
            Assert.IsTrue(target._handleValidationErrorCalled);
            Assert.AreEqual(0, target.ValidationErrors.Count);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(FormBase), LogLevel.INFO));
        }

        ///<summary>
        /// HandleError のテスト
        ///</summary>
        ///<remarks>
        /// resultStringがvalidationFailed、messagesの要素がValidationMessageInfoにキャストできない場合、
        /// データセットにエラーが設定されないことを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void HandleErrorTest07()
        {
            FormBaseEx target = new FormBaseEx();

            string resultString = Coordinator.VALIDATION_FAILED;
            IList<MessageInfo> messages = new List<MessageInfo>();
            MessageInfo info = new MessageInfo("key1", "TERASOLUNA1");
            messages.Add(info);
            FormBaseDataSet01 view = new FormBaseDataSet01();
            view.DataTable1.AddDataTable1Row("row0col1", "row0,col2", "row0col3");
            view.DataTable1.AddDataTable1Row("row1col1", "row1,col2", "row1col3");
            view.DataTable1.AddDataTable1Row("row2col1", "row2,col2", "row2col3");
            view.AcceptChanges();

            // 実施
            target.HandleError(resultString, messages, view);

            // 確認
            Assert.IsFalse(view.HasErrors);
            Assert.IsTrue(target._handleValidationErrorCalled);
            Assert.AreEqual(0, target.ValidationErrors.Count);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(FormBase), LogLevel.INFO));
        }

        ///<summary>
        /// HandleError のテスト
        ///</summary>
        ///<remarks>
        /// resultStringがvalidationFailedで、messagesの要素がValidationMessageInfoにキャストできるが、
        /// ValidationMessageInfoのエラーパスがnullの場合、
        /// データセットにエラーが設定されないことを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void HandleErrorTest08()
        {
            FormBaseEx target = new FormBaseEx();

            string resultString = Coordinator.VALIDATION_FAILED;
            IList<MessageInfo> messages = new List<MessageInfo>();
            ValidationMessageInfo info = new ValidationMessageInfo("col1", "TERASOLUNA1", null);
            messages.Add(info);
            FormBaseDataSet01 view = new FormBaseDataSet01();
            view.DataTable1.AddDataTable1Row("row0col1", "row0,col2", "row0col3");
            view.DataTable1.AddDataTable1Row("row1col1", "row1,col2", "row1col3");
            view.DataTable1.AddDataTable1Row("row2col1", "row2,col2", "row2col3");
            view.AcceptChanges();

            // 実施
            target.HandleError(resultString, messages, view);

            // 確認
            Assert.IsFalse(view.HasErrors);
            Assert.IsTrue(target._handleValidationErrorCalled);
            Assert.AreEqual(0, target.ValidationErrors.Count);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(FormBase), LogLevel.INFO));
        }

        ///<summary>
        /// HandleError のテスト
        ///</summary>
        ///<remarks>
        /// resultStringがvalidationFailedで、messagesの要素がValidationMessageInfoにキャストできるが、
        /// ValidationMessageInfoのエラーパスが空文字の場合、
        /// データセットにエラーが設定されないことを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void HandleErrorTest09()
        {
            FormBaseEx target = new FormBaseEx();

            string resultString = Coordinator.VALIDATION_FAILED;
            IList<MessageInfo> messages = new List<MessageInfo>();
            ValidationMessageInfo info = new ValidationMessageInfo("col1", "TERASOLUNA1", string.Empty);
            messages.Add(info);
            FormBaseDataSet01 view = new FormBaseDataSet01();
            view.DataTable1.AddDataTable1Row("row0col1", "row0,col2", "row0col3");
            view.DataTable1.AddDataTable1Row("row1col1", "row1,col2", "row1col3");
            view.DataTable1.AddDataTable1Row("row2col1", "row2,col2", "row2col3");
            view.AcceptChanges();

            // 実施
            target.HandleError(resultString, messages, view);

            // 確認
            Assert.IsFalse(view.HasErrors);
            Assert.IsTrue(target._handleValidationErrorCalled);
            Assert.AreEqual(0, target.ValidationErrors.Count);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(FormBase), LogLevel.INFO));
        }

        ///<summary>
        /// HandleError のテスト
        ///</summary>
        ///<remarks>
        /// resultStringがvalidationFailedで、messagesの要素がValidationMessageInfoにキャストできるが、
        /// ValidationMessageInfoのエラーパスに対応する項目がデータセットにない場合、
        /// データセットにエラーが設定されないことを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void HandleErrorTest10()
        {
            FormBaseEx target = new FormBaseEx();

            string resultString = Coordinator.VALIDATION_FAILED;
            IList<MessageInfo> messages = new List<MessageInfo>();
            ValidationMessageInfo info = new ValidationMessageInfo("col1", "TERASOLUNA1", "DataTable1[4]/Column1");
            messages.Add(info);
            FormBaseDataSet01 view = new FormBaseDataSet01();
            view.DataTable1.AddDataTable1Row("row0col1", "row0,col2", "row0col3");
            view.DataTable1.AddDataTable1Row("row1col1", "row1,col2", "row1col3");
            view.DataTable1.AddDataTable1Row("row2col1", "row2,col2", "row2col3");
            view.AcceptChanges();

            // 実施
            target.HandleError(resultString, messages, view);

            // 確認
            Assert.IsFalse(view.HasErrors);
            Assert.IsTrue(target._handleValidationErrorCalled);
            Assert.AreEqual(0, target.ValidationErrors.Count);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(FormBase), LogLevel.INFO));
        }

        ///<summary>
        /// HandleError のテスト
        ///</summary>
        ///<remarks>
        /// resultStringがvalidationFailedで、messagesの要素がValidationMessageInfoにキャストできるが、
        /// ValidationMessageInfoのエラーパスに対応する項目がデータセットに1つある場合、
        /// データセットにエラーが設定されることを確認するテスト。
        /// なお、エラーパスには行インデックスを含める。
        ///</remarks>
        [TestMethod()]
        public void HandleErrorTest11()
        {
            FormBaseEx target = new FormBaseEx();

            string resultString = Coordinator.VALIDATION_FAILED;
            IList<MessageInfo> messages = new List<MessageInfo>();
            ValidationMessageInfo info = new ValidationMessageInfo("col1", "TERASOLUNA1", "DataTable1[1]/Column1");
            messages.Add(info);
            FormBaseDataSet01 view = new FormBaseDataSet01();
            view.DataTable1.AddDataTable1Row("row0col1", "row0,col2", "row0col3");
            view.DataTable1.AddDataTable1Row("row1col1", "row1,col2", "row1col3");
            view.DataTable1.AddDataTable1Row("row2col1", "row2,col2", "row2col3");
            view.AcceptChanges();

            // 実施
            target.HandleError(resultString, messages, view);

            // 確認
            Assert.IsTrue(view.HasErrors);
            Assert.IsTrue(target._handleValidationErrorCalled);
            int errorCount = 0;
            foreach (DataRow row in view.DataTable1.GetErrors())
            {
                foreach (DataColumn column in row.GetColumnsInError())
                {
                    errorCount++;
                }
            }
            Assert.AreEqual(1, errorCount);
            Assert.AreEqual("TERASOLUNA1", view.DataTable1[0].GetColumnError("Column1"));
            Assert.AreEqual(1, target.ValidationErrors.Count);
            Assert.AreEqual("TERASOLUNA1", target.ValidationErrors["DataTable1[1]/Column1"]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(FormBase), LogLevel.INFO));
        }

        ///<summary>
        /// HandleError のテスト
        ///</summary>
        ///<remarks>
        /// resultStringがvalidationFailedで、messagesの要素がValidationMessageInfoにキャストできるが、
        /// ValidationMessageInfoのエラーパスに対応する項目がデータセットに3つある場合、
        /// データセットにエラーが設定されることを確認するテスト。
        /// なお、エラーパスには行インデックスを含めない。
        ///</remarks>
        [TestMethod()]
        public void HandleErrorTest12()
        {
            FormBaseEx target = new FormBaseEx();

            string resultString = Coordinator.VALIDATION_FAILED;
            IList<MessageInfo> messages = new List<MessageInfo>();
            ValidationMessageInfo info = new ValidationMessageInfo("col1", "TERASOLUNA1", "DataTable1/Column1");
            messages.Add(info);
            FormBaseDataSet01 view = new FormBaseDataSet01();
            view.DataTable1.AddDataTable1Row("row0col1", "row0,col2", "row0col3");
            view.DataTable1.AddDataTable1Row("row1col1", "row1,col2", "row1col3");
            view.DataTable1.AddDataTable1Row("row2col1", "row2,col2", "row2col3");
            view.AcceptChanges();

            // 実施
            target.HandleError(resultString, messages, view);

            // 確認
            Assert.IsTrue(view.HasErrors);
            Assert.IsTrue(target._handleValidationErrorCalled);
            int errorCount = 0;
            foreach (DataRow row in view.DataTable1.GetErrors())
            {
                foreach (DataColumn column in row.GetColumnsInError())
                {
                    errorCount++;
                }
            }
            Assert.AreEqual(3, errorCount);
            Assert.AreEqual("TERASOLUNA1", view.DataTable1[0].GetColumnError("Column1"));
            Assert.AreEqual("TERASOLUNA1", view.DataTable1[1].GetColumnError("Column1"));
            Assert.AreEqual("TERASOLUNA1", view.DataTable1[2].GetColumnError("Column1"));
            Assert.AreEqual(1, target.ValidationErrors.Count);
            Assert.AreEqual("TERASOLUNA1", target.ValidationErrors["DataTable1/Column1"]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(FormBase), LogLevel.INFO));
        }

        ///<summary>
        /// HandleError のテスト
        ///</summary>
        ///<remarks>
        /// resultStringがvalidationFailedで、messagesの要素がValidationMessageInfoにキャストできるが、
        /// ValidationMessageInfoのエラーパスに対応する項目がデータセットに3つある場合、
        /// データセットにエラーが設定されることを確認するテスト。
        /// なお、エラーパスには行インデックスを含める。
        ///</remarks>
        [TestMethod()]
        public void HandleErrorTest13()
        {
            FormBaseEx target = new FormBaseEx();

            string resultString = Coordinator.VALIDATION_FAILED;
            IList<MessageInfo> messages = new List<MessageInfo>();
            ValidationMessageInfo info1 = new ValidationMessageInfo("col1", "TERASOLUNA1", "DataTable1[1]/Column1");
            ValidationMessageInfo info2 = new ValidationMessageInfo("col2", "TERASOLUNA2", "DataTable1[2]/Column2");
            ValidationMessageInfo info3 = new ValidationMessageInfo("col3", "TERASOLUNA3", "DataTable1[3]/Column3");
            messages.Add(info1);
            messages.Add(info2);
            messages.Add(info3);
            FormBaseDataSet01 view = new FormBaseDataSet01();
            view.DataTable1.AddDataTable1Row("row0col1", "row0,col2", "row0col3");
            view.DataTable1.AddDataTable1Row("row1col1", "row1,col2", "row1col3");
            view.DataTable1.AddDataTable1Row("row2col1", "row2,col2", "row2col3");
            view.AcceptChanges();

            // 実施
            target.HandleError(resultString, messages, view);

            // 確認
            Assert.IsTrue(view.HasErrors);
            Assert.IsTrue(target._handleValidationErrorCalled);
            int errorCount = 0;
            foreach (DataRow row in view.DataTable1.GetErrors())
            {
                foreach (DataColumn column in row.GetColumnsInError())
                {
                    errorCount++;
                }
            }
            Assert.AreEqual(3, errorCount);
            Assert.AreEqual("TERASOLUNA1", view.DataTable1[0].GetColumnError("Column1"));
            Assert.AreEqual("TERASOLUNA2", view.DataTable1[1].GetColumnError("Column2"));
            Assert.AreEqual("TERASOLUNA3", view.DataTable1[2].GetColumnError("Column3"));
            Assert.AreEqual(3, target.ValidationErrors.Count);
            Assert.AreEqual("TERASOLUNA1", target.ValidationErrors["DataTable1[1]/Column1"]);
            Assert.AreEqual("TERASOLUNA2", target.ValidationErrors["DataTable1[2]/Column2"]);
            Assert.AreEqual("TERASOLUNA3", target.ValidationErrors["DataTable1[3]/Column3"]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(FormBase), LogLevel.INFO));
        }

        ///<summary>
        /// ClearError (object) のテスト
        ///</summary>
        ///<remarks>
        /// viewがnullの場合、例外が発生しないことのテスト。
        ///</remarks>
        [TestMethod()]
        public void ClearErrorTest01()
        {
            FormBase target = new FormBase();
          
            object view = null;

            target.ClearError(view);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(FormBase), LogLevel.INFO));
        }

        ///<summary>
        /// ClearError (object) のテスト
        ///</summary>
        ///<remarks>
        /// viewがDataSetではない場合、
        /// 例外TerasolunaExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ClearErrorTest02()
        {
            FormBase target = new FormBase();

            object view = new object();

            try
            {
                target.ClearError(view);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_INVALID_CAST", view.GetType().FullName, typeof(DataSet).FullName), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(FormBase), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(FormBase), typeof(TerasolunaException), "E_INVALID_CAST", view.GetType().FullName, typeof(DataSet).FullName));
        }

        ///<summary>
        /// ClearError (object) のテスト
        ///</summary>
        ///<remarks>
        /// validationErrorsの要素数が0の場合、
        /// DataSetに変化が無いことを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ClearErrorTest03()
        {
            FormBase target = new FormBase();

            FormBaseDataSet01 view = new FormBaseDataSet01();
            view.DataTable1.AddDataTable1Row("row0col1", "row0,col2", "row0col3");
            view.DataTable1.AddDataTable1Row("row1col1", "row1,col2", "row1col3");
            view.DataTable1.AddDataTable1Row("row2col1", "row2,col2", "row2col3");
            view.DataTable1[0].SetColumnError("Column1", "TERASOLUNA1");
            view.DataTable1[1].SetColumnError("Column2", "TERASOLUNA2");
            view.DataTable1[2].SetColumnError("Column3", "TERASOLUNA3");
            view.AcceptChanges();

            // 実行
            target.ClearError(view);

            // 確認
            Assert.IsFalse(view.HasChanges());
            Assert.IsTrue(view.HasErrors);
            int errorCount = 0;
            foreach (DataRow row in view.DataTable1.GetErrors())
            {
                foreach (DataColumn column in row.GetColumnsInError())
                {
                    errorCount++;
                }
            }
            Assert.AreEqual(3, errorCount);
            Assert.AreEqual("TERASOLUNA1", view.DataTable1[0].GetColumnError("Column1"));
            Assert.AreEqual("TERASOLUNA2", view.DataTable1[1].GetColumnError("Column2"));
            Assert.AreEqual("TERASOLUNA3", view.DataTable1[2].GetColumnError("Column3"));

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(FormBase), LogLevel.INFO));
        }

        ///<summary>
        /// ClearError (object) のテスト
        ///</summary>
        ///<remarks>
        /// viewが次の条件を満たす場合、
        /// ・引数 validationErrors が保持している要素(XPath) に対応する行がない
        /// DataSetに変化がないことを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ClearErrorTest04()
        {
            FormBase target = new FormBase();

            IDictionary<string, string> validationErrors = target.ValidationErrors;
            validationErrors.Add("errorPath", "errorMessage");
            
            FormBaseDataSet01 view = new FormBaseDataSet01();
            view.DataTable1.AddDataTable1Row("row0col1", "row0,col2", "row0col3");
            view.DataTable1.AddDataTable1Row("row1col1", "row1,col2", "row1col3");
            view.DataTable1.AddDataTable1Row("row2col1", "row2,col2", "row2col3");
            view.DataTable1[0].SetColumnError("Column1", "TERASOLUNA1");
            view.DataTable1[1].SetColumnError("Column2", "TERASOLUNA2");
            view.DataTable1[2].SetColumnError("Column3", "TERASOLUNA3");
            view.AcceptChanges();

            // 実行
            target.ClearError(view);

            // 確認
            Assert.IsFalse(view.HasChanges());
            Assert.IsTrue(view.HasErrors);
            int errorCount = 0;
            foreach (DataRow row in view.DataTable1.GetErrors())
            {
                foreach (DataColumn column in row.GetColumnsInError())
                {
                    errorCount++;
                }
            }
            Assert.AreEqual(3, errorCount);
            Assert.AreEqual("TERASOLUNA1", view.DataTable1[0].GetColumnError("Column1"));
            Assert.AreEqual("TERASOLUNA2", view.DataTable1[1].GetColumnError("Column2"));
            Assert.AreEqual("TERASOLUNA3", view.DataTable1[2].GetColumnError("Column3"));

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(FormBase), LogLevel.INFO));
        }

        ///<summary>
        /// ClearError (object) のテスト
        ///</summary>
        ///<remarks>
        /// viewが次の条件を満たす場合、
        /// ・引数 validationErrors が保持している要素(XPath) に対応する行がある
        /// ・その行数が１
        /// ・行にエラーがない
        /// DataSetに変化がないことを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ClearErrorTest05()
        {
            FormBase target = new FormBase();

            IDictionary<string, string> validationErrors = target.ValidationErrors;
            validationErrors.Add("DataTable1[1]/Column3", "errorMessage");

            FormBaseDataSet01 view = new FormBaseDataSet01();
            view.DataTable1.AddDataTable1Row("row0col1", "row0,col2", "row0col3");
            view.DataTable1.AddDataTable1Row("row1col1", "row1,col2", "row1col3");
            view.DataTable1.AddDataTable1Row("row2col1", "row2,col2", "row2col3");
            view.DataTable1[0].SetColumnError("Column1", "TERASOLUNA1");
            view.DataTable1[1].SetColumnError("Column2", "TERASOLUNA2");
            view.DataTable1[2].SetColumnError("Column3", "TERASOLUNA3");
            view.AcceptChanges();

            // 実行
            target.ClearError(view);

            // 確認
            Assert.IsFalse(view.HasChanges());
            Assert.IsTrue(view.HasErrors);
            int errorCount = 0;
            foreach (DataRow row in view.DataTable1.GetErrors())
            {
                foreach (DataColumn column in row.GetColumnsInError())
                {
                    errorCount++;
                }
            }
            Assert.AreEqual(3, errorCount);
            Assert.AreEqual("TERASOLUNA1", view.DataTable1[0].GetColumnError("Column1"));
            Assert.AreEqual("TERASOLUNA2", view.DataTable1[1].GetColumnError("Column2"));
            Assert.AreEqual("TERASOLUNA3", view.DataTable1[2].GetColumnError("Column3"));

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(FormBase), LogLevel.INFO));
        }

        ///<summary>
        /// ClearError (IList&lt;string&gt;, object) のテスト
        ///</summary>
        ///<remarks>
        /// viewが次の条件を満たす場合、
        /// ・引数 validationErrors が保持している要素(XPath) に対応する行がある
        /// ・その行数が３
        /// ・全ての行にエラーがない
        /// DataSetに変化がないことを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ClearErrorTest06()
        {
            FormBase target = new FormBase();

            IDictionary<string, string> validationErrors = target.ValidationErrors;
            validationErrors.Add("DataTable1[1]/Column3", "errorMessage1");
            validationErrors.Add("DataTable1[2]/Column1", "errorMessage2");
            validationErrors.Add("DataTable1[3]/Column2", "errorMessage3");

            FormBaseDataSet01 view = new FormBaseDataSet01();
            view.DataTable1.AddDataTable1Row("row0col1", "row0,col2", "row0col3");
            view.DataTable1.AddDataTable1Row("row1col1", "row1,col2", "row1col3");
            view.DataTable1.AddDataTable1Row("row2col1", "row2,col2", "row2col3");
            view.DataTable1[0].SetColumnError("Column1", "TERASOLUNA1");
            view.DataTable1[1].SetColumnError("Column2", "TERASOLUNA2");
            view.DataTable1[2].SetColumnError("Column3", "TERASOLUNA3");
            view.AcceptChanges();

            // 実行
            target.ClearError(view);

            // 確認
            Assert.IsFalse(view.HasChanges());
            Assert.IsTrue(view.HasErrors);
            int errorCount = 0;
            foreach (DataRow row in view.DataTable1.GetErrors())
            {
                foreach (DataColumn column in row.GetColumnsInError())
                {
                    errorCount++;
                }
            }
            Assert.AreEqual(3, errorCount);
            Assert.AreEqual("TERASOLUNA1", view.DataTable1[0].GetColumnError("Column1"));
            Assert.AreEqual("TERASOLUNA2", view.DataTable1[1].GetColumnError("Column2"));
            Assert.AreEqual("TERASOLUNA3", view.DataTable1[2].GetColumnError("Column3"));

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(FormBase), LogLevel.INFO));
        }

        ///<summary>
        /// ClearError (IList&lt;string&gt;, object) のテスト
        ///</summary>
        ///<remarks>
        /// viewが次の条件を満たす場合、
        /// ・引数 validationErrors が保持している要素(XPath) に対応する行がある
        /// ・その行数が３
        /// ・全ての行にエラーがある
        /// ・引数 validationErrors が保持している要素(XPath) がカラムを表現する
        /// XPathで指定したエラーがクリアされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ClearErrorTest07()
        {
            FormBase target = new FormBase();

            IDictionary<string, string> validationErrors = target.ValidationErrors;
            validationErrors.Add("DataTable1[1]/Column1", "errorMessage");

            FormBaseDataSet01 view = new FormBaseDataSet01();
            view.DataTable1.AddDataTable1Row("row0col1", "row0,col2", "row0col3");
            view.DataTable1.AddDataTable1Row("row1col1", "row1,col2", "row1col3");
            view.DataTable1.AddDataTable1Row("row2col1", "row2,col2", "row2col3");
            view.DataTable1[0].SetColumnError("Column1", "TERASOLUNA1");
            view.DataTable1[1].SetColumnError("Column2", "TERASOLUNA2");
            view.DataTable1[2].SetColumnError("Column3", "TERASOLUNA3");
            view.AcceptChanges();

            // 実行
            target.ClearError(view);

            // 確認
            Assert.IsFalse(view.HasChanges());
            Assert.IsTrue(view.HasErrors);
            int errorCount = 0;
            foreach (DataRow row in view.DataTable1.GetErrors())
            {
                foreach (DataColumn column in row.GetColumnsInError())
                {
                    errorCount++;
                }
            }
            Assert.AreEqual(2, errorCount);
            Assert.AreEqual("TERASOLUNA2", view.DataTable1[1].GetColumnError("Column2"));
            Assert.AreEqual("TERASOLUNA3", view.DataTable1[2].GetColumnError("Column3"));

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(FormBase), LogLevel.INFO));
        }

        ///<summary>
        /// ClearError (IList&lt;string&gt;, object) のテスト
        ///</summary>
        ///<remarks>
        /// validationErrorsの要素数が3の場合、
        /// XPathで指定されたすべてのエラーがクリアされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ClearErrorTest08()
        {
            FormBase target = new FormBase();

            IDictionary<string, string> validationErrors = target.ValidationErrors;
            validationErrors.Add("DataTable1[1]/Column1", "errorMessage1");
            validationErrors.Add("DataTable1[2]/Column2", "errorMessage2");
            validationErrors.Add("DataTable1[3]/Column3", "errorMessage3");

            FormBaseDataSet01 view = new FormBaseDataSet01();
            view.DataTable1.AddDataTable1Row("row0col1", "row0,col2", "row0col3");
            view.DataTable1.AddDataTable1Row("row1col1", "row1,col2", "row1col3");
            view.DataTable1.AddDataTable1Row("row2col1", "row2,col2", "row2col3");
            view.DataTable1[0].SetColumnError("Column1", "TERASOLUNA1");
            view.DataTable1[1].SetColumnError("Column2", "TERASOLUNA2");
            view.DataTable1[2].SetColumnError("Column3", "TERASOLUNA3");
            view.AcceptChanges();

            // 実行
            target.ClearError(view);

            // 確認
            Assert.IsFalse(view.HasChanges());
            Assert.IsFalse(view.HasErrors);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(FormBase), LogLevel.INFO));
        }

        ///<summary>
        /// Init () のテスト
        ///</summary>
        ///<remarks>
        /// ・引数にnullを渡したとき、例外がスローされないこと。
        /// ・引数にnullを渡したとき、trueを返却すること。
        ///</remarks>
        [TestMethod()]
        public void InitTest01()
        {
            FormBase target = new FormBase();

            Assert.IsTrue(target.Init(null));

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(FormBase), LogLevel.INFO));
        }

        ///<summary>
        /// Init () のテスト
        ///</summary>
        ///<remarks>
        /// ・引数にFormBaseのインスタンスを渡したとき、例外がスローされないこと。
        /// ・引数にFormBaseのインスタンスを渡したとき、trueを返却すること。
        ///</remarks>
        [TestMethod()]
        public void InitTest02()
        {
            FormBase target = new FormBase();

            Assert.IsTrue(target.Init(new FormBase()));

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(FormBase), LogLevel.INFO));
        }


        ///<summary>
        /// Items のテスト
        ///</summary>
        ///<remarks>
        /// Itemsに値をセットした場合、その値をゲットできることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ItemsTest01()
        {
            FormBase target = new FormBase();

            IDictionary val = new Hashtable();
            TERASOLUNA_Fw_Client_Forms_FormBaseAccessor accessor =
                new TERASOLUNA_Fw_Client_Forms_FormBaseAccessor(target);
            accessor._items = val;

            Assert.AreSame(val, target.Items);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(FormBase), LogLevel.INFO));
        }

        ///<summary>
        /// ViewData のテスト
        ///</summary>
        ///<remarks>
        /// ViewDataプロパティに値をセットした場合、
        /// その値をゲットできることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ViewDataTest01()
        {
            FormBase target = new FormBase();

            DataSet val = new DataSet();
            target.ViewData = val;

            Assert.AreSame(val, target.ViewData);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(FormBase), LogLevel.INFO));
        }

    }

    /// <summary>
    /// HandleValidationError メソッドが呼ばれているかどうかを確認するための FormBase 派生クラス。
    /// </summary>
    public class FormBaseEx : FormBase
    {
        public bool _handleValidationErrorCalled = false;

        protected override void HandleValidationError(IList<MessageInfo> messages, DataSet dataSet)
        {
            _handleValidationErrorCalled = true;
            base.HandleValidationError(messages, dataSet);
        }
    }
}
