// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Text;
using System.Collections.Generic;
using TERASOLUNA.Fw.Client.Configuration.View;
using System.Configuration;
using TERASOLUNA.Fw.Client.Configuration.Conversion;
using TERASOLUNA.Fw.Utlib.Logging;
using TERASOLUNA.Fw.Common.Logging;
using TERASOLUNA.Fw.Common;
namespace TERASOLUNA.Fw.ClientUT.Configuration.View
{
    /// <summary>
    ///TERASOLUNA.Fw.Client.Configuration.View.ViewCollection のテスト クラスです。すべての
    ///TERASOLUNA.Fw.Client.Configuration.View.ViewCollection 単体テストをここに含めます
    ///</summary>
    [TestClass()]
    public class ViewCollectionTest
    {


        private TestContext testContextInstance;

        ///<summary>
        /// 現在のテストの実行についての情報および機能を
        /// 提供するテスト コンテキストを取得または設定します。
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }
        #region 追加のテスト属性
        // 
        //テストを作成するときに、次の追加属性を使用することができます:
        //
        //クラスの最初のテストを実行する前にコードを実行するには、ClassInitialize を使用
        //
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //クラスのすべてのテストを実行した後にコードを実行するには、ClassCleanup を使用
        //
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //各テストを実行する前にコードを実行するには、TestInitialize を使用
        //
        [TestInitialize()]
        public void MyTestInitialize()
        {
            LogUTUtil.Clear();
        }
        //
        //各テストを実行した後にコードを実行するには、TestCleanup を使用
        //
        //[TestCleanup()]
        //public void MyTestCleanup()
        //{
        //}
        //
        #endregion


        ///<summary>
        /// Add (ViewElement) のテスト
        ///</summary>
        ///<remarks>
        /// 要素を追加した場合に例外が発生せず、正しく値が取得できることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void AddTest01()
        {
            // 準備
            ViewCollection target = new ViewCollection();
            ViewElement expectElement = new ViewElement();
            int expectCount = 1;

            // 実行
            target.Add(expectElement);

            // 検証
            Assert.AreEqual(expectCount, target.Count);
            Assert.AreEqual(expectElement, target[0]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ViewCollection), LogLevel.INFO));

        }

        ///<summary>
        /// Add (ViewElement) のテスト
        ///</summary>
        ///<remarks>
        /// 要素を連続で追加した場合に例外が発生せず、FIFOで要素が取得できることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void AddTest02()
        {
            // 準備
            ViewCollection target = new ViewCollection();
            ViewElement expectedElement1 = new ViewElement("View1", "elem1");
            ViewElement expectedElement2 = new ViewElement("View1", "elem2");
            int expectedCount = 2;

            // 実行
            target.Add(expectedElement1);
            target.Add(expectedElement2);

            // 検証
            Assert.AreEqual(expectedCount, target.Count);
            Assert.AreEqual(expectedElement1, target[0]);
            Assert.AreEqual(expectedElement2, target[1]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ViewCollection), LogLevel.INFO));

        }

        ///<summary>
        /// Add (ViewElement) のテスト
        ///</summary>
        ///<remarks>
        /// 同一IDを持つ要素を複数追加した場合、要素がひとつだけ追加されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void AddTest03()
        {
            // 準備
            ViewCollection target = new ViewCollection();
            ViewElement expectedElement = new ViewElement("View1", "same");
            ViewElement duplicateElement = new ViewElement("View1", "same");
            int expectedCount = 1;

            // 実行
            target.Add(expectedElement);
            target.Add(duplicateElement);

            // 検証
            Assert.AreEqual(expectedCount, target.Count);
            Assert.AreEqual(expectedElement, target[0]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ViewCollection), LogLevel.INFO));

        }

        ///<summary>
        /// Add (ViewElement) のテスト
        ///</summary>
        ///<remarks>
        /// nullを追加した場合、ArgumentNullExceptionが発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void AddTest04()
        {
            // 準備
            ViewCollection target = new ViewCollection();

            try
            {
                // 実行
                target.Add(null);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // メッセージ確認
                Assert.AreEqual("element", e.ParamName);
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ViewCollection), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ViewCollection), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "element"));

        }

        ///<summary>
        /// CollectionType のテスト
        ///</summary>
        ///<remarks>
        /// コレクションタイプがBasicMapであることを確認する
        ///</remarks>
        [TestMethod()]
        public void CollectionTypeTest01()
        {
            // 準備
            ViewCollection target = new ViewCollection();

            // 実行
            object actual = target.CollectionType;

            // 検証
            Assert.AreEqual(ConfigurationElementCollectionType.BasicMap, actual);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ViewCollection), LogLevel.INFO));

        }

        ///<summary>
        /// IsReadOnly () のテスト
        ///</summary>
        ///<remarks>
        /// コレクションが読み取り専用でないことを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void IsReadOnlyTest01()
        {
            // 準備
            ViewCollection target = new ViewCollection();

            // 実行
            bool actual = target.IsReadOnly();

            // 検証
            Assert.IsFalse(actual);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ViewCollection), LogLevel.INFO));

        }

        ///<summary>
        /// this[int index] のテスト
        ///</summary>
        ///<remarks>
        /// 追加された要素が正しく返却できることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ItemTest01()
        {
            // 準備
            ViewCollection target = new ViewCollection();
            ViewElement expectedElement = new ViewElement();
            int expectedCount = 1;

            // 実行
            target.Add(expectedElement);

            // 検証
            Assert.AreEqual(expectedCount, target.Count);
            Assert.AreSame(expectedElement, target[0]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ViewCollection), LogLevel.INFO));

        }

        ///<summary>
        /// this[int index] のテスト
        ///</summary>
        /// <remarks>
        ///　複数個追加された要素が正しく返却できることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ItemTest02()
        {
            // 準備
            ViewCollection target = new ViewCollection();
            ViewElement expectedElement1 = new ViewElement(string.Empty, "element1");
            ViewElement expectedElement2 = new ViewElement(string.Empty, "element2");
            ViewElement expectedElement3 = new ViewElement(string.Empty, "element3");
            int expectedCount = 3;

            // 実行
            target.Add(expectedElement1);
            target.Add(expectedElement2);
            target.Add(expectedElement3);

            // 検証
            Assert.AreEqual(expectedCount, target.Count);
            Assert.AreSame(expectedElement1, target[0]);
            Assert.AreSame(expectedElement2, target[1]);
            Assert.AreSame(expectedElement3, target[2]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ViewCollection), LogLevel.INFO));
        }

        ///<summary>
        /// this[string name] のテスト
        ///</summary>
        ///<remarks>
        /// 複数個追加された要素がキー指定で返却できることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ItemTest03()
        {
            // 準備
            ViewCollection target = new ViewCollection();
            ViewElement expectedElement1 = new ViewElement(string.Empty, "element1");
            ViewElement expectedElement2 = new ViewElement(string.Empty, "element2");
            ViewElement expectedElement3 = new ViewElement(string.Empty, "element3");
            int expectedCount = 3;
            target.Add(expectedElement1);
            target.Add(expectedElement2);
            target.Add(expectedElement3);

            // 実行
            object actual = target["element2"];
            int actualCount = target.Count;

            // 検証
            Assert.AreEqual(expectedCount, actualCount);
            Assert.AreSame(expectedElement2, actual);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ViewCollection), LogLevel.INFO));

        }

        ///<summary>
        /// this[int index] のテスト
        ///</summary>
        ///<remarks>
        /// インデックスにnullを指定した場合にArgumentNullExceptionが返却されることを確認する。
        ///</remarks>
        [TestMethod()]
        public void ItemTest04()
        {
            // 準備
            ViewCollection target = new ViewCollection();
            ViewElement element1 = new ViewElement(string.Empty, "element1");
            target.Add(element1);

            try
            {
                // 実行
                object obj = target[null];
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (NullReferenceException)
            {
                // 成功
            }
            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ViewCollection), LogLevel.INFO));

        }

        ///<summary>
        /// this[int index] のテスト
        ///</summary>
        ///<remarks>
        /// インデックスに-1を指定した場合にConfigurationErrorsExceptionが返却されることを確認する。
        ///</remarks>
        [TestMethod()]
        public void ItemTest05()
        {
            // 準備
            ViewCollection target = new ViewCollection();
            ViewElement element1 = new ViewElement(string.Empty, "element1");
            target.Add(element1);

            try
            {
                // 実行
                object obj = target[-1];
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ConfigurationErrorsException)
            {
                // 成功
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ViewCollection), LogLevel.INFO));

        }

        ///<summary>
        /// this[int index] のテスト
        ///</summary>
        ///<remarks>
        /// インデックスにサイズより大きい値を指定した場合にConfigurationErrorsExceptionが返却されることを確認する。
        ///</remarks>
        [TestMethod()]
        public void ItemTest06()
        {
            // 準備
            ViewCollection target = new ViewCollection();
            ViewElement element1 = new ViewElement(string.Empty, "element1");
            target.Add(element1);

            try
            {
                // 実行
                object obj = target[1];
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ConfigurationErrorsException)
            {
                // 成功
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ViewCollection), LogLevel.INFO));

        }

        ///<summary>
        /// this[string name] のテスト
        ///</summary>
        ///<remarks>
        /// インデックスに空文字を指定した場合にnullが返却されることを確認する。
        ///</remarks>
        [TestMethod()]
        public void ItemTest07()
        {
            // 準備
            ViewCollection target = new ViewCollection();
            ViewElement element1 = new ViewElement(string.Empty, "element1");
            target.Add(element1);

            // 実行
            object actual = target[string.Empty];

            // 検証
            Assert.IsNull(actual);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ViewCollection), LogLevel.INFO));

        }

        ///<summary>
        /// this[string name] のテスト
        ///</summary>
        ///<remarks>
        /// インデックスにキーでない文字列を指定した場合にnullが返却されることを確認する。
        ///</remarks>
        [TestMethod()]
        public void ItemTest08()
        {
            // 準備
            ViewCollection target = new ViewCollection();
            ViewElement element1 = new ViewElement(string.Empty, "element1");
            target.Add(element1);

            // 実行
            object actual = target["terasoluna"];

            // 検証
            Assert.IsNull(actual);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ViewCollection), LogLevel.INFO));

        }

        ///<summary>
        /// ViewCollection () のテスト
        ///</summary>
        ///<remarks>
        /// デフォルトコンストラクタによって正常にインスタンスが生成されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ConstructorTest01()
        {
            ViewCollection target = new ViewCollection();
            Assert.IsNotNull(target);
        }

        ///<summary>
        /// GetElementKey (ConfigurationElement) のテスト
        ///</summary>
        ///<remarks>
        /// 引数がViewElementでは無い場合、ArgumentExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void GetElementKeyTest01()
        {
            // 準備
            ViewCollectionImpl target = new ViewCollectionImpl();
            ConvertElement element = new ConvertElement();

            object expected = "view1";
            object actual;

            try
            {
                // 実行
                actual = target.GetElementKey(element);
                Assert.Fail("到達しないコードです。");
            }
            catch (ArgumentException e)
            {
                // 確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_INVALID_CAST", element.GetType().FullName, typeof(ViewElement).FullName), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ViewCollection), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ViewCollection), typeof(ArgumentException), "E_INVALID_CAST", element.GetType().FullName, typeof(ViewElement).FullName));

        }

        ///<summary>
        /// CreateNewElement のテスト
        ///</summary>
        ///<remarks>
        /// 要素を新規作成した場合、作成される要素は常にViewElementであることを確認するテスト。 
        ///</remarks>
        [TestMethod()]
        public void CreateNewElementTest01()
        {
            ViewCollectionImpl target = new ViewCollectionImpl();
            object actual = target.CreateNewElement();
            Assert.IsInstanceOfType(actual, typeof(ViewElement));

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ViewCollection), LogLevel.INFO));

        }
    }


    ///<summary>
    /// protectedメソッドを呼ぶための実装クラスです。 
    ///</summary>
    public class ViewCollectionImpl : ViewCollection
    {

        ///<summary>
        /// CreateNewElementを公開するためのメソッド。
        ///</summary>
        ///<returns>
        /// ConfigurationElement
        ///</returns>
        public new ConfigurationElement CreateNewElement()
        {
            return base.CreateNewElement();
        }

        ///<summary>
        /// GetElementKeyを公開するためのメソッド。
        ///</summary>
        ///<param name="element"></param>
        ///<returns>
        /// 要素のキー
        ///</returns>
        public new object GetElementKey(ConfigurationElement element)
        {
            return base.GetElementKey(element);
        }
    }
}
