﻿// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using System;
using System.Collections.Generic;
using System.Data;
using TERASOLUNA.Fw.Common.Logging;

namespace TERASOLUNA.Fw.Client.Communication
{
    /// <summary>
    /// サーバ処理の結果を保持するクラスです。
    /// </summary>
    public class CommunicationResult
    {
        /// <summary>
        /// <see cref="ILog"/> を保持するインスタンスです。
        /// </summary>
        private static ILog _log = LogFactory.GetLogger(typeof(CommunicationResult));

        /// <summary>
        /// サーバ処理の結果データを保持する <see cref="DataSet"/> です。
        /// </summary>
        private DataSet _resultData = null;

        /// <summary>
        /// レスポンスヘッダの名前と値のペアを格納している <see cref="IDictionary{String, String}"/> です。
        /// </summary>
        private IDictionary<string, string> _responseHeaders = new Dictionary<string,string>();

        /// <summary>
        /// <see cref="CommunicationResult"/> クラスの新しいインスタンスを初期化します。
        /// </summary>
        public CommunicationResult()
        {
        }

        /// <summary>
        /// <see cref="CommunicationResult"/> クラスの新しいインスタンスを初期化します。
        /// </summary>
        /// <param name="resultData">サーバ処理の結果データを保持する <see cref="DataSet"/></param>
        public CommunicationResult(DataSet resultData)
        {
            _resultData = resultData;
        }

        /// <summary>
        /// サーバ処理の結果データを取得または設定します。
        /// </summary>
        /// <value>
        /// サーバ処理の結果データを保持する <see cref="DataSet"/>。
        /// </value>
        public virtual DataSet ResultData
        {
            get 
            { 
                return _resultData; 
            }
            set 
            { 
                _resultData = value; 
            }
        }

        /// <summary>
        /// レスポンスヘッダの名前と値のペアを格納している <see cref="IDictionary{String, String}"/> を取得します。
        /// </summary>
        /// <value>
        /// レスポンスヘッダの名前と値のペアを格納している <see cref="IDictionary{String, String}"/>。
        /// </value>
	    public IDictionary<string,string> ResponseHeaders 
	    {
		    get 
            { 
                return _responseHeaders;
            }
    	}   

        /// <summary>
        /// レスポンスヘッダの名前と値のペアを <see cref="CommunicationResult"/> に設定します。
        /// </summary>
        /// <param name="responseHeaders">
        /// レスポンスヘッダの名前と値のペアを格納している <see cref="IDictionary{String, String}"/>。
        /// </param>
        /// <exception cref="ArgumentNullException">
        /// <paramref name="responseHeaders"/> が null 参照です。
        /// </exception>
        public virtual void AddResponseHeaders(IDictionary<string,string> responseHeaders)
        {
            if (responseHeaders == null)
            {
                ArgumentNullException exception = new ArgumentNullException("responseHeaders");
                if (_log.IsErrorEnabled)
                {
                    _log.Error(string.Format(Properties.Resources.E_NULL_ARGUMENT, "responseHeaders"), exception);
                }
                throw exception;
            }

            foreach (string key in responseHeaders.Keys)
            {
                _responseHeaders.Add(key, responseHeaders[key]);
            }    
	    }
    }
}
