/*
 * Copyright (c) 2007 NTT DATA Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package jp.terasoluna.fw.oxm.mapper.castor;

import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;
import java.io.StringReader;
import java.io.StringWriter;
import java.io.UnsupportedEncodingException;
import java.io.Writer;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;
import java.util.Vector;

import jp.terasoluna.utlib.LogUTUtil;
import jp.terasoluna.utlib.UTUtil;
import jp.terasoluna.fw.oxm.mapper.OXMapper;
import jp.terasoluna.fw.oxm.mapper.castor.CastorOXMapperImpl;
import jp.terasoluna.fw.oxm.mapper.castor.exception.CastorMappingException;
import jp.terasoluna.fw.oxm.mapper.castor.exception.CastorMarshalException;
import jp.terasoluna.fw.oxm.mapper.castor.exception.CastorUnsupportedEncodingException;
import jp.terasoluna.fw.oxm.mapper.castor.exception.CastorValidationException;
import jp.terasoluna.fw.oxm.serialize.XMLSerializerEx;
import junit.framework.TestCase;

import org.apache.xerces.dom.DocumentImpl;
import org.exolab.castor.mapping.Mapping;
import org.exolab.castor.mapping.MappingException;
import org.exolab.castor.mapping.loader.AbstractMappingLoader2;
import org.exolab.castor.mapping.xml.ClassMapping;
import org.exolab.castor.xml.ClassDescriptorResolver;
import org.exolab.castor.xml.MarshalException;
import org.exolab.castor.xml.Marshaller;
import org.exolab.castor.xml.Unmarshaller;
import org.exolab.castor.xml.ValidationException;
import org.exolab.castor.xml.util.XMLClassDescriptorAdapter;
import org.w3c.dom.Document;

/**
 * {@link jp.terasoluna.fw.oxm.mapper.castor.CastorOXMapperImpl}
 * NX̃ubN{bNXeXgB
 * 
 * <p>
 * <h4>yNX̊Tvz</h4>
 * Castor𗘗pIuWFNg-XMLϊNXB<br>
 * OF<br>
 * getCastorMapping̈mappingClassAgetUrl\bḧpathnullɂȂ邱Ƃ͂ȂB
 * </p>
 * 
 * @see jp.terasoluna.fw.oxm.mapper.castor.CastorOXMapperImpl
 */
public class CastorOXMapperImplTest extends TestCase {

    /**
     * ̃NX̃pbP[WpX <br>
     * ̃NXjp.co.sample.SampleTest̏ꍇApbP[WpX́ujp/co/sample/vB <br>
     */
    private String packagePath = this.getClass().getPackage().getName()
            .replace(".", "/")
            + "/";

    /**
     * `t@Csuffix
     */
    private String mappingSuffix = ".xml";

    /**
     * ̃eXgP[Xsׂ GUI AvP[VNB
     * 
     * @param args
     *            java R}hɐݒ肳ꂽp[^
     */
    public static void main(String[] args) {
        junit.swingui.TestRunner.run(CastorOXMapperImplTest.class);
    }

    /**
     * sB
     * 
     * @throws Exception
     *             ̃\bhŔO
     * @see junit.framework.TestCase#setUp()
     */
    @Override
    protected void setUp() throws Exception {
        super.setUp();
    }

    /**
     * IsB
     * 
     * @throws Exception
     *             ̃\bhŔO
     * @see junit.framework.TestCase#tearDown()
     */
    @Override
    protected void tearDown() throws Exception {
        super.tearDown();
    }

    /**
     * RXgN^B
     * 
     * @param name
     *            ̃eXgP[X̖OB
     */
    public CastorOXMapperImplTest(String name) {
        super(name);
    }

    /**
     * testUnmarshalDocumentObject01() <br>
     * <br>
     * 
     * (ُn) <br>
     * ϓ_FA,C,G <br>
     * <br>
     * ͒lF() doc:null<br>
     * () out:not null<br>
     * 
     * <br>
     * ҒlF(ԕω) O:IllegalArgumentException{<br>
     * message = "DOM tree is null!"<br> }<br>
     * (ԕω) O:Ox:error<br>
     * DOM tree is null.<br>
     * 
     * <br>
     * docnull̏ꍇAOX[邱ƂmFB <br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testUnmarshalDocumentObject01() throws Exception {
        // O
        OXMapper oxmapper = new CastorOXMapperImpl();

        Document doc = null;
        Object out = new Object();

        try {
            // eXg{
            oxmapper.unmarshal(doc, out);
            fail();
        } catch (IllegalArgumentException e) {
            // 
            String message = "DOM tree is null.";
            assertEquals(message, e.getMessage());

            // OmF
            assertTrue(LogUTUtil.checkError(message));
        }
    }

    /**
     * testUnmarshalDocumentObject02() <br>
     * <br>
     * 
     * (ُn) <br>
     * ϓ_FA,C,G <br>
     * <br>
     * ͒lF() doc:not null<br>
     * () out:null<br>
     * () createUnmarshaller():ĂяoꂽƂmFB<br>
     * nꂽƂmFB<br>
     * () unmarshaller.unmarshal():ĂяoꂽƂmFB<br>
     * nꂽƂmFB<br>
     * <br>
     * MarshalExceptionX[B<br>
     * 
     * <br>
     * ҒlF(ԕω) O:CastorMarshalException{<br>
     * cause = MarshalException<br> }<br>
     * (ԕω) O:Ox:error<br>
     * Castor unmarshal failure.<br>
     * 
     * <br>
     * A}[VMarshalExceptionX[ꂽꍇAbvCastorMarshalExceptionX[AOo͂邱ƂmFB
     * <br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testUnmarshalDocumentObject02() throws Exception {
        // O
        CastorOXMapperImpl_CastorOXMapperImplStub01 oxmapper = new CastorOXMapperImpl_CastorOXMapperImplStub01();
        CastorOXMapperImpl_UnmarshallerStub01 unmarshaller = new CastorOXMapperImpl_UnmarshallerStub01();
        oxmapper.unmarshaller = unmarshaller;

        Document doc = new DocumentImpl();
        Object out = null;

        try {
            // eXg{
            oxmapper.unmarshal(doc, out);
            fail();
        } catch (CastorMarshalException e) {
            // 
            String message = "Castor unmarshal failure.";
            assertSame(MarshalException.class, e.getCause().getClass());

            // OmF
            assertTrue(LogUTUtil.checkError(message, e.getCause()));

            // createUnmarshaller\bhɈnꂽƂ̊mF
            assertNull(oxmapper.out);

            // unmarshaller.unmarshal\bhɈnꂽƂ̊mF
            assertSame(doc, unmarshaller.node);
        }
    }

    /**
     * testUnmarshalDocumentObject03() <br>
     * <br>
     * 
     * (ُn) <br>
     * ϓ_FA,G <br>
     * <br>
     * ͒lF() doc:not null<br>
     * () out:not null<br>
     * () createUnmarshaller():ĂяoꂽƂmFB<br>
     * nꂽƂmFB<br>
     * () unmarshaller.unmarshal():ĂяoꂽƂmFB<br>
     * nꂽƂmFB<br>
     * <br>
     * ValidationExceptionX[B<br>
     * 
     * <br>
     * ҒlF(ԕω) O:CastorValidationException{<br>
     * cause = ValidationException<br> }<br>
     * (ԕω) O:Ox:error<br>
     * Castor validation error.<br>
     * 
     * <br>
     * A}[VValidationExceptionX[ꂽꍇAbvCastorValudationExceptionX[AOo͂邱ƂmFB
     * <br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testUnmarshalDocumentObject03() throws Exception {
        // O
        CastorOXMapperImpl_CastorOXMapperImplStub01 oxmapper = new CastorOXMapperImpl_CastorOXMapperImplStub01();
        CastorOXMapperImpl_UnmarshallerStub02 unmarshaller = new CastorOXMapperImpl_UnmarshallerStub02();
        oxmapper.unmarshaller = unmarshaller;

        Document doc = new DocumentImpl();
        Object out = new Object();

        try {
            // eXg{
            oxmapper.unmarshal(doc, out);
            fail();
        } catch (CastorValidationException e) {
            // 
            String message = "Castor validation error.";
            assertSame(ValidationException.class, e.getCause().getClass());

            // OmF
            assertTrue(LogUTUtil.checkError(message, e.getCause()));

            // createUnmarshaller\bhɈnꂽƂ̊mF
            assertSame(out, oxmapper.out);

            // unmarshaller.unmarshal\bhɈnꂽƂ̊mF
            assertSame(doc, unmarshaller.node);
        }
    }

    /**
     * testUnmarshalDocumentObject04() <br>
     * <br>
     * 
     * (n) <br>
     * ϓ_FA <br>
     * <br>
     * ͒lF() doc:not null<br>
     * () out:not null<br>
     * () createUnmarshaller():ĂяoꂽƂmFB<br>
     * nꂽƂmFB<br>
     * () unmarshaller.unmarshal():ĂяoꂽƂmFB<br>
     * nꂽƂmFB<br>
     * 
     * <br>
     * ҒlF <br>
     * ñp^[B<br>
     * unmarshal\bhĂяoꂽƂ̊mFsB <br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testUnmarshalDocumentObject04() throws Exception {
        // O
        CastorOXMapperImpl_CastorOXMapperImplStub01 oxmapper = new CastorOXMapperImpl_CastorOXMapperImplStub01();
        CastorOXMapperImpl_UnmarshallerStub03 unmarshaller = new CastorOXMapperImpl_UnmarshallerStub03();
        oxmapper.unmarshaller = unmarshaller;

        Document doc = new DocumentImpl();
        Object out = new Object();

        // eXg{
        oxmapper.unmarshal(doc, out);

        // createUnmarshaller\bhɈnꂽƂ̊mF
        assertSame(out, oxmapper.out);

        // unmarshaller.unmarshal\bhɈnꂽƂ̊mF
        assertSame(doc, unmarshaller.node);
    }

    /**
     * testUnmarshalReaderObject01() <br>
     * <br>
     * 
     * (ُn) <br>
     * ϓ_FA,C,G <br>
     * <br>
     * ͒lF() Reader:null<br>
     * () out:not null<br>
     * 
     * <br>
     * ҒlF(ԕω) O:IllegalArgumentException{<br>
     * message = "Reader is null!!"<br> }<br>
     * (ԕω) O:Ox:error<br>
     * Reader is null.<br>
     * 
     * <br>
     * docnull̏ꍇAOX[邱ƂmFB <br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testUnmarshalReaderObject01() throws Exception {
        // O
        OXMapper oxmapper = new CastorOXMapperImpl();

        Reader reader = null;
        Object out = new Object();

        try {
            // eXg{
            oxmapper.unmarshal(reader, out);
            fail();
        } catch (IllegalArgumentException e) {
            // 
            String message = "Reader is null.";
            assertEquals(message, e.getMessage());

            // OmF
            assertTrue(LogUTUtil.checkError(message));
        }
    }

    /**
     * testUnmarshalReaderObject02() <br>
     * <br>
     * 
     * (ُn) <br>
     * ϓ_FA,C,G <br>
     * <br>
     * ͒lF() Reader:not null<br>
     * () out:null<br>
     * () createUnmarshaller():ĂяoꂽƂmFB<br>
     * nꂽƂmFB<br>
     * () unmarshaller.unmarshal():ĂяoꂽƂmFB<br>
     * nꂽƂmFB<br>
     * <br>
     * MarshalExceptionX[<br>
     * 
     * <br>
     * ҒlF(ԕω) O:CastorMarshalException{<br>
     * cause = MarshalException<br> }<br>
     * (ԕω) O:Ox:error<br>
     * Castor unmarshal failure.<br>
     * 
     * <br>
     * A}[VMarshalExceptionX[ꂽꍇAbvCastorMarshalExceptionX[AOo͂邱ƂmFB
     * <br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testUnmarshalReaderObject02() throws Exception {
        // O
        CastorOXMapperImpl_CastorOXMapperImplStub01 oxmapper = new CastorOXMapperImpl_CastorOXMapperImplStub01();
        CastorOXMapperImpl_UnmarshallerStub01 unmarshaller = new CastorOXMapperImpl_UnmarshallerStub01();
        oxmapper.unmarshaller = unmarshaller;

        Reader reader = new StringReader("");
        Object out = null;

        try {
            // eXg{
            oxmapper.unmarshal(reader, out);
            fail();
        } catch (CastorMarshalException e) {
            // 
            String message = "Castor unmarshal failure.";
            assertSame(MarshalException.class, e.getCause().getClass());

            // OmF
            assertTrue(LogUTUtil.checkError(message, e.getCause()));

            // createUnmarshaller\bhɈnꂽƂ̊mF
            assertNull(oxmapper.out);

            // unmarshaller.unmarshal\bhɈnꂽƂ̊mF
            assertSame(reader, unmarshaller.reader);
        }
    }

    /**
     * testUnmarshalReaderObject03() <br>
     * <br>
     * 
     * (ُn) <br>
     * ϓ_FA,G <br>
     * <br>
     * ͒lF() Reader:not null<br>
     * () out:not null<br>
     * () createUnmarshaller():ĂяoꂽƂmFB<br>
     * nꂽƂmFB<br>
     * () unmarshaller.unmarshal():ĂяoꂽƂmFB<br>
     * nꂽƂmFB<br>
     * <br>
     * ValidationExceptionX[<br>
     * 
     * <br>
     * ҒlF(ԕω) O:CastorValidationException{<br>
     * cause = ValidationException<br> }<br>
     * (ԕω) O:Ox:error<br>
     * Castor validation error.<br>
     * 
     * <br>
     * A}[VValidationExceptionX[ꂽꍇAbvCastorValudationExceptionX[AOo͂邱ƂmFB
     * <br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testUnmarshalReaderObject03() throws Exception {
        // O
        CastorOXMapperImpl_CastorOXMapperImplStub01 oxmapper = new CastorOXMapperImpl_CastorOXMapperImplStub01();
        CastorOXMapperImpl_UnmarshallerStub02 unmarshaller = new CastorOXMapperImpl_UnmarshallerStub02();
        oxmapper.unmarshaller = unmarshaller;

        Reader reader = new StringReader("");
        Object out = new Object();

        try {
            // eXg{
            oxmapper.unmarshal(reader, out);
            fail();
        } catch (CastorValidationException e) {
            // 
            String message = "Castor validation error.";
            assertSame(ValidationException.class, e.getCause().getClass());

            // OmF
            assertTrue(LogUTUtil.checkError(message, e.getCause()));

            // createUnmarshaller\bhɈnꂽƂ̊mF
            assertSame(out, oxmapper.out);

            // unmarshaller.unmarshal\bhɈnꂽƂ̊mF
            assertSame(reader, unmarshaller.reader);
        }
    }

    /**
     * testUnmarshalReaderObject04() <br>
     * <br>
     * 
     * (n) <br>
     * ϓ_FA <br>
     * <br>
     * ͒lF() Reader:not null<br>
     * () out:not null<br>
     * () createUnmarshaller():ĂяoꂽƂmFB<br>
     * nꂽƂmFB<br>
     * () unmarshaller.unmarshal():ĂяoꂽƂmFB<br>
     * nꂽƂmFB<br>
     * 
     * <br>
     * ҒlF <br>
     * ñp^[B<br>
     * unmarshal\bhĂяoꂽƂ̊mFsB <br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testUnmarshalReaderObject04() throws Exception {
        // O
        CastorOXMapperImpl_CastorOXMapperImplStub01 oxmapper = new CastorOXMapperImpl_CastorOXMapperImplStub01();
        CastorOXMapperImpl_UnmarshallerStub03 unmarshaller = new CastorOXMapperImpl_UnmarshallerStub03();
        oxmapper.unmarshaller = unmarshaller;

        Reader reader = new StringReader("");
        Object out = new Object();

        // eXg{
        oxmapper.unmarshal(reader, out);

        // createUnmarshaller\bhɈnꂽƂ̊mF
        assertSame(out, oxmapper.out);

        // unmarshaller.unmarshal\bhɈnꂽƂ̊mF
        assertSame(reader, unmarshaller.reader);

    }

    /**
     * testUnmarshalInputStreamStringObject01() <br>
     * <br>
     * 
     * (ُn) <br>
     * ϓ_FA,C,G <br>
     * <br>
     * ͒lF() is:null<br>
     * () argCharset:not null<br>
     * () out:not null<br>
     * 
     * <br>
     * ҒlF(ԕω) O:IllegalArgumentException{<br>
     * message="InputStream is null."<br> }<br>
     * (ԕω) O:Ox:error<br>
     * InputStream is null.<br>
     * 
     * <br>
     * isnull̏ꍇAOX[AOo͂邱ƂmFB <br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testUnmarshalInputStreamStringObject01() throws Exception {
        // O
        CastorOXMapperImpl oxmapper = new CastorOXMapperImpl();

        // 
        InputStream in = null;
        String argCharset = "";
        Object out = new Object();

        // eXg{
        try {
            oxmapper.unmarshal(in, argCharset, out);
            fail();
        } catch (IllegalArgumentException e) {
            // 
            String message = "InputStream is null.";
            assertEquals(message, e.getMessage());

            // OmF
            assertTrue(LogUTUtil.checkError(message));
        }

    }

    /**
     * testUnmarshalInputStreamStringObject02() <br>
     * <br>
     * 
     * (n) <br>
     * ϓ_FA,C <br>
     * <br>
     * ͒lF() is:not null<br>
     * () argCharset:""<br>
     * () out:not null<br>
     * () unmarshal():ĂяoꂽƂmFB<br>
     * nꂽƂmFB<br>
     * () close():ĂяoꂽƂmFB<br>
     * nꂽƂmFB<br>
     * () this.charset:"UTF-8"<br>
     * 
     * <br>
     * ҒlF <br>
     * argCharset󕶎̏ꍇAR[hƂ"UTF-8"ݒ肳Aunmarshal\bhAisr.close\bhĂяo邱ƂmFB<br>
     * Oo͂邱ƂmFB <br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testUnmarshalInputStreamStringObject02() throws Exception {
        // O
        CastorOXMapperImpl_CastorOXMapperImplStub02 oxmapper = new CastorOXMapperImpl_CastorOXMapperImplStub02();
        UTUtil.setPrivateField(oxmapper, "charset", "UTF-8");

        // 
        CastorOXMapperImpl_InputStreamStub01 in = new CastorOXMapperImpl_InputStreamStub01();
        String argCharset = "";
        Object out = new Object();

        // eXg{
        oxmapper.unmarshal(in, argCharset, out);

        // 
        // unmarshal\bhɈnꂽƂ̊mF
        assertSame(in, UTUtil.getPrivateField(oxmapper.reader, "lock"));
        assertSame(out, oxmapper.out);
        assertEquals("UTF8", oxmapper._charset);

        // close\bhĂяoꂽƂ̊mF
        assertTrue(in.isClose);
    }

    /**
     * testUnmarshalInputStreamStringObject03() <br>
     * <br>
     * 
     * (n) <br>
     * ϓ_FA,C <br>
     * <br>
     * ͒lF() is:not null<br>
     * () argCharset:null<br>
     * () out:null<br>
     * () unmarshal():ĂяoꂽƂmFB<br>
     * nꂽƂmFB<br>
     * () close():IOExceptionX[B<br>
     * () this.charset:"UTF-8"<br>
     * 
     * <br>
     * ҒlF <br>
     * argCharsetnull̏ꍇAR[hƂ"UTF-8"ݒ肳Aunmarshal\bhĂяo邱ƂmFB<br>
     * Xg[N[YۂIOExceptionX[ꂽꍇAOo͂邱ƂmFB <br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testUnmarshalInputStreamStringObject03() throws Exception {
        // O
        CastorOXMapperImpl_CastorOXMapperImplStub02 oxmapper = new CastorOXMapperImpl_CastorOXMapperImplStub02();
        UTUtil.setPrivateField(oxmapper, "charset", "UTF-8");

        // 
        CastorOXMapperImpl_InputStreamStub02 in = new CastorOXMapperImpl_InputStreamStub02();
        String argCharset = null;
        Object out = null;

        // eXg{
        oxmapper.unmarshal(in, argCharset, out);

        // 
        // unmarshal\bhɈnꂽƂ̊mF
        assertSame(in, UTUtil.getPrivateField(oxmapper.reader, "lock"));
        assertNull(out);
        assertEquals("UTF8", oxmapper._charset);

        // OmF
        String message = "Failed to close input stream.";
        assertTrue(LogUTUtil.checkError(message, new IOException()));

    }

    /**
     * testUnmarshalInputStreamStringObject04() <br>
     * <br>
     * 
     * (ُn) <br>
     * ϓ_FA,G <br>
     * <br>
     * ͒lF() is:not null<br>
     * () argCharset:"abc"<br>
     * () out:not null<br>
     * () close():ĂяoȂƂmFB<br>
     * <br>
     * ҒlF(ԕω) O:CastorUnsupportedEncodingException{<br>
     * cause = UnsupportedEncodingException<br> }<br>
     * (ԕω) O:Ox:error<br>
     * Character encoding error.<br>
     * 
     * <br>
     * argCharsetsȕR[h̏ꍇAOX[AAOo͂邱ƂmFB<br>
     * isr.close\bhĂяoȂƂmFB <br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testUnmarshalInputStreamStringObject04() throws Exception {
        // O
        CastorOXMapperImpl_CastorOXMapperImplStub02 oxmapper = new CastorOXMapperImpl_CastorOXMapperImplStub02();

        // 
        CastorOXMapperImpl_InputStreamStub01 in = new CastorOXMapperImpl_InputStreamStub01();
        String argCharset = "abc";
        Object out = new Object();

        // eXg{
        try {
            oxmapper.unmarshal(in, argCharset, out);
            fail();
        } catch (CastorUnsupportedEncodingException e) {
            // 
            assertSame(UnsupportedEncodingException.class, e.getCause()
                    .getClass());

            // OmF
            String message = "Character encoding error.";
            assertTrue(LogUTUtil.checkError(message, e.getCause()));

            // close\bhĂяoȂƂ̊mF
            assertFalse(in.isClose);
        }

    }

    /**
     * testUnmarshalInputStreamStringObject05() <br>
     * <br>
     * 
     * (ُn) <br>
     * ϓ_FA,G <br>
     * <br>
     * ͒lF() is:not null<br>
     * () argCharset:"UTF-8"<br>
     * () out:not null<br>
     * () unmarshal():ĂяoꂽƂmFB<br>
     * nꂽƂmFB<br>
     * () close():ĂяoꂽƂmFB<br>
     * nꂽƂmFB<br>
     * () this.charset:"abc"<br>
     * 0
     * 
     * <br>
     * ҒlF <br>
     * argCharset̕R[hݒ肳Aunmarshal\bhAisr.close\bhĂяo邱ƂmFB <br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testUnmarshalInputStreamStringObject05() throws Exception {
        // O
        CastorOXMapperImpl_CastorOXMapperImplStub02 oxmapper = new CastorOXMapperImpl_CastorOXMapperImplStub02();
        UTUtil.setPrivateField(oxmapper, "charset", "abc");

        // 
        CastorOXMapperImpl_InputStreamStub01 in = new CastorOXMapperImpl_InputStreamStub01();
        String argCharset = "UTF-8";
        Object out = new Object();

        // eXg{
        oxmapper.unmarshal(in, argCharset, out);

        // 
        // unmarshal\bhɈnꂽƂ̊mF
        assertSame(in, UTUtil.getPrivateField(oxmapper.reader, "lock"));
        assertSame(out, oxmapper.out);
        assertEquals("UTF8", oxmapper._charset);

        // close\bhĂяoȂƂ̊mF
        assertTrue(in.isClose);
    }

    /**
     * testCreateUnmarshaller01() <br>
     * <br>
     * 
     * (ُn) <br>
     * ϓ_FA,C,G <br>
     * <br>
     * ͒lF() out:null<br>
     * () this.cache:false<br>
     * () this.mappingFileCache:HashMap<br>
     * 
     * <br>
     * ҒlF(ԕω) O:IllegalArgumentException{<br>
     * message = "Unmarshal object is null."<br> }<br>
     * (ԕω) O:Ox:error<br>
     * Unmarshal object is null.<br>
     * (ԕω) this.mappingFileCache:HashMap<br>
     * 
     * <br>
     * outnull̏ꍇAOX[AOo͂邱ƂmF <br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testCreateUnmarshaller01() throws Exception {
        // O
        CastorOXMapperImpl oxmapper = new CastorOXMapperImpl();

        // LbV
        UTUtil.setPrivateField(oxmapper, "cache", false);
        Map<Class, Mapping> map = new HashMap<Class, Mapping>();
        UTUtil.setPrivateField(oxmapper, "mappingFileCache", map);

        // 
        Object out = null;

        // eXg{
        try {
            oxmapper.createUnmarshaller(out);
            fail();
        } catch (IllegalArgumentException e) {

            // 
            String message = "Unmarshal object is null.";
            assertEquals(message, e.getMessage());

            // OmF
            assertTrue(LogUTUtil.checkError(message));
        }
    }

    /**
     * testCreateUnmarshaller02() <br>
     * <br>
     * 
     * (ُn) <br>
     * ϓ_FA,G <br>
     * <br>
     * ͒lF() out:CastorOXMapperImpl_Stub01<br>
     * () }bsO`t@C:sȃ}bsO`t@C<br>
     * <?xml version="1.0"?><br>
     * <!DOCTYPE mapping PUBLIC "-//EXOLAB/Castor Object Mapping DTD Version
     * 1.0//EN" "http://castor.exolab.org/mapping.dtd"><br>
     * <mapping><br>
     * <class name="jp.terasoluna.fw.oxm.mapper.castor.xxxDTO"><br>
     * <field name="param1" type="int" /><br>
     * <field name="param2" type="date" /><br>
     * <field name="param3" type="string" /><br>
     * </class><br>
     * </mapping><br>
     * () getCastorMapping():ĂяoꂽƂmFB<br>
     * nꂽƂmFB<br>
     * () this.cache:false<br>
     * () this.mappingFileCache:HashMap<br>
     * 
     * <br>
     * ҒlF(ԕω) O:CastorMappingException{<br>
     * cause = MappingException<br> }<br>
     * (ԕω) O:Ox:error<br>
     * Castor mapping file is invalid.<br>
     * (ԕω) this.mappingFileCache:HashMap<br>
     * 
     * <br>
     * }bsO`t@C̒`ɖ肪ꍇAOX[AOo͂邱ƂmFB <br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testCreateUnmarshaller02() throws Exception {
        // O
        CastorOXMapperImpl_CastorOXMapperImplStub03 oxmapper = new CastorOXMapperImpl_CastorOXMapperImplStub03();
        oxmapper.mappingPath = packagePath
                + "CastorOXMapperImpl_JavaBeanStub02" + mappingSuffix;

        // LbV
        UTUtil.setPrivateField(oxmapper, "cache", false);
        Map<Class, Mapping> map = new HashMap<Class, Mapping>();
        UTUtil.setPrivateField(oxmapper, "mappingFileCache", map);

        // 
        Object out = new Object();

        // eXg{
        try {
            oxmapper.createUnmarshaller(out);
            fail();
        } catch (CastorMappingException e) {
            assertSame(MappingException.class, e.getCause().getClass());
            // OmF
            String message = "Castor mapping file is invalid. "
                    + "- [root-classpath]/java/lang/Object.xml";
            assertTrue(LogUTUtil.checkError(message, e.getCause()));
        }

    }

    /**
     * testCreateUnmarshaller03() <br>
     * <br>
     * 
     * (n) <br>
     * ϓ_FA,D <br>
     * <br>
     * ͒lF() out:CastorOXMapperImpl_Stub01<br>
     * () }bsO`t@C:ȃ}bsO`t@C<br>
     * <?xml version="1.0"?><br>
     * <!DOCTYPE mapping PUBLIC "-//EXOLAB/Castor Object Mapping DTD Version
     * 1.0//EN" "http://castor.exolab.org/mapping.dtd"><br>
     * <mapping><br>
     * <class
     * name=""jp.terasoluna.fw.oxm.mapper.castor.CastorOXMapperImpl_Stub01"><br>
     * <field name="param1" type="int" /><br>
     * <field name="param2" type="date" /><br>
     * <field name="param3" type="string" /><br>
     * </class><br>
     * </mapping><br>
     * () getCastorMapping():ĂяoꂽƂmFB<br>
     * nꂽƂmFB<br>
     * () this.cache:false<br>
     * () this.mappingFileCache:HashMap<br>
     * 
     * <br>
     * ҒlF(߂l) Unmarshaller:Unmarshaller{<br>
     * _validate = false;<br>
     * _cdResolver._mappingLoader._javaClass[0]=
     * .CastorOXMapperImpl_Stub01.class<br> }<br>
     * (ԕω) this.mappingFileCache:HashMap<br>
     * 
     * <br>
     * }bsO`ɖ肪ȂꍇA`ꂽNXݒ肵A}[Vԋp邱ƂmFB<br>
     * }bsOIuWFNg̃LbVsȂȂƂmFB <br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testCreateUnmarshaller03() throws Exception {
        // O
        CastorOXMapperImpl_CastorOXMapperImplStub03 oxmapper = new CastorOXMapperImpl_CastorOXMapperImplStub03();
        oxmapper.mappingPath = packagePath
                + "CastorOXMapperImpl_JavaBeanStub01" + mappingSuffix;
        Class testClass = CastorOXMapperImpl_JavaBeanStub01.class;

        // LbV
        UTUtil.setPrivateField(oxmapper, "cache", false);
        Map<Class, Mapping> map = new HashMap<Class, Mapping>();
        UTUtil.setPrivateField(oxmapper, "mappingFileCache", map);

        // 
        Object out = new Object();

        // eXg{
        Unmarshaller unmarshaller = oxmapper.createUnmarshaller(out);
        // 
        // unmarshal\bhɈnꂽƂ̊mF
        assertEquals(out.getClass(), oxmapper.mappingClass);
        // UnmarshallervalidationfalseɂȂĂ邱Ƃ̊mF
        Boolean validate = (Boolean) UTUtil.getPrivateField(unmarshaller,
                "_validate");
        assertFalse(validate.booleanValue());
        // UnmarshallerCastorOXMapperImpl_Stub01NXݒ肳Ă邱ƂmF
        ClassDescriptorResolver _cdResolver = (ClassDescriptorResolver) UTUtil
                .getPrivateField(unmarshaller, "_cdResolver");
        AbstractMappingLoader2 ml = (AbstractMappingLoader2) _cdResolver
                .getMappingLoader();
        Vector descriptors = (Vector) UTUtil
                .getPrivateField(ml, "_descriptors");
        assertEquals(1, descriptors.size());
        XMLClassDescriptorAdapter adapter = (XMLClassDescriptorAdapter) descriptors
                .get(0);
        assertEquals(testClass, adapter.getJavaClass());

        // }bsOIuWFNgLbVĂȂƂmF
        Map resultMap = (HashMap) UTUtil.getPrivateField(oxmapper,
                "mappingFileCache");
        assertEquals(0, resultMap.size());
    }

    /**
     * testCreateUnmarshaller04() <br>
     * <br>
     * 
     * (n) <br>
     * ϓ_FA,D <br>
     * <br>
     * ͒lF() out:CastorOXMapperImpl_Stub01<br>
     * () }bsO`t@C:ȃ}bsO`t@C<br>
     * <?xml version="1.0"?><br>
     * <!DOCTYPE mapping PUBLIC "-//EXOLAB/Castor Object Mapping DTD Version
     * 1.0//EN" "http://castor.exolab.org/mapping.dtd"><br>
     * <mapping><br>
     * <class
     * name=""jp.terasoluna.fw.oxm.mapper.castor.CastorOXMapperImpl_Stub01"><br>
     * <field name="param1" type="int" /><br>
     * <field name="param2" type="date" /><br>
     * <field name="param3" type="string" /><br>
     * </class><br>
     * </mapping><br>
     * () getCastorMapping():ĂяoꂽƂmFB<br>
     * nꂽƂmFB<br>
     * () this.cache:true<br>
     * () this.mappingFileCache:HashMap<br>
     * 
     * <br>
     * ҒlF(߂l) Unmarshaller:Unmarshaller{<br>
     * _validate = false;<br>
     * _cdResolver._mappingLoader._javaClass[0]=
     * .CastorOXMapperImpl_Stub01.class<br> }<br>
     * (ԕω) this.mappingFileCache:vf1HashMap<br>
     * HashMap<br> o<br>
     * ClassCX^X=MappingCX^X<br> }<br>
     * 
     * <br>
     * }bsO`ɖ肪ȂꍇA`ꂽNXݒ肵A}[Vԋp邱ƂmFB<br>
     * }bsOIuWFNg̃LbVsȂĂ邱ƂmFB <br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testCreateUnmarshaller04() throws Exception {
        // O
        CastorOXMapperImpl_CastorOXMapperImplStub03 oxmapper = new CastorOXMapperImpl_CastorOXMapperImplStub03();
        oxmapper.mappingPath = packagePath
                + "CastorOXMapperImpl_JavaBeanStub01" + mappingSuffix;
        Class testClass = CastorOXMapperImpl_JavaBeanStub01.class;

        // LbV
        UTUtil.setPrivateField(oxmapper, "cache", true);
        Map<Class, Mapping> map = new HashMap<Class, Mapping>();
        UTUtil.setPrivateField(oxmapper, "mappingFileCache", map);

        // 
        Object out = new Object();

        // eXg{
        Unmarshaller unmarshaller = oxmapper.createUnmarshaller(out);
        // 
        // unmarshal\bhɈnꂽƂ̊mF
        assertEquals(out.getClass(), oxmapper.mappingClass);
        // UnmarshallervalidationfalseɂȂĂ邱Ƃ̊mF
        Boolean validate = (Boolean) UTUtil.getPrivateField(unmarshaller,
                "_validate");
        assertFalse(validate.booleanValue());
        // UnmarshallerCastorOXMapperImpl_Stub01NXݒ肳Ă邱ƂmF
        ClassDescriptorResolver _cdResolver = (ClassDescriptorResolver) UTUtil
                .getPrivateField(unmarshaller, "_cdResolver");
        AbstractMappingLoader2 ml = (AbstractMappingLoader2) _cdResolver
                .getMappingLoader();
        Vector descriptors = (Vector) UTUtil
                .getPrivateField(ml, "_descriptors");
        assertEquals(1, descriptors.size());
        XMLClassDescriptorAdapter adapter = (XMLClassDescriptorAdapter) descriptors
                .get(0);
        assertEquals(testClass, adapter.getJavaClass());

        // }bsOIuWFNgLbVĂ邱ƂmF
        Map resultMap = (HashMap) UTUtil.getPrivateField(oxmapper,
                "mappingFileCache");
        assertEquals(1, resultMap.size());
        assertSame(oxmapper.mapping, resultMap.get(out.getClass()));

    }

    /**
     * testCreateUnmarshaller05() <br>
     * <br>
     * 
     * (n) <br>
     * ϓ_FA,D <br>
     * <br>
     * ͒lF() out:CastorOXMapperImpl_Stub01<br>
     * () }bsO`t@C:ȃ}bsO`t@C<br>
     * <?xml version="1.0"?><br>
     * <!DOCTYPE mapping PUBLIC "-//EXOLAB/Castor Object Mapping DTD Version
     * 1.0//EN" "http://castor.exolab.org/mapping.dtd"><br>
     * <mapping><br>
     * <class
     * name=""jp.terasoluna.fw.oxm.mapper.castor.CastorOXMapperImpl_Stub01"><br>
     * <field name="param1" type="int" /><br>
     * <field name="param2" type="date" /><br>
     * <field name="param3" type="string" /><br>
     * </class><br>
     * </mapping><br>
     * () getCastorMapping():ĂяoꂽƂmFB<br>
     * nꂽƂmFB<br>
     * () this.cache:true<br>
     * () this.mappingFileCache:vf1HashMap<br>
     * HashMap<br> o<br>
     * ClassCX^X=MappingCX^X<br> }<br>
     * 
     * <br>
     * ҒlF(߂l) Unmarshaller:Unmarshaller{<br>
     * _validate = false;<br>
     * _cdResolver._mappingLoader._javaClass[0]=
     * .CastorOXMapperImpl_Stub01.class<br> }<br>
     * (ԕω) this.mappingFileCache:vf1HashMap<br>
     * HashMap<br> o<br>
     * ClassCX^X=MappingCX^X<br> }<br>
     * 
     * <br>
     * }bsO`ɖ肪ȂꍇA`ꂽNXݒ肵A}[Vԋp邱ƂmFB<br>
     * }bsOIuWFNg̃LbVsȂꂸAvfĂȂƂmFB <br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testCreateUnmarshaller05() throws Exception {
        // O
        CastorOXMapperImpl_CastorOXMapperImplStub03 oxmapper = new CastorOXMapperImpl_CastorOXMapperImplStub03();
        oxmapper.mappingPath = packagePath
                + "CastorOXMapperImpl_JavaBeanStub01" + mappingSuffix;
        Class testClass = CastorOXMapperImpl_JavaBeanStub01.class;

        // 
        Object out = new Object();

        // LbV
        UTUtil.setPrivateField(oxmapper, "cache", true);
        Map<Class, Mapping> map = new HashMap<Class, Mapping>();
        Mapping mapping = new Mapping();
        map.put(out.getClass(), mapping);
        UTUtil.setPrivateField(oxmapper, "mappingFileCache", map);

        // eXg{
        Unmarshaller unmarshaller = oxmapper.createUnmarshaller(out);
        // 
        // unmarshal\bhɈnꂽƂ̊mF
        assertEquals(out.getClass(), oxmapper.mappingClass);
        // UnmarshallervalidationfalseɂȂĂ邱Ƃ̊mF
        Boolean validate = (Boolean) UTUtil.getPrivateField(unmarshaller,
                "_validate");
        assertFalse(validate.booleanValue());
        // UnmarshallerCastorOXMapperImpl_Stub01NXݒ肳Ă邱ƂmF
        ClassDescriptorResolver _cdResolver = (ClassDescriptorResolver) UTUtil
                .getPrivateField(unmarshaller, "_cdResolver");
        AbstractMappingLoader2 ml = (AbstractMappingLoader2) _cdResolver
                .getMappingLoader();
        Vector descriptors = (Vector) UTUtil
                .getPrivateField(ml, "_descriptors");
        assertEquals(1, descriptors.size());
        XMLClassDescriptorAdapter adapter = (XMLClassDescriptorAdapter) descriptors
                .get(0);
        assertEquals(testClass, adapter.getJavaClass());

        // }bsOIuWFNgLbVĂȂƂmF
        Map resultMap = (HashMap) UTUtil.getPrivateField(oxmapper,
                "mappingFileCache");
        assertEquals(1, resultMap.size());
        assertSame(mapping, resultMap.get(out.getClass()));
    }

    /**
     * testCreateUnmarshaller06() <br>
     * <br>
     * 
     * (n) <br>
     * ϓ_FA,D <br>
     * <br>
     * ͒lF<br>
     * () out:CastorOXMapperImpl_Stub01<br>
     * () preserveWhitespaceAtUnmarshal : true<br>
     * () }bsO`t@C:ȃ}bsO`t@C<br>
     * <?xml version="1.0"?><br>
     * <!DOCTYPE mapping PUBLIC "-//EXOLAB/Castor Object Mapping DTD Version
     * 1.0//EN" "http://castor.exolab.org/mapping.dtd"><br>
     * <mapping><br>
     * <class
     * name=""jp.terasoluna.fw.oxm.mapper.castor.CastorOXMapperImpl_Stub01"><br>
     * <field name="param1" type="int" /><br>
     * <field name="param2" type="date" /><br>
     * <field name="param3" type="string" /><br>
     * </class><br>
     * </mapping><br>
     * () getCastorMapping():ĂяoꂽƂmFB<br>
     * nꂽƂmFB<br>
     * () this.cache:false<br>
     * () this.mappingFileCache:HashMap<br>
     * 
     * <br>
     * ҒlF<br>
     * (ԕω) Unmarshaller̃tB[h_wsPreserve : true<br>
     * 
     * <br>
     * setPreserveWhitespaceAtUnmarshalŐݒ肳ꂽIvV UnmarshallerɓnĂ邱ƂmFB<br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testCreateUnmarshaller06() throws Exception {
        // O
        CastorOXMapperImpl_CastorOXMapperImplStub03 oxmapper = new CastorOXMapperImpl_CastorOXMapperImplStub03();
        oxmapper.mappingPath = packagePath
                + "CastorOXMapperImpl_JavaBeanStub01" + mappingSuffix;

        // LbV
        UTUtil.setPrivateField(oxmapper, "cache", false);
        Map<Class, Mapping> map = new HashMap<Class, Mapping>();
        UTUtil.setPrivateField(oxmapper, "mappingFileCache", map);

        // 
        Object out = new Object();

        UTUtil.setPrivateField(oxmapper, "preserveWhitespaceAtUnmarshal", true);

        // eXg{
        Unmarshaller unmarshaller = oxmapper.createUnmarshaller(out);

        // 
        Boolean wsPreserve = (Boolean) UTUtil.getPrivateField(unmarshaller,
                "_wsPreserve");
        assertTrue(wsPreserve);
    }

    /**
     * testCreateUnmarshaller07() <br>
     * <br>
     * 
     * (n) <br>
     * ϓ_FA,D <br>
     * <br>
     * ͒lF<br>
     * () out:CastorOXMapperImpl_Stub01<br>
     * () preserveWhitespaceAtUnmarshal : false<br>
     * () }bsO`t@C:ȃ}bsO`t@C<br>
     * <?xml version="1.0"?><br>
     * <!DOCTYPE mapping PUBLIC "-//EXOLAB/Castor Object Mapping DTD Version
     * 1.0//EN" "http://castor.exolab.org/mapping.dtd"><br>
     * <mapping><br>
     * <class
     * name=""jp.terasoluna.fw.oxm.mapper.castor.CastorOXMapperImpl_Stub01"><br>
     * <field name="param1" type="int" /><br>
     * <field name="param2" type="date" /><br>
     * <field name="param3" type="string" /><br>
     * </class><br>
     * </mapping><br>
     * () getCastorMapping():ĂяoꂽƂmFB<br>
     * nꂽƂmFB<br>
     * () this.cache:false<br>
     * () this.mappingFileCache:HashMap<br>
     * 
     * <br>
     * ҒlF<br>
     * (ԕω) Unmarshaller̃tB[h_wsPreserve : false<br>
     * 
     * <br>
     * setPreserveWhitespaceAtUnmarshalŐݒ肳ꂽIvV UnmarshallerɓnĂ邱ƂmFB<br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testCreateUnmarshaller07() throws Exception {
        // O
        CastorOXMapperImpl_CastorOXMapperImplStub03 oxmapper = new CastorOXMapperImpl_CastorOXMapperImplStub03();
        oxmapper.mappingPath = packagePath
                + "CastorOXMapperImpl_JavaBeanStub01" + mappingSuffix;

        // LbV
        UTUtil.setPrivateField(oxmapper, "cache", false);
        Map<Class, Mapping> map = new HashMap<Class, Mapping>();
        UTUtil.setPrivateField(oxmapper, "mappingFileCache", map);

        // 
        Object out = new Object();

        UTUtil.setPrivateField(oxmapper, "preserveWhitespaceAtUnmarshal", false);

        // eXg{
        Unmarshaller unmarshaller = oxmapper.createUnmarshaller(out);

        // 
        Boolean wsPreserve = (Boolean) UTUtil.getPrivateField(unmarshaller,
                "_wsPreserve");
        assertFalse(wsPreserve);
    }

    /**
     * testMarshal01() <br>
     * <br>
     * 
     * (ُn) <br>
     * ϓ_FA,C,G <br>
     * <br>
     * ͒lF() in:not null<br>
     * () writer:not null<br>
     * () createMarshaller():ĂяoꂽƂmFB<br>
     * nꂽƂmFB<br>
     * () marshaller.marshal():MarshalExceptionX[<br>
     * 
     * <br>
     * ҒlF(ԕω) O:CastorMarshalException{<br>
     * cause = MarshalException<br> }<br>
     * (ԕω) O:Ox:error<br>
     * Writer object is invalid.<br>
     * 
     * <br>
     * sȃIuWFNg}[Vɐݒ肳ꂽꍇAOX[邱ƂmFB <br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testMarshal01() throws Exception {
        // O
        CastorOXMapperImpl_CastorOXMapperImplStub01 oxmapper = new CastorOXMapperImpl_CastorOXMapperImplStub01();
        CastorOXMapperImpl_MarshallerStub01 marshaller = new CastorOXMapperImpl_MarshallerStub01(
                new StringWriter());
        oxmapper.marshaller = marshaller;

        Object in = new Object();
        Writer writer = new StringWriter();

        try {
            // eXg{
            oxmapper.marshal(in, writer);
            fail();
        } catch (CastorMarshalException e) {
            // 
            assertSame(MarshalException.class, e.getCause().getClass());

            // OmF
            String message = "Castor marshal failure.";
            assertTrue(LogUTUtil.checkError(message, e.getCause()));

            // createMarshaller\bhɈnꂽƂ̊mF
            assertSame(in, oxmapper.in);
            assertSame(writer, oxmapper.writer);

        }
    }

    /**
     * testMarshal02() <br>
     * <br>
     * 
     * (ُn) <br>
     * ϓ_FA,C,G <br>
     * <br>
     * ͒lF() in:not null<br>
     * () writer:not null<br>
     * () createMarshaller():ĂяoꂽƂmFB<br>
     * nꂽƂmFB<br>
     * () marshaller.marshal():ValidationExceptionX[<br>
     * 
     * <br>
     * ҒlF(ԕω) O:CastorValidationException{<br>
     * cause = ValidationException<br> }<br>
     * (ԕω) O:Ox:error<br>
     * Castor validation error.<br>
     * 
     * <br>
     * }[VɃof[VG[ꍇAOX[邱ƂmFB<br>
     * iۂɂ̗͂O̓X[Ȃj <br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testMarshal02() throws Exception {
        // O
        CastorOXMapperImpl_CastorOXMapperImplStub01 oxmapper = new CastorOXMapperImpl_CastorOXMapperImplStub01();
        CastorOXMapperImpl_MarshallerStub02 marshaller = new CastorOXMapperImpl_MarshallerStub02(
                new StringWriter());
        oxmapper.marshaller = marshaller;

        Object in = new Object();
        Writer writer = new StringWriter();

        try {
            // eXg{
            oxmapper.marshal(in, writer);
            fail();
        } catch (CastorValidationException e) {
            // 
            assertSame(ValidationException.class, e.getCause().getClass());

            // OmF
            String message = "Castor validation error.";
            assertTrue(LogUTUtil.checkError(message, e.getCause()));

            // createMarshaller\bhɈnꂽƂ̊mF
            assertSame(in, oxmapper.in);
            assertSame(writer, oxmapper.writer);
        }
    }

    /**
     * testMarshal03() <br>
     * <br>
     * 
     * (n) <br>
     * ϓ_FA <br>
     * <br>
     * ͒lF<br>
     * () in:null<br>
     * () writer:null<br>
     * () createMarshaller():ĂяoꂽƂmFB<br>
     * nꂽƂmFB<br>
     * () marshaller.marshal():ĂяoꂽƂmFB<br>
     * nꂽƂmFB<br>
     * 
     * <br>
     * ҒlF <br>
     * ñp^[B<br>
     * out,writernull̏ꍇAmarsharl\bhĂяoꂽƂ̊mFsB <br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testMarshal03() throws Exception {
        // O
        CastorOXMapperImpl_CastorOXMapperImplStub01 oxmapper = new CastorOXMapperImpl_CastorOXMapperImplStub01();
        CastorOXMapperImpl_MarshallerStub03 marshaller = new CastorOXMapperImpl_MarshallerStub03(
                new StringWriter());
        oxmapper.marshaller = marshaller;

        CastorOXMapperImpl_UnmarshallerStub01 unmarshaller = new CastorOXMapperImpl_UnmarshallerStub01();
        oxmapper.unmarshaller = unmarshaller;

        Object in = null;
        Writer writer = null;

        // eXg{
        oxmapper.marshal(in, writer);

        // 
        // createMarshaller\bhɈnꂽƂ̊mF
        assertNull(oxmapper.in);
        assertNull(oxmapper.writer);

        // marshaller\bhɈnꂽƂ̊mF
        assertNull(marshaller.in);

    }

    /**
     * testMarshal04() <br>
     * <br>
     * 
     * (n) <br>
     * ϓ_FA <br>
     * <br>
     * ͒lF<br>
     * () in:not null<br>
     * () writer:not null<br>
     * () createMarshaller():ĂяoꂽƂmFB<br>
     * nꂽƂmFB<br>
     * () marshaller.marshal():ĂяoꂽƂmFB<br>
     * nꂽƂmFB<br>
     * 
     * <br>
     * ҒlF <br>
     * ñp^[B<br>
     * marsharl\bhĂяoꂽƂ̊mFsB <br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testMarshal04() throws Exception {
        // O
        CastorOXMapperImpl_CastorOXMapperImplStub01 oxmapper = new CastorOXMapperImpl_CastorOXMapperImplStub01();
        CastorOXMapperImpl_MarshallerStub03 marshaller = new CastorOXMapperImpl_MarshallerStub03(
                new StringWriter());
        oxmapper.marshaller = marshaller;

        Object in = new Object();
        Writer writer = new StringWriter();

        // eXg{
        oxmapper.marshal(in, writer);

        // 
        // createMarshaller\bhɈnꂽƂ̊mF
        assertSame(in, oxmapper.in);
        assertSame(writer, oxmapper.writer);

        // marshaller\bhɈnꂽƂ̊mF
        assertSame(in, marshaller.in);

    }

    /**
     * testCreateMarshaller01() <br>
     * <br>
     * 
     * (ُn) <br>
     * ϓ_FA,C,G <br>
     * <br>
     * ͒lF() in:null<br>
     * () writer:not null<br>
     * () this.cache:false<br>
     * () this.mappingFileCache:HashMap<br>
     * 
     * <br>
     * ҒlF(ԕω) O:IllegalArgumentException{<br>
     * message="Marshal object is null."<br> }<br>
     * (ԕω) O:Ox:error<br>
     * Marshal object is null.<br>
     * (ԕω) this.mappingFileCache:HashMap<br>
     * 
     * <br>
     * innull̏ꍇAOX[邱ƂmF <br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testCreateMarshaller01() throws Exception {
        // O
        CastorOXMapperImpl oxmapper = new CastorOXMapperImpl();

        // 
        Object in = null;
        Writer writer = new StringWriter();

        // eXg{
        try {
            oxmapper.createMarshaller(in, writer);
            fail();
        } catch (IllegalArgumentException e) {

            // 
            String message = "Marshall object is null.";
            assertEquals(message, e.getMessage());

            // OmF
            assertTrue(LogUTUtil.checkError(message));
        }
    }

    /**
     * testCreateMarshaller02() <br>
     * <br>
     * 
     * (ُn) <br>
     * ϓ_FA,C,G <br>
     * <br>
     * ͒lF() in:not null<br>
     * () writer:null<br>
     * () this.cache:false<br>
     * () this.mappingFileCache:HashMap<br>
     * 
     * <br>
     * ҒlF(ԕω) O:IllegalArgumentException{<br>
     * message="Writer is null."<br> }<br>
     * (ԕω) O:Ox:error<br>
     * Writer is null.<br>
     * (ԕω) this.mappingFileCache:HashMap<br>
     * 
     * <br>
     * writernull̏ꍇAOX[邱ƂmF <br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testCreateMarshaller02() throws Exception {
        // O
        CastorOXMapperImpl oxmapper = new CastorOXMapperImpl();

        // 
        Object in = new Object();
        Writer writer = null;

        // eXg{
        try {
            oxmapper.createMarshaller(in, writer);
            fail();
        } catch (IllegalArgumentException e) {

            // 
            String message = "Writer is null.";
            assertEquals(message, e.getMessage());

            // OmF
            assertTrue(LogUTUtil.checkError(message));
        }
    }

    /**
     * testCreateMarshaller03() <br>
     * <br>
     * 
     * (ُn) <br>
     * ϓ_FA,C,G <br>
     * <br>
     * ͒lF() in:CastorOXMapperImpl_Stub01<br>
     * () writer:not null<br>
     * () }bsO`t@C:sȃ}bsO`t@C<br>
     * <?xml version="1.0"?><br>
     * <!DOCTYPE mapping PUBLIC "-//EXOLAB/Castor Object Mapping DTD Version
     * 1.0//EN" "http://castor.exolab.org/mapping.dtd"><br>
     * <mapping><br>
     * <class name="jp.terasoluna.fw.oxm.mapper.castor.xxxDTO"><br>
     * <field name="param1" type="int" /><br>
     * <field name="param2" type="date" /><br>
     * <field name="param3" type="int" /><br>
     * </class><br>
     * </mapping><br>
     * () getCastorMapping():ĂяoꂽƂmFB<br>
     * nꂽƂmFB<br>
     * () this.cache:false<br>
     * () this.mappingFileCache:HashMap<br>
     * 
     * <br>
     * ҒlF(ԕω) O:CastorMappingException{<br>
     * cause = MappingException<br> }<br>
     * (ԕω) O:Ox:error<br>
     * Castor mapping file is invalid.<br>
     * (ԕω) this.mappingFileCache:HashMap<br>
     * 
     * <br>
     * }bsO`t@CsȏꍇAOX[邱ƂmFB <br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testCreateMarshaller03() throws Exception {
        // O
        CastorOXMapperImpl_CastorOXMapperImplStub03 oxmapper = new CastorOXMapperImpl_CastorOXMapperImplStub03();
        oxmapper.mappingPath = packagePath
                + "CastorOXMapperImpl_JavaBeanStub02" + mappingSuffix;

        // 
        Object in = new Object();
        Writer writer = new StringWriter();

        // eXg{
        try {
            oxmapper.createMarshaller(in, writer);
            fail();
        } catch (CastorMappingException e) {
            assertSame(MappingException.class, e.getCause().getClass());
            // OmF
            String message = "Castor mapping file is invalid. "
                    + "- [root-classpath]/java/lang/Object.xml";
            assertTrue(LogUTUtil.checkError(message, e.getCause()));
        }
    }

    /**
     * testCreateMarshaller04() <br>
     * <br>
     * 
     * (n) <br>
     * ϓ_FA,D <br>
     * <br>
     * ͒lF() in:CastorOXMapperImpl_Stub01<br>
     * () writer:not null<br>
     * () }bsO`t@C:ȃ}bsO`t@C<br>
     * <?xml version="1.0"?><br>
     * <!DOCTYPE mapping PUBLIC "-//EXOLAB/Castor Object Mapping DTD Version
     * 1.0//EN" "http://castor.exolab.org/mapping.dtd"><br>
     * <mapping><br>
     * <class
     * name="jp.terasoluna.fw.oxm.mapper.castor.CastorOXMapperImpl_Stub01"><br>
     * <field name="param1" type="int" /><br>
     * <field name="param2" type="date" /><br>
     * <field name="param3" type="string" /><br>
     * </class><br>
     * </mapping><br>
     * () getCastorMapping():ĂяoꂽƂmFB<br>
     * nꂽƂmFB<br>
     * () this.cache:false<br>
     * () this.mappingFileCache:HashMap<br>
     * 
     * <br>
     * ҒlF(߂l) Marshaller:Marshaller{<br>
     * _validate = false;<br>
     * _cdResolver._mappingLoader._javaClass[0]=CastorOXMapperImpl_Stub01.class<br> }<br>
     * (ԕω) this.mappingFileCache:vf1HashMap<br>
     * HashMap<br> o<br>
     * ClassCX^X=MappingCX^X<br> }<br>
     * 
     * <br>
     * }bsO`t@CꍇAɃ}[V邱ƂmFB<br>
     * }bsOIuWFNg̃LbVsȂȂƂmFB <br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testCreateMarshaller04() throws Exception {
        // O
        CastorOXMapperImpl_CastorOXMapperImplStub03 oxmapper = new CastorOXMapperImpl_CastorOXMapperImplStub03();
        oxmapper.mappingPath = packagePath
                + "CastorOXMapperImpl_JavaBeanStub01" + mappingSuffix;
        Class testClass = CastorOXMapperImpl_JavaBeanStub01.class;

        // LbV
        UTUtil.setPrivateField(oxmapper, "cache", false);
        Map<Class, Mapping> map = new HashMap<Class, Mapping>();
        UTUtil.setPrivateField(oxmapper, "mappingFileCache", map);

        // 
        Object in = new Object();
        Writer writer = new StringWriter();

        // eXg{
        Marshaller marshaller = oxmapper.createMarshaller(in, writer);

        // 
        // marshal\bhɈnꂽƂ̊mF
        // out
        assertSame(in.getClass(), oxmapper.mappingClass);

        // MarshallervalidationfalseɂȂĂ邱Ƃ̊mF
        Boolean validate = (Boolean) UTUtil.getPrivateField(marshaller,
                "_validate");
        assertFalse(validate.booleanValue());
        // MarshallerCastorOXMapperImpl_Stub01NXݒ肳Ă邱ƂmF
        ClassDescriptorResolver _cdResolver = (ClassDescriptorResolver) UTUtil
                .getPrivateField(marshaller, "_cdResolver");
        AbstractMappingLoader2 ml = (AbstractMappingLoader2) _cdResolver
                .getMappingLoader();
        Vector descriptors = (Vector) UTUtil
                .getPrivateField(ml, "_descriptors");
        assertEquals(1, descriptors.size());
        XMLClassDescriptorAdapter adapter = (XMLClassDescriptorAdapter) descriptors
                .get(0);
        assertEquals(testClass, adapter.getJavaClass());

        // }bsOIuWFNgLbVĂȂƂmF
        Map resultMap = (HashMap) UTUtil.getPrivateField(oxmapper,
                "mappingFileCache");
        assertEquals(0, resultMap.size());
    }

    /**
     * testCreateMarshaller05() <br>
     * <br>
     * 
     * (n) <br>
     * ϓ_FA,D <br>
     * <br>
     * ͒lF() in:CastorOXMapperImpl_Stub01<br>
     * () writer:not null<br>
     * () }bsO`t@C:ȃ}bsO`t@C<br>
     * <?xml version="1.0"?><br>
     * <!DOCTYPE mapping PUBLIC "-//EXOLAB/Castor Object Mapping DTD Version
     * 1.0//EN" "http://castor.exolab.org/mapping.dtd"><br>
     * <mapping><br>
     * <class
     * name="jp.terasoluna.fw.oxm.mapper.castor.CastorOXMapperImpl_Stub01"><br>
     * <field name="param1" type="int" /><br>
     * <field name="param2" type="date" /><br>
     * <field name="param3" type="string" /><br>
     * </class><br>
     * </mapping><br>
     * () getCastorMapping():ĂяoꂽƂmFB<br>
     * nꂽƂmFB<br>
     * () this.cache:true<br>
     * () this.mappingFileCache:HashMap<br>
     * 
     * <br>
     * ҒlF(߂l) Marshaller:Marshaller{<br>
     * _validate = false;<br>
     * _cdResolver._mappingLoader._javaClass[0]=CastorOXMapperImpl_Stub01.class<br> }<br>
     * (ԕω) this.mappingFileCache:vf1HashMap<br>
     * HashMap<br> o<br>
     * ClassCX^X=MappingCX^X<br> }<br>
     * 
     * <br>
     * }bsO`t@CꍇAɃ}[V邱ƂmFB<br>
     * }bsOIuWFNg̃LbVsȂĂ邱ƂmFB <br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testCreateMarshaller05() throws Exception {
        // O
        CastorOXMapperImpl_CastorOXMapperImplStub03 oxmapper = new CastorOXMapperImpl_CastorOXMapperImplStub03();
        oxmapper.mappingPath = packagePath
                + "CastorOXMapperImpl_JavaBeanStub01" + mappingSuffix;
        Class testClass = CastorOXMapperImpl_JavaBeanStub01.class;

        // LbV
        UTUtil.setPrivateField(oxmapper, "cache", true);
        Map<Class, Mapping> map = new HashMap<Class, Mapping>();
        UTUtil.setPrivateField(oxmapper, "mappingFileCache", map);

        // 
        Object in = new Object();
        Writer writer = new StringWriter();

        // eXg{
        Marshaller marshaller = oxmapper.createMarshaller(in, writer);

        // 
        // marshal\bhɈnꂽƂ̊mF
        // out
        assertSame(in.getClass(), oxmapper.mappingClass);

        // MarshallervalidationfalseɂȂĂ邱Ƃ̊mF
        Boolean validate = (Boolean) UTUtil.getPrivateField(marshaller,
                "_validate");
        assertFalse(validate.booleanValue());
        // MarshallerCastorOXMapperImpl_Stub01NXݒ肳Ă邱ƂmF
        ClassDescriptorResolver _cdResolver = (ClassDescriptorResolver) UTUtil
                .getPrivateField(marshaller, "_cdResolver");
        AbstractMappingLoader2 ml = (AbstractMappingLoader2) _cdResolver
                .getMappingLoader();
        Vector descriptors = (Vector) UTUtil
                .getPrivateField(ml, "_descriptors");
        assertEquals(1, descriptors.size());
        XMLClassDescriptorAdapter adapter = (XMLClassDescriptorAdapter) descriptors
                .get(0);
        assertEquals(testClass, adapter.getJavaClass());

        // }bsOIuWFNgLbVĂ邱ƂmF
        Map resultMap = (HashMap) UTUtil.getPrivateField(oxmapper,
                "mappingFileCache");
        assertEquals(1, resultMap.size());
        assertSame(oxmapper.mapping, resultMap.get(in.getClass()));

    }

    /**
     * testCreateMarshaller06() <br>
     * <br>
     * 
     * (n) <br>
     * ϓ_:A,D <br>
     * <br>
     * ͒lF() in:CastorOXMapperImpl_Stub01<br>
     * () writer:not null<br>
     * () }bsO`t@C:ȃ}bsO`t@C<br>
     * <?xml version="1.0"?><br>
     * <!DOCTYPE mapping PUBLIC "-//EXOLAB/Castor Object Mapping DTD Version
     * 1.0//EN" "http://castor.exolab.org/mapping.dtd"><br>
     * <mapping><br>
     * <class
     * name="jp.terasoluna.fw.oxm.mapper.castor.CastorOXMapperImpl_Stub01"><br>
     * <field name="param1" type="int" /><br>
     * <field name="param2" type="date" /><br>
     * <field name="param3" type="string" /><br>
     * </class><br>
     * </mapping><br>
     * () getCastorMapping():ĂяoꂽƂmFB<br>
     * nꂽƂmFB<br>
     * () this.cache:true<br>
     * () this.mappingFileCache:vf1HashMap<br>
     * HashMap<br> o<br>
     * ClassCX^X=MappingCX^X<br> }<br>
     * 
     * <br>
     * ҒlF(߂l) Marshaller:Marshaller{<br>
     * _validate = false;<br>
     * _cdResolver._mappingLoader._javaClass[0]=CastorOXMapperImpl_Stub01.class<br> }<br>
     * (ԕω) this.mappingFileCache:vf1HashMap<br>
     * HashMap<br> o<br>
     * ClassCX^X=MappingCX^X<br> }<br>
     * 
     * <br>
     * }bsO`t@CꍇAɃ}[V邱ƂmFB<br>
     * }bsOIuWFNg̃LbVsȂꂸAvfĂȂƂmFB <br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testCreateMarshaller06() throws Exception {
        // O
        CastorOXMapperImpl_CastorOXMapperImplStub03 oxmapper = new CastorOXMapperImpl_CastorOXMapperImplStub03();
        oxmapper.mappingPath = packagePath
                + "CastorOXMapperImpl_JavaBeanStub01" + mappingSuffix;
        Class testClass = CastorOXMapperImpl_JavaBeanStub01.class;

        // 
        Object in = new Object();
        Writer writer = new StringWriter();

        // LbV
        UTUtil.setPrivateField(oxmapper, "cache", true);
        Map<Class, Mapping> map = new HashMap<Class, Mapping>();
        Mapping mapping = new Mapping();
        map.put(in.getClass(), mapping);
        UTUtil.setPrivateField(oxmapper, "mappingFileCache", map);

        // eXg{
        Marshaller marshaller = oxmapper.createMarshaller(in, writer);

        // 
        // marshal\bhɈnꂽƂ̊mF
        // out
        assertSame(in.getClass(), oxmapper.mappingClass);

        // MarshallervalidationfalseɂȂĂ邱Ƃ̊mF
        Boolean validate = (Boolean) UTUtil.getPrivateField(marshaller,
                "_validate");
        assertFalse(validate.booleanValue());
        // MarshallerCastorOXMapperImpl_Stub01NXݒ肳Ă邱ƂmF
        ClassDescriptorResolver _cdResolver = (ClassDescriptorResolver) UTUtil
                .getPrivateField(marshaller, "_cdResolver");
        AbstractMappingLoader2 ml = (AbstractMappingLoader2) _cdResolver
                .getMappingLoader();
        Vector descriptors = (Vector) UTUtil
                .getPrivateField(ml, "_descriptors");
        assertEquals(1, descriptors.size());
        XMLClassDescriptorAdapter adapter = (XMLClassDescriptorAdapter) descriptors
                .get(0);
        assertEquals(testClass, adapter.getJavaClass());

        // }bsOIuWFNgLbVĂȂƂmF
        Map resultMap = (HashMap) UTUtil.getPrivateField(oxmapper,
                "mappingFileCache");
        assertEquals(1, resultMap.size());
        assertSame(mapping, resultMap.get(in.getClass()));
    }

    /**
     * testCreateMarshaller07() <br>
     * <br>
     * 
     * (n) <br>
     * ϓ_:A,D <br>
     * <br>
     * ͒lF() in:CastorOXMapperImpl_Stub01<br>
     * () writer:not null<br>
     * () }bsO`t@C:ȃ}bsO`t@C<br>
     * <?xml version="1.0"?><br>
     * <!DOCTYPE mapping PUBLIC "-//EXOLAB/Castor Object Mapping DTD Version
     * 1.0//EN" "http://castor.exolab.org/mapping.dtd"><br>
     * <mapping><br>
     * <class
     * name="jp.terasoluna.fw.oxm.mapper.castor.CastorOXMapperImpl_Stub01"><br>
     * <field name="param1" type="int" /><br>
     * <field name="param2" type="date" /><br>
     * <field name="param3" type="string" /><br>
     * </class><br>
     * </mapping><br>
     * () this.sharedResolverForMarshaller:LbVĂ邱ƂmFi1ځj<br>
     * () this.sharedResolverForMarshaller:LbVĂ邱ƂmFi2ځj<br>
     * () this.sharedResolverForMarshaller:1ڂ2ڂ̃CX^Xł邱ƂmF<br>
     * 
     * <br>
     * }bsO`t@CꍇAClassDescriptorResolverLbVĂ邱ƂmFB<br>
     * ClassDescriptorResolver̃LbVsȂA2ڂɓCX^Xł邱ƂmFB<br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testCreateMarshaller07() throws Exception {
        // O
        CastorOXMapperImpl_CastorOXMapperImplStub03 oxmapper = new CastorOXMapperImpl_CastorOXMapperImplStub03();
        oxmapper.mappingPath = packagePath
                + "CastorOXMapperImpl_JavaBeanStub01" + mappingSuffix;

        // 
        Object in = new Object();
        Writer writer = new StringWriter();

        // LbV
        UTUtil.setPrivateField(oxmapper, "cache", true);
        Map<Class, Mapping> map = new HashMap<Class, Mapping>();
        Mapping mapping = new Mapping();
        map.put(in.getClass(), mapping);
        UTUtil.setPrivateField(oxmapper, "mappingFileCache", map);

        // eXg{
        Marshaller marshaller = oxmapper.createMarshaller(in, writer);
        assertNotNull(marshaller);

        // ]oLbVĂ邱Ƃ̊mF
        Object resolver1 = UTUtil.getPrivateField(oxmapper,
                "sharedResolverForMarshaller");
        assertNotNull(resolver1);
        assertTrue(resolver1 instanceof org.exolab.castor.xml.XMLClassDescriptorResolver);
        assertTrue(resolver1 instanceof org.exolab.castor.xml.ClassDescriptorResolver);

        // eXg{
        marshaller = oxmapper.createMarshaller(in, writer);
        assertNotNull(marshaller);

        // ]oLbVĂ邱Ƃ̊mF
        Object resolver2 = UTUtil.getPrivateField(oxmapper,
                "sharedResolverForMarshaller");
        assertNotNull(resolver2);
        assertTrue(resolver2 instanceof org.exolab.castor.xml.XMLClassDescriptorResolver);
        assertTrue(resolver2 instanceof org.exolab.castor.xml.ClassDescriptorResolver);

        // CX^Xł邱Ƃ̊mF
        assertEquals(resolver1, resolver2);

    }

    /**
     * testCreateMarshaller08() <br>
     * <br>
     * 
     * (n) <br>
     * ϓ_FA,D <br>
     * <br>
     * ͒lF<br>
     * () in:CastorOXMapperImpl_Stub01<br>
     * () writer:not null<br>
     * () suppressXSIType : true<br>
     * () }bsO`t@C:ȃ}bsO`t@C<br>
     * <?xml version="1.0"?><br>
     * <!DOCTYPE mapping PUBLIC "-//EXOLAB/Castor Object Mapping DTD Version
     * 1.0//EN" "http://castor.exolab.org/mapping.dtd"><br>
     * <mapping><br>
     * <class
     * name="jp.terasoluna.fw.oxm.mapper.castor.CastorOXMapperImpl_Stub01"><br>
     * <field name="param1" type="int" /><br>
     * <field name="param2" type="date" /><br>
     * <field name="param3" type="string" /><br>
     * </class><br>
     * </mapping><br>
     * () getCastorMapping():ĂяoꂽƂmFB<br>
     * nꂽƂmFB<br>
     * () this.cache:true<br>
     * () this.mappingFileCache:HashMap<br>
     * 
     * <br>
     * ҒlF<br>
     * (ԕω) Marshaller  _suppressXSIType : true<br>
     * 
     * <br>
     * suppressXSIType ̐ݒ肪 Marshaller ɓn邱ƂmFB<br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testCreateMarshaller08() throws Exception {
        // O
        CastorOXMapperImpl_CastorOXMapperImplStub03 oxmapper = new CastorOXMapperImpl_CastorOXMapperImplStub03();
        oxmapper.mappingPath = packagePath
                + "CastorOXMapperImpl_JavaBeanStub01" + mappingSuffix;

        // LbV
        UTUtil.setPrivateField(oxmapper, "cache", true);
        Map<Class, Mapping> map = new HashMap<Class, Mapping>();
        UTUtil.setPrivateField(oxmapper, "mappingFileCache", map);

        // 
        Object in = new Object();
        Writer writer = new StringWriter();

        UTUtil.setPrivateField(oxmapper, "suppressXSIType", true);

        // eXg{
        Marshaller marshaller = oxmapper.createMarshaller(in, writer);

        // 
        Boolean suppressXSIType = (Boolean) UTUtil.getPrivateField(marshaller, "_suppressXSIType");
        
        assertTrue(suppressXSIType);
    }

    /**
     * testCreateMarshaller09() <br>
     * <br>
     * 
     * (n) <br>
     * ϓ_FA,D <br>
     * <br>
     * ͒lF<br>
     * () in:CastorOXMapperImpl_Stub01<br>
     * () writer:not null<br>
     * () suppressXSIType : true<br>
     * () }bsO`t@C:ȃ}bsO`t@C<br>
     * <?xml version="1.0"?><br>
     * <!DOCTYPE mapping PUBLIC "-//EXOLAB/Castor Object Mapping DTD Version
     * 1.0//EN" "http://castor.exolab.org/mapping.dtd"><br>
     * <mapping><br>
     * <class
     * name="jp.terasoluna.fw.oxm.mapper.castor.CastorOXMapperImpl_Stub01"><br>
     * <field name="param1" type="int" /><br>
     * <field name="param2" type="date" /><br>
     * <field name="param3" type="string" /><br>
     * </class><br>
     * </mapping><br>
     * () getCastorMapping():ĂяoꂽƂmFB<br>
     * nꂽƂmFB<br>
     * () this.cache:true<br>
     * () this.mappingFileCache:HashMap<br>
     * 
     * <br>
     * ҒlF<br>
     * (ԕω) Marshaller  _suppressXSIType : false<br>
     * 
     * <br>
     * suppressXSIType ̐ݒ肪 Marshaller ɓn邱ƂmFB<br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testCreateMarshaller09() throws Exception {
        // O
        CastorOXMapperImpl_CastorOXMapperImplStub03 oxmapper = new CastorOXMapperImpl_CastorOXMapperImplStub03();
        oxmapper.mappingPath = packagePath
                + "CastorOXMapperImpl_JavaBeanStub01" + mappingSuffix;

        // LbV
        UTUtil.setPrivateField(oxmapper, "cache", true);
        Map<Class, Mapping> map = new HashMap<Class, Mapping>();
        UTUtil.setPrivateField(oxmapper, "mappingFileCache", map);

        // 
        Object in = new Object();
        Writer writer = new StringWriter();

        UTUtil.setPrivateField(oxmapper, "suppressXSIType", false);

        // eXg{
        Marshaller marshaller = oxmapper.createMarshaller(in, writer);

        // 
        Boolean suppressXSIType = (Boolean) UTUtil.getPrivateField(marshaller, "_suppressXSIType");
        
        assertFalse(suppressXSIType);
    }    

    /**
     * testCreateMarshaller10() <br>
     * <br>
     * 
     * (n) <br>
     * ϓ_FA,D <br>
     * <br>
     * ͒lF<br>
     * () in:CastorOXMapperImpl_Stub01<br>
     * () writer:not null<br>
     * () preserveWhitespaceAtMarshal : true<br>
     * () }bsO`t@C:ȃ}bsO`t@C<br>
     * <?xml version="1.0"?><br>
     * <!DOCTYPE mapping PUBLIC "-//EXOLAB/Castor Object Mapping DTD Version
     * 1.0//EN" "http://castor.exolab.org/mapping.dtd"><br>
     * <mapping><br>
     * <class
     * name="jp.terasoluna.fw.oxm.mapper.castor.CastorOXMapperImpl_Stub01"><br>
     * <field name="param1" type="int" /><br>
     * <field name="param2" type="date" /><br>
     * <field name="param3" type="string" /><br>
     * </class><br>
     * </mapping><br>
     * () getCastorMapping():ĂяoꂽƂmFB<br>
     * nꂽƂmFB<br>
     * () this.cache:true<br>
     * () this.mappingFileCache:HashMap<br>
     * 
     * <br>
     * ҒlF<br>
     * (ԕω) Marshaller  ContentHandler  preserveWhitespace : true<br>
     * 
     * <br>
     * preserveWhitespace ̐ݒ肪 Marshaller  ContentHandler ɓn邱ƂmFB<br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testCreateMarshaller10() throws Exception {
        // O
        CastorOXMapperImpl_CastorOXMapperImplStub03 oxmapper = new CastorOXMapperImpl_CastorOXMapperImplStub03();
        oxmapper.mappingPath = packagePath
                + "CastorOXMapperImpl_JavaBeanStub01" + mappingSuffix;

        // LbV
        UTUtil.setPrivateField(oxmapper, "cache", true);
        Map<Class, Mapping> map = new HashMap<Class, Mapping>();
        UTUtil.setPrivateField(oxmapper, "mappingFileCache", map);

        // 
        CastorOXMapperImpl_JavaBeanStub01 in = new CastorOXMapperImpl_JavaBeanStub01();
        Writer writer = new StringWriter();

        UTUtil.setPrivateField(oxmapper, "preserveWhitespaceAtMarshal", true);

        // eXg{
        Marshaller marshaller = oxmapper.createMarshaller(in, writer);
        marshaller.marshal(in);
        
        // 
        XMLSerializerEx handler = (XMLSerializerEx) UTUtil.getPrivateField(marshaller, "_handler");
        Boolean preserveWhitespace = (Boolean) UTUtil.getPrivateField(handler, "preserveWhitespace");
        
        assertTrue(preserveWhitespace);
    }       

    /**
     * testCreateMarshaller11() <br>
     * <br>
     * 
     * (n) <br>
     * ϓ_FA,D <br>
     * <br>
     * ͒lF<br>
     * () in:CastorOXMapperImpl_Stub01<br>
     * () writer:not null<br>
     * () preserveWhitespaceAtMarshal : false<br>
     * () }bsO`t@C:ȃ}bsO`t@C<br>
     * <?xml version="1.0"?><br>
     * <!DOCTYPE mapping PUBLIC "-//EXOLAB/Castor Object Mapping DTD Version
     * 1.0//EN" "http://castor.exolab.org/mapping.dtd"><br>
     * <mapping><br>
     * <class
     * name="jp.terasoluna.fw.oxm.mapper.castor.CastorOXMapperImpl_Stub01"><br>
     * <field name="param1" type="int" /><br>
     * <field name="param2" type="date" /><br>
     * <field name="param3" type="string" /><br>
     * </class><br>
     * </mapping><br>
     * () getCastorMapping():ĂяoꂽƂmFB<br>
     * nꂽƂmFB<br>
     * () this.cache:true<br>
     * () this.mappingFileCache:HashMap<br>
     * 
     * <br>
     * ҒlF<br>
     * (ԕω) Marshaller  ContentHandler  preserveWhitespace : false<br>
     * 
     * <br>
     * preserveWhitespace ̐ݒ肪 Marshaller  ContentHandler ɓn邱ƂmFB<br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testCreateMarshaller11() throws Exception {
        // O
        CastorOXMapperImpl_CastorOXMapperImplStub03 oxmapper = new CastorOXMapperImpl_CastorOXMapperImplStub03();
        oxmapper.mappingPath = packagePath
                + "CastorOXMapperImpl_JavaBeanStub01" + mappingSuffix;

        // LbV
        UTUtil.setPrivateField(oxmapper, "cache", true);
        Map<Class, Mapping> map = new HashMap<Class, Mapping>();
        UTUtil.setPrivateField(oxmapper, "mappingFileCache", map);

        // 
        CastorOXMapperImpl_JavaBeanStub01 in = new CastorOXMapperImpl_JavaBeanStub01();
        Writer writer = new StringWriter();

        UTUtil.setPrivateField(oxmapper, "preserveWhitespaceAtMarshal", false);

        // eXg{
        Marshaller marshaller = oxmapper.createMarshaller(in, writer);
        marshaller.marshal(in);

        // 
        XMLSerializerEx handler = (XMLSerializerEx) UTUtil.getPrivateField(marshaller, "_handler");
        Boolean preserveWhitespace = (Boolean) UTUtil.getPrivateField(handler, "preserveWhitespace");
        
        assertFalse(preserveWhitespace);
    }     
    
    /**
     * testGetCastorMapping01() <br>
     * <br>
     * 
     * (n) <br>
     * ϓ_FA <br>
     * <br>
     * ͒lF() mappingClass:ClassCX^X<br>
     * () this.cache:true<br>
     * () this.mappingFileCache:vf1HashMap<br>
     * HashMap<br> o<br>
     * ClassCX^X=MappingCX^X<br> }<br>
     * 
     * <br>
     * ҒlF(߂l) Mapping:LbVꂽMappingCX^X<br>
     * (ԕω) this.mappingFileCache:vf1HashMap<br>
     * HashMap<br> o<br>
     * ClassCXX=MappingCX^X<br> }<br>
     * 
     * <br>
     * ̃CX^XL[Ƃ}bsOCX^XLbVɑ݂ꍇALbVꂽCX^Xԋp邱ƂmFB <br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    @SuppressWarnings("unchecked")
    public void testGetCastorMapping01() throws Exception {
        // O
        CastorOXMapperImpl oxmapper = new CastorOXMapperImpl();
        UTUtil.setPrivateField(oxmapper, "cache", true);

        Class testClass = CastorOXMapperImpl_JavaBeanStub01.class;
        Mapping mapping = new Mapping();
        HashMap map = (HashMap) UTUtil.getPrivateField(oxmapper,
                "mappingFileCache");
        map.put(testClass, mapping);

        // eXg{
        Mapping result = oxmapper.getCastorMapping(testClass);

        // 
        assertSame(mapping, result);

    }

    /**
     * testGetCastorMapping02() <br>
     * <br>
     * 
     * (n) <br>
     * ϓ_FA <br>
     * <br>
     * ͒lF() mappingClass:ClassCX^X<br>
     * () this.cache:true<br>
     * () this.mappingFileCache:not null<br>
     * () getUrl():null<br>
     * 
     * <br>
     * ҒlF(߂l) null<br>
     * 
     * <br>
     * ̃IuWFNgƓpXɃt@@C݂ȂꍇAnullԋp邱ƂmFB <br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testGetCastorMapping02() throws Exception {
        // O
        CastorOXMapperImpl_CastorOXMapperImplStub04 oxmapper = new CastorOXMapperImpl_CastorOXMapperImplStub04();
        UTUtil.setPrivateField(oxmapper, "cache", true);

        Class testClass = CastorOXMapperImpl_JavaBeanStub01.class;

        // eXg{
        Mapping result = oxmapper.getCastorMapping(testClass);
        assertNull(result);
    }

    /**
     * testGetCastorMapping03() <br>
     * <br>
     * 
     * (ُn) <br>
     * ϓ_FA,G <br>
     * <br>
     * ͒lF() mappingClass:ClassCX^X<br>
     * () this.cache:true<br>
     * () this.mappingFileCache:not null<br>
     * () getUrl():ĂяoꂽƂmFB<br>
     * nꂽƂmFB<br>
     * () }bsO`t@C:sȃ}bsO`t@C<br>
     * <?xml version="1.0"?><br>
     * <!DOCTYPE mapping PUBLIC "-//EXOLAB/Castor Object Mapping DTD Version
     * 1.0//EN" "http://castor.exolab.org/mapping.dtd"><br>
     * <mapping><br>
     * <class name="jp.terasoluna.fw.oxm.mapper.castor.xxxDTO"><br>
     * <field name="param1" type="int" /><br>
     * <field name="param2" type="date" /><br>
     * <field name="param3" type="string" /><br>
     * </class><br>
     * </mapping<br>
     * 
     * <br>
     * ҒlF(ԕω) O:MappingException<br>
     * (ԕω) O:Ox:error<br>
     * Nested error: XML document structures must start and end within the same
     * entity.<br>
     * 
     * <br>
     * }bsO`t@Č`sȏꍇAOX[邱ƂmFB<br>
     * i}bsO`̌`ɖ肪ƁAp[XɔSAXExceptionbvMappingExceptionX[j
     * <br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testGetCastorMapping03() throws Exception {
        // O
        CastorOXMapperImpl_CastorOXMapperImplStub05 oxmapper = new CastorOXMapperImpl_CastorOXMapperImplStub05();
        UTUtil.setPrivateField(oxmapper, "cache", true);

        // getUrl\bȟĂяomFɎgpNX
        Class testClass = CastorOXMapperImpl_JavaBeanStub01.class;

        // `sȃ}bsO`t@C̃pX
        oxmapper.path = packagePath + "CastorOXMapperImpl_JavaBeanStub03"
                + mappingSuffix;

        // eXg{
        try {
            Unmarshaller unmarshaller = new Unmarshaller(new Object());
            Mapping mapping = oxmapper.getCastorMapping(testClass);
            unmarshaller.setMapping(mapping);
            fail();
        } catch (MappingException e) {
            // 
            assertSame(MappingException.class, e.fillInStackTrace().getClass());

            // OmF
            String message = "Nested error: XML document structures must start and end within the same entity.";
            assertTrue(e.getMessage().startsWith(message));
        }
    }

    /**
     * testGetCastorMapping04() <br>
     * <br>
     * 
     * (n) <br>
     * ϓ_FA,D <br>
     * <br>
     * ͒lF() mappingClass:ClassCX^X<br>
     * () this.cache:false<br>
     * () this.mappingFileCache:HashMap<br>
     * () getUrl():ĂяoꂽƂmFB<br>
     * nꂽƂmFB<br>
     * () }bsO`t@C:ȃ}bsO`t@C<br>
     * 
     * <br>
     * ҒlF(߂l) Mapping:ꂽMappingCX^X<br>
     * (ԕω) this.mappingFileCache:HashMap<br>
     * 
     * <br>
     * }bsO`t@Č`ȏꍇA}bsOCX^Xԋp邱ƂmFB this.cachefalsȅꍇ̃p^[B
     * <br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testGetCastorMapping04() throws Exception {
        // O
        CastorOXMapperImpl_CastorOXMapperImplStub05 oxmapper = new CastorOXMapperImpl_CastorOXMapperImplStub05();
        UTUtil.setPrivateField(oxmapper, "cache", false);

        // getUrl\bȟĂяomFɎgpNX
        Class testClass = CastorOXMapperImpl_JavaBeanStub01.class;

        // ȃ}bsO`t@C̃pX
        oxmapper.path = packagePath + "CastorOXMapperImpl_JavaBeanStub01"
                + mappingSuffix;

        // Mapݒ肷
        UTUtil.setPrivateField(oxmapper, "mappingFileCache", new HashMap());

        // eXg{
        Unmarshaller unmarshaller = new Unmarshaller(new Object());
        Mapping mapping = oxmapper.getCastorMapping(testClass);
        unmarshaller.setMapping(mapping);

        // 
        // getUrl\bhɈnĂ邱ƂmF
        assertEquals(testClass, oxmapper.mappingClass);

        // }bsO`[hĂ邱ƂmF
        ClassMapping[] classMappingList = mapping.getRoot().getClassMapping();
        assertEquals(1, classMappingList.length);
        assertEquals(testClass.getName(), classMappingList[0].getName());

    }

    /**
     * testGetCastorMapping05() <br>
     * <br>
     * 
     * (n) <br>
     * ϓ_FA,D <br>
     * <br>
     * ͒lF() mappingClass:ClassCX^X<br>
     * () this.cache:true<br>
     * () this.mappingFileCache:HashMap<br>
     * () getUrl():ĂяoꂽƂmFB<br>
     * nꂽƂmFB<br>
     * () }bsO`t@C:ȃ}bsO`t@C<br>
     * 
     * <br>
     * ҒlF(߂l) Mapping:ꂽMappingCX^X<br>
     * (ԕω) this.mappingFileCache:vf1HashMap<br>
     * HashMap<br> o<br>
     * ClassCXX=ꂽMappingCX^X<br> }<br>
     * 
     * <br>
     * }bsO`t@Č`ȏꍇA}bsOCX^Xԋp邱ƂmFB<br>
     * this.cachetrueŃ}bsOCX^XLbVɑ݂Ȃp^[B <br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testGetCastorMapping05() throws Exception {
        // O
        CastorOXMapperImpl_CastorOXMapperImplStub05 oxmapper = new CastorOXMapperImpl_CastorOXMapperImplStub05();
        UTUtil.setPrivateField(oxmapper, "cache", true);

        // getUrl\bȟĂяomFɎgpNX
        Class testClass = CastorOXMapperImpl_JavaBeanStub01.class;

        // ȃ}bsO`t@C̃pX
        oxmapper.path = packagePath + "CastorOXMapperImpl_JavaBeanStub01"
                + mappingSuffix;

        // LbVɂ
        UTUtil.setPrivateField(oxmapper, "mappingFileCache", new HashMap());

        // eXg{
        Unmarshaller unmarshaller = new Unmarshaller(new Object());
        Mapping mapping = oxmapper.getCastorMapping(testClass);
        unmarshaller.setMapping(mapping);

        // 
        // getUrl\bhɈnĂ邱ƂmF
        assertEquals(testClass, oxmapper.mappingClass);

        // }bsO`[hĂ邱ƂmF
        ClassMapping[] classMappingList = mapping.getRoot().getClassMapping();
        assertEquals(1, classMappingList.length);
        assertEquals(testClass.getName(), classMappingList[0].getName());

    }

    /**
     * testGetUrl01() <br>
     * <br>
     * 
     * (n) <br>
     * ϓ_FA <br>
     * <br>
     * ͒lF() mappingClass:not null<br>
     * iΉ}bsO`t@C݂NXj<br>
     * 
     * <br>
     * ҒlF(߂l) URL:URLCX^X<br>
     * 
     * <br>
     * ̃NXɑΉpXimappingClass̃pbP[WpX + mappingClass̃NX +
     * ".xml"jɃ}bsO`t@C݂ꍇA URLCX^Xԋp邱ƂmFB <br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testGetUrl01() throws Exception {
        // O
        CastorOXMapperImpl oxmapper = new CastorOXMapperImpl();

        // Ή}bsO`t@C݂NX
        Class mappingClass = CastorOXMapperImpl_JavaBeanStub01.class;

        // eXg{
        URL result = oxmapper.getUrl(mappingClass);

        // 
        // ꂽURLCX^Xpath܂ł邱ƂmF
        String mappingPath = packagePath + mappingClass.getSimpleName()
                + mappingSuffix;
        assertTrue(result.getPath().contains(mappingPath));
    }

    /**
     * testGetUrl02() <br>
     * <br>
     * 
     * (n) <br>
     * ϓ_FA <br>
     * <br>
     * ͒lF() mappingClass:not null<br>
     * iΉ}bsO`t@C݂NXj<br>
     * 
     * <br>
     * ҒlF(߂l) URL:null<br>
     * 
     * <br>
     * ̃NXɑΉpXimappingClass̃pbP[WpX + mappingClass̃NX +
     * ".xml"jɃ}bsO`t@C݂ȂꍇA nullԋp邱ƂmFB <br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testGetUrl02() throws Exception {
        // O
        CastorOXMapperImpl oxmapper = new CastorOXMapperImpl();

        // Ή}bsO`t@C݂ȂNX
        Class mappingClass = CastorOXMapperImpl.class;

        // eXg{
        URL result = oxmapper.getUrl(mappingClass);

        // 
        assertNull(result);
    }

    /**
     * testIsCache01() <br>
     * <br>
     * 
     * (n) <br>
     * ϓ_FA <br>
     * <br>
     * ͒lF() this.cache:true<br>
     * 
     * <br>
     * ҒlF(߂l) boolean:true<br>
     * 
     * <br>
     * ԋp邱ƂmFB <br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testIsCache01() throws Exception {
        // O
        CastorOXMapperImpl oxmapper = new CastorOXMapperImpl();
        UTUtil.setPrivateField(oxmapper, "cache", true);

        // eXg{
        boolean result = oxmapper.isCache();

        // 
        assertTrue(result);
    }

    /**
     * testIsIndenting01() <br>
     * <br>
     * 
     * (n) <br>
     * ϓ_FA <br>
     * <br>
     * ͒lF() this.indenting:true<br>
     * 
     * <br>
     * ҒlF(߂l) boolean:true<br>
     * 
     * <br>
     * ԋp邱ƂmFB <br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testIsIndenting01() throws Exception {
        // O
        CastorOXMapperImpl oxmapper = new CastorOXMapperImpl();
        UTUtil.setPrivateField(oxmapper, "indenting", true);

        // eXg{
        boolean result = oxmapper.isIndenting();

        // 
        assertTrue(result);
    }
    
    /**
     * testIsPreserveWhitespaceAtMarshal01() <br>
     * <br>
     * 
     * (n) <br>
     * ϓ_FA <br>
     * <br>
     * ͒lF() this.preserveWhitespaceAtMarshal:true<br>
     * 
     * <br>
     * ҒlF(߂l) boolean:true<br>
     * 
     * <br>
     * ԋp邱ƂmFB <br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testIsPreserveWhitespaceAtMarshal01() throws Exception {
        // O
        CastorOXMapperImpl oxmapper = new CastorOXMapperImpl();
        UTUtil.setPrivateField(oxmapper, "preserveWhitespaceAtMarshal", true);

        // eXg{
        boolean result = oxmapper.isPreserveWhitespaceAtMarshal();

        // 
        assertTrue(result);
    }
    
    /**
     * testIsPreserveWhitespaceAtUnmarshal01() <br>
     * <br>
     * 
     * (n) <br>
     * ϓ_FA <br>
     * <br>
     * ͒lF() this.preserveWhitespaceAtUnmarshal:true<br>
     * 
     * <br>
     * ҒlF(߂l) boolean:true<br>
     * 
     * <br>
     * ԋp邱ƂmFB <br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testIsPreserveWhitespaceAtUnmarshal01() throws Exception {
        // O
        CastorOXMapperImpl oxmapper = new CastorOXMapperImpl();
        UTUtil.setPrivateField(oxmapper, "preserveWhitespaceAtUnmarshal", true);

        // eXg{
        boolean result = oxmapper.isPreserveWhitespaceAtUnmarshal();

        // 
        assertTrue(result);
    }
    
    /**
     * testIsSuppressXSIType01() <br>
     * <br>
     * 
     * (n) <br>
     * ϓ_FA <br>
     * <br>
     * ͒lF() this.suppressXSIType:true<br>
     * 
     * <br>
     * ҒlF(߂l) boolean:true<br>
     * 
     * <br>
     * ԋp邱ƂmFB <br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testIsSuppressXSIType01() throws Exception {
        // O
        CastorOXMapperImpl oxmapper = new CastorOXMapperImpl();
        UTUtil.setPrivateField(oxmapper, "suppressXSIType", true);

        // eXg{
        boolean result = oxmapper.isSuppressXSIType();

        // 
        assertTrue(result);
    }
    
    /**
     * testSetCache01() <br>
     * <br>
     * 
     * (n) <br>
     * ϓ_FA <br>
     * <br>
     * ͒lF() cache:false<br>
     * () this.cache:true<br>
     * 
     * <br>
     * ҒlF(ԕω) this.cache:false<br>
     * 
     * <br>
     * ̒lɐݒ肳邱ƂmFB <br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testSetCache01() throws Exception {
        // O
        CastorOXMapperImpl oxmapper = new CastorOXMapperImpl();
        UTUtil.setPrivateField(oxmapper, "cache", true);

        // eXg{
        oxmapper.setCache(false);

        // 
        assertFalse(((Boolean) UTUtil.getPrivateField(oxmapper, "cache"))
                .booleanValue());
    }

    /**
     * testGetCharset01() <br>
     * <br>
     * 
     * (n) <br>
     * ϓ_FA <br>
     * <br>
     * ͒lF() this.charset:"EUC"<br>
     * 
     * <br>
     * ҒlF(߂l) charset:"EUC"<br>
     * 
     * <br>
     * ̒lԋp邱ƂmFB <br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testGetCharset01() throws Exception {
        // O
        CastorOXMapperImpl oxmapper = new CastorOXMapperImpl();
        UTUtil.setPrivateField(oxmapper, "charset", "EUC");

        // eXg{
        String result = oxmapper.getCharset();

        // 
        assertEquals("EUC", result);

    }

    /**
     * testSetCharset01() <br>
     * <br>
     * 
     * (n) <br>
     * ϓ_FA <br>
     * <br>
     * ͒lF() charset:"EUC"<br>
     * () this.charset:"UTF-8"<br>
     * 
     * <br>
     * ҒlF(ԕω) this.charset:"EUC"<br>
     * 
     * <br>
     * ̒lɐݒ肳邱ƂmFB <br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testSetCharset01() throws Exception {
        // O
        CastorOXMapperImpl oxmapper = new CastorOXMapperImpl();
        UTUtil.setPrivateField(oxmapper, "charset", "UTF-8");

        // eXg{
        oxmapper.setCharset("EUC");

        // 
        assertEquals("EUC", UTUtil.getPrivateField(oxmapper, "charset"));
    }

    /**
     * testSetSuppressXSIType01() <br>
     * <br>
     * 
     * (n) <br>
     * ϓ_FA <br>
     * <br>
     * ͒lF() suppressXSIType:true<br>
     * () this.suppressXSIType:false<br>
     * 
     * <br>
     * ҒlF(ԕω) this.suppressXSIType:true<br>
     * 
     * <br>
     * ̒lɐݒ肳邱ƂmFB <br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testSetSuppressXSIType01() throws Exception {
        // O
        CastorOXMapperImpl oxmapper = new CastorOXMapperImpl();
        UTUtil.setPrivateField(oxmapper, "suppressXSIType", false);

        // eXg{
        oxmapper.setSuppressXSIType(true);

        // 
        assertEquals(true, UTUtil.getPrivateField(oxmapper, "suppressXSIType"));
    }

    /**
     * testSetPreserveWhitespaceAtUnmarshal01() <br>
     * <br>
     * 
     * (n) <br>
     * ϓ_FA <br>
     * <br>
     * ͒lF() whitespacePreserve:true<br>
     * () this.whitespacePreserve:false<br>
     * 
     * <br>
     * ҒlF(ԕω) this.whitespacePreserve:true<br>
     * 
     * <br>
     * ̒lɐݒ肳邱ƂmFB <br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testSetPreserveWhitespaceAtUnmarshal01() throws Exception {
        // O
        CastorOXMapperImpl oxmapper = new CastorOXMapperImpl();
        UTUtil
                .setPrivateField(oxmapper, "preserveWhitespaceAtUnmarshal",
                        false);

        // eXg{
        oxmapper.setPreserveWhitespaceAtUnmarshal(true);

        // 
        assertEquals(true, UTUtil.getPrivateField(oxmapper,
                "preserveWhitespaceAtUnmarshal"));
    }

    /**
     * testSetPreserveWhitespaceAtMarshal01() <br>
     * <br>
     * 
     * (n) <br>
     * ϓ_FA <br>
     * <br>
     * ͒lF() whitespacePreserve:true<br>
     * () this.whitespacePreserve:false<br>
     * 
     * <br>
     * ҒlF(ԕω) this.whitespacePreserve:true<br>
     * 
     * <br>
     * ̒lɐݒ肳邱ƂmFB <br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testSetPreserveWhitespaceAtMarshal01() throws Exception {
        // O
        CastorOXMapperImpl oxmapper = new CastorOXMapperImpl();
        UTUtil.setPrivateField(oxmapper, "preserveWhitespaceAtMarshal", false);

        // eXg{
        oxmapper.setPreserveWhitespaceAtMarshal(true);

        // 
        assertEquals(true, UTUtil.getPrivateField(oxmapper,
                "preserveWhitespaceAtMarshal"));
    }
    
    /**
     * testSetIndenting01() <br>
     * <br>
     * 
     * (n) <br>
     * ϓ_FA <br>
     * <br>
     * ͒lF() indenting:false<br>
     * () this.indenting:true<br>
     * 
     * <br>
     * ҒlF(ԕω) this.indenting:false<br>
     * 
     * <br>
     * ̒lɐݒ肳邱ƂmFB <br>
     * 
     * @throws Exception
     *             ̃\bhŔO
     */
    public void testSetIndenting01() throws Exception {
        // O
        CastorOXMapperImpl oxmapper = new CastorOXMapperImpl();
        UTUtil.setPrivateField(oxmapper, "indenting", true);

        // eXg{
        oxmapper.setIndenting(false);

        // 
        assertEquals(false, UTUtil.getPrivateField(oxmapper, "indenting"));
    }

}
