/* *************************************************** print-callbacks.c *** *
 * ˴ؤؿ 
 *
 * Copyright (C) 2001-2003 Yasuyuki SUGAYA <sugaya@suri.it.okayama-u.ac.jp>
 * Okayama University
 *                                     Time-stamp: <03/06/28 12:21:54 sugaya>
 *
 * The original source code came from gthumb-2.1.2
 * ************************************************************************* */
#include <config.h>
#include <string.h>
#include <libgnome/libgnome.h>
#include <libgnomeprint/gnome-print.h>
#include <libgnomeprint/gnome-print-job.h>
#include <libgnomeprintui/gnome-print-preview.h>
#include <libgnomeprintui/gnome-print-job-preview.h>
#include <libgnomeprintui/gnome-print-dialog.h>
#include <libgnomeprintui/gnome-print-paper-selector.h>
#include <libgnomecanvas/libgnomecanvas.h>
#include <libart_lgpl/libart.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <pango/pango-layout.h>
#include "teoeyes.h"
#include "print-callbacks.h"

#define PARAGRAPH_SEPARATOR	0x2029	
#define CANVAS_ZOOM		0.25

#define g_signal_handlers_block_by_data(instance, data) \
    g_signal_handlers_block_matched ((instance), G_SIGNAL_MATCH_DATA, \
                                     0, 0, NULL, NULL, (data))
#define g_signal_handlers_unblock_by_data(instance, data) \
    g_signal_handlers_unblock_matched ((instance), G_SIGNAL_MATCH_DATA, \
                                       0, 0, NULL, NULL, (data))

typedef struct {
  int                  ref_count;

  GtkWidget           *canvas;
  GnomeCanvasItem     *ci_image;
  GnomeCanvasItem     *ci_comment;

  /* Paper Information. */

  double               paper_width;
  double               paper_height;
  double               paper_lmargin;
  double               paper_rmargin;
  double               paper_tmargin;
  double               paper_bmargin;

  GnomePrintConfig    *config;
  GnomePrintJob       *gpj;

  gboolean             print_comment;
  gboolean             portrait;

  /* Image Information. */

  char               	*image_path;
  char		      	*filename;
	
  GdkPixbuf           *pixbuf;

  double               image_w, image_h;
	
  double               scale_x, scale_y;
  double               trans_x, trans_y;
  double               zoom;

  double               min_x, min_y;
  double               max_x, max_y;

  gboolean             use_colors;
} PrintInfo;


typedef struct {
  GtkWidget     *dialog;
  GtkWidget     *btn_close;
  GtkWidget     *btn_center;
  GtkWidget     *btn_print;
  GtkWidget     *hscale;

  GtkAdjustment *adj;

  PrintInfo     *pi;
} DialogData;

/* ************************************************************************* */
static void 
print_info_unref (PrintInfo *pi)
{
  g_return_if_fail (pi != NULL);
  g_return_if_fail (pi->ref_count > 0);
	
  pi->ref_count--;

  if (pi->ref_count == 0) {
    if (pi->gpj != NULL)
      g_object_unref (pi->gpj);
    gnome_print_config_unref (pi->config);
    g_free (pi->image_path);
    g_object_unref (pi->pixbuf);
    g_free (pi);
  }
}

/* ************************************************************************* */
static void
print_image (GnomePrintContext *pc,
	     PrintInfo         *pi, 
	     gboolean           border) {
  double  	w, h;
  double  	lmargin, rmargin, tmargin, bmargin;
  int     	pw, ph, rs;
  guchar	*p;

  w = pi->paper_width;
  h = pi->paper_height;
  lmargin = pi->paper_lmargin;
  rmargin = pi->paper_rmargin;
  bmargin = pi->paper_bmargin;
  tmargin = pi->paper_tmargin;

  gnome_print_beginpage (pc, NULL);

  if (border) {
    gnome_print_gsave (pc);
    gnome_print_moveto (pc, lmargin, bmargin);
    gnome_print_lineto (pc, lmargin, h - tmargin);
    gnome_print_lineto (pc, w - rmargin, h - tmargin);
    gnome_print_lineto (pc, w - rmargin, bmargin);
    gnome_print_lineto (pc, lmargin, bmargin);
    gnome_print_stroke (pc);
    gnome_print_grestore (pc);
  }

  p  = gdk_pixbuf_get_pixels (pi->pixbuf);
  pw = gdk_pixbuf_get_width (pi->pixbuf);
  ph = gdk_pixbuf_get_height (pi->pixbuf);
  rs = gdk_pixbuf_get_rowstride (pi->pixbuf);

  gnome_print_gsave (pc);
  gnome_print_scale (pc, pi->scale_x, pi->scale_y);
  gnome_print_translate (pc, pi->trans_x, pi->trans_y);
  if (pi->use_colors) {
    if (gdk_pixbuf_get_has_alpha (pi->pixbuf)) {
      gnome_print_rgbaimage (pc, p, pw, ph, rs);
    } else {
      gnome_print_rgbimage  (pc, p, pw, ph, rs);
    }
  } else {
    gnome_print_grayimage (pc, p, pw, ph, rs);
  }
  gnome_print_grestore (pc);
  gnome_print_showpage (pc);
}

/* ************************************************************************* */
static void
add_simulated_page (GnomeCanvas *canvas) {
  GnomeCanvasItem 	*i;
  double		x1, y1, x2, y2, width, height;

  gnome_canvas_get_scroll_region (canvas, &x1, &y1, &x2, &y2);
  width  = x2 - x1;
  height = y2 - y1;

  i = gnome_canvas_item_new (GNOME_CANVAS_GROUP (gnome_canvas_root (canvas)),
			     gnome_canvas_rect_get_type (),
			     "x1",   	 	0.0,
			     "y1",   	 	0.0,
			     "x2",   	 	width,
			     "y2",   	 	height,
			     "fill_color",    	"white",
			     "outline_color", 	"black",
			     "width_pixels",  	1,
			     NULL);
  gnome_canvas_item_lower_to_bottom (i);
  i = gnome_canvas_item_new (GNOME_CANVAS_GROUP (gnome_canvas_root (canvas)),
			     gnome_canvas_rect_get_type (),
			     "x1",   	 	3.0,
			     "y1",   	 	3.0,
			     "x2",   	 	width + 3.0,
			     "y2",   	 	height + 3.0,
			     "fill_color",    	"black",
			     NULL);
  gnome_canvas_item_lower_to_bottom (i);
}

/* ************************************************************************* */
static void
check_bounds (PrintInfo *pi, double *x1, double *y1) {
  *x1 = MAX (*x1, pi->min_x);
  *x1 = MIN (*x1, pi->max_x - pi->image_w * pi->zoom);
  *y1 = MAX (*y1, pi->min_y);
  *y1 = MIN (*y1, pi->max_y - pi->image_h * pi->zoom);
}

/* ************************************************************************* */
static gint
item_event (GnomeCanvasItem *item, 
	    GdkEvent        *event, 
	    PrintInfo       *pi) {
  static double  start_x, start_y;
  static double  img_start_x, img_start_y;
  static int     dragging;
  double         x, y;
  GdkCursor     *fleur;

  x = event->button.x;
  y = event->button.y;

  switch (event->type) {
  case GDK_BUTTON_PRESS:
    switch (event->button.button) {
    case 1:
      start_x = x;
      start_y = y;
      g_object_get (G_OBJECT (item),
		    "x", &img_start_x, "y", &img_start_y, NULL);

      fleur = gdk_cursor_new (GDK_FLEUR);
      gnome_canvas_item_grab (item,
			      (GDK_POINTER_MOTION_MASK 
			       | GDK_BUTTON_RELEASE_MASK),
			      fleur,
			      event->button.time);
      gdk_cursor_unref (fleur);
      dragging = TRUE;
      break;

    default:
      break;
    }
    break;

  case GDK_MOTION_NOTIFY:
    if (dragging && (event->motion.state & GDK_BUTTON1_MASK)) {
      double x1, y1;

      x1 = img_start_x + (x - start_x);
      y1 = img_start_y + (y - start_y);

      check_bounds (pi, &x1, &y1);

      gnome_canvas_item_set (item, "x", x1, "y", y1, NULL);
    }
    break;

  case GDK_BUTTON_RELEASE:
    gnome_canvas_item_ungrab (item, event->button.time);
    dragging = FALSE;
    break;

  default:
    break;
  }

  return FALSE;
}

/* ************************************************************************* */
static void
add_image_preview (PrintInfo	*pi, 
		   gboolean 	border) {
  double            w, h;
  double            lmargin, rmargin, tmargin, bmargin;
  double            iw, ih;
  double            max_w, max_h;
  GnomeCanvasGroup *group;
  double            factor;
  int               layout_width;

  group = GNOME_CANVAS_GROUP (gnome_canvas_root (GNOME_CANVAS (pi->canvas)));

  w = pi->paper_width;
  h = pi->paper_height;
  lmargin = pi->paper_lmargin;
  rmargin = pi->paper_rmargin;
  bmargin = pi->paper_bmargin;
  tmargin = pi->paper_tmargin;
  iw = (double) gdk_pixbuf_get_width (pi->pixbuf); 
  ih = (double) gdk_pixbuf_get_height (pi->pixbuf);

  /* Insert comment */
  layout_width = (int) ((pi->paper_width - lmargin - rmargin) * CANVAS_ZOOM);

  /* Scale if image does not fit on the window. */
  max_w = w - lmargin - rmargin;
  max_h = h - bmargin - tmargin;

  factor = MIN (max_w / iw, max_h / ih);
  ih *= factor;
  iw *= factor;

  pi->image_w = iw;
  pi->image_h = ih;

  /* Center & check bounds. */

  pi->zoom = 1.0;

  pi->min_x = lmargin;
  pi->min_y = tmargin;
  pi->max_x = lmargin + max_w;
  pi->max_y = tmargin + max_h;

  pi->trans_x = MAX ((w - iw) / 2, lmargin);
  pi->trans_y = MAX ((h - ih) / 2, bmargin);
  check_bounds (pi, &pi->trans_x, &pi->trans_y);

  if (border) {
    gnome_canvas_item_new (gnome_canvas_root ( GNOME_CANVAS (pi->canvas)),
			   gnome_canvas_rect_get_type (),
			   "x1",   	 lmargin,
			   "y1",   	 tmargin,
			   "x2",   	 lmargin + max_w,
			   "y2",   	 tmargin + max_h,
			   "outline_color", "gray",
			   "width_pixels",  1,
			   NULL);
  }
  iw = MAX (1.0, iw);
  ih = MAX (1.0, ih);

  pi->ci_image = gnome_canvas_item_new (group,
					gnome_canvas_pixbuf_get_type (),
					"pixbuf",     pi->pixbuf,
					"x",          pi->trans_x,
					"y",          pi->trans_y,
					"width",      iw,
					"width_set",  TRUE,
					"height",     ih,
					"height_set", TRUE,
					"anchor",     GTK_ANCHOR_NW,
					NULL);

  if (pi->ci_image == NULL) g_error ("Cannot create image preview\n");

  g_signal_connect (G_OBJECT (pi->ci_image), 
		    "event",
		    G_CALLBACK (item_event),
		    pi);
}

/* ************************************************************************* */
static void
destroy_cb (GtkWidget	*widget, 
	    DialogData 	*data) {
  print_info_unref (data->pi);
  g_free (data);
}

/* ************************************************************************* */
static void
center_cb (GtkWidget	*widget, 
	   DialogData 	*data) {
  PrintInfo 	*pi = data->pi;
  gdouble	x1, y1, w, h;

  w = pi->max_x - pi->min_x;
  h = pi->max_y - pi->min_y;
  x1 = pi->min_x + (w - pi->image_w * pi->zoom) / 2;
  y1 = pi->min_y + (h - pi->image_h * pi->zoom) / 2;

  gnome_canvas_item_set (pi->ci_image,
			 "x", x1,
			 "y", y1,
			 NULL);
}

/* ************************************************************************* */
static void 
zoom_changed (GtkAdjustment	*adj,
	      DialogData 	*data) {
  PrintInfo	*pi = data->pi;
  gdouble 	w, h;
  gdouble 	x1, y1;

  pi->zoom = adj->value / 100.0;
  w = pi->image_w * pi->zoom;
  h = pi->image_h * pi->zoom;

  gnome_canvas_item_set (pi->ci_image,
			 "width",      w,
			 "width_set",  TRUE,
			 "height",     h,
			 "height_set", TRUE,
			 NULL);

  /* Keep image within page borders. */

  g_object_get (G_OBJECT (pi->ci_image),
		"x", &x1,
		"y", &y1,
		NULL);
  x1 = MAX (x1, pi->min_x);
  x1 = MIN (x1, pi->max_x - pi->image_w * pi->zoom);
  y1 = MAX (y1, pi->min_y);
  y1 = MIN (y1, pi->max_y - pi->image_h * pi->zoom);

  gnome_canvas_item_set (pi->ci_image,
			 "x", x1,
			 "y", y1,
			 NULL);
}

/* ************************************************************************* */
static void 
print_cb (GtkWidget	*widget,
	  DialogData 	*data) {
  PrintInfo		*pi = data->pi;
  GnomePrintContext	*gp_ctx;
  double		x, y;
  char			*title;
  GtkWidget		*dialog;
  int			response;

  g_object_get (G_OBJECT (pi->ci_image),
		"x", &x,
		"y", &y,
		NULL);
  pi->scale_x = pi->image_w * pi->zoom;
  pi->scale_y = pi->image_h * pi->zoom;
  pi->trans_x = x / pi->scale_x;
  pi->trans_y = (pi->paper_height - pi->scale_y - y) / pi->scale_y;

  /* pi is used to print, so we must add a reference. */

  pi->ref_count++;
  gtk_widget_destroy (data->dialog); 

  /* Gnome Print Dialog */

  pi->gpj = gnome_print_job_new (pi->config);

  if (pi->filename != NULL)
    title = g_strdup_printf (_("Print %s"), pi->filename);	
  else
    title = g_strdup (_("Print Image"));
  dialog = gnome_print_dialog_new (pi->gpj, title, 0);
  response = gtk_dialog_run (GTK_DIALOG (dialog));
  gtk_widget_destroy (dialog);

  gp_ctx = gnome_print_job_get_context (pi->gpj);
  print_image (gp_ctx, pi, FALSE);
  gnome_print_job_close (pi->gpj);

  switch (response) {
  case GNOME_PRINT_DIALOG_RESPONSE_PRINT:
    gnome_print_job_print (pi->gpj);	
    break;

  case GNOME_PRINT_DIALOG_RESPONSE_PREVIEW:
    gtk_widget_show (gnome_print_job_preview_new (pi->gpj, title));
    break;

  default:
    break;
  }
 
  g_free (title);

  print_info_unref (pi);
}

/* ************************************************************************* */
static void
clear_canvas (GnomeCanvasGroup *group) {
  GList *list = group->item_list;

  while (list) {
    gtk_object_destroy (GTK_OBJECT (list->data));
    list = group->item_list;
  }
}

/* ************************************************************************* */
static void
page_update (DialogData *data) {
  PrintInfo *pi = data->pi;
  const      GnomePrintUnit *unit;
  const      GnomePrintUnit *ps_unit;
  guchar    *orientation;
  double     width, height, lmargin, tmargin, rmargin, bmargin;
  double     paper_width, paper_height, paper_lmargin, paper_tmargin;
  double     paper_rmargin, paper_bmargin;
  gboolean   portrait;

  ps_unit = gnome_print_unit_get_identity (GNOME_PRINT_UNIT_DIMENSIONLESS);

  if (gnome_print_config_get_length (pi->config, 
				     GNOME_PRINT_KEY_PAPER_WIDTH,
				     &width,
				     &unit))
    gnome_print_convert_distance (&width, unit, ps_unit);
  if (gnome_print_config_get_length (pi->config, 
				     GNOME_PRINT_KEY_PAPER_HEIGHT,
				     &height,
				     &unit))
    gnome_print_convert_distance (&height, unit, ps_unit);
  if (gnome_print_config_get_length (pi->config, 
				     GNOME_PRINT_KEY_PAGE_MARGIN_LEFT,
				     &lmargin,
				     &unit))
    gnome_print_convert_distance (&lmargin, unit, ps_unit);
  if (gnome_print_config_get_length (pi->config, 
				     GNOME_PRINT_KEY_PAGE_MARGIN_RIGHT,
				     &rmargin,
				     &unit))
    gnome_print_convert_distance (&rmargin, unit, ps_unit);
  if (gnome_print_config_get_length (pi->config, 
				     GNOME_PRINT_KEY_PAGE_MARGIN_TOP,
				     &tmargin,
				     &unit))
    gnome_print_convert_distance (&tmargin, unit, ps_unit);
  if (gnome_print_config_get_length (pi->config, 
				     GNOME_PRINT_KEY_PAGE_MARGIN_BOTTOM,
				     &bmargin,
				     &unit))
    gnome_print_convert_distance (&bmargin, unit, ps_unit);

  orientation = gnome_print_config_get (pi->config, GNOME_PRINT_KEY_PAGE_ORIENTATION);
  portrait = ((strcmp (orientation, "R0") == 0) 
	      || (strcmp (orientation, "R180") == 0));
  g_free (orientation);

  if (portrait) {
    paper_width   = width;
    paper_height  = height;
    paper_lmargin = lmargin;
    paper_tmargin = tmargin;
    paper_rmargin = rmargin;
    paper_bmargin = bmargin;
  } else {
    paper_width   = height;
    paper_height  = width;
    paper_lmargin = tmargin;
    paper_tmargin = rmargin;
    paper_rmargin = bmargin;
    paper_bmargin = lmargin;
  }

  pi->portrait = portrait;
  pi->paper_width   = paper_width;
  pi->paper_height  = paper_height;
  pi->paper_lmargin = paper_lmargin;
  pi->paper_tmargin = paper_tmargin;
  pi->paper_rmargin = paper_rmargin;
  pi->paper_bmargin = paper_bmargin;

  clear_canvas (GNOME_CANVAS_GROUP (GNOME_CANVAS (pi->canvas)->root));
  gnome_canvas_set_scroll_region (GNOME_CANVAS (pi->canvas), 
				  0, 0,
				  pi->paper_width, pi->paper_height);

  add_simulated_page (GNOME_CANVAS (pi->canvas));
  add_image_preview (pi, TRUE);

  /* Reset zoom to 100%. */

  g_signal_handlers_block_by_data (G_OBJECT (data->adj), data);
  gtk_adjustment_set_value (data->adj, 100.0);
  g_signal_handlers_unblock_by_data (G_OBJECT (data->adj), data);
}

/* ************************************************************************* */
static void
portrait_toggled_cb (GtkToggleButton  *widget,
		     DialogData       *data) {
  if (! gtk_toggle_button_get_active (widget))
    return;
  gnome_print_config_set (data->pi->config, GNOME_PRINT_KEY_PAGE_ORIENTATION, "R0");
  page_update (data);
}

/* ************************************************************************* */
static void
landscape_toggled_cb (GtkToggleButton  *widget,
		      DialogData       *data) {
  if (! gtk_toggle_button_get_active (widget))
    return;
  gnome_print_config_set (data->pi->config, GNOME_PRINT_KEY_PAGE_ORIENTATION, "R90");
  page_update (data);
}

/* ************************************************************************* */
static void
size_letter_toggled_cb (GtkToggleButton  *widget,
			DialogData       *data) {
  if (! gtk_toggle_button_get_active (widget))
    return;
  gnome_print_config_set (data->pi->config, GNOME_PRINT_KEY_PAPER_SIZE, "USLetter");
  page_update (data);
}

/* ************************************************************************* */
static void
size_legal_toggled_cb (GtkToggleButton  *widget,
		       DialogData       *data) {
  if (! gtk_toggle_button_get_active (widget))
    return;
  gnome_print_config_set (data->pi->config, GNOME_PRINT_KEY_PAPER_SIZE, "USLegal");
  page_update (data);
}

/* ************************************************************************* */
static void
size_executive_toggled_cb (GtkToggleButton  *widget,
			   DialogData       *data) {
  if (! gtk_toggle_button_get_active (widget))
    return;
  gnome_print_config_set (data->pi->config, GNOME_PRINT_KEY_PAPER_SIZE, "Executive");
  page_update (data);
}

/* ************************************************************************* */
static void
size_a3_toggled_cb (GtkToggleButton  *widget,
		    DialogData       *data) {
  if (! gtk_toggle_button_get_active (widget))
    return;
  gnome_print_config_set (data->pi->config, GNOME_PRINT_KEY_PAPER_SIZE, "A3");
  page_update (data);
}

/* ************************************************************************* */
static void
size_a4_toggled_cb (GtkToggleButton  *widget,
		    DialogData       *data) {
  if (! gtk_toggle_button_get_active (widget))
    return;
  gnome_print_config_set (data->pi->config, GNOME_PRINT_KEY_PAPER_SIZE, "A4");
  page_update (data);
}

/* ************************************************************************* */
static void
size_a5_toggled_cb (GtkToggleButton  *widget,
		    DialogData       *data) {
  if (! gtk_toggle_button_get_active (widget))
    return;
  gnome_print_config_set (data->pi->config, GNOME_PRINT_KEY_PAPER_SIZE, "A5");
  page_update (data);
}

/* ************************************************************************* */
static GtkWidget*
page_setup_dialog_new (DialogData	*data) {
  GtkWidget *page_setup_dialog;
  GtkWidget *dialog_vbox;
  GtkWidget *hbox1;
  GtkWidget *hbox2;
  GtkWidget *hbox3;
  GtkWidget *hbox4;  
  GtkWidget *vbox1;
  GtkWidget *vbox2;
  GtkWidget *vbox3;  
  GtkWidget *notebook;
  GtkWidget *frame;
  GtkWidget *image;
  GtkWidget *label;
  GtkWidget *hscale;
  GtkWidget *print_size_a3;
  GtkWidget *print_size_a4;
  GtkWidget *print_size_a5;
  GtkWidget *print_size_letter;
  GtkWidget *print_size_legal;
  GtkWidget *print_size_executive;
  GSList *print_size_a3_group = NULL;
  GtkWidget *print_orient_portrait;
  GtkWidget *print_orient_landscape;
  GSList *print_orient_portrait_group = NULL;
  GtkWidget *scrolledwindow;
  GtkWidget *canvas;
  GtkWidget *dialog_action_area;
  GtkWidget *btn_center;
  GtkWidget *btn_close;
  GtkWidget *btn_print;

  page_setup_dialog = gtk_dialog_new ();
  gtk_container_set_border_width (GTK_CONTAINER (page_setup_dialog), 6);
  gtk_window_set_title (GTK_WINDOW (page_setup_dialog), _("Print"));
  gtk_window_set_resizable (GTK_WINDOW (page_setup_dialog), FALSE);
  gtk_dialog_set_has_separator (GTK_DIALOG (page_setup_dialog), FALSE);
  g_signal_connect (G_OBJECT (page_setup_dialog),
		    "destroy", G_CALLBACK (destroy_cb), data);
  
  dialog_vbox = GTK_DIALOG (page_setup_dialog)->vbox;
  gtk_widget_show (dialog_vbox);

  hbox1 = gtk_hbox_new (FALSE, 12);
  gtk_widget_show (hbox1);
  gtk_box_pack_start (GTK_BOX (dialog_vbox), hbox1, TRUE, TRUE, 0);
  gtk_container_set_border_width (GTK_CONTAINER (hbox1), 6);

  notebook = gtk_notebook_new ();
  gtk_widget_show (notebook);
  gtk_box_pack_start (GTK_BOX (hbox1), notebook, TRUE, TRUE, 0);
  {
    vbox1 = gtk_vbox_new (FALSE, 6);
    gtk_widget_show (vbox1);
    gtk_container_add (GTK_CONTAINER (notebook), vbox1);
    gtk_container_set_border_width (GTK_CONTAINER (vbox1), 12);
    {
      frame = gtk_frame_new (NULL);
      gtk_widget_show (frame);
      gtk_box_pack_end (GTK_BOX (vbox1), frame, FALSE, TRUE, 0);
      gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_OUT);
      {
	hbox2 = gtk_hbox_new (FALSE, 5);
	gtk_widget_show (hbox2);
	gtk_container_add (GTK_CONTAINER (frame), hbox2);
	gtk_container_set_border_width (GTK_CONTAINER (hbox2), 5);
	{
	  image = gtk_image_new_from_stock ("gtk-dialog-info",
					    GTK_ICON_SIZE_BUTTON);
	  gtk_widget_show (image);
	  gtk_box_pack_start (GTK_BOX (hbox2), image, FALSE, FALSE, 0);

	  label = gtk_label_new
	    (_("<i>Drag the image to move\nit in the page.</i>"));
	  gtk_widget_show (label);
	  gtk_box_pack_start (GTK_BOX (hbox2), label, TRUE, TRUE, 0);
	  gtk_widget_set_size_request (label, 150, -1);
	  gtk_label_set_use_markup (GTK_LABEL (label), TRUE);
	  gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
	  gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
	  gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);
	}
      }
      label = gtk_label_new ("");
      gtk_widget_show (label);
      gtk_box_pack_end (GTK_BOX (vbox1), label, TRUE, FALSE, 0);
      gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);

      hbox2 = gtk_hbox_new (FALSE, 12);
      gtk_widget_show (hbox2);
      gtk_box_pack_end (GTK_BOX (vbox1), hbox2, FALSE, TRUE, 0);
      {
	vbox2 = gtk_vbox_new (FALSE, 12);
	gtk_widget_show (vbox2);
	gtk_box_pack_start (GTK_BOX (hbox2), vbox2, TRUE, TRUE, 0);
	{
	  hbox3 = gtk_hbox_new (FALSE, 6);
	  gtk_widget_show (hbox3);
	  gtk_box_pack_start (GTK_BOX (vbox2), hbox3, FALSE, TRUE, 0);
	  {
	    label = gtk_label_new_with_mnemonic (_("_Scale:"));
	    gtk_widget_show (label);
	    gtk_box_pack_start (GTK_BOX (hbox3), label, FALSE, FALSE, 0);
	    gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);

	    hbox4 = gtk_hbox_new (FALSE, 3);
	    gtk_widget_show (hbox4);
	    gtk_box_pack_start (GTK_BOX (hbox3), hbox4, TRUE, TRUE, 0);
	    {
	      hscale =
		gtk_hscale_new (GTK_ADJUSTMENT
				(gtk_adjustment_new (100, 5, 101, 1, 10, 1)));
	      gtk_widget_show (hscale);
	      gtk_box_pack_start (GTK_BOX (hbox4), hscale, TRUE, TRUE, 0);
	      gtk_scale_set_value_pos (GTK_SCALE (hscale), GTK_POS_RIGHT);
	      gtk_scale_set_digits (GTK_SCALE (hscale), 0);
	      
	      label = gtk_label_new (_("%"));
	      gtk_widget_show (label);
	      gtk_box_pack_start (GTK_BOX (hbox4), label, FALSE, FALSE, 0);
	    }
	  }
	  hbox3 = gtk_hbox_new (FALSE, 0);
	  gtk_widget_show (hbox3);
	  gtk_box_pack_start (GTK_BOX (vbox2), hbox3, FALSE, TRUE, 6);
	  {
	    btn_center = gtk_button_new_with_mnemonic (_("Center _on Page"));
	    gtk_widget_show (btn_center);
	    gtk_box_pack_start (GTK_BOX (hbox3), btn_center, TRUE, FALSE, 0);
	    g_signal_connect (G_OBJECT (btn_center),
			      "clicked", G_CALLBACK (center_cb), data);

	  }
	}
	image = gtk_image_new_from_stock ("gtk-print-preview",
					  GTK_ICON_SIZE_DND);
	gtk_widget_show (image);
	gtk_box_pack_start (GTK_BOX (hbox2), image, TRUE, TRUE, 0);
      }
    }
    label = gtk_label_new (_("Layout"));
    gtk_widget_show (label);
    gtk_notebook_set_tab_label (GTK_NOTEBOOK (notebook),
				gtk_notebook_get_nth_page
				(GTK_NOTEBOOK (notebook), 0), label);
    gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
  }
  {
    vbox1 = gtk_vbox_new (FALSE, 12);
    gtk_widget_show (vbox1);
    gtk_container_add (GTK_CONTAINER (notebook), vbox1);
    gtk_container_set_border_width (GTK_CONTAINER (vbox1), 12);
    {
      vbox2 = gtk_vbox_new (FALSE, 6);
      gtk_widget_show (vbox2);
      gtk_box_pack_start (GTK_BOX (vbox1), vbox2, FALSE, TRUE, 0);
      {
	label = gtk_label_new (_("<b>Size</b>"));
	gtk_widget_show (label);
	gtk_box_pack_start (GTK_BOX (vbox2), label, FALSE, FALSE, 0);
	gtk_label_set_use_markup (GTK_LABEL (label), TRUE);
	gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
	gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);

	hbox2 = gtk_hbox_new (FALSE, 0);
	gtk_widget_show (hbox2);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox2, FALSE, TRUE, 0);
	{
	  label = gtk_label_new (_("    "));
	  gtk_widget_show (label);
	  gtk_box_pack_start (GTK_BOX (hbox2), label, FALSE, FALSE, 0);
	  gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);

	  hbox3 = gtk_hbox_new (FALSE, 12);
	  gtk_widget_show (hbox3);
	  gtk_box_pack_start (GTK_BOX (hbox2), hbox3, TRUE, TRUE, 0);
	  {
	    vbox3 = gtk_vbox_new (FALSE, 6);
	    gtk_widget_show (vbox3);
	    gtk_box_pack_start (GTK_BOX (hbox3), vbox3, TRUE, TRUE, 0);
	    {
	      print_size_a3 =
		gtk_radio_button_new_with_mnemonic (NULL,
						    _("A_3 (297 x 420 mm)"));
	      gtk_widget_show (print_size_a3);
	      gtk_box_pack_start (GTK_BOX (vbox3),
				  print_size_a3, FALSE, FALSE, 0);
	      gtk_radio_button_set_group (GTK_RADIO_BUTTON (print_size_a3),
					  print_size_a3_group);
	      print_size_a3_group =
		gtk_radio_button_get_group (GTK_RADIO_BUTTON (print_size_a3));
	      g_signal_connect (G_OBJECT (print_size_a3), "toggled", 
				G_CALLBACK (size_a3_toggled_cb), data);


	      print_size_a4 =
		gtk_radio_button_new_with_mnemonic (NULL,
						    _("A_4 (210 x 297 mm)"));
	      gtk_widget_show (print_size_a4);
	      gtk_box_pack_start (GTK_BOX (vbox3),
				  print_size_a4, FALSE, FALSE, 0);
	      gtk_radio_button_set_group (GTK_RADIO_BUTTON (print_size_a4),
					  print_size_a3_group);
	      print_size_a3_group = 
		gtk_radio_button_get_group (GTK_RADIO_BUTTON (print_size_a4));

	      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (print_size_a4),
					    TRUE);
	      g_signal_connect (G_OBJECT (print_size_a4), "toggled",
				G_CALLBACK (size_a4_toggled_cb), data);


	      print_size_a5 =
		gtk_radio_button_new_with_mnemonic (NULL,
						    _("A_5 (148 x 210 mm)"));
	      gtk_widget_show (print_size_a5);
	      gtk_box_pack_start (GTK_BOX (vbox3),
				  print_size_a5, FALSE, FALSE, 0);
	      gtk_radio_button_set_group (GTK_RADIO_BUTTON (print_size_a5),
					  print_size_a3_group);
	      print_size_a3_group =
		gtk_radio_button_get_group (GTK_RADIO_BUTTON (print_size_a5));
	      g_signal_connect (G_OBJECT (print_size_a5), "toggled",
				G_CALLBACK (size_a5_toggled_cb), data);

	      
	      print_size_letter =
		gtk_radio_button_new_with_mnemonic(NULL,
						  _("_Letter (216 x 279 mm)"));
	      gtk_widget_show (print_size_letter);
	      gtk_box_pack_start (GTK_BOX (vbox3),
				  print_size_letter, FALSE, FALSE, 0);
	      gtk_radio_button_set_group (GTK_RADIO_BUTTON (print_size_letter),
					  print_size_a3_group);
	      print_size_a3_group =
		gtk_radio_button_get_group (GTK_RADIO_BUTTON
					    (print_size_letter));
	      g_signal_connect (G_OBJECT (print_size_letter), "toggled",
				G_CALLBACK (size_letter_toggled_cb), data);

	      
	      print_size_legal =
		gtk_radio_button_new_with_mnemonic (NULL,
						   _("L_egal (216 x 356 mm)"));
	      gtk_widget_show (print_size_legal);
	      gtk_box_pack_start (GTK_BOX (vbox3),
				  print_size_legal, FALSE, FALSE, 0);
	      gtk_radio_button_set_group (GTK_RADIO_BUTTON (print_size_legal),
					  print_size_a3_group);
	      print_size_a3_group =
		gtk_radio_button_get_group(GTK_RADIO_BUTTON(print_size_legal));
	      g_signal_connect (G_OBJECT (print_size_legal), "toggled",
				G_CALLBACK (size_legal_toggled_cb), data);
	      

	      print_size_executive =
		gtk_radio_button_new_with_mnemonic (NULL,
					       _("E_xecutive (190 x 254 mm)"));
	      gtk_widget_show (print_size_executive);
	      gtk_box_pack_start (GTK_BOX (vbox3),
				  print_size_executive, FALSE, FALSE, 0);
	      gtk_radio_button_set_group (GTK_RADIO_BUTTON
					  (print_size_executive),
					  print_size_a3_group);
	      print_size_a3_group =
		gtk_radio_button_get_group (GTK_RADIO_BUTTON
					    (print_size_executive));
	      g_signal_connect (G_OBJECT (print_size_executive), "toggled",
				G_CALLBACK (size_executive_toggled_cb), data);
	    }
	    image = gtk_image_new_from_stock ("gtk-dnd", GTK_ICON_SIZE_DIALOG);
	    gtk_widget_show (image);
	    gtk_box_pack_start (GTK_BOX (hbox3), image, TRUE, TRUE, 0);
	  }
	}
      }
      vbox2 = gtk_vbox_new (FALSE, 6);
      gtk_widget_show (vbox2);
      gtk_box_pack_start (GTK_BOX (vbox1), vbox2, FALSE, TRUE, 0);
      {
	label = gtk_label_new (_("<b>Orientation</b>"));
	gtk_widget_show (label);
	gtk_box_pack_start (GTK_BOX (vbox2), label, FALSE, FALSE, 0);
	gtk_label_set_use_markup (GTK_LABEL (label), TRUE);
	gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
	gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);

	hbox2 = gtk_hbox_new (FALSE, 0);
	gtk_widget_show (hbox2);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox2, TRUE, TRUE, 0);
	{
	  label = gtk_label_new (_("    "));
	  gtk_widget_show (label);
	  gtk_box_pack_start (GTK_BOX (hbox2), label, FALSE, FALSE, 0);
	  gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);

	  vbox3 = gtk_vbox_new (FALSE, 6);
	  gtk_widget_show (vbox3);
	  gtk_box_pack_start (GTK_BOX (hbox2), vbox3, TRUE, TRUE, 0);
	  {
	    print_orient_portrait =
	      gtk_radio_button_new_with_mnemonic (NULL, _("P_ortrait"));
	    gtk_widget_show (print_orient_portrait);
	    gtk_box_pack_start (GTK_BOX (vbox3),
				print_orient_portrait, FALSE, FALSE, 0);
	    gtk_radio_button_set_group (GTK_RADIO_BUTTON
					(print_orient_portrait),
					print_orient_portrait_group);
	    print_orient_portrait_group =
	      gtk_radio_button_get_group (GTK_RADIO_BUTTON
					  (print_orient_portrait));

	    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON 
					  (print_orient_portrait), TRUE);
	    g_signal_connect (G_OBJECT (print_orient_portrait), "toggled",
			      G_CALLBACK (portrait_toggled_cb), data);
	    

	    print_orient_landscape =
	      gtk_radio_button_new_with_mnemonic (NULL, _("Lan_dscape"));
	    gtk_widget_show (print_orient_landscape);
	    gtk_box_pack_start (GTK_BOX (vbox3),
				print_orient_landscape, FALSE, FALSE, 0);
	    gtk_radio_button_set_group (GTK_RADIO_BUTTON
					(print_orient_landscape),
					print_orient_portrait_group);
	    print_orient_portrait_group =
	      gtk_radio_button_get_group (GTK_RADIO_BUTTON
					  (print_orient_landscape));
	    g_signal_connect (G_OBJECT (print_orient_landscape), "toggled",
			      G_CALLBACK (landscape_toggled_cb), data);
	  }
	}
      }
    }
    label = gtk_label_new (_("Paper Details"));
    gtk_widget_show (label);
    gtk_notebook_set_tab_label (GTK_NOTEBOOK (notebook),
				gtk_notebook_get_nth_page
				(GTK_NOTEBOOK (notebook), 1), label);
    gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
  }
  scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
  gtk_widget_show (scrolledwindow);
  gtk_box_pack_start (GTK_BOX (hbox1), scrolledwindow, TRUE, TRUE, 0);
  gtk_widget_set_size_request (scrolledwindow, 310, 310);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow),
				  GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrolledwindow),
				       GTK_SHADOW_ETCHED_IN);
  {
    canvas = gnome_canvas_new ();
    gtk_widget_show (canvas);
    gtk_container_add (GTK_CONTAINER (scrolledwindow), canvas);
    gnome_canvas_set_scroll_region (GNOME_CANVAS (canvas), 0, 0, 100, 100);
  }
  dialog_action_area = GTK_DIALOG (page_setup_dialog)->action_area;
  gtk_widget_show (dialog_action_area);
  gtk_button_box_set_layout (GTK_BUTTON_BOX (dialog_action_area),
			     GTK_BUTTONBOX_END);
  {
    btn_close = gtk_button_new_from_stock ("gtk-cancel");
    gtk_widget_show (btn_close);
    gtk_dialog_add_action_widget (GTK_DIALOG (page_setup_dialog),
				  btn_close, GTK_RESPONSE_CANCEL);
    GTK_WIDGET_SET_FLAGS (btn_close, GTK_CAN_DEFAULT);
    g_signal_connect_swapped (G_OBJECT (btn_close),
			      "clicked", G_CALLBACK (gtk_widget_destroy),
			      G_OBJECT (page_setup_dialog));

    btn_print = gtk_button_new_from_stock ("gtk-print");
    gtk_widget_show (btn_print);
    gtk_dialog_add_action_widget (GTK_DIALOG (page_setup_dialog),
				  btn_print, GTK_RESPONSE_OK);
    GTK_WIDGET_SET_FLAGS (btn_print, GTK_CAN_DEFAULT);
    g_signal_connect (G_OBJECT (btn_print),
		      "clicked", G_CALLBACK (print_cb), data);
  }
#if 0
  gtk_label_set_mnemonic_widget (GTK_LABEL (label), scrolledwindow);
#endif

  data->adj = gtk_range_get_adjustment (GTK_RANGE (hscale));
  g_signal_connect (G_OBJECT (data->adj), 
		    "value_changed", G_CALLBACK (zoom_changed), data);
  data->pi->canvas = canvas;
  data->hscale	   = hscale;
  data->btn_close  = btn_close;
  data->btn_center = btn_center;
  data->btn_print  = btn_print;

  return page_setup_dialog;
}

/* ************************************************************************* */
void
print_dialog_show (GtkWindow	*parent, 
		   GList	*list) {
  PrintInfo    *pi;
  DialogData   *data;
  char         *title = NULL;

  pi = g_new0 (PrintInfo, 1);
  pi->ref_count = 1;
  pi->zoom = 1.0;

  pi->config	 = gnome_print_config_default ();
  pi->image_path = g_strdup (ti_get_file_path (list));
  pi->filename	 = (gchar *) ti_get_filename (list);
  pi->pixbuf	 = gdk_pixbuf_copy (ti_get_pixbuf (list));

  if (pi->pixbuf == NULL) {
    g_warning ("Cannot load image %s", pi->image_path);
    print_info_unref (pi);
    return;
  }
  g_object_ref (pi->pixbuf);

  pi->image_w	 = (double) gdk_pixbuf_get_width  (pi->pixbuf);
  pi->image_h	 = (double) gdk_pixbuf_get_height (pi->pixbuf);
  pi->portrait	 = TRUE;
  pi->use_colors = TRUE;

  data = g_new (DialogData, 1);
  data->pi = pi;
  data->dialog = page_setup_dialog_new (data);

  title = g_strdup_printf (_("Print %s"), pi->filename);

  gtk_window_set_title (GTK_WINDOW (data->dialog), title); 
  g_free (title);

  gtk_window_set_transient_for (GTK_WINDOW (data->dialog), parent);

  gnome_canvas_set_pixels_per_unit (GNOME_CANVAS (pi->canvas), 
				    CANVAS_ZOOM);
  page_update (data);

  gtk_widget_show (data->dialog);
}

/* *************************************************** print-callbacks.c *** */
