/* ************************************************************ player.c *** *
 * ץ졼
 *
 * Copyright (C) 2001-2003 Yasuyuki SUGAYA <sugaya@suri.it.okayama-u.ac.jp>
 * Okayama University
 *                                    Time-stamp: <02/06/28 16:33:54 sugaya>
 * ************************************************************************* */
#include <teo.h>
#include "teoeyes.h"
#include "draw.h"
#include "image_io.h"
#include "image_window.h"
#include "player.h"

#define	MIN_SPEED	1
#define	MAX_SPEED	1000
#define	DEFAULT_SPEED	50

/* ؿ ************************************************************ */
static void	player_update_controller	(gint		value);
static void	player_show_nth 		(gint		n);
static void	player_show_next		(void);
static void	player_show_previous		(void);
static void	play_forward			(void);
static void	play_backward			(void);
static void	start_play_forward		(void);
static void	start_play_backward	 	(void);
static void	stop_play_image			(void);
static void	exit_player			(void);
static void	set_play_speed 			(float		value);

static int	player_scrollbar_changed	(GtkWidget	*widget,
						 GtkAdjustment	*adj);
static int	player_spinbutton_changed	(GtkWidget	*widget,
						 GtkAdjustment	*adj);

/* ޥ֤Υåؿ ************************************************ */
static int	check_control_widget_proximity 	(GtkWidget	*widget,
						 gint		x,
						 gint		y);
static int	check_each_button_proximity	(ButtonData	*button,
						 gint		x,
						 gint		y);
static int	check_button_proximity		(GtkWidget	*window,
						 gint		x,
						 gint		y);
static int	check_slider_press_proximity	(GtkWidget	*window,
						 SliderData	*slider,
						 gint		x,
						 gint		y);
static void	check_font_motion_proximity 	(FontData	*font,
						 gint		x,
						 gint		y);
#if 0
static void	check_font_leave_proximity 	(FontData	*font);
#endif
static int	player_mouse_motion		(GtkWidget	*widget,
						 GdkEventMotion	*ev,
						 gpointer	data);
static int	player_mouse_pressed		(GtkWidget	*widget,
						 GdkEventMotion	*ev,
						 gpointer	data);
static int	player_mouse_released 		(GtkWidget	*widget,
						 GdkEventMotion	*ev,
						 gpointer	data);
static int	player_mouse_leave		(GtkWidget	*widget,
						 GdkEventMotion	*ev,
						 gpointer	data);

/* ǡؿ **************************************************** */
static void	draw_button 			(GtkWidget	*window,
						 ButtonData	*button,
						 gint		type);
static void	draw_item 			(GtkWidget	*window,
						 ItemData	*item,
						 gint		type);
static void	draw_slider_private		(GtkWidget	*window,
						 SliderData	*slider,
						 float		position,
						 gint		absolute,
						 gint		move,
						 gint		force);
static void	slider_motion			(GtkWidget	*window,
						 SliderData	*slider,
						 gint		x,
						 gint		y);
static void	draw_font 			(GtkWidget	*window,
						 FontData	*font);
static void	update_font			(GtkWidget	*window);
static void	set_font_text 			(GtkWidget	*window,
						 const gchar	*text);
static void	draw_digit			(GtkWidget	*window,
						 NumberData	*digit,
						 int		data);

/* ǡɤ߹ߴؿ ************************************************ */
static ButtonData*	button_new		(gchar		*filename, 
						 gint		cx,
						 gint		cy,
						 gint		x,
						 gint		y,
						 gint		area,
						 gint		type);
static ItemData*	item_new		(gchar		*filename,
						 gint		sections,
						 gint		cx,
						 gint		cy,
						 gint		area);
static SliderData*	slider_new		(gchar		*filename,
						 ItemData	*item,
						 gint		prelight,
						 gint		verticle,
						 gint		reversed,
						 gint		length,
						 gint		cx,
						 gint		cy,
						 gint		x,
						 gint		y,
						 gint		area);
static FontData*	font_new		(gchar		*filename, 
						 gint		length,
						 gint		cx,
						 gint		cy,
						 gint		x,
						 gint		y,
						 gint		area);
static SkinData*	player_skin_new		(gchar		*skin_data);

static gint		play_timer;
static gint		play_speed = DEFAULT_SPEED;
static gint		repeat_flg;

static const gchar	iso_ascii[] = {
  ' ','|','c','L','c','Y','|','S','"','c',' ','<','!','-','r','~',
  'o',' ','2','3','\'','u','p','.',',','1',' ','>',' ',' ',' ','?',
  'A','A','A','A','A','A','A','C','E','E','E','E','I','I','I','I',
  'D','N','O','O','O','O','O','x','0','U','U','U','U','Y','P','B',
  'a','a','a','a','a','a','a','c','e','e','e','e','i','i','i','i',
  'o','n','o','o','o','o','o','/','o','u','u','u','u','y','p','y'
};

/* ************************************************************************* *
 * ؿ
 * ************************************************************************* */

/* 楦åȤι ************************************************** */
static void
player_update_controller (gint	value) {
  GtkWidget	*widget;
  GtkAdjustment	*adj;
  gchar		buf[32];

  widget     = G_GET_WIDGET (player, "spinbutton");
  adj        = (GtkAdjustment *) G_GET_OBJECT (player, "spinbutton_adj");
  adj->value = value;
  sprintf (buf, "%d", value);
  gtk_entry_set_text (GTK_ENTRY (widget), buf);

  widget     = G_GET_WIDGET (player, "scrollbar");
  adj        = (GtkAdjustment *) G_GET_OBJECT (player, "scrollbar_adj");
  adj->value = value;
  gtk_widget_queue_draw (widget);
}

/* ꤵ줿ɽ **************************************************** */
static void
player_show_nth (gint	n) {
  if (g_list_nth (g_list_first (image_list), n)) {
    image_list = g_list_nth (g_list_first (image_list), n);
    teoeyes_image_set (ti_get_image (image_list), 0, to->min, to->max);
    set_font_text (player, ti_get_filename (image_list));

    /*  */
    gtk_widget_set_size_request (G_GET_WIDGET (image_window, "canvas"),
				 ti_get_current_width  (image_list),
				 ti_get_current_height (image_list));      
  }
}

/* βɽ ********************************************************** */
static void
player_show_next (void) {
  if (ti_get_frame (image_list) < ti_get_nframes (image_list) - 1) {
    /* Υե졼ǡΥå */
    if (g_list_length (ti_get_pixbuf_list (image_list)) == 1) {
      teoeyes_image_load (ti_get_image (image_list), 
			  ti_get_frame (image_list) + 1, to->min, to->max);
    } else {
      teoeyes_image_set (ti_get_image (image_list), 
			 ti_get_frame (image_list) + 1, to->min, to->max);
    }
  } else {
    /* βɽ */
    if (g_list_next (image_list)) {
      image_list = g_list_next (image_list);
      teoeyes_image_set (ti_get_image (image_list), 0, to->min, to->max);
      set_font_text (player, ti_get_filename (image_list));
      player_update_controller (g_list_length (g_list_first (image_list)) -
				g_list_length (image_list));
    }
  }
  /*  */
  gtk_widget_set_size_request (get_widget (image_window, "canvas"),
			       ti_get_width  (image_list),
			       ti_get_height (image_list));
}

/* βɽ ********************************************************** */
static void
player_show_previous (void) {
  if (ti_get_frame (image_list) > 0) {
    /* Υե졼ǡΥå */
    if (g_list_length (ti_get_pixbuf_list (image_list)) == 1) {
      teoeyes_image_load (ti_get_image (image_list),
			  ti_get_frame (image_list) - 1, to->min, to->max);
    } else {
      teoeyes_image_set (ti_get_image (image_list),
			 ti_get_frame (image_list) - 1, to->min, to->max);
    }
  } else {
    /* βɽ */
    if (g_list_previous (image_list)) {
      image_list = g_list_previous (image_list);
      if (g_list_length (ti_get_pixbuf_list (image_list)) == 1) {
	teoeyes_image_load (ti_get_image (image_list), -1, to->min, to->max);
      } else {
	teoeyes_image_set (ti_get_image (image_list), -1, to->min, to->max);
      }
      set_font_text (player, ti_get_filename (image_list));
      player_update_controller (g_list_length
				(g_list_first(image_list))-
				g_list_length (image_list));
    }
  }
  /*  */
  gtk_widget_set_size_request (get_widget (image_window, "canvas"),
			       ti_get_width  (image_list),
			       ti_get_height (image_list));
}

/* ꥹȤϢ³ɽ() ******************************************** */
static void
play_forward (void) {
  if (ti_get_frame (image_list) < ti_get_nframes (image_list) - 1) {
    /* ޥե졼TEOļΥե졼ब... */

    /* Υե졼ǡΥå */
    if (g_list_length (ti_get_pixbuf_list (image_list)) == 1) {
      teoeyes_image_load (ti_get_image (image_list),
			  ti_get_frame (image_list) + 1, to->min, to->max);
    } else {
      teoeyes_image_set (ti_get_image (image_list), 
			 ti_get_frame (image_list) + 1, to->min, to->max);
    }
  } else if (g_list_next (image_list)) {

    /* β... */
    image_list = g_list_next (image_list);
    teoeyes_image_set (ti_get_image (image_list), 0, to->min, to->max);
    set_font_text (player, ti_get_filename (image_list));
    player_update_controller (g_list_length(g_list_first(image_list))-
			      g_list_length (image_list));
      
  } else if (repeat_flg) {
      
    /* ֤⡼ɤξ... */
    image_list = g_list_first (image_list);
    teoeyes_image_set (ti_get_image (image_list), 0, to->min, to->max);
    set_font_text (player, ti_get_filename (image_list));
    player_update_controller (0);

  } else {
    /*  */
    if (play_timer) gtk_timeout_remove (play_timer);
    play_timer = 0;
  }
  /*  */
  gtk_widget_set_size_request (get_widget (image_window, "canvas"),
			       ti_get_width  (image_list),
			       ti_get_height (image_list));
}

/* ꥹȤϢ³ɽ() ******************************************** */
static void
play_backward (void) {
  if (ti_get_frame (image_list) > 0) {    
    /* ޥե졼TEOΥե졼ब... */

    /* Υե졼ǡΥå */
    if (g_list_length (ti_get_pixbuf_list (image_list)) == 1) {
      teoeyes_image_load (ti_get_image (image_list),
			  ti_get_frame (image_list) - 1, to->min, to->max);
    } else {
      teoeyes_image_set (ti_get_image (image_list),
			 ti_get_frame (image_list) - 1, to->min, to->max);
    }
  } else if (g_list_previous (image_list)) {
    /* β... */
    image_list = g_list_previous (image_list);
    if (g_list_length (ti_get_pixbuf_list (image_list)) == 1) {
      teoeyes_image_load (ti_get_image (image_list), -1, to->min, to->max);
    } else {
      teoeyes_image_set (ti_get_image (image_list), -1, to->min, to->max);
    }
    set_font_text (player, ti_get_filename (image_list));
    player_update_controller (g_list_length(g_list_first(image_list))-
			      g_list_length (image_list));
  } else if (repeat_flg) {
    /* ֤⡼ɤξ... */
    image_list = g_list_last (image_list);
    if (g_list_length (ti_get_pixbuf_list (image_list)) == 1) {
      teoeyes_image_load (ti_get_image (image_list), -1, to->min, to->max);
    } else {
      teoeyes_image_set (ti_get_image (image_list), -1, to->min, to->max);
    }
    set_font_text (player, ti_get_filename (image_list));
    player_update_controller(g_list_length(g_list_first(image_list))-1);
  } else {
    /*  */
    if (play_timer) gtk_timeout_remove (play_timer);
    play_timer = 0;
  }
  /*  */
  gtk_widget_set_size_request (get_widget (image_window, "canvas"),
			       ti_get_width  (image_list),
			       ti_get_height (image_list));
}

/* ץ쥤ܥ󤬲줿ν ****************************************** */
static void 
start_play_forward (void) {
  if (play_timer != 0) return;
  play_timer = gtk_timeout_add (play_speed,
				(GtkFunction) play_forward, NULL);
}

/* Хåܥ󤬲줿ν ****************************************** */
static void
start_play_backward (void) {
  if (play_timer != 0) return;  
  play_timer = gtk_timeout_add (play_speed,
				(GtkFunction) play_backward, NULL);
}

/* ȥåץܥ󤬲줿ν **************************************** */
static void
stop_play_image (void) {
  if (play_timer) gtk_timeout_remove (play_timer);
  play_timer = 0;
}

/* ץ졼Ĥ ******************************************************** */
static void
exit_player (void) {
  stop_play_image ();
  gtk_widget_hide (player);
  gtk_widget_destroy (player);
  player = NULL;
  if (update_timer) gtk_timeout_remove (update_timer);
  update_timer = 0;
}

/* ®٤ ********************************************************** */
static void
set_play_speed (float	value) {
  play_speed = MIN_SPEED + ((MAX_SPEED - MIN_SPEED) / 100.0) * (value * 100.0);
}

/* Сˤ ****************************************** */
static int
player_scrollbar_changed	(GtkWidget	*widget,
				 GtkAdjustment	*adj) {
  player_update_controller ((gint) adj->value);
  player_show_nth ((int) adj->value);
  
  return TRUE;
}

/* ԥܥˤ ******************************************** */
static int
player_spinbutton_changed	(GtkWidget	*widget,
				 GtkAdjustment	*adj) {
  player_update_controller ((gint) adj->value);
  player_show_nth ((int) adj->value);
  
  return TRUE;
}

/* ************************************************************************* *
 * ޥåؿ
 * ************************************************************************* */

/* ޥ뤬ȥ륦åȤΥեäƤ뤫Ĵ٤  */
static int
check_control_widget_proximity (GtkWidget	*widget,
				gint		x,
				gint		y) {
  SkinData	*skin;

  skin = (SkinData *) G_GET_PARAMETER (widget, "skin");
  if (x >= skin->cx && x < skin->cx + skin->cw &&
      y >= skin->cy && y < skin->cy + skin->ch) {
    return TRUE;
  } else {
    return FALSE;
  }
}

/* ޥ뤬ܥΥեäƤ뤫Ĵ٤ ****************** */
static int
check_each_button_proximity (ButtonData	*button,	
			     gint	x,
			     gint	y) {
  if (x >= button->x && x < button->x + button->w &&
      y >= button->y && y < button->y + button->h) {
    return TRUE;
  } else {
    return FALSE;
  }
}

/* ޥ뤬ɤΥܥΥեäƤ뤫Ĵ٤ ************** */
static int
check_button_proximity (GtkWidget	*window,
			gint		x,
			gint		y) {
  SkinData	*skin;

  skin = (SkinData *) gtk_object_get_data (GTK_OBJECT (window), "skin");

  if (check_each_button_proximity (skin->play_button, x, y))
    return 1;
  if (check_each_button_proximity (skin->back_button, x, y))
    return 2;
  if (check_each_button_proximity (skin->next_button, x, y))
    return 3;
  if (check_each_button_proximity (skin->prev_button, x, y))
    return 4;
  if (check_each_button_proximity (skin->stop_button, x, y))
    return 5;
  if (check_each_button_proximity (skin->rept_button, x, y))
    return 6;
  if (check_each_button_proximity (skin->exit_button, x, y))
    return 7;
  return 0;
}

/* ޥ뤬饤äƤ뤫Ĵ٤ ************************** */
static int
check_slider_press_proximity (GtkWidget		*window,
			      SliderData	*slider,
			      gint		x,
			      gint		y) {
  float	pos;

  if ((slider->verticle &&
       x >= slider->x && x < slider->x + slider->h &&
       y >= slider->y && y < slider->y + slider->w) ||
      (!slider->verticle &&
       x >= slider->x && x < slider->x + slider->w &&
       y >= slider->y && y < slider->y + slider->h)) {

    if (slider->verticle) {
      pos = y - slider->y - (slider->handle_width / 2);
    } else {
      pos = x - slider->x - (slider->handle_width / 2);
    }
    if (pos < 0) pos = 0;
    if (pos > slider->w - slider->handle_width)
      pos = slider->w - slider->handle_width;
    draw_slider_private (window, slider, pos, TRUE, TRUE, FALSE);
    if (slider->reversed) {
      set_play_speed (1.0 - (pos / (slider->w - slider->handle_width)));
    } else {
      set_play_speed (pos / (slider->w - slider->handle_width));
    }
    draw_digit (window,
		((SkinData *) G_GET_PARAMETER (window, "skin"))->digit,
		play_speed);
    return TRUE;
  }
  return FALSE;
}

/* ޥ뤬ȥɽϰϤäƤ뤫Ĵ٤ ******************** */
static void
check_font_motion_proximity (FontData	*font,
			     gint	x,
			     gint	y) {
  if (!font) return;

  if (x >= font->x && x < font->x + font->w &&
      y >= font->y && y < font->y + font->h) {
    font->scroll = 1;
  } else {
    font->scroll = -1;
  }
} 

/* ޥ뤬ɥФΥȥɽ **************** */
#if 0
static void
check_font_leave_proximity (FontData	*font) {
  if (!font) return;
  font->scroll = -1;
}
#endif

/* ޥưν ************************************************** */
static int
player_mouse_motion (GtkWidget		*widget,
			  GdkEventMotion	*ev,
			  gpointer		data) {
  GdkModifierType	mod;
  SkinData		*skin;
  int			id;
  int			move_flg;
  int			slider_flg;
  gint			pos_x, pos_y;
  gint			new_x, new_y;
  gint			xmove, ymove;
  gint			x = (gint) ev->x;
  gint			y = (gint) ev->y;

  skin       = (SkinData *) G_GET_PARAMETER (widget, "skin");
  move_flg   = (int)        G_GET_PARAMETER (widget, "move_flg");
  slider_flg = (int)        G_GET_PARAMETER (widget, "slider_flg");  
  xmove      = (int)        G_GET_PARAMETER (widget, "move_x_pos");
  ymove      = (int)        G_GET_PARAMETER (widget, "move_y_pos");

  if (check_control_widget_proximity (widget, x, y)) return TRUE;

  if (move_flg) {
    gdk_window_get_pointer (NULL, &pos_x, &pos_y, &mod);
    new_x = pos_x - xmove;	
    new_y = pos_y - ymove;
    gdk_window_move (widget->window, new_x, new_y);
  } else if (slider_flg) {
    slider_motion (widget, skin->slider, x, y);
  } else {
    if ((id = check_button_proximity (widget, x, y)) != 0) {
      switch (id) {
      case 1:
	draw_button (widget, skin->play_button, 2);
	break;
      case 2:
	draw_button (widget, skin->back_button, 2);
	break;
      case 3:
	draw_button (widget, skin->next_button, 2);
	break;
      case 4:
	draw_button (widget, skin->prev_button, 2);
	break;
      case 5:
	draw_button (widget, skin->stop_button, 2);
	break;
      case 6:
	draw_button (widget, skin->rept_button, 2);
	break;
      case 7:
	draw_button (widget, skin->exit_button, 2);
	break;
      }
    } else {
      draw_button (widget, skin->play_button, 0);
      draw_button (widget, skin->back_button, 0);
      draw_button (widget, skin->next_button, 0);
      draw_button (widget, skin->prev_button, 0);
      draw_button (widget, skin->stop_button, 0);
      draw_button (widget, skin->rept_button, 0);
      draw_button (widget, skin->exit_button, 0);
    }
  }
  check_font_motion_proximity (skin->font, x, y);
  
  return TRUE;
}

/* ޥܥ󤬥å줿ν ************************************ */
static int
player_mouse_pressed (GtkWidget		*widget,
			   GdkEventMotion	*ev,
			   gpointer		data) {
  SkinData		*skin;
  GdkModifierType	mod;
  int			id;
  int			rept_flg;
  int			x = (gint) ev->x;
  int			y = (gint) ev->y;

  skin = (SkinData *) G_GET_PARAMETER (widget, "skin");
  G_SET_PARAMETER (widget, "move_x_pos", x);
  G_SET_PARAMETER (widget, "move_y_pos", y);

  if ((id = check_button_proximity (widget, x, y)) != 0) {
    switch (id) {
    case 1:
      draw_button (widget, skin->play_button, 1);
      break;
    case 2:
      draw_button (widget, skin->back_button, 1);
      break;
    case 3:
      draw_button (widget, skin->next_button, 1);
      break;
    case 4:
      draw_button (widget, skin->prev_button, 1);
      break;
    case 5:
      draw_button (widget, skin->stop_button, 1);
      break;
    case 6:

      rept_flg = (int) G_GET_PARAMETER (widget, "rept_flg");
      G_SET_PARAMETER (widget, "rept_flg", !rept_flg);
      repeat_flg = !rept_flg;
      draw_button (widget, skin->rept_button, 1);
      draw_item   (widget, skin->rept_item, !rept_flg);
      break;
    case 7:
      draw_button (widget, skin->exit_button, 1);
      break;
    }
  } else if (check_slider_press_proximity (widget, skin->slider, x, y)) {
    G_SET_PARAMETER (widget, "slider_flg", 1);
  } else {
    gdk_window_get_pointer (widget->window, &x, &y, &mod);
    if (!check_control_widget_proximity (widget, x, y)) {
      G_SET_PARAMETER (widget, "move_flg", 1);
    }
  }
  return FALSE;
}

/* ޥܥΥ줿ν ****************************************** */
static int
player_mouse_released (GtkWidget		*widget,
			    GdkEventMotion	*ev,
			    gpointer		data) {
  SkinData	*skin;
  int		id;
  int		xpos, ypos;

  skin = (SkinData *) G_GET_PARAMETER (widget, "skin");  
  xpos = (int) G_GET_PARAMETER (widget, "move_x_pos");
  ypos = (int) G_GET_PARAMETER (widget, "move_y_pos");
  G_SET_PARAMETER (widget, "move_flg",   0);
  G_SET_PARAMETER (widget, "slider_flg", 0);

  id = check_button_proximity (widget, xpos, ypos);

  switch (id) {
  case 1:
    draw_button (widget, skin->play_button, 2);
    start_play_forward ();
    break;
  case 2:
    draw_button (widget, skin->back_button, 2);
    start_play_backward ();
    break;
  case 3:
    draw_button (widget, skin->next_button, 2);
    player_show_next ();
    break;
  case 4:
    draw_button (widget, skin->prev_button, 2);
    player_show_previous ();
    break;
  case 5:
    stop_play_image ();
    draw_button (widget, skin->stop_button, 2);
    break;
  case 6:
    draw_button (widget, skin->rept_button, 2);
    break;
  case 7:
    exit_player ();
#if 0      
    draw_button (widget, skin->exit_button, 2);
#endif
    break;
  default:
    break;
  }
  return FALSE;
}

/* ޥݥ󥿤ե鳰줿ν **************************** */
static int
player_mouse_leave (GtkWidget		*widget,
			 GdkEventMotion		*ev,
			 gpointer		data) {
  SkinData	*skin;

  skin = (SkinData *) G_GET_PARAMETER (widget, "skin");
  draw_button (widget, skin->play_button, 0);
  draw_button (widget, skin->back_button, 0);
  draw_button (widget, skin->next_button, 0);
  draw_button (widget, skin->prev_button, 0);
  draw_button (widget, skin->stop_button, 0);
  draw_button (widget, skin->rept_button, 0);
  draw_button (widget, skin->exit_button, 0);

  return TRUE;
}

/* ************************************************************************* *
 * ؿ
 * ************************************************************************* */
#if 0
/* κ ********************************************************** */
static void
redraw_skin (GtkWidget	*window) {
  SkinData	*skin;
  int		w, h;

  skin = (SkinData *) G_GET_PARAMETER (window, "skin");
  gdk_window_get_size (skin->mask[4], &w, &h);
  gtk_widget_set_usize (window, w, h);
  gtk_widget_shape_combine_mask (window, skin->mask[4], 0, 0);
}
#endif

/* ܥ ************************************************************ */
static void
draw_button (GtkWidget	*window,
	     ButtonData	*button,
	     int	type) {
  GtkWidget	*canvas;
  SkinData	*skin;
  char		name[64];

  skin = (SkinData *) G_GET_PARAMETER (window, "skin");
  sprintf (name, "canvas%d", button->area);
  canvas = G_GET_WIDGET (window, name);
  gdk_draw_pixmap (skin->background[button->area], 
		   canvas->style->fg_gc[GTK_WIDGET_STATE (canvas)],
		   button->pixmap, button->w * type, 0, 
		   button->cx, button->cy, button->w, button->h);
  gdk_window_clear (canvas->window);
}

/* ƥ ********************************************************** */
static void
draw_item (GtkWidget	*window,
	   ItemData	*item,
	   int		type) {
  GtkWidget	*canvas;
  SkinData	*skin;
  char		name[64];

  skin = (SkinData *) G_GET_PARAMETER (window, "skin");
  sprintf (name, "canvas%d", item->area);
  canvas = G_GET_WIDGET (window, name);

  gdk_draw_pixmap (skin->background[item->area], 
		   canvas->style->fg_gc[GTK_WIDGET_STATE (canvas)],
		   item->pixmap, 0, item->h * type,
		   item->cx, item->cy, item->w, item->h);
  gdk_window_clear (canvas->window);
}

/* 饤 ******************************************************** */
static void
draw_slider_private (GtkWidget	*window,
		     SliderData	*slider,
		     float	position,
		     gint	absolute,
		     gint	move,
		     gint	force) {
  gint	new_pos;

  if (move) {
    if (absolute) {
      new_pos = (float) position;
    } else {
      new_pos = (float) (slider->w - slider->handle_width) * position;
    }
    if (new_pos < 0) new_pos = 0;
    if (new_pos > slider->w - slider->handle_width)
      new_pos = slider->w - slider->handle_width;
  } else {
    new_pos = slider->position;
  }

  if (force) {
    if (move) slider->position = new_pos;
    if (slider->item) {
      gint section =
	(float) slider->position / (slider->w - slider->handle_width)
	* (slider->item->sections - 1);
      draw_item (window, slider->item, section);
    }
    return;
  }

  if (slider->position != new_pos) {
    if (slider->item) {
      gint section =
	(float) new_pos / (slider->w - slider->handle_width)
	* (slider->item->sections - 1);
      draw_item(window, slider->item, section);
    }
  }
}

/* ޥ뤬饤äƤ뤫Ĵ٤ ************************** */
static void
slider_motion (GtkWidget	*window,
	       SliderData	*slider,
	       gint		x,
	       gint		y) {
  float	pos;

  if (slider->verticle) {
    pos = y - slider->y - (slider->handle_width / 2);
  } else {
    pos = x - slider->x - (slider->handle_width / 2);
  }
  if (pos < 0) pos = 0;
  if (pos > slider->w - slider->handle_width)
    pos = slider->w - slider->handle_width;
  if (slider->position != pos) {
    if (slider->reversed) {
      set_play_speed (1.0 - (pos / (slider->w - slider->handle_width)));
    } else {
      set_play_speed (pos / (slider->w - slider->handle_width));
    }
    draw_digit (window,
		((SkinData *)
		 gtk_object_get_data (GTK_OBJECT (window), "skin"))->digit,
		play_speed);
    draw_slider_private (window, slider, pos, TRUE, TRUE, FALSE);
  }
}

/* ȥեȤ ************************************************** */
static void
draw_font (GtkWidget	*window,
	   FontData	*font) {
  GtkWidget	*canvas;
  SkinData	*skin;
  gint		n;
  gint		px, py;
  guint8	c;
  gint		l;
  gchar		*start;
  char		name[64];

  if (!font) return;
  if (font->text) {
    l = strlen (font->text);
  } else {
    l = 0;
  }
  if (font->offset >= l) font->offset = l - 1;
  if (font->offset < 0)  font->offset = 0;
  if (font->text) {
    start = font->text + font->offset;
  } else {
    start = NULL;
  }
  l -= font->offset;

  skin = (SkinData *) G_GET_PARAMETER (window, "skin");
  sprintf (name, "canvas%d", font->area);
  canvas = G_GET_WIDGET (window, name);

  for (n = 0; n < font->length; n++) {
    if (n < l) {
      c = start[n];
      if (c >= 32 && c < 128) {
	c -= 32;
      } else if (c >= 160 && c < 256) {
	c = iso_ascii[c-160];
	c -= 32;
      } else {
	c = 0;
      }
      py = c / 32;
      px = c - (py * 32);
      px = px * font->char_width;
      py = py * font->char_height;
      gdk_draw_pixmap (skin->background[font->area],
		       canvas->style->fg_gc[GTK_WIDGET_STATE (canvas)],
		       font->pixmap,
		       px, py,
		       font->cx + n * font->char_width, font->cy,
		       font->char_width, font->char_height);
    } else {
      gdk_draw_pixmap (skin->background[font->area],
		       canvas->style->fg_gc[GTK_WIDGET_STATE (canvas)],
		       font->pixmap,
		       0, 0,
		       font->cx + n * font->char_width, font->cy,
		       font->char_width, font->char_height);
    }
  }
  gdk_window_clear (canvas->window);
}

/* ȥեȤΥåץǡ ****************************************** */
static void
update_font (GtkWidget	*window) {
  FontData	*font;

  font = ((SkinData *) G_GET_PARAMETER (window, "skin"))->font;

  if (!font || !font->text) return;
  if (font->scroll > 0) {
    gint l = strlen (font->text);
    if (font->offset < l - font->length && l > font->length) {
      font->offset++;
      draw_font (window, font);
    }
  } else if (font->offset > 0) {
    while (font->offset > 0) {
      font->offset--;
      draw_font (window, font);
    }
  }
}

/* ȥեȤ ************************************************** */
static void
set_font_text (GtkWidget	*window,
	       const gchar	*text) {
  SkinData	*skin;
  FontData	*font;

  skin = (SkinData *) G_GET_PARAMETER (window, "skin");
  font = skin->font;
  
  if (!font) return;
  g_free (font->text);
  if (text) {
    font->text = g_strdup (text);
  } else {
    font->text = NULL;
  }
  font->offset = 0;
  draw_font (window, font);
}

/* ȥեȤ ************************************************** */
static void
draw_digit (GtkWidget	*window,
	    NumberData	*digit,
	    int		data) {
  GtkWidget	*canvas;
  SkinData	*skin;
  int		n;
  int		pos;
  int		num;
  int		base = 1000;
  char		name[64];

  skin = (SkinData *) G_GET_PARAMETER (window, "skin");
  sprintf (name, "canvas%d", digit->area);
  canvas = G_GET_WIDGET (window, name);

  for (num = data, n = 0; n < 4; n++) {
    /*  */
    pos = (int) (num / base);
    if (pos != 0) num -= pos * base;
    base /= 10;
    /* 褹 */
    gdk_draw_pixmap (skin->background[digit->area], 
		     canvas->style->fg_gc[GTK_WIDGET_STATE (canvas)],
		     digit->pixmap,
		     digit->w * pos, 0,
		     digit->cx + n * digit->w, digit->cy,
		     digit->w, digit->h);
  }
  gdk_window_clear (canvas->window);
}

/* ************************************************************************* *
 * ǡ
 * ************************************************************************* */

/* ܥǡ ****************************************************** */
static ButtonData*
button_new (gchar		*filename,
	    int			cx,
	    int			cy,
	    int			x,
	    int			y,
	    int			area,	
	    int			type) {
  ButtonData	*b;
  int		w, h;
  GdkPixbuf	*pbuf;
  GdkBitmap	*mask;

  b = g_new (ButtonData, 1);

  pbuf = gdk_pixbuf_new_from_file (filename, NULL);
  gdk_pixbuf_render_pixmap_and_mask (pbuf, &(b->pixmap), &mask, 255);
  gdk_pixbuf_unref (pbuf);
  if (mask) gdk_bitmap_unref (mask);

  gdk_window_get_size (b->pixmap, &w, &h);
  b->cx   = cx;
  b->cy	  = cy;
  b->x	  = x;
  b->y	  = y;
  b->w	  = (type == 0) ? w / 3 : w / 2;
  b->h	  = h;
  b->area = area;
  
  return b;	
}

/* ƥǡ **************************************************** */
static ItemData*
item_new (gchar		*filename,
	  int 		sections,
	  int		cx,
	  int		cy,
	  int		area) {
  ItemData	*i;
  int		w, h;
  GdkPixbuf	*pbuf;
  GdkBitmap	*mask;

  i = g_new (ItemData, 1);

  pbuf = gdk_pixbuf_new_from_file (filename, NULL);
  gdk_pixbuf_render_pixmap_and_mask (pbuf, &(i->pixmap), &mask, 255);
  gdk_pixbuf_unref (pbuf);
  if (mask) gdk_bitmap_unref (mask);

  gdk_window_get_size (i->pixmap, &w, &h);

  i->sections	= sections;
  i->w		= w;
  i->h		= h / sections;
  i->cx		= cx;
  i->cy		= cy;
  i->area	= area;
  
  return i;
}


/* 饤ǡ ************************************************** */
static SliderData*
slider_new (gchar		*filename,
	    ItemData		*item,
	    int			prelight,
	    int			verticle,
	    int			reversed,
	    int			length,
	    int			cx,
	    int			cy,
	    int			x,
	    int			y,
	    int			area) {
  SliderData	*s;
  int		w, h;
  GdkPixbuf	*pbuf;
  GdkBitmap	*mask;

  s = g_new (SliderData, 1);

  pbuf = gdk_pixbuf_new_from_file (filename, NULL);
  gdk_pixbuf_render_pixmap_and_mask (pbuf, &(s->pixmap), &mask, 255);
  gdk_pixbuf_unref (pbuf);
  if (mask) gdk_bitmap_unref (mask);
  gdk_window_get_size (s->pixmap, &w, &h);

  s->prelight = prelight;
  s->verticle = verticle;
  s->reversed = reversed;
  s->cx	      = cx;
  s->cy	      = cy;
  s->x	      = x;
  s->y	      = y;
  s->position = 0;
  s->item     = item;
  s->area     = area;
  
  if (verticle) {
    s->w	     = length;
    s->h	     = w;
    s->handle_width  = (h - length) / (2 + prelight);
    s->handle_height = w;
  } else {
    s->w	     = length;
    s->h	     = h;
    s->handle_width  = (w - length) / (2 + prelight);
    s->handle_height = h;
  }

  return s;
}


/* եȥǡ(ե) ******************************** */
static FontData*
font_new (gchar		*filename, 
	  int		length,
	  int		cx,
	  int		cy,
	  int		x,
	  int		y,
	  int		area) {
  FontData	*f;
  int		w, h;
  GdkPixbuf	*pbuf;
  GdkBitmap	*mask;

  f = g_new (FontData, 1);

  pbuf = gdk_pixbuf_new_from_file (filename, NULL);
  gdk_pixbuf_render_pixmap_and_mask (pbuf, &(f->pixmap), &mask, 255);
  gdk_pixbuf_unref (pbuf);
  if (mask) gdk_bitmap_unref (mask);
  gdk_window_get_size (f->pixmap, &w, &h);

  f->char_width = w / 32;
  f->char_height= h / 3;
  f->w		= f->char_width * length;
  f->h		= f->char_height;
  f->length	= length;
  f->cx		= cx;
  f->cy		= cy;
  f->x		= x;
  f->y		= y;
  f->text	= NULL;
  f->offset	= 0;
  f->extended	= 0;
  f->area	= area;
  
  return f;
}


/* ʥСǡ(ե) ******************************** */
static NumberData*
digit_new (gchar	*filename,
	   int		cx,
	   int		cy,
	   int		area) {
  NumberData	*d;
  int		w, h;
  GdkPixbuf	*pbuf;
  GdkBitmap	*mask;

  d = g_new (NumberData, 1); 

  pbuf = gdk_pixbuf_new_from_file (filename, NULL);
  gdk_pixbuf_render_pixmap_and_mask (pbuf, &(d->pixmap), &mask, 255);
  gdk_pixbuf_unref (pbuf);
  if (mask) gdk_bitmap_unref (mask);
  gdk_window_get_size (d->pixmap, &w, &h);

  d->w    = w / 11;
  d->h	  = h;
  d->cx	  = cx;
  d->cy	  = cy;
  d->area = area;
  
  return d;
}


/* ǡ ****************************************************** */
static SkinData*
player_skin_new (gchar	*skin_data) {
  SkinData	*skin;
  GdkPixbuf 	*pbuf;
  GdkBitmap	*dummy;
  ItemData	*slider_item;
  FILE		*fp;
  gchar		buf[1024];
  gchar		b1[64], b2[64], b3[64];
  gchar		*skin_path;
  gchar		*name;
  gint		cx, cy, a, x, y, l, n;

  /* եɤ߹ */
  fp = fopen (skin_data, "r");
  if (!fp) return NULL;

  /* ¤Τΰ */
  skin = g_new (SkinData, 1);

  /* ǡѥγ */
  g_separate_path (skin_data, &skin_path, &name);
  g_free (name);

  /* åȥɤ߹ */
  name = g_build_filename (skin_path, "gtkrc", NULL);
  if (g_file_exist (name)) {
    gtk_rc_parse (name);
    g_free (name);
  }
  /* Υޥǡ */
  fscanf (fp, "%s", buf);
  name = g_build_filename (skin_path, buf, NULL);
  pbuf = gdk_pixbuf_new_from_file (name, NULL);
  gdk_pixbuf_render_pixmap_and_mask (pbuf, &dummy, &(skin->mask[4]), 255);
  gdk_pixbuf_unref (pbuf);
  gdk_pixmap_unref (dummy);
  g_free (name);

  /* طʥǡ */
  for (n = 0; n < 4; n++) {
    fscanf (fp, "%s", buf);
    name = g_build_filename (skin_path, buf, NULL);
    pbuf = gdk_pixbuf_new_from_file (name, NULL);
    skin->mask[n] = NULL;
    gdk_pixbuf_render_pixmap_and_mask (pbuf,
				       &(skin->background[n]),
				       &(skin->mask[n]), 255);
    gdk_pixbuf_unref (pbuf);
    g_free (name);
  }
  /* ץ쥤ܥ */
  fscanf (fp, "%s %d %d %d %d %d", buf, &cx, &cy, &x, &y, &a);  
  name = g_build_filename (skin_path, buf, NULL);
  skin->play_button = button_new (name, cx, cy, x, y, a, FALSE);
  g_free (name);

  /* Хåܥ */
  fscanf (fp, "%s %d %d %d %d %d", buf, &cx, &cy, &x, &y, &a);
  name = g_build_filename (skin_path, buf, NULL);
  skin->back_button = button_new (name, cx, cy, x, y, a, FALSE);
  g_free (name);

  /* ܥ */
  fscanf (fp, "%s %d %d %d %d %d", buf, &cx, &cy, &x, &y, &a);
  name = g_build_filename (skin_path, buf, NULL);
  skin->next_button = button_new (name, cx, cy, x, y, a, FALSE);
  g_free (name);

  /* ܥ */
  fscanf (fp, "%s %d %d %d %d %d", buf, &cx, &cy, &x, &y, &a);
  name = g_build_filename (skin_path, buf, NULL);
  skin->prev_button = button_new (name, cx, cy, x, y, a, FALSE);
  g_free (name);

  /* ȥåץܥ */
  fscanf (fp, "%s %d %d %d %d %d", buf, &cx, &cy, &x, &y, &a);
  name = g_build_filename (skin_path, buf, NULL);
  skin->stop_button = button_new (name, cx, cy, x, y, a, FALSE);
  g_free (name);

  /* ԡȥܥ */
  fscanf (fp, "%s %d %d %d %d %d", buf, &cx, &cy, &x, &y, &a);
  name = g_build_filename (skin_path, buf, NULL);
  skin->rept_button = button_new (name, cx, cy, x, y, a, FALSE);
  g_free (name);

  /* ܥ */
  fscanf (fp, "%s %d %d %d %d %d", buf, &cx, &cy, &x, &y, &a);
  name = g_build_filename (skin_path, buf, NULL);
  skin->exit_button = button_new (name, cx, cy, x, y, a, FALSE);
  g_free (name);

  /* ԡȥƥ */
  fscanf (fp, "%s %d %d %d %d", buf, &l, &cx, &cy, &a);  
  name = g_build_filename (skin_path, buf, NULL);
  skin->rept_item = item_new (name, l, cx, cy, a);
  g_free (name);

  /* 饤ƥ */
  fscanf (fp, "%s %d %d %d %d", buf, &l, &cx, &cy, &a);
  name = g_build_filename (skin_path, buf, NULL);
  slider_item = item_new (name, l, cx, cy, a);
  g_free (name);

  /* 饤 */
  fscanf (fp, "%s %s %s %s %d %d %d %d",
	  buf, b1, b2, b3, &l, &x, &y, &a);
  name = g_build_filename (skin_path, buf, NULL);
  skin->slider = slider_new (name,
			     slider_item,
			     (strcmp (b1, "FALSE") == 0) ? 0 : 1,
			     (strcmp (b2, "FALSE") == 0) ? 0 : 1,
			     (strcmp (b3, "FALSE") == 0) ? 0 : 1,
			     l, x, y, x, y, a);
  g_free (name);

  /* ե */
  fscanf (fp, "%s %d %d %d %d %d %d", buf, &l, &cx, &cy, &x, &y, &a);
  name = g_build_filename (skin_path, buf, NULL);
  skin->font = font_new (name, l, cx, cy, x, y, a);
  g_free (name);

  /* ǥ */
  fscanf (fp, "%s %d %d %d", buf, &cx, &cy, &a);
  name = g_build_filename (skin_path, buf, NULL);
  skin->digit = digit_new (name, cx, cy, a);
  g_free (name);

  /* 泰ΰ */
  fscanf (fp, "%d %d %d %d",
	  &(skin->cx), &(skin->cy), &(skin->cw), &(skin->ch));

  g_free (skin_path);
  fclose (fp);

  return skin;
}

/* ************************************************************************* *
 * Хؿ
 *
 * player_new
 * player_sync_to_skin
 * ************************************************************************* */

/* ץ졼 ********************************************************** */
GtkWidget*
player_new (gchar	*skin_data) {
  GtkWidget 	*window;
  GtkWidget 	*table;
  GtkWidget 	*hbox;
  GtkWidget 	*hscrollbar;
  GtkWidget 	*spinbutton;
  GtkWidget 	*canvas;
  GtkObject 	*hscrollbar_adj;
  GtkObject 	*spinbutton_adj;
  SkinData	*skin;
  int		length;

  /* ǡɤ߹ */
  skin = player_skin_new (skin_data);
  if (!skin) return NULL;
  
  /* ɥ */
  window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_widget_ref (window);
  gtk_widget_set_name (window, "player_widget");
  gtk_container_border_width (GTK_CONTAINER (window), 0);
  gtk_window_set_policy (GTK_WINDOW (window), FALSE, FALSE, TRUE);
  gtk_widget_set_events (window,
			 GDK_ENTER_NOTIFY_MASK | 
			 GDK_LEAVE_NOTIFY_MASK |
			 GDK_POINTER_MOTION_MASK |
			 GDK_BUTTON_PRESS_MASK |
			 GDK_BUTTON_RELEASE_MASK);

  gtk_signal_connect (GTK_OBJECT (window), "destroy",
		      (GtkSignalFunc) exit_player, NULL);
  gtk_signal_connect (GTK_OBJECT (window), "delete_event",
		      (GtkSignalFunc) exit_player, NULL);
  gtk_signal_connect (GTK_OBJECT (window), "motion_notify_event",
		      (GtkSignalFunc) player_mouse_motion, NULL);
  gtk_signal_connect (GTK_OBJECT (window), "button_press_event",
		      (GtkSignalFunc) player_mouse_pressed, NULL);
  gtk_signal_connect (GTK_OBJECT (window), "button_release_event",
		      (GtkSignalFunc) player_mouse_released, NULL);
  gtk_signal_connect (GTK_OBJECT (window), "leave_notify_event",
		      (GtkSignalFunc) player_mouse_leave, NULL);

  gtk_widget_realize (window);
  gdk_window_set_decorations (window->window, FALSE);

  /* ơ֥ */
  table = gtk_table_new (3, 3, FALSE);
  gtk_widget_show (table);
  gtk_container_add (GTK_CONTAINER (window), table);
  G_SET_WIDGET (window, "table", table);

  /* ʿBOX */
  hbox = gtk_hbox_new (FALSE, 0);
  gtk_widget_show (hbox);
  gtk_table_attach (GTK_TABLE (table), hbox, 1, 2, 1, 2,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (GTK_FILL), 0, 0);
  G_SET_WIDGET (window, "hbox", hbox);
  {
    /* С */
    length = g_list_length (g_list_first (image_list));
    hscrollbar_adj = gtk_adjustment_new (0, 0, length, 1,
					 (length > 10) ? 10 : 1, 1);
    hscrollbar = gtk_hscrollbar_new (GTK_ADJUSTMENT (hscrollbar_adj));
    gtk_widget_show (hscrollbar);
    gtk_box_pack_start (GTK_BOX (hbox), hscrollbar, TRUE, TRUE, 0);
    G_SET_OBJECT (window, "scrollbar_adj", hscrollbar_adj);
    G_SET_WIDGET (window, "scrollbar",     hscrollbar);

    /* ԥܥ */
    spinbutton_adj = gtk_adjustment_new (0, 0, length-1, 1, 10, 0);
    spinbutton = gtk_spin_button_new (GTK_ADJUSTMENT (spinbutton_adj), 1, 0);
    gtk_widget_show (spinbutton);
    gtk_box_pack_start (GTK_BOX (hbox), spinbutton, FALSE, FALSE, 0);
    gtk_signal_connect (GTK_OBJECT (hscrollbar_adj), "value_changed",
			GTK_SIGNAL_FUNC (player_scrollbar_changed),
			(gpointer) hscrollbar_adj);

    gtk_signal_connect (GTK_OBJECT (spinbutton_adj), "value_changed",
			GTK_SIGNAL_FUNC (player_spinbutton_changed),
			(gpointer) spinbutton_adj);
    G_SET_OBJECT (window, "spinbutton_adj", spinbutton_adj);
    G_SET_WIDGET (window, "spinbutton",     spinbutton);
  }
  /* ǡѥåȤ */
  canvas = gtk_drawing_area_new ();
  gtk_widget_show (canvas);
  gtk_table_attach (GTK_TABLE (table), canvas, 0, 1, 0, 3,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), 0, 0);
  G_SET_WIDGET (window, "canvas0", canvas);

  canvas = gtk_drawing_area_new ();
  gtk_widget_show (canvas);
  gtk_table_attach (GTK_TABLE (table), canvas, 2, 3, 0, 3,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_FILL), 0, 0);
  G_SET_WIDGET (window, "canvas1", canvas);

  canvas = gtk_drawing_area_new ();
  gtk_widget_show (canvas);
  gtk_table_attach (GTK_TABLE (table), canvas, 1, 2, 0, 1,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (GTK_FILL), 0, 0);
  G_SET_WIDGET (window, "canvas2", canvas);

  canvas = gtk_drawing_area_new ();
  gtk_widget_show (canvas);
  gtk_table_attach (GTK_TABLE (table), canvas, 1, 2, 2, 3,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_FILL), 0, 0);
  G_SET_WIDGET (window, "canvas3", canvas);

  G_SET_PARAMETER (window, "skin",       skin);
  G_SET_PARAMETER (window, "move_flg",   0);
  G_SET_PARAMETER (window, "rept_flg",	 0);
  G_SET_PARAMETER (window, "move_x_pos", 0);
  G_SET_PARAMETER (window, "move_y_pos", 0);
  G_SET_PARAMETER (window, "pos_x",	 0);
  G_SET_PARAMETER (window, "pos_y",	 0);

  return window;
}

/* ԥåޥåפ ********************************************** */
void
player_sync_to_skin (GtkWidget	*window) {
  GtkWidget	*widget;
  SkinData	*skin;
  char		name[64];
  int		w, h, n;

  skin = (SkinData *) G_GET_PARAMETER (window, "skin");

  /* ΤΥޥǡΥå */
  gdk_window_get_size (skin->mask[4], &w, &h);
  gtk_widget_set_size_request (window, w, h);
  gtk_widget_shape_combine_mask (window, skin->mask[4], 0, 0);

  /* ǡ */
  for (n = 0; n < 4; n++) {
    sprintf (name, "canvas%d", n);
    widget = G_GET_WIDGET (window, name);
    gdk_window_get_size (skin->background[n], &w, &h);
    gtk_widget_set_size_request (widget, w, h);
    gtk_drawing_area_size (GTK_DRAWING_AREA (widget), w, h);
  
    gtk_widget_shape_combine_mask (widget, skin->mask[n], 0, 0);
    gdk_window_set_back_pixmap (widget->window, skin->background[n], FALSE);
    gdk_window_clear (widget->window);
  }
}

/* ץ졼Υåץǡ ************************************************** */
void
player_update_window (void) {
  SkinData	*skin = NULL;

  skin = (SkinData *) G_GET_PARAMETER (player, "skin");
  if (skin) {
    if (!skin->font->text) {
      set_font_text (player, ti_get_filename (image_list));
    }
    update_font (player);
    draw_digit (player, skin->digit, play_speed);
  }
}

/* ***************************************************** End of player.c *** */
