#ifndef _TMVECTOR_HPP
#define _TMVECTOR_HPP
/** 
	@file vector.hpp
	@brief Defines tempest::vector<T,Sz>.
	@author ototoi / Toru Matsuoka
	@date 2004/05/15 
*/

//-----------------------------------------------
#include<cstddef>//std::size_t
#include<cmath>//std::sqrt
#include<algorithm>//copy
#include<stdexcept>
#include<cassert>

#include"vector_base.hpp" //For static polimorphizm.




//_TEMPEST_USE_META_
//_TEMPEST_USE_FASTMATH_

#ifdef _TEMPEST_USE_FASTMATH_
#include"../util/math.hpp"
namespace tempest{
	template<class T>
	inline T RSQRT(T x){return rsqrt(x);}
}
#else
namespace tempest{
	template<class T>
	inline T RSQRT(T x){return T(1)/std::sqrt(x);}
}
#endif


namespace tempest{
	
template <std::size_t i>
struct node_loop;
	
template <std::size_t i>
struct node_loop{
	template<class P, class Q>
	inline static void eval(P & lhs, const Q & rhs){
		lhs[i] = rhs[i];
		node_loop<i-1>::eval(lhs,rhs);
	}
};
	
template <>
struct node_loop<0>{
	template<class P, class Q>
	inline static void eval(P & lhs, const Q & rhs){
		lhs[0] = rhs[0];
	}
};
	
	/*
	template<> class vector<float,2>;
	template<> class vector<double,2>;
	template<> class vector<long double,2>;

	template<> class vector<float,3>;
	template<> class vector<double,3>;
	template<> class vector<long double,3>;
	
	template<> class vector<float,4>;
	template<> class vector<double,4>;
	template<> class vector<long double,4>;
	*/
	
//
template<class Self, class T, std::size_t Sz>struct vector_expression;//pre
template <std::size_t i>struct node_loop;
	
template<class T,std::size_t Sz>class vector;//Predacal.
	
/** 
 * @class vector
 * @brief Vector class template. 
 * @todo	
 * @code
 * 	tempest::vector<double,3> v;	//3D vector class  v
 * @endcode
 * @bug
 */

	
template<class T,std::size_t Sz>
class vector:public vector_base< vector<T,Sz>, T, Sz >{	
	//template<class,std::size_t> friend class vector	
public:
	//-----------------------------------------------
	//type define

	typedef		T										value_type;		/**< type of element*/
	typedef		T&										reference;		/**< reference*/					
	typedef		const T&								const_reference;/**< const reference*/
	typedef		vector<T,Sz>							this_type;		/**< self*/

	typedef		std::size_t								size_type;		/**< type of element size*/
	typedef		std::ptrdiff_t							difference_type;/**< difference type of pointer*/
	
	typedef		T*										iterator;		/**< iterator*/
	typedef		const T*								const_iterator;	/**< const iterator*/

public:
	static const size_type c_size = Sz;//container size
	
public:
	//-----------------------------------------------
	//function for iterator
	/** 
	 * @name function for iterator
	 */
	//@{ 
	iterator		begin()			{return element.begin();}	/**< Return iterator for first element. */	 
	iterator		end()			{return element.end();}		/**< Return iterator for last element.  */
	const_iterator	begin()	const	{return element.begin();}	/**< Return iterator for first element. */	 
	const_iterator	end()	const	{return element.end();}		/**< Return iterator for last element.  */
	//@}
	
	//-----------------------------------------------
	//front&back
	/**
	 * @name front&back
	 */
	//@{
	value_type front()				{ return element[0]; }
	value_type back()				{ return element[c_size-1]; }
	const_reference front() const 	{ return element[0]; }
	const_reference back()  const 	{ return element[c_size-1]; }
	//@}
	
	//-----------------------------------------------
	//capacity
	/** 
	 * @name size & capacity
	 */
	//@{ 
	size_type size ()	  const { return Sz; }					/**< Return size.			*/					
	size_type max_size () const { return Sz; }					/**< Return size.			*/
	bool	  empty ()	  const { return false; }				/**< Return true always.  	*/
	//@}
	
	//-----------------------------------------------
	//constructor & destructor
	/** 
	 * @name constructor & destructor
	 */
	//@{
	vector(){}
	
	vector(const vector<T,Sz> & rhs)
		//:element(rhs.element)//std::copy(rhs.begin(),rhs.end(),begin());
	{
		node_loop<Sz-1>::eval(element,rhs);
	}
	
	template<class X>
	explicit vector(const vector<X,Sz> & rhs){
		std::copy(rhs.begin(),rhs.end(),begin());
	}
		
	template<class X>
	explicit vector(const X * rhs){
		std::copy(&(rhs[0]),&(rhs[Sz]),begin());
	}
		
	template<class Self_X,class Type_X>
	explicit vector(const vector_expression<Self_X,Type_X,Sz> & rhs){
		node_loop<Sz-1>::eval(*this,static_cast<const Self_X &>(rhs));
		//for (std::size_t i = 0;i<=Sz;i++){(*this)[i] = static_cast<T>(static_cast<const Self_X &>(rhs)[i]);}		
	}
		
	template<class Self,class X>
	explicit vector(const vector_base<Self,X,Sz> & rhs){
		std::copy(static_cast<const Self &>(rhs).begin(),static_cast<const Self &>(rhs).end(),begin());
	}
		
	~vector(){}
	//@}
	
	//swap :-)
	//void swap(this_type& rhs){std::swap(m,rhs.m);}

	
	
	//-----------------------------------------------
	//substitutioner
	/** 
	 * @name substitutioner
	 */
	//@{ 
	this_type& operator= (const this_type &rhs){
		element = rhs.element;
		return *this;
	}	
	template<class X>
	this_type& operator= (const vector<X,Sz> &rhs){
		std::copy(rhs.begin(),rhs.end(),begin());	
		return *this;
	}
		
	template<class Self_X,class Type_X>
	this_type& operator= (const vector_expression<Self_X,Type_X,Sz> & rhs){	
		node_loop<Sz-1>::eval(*this,static_cast<const Self_X &>(rhs));
		//for (std::size_t i = 0;i<=Sz;i++){(*this)[i] = static_cast<T>(static_cast<const Self_X &>(rhs)[i]);}		
		return *this;
	}
		
	template<class Self,class X>
	this_type& operator= (const vector_base<Self,X,Sz> & rhs){
		std::copy(static_cast<const Self &>(rhs).begin(),static_cast<const Self &>(rhs).end(),begin());
		return *this;
	}
	
	template<class IT>
	void assign( IT start, IT end ){
		assert(std::distance(start,end)<=c_size);//debug
		
		std::copy(start,end,begin());				
	}
	
	void assign( size_type num, T val ){
		std::fill_n(begin(),(num<c_size)?num:c_size,val);		
	}

		
	//@}
	
	//-----------------------------------------------
	//operator
	/** 
	 * @name operator
	 */
	//@{
		
	//--------------------------------
	this_type& negate (){
		for(size_type i=0;i<Sz;i++){element[i] = -element[i];}
		return *this;
	}
		
	//--------------------------------
#define DECLARE_OPERATOR_EQ( OP )							\
template<class X>											\
this_type& operator OP (const vector<X,Sz>& rhs){			\
	for(size_type i=0;i<Sz;i++){element[i] OP rhs[i];}		\
	return *this;											\
}
	
	DECLARE_OPERATOR_EQ(+=)
	DECLARE_OPERATOR_EQ(-=)
	DECLARE_OPERATOR_EQ(*=)
	DECLARE_OPERATOR_EQ(/=)
		
#undef DECLARE_OPERATOR_EQ
		
	
	//--------------------------------
	template<class X>
	this_type& operator*= (const X rhs){
		for(size_type i=0;i<Sz;i++){element[i] *= rhs;}
		return *this;
	}
		
	template<class X>
	this_type& operator/= (const X rhs){
		for(size_type i=0;i<Sz;i++){element[i] /= rhs;}
		return *this;
	}
		
	//--------------------------------
	//@}
	
	//-----------------------------------------------
	//indexer
	/** 
	 * @name indexer
	 */
	//@{ 	
	reference operator[](size_type i){
		return element[i];
	}
	
	const value_type operator[](size_type i) const {
		return element[i];
	}
	
	reference at(size_type i){
		if(Sz<=i){throw std::out_of_range("tempest::vector");}
		return element[i];
	}
	const_reference at(size_type i) const {
		if(Sz<=i){throw std::out_of_range("tempest::vector");}
		return element[i];
	}
	//@}
		
	//-----------------------------------------------
	//utility
	/** 
	 * @name utility
	 */
	//@{ 
	value_type length() const {
		return std::sqrt( sqr_length() );
	}
		
	value_type sqr_length() const {
		T temp=T();
		for(size_type i=0;i<Sz;i++){ temp += element[i] * element[i]; }
		return temp;
	}
		
	value_type sum() const {
		T temp=T();
		for(size_type i=0;i<Sz;i++){ temp += element[i]; }
		return temp;
	}
		
	this_type& normalize(){
		T length = sqr_length();		//||V||^2
		if (length == T()) return *this;
		
		length = RSQRT(length);
		
		for(size_type i=0;i<Sz;i++){ element[i] *= length; }

		return *this;
	}
	//@}
	
	/** 
	 *  @name others
 	 */
	//@{ 
	const char* debug()const{return "tempest::vector<T,Sz>"; }
	//@}
	
private:	
	struct ___m{
		T m[c_size];
		
		T*			begin()			{return m;}
		T*			end()			{return m+c_size;}
		const T*	begin()	const	{return m;}
		const T*	end()	const	{return m+c_size;}
		T & operator[](size_type i)		{return m[i];}
		const T   operator[](size_type i)const{return m[i];}
	}element;
	
};//End of class.

}//End of namespace.
	
#include "vector_functions.hpp"

#endif

