#ifndef _TMMATRIX_GENERATOR_HPP
#define _TMMATRIX_GENERATOR_HPP
/** 
	@file matrix_generator.hpp
	@brief Defines class template tempest::matrix_generator<T>,which is helper for generating matrix.
	@author ototoi / Toru Matsuoka
	@date 2004/05/15 
*/
#include<cmath>
#include<cstddef>

namespace std{
	template<class T>
	class complex;	
}

namespace tempest{
	
template<class T,std::size_t Sz>
class vector;  
template<class T,std::size_t RowSz,std::size_t ColumnSz>
class matrix;  
template<class T>
class quaternion;


/** 
 *  @struct matrix_generator
 *  @brief  Helper class template for matrix generation.
 *  
 *  @todo 
 *  @code
 *  typedef tempest::matrix<double,2> mat2;                         //matrix<double,2>
 *  typedef tempest::matrix_generator< mat2 > mgen2;  //generator for matrix<double,2>
 *  mat2 m(gen2::scaling(1.5,1.5));
 *  @endcode    
 *  @bug
 */
	
/** 
 *  @struct matrix_generator< matrix<T, Sz> >
 *  @brief  Helper class template for matrix generation.
 *  
 *  @todo 
 *  @code
 *  typedef tempest::matrix<double,10> mat10;                         //matrix<double,2>
 *  typedef tempest::matrix_generator< mat10 > mgen10;  //generator for matrix<double,2>
 *  mat10 m(gen10::zero());								//0
 *  @endcode    
 *  @bug
 */
	
/** 
 *  @struct matrix_generator< matrix<T,2,2> >
 *  @brief  Helper class template for matrix generation.
 *  
 *  @todo 
 *  @code
 *  typedef tempest::matrix<double,2> mat2;                         //matrix<double,2>
 *  typedef tempest::matrix_generator< mat2 > mgen2;  //generator for matrix<double,2>
 *  mat2 m(gen2::scaling(1.5,1.5));
 *  @endcode    
 *  @bug
 */
	
/** 
 *  @struct matrix_generator< matrix<T,3,3> >
 *  @brief  Helper class template for matrix generation.
 *  
 *  @todo 
 *  @code
 *  typedef tempest::matrix<double,3> mat3;                         //matrix<double,2>
 *  typedef tempest::matrix_generator< mat3 > mgen3;  //generator for matrix<double,2>
 *  mat3 m(gen2::scaling(1.5,1.5));
 *  @endcode    
 *  @bug
 */
	
/** 
 *  @struct matrix_generator< matrix<T,4,4> >
 *  @brief  Helper class template for matrix generation.
 *  
 *  @todo 
 *  @code
 *  typedef tempest::matrix<double,4> mat4;                         //matrix<double,2>
 *  typedef tempest::matrix_generator< mat4 > mgen4;  //generator for matrix<double,2>
 *  mat4 m(gen2::scaling(1.5,1.5));
 *  @endcode    
 *  @bug
 */

	
template<class T>
struct matrix_generator{};
	
template<class T, std::size_t Sz>
struct matrix_generator< matrix<T,Sz> >{
	static matrix<T,Sz> zero(){
		matrix<T,Sz> temp;
		for(std::size_t i = 0;i<Sz;i++){
			for(std::size_t j = 0;j<Sz;j++){
				temp[i][j] = T();
			}
		}
		
		return temp;
	}
	
	static matrix<T,Sz> identity(){
		matrix<T,Sz> temp;
		for(std::size_t i = 0;i<Sz;i++){
			for(std::size_t j = 0;j<Sz;j++){
					if(i != j){temp[i][j] = T();}
					else{temp[i][j] = T(1);}
			}
		}
		
		return temp;
	}	
};

//-------------------------------------------------------------------
//For "matrix2"
template<class T>
struct matrix_generator< matrix<T,2,2> >{
	inline static matrix<T,2,2> zero(){
		return matrix<T,2,2>(0,0,0,0);
	}
	
	inline static matrix<T,2,2> identity(){
		return matrix<T,2,2>(1,0,0,1);
	}

	template<class X>
	inline static matrix<T,2,2> scaling(const X _x,const X _y){
		return matrix<T,2,2>(_x,0,0,_y);
	}

	template<class X>
	inline static matrix<T,2,2> rotation_z(const X sin,const X cos){
		return matrix<T,2,2>(cos,-sin,sin,cos);
	}

	template<class X>
	inline static matrix<T,2,2> rotation_z(const X radian){
		T _sin = std::sin(radian);
		T _cos = std::cos(radian);
		
		return matrix<T,2,2>(_cos,-_sin,_sin,_cos);
	}
	
	template<class X>
	inline static matrix<T,2,2> convert(const std::complex<X> & rhs){
		//T _sin = std::sin(radian);
		//T _cos = std::cos(radian);
		T _r = rhs.real();
		T _i = rhs.imag();
		
		return matrix<T,2,2>(
			_r,-_i,
			_i,_r
		);
	}
};
	

//-------------------------------------------------------------------
//For matrix3
template<class T>
struct matrix_generator< matrix<T,3,3> >{
	
	inline static matrix<T,3,3> zero(){
		return
			matrix<T,3,3>(
				0,0,0,
				0,0,0,
				0,0,0
			);
	}
	
	inline static matrix<T,3,3> identity(){
		return
			matrix<T,3,3>(
				1,0,0,
				0,1,0,
				0,0,1
			);
	}

	template<class X>
	inline static matrix<T,3,3> translation(const X tx,const X ty){
		return
			matrix<T,3,3>(
				1,0,tx,
				0,1,ty,
				0,0,1
			);
	}

	template<class X>
	inline static matrix<T,3,3> scaling(const X sx,const X sy,const X sz){
		return
			matrix<T,3,3>(
				sx,0,0,
				0,sy,0,
				0,0,sz
			);
	}

	template<class X>
	inline static matrix<T,3,3> rotation_z(const X sin,const X cos){
		return
			matrix<T,3,3>(
				cos,-sin,0,
				sin,cos,0,
				0,0,1
			);
	}

	template<class X>
	inline static matrix<T,3,3> rotation_z(const X radian){
		T _sin = std::sin(radian);
		T _cos = std::cos(radian);
		return rotation_z(_sin,_cos);
	}
//-----------------------------------------------
	template<class X>
	inline static matrix<T,3,3> rotation_x(const X sin,const X cos){
		return
			matrix<T,3,3>(
				1,0,0,
				0,cos,-sin,
				0,sin,cos
			);
	}

	template<class X>
	inline static matrix<T,3,3> rotation_x(const X radian){
		T _sin = std::sin(radian);
		T _cos = std::cos(radian);
		return rotation_x(_sin,_cos);
	}

//-----------------------------------------------
	template<class X>
	inline static matrix<T,3,3> rotation_y(const X sin,const X cos){
		return
			matrix<T,3,3>(
				cos,0,sin,
				0,1,0,
				-sin,0,cos
			);
	}

	template<class X>
	inline static matrix<T,3,3> rotation_y(const X radian){
		T _sin = std::sin(radian);
		T _cos = std::cos(radian);
		return rotation_y(_sin,_cos);
	}

	template<class X>
	static matrix<T,3,3> convert(const quaternion<X>& rhs){
		//4C2 = 4*5/2*1 = 10
		//tww = 1;
		//10 - 1 = 9
		//T tww = 2*rhs[0]*rhs[0];//0
		T tx2 = rhs[1]*2;
		T ty2 = rhs[2]*2;
		T tz2 = rhs[3]*2;

		T twx = static_cast<T>(tx2*rhs[0]);//1
		T twy = static_cast<T>(ty2*rhs[0]);//2
		T twz = static_cast<T>(tz2*rhs[0]);//3
		
		T txx = static_cast<T>(tx2*rhs[1]);//4
		T txy = static_cast<T>(ty2*rhs[1]);//5
		T txz = static_cast<T>(tz2*rhs[1]);//6
		
		T tyy = static_cast<T>(ty2*rhs[2]);//7
		T tyz = static_cast<T>(tz2*rhs[2]);//8

		T tzz = static_cast<T>(tz2*rhs[3]);//9
		
		return
			matrix<T,3,3>(
				1 - tyy - tzz,	txy - twz,		txz + twy,
				txy + twz,		1 - txx - tzz,	tyz - twx,
				txz - twy,		tyz + twx,		1 - txx -tyy
			);		
	}

	template<class X>
	inline static matrix<T,3,3> world2camera(const vector<X,3> & camera_pos,const vector<X,3> & target_pos,const vector<X,3> & camera_up){
		typedef vector<X,3> Vector;
		typedef matrix<T,3,3> Matrix;
		
		//Matrix T ( translation(-camera_pos.x,-camera_pos.y,-camera_pos.z) );
		Vector nz = normalize(target_pos-camera_pos);                  	//z
    	Vector ny = normalize(camera_up - (dot(nz,camera_up) * nz)); 	//y
    	Vector nx = cross(nz,ny);                        				//x
    
		//Matrix R(nx[0],nx[1],nx[2],0,  ny[0],ny[1],ny[2],0,  nz[0],nz[1],nz[2],0,  0,0,0,1);
        
		//return R*T;
    	return Matrix(nx[0],nx[1],nx[2],  ny[0],ny[1],ny[2],  nz[0],nz[1],nz[2]) 
				* translation(-camera_pos[0],-camera_pos[1],-camera_pos[2]);		
	}
	
};

//-------------------------------------------------------------------
//-------------------------------------------------------------------
//For matrix4
	
template<class T>
struct matrix_generator< matrix<T,4,4> >{
	//typedef typename matrix_generator< matrix<T,4,4> > this_type;

	inline static matrix<T,4,4> zero(){
		return
			matrix<T,4,4>(
				0,0,0,0,
				0,0,0,0,
				0,0,0,0,
				0,0,0,0
			);
	}
	
	inline static matrix<T,4,4> identity(){
		return
			matrix<T,4,4>(
				1,0,0,0,
				0,1,0,0,
				0,0,1,0,
				0,0,0,1
			);
	}

	template<class X>
	inline static matrix<T,4,4> translation(const X tx,const X ty,const X tz){
		return
			matrix<T,4,4>(
				1,0,0,tx,
				0,1,0,ty,
				0,0,1,tz,
				0,0,0,1
			);
	}

	template<class X>
	inline static matrix<T,4,4> scaling(const X sx,const X sy,const X sz){
		return
			matrix<T,4,4>(
				sx,0,0,0,
				0,sy,0,0,
				0,0,sz,0,
				0,0,0,1
			);
	}

	template<class X>
	inline static matrix<T,4,4> rotation_z(const X sin,const X cos){
		return
			matrix<T,4,4>(
				cos,-sin,0,0,
				sin, cos,0,0,
				  0,   0,1,0,
				  0,   0,0,1
			);
	}

	template<class X>
	inline static matrix<T,4,4> rotation_z(const X radian){
		T _sin = std::sin(radian);
		T _cos = std::cos(radian);
		return rotation_z(_sin,_cos);
	}
//-----------------------------------------------
	template<class X>
	inline static matrix<T,4,4> rotation_x(const X sin,const X cos){
		return
			matrix<T,4,4>(
				1,0,0,0,
				0,cos,-sin,0,
				0,sin,cos,0,
				0,0,0,1
			);
	}

	template<class X>
	inline static matrix<T,4,4> rotation_x(const X radian){
		T _sin = std::sin(radian);
		T _cos = std::cos(radian);
		return rotation_x(_sin,_cos);
	}

//-----------------------------------------------
	template<class X>
	inline static matrix<T,4,4> rotation_y(const X sin,const X cos){
		return
			matrix<T,4,4>(
				cos, 0, sin,0,
				0,   1,   0,0,
				-sin,0, cos,0,
				0,   0,   0,1
			);
	}

	template<class X>
	inline static matrix<T,4,4> rotation_y(const X radian){
		T _sin = std::sin(radian);
		T _cos = std::cos(radian);
		return rotation_y(_sin,_cos);
	}
	

	template<class X>
	inline static matrix<T,4,4> world2camera(const vector<X,3> & camera_pos,const vector<X,3> & target_pos,const vector<X,3> & camera_up){
		typedef vector<X,3> Vector;
		typedef matrix<T,4,4> Matrix;
		
		//Matrix T ( translation(-camera_pos.x,-camera_pos.y,-camera_pos.z) );
		Vector nz = normalize(target_pos-camera_pos);                  	//z
    	Vector ny = normalize(camera_up - (dot(nz,camera_up) * nz)); 	//y
    	Vector nx = cross(nz,ny);                        				//x
    
		//Matrix R(nx[0],nx[1],nx[2],0,  ny[0],ny[1],ny[2],0,  nz[0],nz[1],nz[2],0,  0,0,0,1);
        
		//return R*T;
    	return Matrix(nx[0],nx[1],nx[2],0,  ny[0],ny[1],ny[2],0,  nz[0],nz[1],nz[2],0,  0,0,0,1) 
				* translation(-camera_pos[0],-camera_pos[1],-camera_pos[2]);		
	}
	
	template<class X>
	inline static matrix<T,4,4> convert(const quaternion<X>& rhs){
		
		//4C2 = 4*5/2*1 = 10
		//tww = 1;
		//10 - 1 = 9
		//T tww = 2*rhs[0]*rhs[0];//0
		T tx2 = rhs[1]*2;
		T ty2 = rhs[2]*2;
		T tz2 = rhs[3]*2;

		T twx = static_cast<T>(tx2*rhs[0]);//1
		T twy = static_cast<T>(ty2*rhs[0]);//2
		T twz = static_cast<T>(tz2*rhs[0]);//3
		
		T txx = static_cast<T>(tx2*rhs[1]);//4
		T txy = static_cast<T>(ty2*rhs[1]);//5
		T txz = static_cast<T>(tz2*rhs[1]);//6
		
		T tyy = static_cast<T>(ty2*rhs[2]);//7
		T tyz = static_cast<T>(tz2*rhs[2]);//8

		T tzz = static_cast<T>(tz2*rhs[3]);//9
		
		return
			matrix<T,4,4>(
				1 - tyy - tzz,	txy - twz,		txz + twy,		0,
				txy + twz,		1 - txx - tzz,	tyz - twx,		0,
				txz - twy,		tyz + twx,		1 - txx -tyy,	0,
				0,				0,				0,				1
			);		
	}

	
};

}//end of namespace tempest;
#endif
