#ifndef _TMMATRIX_GENERATOR_HPP
#define _TMMATRIX_GENERATOR_HPP
/** 
	@file matrix_generator.hpp
	@brief Defines class template tempest::matrix_generator<T>,which is helper for generating matrix.
	@author ototoi / Toru Matsuoka
	@date 2004/05/15 
*/
#include<cmath>
#include<cstddef>

namespace tempest{
	
template<class T,std::size_t Sz>
class vector;  
template<class T,std::size_t RowSz,std::size_t ColumnSz>
class matrix;  
template<class T>
class quaternion;
	


/** 
 *	@struct matrix_generator
 *	@brief	Helper class template for matrix generation.
 *	
 *	@todo 
 *	@code
 *		typedef tempest::matrix<double,2> mat2;							//matrix<double,2>
 *		typedef tempest::matrix_generator< tempest::matrix<T,2,2> > mgen2;	//generator for matrix<double,2>
 *		mat2 m(gen2::scaling(1.5,1.5));
 *	@endcode	
 *	@bug
 */

	
template<class T>
struct matrix_generator{};
	
template<class T, std::size_t Sz>
struct matrix_generator< matrix<T,Sz> >{
	static matrix<T,Sz> zero(){
		matrix<T,Sz> temp;
		for(std::size_t i = 0;i<Sz;i++){
			for(std::size_t j = 0;j<Sz;j++){
				temp[i][j] = T();
			}
		}
		
		return temp;
	}
	
	static matrix<T,Sz> identity(){
		matrix<T,Sz> temp;
		for(std::size_t i = 0;i<Sz;i++){
			for(std::size_t j = 0;j<Sz;j++){
					if(i != j){temp[i][j] = T();}
					else{temp[i][j] = T(1);}
			}
		}
		
		return temp;
	}	
};

//-------------------------------------------------------------------
//For "matrix2"
template<class T>
struct matrix_generator< matrix<T,2,2> >{
	inline static matrix<T,2,2> zero(){
		return matrix<T,2,2>(0,0,0,0);
	}
	/**
		@brief identity
	*/
	inline static matrix<T,2,2> identity(){
		return matrix<T,2,2>(1,0,0,1);
	}

	template<class X>
	inline static matrix<T,2,2> scaling(const X _x,const X _y){
		return matrix<T,2,2>(_x,0,0,_y);
	}

	template<class X>
	inline static matrix<T,2,2> rotation_z(const X sin,const X cos){
		return matrix<T,2,2>(cos,-sin,sin,cos);
	}

	template<class X>
	inline static matrix<T,2,2> rotation_z(const X radian){
		X _sin = std::sin(radian);
		X _cos = std::cos(radian);
		
		return matrix<T,2,2>(_cos,-_sin,_sin,_cos);
	}
};
	

//-------------------------------------------------------------------
//For matrix3
template<class T>
struct matrix_generator< matrix<T,3,3> >{
	
	inline static matrix<T,3,3> zero(){
		return
			matrix<T,3,3>(
				0,0,0,
				0,0,0,
				0,0,0
			);
	}
	
	inline static matrix<T,3,3> identity(){
		return
			matrix<T,3,3>(
				1,0,0,
				0,1,0,
				0,0,1
			);
	}

	template<class X>
	inline static matrix<T,3,3> translation(const X tx,const X ty){
		return
			matrix<T,3,3>(
				1,0,tx,
				0,1,ty,
				0,0,1
			);
	}

	template<class X>
	inline static matrix<T,3,3> scaling(const X sx,const X sy,const X sz){
		return
			matrix<T,3,3>(
				sx,0,0,
				0,sy,0,
				0,0,sz
			);
	}

	template<class X>
	inline static matrix<T,3,3> rotation_z(const X sin,const X cos){
		return
			matrix<T,3,3>(
				cos,-sin,0,
				sin,cos,0,
				0,0,1
			);
	}

	template<class X>
	inline static matrix<T,3,3> rotation_z(const X radian){
		X _sin = std::sin(radian);
		X _cos = std::cos(radian);
		return rotation_z(_sin,_cos);
	}
//-----------------------------------------------
	template<class X>
	inline static matrix<T,3,3> rotation_x(const X sin,const X cos){
		return
			matrix<T,3,3>(
				1,0,0,
				0,cos,-sin,
				0,sin,cos
			);
	}

	template<class X>
	inline static matrix<T,3,3> rotation_x(const X radian){
		X _sin = std::sin(radian);
		X _cos = std::cos(radian);
		return rotation_x(_sin,_cos);
	}

//-----------------------------------------------
	template<class X>
	inline static matrix<T,3,3> rotation_y(const X sin,const X cos){
		return
			matrix<T,3,3>(
				cos,0,sin,
				0,1,0,
				-sin,0,cos
			);
	}

	template<class X>
	inline static matrix<T,3,3> rotation_y(const X radian){
		X _sin = std::sin(radian);
		X _cos = std::cos(radian);
		return rotation_y(_sin,_cos);
	}

	template<class X>
	static matrix<T,3,3> q2m(const quaternion<X>& rhs){
		//4C2 = 4*5/2*1 = 10
		//tww = 1;
		//10 - 1 = 9
		//X tww = 2*rhs[0]*rhs[0];//0
		X twx = 2*rhs[0]*rhs[1];//1
		X twy = 2*rhs[0]*rhs[2];//2
		X twz = 2*rhs[0]*rhs[3];//3
		
		X txx = 2*rhs[1]*rhs[1];//4
		X txy = 2*rhs[1]*rhs[2];//5
		X txz = 2*rhs[1]*rhs[3];//6
		
		X tyy = 2*rhs[2]*rhs[2];//7
		X tyz = 2*rhs[2]*rhs[3];//8

		X tzz = 2*rhs[3]*rhs[3];//9
		
		return
			matrix<T,3,3>(
				1 - tyy - tzz,	txy - twz,		txz + twy,
				txy + twz,		1 - txx - tzz,	tyz - twx,
				txz - twy,		tyz + twx,		1 - txx -tyy
			);		
	}

	template<class X>
	inline static matrix<T,3,3> world2camera(const vector<X,3> & camera_pos,const vector<X,3> & target_pos,const vector<X,3> & camera_up){
		typedef vector<X,3> Vector;
		typedef matrix<T,3,3> Matrix;
		
		//Matrix T ( translation(-camera_pos.x,-camera_pos.y,-camera_pos.z) );
		Vector nz = normalize(target_pos-camera_pos);                  	//z
    	Vector ny = normalize(camera_up - (dot(nz,camera_up) * nz)); 	//y
    	Vector nx = cross(nz,ny);                        				//x
    
		//Matrix R(nx[0],nx[1],nx[2],0,  ny[0],ny[1],ny[2],0,  nz[0],nz[1],nz[2],0,  0,0,0,1);
        
		//return R*T;
    	return Matrix(nx[0],nx[1],nx[2],  ny[0],ny[1],ny[2],  nz[0],nz[1],nz[2]) 
				* translation(-camera_pos[0],-camera_pos[1],-camera_pos[2]);		
	}
	
};

//-------------------------------------------------------------------
//-------------------------------------------------------------------
//For matrix4
	
template<class T>
struct matrix_generator< matrix<T,4,4> >{
	//typedef typename matrix_generator< matrix<T,4,4> > this_type;

	inline static matrix<T,4,4> zero(){
		return
			matrix<T,4,4>(
				0,0,0,0,
				0,0,0,0,
				0,0,0,0,
				0,0,0,0
			);
	}
	
	inline static matrix<T,4,4> identity(){
		return
			matrix<T,4,4>(
				1,0,0,0,
				0,1,0,0,
				0,0,1,0,
				0,0,0,1
			);
	}

	template<class X>
	inline static matrix<T,4,4> translation(const X tx,const X ty,const X tz){
		return
			matrix<T,4,4>(
				1,0,0,tx,
				0,1,0,ty,
				0,0,1,tz,
				0,0,0,1
			);
	}

	template<class X>
	inline static matrix<T,4,4> scaling(const X sx,const X sy,const X sz){
		return
			matrix<T,4,4>(
				sx,0,0,0,
				0,sy,0,0,
				0,0,sz,0,
				0,0,0,1
			);
	}

	template<class X>
	inline static matrix<T,4,4> rotation_z(const X sin,const X cos){
		return
			matrix<T,4,4>(
				cos,-sin,0,0,
				sin,cos,0,0,
				0,0,1,0,
				0,0,0,1
			);
	}

	template<class X>
	inline static matrix<T,4,4> rotation_z(const X radian){
		X _sin = std::sin(radian);
		X _cos = std::cos(radian);
		return rotation_z(_sin,_cos);
	}
//-----------------------------------------------
	template<class X>
	inline static matrix<T,4,4> rotation_x(const X sin,const X cos){
		return
			matrix<T,4,4>(
				1,0,0,0,
				0,cos,-sin,0,
				0,sin,cos,0,
				0,0,0,1
			);
	}

	template<class X>
	inline static matrix<T,4,4> rotation_x(const X radian){
		X _sin = std::sin(radian);
		X _cos = std::cos(radian);
		return rotation_x(_sin,_cos);
	}

//-----------------------------------------------
	template<class X>
	inline static matrix<T,4,4> rotation_y(const X sin,const X cos){
		return
			matrix<T,4,4>(
				cos,0,sin,0,
				0,1,0,0,
				-sin,0,cos,0,
				0,0,0,1
			);
	}

	template<class X>
	inline static matrix<T,4,4> rotation_y(const X radian){
		X _sin = std::sin(radian);
		X _cos = std::cos(radian);
		return rotation_y(_sin,_cos);
	}
	/*
	template<class X, class V>
	inline static matrix<T,4,4> rotation(V x, V y, V z, X radian){
		return zero();
	}
	
	template<class X, class V>
	inline static matrix<T,4,4> rotation(const vector<V,3> & v, X radian){
		return rotation(v[0],v[1],v[2],radian);
	}
	
	template<class X>
	inline static matrix<T,4,4> rotation_ypr(X yew, X pitch, X roll);
	
	
	
	template<class X>
	inline static matrix<T,4,4> ortho(X width, X height){
		typedef matrix<T,4,4> Matrix;
		return Scaling(T(1)/width,T(1)/height,T(1));	
		
	}
	
	template<class X>
	inline static matrix<T,4,4> perspective_fov(X fov, X ratio, X near, X far){
		typedef matrix<T,4,4> Matrix;
		//y = y * z * std::tan(fov);
		//x = x * ratio * z * std::tan(fov);
		return Scaling(T(1)/width,T(1)/height,T(1));	
		
	}
	
	
	template<class X>
	inline static matrix<T,4,4> reflection(const vector<V,4> & plane);
	
	
	
	*/

	template<class X>
	inline static matrix<T,4,4> world2camera(const vector<X,3> & camera_pos,const vector<X,3> & target_pos,const vector<X,3> & camera_up){
		typedef vector<X,3> Vector;
		typedef matrix<T,4,4> Matrix;
		
		//Matrix T ( translation(-camera_pos.x,-camera_pos.y,-camera_pos.z) );
		Vector nz = normalize(target_pos-camera_pos);                  	//z
    	Vector ny = normalize(camera_up - (dot(nz,camera_up) * nz)); 	//y
    	Vector nx = cross(nz,ny);                        				//x
    
		//Matrix R(nx[0],nx[1],nx[2],0,  ny[0],ny[1],ny[2],0,  nz[0],nz[1],nz[2],0,  0,0,0,1);
        
		//return R*T;
    	return Matrix(nx[0],nx[1],nx[2],0,  ny[0],ny[1],ny[2],0,  nz[0],nz[1],nz[2],0,  0,0,0,1) 
				* translation(-camera_pos[0],-camera_pos[1],-camera_pos[2]);		
	}
	
	template<class X>
	inline static matrix<T,4,4> q2m(const quaternion<X>& rhs){
		
		//4C2 = 4*5/2*1 = 10
		//tww = 1;
		//10 - 1 = 9
		//X tww = 2*rhs[0]*rhs[0];//0
		X twx = 2*rhs[0]*rhs[1];//1
		X twy = 2*rhs[0]*rhs[2];//2
		X twz = 2*rhs[0]*rhs[3];//3
		
		X txx = 2*rhs[1]*rhs[1];//4
		X txy = 2*rhs[1]*rhs[2];//5
		X txz = 2*rhs[1]*rhs[3];//6
		
		X tyy = 2*rhs[2]*rhs[2];//7
		X tyz = 2*rhs[2]*rhs[3];//8

		X tzz = 2*rhs[3]*rhs[3];//9
		
		return
			matrix<T,4,4>(
				1 - tyy - tzz,	txy - twz,		txz + twy,		0,
				txy + twz,		1 - txx - tzz,	tyz - twx,		0,
				txz - twy,		tyz + twx,		1 - txx -tyy,	0,
				0,				0,				0,				1
			);		
	}

	
};

}//end of namespace tempest;
#endif
