/* 
 * Copyright 2007 Tatooine Project <http://tatooine.sourceforge.jp/> 
 *  
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *  
 *     http://www.apache.org/licenses/LICENSE-2.0
 *  
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.sf.tatooine.gtx.node;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import jp.sf.tatooine.gtx.ELUtils;
import jp.sf.tatooine.gtx.GtxContext;
import jp.sf.tatooine.gtx.GtxSyntaxException;
import jp.sf.tatooine.gtx.LoopRange;
import jp.sf.tatooine.gtx.Tag;

/**
 * タグ要素.
 *
 * @author  Tooru Noda
 * @version 1.0 2007/09/07
 * @since   JDK5.0 Tiger
 */
public class ElementNode extends Node {
	
	/** 要素名. */
	private String _name;
	
	/** 子要素. */
	private List<Node> _childNodes = new ArrayList<Node>();
	
	/** 属性値. */
	private List<Attribute> _attrs = new ArrayList<Attribute>();
	
	private Map<GtxAttributeType, GtxAttribute> _gtxAttrs = 
			new HashMap<GtxAttributeType, GtxAttribute>();
	
	/**
	 * コンストラクタ.
	 * 
	 * @param name		要素名
	 * @param parent	親要素への参照
	 */
	public ElementNode(String name) {
		_name = name;
	}
	
	public void addGtxAttribute(GtxAttribute attr) {
		_gtxAttrs.put(attr.getType(), attr);
		_attrs.add(attr);
	}
	
	public String getName() {
		return _name;
	}
	
	/**
	 * 属性を追加する.
	 * 
	 * @param attr	属性
	 */
	public void addAttribute(Attribute attr) {
		_attrs.add(attr);
	}
	
	/**
	 * 子要素を追加する.
	 * 
	 * @param node	子要素
	 */
	public void appendChild(Node node) {
		node.setParent(this);
		_childNodes.add(node);
	}
	
	/**
	 * 子要素のリストを取得する.
	 * 
	 * @return	子要素のリスト
	 */
	public List<Node> getChildNodes() {
		return _childNodes;
	}

	public void evaluate(StringBuilder builder, GtxContext context) 
			throws GtxSyntaxException {

		GtxAttribute gtxAttr = null;
		String value = null;
		boolean ignore = false;
		
		/* gtx:ignore */
		if (_gtxAttrs.containsKey(GtxAttributeType.IGNORE)) {
			gtxAttr = _gtxAttrs.get(GtxAttributeType.IGNORE);
			value = gtxAttr.getValue();
			if (!value.matches("true|false")) {
				throw new GtxSyntaxException(
						String.format("ignore=%s : ignoreにはboolean値を指定してください。", value));
			}
			ignore = Boolean.valueOf(ELUtils.evaluateEL(context, value));
		}
		/* gtx:if */
		if (!ignore && _gtxAttrs.containsKey(GtxAttributeType.IF)) {
			gtxAttr = _gtxAttrs.get(GtxAttributeType.IF);
			value = gtxAttr.getValue();
			if (!value.matches("true|false")) {
				throw new GtxSyntaxException(
						String.format("if=%s : ifにはboolean値を指定してください。", value));
			}
			boolean bif = Boolean.valueOf(ELUtils.evaluateEL(context, value));
			if (!bif) {
				return;
			}
		}
		/* gtx:unless */
		if (!ignore && _gtxAttrs.containsKey(GtxAttributeType.UNLESS)) {
			gtxAttr = _gtxAttrs.get(GtxAttributeType.UNLESS);
			value = gtxAttr.getValue();
			if (!value.matches("true|false")) {
				throw new GtxSyntaxException(
						String.format("unless=%s : unlessにはboolean値を指定してください。", value));
			}
			boolean unless = Boolean.valueOf(ELUtils.evaluateEL(context, value));
			if (unless) {
				return;
			}
		}
		/* gtx:for */
		LoopRange range = new LoopRange(1, 1);
		int step = 1;
		String index = null;
		boolean inside = false;
		GtxContext localContext = new GtxContext();
		localContext.setParent(context);
		
		if (!ignore && _gtxAttrs.containsKey(GtxAttributeType.FOR)) {
			GtxFor gtxFor = (GtxFor) _gtxAttrs.get(GtxAttributeType.FOR);
			value = gtxFor.getValue();
			range = LoopRange.parse(ELUtils.evaluateEL(localContext, value));
			
			step = Integer.valueOf(ELUtils.evaluateEL(localContext, gtxFor.getStep()));
			index = gtxFor.getIndex();
			if (index != null) {
				index = ELUtils.evaluateEL(localContext, index);
			}
			value = gtxFor.getInside();
			if (!value.matches("true|false")) {
				throw new GtxSyntaxException(
						String.format("inside=%s : insideにはboolean値を指定してください。", value));
			}
			inside = Boolean.valueOf(ELUtils.evaluateEL(localContext, value));
		}
		if (inside) {
			
			builder.append(START_TAG_PREFIX);
			builder.append(_name);
			
			/* gtx属性命令でないものはelを評価して出力 */
			for (Attribute attr : _attrs) {
				if (attr.getType() == GtxAttributeType.NA) {
					builder.append(ATTR_SEPARATOR);
					attr.evaluate(builder, localContext);
					if (attr.getName().equalsIgnoreCase("id")) {
						evaluateTag(ELUtils.evaluateEL(context, attr.getValue()), builder, localContext);
					}
				}
			}
			if (!ignore && _gtxAttrs.containsKey(GtxAttributeType.INNER_TEXT)) {
				gtxAttr = _gtxAttrs.get(GtxAttributeType.INNER_TEXT);
				value = gtxAttr.getValue();
				builder.append(TAG_POSTFIX);
				for (int i = range.getBegin(); i <= range.getEnd(); i += step) {
					if (index != null) {
						localContext.put(index, i);
					}
					builder.append(ELUtils.evaluateEL(localContext, value));
				}
				builder.append(END_TAG_PREFIX);
				builder.append(_name);
				builder.append(TAG_POSTFIX);
			}
			else {
				if (_childNodes.isEmpty()) {
					builder.append(EMPTY_TAG_POSTFIX);
				}
				else {
					builder.append(TAG_POSTFIX);
					for (int i = range.getBegin(); i <= range.getEnd(); i += step) {
						if (index != null) {
							localContext.put(index, i);
						}
						for (Node node : _childNodes) {
							node.evaluate(builder, localContext);
						}
					}
					builder.append(END_TAG_PREFIX);
					builder.append(_name);
					builder.append(TAG_POSTFIX);
				}
			}
		}
		else {
			for (int i = range.getBegin(); i <= range.getEnd(); i += step) {
				
				if (index != null) {
					localContext.put(index, i);
				}
				builder.append(START_TAG_PREFIX);
				builder.append(_name);
				
				/* gtx属性命令でないものはelを評価して出力 */
				for (Attribute attr : _attrs) {
					if (attr.getType() == GtxAttributeType.NA) {
						builder.append(ATTR_SEPARATOR);
						attr.evaluate(builder, localContext);
						if (attr.getName().equalsIgnoreCase("id")) {
							evaluateTag(ELUtils.evaluateEL(context, attr.getValue()), builder, localContext);
						}
					}
				}
				if (!ignore && _gtxAttrs.containsKey(GtxAttributeType.INNER_TEXT)) {
					gtxAttr = _gtxAttrs.get(GtxAttributeType.INNER_TEXT);
					value = gtxAttr.getValue();
					builder.append(TAG_POSTFIX);
					builder.append(ELUtils.evaluateEL(localContext, value));
					builder.append(END_TAG_PREFIX);
					builder.append(_name);
					builder.append(TAG_POSTFIX);
				}
				else {
					if (_childNodes.isEmpty()) {
						builder.append(EMPTY_TAG_POSTFIX);
					}
					else {
						builder.append(TAG_POSTFIX);
						for (Node node : _childNodes) {
							node.evaluate(builder, localContext);
						}
						builder.append(END_TAG_PREFIX);
						builder.append(_name);
						builder.append(TAG_POSTFIX);
					}
				}
			}
		}
		/* gtx:null */
		if (!ignore && _gtxAttrs.containsKey(GtxAttributeType.NULL)) {
			gtxAttr = _gtxAttrs.get(GtxAttributeType.NULL);
			value = gtxAttr.getValue();
			
			/* elの評価のみ */
			ELUtils.evaluateEL(context, value);
		}
	}
	
	private void evaluateTag(String id, StringBuilder builder, GtxContext context) 
			throws GtxSyntaxException {
		
		Tag tag = context.getTag(id);
		if (tag != null) {
			String tagName = tag.getTagName();
			if (tagName != null && !_name.equalsIgnoreCase(tagName)) {
				throw new IllegalArgumentException(
						String.format("id(%s)の整合性が取れていません：%s", 
								_name, tagName));
			}
			Map<String, String> attrMap = tag.getRawMap();
			for (Map.Entry<String, String> entry : attrMap.entrySet()) {
				Attribute attr = new Attribute(GtxAttributeType.NA, 
						entry.getKey(), entry.getValue());
				builder.append(ATTR_SEPARATOR);
				attr.evaluate(builder, context);
			}
		}
	}

	/**
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		
		StringBuilder builder = new StringBuilder();
		builder.append(START_TAG_PREFIX);
		builder.append(_name);
		for (Attribute attr : _attrs) {
			builder.append(ATTR_SEPARATOR);
			builder.append(attr.toString());
		}
		if (_childNodes.isEmpty()) {
			builder.append(EMPTY_TAG_POSTFIX);
		}
		else {
			builder.append(TAG_POSTFIX);
			for (Node node : _childNodes) {
				builder.append(node.toString());
			}
			builder.append(END_TAG_PREFIX);
			builder.append(_name);
			builder.append(TAG_POSTFIX);
		}
		return builder.toString();
	}
	
}
