/* 
 * Copyright 2007 Tatooine Project <http://tatooine.sourceforge.jp/> 
 *  
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *  
 *     http://www.apache.org/licenses/LICENSE-2.0
 *  
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.sf.tatooine.gtx;

import java.util.ArrayList;
import java.util.List;
import java.util.Stack;

import jp.sf.tatooine.gtx.node.Attribute;
import jp.sf.tatooine.gtx.node.CommentNode;
import jp.sf.tatooine.gtx.node.DTDNode;
import jp.sf.tatooine.gtx.node.ElementNode;
import jp.sf.tatooine.gtx.node.GtxAttributeType;
import jp.sf.tatooine.gtx.node.GtxFor;
import jp.sf.tatooine.gtx.node.GtxIf;
import jp.sf.tatooine.gtx.node.GtxIgnore;
import jp.sf.tatooine.gtx.node.GtxInnerText;
import jp.sf.tatooine.gtx.node.GtxNull;
import jp.sf.tatooine.gtx.node.GtxUnless;
import jp.sf.tatooine.gtx.node.Namespace;
import jp.sf.tatooine.gtx.node.Node;
import jp.sf.tatooine.gtx.node.ProcessingInstructionNode;
import jp.sf.tatooine.gtx.node.TextNode;

import static jp.sf.tatooine.gtx.GtxConsts.*;

import org.apache.commons.lang.StringUtils;
import org.xml.sax.Attributes;
//import org.xml.sax.Locator;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.xml.sax.ext.DefaultHandler2;

/**
 * GtxTemplateHandler.
 *
 * @author  Tooru Noda
 * @version 1.0 2007/09/08
 * @since   JDK5.0 Tiger
 */
public class GtxTemplateHandler extends DefaultHandler2 {
	
	/** テンプレート. */
	private GtxTemplate _gtxTemplate;
	
	private String _gtxPrefix;
	
	private boolean _ignoreSpace = false;
	
	private boolean _ignoreComment = false;
	
	private List<Node> _topLevelList = new ArrayList<Node>();
	
	private Stack<ElementNode> _elementStack = new Stack<ElementNode>();
	
	private List<Namespace> _namespaceList = new ArrayList<Namespace>();

	private boolean _DTDprocessed = false;
	
	public GtxTemplate getGtxTemplate() {
		return _gtxTemplate;
	}

	public void setIgnoreComment(boolean ignoreComment) {
		_ignoreComment = ignoreComment;
	}

	public void setIgnoreSpace(boolean ignoreSpace) {
		_ignoreSpace = ignoreSpace;
	}
	
	/**
	 * @see org.xml.sax.ext.DefaultHandler2#comment(char[], int, int)
	 */
	@Override
	public void comment(char[] ch, int start, int length) throws SAXException {
		if (_DTDprocessed || _ignoreComment) {
			return;
		}
		String value = new String(ch, start, length);
		if (_ignoreSpace) {
			value = value.trim();
		}
		CommentNode commentNode = new CommentNode();
		if (_elementStack.isEmpty()) {
			_topLevelList.add(commentNode);
		}
		else {
			_elementStack.peek().appendChild(commentNode);
		}
		commentNode.setValue(value);
	}

	/**
	 * @see org.xml.sax.ext.DefaultHandler2#startDTD(
	 * 		java.lang.String, java.lang.String, java.lang.String)
	 */
	@Override
	public void startDTD(String name, String publicId, String systemId) throws SAXException {
		Node node = new DTDNode(name, publicId, systemId);
		_topLevelList.add(node);
		_DTDprocessed = true;
	}
	
	/**
	 * @see org.xml.sax.ext.DefaultHandler2#endDTD()
	 */
	@Override
	public void endDTD() throws SAXException {
		_DTDprocessed = false;
	}

	/**
	 * @see org.xml.sax.helpers.DefaultHandler#startDocument()
	 */
	@Override
	public void startDocument() throws SAXException {
		_gtxTemplate = new GtxTemplate();
		_gtxTemplate.setTopLevelList(_topLevelList);
	}

	/**
	 * @see org.xml.sax.helpers.DefaultHandler#processingInstruction(java.lang.String, java.lang.String)
	 */
	@Override
	public void processingInstruction(String target, String data) throws SAXException {
		
		Node node = new ProcessingInstructionNode(target, StringUtils.trimToEmpty(data));
		if (_elementStack.isEmpty()) {
			_topLevelList.add(node);
		}
		else {
			_elementStack.peek().appendChild(node);
		}
	}

	/**
	 * @see org.xml.sax.helpers.DefaultHandler#startPrefixMapping(
	 * 		java.lang.String, java.lang.String)
	 */
	@Override
	public void startPrefixMapping(String prefix, String uri) throws SAXException {
		Namespace namespace = new Namespace(prefix, uri);
		_namespaceList.add(namespace);
		if (GTX_SCHEMA.equals(uri)) {
			_gtxPrefix = prefix;
		}
	}

	/**
	 * @see org.xml.sax.helpers.DefaultHandler#endPrefixMapping(java.lang.String)
	 */
	@Override
	public void endPrefixMapping(String prefix) throws SAXException {
		if (_gtxPrefix != null && _gtxPrefix.equals(prefix)) {
			_gtxPrefix = null;
		}
	}

	/**
	 * @see org.xml.sax.helpers.DefaultHandler#startElement(
	 * 		java.lang.String, java.lang.String, 
	 * 		java.lang.String, org.xml.sax.Attributes)
	 */
	@Override
	public void startElement(
			String uri, 
			String localName, 
			String qName, 
			Attributes attributes) throws SAXException {
		
		ElementNode elementNode = new ElementNode(qName);
		if (_elementStack.isEmpty()) {
			_topLevelList.add(elementNode);
		}
		else {
			_elementStack.peek().appendChild(elementNode);
		}
		for (Namespace namespace : _namespaceList) {
			elementNode.addAttribute(namespace);
		}
		_namespaceList.clear();
		
		String value = null;
		try {
			/* gtx:ignore */
			value = attributes.getValue(GTX_SCHEMA, IGNORE);
			if (value != null) {
				GtxIgnore attr = new GtxIgnore(_gtxPrefix, value);
				elementNode.addGtxAttribute(attr);
			}
			/* gtx:if */
			value = attributes.getValue(GTX_SCHEMA, IF);
			if (value != null) {
				GtxIf attr = new GtxIf(_gtxPrefix, value);
				elementNode.addGtxAttribute(attr);
			}
			/* gtx:unless */
			value = attributes.getValue(GTX_SCHEMA, UNLESS);
			if (value != null) {
				GtxUnless attr = new GtxUnless(_gtxPrefix, value);
				elementNode.addGtxAttribute(attr);
			}
			/* gtx:for */
			value = attributes.getValue(GTX_SCHEMA, FOR);
			if (value != null) {
				GtxFor attr = new GtxFor(_gtxPrefix, value);
				
				/* gtx:index */
				String index = attributes.getValue(GTX_SCHEMA, INDEX);
				if (index != null) {
					attr.setIndex(index);
				}
				/* gtx:inside */
				String inside = attributes.getValue(GTX_SCHEMA, INSIDE);
				if (inside != null) {
					attr.setInside(inside);
				}
				/* gtx:step */
				String step = attributes.getValue(GTX_SCHEMA, STEP);
				if (step != null) {
					attr.setStep(step);
				}
				elementNode.addGtxAttribute(attr);
			}
			/* gtx:inner-text */
			value = attributes.getValue(GTX_SCHEMA, INNER_TEXT);
			if (value != null) {
				GtxInnerText attr = new GtxInnerText(_gtxPrefix, value);
				elementNode.addGtxAttribute(attr);
			}
			/* gtx:null */
			value = attributes.getValue(GTX_SCHEMA, NULL);
			if (value != null) {
				GtxNull attr = new GtxNull(_gtxPrefix, value);
				elementNode.addGtxAttribute(attr);
			}
		}
		catch (Exception e) {
			throw new SAXException(e.getMessage());
		}
		for (int i = 0; i < attributes.getLength(); i++) {
			if (GTX_SCHEMA.equals(attributes.getURI(i))) {
				continue;	/* gtx命令は処理済 */
			}
			Attribute attr = new Attribute(
					GtxAttributeType.NA,
					attributes.getQName(i), 
					attributes.getValue(i));
			elementNode.addAttribute(attr);
//			
//			if (attr.getName().equalsIgnoreCase("id")) {
//				elementNode.setId(attr.getValue());
//			}
		}
		_elementStack.push(elementNode);
	}

	/**
	 * @see org.xml.sax.helpers.DefaultHandler#endElement(
	 * 		java.lang.String, java.lang.String, java.lang.String)
	 */
	@Override
	public void endElement(
			String uri, 
			String localName, 
			String qName) throws SAXException {
		
		_elementStack.pop();
	}

	/**
	 * @see org.xml.sax.helpers.DefaultHandler
	 * 		#ignorableWhitespace(char[], int, int)
	 */
	@Override
	public void ignorableWhitespace(
			char[] ch, 
			int start, 
			int length) throws SAXException {
		
		if (!_ignoreSpace) {
			_processTextNode(ch, start, length);
		}
	}

	/**
	 * @see org.xml.sax.helpers.DefaultHandler#characters(char[], int, int)
	 */
	@Override
	public void characters(
			char[] ch, 
			int start, 
			int length) throws SAXException {
		
		_processTextNode(ch, start, length);
	}
	
	private void _processTextNode(
			char[] ch, 
			int start, 
			int length) throws SAXException {
		
		String text = new String(ch, start, length);
		if (_ignoreSpace) {
			text = text.trim();
		}
		TextNode textNode = new TextNode();
		textNode.setParent(_elementStack.peek());
		textNode.setValue(text);
		_elementStack.peek().appendChild(textNode);
	}
	
	public void warning(SAXParseException e) {
		
		throw new RuntimeException(String.format("警告: %s %s行目\n%s", 
				"", e.getLineNumber(), e.getMessage()));
	}
	
	public void error(SAXParseException e) {
		throw new RuntimeException(String.format("エラー: %s %s行目\n%s", 
				"", e.getLineNumber(), e.getMessage()));
	}
	
	public void fatalError(SAXParseException e) {
		throw new RuntimeException(String.format("深刻なエラー: %s %s行目\n%s", 
				"", e.getLineNumber(), e.getMessage()));
	}

}
