package jp.sourceforge.talisman.hermes;

/*
 * $Id: HermesTest.java 198 2009-05-30 14:45:26Z tama3 $
 */

import java.io.File;
import java.util.Iterator;

import jp.sourceforge.talisman.hermes.maven.Artifact;
import jp.sourceforge.talisman.hermes.maven.DependencyScope;
import jp.sourceforge.talisman.hermes.maven.License;
import jp.sourceforge.talisman.hermes.maven.Repository;

import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

/**
 * @author Haruaki Tamada
 * @version $Revision: 198 $
 */
public class HermesTest{
    private HermesContext context;

    @Before
    public void setUp(){
        context = new HermesContext(
            new Repository("local", "local repository for testing", getClass().getResource("/repository/"))
        );
        context.setRepositoryScope(RepositoryScope.BOTH);
        new File("target/destination").mkdirs();
    }

    @After
    public void tearDown(){
        File file = new File("target/destination");
        for(File f: file.listFiles()){
            f.deleteOnExit();
        }
        file.deleteOnExit();
    }

    @Test
    public void testGettingArtifact() throws Exception{
        Hermes hermes = new Hermes(context);

        Artifact artifact = hermes.findArtifact("jp.sourceforge", "stigmata", "2.0.0");
        Assert.assertEquals(artifact.getGroupId(), "jp.sourceforge");
        Assert.assertEquals(artifact.getArtifactId(), "stigmata");
        Assert.assertEquals(artifact.getVersion(), "2.0.0");

        Assert.assertEquals(artifact.getPom().getDependencyCount(), 8);
        Iterator<Artifact> iterator = artifact.dependencies();
        Artifact a1 = iterator.next();
        Assert.assertEquals(a1.getGroupId(), "asm");
        Assert.assertEquals(a1.getArtifactId(), "asm-all");
        Assert.assertEquals(a1.getVersion(), "3.1");
        Assert.assertEquals(a1.getScope(), DependencyScope.COMPILE);

        Artifact a2 = iterator.next();
        Assert.assertEquals(a2.getGroupId(), "commons-dbutils");
        Assert.assertEquals(a2.getArtifactId(), "commons-dbutils");
        Assert.assertEquals(a2.getVersion(), "1.1");
        Assert.assertEquals(a2.getScope(), DependencyScope.COMPILE);

        Artifact a3 = iterator.next();
        Assert.assertEquals(a3.getGroupId(), "commons-beanutils");
        Assert.assertEquals(a3.getArtifactId(), "commons-beanutils");
        Assert.assertEquals(a3.getVersion(), "1.7.0");
        Assert.assertEquals(a3.getScope(), DependencyScope.COMPILE);

        Artifact a4 = iterator.next();
        Assert.assertEquals(a4.getGroupId(), "jp.sourceforge.talisman");
        Assert.assertEquals(a4.getArtifactId(), "mds");
        Assert.assertEquals(a4.getVersion(), "1.0.0");
        Assert.assertEquals(a4.getScope(), DependencyScope.COMPILE);

        Artifact a5 = iterator.next();
        Assert.assertEquals(a5.getGroupId(), "jp.sourceforge.talisman");
        Assert.assertEquals(a5.getArtifactId(), "xmlcli");
        Assert.assertEquals(a5.getVersion(), "1.2.2");
        Assert.assertEquals(a5.getScope(), DependencyScope.COMPILE);

        Artifact a6 = iterator.next();
        Assert.assertEquals(a6.getGroupId(), "jp.sourceforge.talisman");
        Assert.assertEquals(a6.getArtifactId(), "i18n");
        Assert.assertEquals(a6.getVersion(), "1.0.1");
        Assert.assertEquals(a6.getScope(), DependencyScope.COMPILE);

        Artifact a7 = iterator.next();
        Assert.assertEquals(a7.getGroupId(), "jp.sourceforge.stigmata");
        Assert.assertEquals(a7.getArtifactId(), "digger");
        Assert.assertEquals(a7.getVersion(), "1.1.0");
        Assert.assertEquals(a7.getScope(), DependencyScope.COMPILE);

        Artifact a8 = iterator.next();
        Assert.assertEquals(a8.getGroupId(), "junit");
        Assert.assertEquals(a8.getArtifactId(), "junit");
        Assert.assertEquals(a8.getVersion(), "4.3");
        Assert.assertEquals(a8.getScope(), DependencyScope.TEST);

        License[] licenses = artifact.getPom().getLicenses();
        Assert.assertEquals(licenses[0].getName(), "Apache License 2.0");
        Assert.assertEquals(licenses[0].getUrl(), "http://www.apache.org/licenses/LICENSE-2.0.txt");
    }

    @Test
    public void testGettingLatestVersion() throws Exception{
        Hermes hermes = new Hermes(context);

        Artifact artifact = hermes.findArtifact("jp.sourceforge", "stigmata");
        Assert.assertEquals(artifact.getGroupId(), "jp.sourceforge");
        Assert.assertEquals(artifact.getArtifactId(), "stigmata");
        Assert.assertEquals(artifact.getVersion(), "2.1.0");

        Assert.assertEquals(artifact.getPom().getDependencyCount(), 8);
        Iterator<Artifact> iterator = artifact.dependencies();
        Artifact a1 = iterator.next();
        Assert.assertEquals(a1.getGroupId(), "asm");
        Assert.assertEquals(a1.getArtifactId(), "asm-all");
        Assert.assertEquals(a1.getVersion(), "3.1");
        Assert.assertEquals(a1.getScope(), DependencyScope.COMPILE);

        Artifact a2 = iterator.next();
        Assert.assertEquals(a2.getGroupId(), "commons-dbutils");
        Assert.assertEquals(a2.getArtifactId(), "commons-dbutils");
        Assert.assertEquals(a2.getVersion(), "1.1");
        Assert.assertEquals(a2.getScope(), DependencyScope.COMPILE);

        Artifact a3 = iterator.next();
        Assert.assertEquals(a3.getGroupId(), "commons-beanutils");
        Assert.assertEquals(a3.getArtifactId(), "commons-beanutils");
        Assert.assertEquals(a3.getVersion(), "1.7.0");
        Assert.assertEquals(a3.getScope(), DependencyScope.COMPILE);

        Artifact a4 = iterator.next();
        Assert.assertEquals(a4.getGroupId(), "jp.sourceforge.talisman");
        Assert.assertEquals(a4.getArtifactId(), "mds");
        Assert.assertEquals(a4.getVersion(), "1.0.0");
        Assert.assertEquals(a4.getScope(), DependencyScope.COMPILE);

        Artifact a5 = iterator.next();
        Assert.assertEquals(a5.getGroupId(), "jp.sourceforge.talisman");
        Assert.assertEquals(a5.getArtifactId(), "xmlcli");
        Assert.assertEquals(a5.getVersion(), "1.2.2");
        Assert.assertEquals(a5.getScope(), DependencyScope.COMPILE);

        Artifact a6 = iterator.next();
        Assert.assertEquals(a6.getGroupId(), "jp.sourceforge.talisman");
        Assert.assertEquals(a6.getArtifactId(), "i18n");
        Assert.assertEquals(a6.getVersion(), "1.0.1");
        Assert.assertEquals(a6.getScope(), DependencyScope.COMPILE);

        Artifact a7 = iterator.next();
        Assert.assertEquals(a7.getGroupId(), "jp.sourceforge.stigmata");
        Assert.assertEquals(a7.getArtifactId(), "digger");
        Assert.assertEquals(a7.getVersion(), "1.2.0");
        Assert.assertEquals(a7.getScope(), DependencyScope.COMPILE);

        Artifact a8 = iterator.next();
        Assert.assertEquals(a8.getGroupId(), "junit");
        Assert.assertEquals(a8.getArtifactId(), "junit");
        Assert.assertEquals(a8.getVersion(), "4.3");
        Assert.assertEquals(a8.getScope(), DependencyScope.TEST);

        License[] licenses = artifact.getPom().getLicenses();
        Assert.assertEquals(licenses[0].getName(), "Apache License 2.0");
        Assert.assertEquals(licenses[0].getUrl(), "http://www.apache.org/licenses/LICENSE-2.0.txt");
    }

    @Test
    public void testLoadingDependencies() throws Exception{
        Hermes hermes = new Hermes(context);

        Artifact artifact1 = new Artifact("jp.sourceforge", "stigmata", "2.0.0");
        Artifact[] artifacts = hermes.getDependencies(artifact1);

        Assert.assertEquals(artifacts.length, 11);
        Assert.assertEquals(artifacts[0].getGroupId(), "jp.sourceforge");
        Assert.assertEquals(artifacts[0].getArtifactId(), "stigmata");
        Assert.assertEquals(artifacts[0].getVersion(), "2.0.0");
        Assert.assertEquals(artifacts[0].getScope(), DependencyScope.COMPILE);

        Assert.assertEquals(artifacts[1].getGroupId(), "asm");
        Assert.assertEquals(artifacts[1].getArtifactId(), "asm-all");
        Assert.assertEquals(artifacts[1].getVersion(), "3.1");
        Assert.assertEquals(artifacts[1].getScope(), DependencyScope.COMPILE);

        Assert.assertEquals(artifacts[2].getGroupId(), "commons-dbutils");
        Assert.assertEquals(artifacts[2].getArtifactId(), "commons-dbutils");
        Assert.assertEquals(artifacts[2].getVersion(), "1.1");
        Assert.assertEquals(artifacts[2].getScope(), DependencyScope.COMPILE);

        Assert.assertEquals(artifacts[3].getGroupId(), "commons-beanutils");
        Assert.assertEquals(artifacts[3].getArtifactId(), "commons-beanutils");
        Assert.assertEquals(artifacts[3].getVersion(), "1.7.0");
        Assert.assertEquals(artifacts[3].getScope(), DependencyScope.COMPILE);

        Assert.assertEquals(artifacts[4].getGroupId(), "commons-logging");
        Assert.assertEquals(artifacts[4].getArtifactId(), "commons-logging");
        Assert.assertEquals(artifacts[4].getVersion(), "1.0.3");
        Assert.assertEquals(artifacts[4].getScope(), DependencyScope.COMPILE);

        Assert.assertEquals(artifacts[5].getGroupId(), "jp.sourceforge.talisman");
        Assert.assertEquals(artifacts[5].getArtifactId(), "mds");
        Assert.assertEquals(artifacts[5].getVersion(), "1.0.0");
        Assert.assertEquals(artifacts[5].getScope(), DependencyScope.COMPILE);

        Assert.assertEquals(artifacts[6].getGroupId(), "jp.sourceforge.talisman");
        Assert.assertEquals(artifacts[6].getArtifactId(), "csvio");
        Assert.assertEquals(artifacts[6].getVersion(), "1.0.1");
        Assert.assertEquals(artifacts[6].getScope(), DependencyScope.COMPILE);

        Assert.assertEquals(artifacts[7].getGroupId(), "jp.sourceforge.talisman");
        Assert.assertEquals(artifacts[7].getArtifactId(), "i18n");
        Assert.assertEquals(artifacts[7].getVersion(), "1.0.1");
        Assert.assertEquals(artifacts[7].getScope(), DependencyScope.COMPILE);

        Assert.assertEquals(artifacts[8].getGroupId(), "jp.sourceforge.talisman");
        Assert.assertEquals(artifacts[8].getArtifactId(), "xmlcli");
        Assert.assertEquals(artifacts[8].getVersion(), "1.2.2");
        Assert.assertEquals(artifacts[8].getScope(), DependencyScope.COMPILE);

        Assert.assertEquals(artifacts[9].getGroupId(), "commons-cli");
        Assert.assertEquals(artifacts[9].getArtifactId(), "commons-cli");
        Assert.assertEquals(artifacts[9].getVersion(), "1.1");
        Assert.assertEquals(artifacts[9].getScope(), DependencyScope.COMPILE);

        Assert.assertEquals(artifacts[10].getGroupId(), "jp.sourceforge.stigmata");
        Assert.assertEquals(artifacts[10].getArtifactId(), "digger");
        Assert.assertEquals(artifacts[10].getVersion(), "1.1.0");
        Assert.assertEquals(artifacts[10].getScope(), DependencyScope.COMPILE);
    }

    @Test
    public void testLoadingDependenciesAndIgnores() throws Exception{
        context = new HermesContextParser().parse(getClass().getResource("/hermes/hermes2.xml"));
        Hermes hermes = new Hermes(context);

        Artifact artifact1 = new Artifact("jp.sourceforge", "stigmata", "2.0.0");
        Artifact[] artifacts = hermes.getDependencies(artifact1);

        Assert.assertEquals(artifacts.length, 9);
        Assert.assertEquals(artifacts[0].getGroupId(), "jp.sourceforge");
        Assert.assertEquals(artifacts[0].getArtifactId(), "stigmata");
        Assert.assertEquals(artifacts[0].getVersion(), "2.0.0");
        Assert.assertEquals(artifacts[0].getScope(), DependencyScope.COMPILE);

        Assert.assertEquals(artifacts[1].getGroupId(), "asm");
        Assert.assertEquals(artifacts[1].getArtifactId(), "asm-all");
        Assert.assertEquals(artifacts[1].getVersion(), "3.1");
        Assert.assertEquals(artifacts[1].getScope(), DependencyScope.COMPILE);

        Assert.assertEquals(artifacts[2].getGroupId(), "commons-dbutils");
        Assert.assertEquals(artifacts[2].getArtifactId(), "commons-dbutils");
        Assert.assertEquals(artifacts[2].getVersion(), "1.1");
        Assert.assertEquals(artifacts[2].getScope(), DependencyScope.COMPILE);

        Assert.assertEquals(artifacts[3].getGroupId(), "jp.sourceforge.talisman");
        Assert.assertEquals(artifacts[3].getArtifactId(), "mds");
        Assert.assertEquals(artifacts[3].getVersion(), "1.0.0");
        Assert.assertEquals(artifacts[3].getScope(), DependencyScope.COMPILE);

        Assert.assertEquals(artifacts[4].getGroupId(), "jp.sourceforge.talisman");
        Assert.assertEquals(artifacts[4].getArtifactId(), "csvio");
        Assert.assertEquals(artifacts[4].getVersion(), "1.0.1");
        Assert.assertEquals(artifacts[4].getScope(), DependencyScope.COMPILE);

        Assert.assertEquals(artifacts[5].getGroupId(), "jp.sourceforge.talisman");
        Assert.assertEquals(artifacts[5].getArtifactId(), "i18n");
        Assert.assertEquals(artifacts[5].getVersion(), "1.0.1");
        Assert.assertEquals(artifacts[5].getScope(), DependencyScope.COMPILE);

        Assert.assertEquals(artifacts[6].getGroupId(), "jp.sourceforge.talisman");
        Assert.assertEquals(artifacts[6].getArtifactId(), "xmlcli");
        Assert.assertEquals(artifacts[6].getVersion(), "1.2.2");
        Assert.assertEquals(artifacts[6].getScope(), DependencyScope.COMPILE);

        Assert.assertEquals(artifacts[7].getGroupId(), "commons-cli");
        Assert.assertEquals(artifacts[7].getArtifactId(), "commons-cli");
        Assert.assertEquals(artifacts[7].getVersion(), "1.1");
        Assert.assertEquals(artifacts[7].getScope(), DependencyScope.COMPILE);

        Assert.assertEquals(artifacts[8].getGroupId(), "jp.sourceforge.stigmata");
        Assert.assertEquals(artifacts[8].getArtifactId(), "digger");
        Assert.assertEquals(artifacts[8].getVersion(), "1.1.0");
        Assert.assertEquals(artifacts[8].getScope(), DependencyScope.COMPILE);
    }

    @Test
    public void testDependenciesFromContextParser() throws Exception{
        context = new HermesContextParser().parse(getClass().getResource("/hermes/hermes.xml"));
        context.setDestination("target/destination");
        Hermes hermes = new Hermes(context);

        Artifact[] contextArtifacts = context.getDependencies();
        Assert.assertEquals(contextArtifacts.length, 1);
        Assert.assertEquals(contextArtifacts[0].getGroupId(), "jp.sourceforge");
        Assert.assertEquals(contextArtifacts[0].getArtifactId(), "stigmata");
        Assert.assertEquals(contextArtifacts[0].getVersion(), "2.0.0");

        Artifact[] artifacts = hermes.getDependencies();
        Assert.assertEquals(artifacts.length, 11);
        Assert.assertEquals(artifacts[0].getGroupId(), "jp.sourceforge");
        Assert.assertEquals(artifacts[0].getArtifactId(), "stigmata");
        Assert.assertEquals(artifacts[0].getVersion(), "2.0.0");
        Assert.assertEquals(artifacts[0].getScope(), DependencyScope.COMPILE);

        Assert.assertEquals(artifacts[1].getGroupId(), "asm");
        Assert.assertEquals(artifacts[1].getArtifactId(), "asm-all");
        Assert.assertEquals(artifacts[1].getVersion(), "3.1");
        Assert.assertEquals(artifacts[1].getScope(), DependencyScope.COMPILE);

        Assert.assertEquals(artifacts[2].getGroupId(), "commons-dbutils");
        Assert.assertEquals(artifacts[2].getArtifactId(), "commons-dbutils");
        Assert.assertEquals(artifacts[2].getVersion(), "1.1");
        Assert.assertEquals(artifacts[2].getScope(), DependencyScope.COMPILE);

        Assert.assertEquals(artifacts[3].getGroupId(), "commons-beanutils");
        Assert.assertEquals(artifacts[3].getArtifactId(), "commons-beanutils");
        Assert.assertEquals(artifacts[3].getVersion(), "1.7.0");
        Assert.assertEquals(artifacts[3].getScope(), DependencyScope.COMPILE);

        Assert.assertEquals(artifacts[4].getGroupId(), "commons-logging");
        Assert.assertEquals(artifacts[4].getArtifactId(), "commons-logging");
        Assert.assertEquals(artifacts[4].getVersion(), "1.0.3");
        Assert.assertEquals(artifacts[4].getScope(), DependencyScope.COMPILE);

        Assert.assertEquals(artifacts[5].getGroupId(), "jp.sourceforge.talisman");
        Assert.assertEquals(artifacts[5].getArtifactId(), "mds");
        Assert.assertEquals(artifacts[5].getVersion(), "1.0.0");
        Assert.assertEquals(artifacts[5].getScope(), DependencyScope.COMPILE);

        Assert.assertEquals(artifacts[6].getGroupId(), "jp.sourceforge.talisman");
        Assert.assertEquals(artifacts[6].getArtifactId(), "csvio");
        Assert.assertEquals(artifacts[6].getVersion(), "1.0.1");
        Assert.assertEquals(artifacts[6].getScope(), DependencyScope.COMPILE);

        Assert.assertEquals(artifacts[7].getGroupId(), "jp.sourceforge.talisman");
        Assert.assertEquals(artifacts[7].getArtifactId(), "i18n");
        Assert.assertEquals(artifacts[7].getVersion(), "1.0.1");
        Assert.assertEquals(artifacts[7].getScope(), DependencyScope.COMPILE);

        Assert.assertEquals(artifacts[8].getGroupId(), "jp.sourceforge.talisman");
        Assert.assertEquals(artifacts[8].getArtifactId(), "xmlcli");
        Assert.assertEquals(artifacts[8].getVersion(), "1.2.2");
        Assert.assertEquals(artifacts[8].getScope(), DependencyScope.COMPILE);

        Assert.assertEquals(artifacts[9].getGroupId(), "commons-cli");
        Assert.assertEquals(artifacts[9].getArtifactId(), "commons-cli");
        Assert.assertEquals(artifacts[9].getVersion(), "1.1");
        Assert.assertEquals(artifacts[9].getScope(), DependencyScope.COMPILE);

        Assert.assertEquals(artifacts[10].getGroupId(), "jp.sourceforge.stigmata");
        Assert.assertEquals(artifacts[10].getArtifactId(), "digger");
        Assert.assertEquals(artifacts[10].getVersion(), "1.1.0");
        Assert.assertEquals(artifacts[10].getScope(), DependencyScope.COMPILE);
    }

    @Test
    public void testBasic() throws Exception{
        context = new HermesContextParser().parse(getClass().getResource("/hermes/hermes.xml"));
        context.setDestination("target/destination");
        Repository[] repositories = context.getRepositories();
        for(Repository r: repositories){
            context.removeRepository(r);
        }
        context.addRepository(
            new Repository("local", "local repository for testing", getClass().getResource("/repository/"))
        );
        context.setPolicy(UpdatingLibraryCheckPolicy.REPOSITORY_CHECK);
        Hermes hermes = new Hermes(context);

        Assert.assertTrue(hermes.canUpdate());

        HermesListener listener = new HermesListener(){
            public void downloadDone(HermesEvent e){
            }

            public void fileSizeGotten(HermesEvent e){
            }

            public void finish(HermesEvent e){
            }

            public void fileSizeUpdated(HermesEvent e){
            }

            public void targetResolved(HermesEvent e){
                Artifact[] artifacts = e.getArtifacts();
                Assert.assertEquals(artifacts.length, 2);
                Assert.assertEquals(artifacts[0].getGroupId(), "jp.sourceforge");
                Assert.assertEquals(artifacts[0].getArtifactId(), "stigmata");
                Assert.assertEquals(artifacts[0].getVersion(), "2.1.0");

                Assert.assertEquals(artifacts[1].getGroupId(), "jp.sourceforge.stigmata");
                Assert.assertEquals(artifacts[1].getArtifactId(), "digger");
                Assert.assertEquals(artifacts[1].getVersion(), "1.2.0");
            }
        };
        hermes.addHermesListener(listener);
        hermes.update();

        File file = new File("target/destination");
        File[] children = file.listFiles();
        Assert.assertEquals(children.length, 2);
        Assert.assertEquals(children[0].getName(), "digger-1.2.0.jar");
        Assert.assertEquals(children[1].getName(), "stigmata-2.1.0.jar");
    }
}
