/********************************************************************/
/* Copyright (c) 2019 System fugen G.K. and Yuzi Mizuno          */
/* All rights reserved.                                             */
/********************************************************************/

/** @file QSEval.h
 *  This is an internal header file, included by other QS headers.
 *  You should not attempt to use it directly.
 *
 *  Declaration for evaluation functions. eval_smoothness() and eval_gap().
 */
#if !defined(__QSEVAL_H__)
#define __QSEVAL_H__
#include <vector>

#include "mg/LBrep.h"
class MGSBRep;
class MGPosition;
class MGSBRepTP;
class MGSBRepVecTP;

namespace QS{

//! @brief Evaluation of the difference between @a qsurf and @a tpcrv.
//! @return The maximum value of angle difference.
//!
//! Returns the maximum difference of angle between a sequence of unit 
//! length normal vectors along a perimter curve of @a qsurf and @a tpcrv. 
//! @a tpcrv must be the same object as one when qsurf is constructed.
double eval_smoothness(
	const MGSBRep&   qsurf,   //!< quadangular surface we evaluate.
	const MGSBRepTP& tpcrv,   //!< TP curve that used when @a qsurf is created.
	int              iperi,   //!< 0: vmin, 1: umax, 2: vmax, and 3: umin. Based on class %MGSBRepTP.
	MGPosition&      uv       //!< a candidate parameter of @a qsurf.
);

//! @brief Evaluation of the difference between @a qsurf and @a pericrv.
//! @return The maximum value of gap.
//!
//! Returns the maximum gap of between a sequence of position
//! along a perimter curve of @a qsurf and the curve that is used when @a qsurf is constructed.
double eval_gap(
	const MGSBRep& qsurf,   //!< quadangular surface we evaluate.
	const MGCurve& pericrv, //!< perimeter curve that is used when qsurf is created.
	int            iperi,   //!< 0: vmin, 1: umax, 2: vmax, and 3: umin. Based on class %MGSBRepTP.
	MGPosition&    uv       //!< a candidate parameter of @a qsurf.
);

//! @brief Evaluation smoothness at perimeters.
//! @return The maximum value of angle.
double eval_smoothness(
	const MGSBRep& qsurf,   //!< a quadrangular surface to evaluate.
	const MGSBRepTP& tp,    //!< the TP data of @a qsurf.
	MGPosition& uv          //!< a candidate parameter of @a qsurf the max xx of gap takes at.
);

//! @brief Evaluation gaps.
//! @return The maximum value of gaps.
double eval_gap(
	const MGSBRep& qsurf,   //!< a quadrangular surface to evaluate.
	const MGCurve* perim[4],//!< the perimeter curves of @a qsurf.
	MGPosition& uv          //!< a candidate parameter of @a qsurf the max length of gap takes at.
);
double eval_gap(
	const MGSBRep& qsurf,   //!< a quadrangular surface to evaluate.
	const MGLBRep* perim[4],//!< the perimeter curves of @a qsurf.
	MGPosition& uv          //!< a candidate parameter of @a qsurf the max length of gap takes at.
);

void eval_deviation(
	const MGCurve&      crv1,
	const MGCurve&      crv2,
	std::vector<MGPosition>& pos1,
	std::vector<MGPosition>& pos2,
	std::vector<double>&     gap
);

void eval_deviation(
	const MGCurve& curve,
	const MGFace& face,
	std::vector<MGPosition>& pos1,
	std::vector<MGPosition>& pos2,
	std::vector<double>& gap
);

void eval_deviation(
	const MGFSurface& face1,
	const MGFSurface& face2,
	std::vector<MGPosition>& pos1,
	std::vector<MGPosition>& pos2,
	std::vector<double>& gap,
	std::vector<double>& angle
);

} // QS

#endif // __QSEVAL_H__
