/********************************************************************/
/* Copyright (c) 2019 System fugen G.K. and Yuzi Mizuno          */
/* All rights reserved.                                             */
/* ***************************************************** */
/********************************************************************/
/**
 * @file CircleTTRTool.cpp
 * @brief MGCircleTTRTool NX̃Cve[V
 */
#include "stdafx.h"
#include "fugenDoc.h"
#include "fugenView.h"
#include "CurveCmd/CircleTTRTool.h"
#include "Common/CommandStateOwner.h"
#include "GLInputRealDlg.h"
#include "mg/Ellipse.h"
#include "Misc/UserPreference.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

// MGCircleTTRTool

MGCircleTTRTool::MGCircleTTRTool(fugenDoc* pDoc)
: MGLocateState(pDoc,ID_CURVE_CIRCLE_TTR,NO_RUBBER,POINT_IPDRAW),m_dRadius(10.0){
	turn_on_near();
	turn_on_end();
}

MGCircleTTRTool::MGCircleTTRTool(MGCommandStateOwner* owner)
: MGLocateState(owner, UNLOCK_SNAP_ATTRIB, NO_RUBBER, POINT_IPDRAW), m_dRadius(10.0){
	turn_on_near();
	turn_on_end();
}

MGCommandBase* MGCircleTTRTool::initial_clone(fugenDoc* pDoc)const{
	return new MGCircleTTRTool(pDoc);
}

bool MGCircleTTRTool::initiate_tool(){
	MGLocateState::initiate_tool();
	UserPreference& pref = UserPreference::getInstance();
	m_dRadius = pref.GetDoubleValue(upv_Curve_CircleTTR_Radius);
	clear_pick_object();
	return false;
}

bool MGCircleTTRTool::terminate_tool(bool cancel)
{
	UserPreference& pref = UserPreference::getInstance();
	pref.SetDoubleValue(upv_Curve_CircleTTR_Radius, m_dRadius);

	return MGLocateState::terminate_tool(cancel);
}

bool MGCircleTTRTool::input_param(){
	SetStatusMessage(IDS_PROMPT_CIRCLE_RADIUS);
	CString str;
	VERIFY(str.LoadString(IDS_CAPTION_CORNER_R));

	CGLInputRealDlg dlg(CString(), str);
	dlg.SetValue(m_dRadius);
	bool bOK = (IDOK == dlg.DoModal());
	if(bOK)
		m_dRadius = dlg.GetValue();
	return bOK;
}

bool MGCircleTTRTool::OnCommandEnd(
	UINT nIDS,	//=0: erase the current message, and display no messages.
			//=1: display "xxxx" normally end.
			//otherwise: nIDS is a string id, and load the message from string table to display.
	bool erase_temporary_display
){
	const LInfoVec& linfos=locates();
	if(linfos.size()<2)
		return MGLocateState::OnCommandEnd(3);
	if(!input_param())
		return MGLocateState::OnCommandEnd(3);

	// vZJn
	SetStatusMessage(IDS_PROMPT_COMPUTE);
	CWaitCursor sandglass;
	
	const MGCurve& c1=*(dynamic_cast<const MGCurve*>(linfos[0]->object()));
	double t1=linfos[0]->curve_parameter();
	const MGCurve& c2=*(dynamic_cast<const MGCurve*>(linfos[1]->object()));
	double t2=linfos[1]->curve_parameter();

	MGPosition closestParam12 = c1.closest(c2);
	MGPosition p0=(c1.eval(closestParam12[0])+c2.eval(closestParam12[1]))/2.;
	MGPosition p1 = c1.eval(t1);
	MGPosition p2 = c2.eval(t2);
	MGUnit_vector normal = (p1 - p0) * (p2 - p0);
	int ret = 0;
	std::unique_ptr<MGEllipse> arc(new MGEllipse(c1,c2,normal,m_dRadius,t1,t2,ret));
	if(ret){
		// failed
		return MGLocateState::OnCommandEnd(IDS_FAIL_GENERATE_CURVE);
	}
	arc->unlimit();
	add_object_to_current_group(arc.release());
	return MGLocateState::OnCommandEnd(nIDS);
}

bool MGCircleTTRTool::OnLocated(const MGLocateInfo& info){
	if(info.is_deleted()){
		return false;
	}

	const MGCurve* curve = dynamic_cast<const MGCurve*>(info.object());
	if(!curve){
		// Ȑ̓_ȊÕNbN͔F߂Ȃ
		cancel_last_locate();
		return false;
	}
	
	const LInfoVec& linfos=locates();
	size_t np=linfos.size();
	if(np==2){
		const MGCurve* c1=dynamic_cast<const MGCurve*>(linfos[0]->object());
		if(c1 == curve){
			// ŐȐƓł͂Ȃ(MGCL)?
			cancel_last_locate();
		}
	}

	if(linfos.size()<2){
		return false;
	}

	// vZJn
	return OnCommandEnd(1);
}

void MGCircleTTRTool::prompt_message() const{
	SetStatusMessage(IDS_PROMPT_CIRCLE_BASELINE, locates().size()+1);
}
