/********************************************************************/
/* Copyright (c) 2019 System fugen G.K. and Yuzi Mizuno          */
/* All rights reserved.                                             */
/* ***************************************************** */
/********************************************************************/
/**
 * @file LocateOnObjects.cpp
 * @brief LocateOnObjects.h ̎B
 */
#include "stdafx.h"
#include "fugenDoc.h"
#include "fugenView.h"
#include "mg/Straight.h"
#include "mg/CSisects.h"
#include "topo/Shell.h"
#include "Common/LocateOnObjects.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

//Implements MGLocateOnObjects Class.
//MGLocateOnObjects is a class to get point data(locate) on objects
//
//////////Constructor//////////

// Standard constructor. This state belongs to an owner.
//constructor type 1(objects and owner).
MGLocateOnObjects::MGLocateOnObjects(
	std::list<const MGGel*>* objectives,
	MGCommandStateOwner* owner,//Owner of this MGLocateOnObjects.
	SNAP_ATTRIB_LOCKKIND lock_snap_attrib,//indicates if lock the snap attrib.
	RUBBER_KIND rubberband,
		//Specify what type of rubberband is necessary.
	IPOINT_DRAWER_KIND input_drawer,
		//specify what type of input points drawer be used.
	bool only_on_object	///<only_on_object==true means
		///<OnLocated is not invoked when a point is not on an object.
):MGLocateState(owner,lock_snap_attrib,rubberband,input_drawer,false),
m_only_on_object(only_on_object),m_objectives2(*objectives){
	set_snap_objectives(&m_objectives2);
}

// constructor that does not have an owner.
//constructor type 2(objects and fugenDoc).
MGLocateOnObjects::MGLocateOnObjects(
	std::list<const MGGel*>* objectives,
	fugenDoc*  pDoc,		//document.
	UINT	command_id,	//command id
	RUBBER_KIND rubberband,
		//Specify what type of rubberband is necessary.
	IPOINT_DRAWER_KIND input_drawer,
		//specify what type of input points drawer be used.
	bool only_on_object	///<only_on_object==true means
		///<OnLocated is not invoked when a point is not on an object.
):MGLocateState(pDoc,command_id,rubberband,input_drawer,false),
m_only_on_object(only_on_object),m_objectives2(*objectives){
	set_snap_objectives(&m_objectives2);
}

// Objectives is only one constructor. This state belongs to an owner.
//constructor type 3(an object and owner).
MGLocateOnObjects::MGLocateOnObjects(
	const MGObject* objective,
	MGCommandStateOwner* owner,//Owner of this MGLocateOnObjects.
	SNAP_ATTRIB_LOCKKIND lock_snap_attrib,//indicates if lock the snap attrib.
	RUBBER_KIND rubberband,
		//Specify what type of rubberband is necessary.
	IPOINT_DRAWER_KIND input_drawer,
		//specify what type of input points drawer be used.
	bool only_on_object	///<only_on_object==true means
		///<OnLocated is not invoked when a point is not on an object.
):MGLocateState(owner,lock_snap_attrib,rubberband,input_drawer,false),
m_objectives2(1,objective),m_only_on_object(only_on_object){
	set_snap_objectives(&m_objectives2);
}

// Objectives is only one constructor. Constructor that does not have an owner.
//constructor type 4(an object and fugenDoc).
MGLocateOnObjects::MGLocateOnObjects(
	const MGObject* objective,
	fugenDoc*  pDoc,		//document.
	UINT	command_id,	//command id
	RUBBER_KIND rubberband,
		//Specify what type of rubberband is necessary.
	IPOINT_DRAWER_KIND input_drawer,
		//specify what type of input points drawer be used.
	bool only_on_object	///<only_on_object==true means
		///<OnLocated is not invoked when a point is not on an object.
):MGLocateState(pDoc,command_id,rubberband,input_drawer,false),
	m_objectives2(1,objective),m_only_on_object(only_on_object){
	set_snap_objectives(&m_objectives2);
}

//Constructor that does have an owner.
//constructor type 5(no objects and owner).
//Objects must be set by set_locate_objectives.
MGLocateOnObjects::MGLocateOnObjects(
	MGCommandStateOwner* owner,//Owner of this MGLocateOnObjects.
	SNAP_ATTRIB_LOCKKIND lock_snap_attrib,//indicates if lock the snap attrib.
	RUBBER_KIND rubberband,
		//Specify what type of rubberband is necessary.
	IPOINT_DRAWER_KIND input_drawer,
		//specify what type of input points drawer be used.
	bool only_on_object	///<only_on_object==true means
		///<OnLocated is not invoked when a point is not on an object.
):MGLocateState(owner,lock_snap_attrib,rubberband,input_drawer,false),
m_only_on_object(only_on_object){
}

//Constructor that does not have an owner.
//constructor type 6(no objects and fugenDoc)
//Objects must be set by set_locate_objectives.
MGLocateOnObjects::MGLocateOnObjects(
	fugenDoc*  pDoc,		//document.
	UINT	command_id,	//command id
	RUBBER_KIND rubberband,
		//Specify what type of rubberband is necessary.
	IPOINT_DRAWER_KIND input_drawer,
		//specify what type of input points drawer be used.
	bool only_on_object	///<only_on_object==true means
		///<OnLocated is not invoked when a point is not on an object.
):MGLocateState(pDoc,command_id,rubberband,input_drawer,false),
	m_only_on_object(only_on_object){
}

//////////Member Function//////////

//Initiate the class.
bool MGLocateOnObjects::initiate_tool(){
	MGLocateState::initiate_tool();
	return false;
}

//Get a point on surface srf that is the intesection points of sl and srf.
bool get_locate_point_on_fsurface(
	const MGFSurface* srf,
	const MGStraight& sl,
	MGPosition& point,	//intersection point will be output
	const MGObject*& obj,//srf's MGObject pointer will be rturned when true returned.
	MGPosition& param	//intersection point's surface parameter will be returned.
){
	MGCSisects ises=srf->isectFS(sl);
	if(!ises.size()) return false;

	MGCSisects::iterator j=ises.begin(), je=ises.end();
	auto& csi = isectCast<MGCSisect>(j);
	double t= csi.param_curve();
	param= csi.param_surface();
	for(j++; j!=je; j++){
		auto& csij = isectCast<MGCSisect>(j);
		double t2= csij.param_curve();
		if(t2>t){
			t=t2; param= csij.param_surface();
		}
	}
	obj=srf->object_pointer();
	point=srf->eval(param);
	return true;
}

MGSnapPositions::snap_kind MGLocateOnObjects::locate_on_face(
	const MGStraight& sl,//straight line that is a point on the screen.
	MGPosition& point,	//located point will be output
	const MGObject*& obj,//When function's return value is nearpos, end, or knot,
		//the object pointer and the point's parameter value of the object will be returned.
		//When center, only MGObject pointer will be rturned.
	MGPosition& param	
)const{
	std::list<const MGGel*>::const_iterator i=m_objectives2.begin(), ie=m_objectives2.end();
	for(; i!=ie;i++){
		const MGFSurface* srf=dynamic_cast<const MGFSurface*>(*i);
		if(srf){
			if(get_locate_point_on_fsurface(srf,sl,point,obj,param)){
				return MGSnapPositions::ON_SURFACE;
			}
		}else{
			const MGShell* shell=dynamic_cast<const MGShell*>(*i);
			if(!shell)
				continue;
			MGShell::const_iterator j=shell->pcell_begin(), je=shell->pcell_end();
			for(; j!=je; j++){
				const MGFace* fj=static_cast<const MGFace*>((*j).get());
				if(get_locate_point_on_fsurface(fj,sl,point,obj,param)){
					return MGSnapPositions::ON_SURFACE;
				}
			}
		}
	}
	return MGSnapPositions::nopos;
}

//Locate a point from the screen coordinate point_in to world coordinate point.
//Function's return value is snap_kind of the point located:
//nopos: no points located:curve.
//endpos: end point located:curve.
//knotpos: knot point located:curve.
//centerpos:center of curve or fsurface.
//vertexpos: vertex point located:fsurface.
//nearpos: near point located:curve.
//ON_SURFACE: point on MGFSurface is located.
MGSnapPositions::snap_kind MGLocateOnObjects::locate_object_snap_point(
	fugenView* window,//The fugenView pointer where point_in  belongs to.
	const CPoint& point_in,	//Window's point coordinates.
	MGPosition& point,	//located point will be output
	const MGObject*& obj,//When function's return value is nearpos, end, knot(Curve),
				//vertex(FSurface), or center(Curve&FSurface),
				//the point's parameter value of the object be returned.
	MGPosition& param	
){
	obj=0;
	MGSnapPositions::snap_kind sk=
		MGLocateState::m_snapKind=MGLocateState::locate_object_snap_point(
		window,point_in,point,obj,param);
	if(!m_only_on_object)
		return sk;//When !m_only_on_object, other snap is valid.

	if(obj){
		std::list<const MGGel*>::iterator i, iend=m_objectives2.end();
		i=std::find(m_objectives2.begin(), iend, static_cast<const MGGel*>(obj));
		if(i!=iend)
			return sk;
	}

	MGStraight sl;
		//Obtain the sl from snapped point point.
	window->unproject_to_sl(point,sl);
	return locate_on_face(sl,point,obj,param);
}

///set the objectives to locate on.
///After set_locate_objectives(clear_locate_objectives), 
///MGLocateOnObjects::initiate_tool() or initiate_snap_data() must be invoked.
void MGLocateOnObjects::clear_locate_objectives(){
	m_objectives2.clear();
}

///set the objectives to locate on.
///After set_locate_objectives(clear_locate_objectives), MGLocateOnObjects::initiate_tool()
///or initiate_snap_data() must be invoked.
void MGLocateOnObjects::set_locate_objectives(const std::list<const MGGel*>& objectives){
	if(m_only_on_object)
		set_snap_objectives(&objectives);
	else
		add_snap_objectives(objectives);
	m_objectives2=objectives;
}
void MGLocateOnObjects::set_locate_objectives(const MGObject* objective){
	std::list<const MGGel*> objectives(1,objective);
	set_locate_objectives(objectives);
}

//Terminate the class.
//Overrided terminate must invoke MGLocateTool::terminate_tool last.
//The function's return value is true, if the tool termination is accepted.
//False, if the termination is refuses.
bool MGLocateOnObjects::terminate_tool(bool cancel){
	return MGLocateState::terminate_tool(cancel);
}

bool MGLocateOnObjects::OnKeyDown(
	fugenView* window,//The fugenView pointer where this event took place.
	UINT nChar,	UINT nRepCnt, UINT nFlags
		//These parameters are of CWnd::OnKeyDown. See the document.
){
	bool terminate=false;
	if(nChar==VK_F8 && m_only_on_object){
		if(F8Locate(window,m_located_object)){
			terminate=OnLocated(*(locates().back()));
			if(!terminate){
				prompt_message();
				draw_temporary();
			}
			if(MGCommandBase::OnKeyDown(window,nChar,nRepCnt,nFlags))
				terminate=true;
		}
	}else if(MGLocateState::OnKeyDown(window,nChar,nRepCnt,nFlags))
		terminate=true;
	return terminate;
}

//When a point is located, the data will be output to documnet's cursor().
bool MGLocateOnObjects::OnLButtonDown(
	fugenView* window,//The fugenView pointer where this event took place.
	UINT nFlags, CPoint point
		//These parameters are of CWnd::OnLButtonDown. See the document.
){
	return processLButtonDown(window,nFlags,point,m_located_object||!m_only_on_object);
}
