/********************************************************************/
/* Copyright (c) 2019 System fugen G.K. and Yuzi Mizuno          */
/* All rights reserved.                                             */
/* ***************************************************** */

// chamfer.cpp
// Implementation for chamfer module
#include "stdafx.h"
#include "Calc/curve.h"
#include "Calc/chamfer.h"
#include "mg/Straight.h"
#include "mg/CCisects.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

// [JȎ
namespace{
	using namespace mgcalc;

	/**
	 *  class ChamferInfo
	 *
	 *  БȐɊւAʎɕKvȏLNX
	 *  gvZɎgp
	 */
	class ChamferInfo{
	public:
		virtual ~ChamferInfo();

		// chamfer ɂăgۂ̋ȐԂ
		virtual std::unique_ptr<MGCurve> create_trimmed_curve(const ChamferParam* param) const = 0;
		virtual MGPosition trim_position(const ChamferParam* param) const = 0;

		bool is_start_nearer() const{ return m_start_is_nearer;}
		double trim_param() const{ return m_trim_param;}
		bool trim_start() const{ return m_trim_start;}

	protected:
		ChamferInfo(double trim_param, bool trim_start, bool start_is_nearer_to_chamfer_point)
			 : m_trim_param(trim_param),
			   m_trim_start(trim_start),
			   m_start_is_nearer(start_is_nearer_to_chamfer_point)
			{}

		double m_trim_param;      // gȐ̃gʒuł̃p[^
		bool   m_trim_start;      // gȐ̂ǂ瑤g邩
		bool   m_start_is_nearer; // chamfer point ɋ߂͎̂n_I_
	};

	/**
	 *  class ChamferInfoOnCurve
	 *
	 *  ʎɂgʒu^ꂽȐɂꍇ
	 *  gvZNX
	 */
	class ChamferInfoOnCurve : public ChamferInfo{
	public:
		ChamferInfoOnCurve(double trim_param, bool trim_start, bool start_is_nearer_to_chamfer_point)
			 : ChamferInfo(trim_param, trim_start, start_is_nearer_to_chamfer_point){}

		// chamfer ɂăgۂ̋ȐԂ
		virtual std::unique_ptr<MGCurve> create_trimmed_curve(const ChamferParam* param) const;
		virtual MGPosition trim_position(const ChamferParam* param) const;
	};

	/**
	 *  class ChamferInfoOnExtension
	 *
	 *  ʎɂgʒuA^ꂽȐɂ͂Ȃ
	 *  ɂꍇɃgvZNX
	 */
	class ChamferInfoOnExtension : public ChamferInfo{
	public:
		ChamferInfoOnExtension(
			double trim_param,
			MGStraight* extension,
			bool extend_start
			)
			 : ChamferInfo(trim_param, extend_start, extend_start),
			   m_extension(extension),
			   m_extend_start(extend_start){}

		// chamfer ɂăgۂ̋ȐԂ
		// ̈ꕔƃhbLOĂ
		virtual std::unique_ptr<MGCurve> create_trimmed_curve(const ChamferParam* param) const;
		virtual MGPosition trim_position(const ChamferParam* param) const;

	private:
		std::unique_ptr<MGStraight>  m_extension;  // ̐()
		bool                       m_extend_start; // IWiȐ̂ǂ瑤牄̂
	};

	/**
	 *  ̌vZAgʒuȐɋ߂
	 */
	std::unique_ptr<ChamferInfo> seek_trim_position(const ChamferParam*, double chamfer_position);

	/** 
	 *  KvȂƂo[W
	 */
	std::unique_ptr<ChamferInfo> seek_trim_position(const ChamferParam*, const MGStraight* ext, double chamfer_position, bool extend_start);

	ChamferInfo::~ChamferInfo(){
	}

	// 

	std::unique_ptr<MGCurve> ChamferInfoOnCurve::create_trimmed_curve(const ChamferParam* param) const{
		std::unique_ptr<MGCurve> curve(param->curve()->clone());
		double t=trim_param();
		if(trim_start()){
			// n_̂Ă
			curve->trim_start(t);
		}else{
			// I_̂Ă
			curve->trim_end(t);
		}
		return curve;
	}

	MGPosition ChamferInfoOnCurve::trim_position(const ChamferParam* param) const{
		// IWiȐɑ݂
		return param->curve()->eval(trim_param());
	}

	std::unique_ptr<MGCurve> ChamferInfoOnExtension::create_trimmed_curve(const ChamferParam* param) const{
		// IWiJ[ȗŚ{
		const MGCurve& curve = *param->curve();

		// ̃g
		double s = __min(0, trim_param());
		double e = __max(0, trim_param());
		std::unique_ptr<MGCurve> line(m_extension->part(s, e));

		// ڍ̂Ԃ
		return connect(curve, *line);
	}

	MGPosition ChamferInfoOnExtension::trim_position(const ChamferParam*) const{
		// ɑ݂
		return m_extension->eval(trim_param());
	}

	std::unique_ptr<ChamferInfo> seek_trim_position(const ChamferParam* param, double chamfer_position){
		assert(!param->is_null());
		assert(param->curve()->in_range(chamfer_position));

		// alias
		const MGCurve& curve = *param->curve();

		double distance = param->distance();
		double hint = param->hint();

		// sbNʒu chamfer position 猩ċȐ̎n_ł邩ǂ
		bool to_start = (hint <= chamfer_position);
		double len = to_start
			? curve.length(curve.param_s(), chamfer_position)
				: curve.length(chamfer_position, curve.param_e());

		if(distance > len){
			// ʎw苗ăAEc
			return std::unique_ptr<ChamferInfo>(nullptr);
		}

		double trim_position = curve.length_param(chamfer_position, (to_start ? -distance : distance));
		assert(curve.in_range(trim_position));

		// OK
		return std::unique_ptr<ChamferInfo>(new ChamferInfoOnCurve(trim_position, !to_start, !to_start));
	}

	std::unique_ptr<ChamferInfo> seek_trim_position(
		const ChamferParam* param,
		const MGStraight*   ext,
		double              chamfer_position, // ext ̃p[^ł
		bool                extend_start  // IWiȐ̉n_ǂ
		){
		assert(!param->is_null());
		assert(ext->in_range(chamfer_position));

		// alias
		const MGCurve& curve = *param->curve();

		double distance = param->distance();
		double hint = param->hint();

		double lenorg = curve.length();
		double lenext = abs(ext->length(0, chamfer_position));
		if(distance > lenorg + lenext){
			// ʎ苗ăAEc
			return std::unique_ptr<ChamferInfo>(nullptr);
		}

		// ʎ_vZ
		if(distance < lenext){
			// ɖʎ_
			double trim_param = ext->length_param(chamfer_position, (ext->param_e() == 0.) ? distance : -distance);

			return std::unique_ptr<ChamferInfo>(
				new ChamferInfoOnExtension(trim_param, new MGStraight(*ext), extend_start)
				);
		}
		else{
			double start_or_end = extend_start ? curve.param_s() : curve.param_e();
			// ̋Ȑɖʎ_
			double trim_param = curve.length_param(
				start_or_end,
				(extend_start ? (distance - lenext) : (lenext - distance))
				);
			return std::unique_ptr<ChamferInfo>(
				new ChamferInfoOnCurve(trim_param, extend_start, extend_start)
				);
		}
	}
} // namespace

namespace mgcalc{
	/**
	 *  class ChamferResult::Private
	 *
	 *  class ChamferResult ̃vCx[gȎS
	 */
	class ChamferResult::Private{
	public:
		Private(const ChamferParam* param1, const ChamferParam* param2)
			: m_param1(param1), m_param2(param2){}

		bool create_info();
		bool is_null() const;

		bool seek_with_extension(
			const MGStraight* ext1, bool extend_start1,
			const MGStraight* ext2, bool extend_start2);

		const ChamferParam*         m_param1;
		const ChamferParam*         m_param2;
		std::unique_ptr<ChamferInfo>  m_info1;
		std::unique_ptr<ChamferInfo>  m_info2;
		MGPosition                  m_point;
	};

	// class ChamferParam

	ChamferResult::~ChamferResult(){
		// (ł)fXgN^`ĂȂƃRpCʂȂ
	}

	bool ChamferParam::set(const MGCurve* curve, double distance, double hint){
		if(!curve) return false;
		if(distance < 0) return false;
		if(!curve->in_range(hint)) return false;

		m_curve = curve;
		m_distance = distance;
		m_hint = hint;
		return true;
	}

	// class ChamferResult

	ChamferResult::ChamferResult(
		const ChamferParam* param1, 
		const ChamferParam* param2
		) : m_impl(new Private(param1, param2)){
	}

	bool ChamferResult::chamfer(){
		if(m_impl->is_null()){
			return false;
		}
		// ChamferInfo oZbg
		if(!m_impl->create_info()){
			return false;
		}
		return true;
	}

	MGCurve* ChamferResult::chamfer_line() const{
		if(!m_impl->m_info1.get() || !m_impl->m_info2.get()){
			return 0;
		}
		if(m_impl->m_param1->distance() == 0 && m_impl->m_param2->distance() == 0){
			return 0;
		}
		// fɐ𐶐
		MGPosition start = m_impl->m_info1->trim_position(m_impl->m_param1);
		MGPosition end   = m_impl->m_info2->trim_position(m_impl->m_param2);
		return new MGStraight(end, start);
	}

	std::unique_ptr<MGCurve> ChamferResult::trimmed_curve(int i) const{
		const ChamferInfo* info = 0;
		const ChamferParam* param = 0;
		switch(i){
		case 1:
			info = m_impl->m_info1.get();
			param = m_impl->m_param1;
			break;
		case 2:
			info = m_impl->m_info2.get();
			param = m_impl->m_param2;
			break;
		}
		return info->create_trimmed_curve(param);
	}

	// class ChamferResult::Private

	bool ChamferResult::Private::seek_with_extension(
		const MGStraight* ext1, bool extend_start1,
		const MGStraight* ext2, bool extend_start2)
	{
		// alias
		const MGCurve& curve1 = *m_param1->curve();
		const MGCurve& curve2 = *m_param2->curve();
		const double hint1 = m_param1->hint();
		const double hint2 = m_param2->hint();

		// ʒuLĂ
		MGPosition posFrom1, posFrom2;
		if(ext1){
			posFrom1 = extend_start1 ? curve1.start_point() : curve1.end_point();
		}
		if(ext2){
			posFrom2 = extend_start2 ? curve2.start_point() : curve2.end_point();
		}

		double dLenPrev = -1;

		// curve1 ̂̂܂͂̉
		// curve2 ̂̂܂͂̉Ƃ̌vZ
		MGCCisects ls;
		if(ext1 && !ext2){
			ls = ext1->isect(curve2);
		}
		else if(!ext1 && ext2){
			ls = curve1.isect(*ext2);
		}
		else if(ext1 && ext2){
			ls = ext1->isect(*ext2);
		}
		else{
			ls = curve1.isect(curve2);
		}

		MGCCisects::iterator first = ls.begin(), last = ls.end();
		for(; first != last; ++first){
			std::unique_ptr<ChamferInfo> info1, info2;
			const MGCCisect& is = isectCast<const MGCCisect>(first);

			if(ext1){
				// 狁߂
				info1 = seek_trim_position(m_param1, ext1, is.param1(), extend_start1);
			}
			else{
				// Ȑ狁߂
				info1 = seek_trim_position(m_param1, is.param1());
			}
			if(!info1.get()){
				continue;
			}

			if(ext2){
				// 狁߂
				info2 = seek_trim_position(m_param2, ext2, is.param2(), extend_start2);
			}
			else{
				// Ȑ狁߂
				info2 = seek_trim_position(m_param2, is.param2());
			}
			if(!info2.get()){
				continue;
			}

			if(ls.size() == 1){
				// _XV
				m_info1 = std::move(info1);
				m_info2 = std::move(info2);
				m_point = is.point();
				return true;
			}
			else{
				if(dLenPrev < 0){
					double dLenCur = 0;
					if(ext1){
						dLenCur = is.point().distance(posFrom1);
					}
					else{
						dLenCur = fabs(curve1.length(hint1, is.param1()));
					}
					
					if(ext2){
						dLenCur += is.point().distance(posFrom2);
					}
					else{
						dLenCur = fabs(curve2.length(hint2, is.param2()));
					}
					dLenPrev = dLenCur;

					// _XV
					m_info1 = std::move(info1);
					m_info2 = std::move(info2);
					m_point = is.point();
				}
				else{
					double dLenCur = 0;
					if(ext1){
						dLenCur = is.point().distance(posFrom1);
					}
					else{
						dLenCur = fabs(curve1.length(hint1, is.param1()));
					}
					
					if(ext2){
						dLenCur += is.point().distance(posFrom2);
					}
					else{
						dLenCur = fabs(curve2.length(hint2, is.param2()));
					}

					if(dLenCur < dLenPrev){
						dLenPrev = dLenCur;

						// _XV
						m_info1 = std::move(info1);
						m_info2 = std::move(info2);
						m_point = is.point();
					}
				}
			}
		}
		// ߂ꂽB
		if(m_info1.get() && m_info2.get()){
			return true;
		}

		return false;
	}

	// m_info1, m_info2 o𐶐
	bool ChamferResult::Private::create_info()
	{
		const MGCurve& curve1 = *m_param1->curve();
		const MGCurve& curve2 = *m_param2->curve();
		const double hint1 = m_param1->hint();
		const double hint2 = m_param2->hint();

		// curve1 ̎n_Ȃ true
		const bool extend_start1 = (curve1.param_se(hint1) == curve1.param_s());
		// curve2 ̎n_Ȃ true
		const bool extend_start2 = (curve2.param_se(hint2) == curve2.param_s());

		// 1. f̂܂܂Ō_vZł邩?
		if(seek_with_extension(0, extend_start1, 0, extend_start2)){
			return true;
		}

		// 2. ǂ炩Ȃ΂ȂȂ
		// ܂ curve1 Ă݂
		const std::unique_ptr<MGStraight> ext1(mgcalc::extension(curve1, hint1));
		if(seek_with_extension(ext1.get(), extend_start1, 0, extend_start2)){
			return true;
		}

		// 3. curve2 āAIWi curve1 ƌeXgB
		const std::unique_ptr<MGStraight> ext2(mgcalc::extension(curve2, hint2));
		if(seek_with_extension(0, extend_start1, ext2.get(), extend_start2)){
			return true;
		}

		// 4. ̉m̌_ɖʎ_P[Xm
		return seek_with_extension(ext1.get(), extend_start1, ext2.get(), extend_start2);
	}

	bool ChamferResult::Private::is_null() const{
		return (   !m_param1
				|| !m_param2
				|| m_param1->is_null()
				|| m_param2->is_null()
				);
	}
} // mgcalc
