package jp.sourceforge.sxdbutils.handlers;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.LinkedHashMap;
import java.util.Map;

import org.apache.commons.dbutils.ResultSetHandler;

import jp.sourceforge.sxdbutils.SxRowProcessor;
import jp.sourceforge.sxdbutils.processors.ValueRowProcessor;

/**
 * 結果をMapで返すハンドラです。特定の項目がキーになります。
 * 
 * @author chinpei
 * 
 */
public class KeyedHandler<K, V> implements ResultSetHandler<Map<K, V>> {

	protected final SxRowProcessor<K> keyProcessor;

	protected final SxRowProcessor<V> valueProcessor;

	protected SxRowProcessor<K> getKeyProcessor() {
		if (keyProcessor == null) {
			throw new IllegalStateException("キーを処理するSxRowProcessorが指定されていません。" + "引数なしのコンストラクタを使用する場合、"
					+ "必ずgetKeyProcessorとgetValueProcessorメソッドをオーバーライドしてください。");
		}

		return keyProcessor;
	}

	protected SxRowProcessor<V> getValueProcessor() {
		if (valueProcessor == null) {
			if (keyProcessor == null) {
				throw new IllegalStateException("値を処理するSxRowProcessorが指定されていません。" + "引数なしのコンストラクタを使用する場合、"
						+ "必ずgetKeyProcessorとgetValueProcessorメソッドをオーバーライドしてください。");
			}

		}
		return valueProcessor;
	}

	public KeyedHandler() {
		this.keyProcessor = null;
		this.valueProcessor = null;
	}

	public KeyedHandler(String keyColumnName, Class<K> type, SxRowProcessor<V> processor) {
		this(new ValueRowProcessor<K>(keyColumnName, type), processor);
	}

	public KeyedHandler(String keyColumnName, SxRowProcessor<V> processor) {
		this(new ValueRowProcessor<K>(keyColumnName), processor);
	}

	public KeyedHandler(int keyColumnIndex, Class<K> keyType, SxRowProcessor<V> processor) {
		this(new ValueRowProcessor<K>(keyColumnIndex, keyType), processor);
	}

	public KeyedHandler(int columnIndex, SxRowProcessor<V> processor) {
		this(new ValueRowProcessor<K>(columnIndex), processor);
	}

	public KeyedHandler(SxRowProcessor<K> keyProcessor, SxRowProcessor<V> processor) {
		this.keyProcessor = keyProcessor;
		this.valueProcessor = processor;
	}

	public Map<K, V> handle(ResultSet rs) throws SQLException {
		Map<K, V> result = createMap();

		if (rs.next()) {
			SxRowProcessor<K> kproc = getKeyProcessor();
			SxRowProcessor<V> vproc = getValueProcessor();
			kproc.init(rs.getMetaData());
			vproc.init(rs.getMetaData());
			do {
				result.put(kproc.process(rs), vproc.process(rs));
			} while (rs.next());
		}
		return result;
	}

	protected Map<K, V> createMap() {
		return new LinkedHashMap<K, V>();
	}

}
