package jp.sourceforge.sxdbutils.processors;

import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import jp.sourceforge.sxdbutils.SxRowProcessor;
import jp.sourceforge.sxdbutils.TypeMappings;
import jp.sourceforge.sxdbutils.ValueType;
import jp.sourceforge.sxdbutils.mapping.ColumnNameMapping;
import jp.sourceforge.sxdbutils.mapping.NameMapping;
import jp.sourceforge.sxdbutils.meta.AttributeDescpriotr;
import jp.sourceforge.sxdbutils.meta.BasicResultSetToObjectEntry;
import jp.sourceforge.sxdbutils.meta.PropertyAttributeDescpritor;
import jp.sourceforge.sxdbutils.meta.ResultSetToObjectEntry;
import jp.sourceforge.sxdbutils.util.CaseInsensitiveHashMap;
import jp.sourceforge.sxdbutils.util.ReflectionUtil;

/**
 * 
 * @author chinpei
 * 
 */
abstract class AttributeRowProcessor<E> implements SxRowProcessor<E> {

	protected final Class<E> baseClass;
	protected final NameMapping nameMapping;
	private static final NameMapping DEFAULT_NAME_MAPPING = new ColumnNameMapping();

	public AttributeRowProcessor(Class<E> baseClass) {
		this(baseClass, DEFAULT_NAME_MAPPING);
	}

	public AttributeRowProcessor(Class<E> clazz, NameMapping nameMapping) {
		this.baseClass = clazz;
		this.nameMapping = nameMapping;
	}

	protected ResultSetToObjectEntry[] bindingEntries;

	public void init(ResultSetMetaData rsmd) throws SQLException {
		Map<String, AttributeDescpriotr> attributeDescpriotrNameMap = new HashMap<String, AttributeDescpriotr>();
		putDescriptorToMap(this.baseClass, attributeDescpriotrNameMap);

		Map<String, AttributeDescpriotr> attributeDescpriotrMap = new CaseInsensitiveHashMap<AttributeDescpriotr>();
		for (Iterator<Map.Entry<String, AttributeDescpriotr>> iterator = attributeDescpriotrNameMap.entrySet()
				.iterator(); iterator.hasNext();) {
			Map.Entry<String, AttributeDescpriotr> entry = iterator.next();
			attributeDescpriotrMap.put(this.nameMapping.toIntermediateNameFromAttrName(entry.getKey()), entry
					.getValue());
		}

		List<ResultSetToObjectEntry> list = new ArrayList<ResultSetToObjectEntry>();
		for (int i = 0; i < rsmd.getColumnCount(); i++) {
			ResultSetToObjectEntry bindingEntry = getBindingEntry(rsmd, i + 1, attributeDescpriotrMap);
			if (bindingEntry != null)
				list.add(bindingEntry);
		}
		this.bindingEntries = list.toArray(new ResultSetToObjectEntry[list.size()]);
	}

	@SuppressWarnings("unchecked")
	public E process(ResultSet rs) throws SQLException {
		E bean = (E) ReflectionUtil.newInstance(this.baseClass);
		ResultSetToObjectEntry bindingEntry = null;
		for (int i = 0; i < bindingEntries.length; i++) {
			bindingEntry = bindingEntries[i];
			bindingEntry.set(bean, rs);
		}

		return bean;
	}

	protected abstract void putDescriptorToMap(Class<E> clazz, Map<String, AttributeDescpriotr> descriptorMap);

	/**
	 * カラムとプロパティのマッピング情報を返します。
	 * @param rsmd
	 * @param columnIndex
	 * @param attributeDescpriotrMap
	 * @return
	 * @throws SQLException
	 */
	protected ResultSetToObjectEntry getBindingEntry(ResultSetMetaData rsmd, int columnIndex,
			Map<String, AttributeDescpriotr> attributeDescpriotrMap) throws SQLException {

		String columnName = rsmd.getColumnLabel(columnIndex);
		AttributeDescpriotr attrDescpriotr = attributeDescpriotrMap.get(nameMapping
				.toIntermediateNameFromColumnName(columnName));
		if (attrDescpriotr == null)
			return null;
		if (!attrDescpriotr.isWriteable())
			return null;

		int sqlType = rsmd.getColumnType(columnIndex);
		ValueType valueType = getValueType(attrDescpriotr, sqlType);
		if (valueType == null)
			return null;

		ResultSetToObjectEntry propertySetter = new BasicResultSetToObjectEntry(columnName, valueType, sqlType,
				columnIndex, attrDescpriotr);
		return propertySetter;
	}

	/**
	 * Attribute名をキーにしたMapを返します。
	 * 
	 * @return
	 */
	protected Map<String, AttributeDescpriotr> attributeDescpriotrNameMap() {
		Map<String, AttributeDescpriotr> attributeDescpriotrNameMap = new HashMap<String, AttributeDescpriotr>();
		PropertyAttributeDescpritor.putDescriptorToMap(this.baseClass, attributeDescpriotrNameMap);
		return attributeDescpriotrNameMap;
	}

	protected ValueType getValueType(AttributeDescpriotr attrDescriptor, int sqlType) {
		return TypeMappings.getValueType(attrDescriptor.getType(), sqlType);
	}

	protected ValueType getValueType(Class<E> type, int sqlType) {
		return TypeMappings.getValueType(type, sqlType);
	}

	/**
	 * マッピング対象クラスを返します。
	 * 
	 * @return
	 */
	public Class<E> getBaseClass() {
		return baseClass;
	}

}
