package jp.sourceforge.sxdbutils.processors;

import java.beans.PropertyDescriptor;
import java.lang.reflect.Method;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import jp.sourceforge.sxdbutils.SxRowProcessor;
import jp.sourceforge.sxdbutils.TypeMappings;
import jp.sourceforge.sxdbutils.ValueType;
import jp.sourceforge.sxdbutils.bean.ColumnNameMapping;
import jp.sourceforge.sxdbutils.bean.NameMapping;
import jp.sourceforge.sxdbutils.util.ReflectionUtil;

/**
 * Beanにマッピングする {@link SxRowProcessor}。 カラム名とプロパティ名の名前が、 {@link NameMapping}
 * によって一致したものがマッピングされます。 デフォルトでは、キャメル形式でマッピングされます。
 * 
 * @author chinpei
 * 
 */
public class BeanRowProcessor implements SxRowProcessor {

	protected final Class beanClass;
	protected final NameMapping nameMapping;
	private static final NameMapping DEFAULT_NAME_MAPPING = new ColumnNameMapping();

	public BeanRowProcessor(Class beanClass) {
		this(beanClass, DEFAULT_NAME_MAPPING);
	}

	public BeanRowProcessor(Class beanClass, NameMapping nameMapping) {
		this.beanClass = beanClass;
		this.nameMapping = nameMapping;
	}

	protected PropertyMpping[] propertyMppings;

	/**
	 * 初期化。 {@link ResultSetMetaData}とマッピングするクラスとの情報を作成し、
	 * protectedなフィールド、propertyMppingとして保持させます。
	 */
	public void init(ResultSetMetaData rsmd) throws SQLException {
		Map propertyDescriptorMap = readPropertyDescriptor(this.beanClass);
		List list = new ArrayList();
		for (int i = 0; i < rsmd.getColumnCount(); i++) {
			PropertyMpping propertyMapping = getPropertyMapping(rsmd, i + 1,
					propertyDescriptorMap);
			if (propertyMapping != null)
				list.add(propertyMapping);
		}
		this.propertyMppings = (PropertyMpping[]) list
				.toArray(new PropertyMpping[list.size()]);
	}

	/**
	 * マッピングされたオブジェクトを返します。
	 * ここでは、initで初期化されたpropertyMppingsフィールドを使用して、値をセットしています。
	 */
	public Object process(ResultSet rs) throws SQLException {
		Object bean = ReflectionUtil.newInstance(this.beanClass);
		PropertyMpping propertyMpping = null;
		for (int i = 0; i < propertyMppings.length; i++) {
			propertyMpping = propertyMppings[i];
			propertyMpping.writeProperty(bean, rs);
		}

		return bean;
	}

	/**
	 * カラムとプロパティのマッピング情報を返します。
	 * 
	 * @param rsmd
	 * @param propertyDescriptorMap
	 * @param columnIndex
	 * @return
	 * @throws SQLException
	 */
	protected PropertyMpping getPropertyMapping(ResultSetMetaData rsmd,
			int columnIndex, Map propertyDescriptorMap) throws SQLException {

		String columnName = rsmd.getColumnLabel(columnIndex);

		PropertyDescriptor propertyDescriptor = (PropertyDescriptor) propertyDescriptorMap
				.get(nameMapping.toIntermediateName(columnName));
		if (propertyDescriptor == null)
			return null;
		Method setter = propertyDescriptor.getWriteMethod();
		if (setter == null || "class".equals(propertyDescriptor.getName()))
			return null;

		int sqlType = rsmd.getColumnType(columnIndex);
		ValueType valueType = getValueType(propertyDescriptor, sqlType);
		if (valueType == null)
			return null;

		PropertyMpping propertySetter = new PropertyMpping();
		propertySetter.columnIndex = columnIndex;
		propertySetter.columnName = columnName;
		propertySetter.sqlType = sqlType;
		propertySetter.setter = setter;
		propertySetter.valueType = valueType;
		return propertySetter;
	}

	protected ValueType getValueType(PropertyDescriptor propertyDescriptor,
			int sqlType) {
		return TypeMappings.getValueType(propertyDescriptor.getPropertyType(),
				sqlType);
	}

	/**
	 * クラスから{@link PropertyDescriptor}を読み込み、マッピングするカラム名をキーにしたMapを返す。
	 * 親クラスからも情報を取得し、{@link Object}になるまで再起します。
	 * 
	 * @param beanClass
	 * @return
	 */
	protected Map readPropertyDescriptor(Class beanClass) {
		Map propertyDescriptorMap = this.nameMapping.createIntermediateMap();
		readPropertyDescriptorToMap(beanClass, propertyDescriptorMap);
		return propertyDescriptorMap;
	}

	/**
	 * クラスから{@link PropertyDescriptor}を読み込み、マッピングするカラム名をキーにしたMapに詰め込む。
	 * 親クラスからも情報を取得し、 {@link Object}になるまで再起します。
	 * 
	 * @param type
	 * @param propertyDescriptorMap
	 */
	protected void readPropertyDescriptorToMap(Class type,
			Map propertyDescriptorMap) {

		if ("java.lang.Object".equals(type.getName()))
			return;
		else {
			readPropertyDescriptorToMap(type.getSuperclass(),
					propertyDescriptorMap);
		}

		PropertyDescriptor[] propertyDescriptors = ReflectionUtil
				.propertyDescriptors(type);
		for (int i = 0; i < propertyDescriptors.length; i++) {
			PropertyDescriptor propertyDescriptor = propertyDescriptors[i];
			String intermediateName = this.nameMapping
					.toIntermediateName(propertyDescriptor);
			propertyDescriptorMap.put(intermediateName, propertyDescriptor);
		}
	}

	/**
	 * カラムとプロパティのマッピング
	 */
	protected static class PropertyMpping {
		String columnName;
		ValueType valueType;
		Method setter;
		int sqlType;
		int columnIndex;

		void writeProperty(Object bean, ResultSet rset) throws SQLException {
			try {
				ReflectionUtil.invoke(bean, setter, valueType.getValue(rset,
						this.columnIndex));
			} catch (Exception e) {
				throw new RuntimeException(toString() + "で処理する際に、例外が発生しました。", e);
			}
		}

		public String toString() {
			return "columnName=" + columnName + "; sqlType=" + sqlType
					+ "; setter=" + setter.getName() + "; resultReader="
					+ valueType.getClass().getName();
		}
	}

}
