package jp.sourceforge.sxdbutils.query;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import jp.sourceforge.sxdbutils.TypeMappings;
import jp.sourceforge.sxdbutils.ValueType;
import jp.sourceforge.sxdbutils.util.ColumnInfo;

/**
 * MapQueryFactoryのビルダーです。
 * 
 * @author chinpei
 * 
 */
public class MapQueryFactoryBuilder {

	protected final String tableName;

	public MapQueryFactoryBuilder(String tableName) {
		this.tableName = tableName;
	}

	/**
	 * Keyがカラム名で ValueがValueTypeです。
	 * 
	 */
	protected Map columns = new LinkedHashMap();
	protected Set updateKeyColumnSet;

	public MapQueryFactoryBuilder addColumn(String columnName) {
		columns.put(columnName, TypeMappings.OBJECT_TYPE);
		return this;
	}

	public MapQueryFactoryBuilder addColumn(String columnName, int sqlType) {
		columns.put(columnName, TypeMappings.getValueType(sqlType));
		return this;
	}

	public MapQueryFactoryBuilder addColumn(String columnName, Class valueClass) {
		columns.put(columnName, TypeMappings.getValueType(valueClass));
		return this;
	}

	public MapQueryFactoryBuilder addColumn(String columnName, int sqlType,
			Class valueClass) {
		columns.put(columnName, TypeMappings.getValueType(valueClass, sqlType));
		return this;
	}

	public MapQueryFactoryBuilder addColumns(String[] columnNames) {
		for (int i = 0; i < columnNames.length; i++) {
			addColumn(columnNames[i]);
		}
		return this;
	}

	public MapQueryFactoryBuilder addColumns(Collection columnNames) {
		for (Iterator iterator = columnNames.iterator(); iterator.hasNext();) {
			addColumn((String) iterator.next());
		}
		return this;
	}

	public MapQueryFactoryBuilder updateKeyColumn(String columnNames) {
		if (this.updateKeyColumnSet == null) {
			this.updateKeyColumnSet = new LinkedHashSet();
		}
		this.updateKeyColumnSet.add(columnNames.toLowerCase());
		return this;
	}

	private boolean isUpdateKey(String columnName) {
		if (this.updateKeyColumnSet == null)
			return false;
		return this.updateKeyColumnSet.contains(columnName.toLowerCase());
	}

	public QueryFactory buildUpdate() {
		ColumnInfo[] columnInfos = toColumnInfos();

		List updateColumnInfos = new ArrayList();
		List whereColumnInfos = new ArrayList();

		StringBuffer columnBuffer = new StringBuffer(columnInfos.length * 7);

		columnBuffer.append("update ").append(this.tableName).append(" set ");
		boolean flg = false;
		for (int i = 0; i < columnInfos.length; i++) {

			ColumnInfo columnInfo = columnInfos[i];

			// Where句になる。
			if (isUpdateKey(columnInfo.getColumnName())) {
				whereColumnInfos.add(columnInfo);
				continue;
			}
			if (flg) {
				columnBuffer.append(',');
			} else {
				flg = true;
			}

			columnBuffer.append(columnInfo.getColumnName()).append("=?");
			updateColumnInfos.add(columnInfo);
		}

		StringBuffer whereBuffer = null;
		if (!whereColumnInfos.isEmpty()) {
			for (Iterator iterator = whereColumnInfos.iterator(); iterator
					.hasNext();) {
				if (whereBuffer == null)
					whereBuffer = new StringBuffer(" where ");
				else
					whereBuffer.append(" and ");
				ColumnInfo columnInfo = (ColumnInfo) iterator.next();
				whereBuffer.append(columnInfo.getColumnName()).append("=?");
			}
		}

		List list = new ArrayList(updateColumnInfos);
		list.addAll(whereColumnInfos);
		if (whereBuffer != null)
			columnBuffer.append(whereBuffer);

		String sql = columnBuffer.toString();
		return new MapQueryFactory(sql, (ColumnInfo[]) list
				.toArray(new ColumnInfo[list.size()]));
	}

	public QueryFactory buildInsert() {
		ColumnInfo[] columnInfos = toColumnInfos();
		StringBuffer columnBuffer = new StringBuffer(columnInfos.length * 7);
		StringBuffer valueBuffer = new StringBuffer(columnInfos.length * 2);

		columnBuffer.append("insert into ").append(this.tableName).append("(");
		valueBuffer.append(" values ( ");
		boolean flg = false;
		for (int i = 0; i < columnInfos.length; i++) {

			if (flg) {
				columnBuffer.append(',');
				valueBuffer.append(',');
			} else {
				flg = true;
			}
			ColumnInfo columnInfo = columnInfos[i];
			columnBuffer.append(columnInfo.getColumnName());
			valueBuffer.append('?');
		}
		columnBuffer.append(" ) ");
		valueBuffer.append(" ) ");
		String sql = columnBuffer.append(valueBuffer).toString();

		return new MapQueryFactory(sql, columnInfos);

	}

	protected ColumnInfo[] toColumnInfos() {
		ColumnInfo[] result = new ColumnInfo[columns.size()];
		int i = 0;
		for (Iterator iterator = columns.entrySet().iterator(); iterator
				.hasNext();) {
			Map.Entry entry = (Map.Entry) iterator.next();
			result[i++] = new ColumnInfo((String) entry.getKey(),
					(ValueType) entry.getValue());
		}
		return result;
	}

}
