/*
 * Copyright 2003-2006 Stream Converter Project Team.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sf.stconv.storage.impl;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import jp.sf.stconv.storage.StreamStorage;
import jp.sf.stconv.storage.StreamStorageException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public class ByteArrayStreamStorageImpl extends AbstractStreamStorage implements
        StreamStorage {
    /**
     * Logger for this class
     */
    private static final Log log = LogFactory
            .getLog(ByteArrayStreamStorageImpl.class);

    private int DEFAULT_ARRAY_SIZE = 512;

    private InputStream inputStream;

    private ByteArrayOutputStream outputStream;

    private String encoding;

    private byte[] buffer = null;

    public ByteArrayStreamStorageImpl(InputStream in, String encoding)
            throws StreamStorageException {
        init(in, encoding);
    }

    public void init(InputStream in, String encoding)
            throws StreamStorageException {
        // TODO better to wrap BufferredInputStream
        inputStream = in;
        this.encoding = encoding;
        outputStream = new ByteArrayOutputStream(DEFAULT_ARRAY_SIZE);
    }

    public void destroy() {
        buffer = null;
        outputStream = null;
    }

    /*
     * (non-Javadoc)
     * 
     * @see jp.sf.stconv.storage.impl.StreamStorage#commit()
     */
    public void commit() throws StreamStorageException {
        try {
            if (inputStream != null) {
                inputStream.close();
            }
            if (outputStream != null) {
                outputStream.flush();
                outputStream.close();
            }

            buffer = outputStream.toByteArray();
            inputStream = new ByteArrayInputStream(buffer);
            outputStream = new ByteArrayOutputStream(DEFAULT_ARRAY_SIZE);
        }

        catch (IOException e) {
            log.error("I/O error occurs. ", e);
            throw new StreamStorageException((Throwable) e);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see jp.sf.stconv.storage.impl.StreamStorage#getResultInputStream()
     */
    public InputStream getResultInputStream() throws StreamStorageException {
        try {

            if (buffer == null) {
                if (inputStream.markSupported()) {
                    inputStream.reset();
                }
                return inputStream;
            }

            return new ByteArrayInputStream(buffer);
        }

        catch (IOException e) {
            log.error("I/O error occurs. ", e);
            throw new StreamStorageException((Throwable) e);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see jp.sf.stconv.storage.impl.StreamStorage#getEncoding()
     */
    public String getEncoding() {
        return encoding;
    }

    /**
     * @param encoding
     *            The encoding to set.
     */
    public void setEncoding(String encoding) {
        this.encoding = encoding;
    }

    /*
     * (non-Javadoc)
     * 
     * @see jp.sf.stconv.storage.impl.StreamStorage#getInputStream()
     */
    public InputStream getInputStream() {
        return inputStream;
    }

    /**
     * @param inputStream
     *            The inputStream to set.
     */
    public void setInputStream(InputStream inputStream) {
        this.inputStream = inputStream;
    }

    /*
     * (non-Javadoc)
     * 
     * @see jp.sf.stconv.storage.impl.StreamStorage#getOutputStream()
     */
    public OutputStream getOutputStream() {
        return outputStream;
    }

    /**
     * @param outputStream
     *            The outputStream to set.
     */
    public void setOutputStream(OutputStream outputStream) {
        if (outputStream instanceof ByteArrayOutputStream) {
            this.outputStream = (ByteArrayOutputStream) outputStream;
        } else {
            throw new IllegalArgumentException(
                    "The output stream is not ByteArrayOutputStream.");
        }
    }
}
