/*
 * Copyright 2003-2006 Stream Converter Project Team.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sf.stconv.storage.impl;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import jp.sf.stconv.storage.StreamStorage;
import jp.sf.stconv.storage.StreamStorageException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public class FileStreamStorageImpl extends AbstractStreamStorage implements
        StreamStorage {
    /**
     * Logger for this class
     */
    private static final Log log = LogFactory
            .getLog(FileStreamStorageImpl.class);

    private String DEFAULT_TMPFILE_PREFIX = "STREAMSTORAGE_";

    private String DEFAULT_TMPFILE_SUFFIX = ".tmp";

    private InputStream inputStream;

    private OutputStream outputStream;

    private String encoding;

    private File previousFile;

    private File currentFile;

    public FileStreamStorageImpl(InputStream in, String encoding)
            throws StreamStorageException {
        init(in, encoding);
    }

    public void init(InputStream in, String encoding)
            throws StreamStorageException {
        try {
            // TODO better to wrap BufferredInputStream
            inputStream = in;
            this.encoding = encoding;
            this.previousFile = null;
            this.currentFile = File.createTempFile(DEFAULT_TMPFILE_PREFIX,
                    DEFAULT_TMPFILE_SUFFIX);
            outputStream = new FileOutputStream(currentFile);
        } catch (FileNotFoundException e) {
            log.error("Cannot find the temp file: " + currentFile.getName(), e);
            throw new StreamStorageException((Throwable) e);
        } catch (IOException e) {
            log.error("I/O error occurs. ", e);
            throw new StreamStorageException((Throwable) e);
        }
    }

    public void destroy() {
        if (previousFile != null && previousFile.isFile()) {
            previousFile.deleteOnExit();
        }
        if (currentFile != null && currentFile.isFile()) {
            currentFile.deleteOnExit();
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see jp.sf.stconv.storage.impl.StreamStorage#commit()
     */
    public void commit() throws StreamStorageException {
        try {
            if (inputStream != null) {
                inputStream.close();
            }
            if (outputStream != null) {
                outputStream.flush();
                outputStream.close();
            }

            if (previousFile != null && previousFile.isFile()) {
                previousFile.deleteOnExit();
            }

            previousFile = currentFile;
            currentFile = File.createTempFile(DEFAULT_TMPFILE_PREFIX,
                    DEFAULT_TMPFILE_SUFFIX);
            inputStream = new FileInputStream(previousFile);
            outputStream = new FileOutputStream(currentFile);
        } catch (FileNotFoundException e) {
            log.error("Cannot find the temp file. ", e);
            throw new StreamStorageException((Throwable) e);
        } catch (IOException e) {
            log.error("I/O error occurs. ", e);
            throw new StreamStorageException((Throwable) e);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see jp.sf.stconv.storage.impl.StreamStorage#getResultInputStream()
     */
    public InputStream getResultInputStream() throws StreamStorageException {
        try {
            if (log.isDebugEnabled()) {
                log.debug("getResultInputStream() - previousFile="
                        + previousFile);
            }

            if (previousFile == null) {
                if (inputStream.markSupported()) {
                    inputStream.reset();
                }
                return inputStream;
            }

            return new FileInputStream(previousFile);
        } catch (FileNotFoundException e) {
            log
                    .error("Cannot find the temp file: "
                            + previousFile.getName(), e);
            throw new StreamStorageException((Throwable) e);
        } catch (IOException e) {
            log.error("I/O error occurs. ", e);
            throw new StreamStorageException((Throwable) e);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see jp.sf.stconv.storage.impl.StreamStorage#getEncoding()
     */
    public String getEncoding() {
        return encoding;
    }

    /**
     * @param encoding
     *            The encoding to set.
     */
    public void setEncoding(String encoding) {
        this.encoding = encoding;
    }

    /*
     * (non-Javadoc)
     * 
     * @see jp.sf.stconv.storage.impl.StreamStorage#getInputStream()
     */
    public InputStream getInputStream() {
        return inputStream;
    }

    /**
     * @param inputStream
     *            The inputStream to set.
     */
    public void setInputStream(InputStream inputStream) {
        this.inputStream = inputStream;
    }

    /*
     * (non-Javadoc)
     * 
     * @see jp.sf.stconv.storage.impl.StreamStorage#getOutputStream()
     */
    public OutputStream getOutputStream() {
        return outputStream;
    }

    /**
     * @param outputStream
     *            The outputStream to set.
     */
    public void setOutputStream(OutputStream outputStream) {
        this.outputStream = outputStream;
    }
}
