/*
 * Copyright (C) 2021- TeraTerm Project
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <windows.h>
#define _CRTDBG_MAP_ALLOC
#include <stdlib.h>
#include <crtdbg.h>

#include "asprintf.h"

#include "win32helper.h"

/**
 *	GetModuleFileNameW() ̓Iobt@
 *
 *	@param buf	i[obt@
 *				svɂȂfree()
 *	@return	G[R[h,0(=NO_ERROR)̂ƂG[Ȃ
 */
DWORD hGetModuleFileNameW(HMODULE hModule, wchar_t **buf)
{
	size_t size = MAX_PATH;
	wchar_t *b = (wchar_t*)malloc(sizeof(wchar_t) * size);
	DWORD error;
	if (b == NULL) {
		error = ERROR_NOT_ENOUGH_MEMORY;
		goto error_return;
	}

	for(;;) {
		DWORD r = GetModuleFileNameW(hModule, b, (DWORD)size);
		if (r == 0) {
			// ֐s
			error = GetLastError();
			break;
		} else if (r < size - 1) {
			// 擾
			size = r + 1;
			b = (wchar_t*)realloc(b, sizeof(wchar_t) * size);
			*buf = b;
			return NO_ERROR;
		} else {
			size *= 2;
			wchar_t *p = (wchar_t*)realloc(b, sizeof(wchar_t) * size);
			if (p == NULL) {
				free(b);
				error = ERROR_NOT_ENOUGH_MEMORY;
				break;
			}
			b = p;
		}
    }

	// error
	free(b);
error_return:
	*buf = NULL;
	return error;
}

/**
 *	GetPrivateProfileStringW() ̓Iobt@
 *
 *	@param section
 *	@param key
 *	@param def
 *	@param ini		init@C̃pX
 *					NULL̂Ƃ̓t@CwȂ
 *	@param str		i[obt@
 *					svɂȂfree()
 *	@return	G[R[h,0(=NO_ERROR)̂ƂG[Ȃ
 *
 *		̏ꍇ str = L"" Ԃ (free()邱)
 *			keyɋ󂪐ݒ肳Ă 'key='ƋLq ("="̌ɉĂȂ)
 *				߂l NO_ERROR
 *			key݂def=NULL ꍇ
 *				߂l ERROR_FILE_NOT_FOUND
 *			GetPrivateProfileStringW() ŃG[ꍇ
 *				߂l G[l
 *			mۂłȂꍇ
 *				߂l ERROR_NOT_ENOUGH_MEMORY
 */
DWORD hGetPrivateProfileStringW(const wchar_t *section, const wchar_t *key, const wchar_t *def, const wchar_t *ini, wchar_t **str)
{
	size_t size;
	wchar_t *b;
	DWORD error;

	if (ini == NULL) {
		// init@CwȂ̂ƂftHglԂ
		//		GetPrivateProfileStringW(A)() ̎dlɂ
		//		t@CNULLnėǂƂ̋Lq͂Ȃ
		//			NULLnƂA
		//			Windows10,11 ł̓t@CȂƂƓ
		//			Windows95 ͎dlO̓
		//				߂l=obt@TCY+2ԂĂ
		*str = _wcsdup(def);
		if (*str == NULL) {
			return ERROR_NOT_ENOUGH_MEMORY;
		}
		return NO_ERROR;
	}
	size = 256;
	b = (wchar_t*)malloc(sizeof(wchar_t) * size);
	if (b == NULL) {
		error = ERROR_NOT_ENOUGH_MEMORY;
		goto error_return;
	}
	b[0] = 0;
	for(;;) {
		DWORD r = GetPrivateProfileStringW(section, key, def, b, (DWORD)size, ini);
		if (r == 0 || b[0] == L'\0') {
			error = GetLastError();
			free(b);
			*str = _wcsdup(L"");
			return error;
		} else if (r < size - 2) {
			size = r + 1;
			b = (wchar_t *)realloc(b, sizeof(wchar_t) * size);
			*str = b;
			return NO_ERROR;
		} else {
			wchar_t *p;
			size *= 2;
			p = (wchar_t*)realloc(b, sizeof(wchar_t) * size);
			if (p == NULL) {
				error = ERROR_NOT_ENOUGH_MEMORY;
				break;
			}
			b = p;
		}
	}

	// error
	free(b);
error_return:
	*str = NULL;
	return error;
}

/**
 *	GetFullPathNameW() ̓Iobt@
 *
 *	@param fullpath		fullpathi[obt@
 *						svɂȂfree()
 *	@return	G[R[h,0(=NO_ERROR)̂ƂG[Ȃ
 */
DWORD hGetFullPathNameW(const wchar_t *lpFileName, wchar_t **fullpath, wchar_t **filepart)
{
	size_t len = GetFullPathNameW(lpFileName, 0, NULL, NULL);		// include L'\0'
	if (len == 0) {
		*fullpath = NULL;
		if (filepart != NULL) {
			*filepart = NULL;
		}
		return GetLastError();
	}
	wchar_t *path = (wchar_t *)malloc(sizeof(wchar_t) * len);
	wchar_t *file;
	len = GetFullPathNameW(lpFileName, (DWORD)len, path, &file);
	if (len == 0) {
		free(path);
		return GetLastError();
	}
	if (filepart != NULL) {
		*filepart = file;
	}
	*fullpath = path;
	return NO_ERROR;
}

/**
 *	GetCurrentDirectoryW() ̓Iobt@
 *
 *	@param[out]	dir		tH_
 *						svɂȂfree()
 *	@return	G[R[h,0(=NO_ERROR)̂ƂG[Ȃ
 */
DWORD hGetCurrentDirectoryW(wchar_t **dir)
{
	DWORD len = GetCurrentDirectoryW(0, NULL);
	if (len == 0) {
		*dir = NULL;
		return GetLastError();
	}
	wchar_t *d = (wchar_t *)malloc(sizeof(wchar_t) * len);
	len = GetCurrentDirectoryW(len, d);
	if (len == 0) {
		free(d);
		*dir = NULL;
		return GetLastError();
	}
	*dir = d;
	return 0;
}

/**
 *	hWnd̕擾
 *	svɂȂ free() 邱
 *
 *	@param[out]	text	ݒ肳Ă镶
 *						svɂȂfree()
 *	@return	G[R[h,0(=NO_ERROR)̂ƂG[Ȃ
 */
DWORD hGetWindowTextW(HWND hWnd, wchar_t **text)
{
	// GetWindowTextLengthW()  0 ԂƂA
	// G[ȂG[Zbg邪A
	// G[ł͂ȂƂ(I)AG[NAȂ(G[ȂZbgȂ)
	// ŃG[NAĂ
	SetLastError(NO_ERROR);
	int len = GetWindowTextLengthW(hWnd);
	if (len == 0) {
		DWORD err = GetLastError();
		if (err != NO_ERROR) {
			*text = NULL;
			return err;
		}
		*text = _wcsdup(L"");
		return 0;
	}

	wchar_t *strW = (wchar_t *)malloc(sizeof(wchar_t) * (len + 1));
	if (strW == NULL) {
		*text = NULL;
		return ERROR_NOT_ENOUGH_MEMORY;
	}

	GetWindowTextW(hWnd, strW, len + 1);
	strW[len] = 0;
	*text = strW;
	return 0;
}

DWORD hGetDlgItemTextW(HWND hDlg, int id, wchar_t **text)
{
	HWND hWnd = GetDlgItem(hDlg, id);
	return hGetWindowTextW(hWnd, text);
}

DWORD hExpandEnvironmentStringsW(const wchar_t *src, wchar_t **expanded)
{
	size_t len = (size_t)ExpandEnvironmentStringsW(src, NULL, 0);
	wchar_t *dest = (wchar_t *)malloc(sizeof(wchar_t) * len);
	if (dest == NULL) {
		*expanded = NULL;
		return ERROR_NOT_ENOUGH_MEMORY;
	}
	ExpandEnvironmentStringsW(src, dest, (DWORD)len);
	*expanded = dest;
	return NO_ERROR;
}
